/* Copyright (C) 2006 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

// SerialLogTransaction.cpp: implementation of the SerialLogTransaction class.
//
//////////////////////////////////////////////////////////////////////

#include <memory.h>
#include "Engine.h"
#include "SerialLogTransaction.h"
#include "SerialLog.h"
#include "SerialLogControl.h"
#include "SerialLogWindow.h"
#include "Transaction.h"
#include "Database.h"

#ifdef _DEBUG
static char THIS_FILE[]=__FILE__;
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

SerialLogTransaction::SerialLogTransaction(SerialLog *serialLog, TransId transId) : SerialLogAction(serialLog)
{
	transactionId = transId;
	state = sltUnknown;
	window = NULL;
	finished = false;
	transaction = NULL;
	maxBlockNumber = minBlockNumber = physicalBlockNumber = 0;
	xidLength = 0;
	xid = NULL;
}

SerialLogTransaction::~SerialLogTransaction()
{
	if (window)
		window->release();
		
	log->transactionDelete(this);
	delete [] xid;
}

void SerialLogTransaction::commit()
{
	SerialLogControl control(log);
	window->activateWindow(true);
	SerialLogBlock *block = (SerialLogBlock*) (window->buffer + blockOffset);
	control.setWindow(window, block, recordOffset);
	window->deactivateWindow();
	finished = false;

	for (SerialLogRecord *record; !finished && (record = control.nextRecord());)
		if (record->transactionId == transactionId)
			record->commit();
	
	if (transaction)
		transaction->writeComplete();
}

void SerialLogTransaction::rollback()
{
	SerialLogControl control(log);
	window->activateWindow(true);
	SerialLogBlock *block = (SerialLogBlock*) (window->buffer + blockOffset);
	control.setWindow(window, block, recordOffset);
	window->deactivateWindow();
	finished = false;

	for (SerialLogRecord *record; !finished && (record = control.nextRecord());)
		if (record->transactionId == transactionId)
			record->rollback();
}

void SerialLogTransaction::setStart(const UCHAR *record, SerialLogBlock *block, SerialLogWindow *win)
{
	if (win != window)
		{
		win->addRef();
		
		if (window)
			window->release();
		}
		
	window = win;
	blockNumber = block->blockNumber;
	minBlockNumber = blockNumber;
	maxBlockNumber = blockNumber;
	blockOffset = (UCHAR*) block - window->buffer;
	recordOffset = record - block->data;
}

void SerialLogTransaction::setState(sltState newState)
{
	state = newState;
}

void SerialLogTransaction::setFinished()
{
	finished = true;
}

bool SerialLogTransaction::isRipe()
{
	return (state == sltCommitted ||
		    state == sltRolledBack);
}

void SerialLogTransaction::doAction()
{
	if (state == sltCommitted)
		commit();
	else
		rollback();

}

void SerialLogTransaction::preRecovery()
{
	if (state == sltUnknown)
		setState(sltRolledBack);
}

bool SerialLogTransaction::completedRecovery()
{
	if (state == sltPrepared)
		return false;

	return true;
}

bool SerialLogTransaction::isTransaction()
{
	return true;
}

uint64 SerialLogTransaction::getBlockNumber()
{
	return blockNumber;
}


void SerialLogTransaction::setPhysicalBlock()
{
	if (log->writeBlock->blockNumber > physicalBlockNumber)
		{
		physicalBlockNumber = log->writeBlock->blockNumber;
		maxBlockNumber = MAX(physicalBlockNumber, maxBlockNumber);
		}
}

void SerialLogTransaction::setXID(int length, const UCHAR* xidPtr)
{
	xidLength = length;
	xid = new UCHAR[xidLength];
	memcpy(xid, xidPtr, xidLength);
}

bool SerialLogTransaction::isXidEqual(int testLength, const UCHAR* test)
{
	if (testLength != xidLength)
		return false;
	
	return memcmp(xid, test, xidLength) == 0;
}
