/* Copyright (C) 2006 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

#include <stdio.h>
#include "Engine.h"
#include "SRLUpdateIndex.h"
#include "DeferredIndex.h"
#include "Index.h"
#include "IndexRootPage.h"
#include "Index2RootPage.h"
#include "IndexKey.h"
#include "DeferredIndexWalker.h"
#include "Transaction.h"
#include "SerialLogTransaction.h"
#include "SerialLogControl.h"
#include "Dbb.h"

SRLUpdateIndex::SRLUpdateIndex(void)
{
}

SRLUpdateIndex::~SRLUpdateIndex(void)
{
}

void SRLUpdateIndex::append(Transaction *transaction, DeferredIndex* deferredIndex)
{
	DeferredIndexWalker walker(deferredIndex, NULL);
	uint indexId = deferredIndex->index->indexId;
	int idxVersion = deferredIndex->index->indexVersion;
		
	for (DINode *node = walker.next(); node;)
		{
		START_RECORD(srlUpdateIndex, "SRLUpdateIndex::append");
		UCHAR *org = log->writePtr;
		
		if (indexId >= log->indexUseVector.length)
			log->extendIndexVector(indexId + 10);
		
		++log->indexUseVector.vector[indexId];
		SerialLogTransaction *trans = log->getTransaction(transaction->transactionId);
		trans->transaction = transaction;
		ASSERT(transaction->writePending);
		putInt(transaction->transactionId);
		putInt(indexId);
		putInt(idxVersion);
		UCHAR *lengthPtr = putFixedInt(0);
		UCHAR *start = log->writePtr;
		UCHAR *end = log->writeWarningTrack;

		for (; node; node = walker.next())
			{
			if (log->writePtr + 
				byteCount(node->pageNumber) + byteCount(node->keyLength) + node->keyLength >= end)
				break;
			
			putInt(node->pageNumber);
			putInt(node->keyLength);
			log->putData(node->keyLength, node->key);
			}
		
		int len = log->writePtr - start;
		//printf("SRLUpdateIndex::append tid %d, index %d, length %d, ptr %x (%x)\n",  transaction->transactionId, indexId, len, lengthPtr, org);
		ASSERT(len >= 0);
		putFixedInt(len, lengthPtr);
		const UCHAR *p = lengthPtr;
		ASSERT(getInt(&p) == len);
		
		if (node)
			log->flush(true, 0, &sync);
		else
			sync.unlock();
		}

}

void SRLUpdateIndex::read(void)
{
	transactionId = getInt();
	indexId = getInt();
	
	if (control->version >= srlVersion6)
		indexVersion = getInt();
	else
		indexVersion = INDEX_VERSION_1;

	const UCHAR *p = control->input;
	dataLength = getInt();
	data = getData(dataLength);
}

void SRLUpdateIndex::print(void)
{
	logPrint("UpdateIndex: transaction %d, length %d\n",
			transactionId, dataLength);

	for (const UCHAR *p = data, *end = data + dataLength; p < end;)
		{
		int recordNumber = getInt(&p);
		int length = getInt(&p);
		char temp[40];
		printf("   rec %d to index %d %s\n", recordNumber, indexId, format(length, p, sizeof(temp), temp));
		p += length;
		}
}

void SRLUpdateIndex::pass1(void)
{
	control->getTransaction(transactionId);
}

void SRLUpdateIndex::redo(void)
{
	execute();
}

void SRLUpdateIndex::commit(void)
{
	Sync sync(&log->syncIndexes, "SRLUpdateIndex::commit");
	sync.lock(Shared);
	--log->indexUseVector.vector[indexId];
	execute();
	log->setPhysicalBlock(transactionId);
}

void SRLUpdateIndex::execute(void)
{
	if (!log->isIndexActive(indexId))
		return;

	SerialLogTransaction *transaction = control->getTransaction(transactionId);
	ptr = data;
	end = ptr + dataLength;
	
	switch (indexVersion)
		{
		case INDEX_VERSION_0:
			Index2RootPage::indexMerge(log->dbb, indexId, this, NO_TRANSACTION);
			break;
		
		case INDEX_VERSION_1:
			IndexRootPage::indexMerge(log->dbb, indexId, this, NO_TRANSACTION);
			break;
		
		default:
			ASSERT(false);
		}
		
	/***
	IndexKey indexKey;
	
	for (int recordNumber; (recordNumber = nextKey(&indexKey)) != -1;)
		log->dbb->addIndexEntry(indexId, &indexKey, recordNumber, NO_TRANSACTION);
	***/
}

int SRLUpdateIndex::nextKey(IndexKey *indexKey)
{
	if (ptr >= end)
		return -1;
		
	int recordNumber = getInt(&ptr);
	int length = getInt(&ptr);
	
	indexKey->setKey(length, ptr);
	ptr += length;
	
	return recordNumber;	
}
