/* Copyright (C) 2006 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

// RecordVersion.cpp: implementation of the RecordVersion class.
//
//////////////////////////////////////////////////////////////////////

#include "Engine.h"
#include "RecordVersion.h"
#include "Transaction.h"
#include "Table.h"
#include "Connection.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

RecordVersion::RecordVersion(Table *tbl, Format *format, Transaction *trans, Record *oldVersion) :
	Record (tbl, format)
{
	transaction = trans;
	transactionId = transaction->transactionId;
	savePointId = transaction->curSavePointId;
	superceded = false;

	if ((priorVersion = oldVersion))
		{
		priorVersion->addRef();
		recordNumber = oldVersion->recordNumber;
		if (trans == priorVersion->getTransaction())
			oldVersion->setSuperceded (true);
		}
	else
		recordNumber = -1;
}

RecordVersion::~RecordVersion()
{
	Record *prior = priorVersion;
	priorVersion = NULL;

	// Avoid recursion here. May crash from too many levels
	// if the same record is updated too often and quickly.
	
	while (prior)
		prior = prior->releaseNonRecursive();
}

// Release the priorRecord reference without doing it recursively.
// The caller needs to do this for what is returned it is if not null;

Record* RecordVersion::releaseNonRecursive()
{
	Record *prior = NULL;

	if (useCount == 1)
		{
		prior = priorVersion;
		priorVersion = NULL;
		}

	release();

	return prior;
}

Record* RecordVersion::fetchVersion(Transaction * trans)
{
	// Unless the record is at least as old as the transaction, it's not for us

	if (trans->isolationLevel == TRANSACTION_READ_COMMITTED)
		{
		if (!transaction || transaction->state == Committed || transaction == trans)
			return (data.record) ? this : NULL;
		}
	else if (transactionId <= trans->transactionId)
		if (trans->visible (transaction))
			return (data.record) ? this : NULL;

	if (!priorVersion)
		return NULL;
		
	return priorVersion->fetchVersion (trans);
}

Record* RecordVersion::rollback()
{
	if (superceded)
		return NULL;

	return table->rollbackRecord (this);
}

bool RecordVersion::isVersion()
{
	return true;
}

/*
 *	Parent transaction is now fully mature (and about to go
 *	away).  Cleanup any multiversion stuff.
 */

void RecordVersion::commit()
{
	transaction = NULL;
	poke();
}

// Scavenge record versions by the scavenger thread.  Return true if the
// record is a scavenge candidate

bool RecordVersion::scavenge(TransId oldestActive)
{
	if (transaction || (transactionId >= oldestActive))
		{
		table->activeVersions = true;

		if (priorVersion)
			priorVersion->scavenge(oldestActive);

		return false;
		}

	if (priorVersion)
		table->expungeRecordVersions (this);

	return true;
}

// Scavenge record versions replaced within a savepoint.

void RecordVersion::scavenge(TransId targetTransactionId, int oldestActiveSavePointId)
{
	if (!priorVersion)
		return;

	Record *rec = priorVersion;
	Record *ptr = NULL;
	
	// Loop through versions 'till we find somebody rec (or run out of versions looking
	
	for (; rec && rec->getTransactionId() == targetTransactionId && rec->getSavePointId() >= savePointId;
		  rec = rec->getPriorVersion())
		{
		ptr = rec;
#ifdef CHECK_RECORD_ACTIVITY
		rec->active = false;
#endif
		Transaction *trans = rec->getTransaction();
		if (trans)
			trans->removeRecord( (RecordVersion*) rec);
		}
	
	// If we didn't find anyone, there's nothing to do
	
	if (!ptr)
		return;
	
	// There are intermediate versions to collapse.  Splice the unnecessary ones out of the loop
	
	Record *prior = priorVersion;
	prior->addRef();
	setPriorVersion(rec);
	ptr->setPriorVersion(NULL);
	table->garbageCollect(prior, this, transaction, false);
	prior->release();
}

Record* RecordVersion::getPriorVersion()
{
	return priorVersion;
}

void RecordVersion::setSuperceded(bool flag)
{
	superceded = flag;
}

Transaction* RecordVersion::getTransaction()
{
	return transaction;
}

bool RecordVersion::isSuperceded()
{
	return superceded;
}

void RecordVersion::setPriorVersion(Record *oldVersion)
{
	if (oldVersion)
		oldVersion->addRef();

	if (priorVersion)
		priorVersion->release();

	priorVersion = oldVersion;
}

TransId RecordVersion::getTransactionId()
{
	return transactionId;
}

int RecordVersion::getSavePointId()
{
	return savePointId;
}
