/* Copyright (C) 2006 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

// IndexRootPage.cpp: implementation of the IndexRootPage class.
//
//////////////////////////////////////////////////////////////////////

#include <memory.h>
#include <stdio.h>
#include "Engine.h"
#include "IndexRootPage.h"
#include "IndexPage.h"
#include "Dbb.h"
#include "BDB.h"
#include "Section.h"
#include "SectionPage.h"
#include "Bitmap.h"
#include "IndexNode.h"
#include "SQLError.h"
#include "Log.h"
#include "IndexKey.h"
#include "SerialLogControl.h"
#include "Transaction.h"
#include "Index.h"
#include "SRLUpdateIndex.h"

static IndexKey dummyKey;
static int debugSplits;

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

void IndexRootPage::create(Dbb * dbb, TransId transId)
{
	Bdb *bdb = dbb->fakePage (INDEX_ROOT, PAGE_sections, transId);
	IndexRootPage *sections = (IndexRootPage*) bdb->buffer;
	sections->section = -1;
	sections->level = 0;
	sections->sequence = 0;
	bdb->release();
}

int32 IndexRootPage::createIndex(Dbb * dbb, TransId transId)
{
	int sequence = -1;
	Bdb *sectionsBuffer = NULL;
	SectionPage	*sections;
	int skipped = 0;
	
	for (int32 id = dbb->nextIndex;; ++id)
		{
		int n = id / dbb->pagesPerSection;

		if (n != sequence)
			{
			sequence = n;
			
			if (sectionsBuffer)
				sectionsBuffer->release();
				
			sectionsBuffer = Section::getSectionPage(dbb, INDEX_ROOT, n, Exclusive, transId);
			sections = (SectionPage*) sectionsBuffer->buffer;
			sequence = n;
			}

		int slot = id % dbb->pagesPerSection;

		if (sections->pages [slot] == 0)
			{
			if (dbb->indexInUse(id))
				{
				if (!skipped)
					skipped = id;
				}
			else
				{
				sectionsBuffer->mark(transId);
				Bdb *indexBdb = createIndexRoot(dbb, transId);
				sections->pages [slot] = indexBdb->pageNumber;
				IndexPage::logIndexPage(indexBdb, transId);
				indexBdb->release();
				sectionsBuffer->release();
				dbb->nextIndex = (skipped) ? skipped : id + 1;

				return id;
				}
			}
		}
}

bool IndexRootPage::addIndexEntry(Dbb * dbb, int32 indexId, IndexKey *key, int32 recordNumber, TransId transId)
{
	IndexKey searchKey(key);
	searchKey.appendRecordNumber(recordNumber);
	
	if (dbb->debug)
		{
		Sync sync(&Log::syncObject, "IndexRootPage::addIndexEntry");
		sync.lock(Exclusive);
		Btn::printKey ("insertion key: ", key, 0, false);
		Btn::printKey (" appended key: ", &searchKey, 0, false);
		}
		
	for (int n = 0; n < 3; ++n)
		{
		/***
		if (recordNumber == 172303)
			Btn::printKey("page 172303", length, key, 0, false);
		***/

		/* Find insert page and position on page */

		Bdb *bdb = findInsertionLeaf(dbb, indexId, &searchKey, recordNumber, transId);

		if (!bdb)
			return false;

		IndexPage *page;

		for (;;)
			{
			IndexKey priorKey;
			page = (IndexPage*) bdb->buffer;
			Btn *node = page->findNode(key, &priorKey);
			Btn *bucketEnd = (Btn*) ((char*) page + page->length);

			if (node < bucketEnd || page->nextPage == 0)
				break;

			ASSERT (bdb->pageNumber != page->nextPage);
			bdb = dbb->handoffPage (bdb, page->nextPage, PAGE_btree, Exclusive);
			}

		bdb->mark(transId);

		/* If the node fits on page, we're done */

		AddNodeResult result;

		for (;;)
			{
			result = page->addNode(dbb, key, recordNumber);

			if (result != NextPage)
				break;

			bdb = dbb->handoffPage(bdb, page->nextPage, PAGE_btree, Exclusive);
			bdb->mark(transId);
			page = (IndexPage*) bdb->buffer;
			}

		if (result == NodeAdded || result == Duplicate)
			{
			if (dbb->debug)
				page->printPage (bdb, false);
				
			//page->validateInsertion (length, key, recordNumber);
			bdb->release();

			return true;
			}

		/* Node didn't fit.  Split the page and propogate the
		   split upward.  Sooner or laster we'll go back and re-try
		   the original insertion */

		dbb->debug += debugSplits;
		
		if (splitIndexPage (dbb, indexId, bdb, transId, result, key, recordNumber))
			{
			dbb->debug -= debugSplits;
			return true;
			}

		dbb->debug -= debugSplits;

#ifdef _DEBUG
		if (n)
			{
			++dbb->debug;
			Btn::printKey ("Key: ", key, 0, false);
			Btn::printKey ("SearchKey: ", &searchKey, 0, false);
			Bdb *bdb = findInsertionLeaf (dbb, indexId, &searchKey, recordNumber, transId);
			IndexPage *page = (IndexPage*) bdb->buffer;

			while (page->nextPage)
				{
				bdb = dbb->handoffPage (bdb, page->nextPage, PAGE_btree, Exclusive);
				page = (IndexPage*) bdb->buffer;
				page->printPage(bdb, false);
				}

			bdb->release();
			--dbb->debug;
			}
#endif
		}


	ASSERT (false);
	FATAL ("index split failed");

	return false;
}

Bdb* IndexRootPage::findLeaf(Dbb *dbb, int32 indexId, int32 rootPage, IndexKey *indexKey, LockType lockType, TransId transId)
{
	Bdb *bdb = findRoot(dbb, indexId, rootPage, lockType, transId);

	if (!bdb)
		return NULL;

	IndexPage *page = (IndexPage*) bdb->buffer;
	UCHAR *key = indexKey->key;
	int length = indexKey->keyLength;

	if (dbb->debug)
		page->printPage (bdb, false);

	while (page->level > 0)
		{
		IndexNode node (page->findIndex (indexKey));
		int32 pageNumber = node.getNumber();

		if (pageNumber == END_BUCKET)
			pageNumber = page->nextPage;

		if (pageNumber == 0)
			{
			page->printPage (bdb, false);
			node = page->findIndex (indexKey);
			bdb->release();
			throw SQLError (DATABASE_CORRUPTION, "index %d damaged", indexId);
			}

		bdb = dbb->handoffPage(bdb, pageNumber, PAGE_btree,
								(page->level > 1) ? Shared : lockType);
		page = (IndexPage*) bdb->buffer;

		if (dbb->debug)
			page->printPage (bdb, false);
		}

	return bdb;
}
Bdb* IndexRootPage::findInsertionLeaf(Dbb *dbb, int32 indexId, IndexKey *indexKey, int32 recordNumber, TransId transId)
{
	Bdb *bdb = findRoot(dbb, indexId, 0, Shared, transId);

	if (!bdb)
		return NULL;

	IndexPage *page = (IndexPage*) bdb->buffer;
	
	if (dbb->debug)
		page->printPage (bdb, false);

	if (page->level == 0)
		{
		bdb->release();
		
		if ( !(bdb = findRoot(dbb, indexId, 0, Exclusive, transId)) )
			return NULL;
			
		page = (IndexPage*) bdb->buffer;
		
		if (page->level == 0)
			return bdb;
		}
		
	UCHAR *key = indexKey->key;
	int length = indexKey->keyLength;
	
	while (page->level > 0)
		{
		//IndexNode node (page->findIndexNode(indexKey, recordNumber));
		IndexNode node (page->findIndex(indexKey));
		int32 pageNumber = node.getNumber();

		if (pageNumber == END_BUCKET)
			pageNumber = page->nextPage;

		if (pageNumber == 0)
			{
			page->printPage (bdb, false);
			node = page->findIndex (indexKey);
			bdb->release();
			throw SQLError (DATABASE_CORRUPTION, "index %d damaged", indexId);
			}

		bdb = dbb->handoffPage(bdb, pageNumber, PAGE_btree, 
								(page->level > 1) ? Shared : Exclusive);
		page = (IndexPage*) bdb->buffer;

		if (dbb->debug)
			page->printPage (bdb, false);
		}

	return bdb;
}

Bdb* IndexRootPage::findRoot(Dbb *dbb, int32 indexId, int32 rootPage, LockType lockType, TransId transId)
{
	if (rootPage)
		return dbb->fetchPage(rootPage, PAGE_btree, lockType);
		
	if (indexId < 0)
		return NULL;

	Bdb *bdb = Section::getSectionPage (dbb, INDEX_ROOT, indexId / dbb->pagesPerSection, Shared, transId);

	if (!bdb)
		return NULL;

	SectionPage *sections = (SectionPage*) bdb->buffer;
	int32 pageNumber = sections->pages [indexId % dbb->pagesPerSection];

	if (pageNumber == 0)
		{
		bdb->release();
		return NULL;
		}

	return dbb->handoffPage (bdb, pageNumber, PAGE_btree, lockType);
}

void IndexRootPage::scanIndex(Dbb * dbb, int32 indexId, int32 rootPage, IndexKey* lowKey, IndexKey* highKey, bool partial, TransId transId, Bitmap *bitmap)
{
	IndexKey key;
	uint	offset;

	if (dbb->debug)
		{
		Sync sync(&Log::syncObject, "IndexRootPage::scanIndex");
		sync.lock(Exclusive);
		Btn::printKey ("lower: ", lowKey, 0, false);
		Btn::printKey ("upper: ", highKey, 0, false);
		}

	if (!lowKey)
		lowKey = &dummyKey;
		
	/* Find leaf page and position on page */

	Bdb *bdb = findLeaf (dbb, indexId, rootPage, lowKey, Shared, transId);

	if (!bdb)
		throw SQLError (RUNTIME_ERROR, "can't find index %d", indexId);

	IndexPage *page = (IndexPage*) bdb->buffer;
	IndexNode node (page->findNode (lowKey, &key));
	Btn *end = (Btn*) ((UCHAR*) page + page->length);
	UCHAR *endKey = (highKey) ? highKey->key + highKey->keyLength : 0;

	/* If we didn't find it here, try the next page */

	while (node.node >= end)
		{
		if (!page->nextPage)
			{
			bdb->release();
			
			return;
			}
			
		bdb = dbb->handoffPage (bdb, page->nextPage, PAGE_btree, Shared);
		page = (IndexPage*) bdb->buffer;
		node.parseNode (page->findNode (lowKey, &key));
		end = (Btn*) ((UCHAR*) page + page->length);
		}

	if (highKey && node.node < end)
		{
		ASSERT (node.offset + node.length < sizeof(lowKey->key));
		node.expandKey (&key);
		offset = page->computePrefix (node.offset + node.length, key.key, highKey->keyLength, highKey->key);
		}

	/* Scan index setting bits */

	for (;;)
		{
		for (; node.node < end; node.getNext())
			{
			if (highKey)
				{
				if (node.offset <= offset)
					{
					UCHAR *p = highKey->key + node.offset;
					UCHAR *q = node.key;
					
					for (int length = node.length; length; --length)
						{
						if (p >= endKey)
							{
							if (partial)		// this is highly suspect
								break;
								
							bdb->release();
							
							return;
							}
							
						if (*p < *q)
							{
							bdb->release();
							
							return;
							}
							
						if (*p++ > *q++)
							break;
							
						offset = p - highKey->key;
						}
					}
				}
				
			int number = node.getNumber();
			
			if (number < 0)
				break;
				
			bitmap->set (number);
			}

		if (!page->nextPage)
			{
			bdb->release();
			
			return;
			}
			
		bdb = dbb->handoffPage (bdb, page->nextPage, PAGE_btree, Shared);
		page = (IndexPage*) bdb->buffer;
		node.parseNode (page->nodes);
		end = (Btn*) ((UCHAR*) page + page->length);
		offset = 0;
		
		if (dbb->debug)
			page->printPage (bdb, false);
		}
}

bool IndexRootPage::splitIndexPage(Dbb * dbb, int32 indexId, Bdb * bdb, TransId transId,
								   AddNodeResult addResult, IndexKey *indexKey, int recordNumber)
{
	// Start by splitting page (allocating new page)

	IndexPage *page = (IndexPage*) bdb->buffer;
	IndexKey splitKey(indexKey->index);
	bool inserted = false;
	Bdb *splitBdb;
	
	if (addResult == SplitEnd)
		{
		//++dbb->debug;
		splitBdb = page->splitIndexPageEnd (dbb, bdb, transId, indexKey, recordNumber);
		//--dbb->debug;
		inserted = true;
		}
	else
		splitBdb = page->splitIndexPageMiddle (dbb, bdb, &splitKey, transId);

	IndexPage *splitPage = (IndexPage*) splitBdb->buffer;
	IndexNode splitNode(splitPage);
	splitKey.setKey(splitNode.keyLength(), splitNode.key);
	int32 splitRecordNumber = 0;
	
	if (splitPage->level == 0)
		{
		splitRecordNumber = splitNode.getNumber();
		splitKey.appendRecordNumber(splitRecordNumber);
		}

	// If there isn't a parent page, we need to create a new level.  Do so.

	if (!page->parentPage)
		{
		// Allocate and copy a new left-most index page
		
		Bdb *leftBdb = dbb->allocPage(PAGE_btree, transId);
		IndexPage *leftPage = (IndexPage*) leftBdb->buffer;
		memcpy(leftPage, page, page->length);
		splitPage->priorPage = leftBdb->pageNumber;
		
		// Reinitialize the parent page
		
		/***
		Bdb *parentBdb = IndexPage::createNewLevel (dbb, 
				splitPage->level + 1, page->version, bdb->pageNumber, splitBdb->pageNumber,
				&splitKey, transId);
		***/
		
		page->level = splitPage->level + 1;
		page->version = splitPage->version;
		IndexKey dummy(indexKey->index);
		dummy.keyLength = 0;
		page->length = OFFSET (IndexPage*, nodes);
		page->addNode (dbb, &dummy, END_LEVEL);
		page->addNode (dbb, &dummy, leftBdb->pageNumber);
		page->addNode (dbb, &splitKey, splitBdb->pageNumber);
		
		
		leftPage->parentPage = bdb->pageNumber;
		splitPage->parentPage = bdb->pageNumber;
		IndexPage::logIndexPage(bdb, transId);
		IndexPage::logIndexPage(splitBdb, transId);
		IndexPage::logIndexPage(leftBdb, transId);

		/***
		bdb->release();
		dbb->setPrecedence(bdb, splitBdb->pageNumber);
		dbb->setPrecedence(parentBdb, bdb->pageNumber);
		splitBdb->release();
		parentBdb->release();
		setIndexRoot(dbb, indexId, pageNumber, transId);
		***/
		
		dbb->setPrecedence(leftBdb, splitBdb->pageNumber);
		dbb->setPrecedence(bdb, leftBdb->pageNumber);
		splitBdb->release();
		leftBdb->release();
		bdb->release();
		
		return false;
		}

	// We need to propogate the split upward.  Start over from the top
	// to find the insertion point.  Try to insert.  If successful, be happy

	int level = splitPage->level + 1;
	IndexPage::logIndexPage(bdb, transId);
	bdb->release();
	int splitPageNumber = splitBdb->pageNumber;
	splitBdb->release();
		
	for (;;)
		{
		bdb = findRoot (dbb, indexId, 0, Exclusive, transId);
		page = (IndexPage*) bdb->buffer;
		bdb = IndexPage::findLevel (dbb, bdb, level, &splitKey, splitRecordNumber);
		bdb->mark(transId);
		page = (IndexPage*) bdb->buffer;

		// If we can add the node, we're happy

		AddNodeResult result = page->addNode (dbb, &splitKey, splitPageNumber);

		if (result == NodeAdded || result == Duplicate)
			{
			dbb->setPrecedence(bdb, splitPageNumber);
			splitBdb = dbb->fetchPage (splitPageNumber, PAGE_btree, Exclusive);
			splitBdb->mark (transId);
			splitPage = (IndexPage*) splitBdb->buffer;
			splitPage->parentPage = bdb->pageNumber;
			bdb->release();
			splitBdb->release();

			return false;
			}

		// That page needs to be split.  Recurse

		if (splitIndexPage (dbb, indexId, bdb, transId, 
							result, &splitKey, splitPageNumber))
			return true;
		}
}

bool IndexRootPage::deleteIndexEntry(Dbb * dbb, int32 indexId, IndexKey *key, int32 recordNumber, TransId transId)
{
	IndexPage *page;
	IndexKey searchKey(key);
	searchKey.appendRecordNumber(recordNumber);

	if (dbb->debug)
		{
		Btn::printKey ("deletion key: ", key, 0, false);
		Btn::printKey (" search key: ", &searchKey, 0, false);
		}
		
	// Try to delete node.  If necessary, chase to next page.

	for (Bdb *bdb = findInsertionLeaf (dbb, indexId, &searchKey, Exclusive, transId); bdb;
		 bdb = dbb->handoffPage (bdb, page->nextPage, PAGE_btree, Exclusive))
		{
		page = (IndexPage*) bdb->buffer;
		bdb->mark(transId);
		int result = page->deleteNode (dbb, key, recordNumber);

		if (result || page->nextPage == 0)
			{
			bdb->release();

			return result > 0;
			}
		}	

	return false;
}

void IndexRootPage::deleteIndex(Dbb *dbb, int32 indexId, TransId transId)
{
	Bdb *bdb = Section::getSectionPage (dbb, INDEX_ROOT, indexId / dbb->pagesPerSection, Exclusive, transId);

	if (!bdb)
		return;

	bdb->mark (transId);
	SectionPage *sections = (SectionPage*) bdb->buffer;
	int32 nextLevel = sections->pages [indexId % dbb->pagesPerSection];
	sections->pages [indexId % dbb->pagesPerSection] = 0;
	dbb->nextIndex = MIN(dbb->nextIndex, indexId);
	bdb->release();

	while (nextLevel)
		{
		bdb = dbb->fetchPage (nextLevel, PAGE_any, Exclusive);
		IndexPage *page = (IndexPage*) bdb->buffer;
		
		if (page->pageType != PAGE_btree)
			{
			Log::logBreak ("Drop index %d: bad level page %d\n", indexId, bdb->pageNumber);
			bdb->release();
			
			break;
			}
			
		IndexNode node (page->nodes);
		nextLevel = (page->level) ? node.getNumber() : 0;
		
		for (;;)
			{
			int32 nextPage = page->nextPage;
			dbb->freePage (bdb, transId);
			
			if (!nextPage)
				break;
				
			bdb = dbb->fetchPage (nextPage, PAGE_any, Exclusive);
			page = (IndexPage*) bdb->buffer;
			
			if (page->pageType != PAGE_btree)
				{
				Log::logBreak ("Drop index %d: bad index page %d\n", indexId, bdb->pageNumber);
				bdb->release();
				break;
				}
			}
		}
}


void IndexRootPage::debugBucket(Dbb *dbb, int indexId, int recordNumber, TransId transactionId)
{
	Bdb *bdb = findRoot (dbb, indexId, 0, Exclusive, transactionId);
	IndexPage *page;
	IndexNode node;
	int debug = 1;

	// Find leaf

	for (;;)
		{
		page = (IndexPage*) bdb->buffer;
		
		if (debug)
			page->printPage (bdb, false);
			
		node.parseNode (page->nodes);
		
		if (page->level == 0)
			break;
			
		bdb = dbb->handoffPage (bdb, node.getNumber(), PAGE_btree, Exclusive);
		}

	// Scan index

	Btn *end = (Btn*) ((UCHAR*) page + page->length);
	int pages = 0;
	int nodes = 0;

	/* If we didn't find it here, try the next page */

	for (;;)
		{
		++pages;
		
		for (; node.node < end; node.getNext())
			{
			++nodes;
			int number = node.getNumber();
			
			if (number < 0)
				break;
				
			if (number == recordNumber)
				page->printPage (bdb, false);
			}
			
		if (!page->nextPage)
			break;
			
		bdb = dbb->handoffPage (bdb, page->nextPage, PAGE_btree, Exclusive);
		page = (IndexPage*) bdb->buffer;
		
		if (debug)
			page->printPage (bdb, false);
			
		node.parseNode (page->nodes);
		end = (Btn*) ((UCHAR*) page + page->length);
		}

	bdb->release();
}

void IndexRootPage::redoIndexPage(Dbb* dbb, int32 pageNumber, int32 parentPage, int level, int32 priorPage, int32 nextPage, int length, const UCHAR *data)
{
	//Log::debug("redoIndexPage %d -> %d -> %d level %d, parent %d)\n", priorPage, pageNumber, nextPage, level, parentPage);
	Bdb *bdb = dbb->fakePage(pageNumber, PAGE_any, 0);
	IndexPage *indexPage = (IndexPage*) bdb->buffer;

	//  Try to read page.  If it looks OK, keep it.  Otherwise, rebuild it

	if (!dbb->trialRead(bdb) ||
		 indexPage->pageType != PAGE_btree ||
		 indexPage->parentPage != parentPage ||
		 indexPage->level == level)
		{
		memset(indexPage, 0, dbb->pageSize);
		indexPage->pageType = PAGE_btree;
		}

	indexPage->level = level;
	indexPage->parentPage = parentPage;
	indexPage->nextPage = nextPage;
	indexPage->priorPage = priorPage;
	indexPage->length = length + (int32) OFFSET (IndexPage*, nodes);
	memcpy(indexPage->nodes, data, length);
	
	// If we have a parent page, propogate the first node upward

	if (parentPage)
		{
		IndexNode node(indexPage);
		int number = node.getNumber();

		if (number >= 0)
			{
			IndexKey indexKey(node.keyLength(), node.key);
			Bdb *parentBdb = dbb->fetchPage(parentPage, PAGE_btree, Exclusive);
			IndexPage *parent = (IndexPage*) parentBdb->buffer;
		
			if (level == 0)
				indexKey.appendRecordNumber(number);
				
			parentBdb->mark(NO_TRANSACTION);
			AddNodeResult result = parent->addNode(dbb, &indexKey, pageNumber);
			parentBdb->release();
			}
		}

	bdb->release();
	
	if (nextPage)
		{
		bdb = dbb->trialFetch(nextPage, PAGE_any, Exclusive);
				
		if (bdb)
			{
			indexPage = (IndexPage*) bdb->buffer;
			
			if (indexPage->pageType == PAGE_btree &&
				indexPage->parentPage == parentPage &&
				indexPage->level == level)
				{
				if (indexPage->priorPage != pageNumber)
					{
					bdb->mark(NO_TRANSACTION);
					indexPage->priorPage = pageNumber;
					}
				}
			
			bdb->release();
			}
		}
		
	if (priorPage)
		{
		bdb = dbb->trialFetch(priorPage, PAGE_any, Exclusive);
		
		if (bdb)
			{
			indexPage = (IndexPage*) bdb->buffer;
			
			if (indexPage->pageType == PAGE_btree &&
				indexPage->parentPage == parentPage &&
				indexPage->level == level)
				{
				if (indexPage->nextPage != pageNumber)
					{
					bdb->mark(NO_TRANSACTION);
					indexPage->nextPage = pageNumber;
					}
				}
				
			bdb->release();
			}
		}
}

void IndexRootPage::setIndexRoot(Dbb* dbb, int indexId, int32 pageNumber, TransId transId)
{
	int sequence = indexId / dbb->pagesPerSection;
	int slot = indexId % dbb->pagesPerSection;
	Bdb *bdb = Section::getSectionPage (dbb, INDEX_ROOT, sequence, Exclusive, transId);
	dbb->setPrecedence(bdb, pageNumber);
	bdb->mark(transId);
	SectionPage *sections = (SectionPage*) bdb->buffer;
	sections->pages[slot] = pageNumber;
	
	if (!dbb->recovering)
		dbb->serialLog->logControl->sectionPage.append(bdb->pageNumber, pageNumber, slot, INDEX_ROOT, sequence, 0);

	bdb->release();
}

void IndexRootPage::redoIndexDelete(Dbb* dbb, int indexId)
{
	setIndexRoot(dbb, indexId, 0, NO_TRANSACTION);
}

void IndexRootPage::indexMerge(Dbb *dbb, int indexId, SRLUpdateIndex *logRecord, TransId transId)
{
	IndexKey key;
	int recordNumber = logRecord->nextKey(&key);
	int duplicates = 0;
	int insertions = 0;
	int punts = 0;
	int rollovers = 0;
	
	for (; recordNumber != -1;)
		{
		// Position to insert first key (clone of addIndexEntry)
		
		IndexKey searchKey(key);
		searchKey.appendRecordNumber(recordNumber);
			
		if (dbb->debug)
			{
			Sync sync(&Log::syncObject, "IndexRootPage::indexMerge");
			sync.lock(Exclusive);
			Btn::printKey ("insertion key: ", &key, 0, false);
			Btn::printKey (" appended key: ", &searchKey, 0, false);
			}
		
		Bdb *bdb = findInsertionLeaf(dbb, indexId, &searchKey, recordNumber, transId);
		ASSERT(bdb);
		IndexPage *page = (IndexPage*) bdb->buffer;
		Btn *bucketEnd = (Btn*) ((char*) page + page->length);
		IndexKey priorKey;
		IndexNode node(page->findInsertionPoint(&key, recordNumber, &priorKey));
		IndexKey nextKey;
		nextKey.setKey(0, node.offset, priorKey.key);
		node.expandKey(&nextKey);
		int number = node.getNumber();
		
		// If we need to go to the next page, do it now
		
		while (number == END_BUCKET && nextKey.compare(&key) > 0)
			{
			if (!page->nextPage)
				{
				bdb->release();
				bdb = NULL;
				++punts;
				addIndexEntry(dbb, indexId, &key, recordNumber, transId);
				
				if ( (recordNumber = logRecord->nextKey(&key)) == -1)
					//return;
					goto exit;
				
				break;
				}
				
			nextKey.compare(&key);
			ASSERT (bdb->pageNumber != page->nextPage);
			bdb = dbb->handoffPage (bdb, page->nextPage, PAGE_btree, Exclusive);
			page = (IndexPage*) bdb->buffer;
			node.parseNode(page->findInsertionPoint(&key, recordNumber, &priorKey));
			nextKey.setKey(0, node.offset, priorKey.key);
			node.expandKey(&nextKey);
			number = node.getNumber();
			++rollovers;
			}
		
		if (!bdb)
			continue;

		bdb->mark(transId);

		for (;;)
			{
			if (number == recordNumber && nextKey.compare(&key) == 0)
				++duplicates;
			else
				{
				int offset1 = IndexPage::computePrefix (&priorKey, &key);
				int length1 = key.keyLength - offset1;
				int delta = IndexNode::nodeLength(offset1, length1, recordNumber);
				int32 nextNumber;
				int offset2;
				
				if ((UCHAR*) node.node == (UCHAR*) page + page->length)
					offset2 = -1;
				else
					{
					node.expandKey(&nextKey);
					offset2 = IndexPage::computePrefix(key.keyLength, key.key, node.offset + node.length, nextKey.key);
					int deltaOffset = offset2 - node.offset;
					
					if (node.length >= 128 && (node.length - deltaOffset) < 128)
						--delta;

					if (node.offset < 128 && (node.offset + deltaOffset) >= 128)
						++delta;
						
					delta -= deltaOffset;
					nextNumber = node.getNumber();
					}

				// If node doesn't fit, punt and let someone else do it
				
				if (page->length + delta > dbb->pageSize)
					{
					bdb->release();
					bdb = NULL;
					++punts;
					addIndexEntry(dbb, indexId, &key, recordNumber, transId);
					
					if ( (recordNumber = logRecord->nextKey(&key)) == -1)
						//return;
						goto exit;
					
					break;
					}
				
				// Add insert node into page
				
				if (offset2 >= 0)
					{
					UCHAR *tail = (UCHAR*) node.nextNode;
					int tailLength = (UCHAR*) page + page->length - tail;
					ASSERT (tailLength >= 0);
					
					if (tailLength > 0)
						memmove (tail + delta, tail, tailLength);
					}

				// Insert new node

				++insertions;
				IndexNode newNode;
				newNode.insert(node.node, offset1, length1, key.key, recordNumber);

				// If necessary, rebuild next node

				if (offset2 >= 0)
					newNode.insert(newNode.nextNode, offset2, nextKey.keyLength - offset2, nextKey.key, nextNumber);

				page->length += delta;
				//page->validate(NULL);

				/***
				if (dbb->debug)
					page->printPage(bdb, false);
				***/
				}
				
			int priorRecordNumber = recordNumber;
			priorKey.setKey(&key);
			
			// Get next key
			
			if ( (recordNumber = logRecord->nextKey(&key)) == -1)
				break;

			// If the key is out of order, somebody screwed up.  Punt out of here
			
			if (key.compare(&priorKey) > 0)
				//break;
				ASSERT(false);
			
			// Find the next insertion point, compute the next key, etc.
			
			bucketEnd = (Btn*) ((char*) page + page->length);
			node.parseNode(IndexPage::findInsertionPoint(0, &key, recordNumber, &priorKey, node.node, bucketEnd));
			nextKey.setKey(0, node.offset, priorKey.key);
			node.expandKey(&nextKey);
			number = node.getNumber();
			
			if (number != END_LEVEL && nextKey.compare(&key) > 0)
				break;
			}
		
		if (bdb)	
			bdb->release();
		}	

	exit:;
	
	//Log::debug("indexMerge: index %d, %d insertions, %d punts, %d duplicates, %d rollovers\n",  indexId, insertions, punts, duplicates, rollovers);
}

void IndexRootPage::redoCreateIndex(Dbb* dbb, int indexId)
{
	Bdb *bdb = Section::getSectionPage (dbb, INDEX_ROOT, indexId / dbb->pagesPerSection, Exclusive, NO_TRANSACTION);
	ASSERT(bdb);
	SectionPage *sections = (SectionPage*) bdb->buffer;
	int slot = indexId % dbb->pagesPerSection;

	if (!sections->pages [slot])
		{
		bdb->mark(NO_TRANSACTION);
		Bdb *pageBdb = createIndexRoot(dbb, NO_TRANSACTION);
		sections->pages [slot] = pageBdb->pageNumber;
		pageBdb->release();
		}

	bdb->release();
}

Bdb* IndexRootPage::createIndexRoot(Dbb* dbb, TransId transId)
{
	Bdb *bdb = dbb->allocPage (PAGE_btree, transId);
	IndexPage *btree = (IndexPage*) bdb->buffer;
	btree->level = 0;
	btree->version = INDEX_CURRENT_VERSION;
	btree->length = OFFSET (IndexPage*, nodes);
	IndexKey key;
	key.keyLength = 0;
	btree->addNode (dbb, &key, END_LEVEL);

	return bdb;
}

void IndexRootPage::analyzeIndex(Dbb* dbb, int indexId, IndexAnalysis *indexAnalysis)
{
	Bdb *bdb = findRoot(dbb, indexId, 0, Shared, NO_TRANSACTION);

	if (!bdb)
		return;

	IndexPage *page = (IndexPage*) bdb->buffer;
	indexAnalysis->levels = page->level + 1;
	int32 nextLevel;
	bool first = true;
	
	for (;;)
		{
		if (first)
			{
			IndexNode node(page);
			nextLevel = node.getNumber();
			}

		if (page->level == 0)
			{
			++indexAnalysis->leafPages;
			indexAnalysis->leafSpaceUsed += page->length;
			}
		else
			++indexAnalysis->upperLevelPages;
	
		if (page->nextPage)
			bdb = dbb->handoffPage(bdb, page->nextPage, PAGE_btree, Shared);
		else 
			{
			bdb->release();

			if (page->level == 0)
				break;
				
			bdb = dbb->fetchPage(nextLevel, PAGE_btree, Shared);
			first = true;
			}
		
		page = (IndexPage*) bdb->buffer;
		}
}

int32 IndexRootPage::getIndexRoot(Dbb* dbb, int indexId)
{
	Bdb *bdb = Section::getSectionPage (dbb, INDEX_ROOT, indexId / dbb->pagesPerSection, Shared, NO_TRANSACTION);

	if (!bdb)
		return 0;

	SectionPage *sections = (SectionPage*) bdb->buffer;
	int32 pageNumber = sections->pages [indexId % dbb->pagesPerSection];
	bdb->release();
	
	return pageNumber;
}
