/* Copyright (C) 2006 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

// Database.cpp: implementation of the Database class.
//
//////////////////////////////////////////////////////////////////////

#include <string.h>
#include <stdio.h>
#include <time.h>
#include "Engine.h"
#include "Database.h"
#include "Dbb.h"
#include "PreparedStatement.h"
#include "CompiledStatement.h"
#include "Table.h"
#include "UnTable.h"
#include "Index.h"
#include "Connection.h"
#include "Transaction.h"
#include "Stream.h"
#include "SQLError.h"
#include "ResultSet.h"
#include "Search.h"
#include "ImageManager.h"
#include "SequenceManager.h"
#include "Inversion.h"
#include "Sync.h"
#include "Threads.h"
#include "Thread.h"
#include "Scheduler.h"
#include "Scavenger.h"
#include "RoleModel.h"
#include "User.h"
#include "Registry.h"
#include "SymbolManager.h"
#include "FilterSetManager.h"
#include "Log.h"
#include "DateTime.h"
#include "Interlock.h"
#include "Cache.h"
#include "NetfraVersion.h"
#include "OpSys.h"
#include "SearchWords.h"
#include "Repository.h"
#include "RepositoryManager.h"
#include "Schema.h"
#include "Configuration.h"
#include "SerialLog.h"
#include "SerialLogControl.h"
#include "PageWriter.h"
#include "Trigger.h"
#include "TransactionManager.h"

#ifndef STORAGE_ENGINE
#include "Applications.h"
#include "Application.h"
#include "JavaVM.h"
#include "Java.h"
#include "Template.h"
#include "Templates.h"
#include "TemplateContext.h"
#include "TemplateManager.h"
#include "Session.h"
#include "SessionManager.h"
#include "DataResourceLocator.h"
#else
extern unsigned int falcon_page_size;
#endif

#ifdef LICENSE
#include "LicenseManager.h"
#include "LicenseProduct.h"
#endif

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#endif

#define TRANSACTION_ID(transaction)		((transaction) ? transaction->transactionId : 0)
#define HASH(address,size)				(((UIPTR) address >> 2) % size)
#define EXPIRATION_DAYS					180

#define STATEMENT_RETIREMENT_AGE	60
#define RECORD_RETIREMENT_AGE		60

static const char *createTables = 
	"create table Tables (\
			tableName varchar (128) not null,\
			schema varchar (128) not null,\
			type varchar (16),\
			dataSection int,\
			blobSection int,\
			tableId int,\
			currentVersion int,\
			remarks text,\
			viewDefinition clob,\
			primary key (tableName, schema));";

static const char *createOds2Fields = 
	"create table Fields (\
			field varchar (128) not null,\
			tableName varchar (128) not null,\
			schema varchar (128) not null,"
			//"domainName varchar (128),"
			"collationsequence varchar (128),"
			"fieldId int,\
			dataType int,\
			length int,\
			scale int,\
			flags int,\
			remarks text,\
			primary key (schema, tableName, field));";

static const char *createOds3Fields = 
	"upgrade table Fields ("
		"field varchar (128) not null,"
		"tableName varchar (128) not null,"
		"schema varchar (128) not null,"
		"domainName varchar (128),"
		"repositoryName varchar (128),"
		"collationsequence varchar (128),"
		"fieldId int,"
		"dataType int,"
		"length int,"
		"scale int,"
		"precision int,"
		"flags int,"
		"remarks text,"
		"primary key (schema, tableName, field));";

static const char *createFieldDomainName = 
	"upgrade index FieldDomainName on Fields (domainName);";

static const char *createFieldCollationSequenceName = 
	"create index FieldCollationSequenceName on Fields (collationsequence);";

static const char *createFormats = 
	"create table Formats (\
			tableId int not null,\
			fieldId int not null,\
			version int not null,\
			dataType int,\
			offset int,\
			length int,\
			scale int,\
			maxId int,\
			primary key (tableId, version, fieldId));";

static const char *createIndexes = 
	"create table Indexes (\
			indexName varchar (128) not null,\
			tableName varchar (128) not null,\
			schema varchar (128) not null,\
			indexType int,\
			fieldCount int,\
			indexId int,\
			primary key (schema, indexName));";

static const char *createOd3IndexIndex =
	"create index index_table on indexes (schema, tableName)";

static const char *createIndexFields = 
	"create table IndexFields (\
			indexName varchar (128) not null,\
			schema varchar (128) not null,\
			tableName varchar (128) not null,\
			field varchar (128) not null,\
			position int,\
			primary key (schema, indexName, field));";

static const char *createOd3IndexFieldsIndex =
	"create index indexfields_table on IndexFields (schema, tableName)";

static const char *createOds3IndexFields = 
	"upgrade table IndexFields (\
			indexName varchar (128) not null,\
			schema varchar (128) not null,\
			tableName varchar (128) not null,\
			field varchar (128) not null,\
			position int,\
			partial int,\
			primary key (schema, indexName, field));";

static const char *createForeignKeys = 
	"create table ForeignKeys (\
			primaryTableId int not null,\
			primaryFieldId int not null,\
			foreignTableId int not null,\
			foreignFieldId int not null,\
			numberKeys int,\
			position int,\
			updateRule smallint,\
			deleteRule smallint,\
			deferrability smallint,\
			primary key (foreignTableId, primaryTableId, foreignFieldId));";

static const char *createForeignKeysIndex = 
	"create index ForeignKeysIndex on ForeignKeys\
			(primaryTableId);";

static const char *createDomains = 
	"create table Domains ("
			"domainName varchar (128) not null,"
			"schema varchar (128) not null,"
		    "dataType int,"
		    "length int,"
		    "scale int,"
		    "remarks text,"
			"primary key (domainName, schema));";

static const char *createOds3Domains = 
	"create table Domains ("
			"domainName varchar (128) not null,"
			"schema varchar (128) not null,"
		    "dataType int,"
		    "length int,"
		    "scale int,"
		    "remarks text,"
			"primary key (domainName, schema));";

static const char *createView_tables = 
	"upgrade table view_tables ("
		"viewName varchar (128) not null,"
		"viewSchema varchar (128) not null,"
		"sequence int not null,"
		"tableName varchar (128) not null,"
		"schema varchar (128) not null,"
		"primary key (viewSchema,viewName,sequence))";

static const char *createRepositories = 
	"upgrade table repositories ("
		"repositoryName varchar (128) not null,"
		"schema varchar (128) not null,"
		"sequenceName varchar (128),"
		"filename varchar (128),"
		"rollovers varchar (128),"
		"currentVolume int,"
		"primary key (repositoryName, schema))";

static const char *createSchemas =
	"upgrade table schemas ("
		"schema varchar (128) not null primary key,"
		"sequence_interval int,"
		"system_id int)";

static const char *ods2Statements [] =
    {
	createTables,
	createOds2Fields,
	createFieldCollationSequenceName,
	createFormats,
	createIndexes,
	createIndexFields,
	createForeignKeys,
	createForeignKeysIndex,
	createDomains,

	"grant select on tables to public",
	"grant select on fields to public",
	"grant select on formats to public",
	"grant select on indexes to public",
	"grant select on indexFields to public",
	"grant select on foreignKeys to public",
	"grant select on Domains to public",

	NULL
	};
	
static const char *ods2UpgradeStatements [] = {
	createView_tables,
	createRepositories,
	createSchemas,
	"grant select on repositories to public",
	"grant select on schemas to public",

	NULL
	};

static const char *ods3Upgrade  [] = {
	createOds3Fields,
	createFieldDomainName,
	createOds3IndexFields,
	createOd3IndexIndex,
	createOd3IndexFieldsIndex,
	NULL
	};

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

Database::Database(const char *dbName, Configuration *config, Threads *parent)
{
	panicShutdown = false;
	name = dbName;
	configuration = config;
	useCount = 1;
	nextTableId = 0;
	compiledStatements = NULL;
	memset (tables, 0, sizeof (tables));
	memset (tablesModId, 0, sizeof (tablesModId));
	memset (unTables, 0, sizeof (unTables));
	memset (schemas, 0, sizeof (schemas));
	currentAgeGroup = 1;
	overflowSize = 0;
	memset ((void*) ageGroupSizes, 0, sizeof (ageGroupSizes));
	tableList = NULL;
	recordMemoryLower = configuration->recordMemoryLower;
	recordMemoryUpper = configuration->recordMemoryUpper;
	lastRecordMemory = 0;
	utf8 = false;
	stepNumber = 0;
	shuttingDown = false;

	noSchedule = 0;
	//noSchedule = 1;

	fieldExtensions = false;
	cache = NULL;
	dbb = new Dbb(this);
	numberQueries = 0;
	numberRecords = 0;
	numberTemplateEvals = 0;
	numberTemplateExpands = 0;
	threads = new Threads (parent);

	symbolManager = NULL;
	templateManager = NULL;
	imageManager = NULL;
	roleModel = NULL;
	licenseManager = NULL;
	transactionManager = NULL;
	applications = NULL;
	systemConnection = NULL;
	searchWords = NULL;
	java = NULL;
	scheduler = NULL;
	internalScheduler = NULL;
	scavenger = NULL;
	garbageCollector = NULL;
	sequenceManager = NULL;
	repositoryManager = NULL;
	sessionManager = NULL;
	filterSetManager = NULL;
	//firstTransaction = lastTransaction = NULL;
	timestamp = time (NULL);
	tickerThread = NULL;
	serialLog = NULL;
	pageWriter = NULL;
	zombieTables = NULL;
}


void Database::start()
{
	symbolManager = new SymbolManager;

#ifdef LICENSE
	licenseManager = new LicenseManager (this);
#endif

#ifndef STORAGE_ENGINE
	templateManager = new TemplateManager (this);
	java = new Java (this, applications, configuration->classpath);
	sessionManager = new SessionManager (this);
	applications = new Applications (this);
#endif

	imageManager = new ImageManager (this);
	roleModel = new RoleModel (this);
	systemConnection = new Connection (this);
	dbb->serialLog = serialLog = new SerialLog(this, configuration->checkpointSchedule);
	pageWriter = new PageWriter(this);
	searchWords = new SearchWords (this);
	systemConnection->setLicenseNotRequired (true);
	systemConnection->pushSchemaName ("SYSTEM");
	addSystemTables();
	scheduler = new Scheduler (this);
	internalScheduler = new Scheduler (this);
	scavenger = new Scavenger (this, scvRecords, configuration->scavengeSchedule);
	garbageCollector = new Scavenger (this, scvJava, configuration->gcSchedule);
	sequenceManager = new SequenceManager (this);
	repositoryManager = new RepositoryManager (this);
	transactionManager = new TransactionManager(this);
	internalScheduler->addEvent (repositoryManager);
	filterSetManager = new FilterSetManager (this);
	//firstTransaction = lastTransaction = NULL;
	timestamp = time (NULL);
	tickerThread = threads->start ("Database::Database", &Database::ticker, this);
	internalScheduler->addEvent (scavenger);
	internalScheduler->addEvent (garbageCollector);
	internalScheduler->addEvent (serialLog);
	pageWriter->start();
	cache->setPageWriter(pageWriter);
}

Database::~Database()
{
	for (Table *table; (table = tableList);)
		{
		tableList = table->next;
		delete table;
		}

	int n;

	for (n = 0; n < TABLE_HASH_SIZE; ++n)
		for (UnTable *untable; (untable = unTables [n]);)
			{
			unTables [n] = untable->collision;
			delete untable;
			}
					
	for (n = 0; n < TABLE_HASH_SIZE; ++n)
		for (Schema *schema; (schema = schemas [n]);)
			{
			schemas [n] = schema->collision;
			delete schema;
			}
					
	if (systemConnection)
		{
		systemConnection->rollback();
		systemConnection->close();
		}

	for (CompiledStatement *statement = compiledStatements; statement;)
		{
		CompiledStatement *object = statement;
		statement = statement->next;
		if (object->useCount > 1)
			Log::debug ("~Database: '%s' in use\n", (const char*) object->sqlString);
		object->release();
		}

	delete dbb;

	if (scavenger)
		scavenger->release();

	if (garbageCollector)
		garbageCollector->release();

	if (threads)
		{
		threads->shutdownAll();
		threads->waitForAll();
		threads->release();
		}

#ifndef STORAGE_ENGINE
	delete java;
	delete templateManager;
	delete sessionManager;
	delete applications;
#endif

#ifdef LICENSE
	delete licenseManager;
#endif

	if (scheduler)
		scheduler->release();

	if (internalScheduler)
		internalScheduler->release();


	delete imageManager;
	delete sequenceManager;
	delete searchWords;
	delete serialLog;
	delete pageWriter;
	delete cache;
	delete roleModel;
	delete symbolManager;
	delete filterSetManager;
	delete repositoryManager;
	delete transactionManager;
}

void Database::createDatabase(const char * filename)
{
	const char *p = strrchr(filename, '.');
	JString logRoot = (p) ? JString(filename, p - filename) : name;
#ifdef STORAGE_ENGINE
	int page_size = falcon_page_size;
#else
	int page_size = PAGE_SIZE;
#endif

	cache = dbb->create (filename, page_size, configuration->pageCacheSize, HdrDatabaseFile, 0, logRoot);
	
	try
		{
		start();
		serialLog->open(logRoot, true);
		serialLog->start();
		sequence = dbb->sequence;
		odsVersion = dbb->odsVersion;
		dbb->createInversion(0);
		Table *table;
		Transaction *sysTrans = getSystemTransaction();
		
		for (table = tableList; table; table = table->next)
			table->create ("SYSTEM TABLE", sysTrans);

		for (table = tableList; table; table = table->next)
			table->save();

		roleModel->createTables();
		sequenceManager->initialize();
		Trigger::initialize (this);
		systemConnection->commit();

#ifndef STORAGE_ENGINE
		java->initialize();
		checkManifest();
		templateManager->getTemplates ("base");

#ifdef LICENSE
		licenseManager->initialize();
		licenseCheck();
#endif

		startSessionManager();
#endif

		imageManager->getImages ("base", NULL);
		filterSetManager->initialize();
		scheduler->start();
		internalScheduler->start();
		upgradeSystemTables();
		//dbb->flush();
		commitSystemTransaction();
		serialLog->checkpoint(true);
		}
	catch (...)
		{
		dbb->closeFile();
		dbb->deleteFile();
		
		throw;
		}
}

void Database::openDatabase(const char * filename)
{
	cache = dbb->open (filename, configuration->pageCacheSize, 0);
	start();

	if (dbb->logRoot.IsEmpty())
		{
		const char *p = strrchr(filename, '.');
		dbb->logRoot = (p) ? JString(filename, p - filename) : name;
		}
		
	if (serialLog)
		{
		if (COMBINED_VERSION(dbb->odsVersion, dbb->odsMinorVersion) >= VERSION_SERIAL_LOG)
			{
			if (dbb->logLength)
				serialLog->copyClone(dbb->logRoot, dbb->logOffset, dbb->logLength);

			try
				{
				serialLog->open(dbb->logRoot, false);
				}
			catch (SQLException&)
				{
				const char *p = strrchr(filename, '.');
				JString logRoot = (p) ? JString(filename, p - filename) : name;
				bool failed = true;
				
				try
					{
					serialLog->open(logRoot, false);
					failed = false;
					}
				catch (...)
					{
					}
				
				if (failed)
					throw;
				}
			
			serialLog->recover();
			serialLog->start();
			}
		else
			{
			dbb->enableSerialLog();
			serialLog->open(dbb->logRoot, true);
			serialLog->start();
			}
		}

	sequence = dbb->sequence;
	odsVersion = dbb->odsVersion;
	utf8 = dbb->utf8;
	int indexId = 0;
	int sectionId = 0;
	
	for (Table *table = tableList; table; table = table->next)
		{
		table->setDataSection (sectionId++);
		table->setBlobSection (sectionId++);
		
		FOR_INDEXES (index, table)
			index->setIndex (indexId++);
		END_FOR;
		}

	PreparedStatement *statement = prepareStatement ("select tableid from tables");
	ResultSet *resultSet = statement->executeQuery();

	while (resultSet->next())
		{
		int n = resultSet->getInt (1);
		
		if (n >= nextTableId)
			nextTableId = n + 1;
		}

	resultSet->close();
	statement->close();
	upgradeSystemTables();
	Trigger::initialize (this);

	//validate(0);

#ifndef STORAGE_ENGINE

#ifdef LICENSE
	licenseManager->initialize();
	licenseCheck();
#endif

	java->initialize();
	checkManifest();
	startSessionManager();
#endif

	sequenceManager->initialize();
	filterSetManager->initialize();
	searchWords->initialize();
	roleModel->initialize();

	if (serialLog)
		serialLog->recoverLimboTransactions();
		
#ifndef STORAGE_ENGINE
	getApplication ("base");
#endif

	internalScheduler->start();

	if (configuration->schedulerEnabled)
		scheduler->start();
}

#ifndef STORAGE_ENGINE
void Database::startSessionManager()
{
	try
		{
		java->run (systemConnection);
		sessionManager->start (systemConnection);
		}
	catch (SQLException &exception)
		{
		Log::debug ("Exception during Java initialization: %s\n", (const char*) exception.getText());
		const char *stackTrace = exception.getTrace();
		if (stackTrace && stackTrace [0])
			Log::debug ("Stack trace:\n%s\n", stackTrace);
		}
}

void Database::genHTML(ResultSet * resultSet, const char *series, const char *type, TemplateContext *context, Stream *stream, JavaCallback *callback)
{
	Templates *tmpls = templateManager->getTemplates (series);
	tmpls->genHTML (type, resultSet, context, 0, stream, callback);
}

void Database::genHTML(ResultSet *resultSet, const WCString *series, const WCString *type, TemplateContext *context, Stream *stream, JavaCallback *callback)
{
	Templates *tmpls = templateManager->getTemplates (series);
	tmpls->genHTML (symbolManager->getSymbol(type), resultSet, context, 0, stream, callback);
}


JString Database::expandHTML(ResultSet *resultSet, const WCString *applicationName, const char *source, TemplateContext *context, JavaCallback *callback)
{
	Templates *tmpls = templateManager->getTemplates (applicationName);

	return tmpls->expandHTML (source, resultSet, context, callback);
}


const char* Database::fetchTemplate(JString applicationName, JString templateName, TemplateContext *context)
{
	Templates *templates = templateManager->getTemplates (applicationName);
	Template *pTemplate = templates->findTemplate (context, templateName);

	if (!pTemplate)
		return NULL;

	return pTemplate->body;
}


void Database::zapLinkages()
{
	for (Table *table = tableList; table; table = table->next)
		table->zapLinkages();

	sessionManager->zapLinkages();
}

void Database::checkManifest()
{
	if (!java->findManifest ("netfrastructure/model/Application"))
		throw SQLError (LICENSE_ERROR, "missing manifest for base application");
}

int Database::attachDebugger()
{
	return java->attachDebugger();
}

JString Database::debugRequest(const char *request)
{
	return java->debugRequest(request);
}

void Database::detachDebugger()
{
	java->detachDebugger();
}

Application* Database::getApplication(const char * applicationName)
{
	return applications->getApplication (applicationName);
}

int Connection::initiateService(const char *applicationName, const char *service)
{
	Application *application = database->getApplication (applicationName);

	if (!application)
		throw SQLEXCEPTION (RUNTIME_ERROR, "application '%s' is unknown",
								(const char*) applicationName);

	application->pushNameSpace (this);
	Session *session = NULL;
	int port;
	
	try
		{
		session = database->sessionManager->createSession (application);
		port = database->sessionManager->initiateService (this, session, service);
		}
	catch (...)
		{
		if (session)
			session->release();
		throw;
		}

	session->release();

	return port;
}

PreparedStatement* Connection::prepareDrl(const char * drl)
{
	DataResourceLocator locator;

	return locator.prepareStatement (this, drl);
}

#endif	// STORAGE_ENGINE

void Database::serverOperation(int op, Parameters *parameters)
{
	switch (op)
		{
#ifndef STORAGE_ENGINE
		case opTraceAll:
			java->traceAll();
			break;
#endif

#ifdef LICENSE
		case opInstallLicense:
			{
			const char *license = parameters->findValue ("license", NULL);
			if (!license)
				throw SQLEXCEPTION (RUNTIME_ERROR, "installLicense requires license");
			licenseManager->installLicense (license);
			licenseCheck();
			break;
			}
#endif

		case opShutdown:
			break;

		case opClone:
		case opCreateShadow:
			{
			const char *fileName = parameters->findValue("fileName", NULL);

			if (!fileName)
				throw SQLEXCEPTION (RUNTIME_ERROR, "Filename required to shadow database");

			dbb->cloneFile(this, fileName, op == opCreateShadow);
			}
			break;

		default:
			throw SQLEXCEPTION (RUNTIME_ERROR, "Server operation %d is not currently supported", op);
		}

}

Statement* Database::createStatement()
{
	return systemConnection->createStatement();
}

Table* Database::findTable (const char *schema, const char *name)
{
	if (!schema)
		return NULL;

	schema = symbolManager->getSymbol (schema);
	name = symbolManager->getSymbol (name);
	Sync sync (&syncTables, "Database::findTable");
	sync.lock (Shared);
	int slot = HASH (name, TABLE_HASH_SIZE);
	Table *table;

	for (table = tables [slot]; table; table = table->collision)
		if (table->name == name && table->schemaName == schema)
			return table;

	sync.unlock();

	for (UnTable *untable = unTables [slot]; untable;
		 untable = untable->collision)
		if (untable->name == name && untable->schemaName == schema)
			return NULL;

	Sync syncSystemTransaction(&syncSysConnection, "Database::findTable");
	syncSystemTransaction.lock(Shared);
	PreparedStatement *statement = prepareStatement (
		"select tableName,tableId,dataSection,blobSection,currentVersion,schema,viewDefinition \
		 from Tables where tableName=? and schema=?");
	statement->setString (1, name);
	statement->setString (2, schema);
	ResultSet *resultSet = statement->executeQuery();
	table = loadTable (resultSet);
	resultSet->close();
	statement->close();

	if (!table)
		{
		UnTable *untable = new UnTable (schema, name);
		untable->collision = unTables [slot];
		unTables [slot] = untable;
		}

	return table;
}

Table* Database::addTable(User *owner, const char *name, const char *schema)
{
	if (!schema)
		throw SQLEXCEPTION(DDL_ERROR, "no schema defined for table %s\n", name);
		
	if (!formatting && findTable (schema, name))
		throw SQLEXCEPTION (DDL_ERROR, "table %s is already defined", name);

	Table *table = new Table (this, nextTableId++, schema, name);
	addTable (table);

	return table;
}

int32 Database::createSection(Transaction *transaction)
{
	return dbb->createSection(TRANSACTION_ID(transaction));
}

void Database::addSystemTables()
{
	formatting = true;
	Statement *statement = createStatement();

	for (const char **sql = ods2Statements; *sql; ++sql)
		statement->execute (*sql);

	statement->close();
	formatting = false;
}

PreparedStatement* Database::prepareStatement(const char * sqlStr)
{
	return systemConnection->prepareStatement (sqlStr);
}

void Database::addRef()
{
	++useCount;
}

void Database::release()
{
	int n = --useCount;

	if (n == 1 && systemConnection)
		{
		Connection *temp = systemConnection;
		temp->commit();
		systemConnection = NULL;
		temp->close();
		}
	else if (n == 0)
		{
		shutdown();
		delete this;
		}
}

Statement* Database::createStatement(Connection * connection)
{
	return new Statement (connection, this);
}

PreparedStatement* Database::prepareStatement(Connection * connection, const char *sqlStr)
{
	PreparedStatement *statement = new PreparedStatement (connection, this);

	try
		{
		statement->setSqlString (sqlStr);
		}
	catch (...)
		{
		statement->close();
		throw;
		}

	return statement;
}


PreparedStatement* Database::prepareStatement(Connection *connection, const WCString *sqlStr)
{
	PreparedStatement *statement = new PreparedStatement (connection, this);

	try
		{
		statement->setSqlString (sqlStr);
		}
	catch (...)
		{
		statement->close();
		throw;
		}

	return statement;
}

CompiledStatement* Database::getCompiledStatement(Connection *connection, const char * sqlString)
{
	Sync sync (&syncStatements, "Database::getCompiledStatement(1)");
	sync.lock (Shared);
	//printf("%s\n", (const char*) sqlString);

	for (CompiledStatement *statement = compiledStatements; statement; statement = statement->next)
		if (statement->sqlString == sqlString && statement->validate (connection))
			{
			statement->addRef();
			sync.unlock();
			try
				{
				statement->checkAccess (connection);
				}
			catch (...)
				{
				statement->release();
				throw;
				}
			return statement;
			}

	sync.unlock();

	return compileStatement (connection, sqlString);
}


CompiledStatement* Database::getCompiledStatement(Connection *connection, const WCString *sqlString)
{
	Sync sync (&syncStatements, "Database::getCompiledStatement(WC)");
	sync.lock (Shared);
	//JString str(sqlString);
	//printf("%s\n", (const char*) str);
	
	for (CompiledStatement *statement = compiledStatements; statement; statement = statement->next)
		if (statement->sqlString == sqlString && statement->validate (connection))
			{
			statement->addRef();
			sync.unlock();
			try
				{
				statement->checkAccess (connection);
				}
			catch (...)
				{
				statement->release();
				throw;
				}
			return statement;
			}

	sync.unlock();
	JString sql (sqlString);

	return compileStatement (connection, sql);
}

CompiledStatement* Database::compileStatement(Connection *connection, JString sqlString)
{
	CompiledStatement *statement = new CompiledStatement (connection);

	try
		{
		statement->compile (sqlString);
		statement->checkAccess (connection);
		}
	catch (...)
		{
		delete statement;
		throw;
		}

	if (statement->useable && 
		(statement->numberParameters > 0 || !statement->filters.isEmpty()))
		{
		Sync sync (&syncStatements, "Database::compileStatement(1)");
		sync.lock (Shared);
		Sync sync2 (&syncAddStatement, "Database::compileStatement(2)");
		sync2.lock (Exclusive);
		statement->addRef();
		statement->next = compiledStatements;
		compiledStatements = statement;
		}

	return statement;
}

Transaction* Database::startTransaction(Connection *connection)
{
	return transactionManager->startTransaction(connection);
}


int32 Database::insertStub(int32 recordSection, Transaction *transaction)
{
	int32 recordNumber = dbb->insertStub (recordSection, TRANSACTION_ID(transaction));
	//Log::debug ("Created %d/%d\n", recordSection, recordNumber);

	return recordNumber;
}

void Database::logRecord(int32 section, int32 recordNumber, Stream * stream, Transaction *transaction)
{
	dbb->logRecord (section, recordNumber, stream, transaction);
}

void Database::updateRecord(int32 section, int32 recordNumber, Stream *stream, Transaction *transaction)
{
	dbb->updateRecord (section, recordNumber, stream, TRANSACTION_ID(transaction), false);
}

void Database::updateBlob(int32 section, int32 recordNumber, Stream *stream, Transaction *transaction)
{
	dbb->updateBlob (section, recordNumber, stream, TRANSACTION_ID(transaction));
}

void Database::expungeRecord(int32 section, int32 recordNumber)
{
	dbb->expungeRecord (section, recordNumber);
}

int32 Database::findNextRecord(int32 sectionId, int32 recordNumber, Stream *stream)
{
	/***
	int32 fetches = dbb->fetches;
	int32 reads = dbb->reads;
	int32 writes = dbb->writes;
	int32 fakes = dbb->fakes;
	***/
	int32 number = dbb->findNextRecord (sectionId, recordNumber, stream);

	/***
	fetches = dbb->fetches - fetches;
	fakes = dbb->fakes - fakes;
	reads = dbb->reads - reads;
	writes = dbb->writes - writes;
	***/

	return number;
}

bool Database::fetchRecord(int32 sectionId, int32 recordNumber, Stream * stream)
{
	return dbb->fetchRecord (sectionId, recordNumber, stream);
}

void Database::flush()
{
	dbb->flush();
}

void Database::commitSystemTransaction()
{
	Sync sync (&syncSysConnection, "Database::commitSystemTransaction");
	sync.lock (Exclusive);
	systemConnection->commit();
}

int32 Database::createIndex(Transaction *transaction)
{
	return dbb->createIndex(TRANSACTION_ID(transaction));
}

bool Database::addIndexEntry(int32 indexId, int indexVersion, IndexKey *key, int32 recordNumber, Transaction *transaction)
{
	return dbb->addIndexEntry (indexId, indexVersion, key, recordNumber, TRANSACTION_ID(transaction));
}

void Database::deleteIndex(int32 indexId, int indexVersion, Transaction *transaction)
{
	dbb->deleteIndex (indexId, indexVersion, TRANSACTION_ID(transaction));
}

/***
void Database::scanIndex(int32 indexId, int indexVersion, IndexKey* lowKey, IndexKey* highKey, bool partial, Bitmap *bitmap)
{
	dbb->scanIndex (indexId, indexVersion, lowKey, highKey, partial, bitmap);
}
***/

void Database::setDebug()
{
	dbb->setDebug();
}

void Database::clearDebug()
{
	dbb->setDebug();
}

int32 Database::addInversion(InversionFilter * filter, Transaction *transaction)
{
	return dbb->inversion->addInversion (filter, TRANSACTION_ID(transaction), true);
}

void Database::removeFromInversion(InversionFilter *filter, Transaction *transaction)
{
	dbb->inversion->addInversion (filter, TRANSACTION_ID(transaction), false);
}

void Database::search(ResultList *resultList, const char * string)
{
	Search search (string, searchWords);
	search.search (resultList);
}

void Database::reindex(Transaction *transaction)
{
	//rebuildIndexes();

	dbb->inversion->deleteInversion (TRANSACTION_ID(transaction));
	dbb->inversion->createInversion (TRANSACTION_ID(transaction));

	for (Table *table = tableList; table; table = table->next)
		table->reIndexInversion(transaction);
}

Table* Database::getTable(int tableId)
{
	Table *table;

	for (table = tablesModId [tableId % TABLE_HASH_SIZE]; table; 
		 table = table->idCollision)
		if (table->tableId == tableId)
			return table;

	PreparedStatement *statement = prepareStatement (
		"select tableName,tableId,dataSection,blobSection,currentVersion,schema,viewDefinition \
		 from system.Tables where tableid=?");
	statement->setInt (1, tableId);
	ResultSet *resultSet = statement->executeQuery();

	table = loadTable (resultSet);
	resultSet->close();
	statement->close();

	return table;
}

Table* Database::loadTable(ResultSet * resultSet)
{
	Sync sync (&syncTables, "Database::loadTable");

	if (!resultSet->next())
		return NULL;

	const char *name = getString (resultSet->getString(1));
	int version = resultSet->getInt (5);
	const char *schemaName = getString (resultSet->getString (6));
	Table *table = new Table (this, schemaName, name, resultSet->getInt (2), version);

	int dataSection = resultSet->getInt (3);

	if (dataSection)
		{
		table->setDataSection (dataSection);
		table->setBlobSection (resultSet->getInt (4));
		}
	else
		{
		const char *viewDef = resultSet->getString (7);
		if (viewDef [0])
			{
			CompiledStatement statement (systemConnection);
			JString string;
			// Do a little backward compatibility
			if (strncmp (viewDef, "create view ", strlen ("create view ")) == 0)
				string = viewDef;
			else
				string.Format ("create view %s.%s %s", schemaName, name, viewDef);
			table->setView (statement.getView (string));
			}
		}

	table->loadStuff();
	sync.lock (Exclusive);
	addTable (table);

	return table;
}


bool Database::matches(const char * fileName)
{
	return strcasecmp (dbb->fileName, fileName) == 0;
}

void Database::flushInversion(Transaction *transaction)
{
	dbb->inversion->flush (TRANSACTION_ID(transaction));
}

void Database::dropTable(Table * table, Transaction *transaction)
{
	table->checkDrop();

	// Check for records in active transactions.  If so, barf

	if (hasUncommittedRecords(table, transaction))
			throw SQLError(UNCOMMITTED_UPDATES, "table %s.%s has uncommitted updates and can't be dropped", 
						   table->schemaName, table->name);
			
	// OK, now make sure any records are purged out of committed transactions as well
	
	transactionManager->dropTable(table, transaction);		
	Sync sync (&syncTables, "Database::dropTable");
	sync.lock (Exclusive);

	// Remove table from linear table list

	Table **ptr;

	for (ptr = &tableList; *ptr; ptr = &((*ptr)->next))
		if (*ptr == table)
			{
			*ptr = table->next;
			break;
			}

	// Remove table from name hash table

	for (ptr = tables + HASH (table->name, TABLE_HASH_SIZE); *ptr;
		 ptr = &((*ptr)->collision))
		if (*ptr == table)
			{
			*ptr = table->collision;
			break;
			}

	// Remove table from id hash table

	for (ptr = tablesModId + table->tableId % TABLE_HASH_SIZE; *ptr;
		 ptr = &((*ptr)->idCollision))
		if (*ptr == table)
			{
			*ptr = table->idCollision;
			break;
			}

	invalidateCompiledStatements(table);
	sync.unlock();
	table->drop(transaction);
	table->expunge(getSystemTransaction());
	commitSystemTransaction();
	delete table;
}


void Database::addTable(Table * table)
{
	Sync sync (&syncTables, "Database::addTable");
	sync.lock (Exclusive);

	if (formatting)
		{
		Table **ptr;
		for (ptr = &tableList; *ptr; ptr = &((*ptr)->next))
			;
		table->next = *ptr;
		*ptr = table;
		}
	else
		{
		table->next = tableList;
		tableList = table;
		}

	int slot = HASH (table->name, TABLE_HASH_SIZE);
	table->collision = tables [slot];
	tables [slot] = table;

	slot = table->tableId % TABLE_HASH_SIZE;
	table->idCollision = tablesModId [slot];
	tablesModId [slot] = table;

	// Notify folks who track stuff that there's a new table

#ifndef STORAGE_ENGINE
	templateManager->tableAdded (table);
	applications->tableAdded (table);
#endif

#ifdef LICENSE
	licenseManager->tableAdded (table);
#endif

	imageManager->tableAdded (table);
	searchWords->tableAdded (table);
}

void Database::execute(const char * sql)
{
	Statement *statement = createStatement();
	statement->execute (sql);
	statement->close();
}

void Database::shutdown()
{
	if (shuttingDown)
		return;
	
	shuttingDown = true;
	
	if (systemConnection && 
		systemConnection->transaction && 
		systemConnection->transaction->state == Active)
		systemConnection->commit();
		
	if (repositoryManager)
		repositoryManager->close();

	flush();

	if (scheduler)
		scheduler->shutdown(false);

	if (internalScheduler)
		internalScheduler->shutdown(false);
	
	if (pageWriter)
		pageWriter->shutdown(false);

#ifndef STORAGE_ENGINE
	if (java)
		java->shutdown (false);
#endif

	if (threads)
		{
		threads->shutdownAll();
		threads->waitForAll();
		}

	if (serialLog)
		serialLog->shutdown();

	dbb->shutdown(0);

	if (serialLog)
		serialLog->close();
}

void Database::deleteSection(int32 sectionId, Transaction *transaction)
{
	dbb->deleteSection (sectionId, TRANSACTION_ID(transaction));
	if (transaction)
		transaction->hasUpdates = true;

}

void Database::invalidateCompiledStatements(Table * table)
{
	Sync sync (&syncStatements, "Database::invalidateCompiledStatements");
	sync.lock (Exclusive);

	for (CompiledStatement *statement, **ptr = &compiledStatements; (statement = *ptr);)
		if (statement->references (table))
			{
			statement->invalidate();
			*ptr = statement->next;
			statement->release();
			}
		else
			ptr = &(*ptr)->next;
}


User* Database::createUser(const char * account, const char * password, bool encrypted, Coterie *coterie)
{
	return roleModel->createUser (getSymbol (account), password, encrypted, coterie);
}

User* Database::findUser(const char * account)
{
	return roleModel->findUser (symbolManager->getSymbol (account));
}

Role* Database::findRole(const char * schemaName, const char * roleName)
{
	const char *schema = symbolManager->getSymbol (schemaName);
	const char *role = symbolManager->getSymbol (roleName);
	return roleModel->findRole (schema, role);
}


Role* Database::findRole(const WCString *schemaName, const WCString *roleName)
{
	return roleModel->findRole (symbolManager->getSymbol (schemaName), 
								symbolManager->getSymbol (roleName));
}

void Database::validate(int optionMask)
{
	Sync sync (&syncObject, "Database::validate");
	sync.lock (Exclusive);
	dbb->validate (optionMask);

	if (optionMask & validateBlobs)
		{
		Sync sync2 (&syncSysConnection, "Database::validate");
		sync2.lock (Shared);
		PreparedStatement *statement = prepareStatement (
			"select tableId,viewDefinition from system.tables");
		ResultSet *resultSet = statement->executeQuery();

		while (resultSet->next())
			if (!resultSet->getString (2)[0])
				{
				Table *table = getTable (resultSet->getInt (1));
				table->validateBlobs (optionMask);
				}

		resultSet->close();
		statement->close();
		}

	Log::debug ("Database::validate: validation complete\n");
}

void Database::scavenge()
{
	//OpSys::logParameters();

	// Start by scavenging compiled statements.  If they're moldy and not in use,
	// off with their heads!

	Sync sync (&syncStatements, "Database::scavenge");
	sync.lock (Exclusive);
	time_t threshold = timestamp - STATEMENT_RETIREMENT_AGE;
	int n = 0;

	for (CompiledStatement *statement, **ptr = &compiledStatements; (statement = *ptr);)
		if (statement->useCount > 1 || statement->lastUse > threshold)
			ptr = &statement->next;
		else
			{
			*ptr = statement->next;
			statement->release();
			}
	
	sync.unlock();

	// Next, scanvenge tables

	retireRecords();				// age group based scavenger

	// Scavenge expired licenses
	
	DateTime now;
	now.setNow();

#ifdef LICENSE
	licenseManager->scavenge (&now);
#endif
	
#ifndef STORAGE_ENGINE
	sessionManager->scavenge (&now);
#endif
	
	dbb->reportStatistics();
	repositoryManager->reportStatistics();
}


void Database::retireRecords()
{
	if (systemConnection->transaction)
		commitSystemTransaction();
		
	int threshold = 0;
	int64 total = 0;
	TransId oldestActiveTransaction = transactionManager->findOldestActive();
	int n;

	// Compute record memory in use.  Note point we pass lower limit

	for (n = 0; n < AGE_GROUPS; ++n)
		{
		int32 size = ageGroupSizes [n];
		
		if (size > 0)
			total += size;
			
		if (!threshold && total > recordMemoryLower)
			threshold = currentAgeGroup - n;
		}

	total += overflowSize;

	// If we passed the upper limit, scavenge.  If we didn't pick up
	// a significant amount of memory since the last cycle, don't bother
	// bumping the age group.

	if (total > recordMemoryUpper)
		{
		printRecordMemory (threshold, total);
		Sync sync (&syncTables, "Database::retireRecords");
		sync.lock (Shared);
		int count = 0;
		int skipped = 0;
		
		for (Table *table = tableList; table; table = table->next)
			{
			try
				{
				int n = table->retireRecords (threshold, oldestActiveTransaction);
				
				if (n >= 0)
					count += n;
				else
					++skipped;
				}
			catch (SQLException &exception)
				{
				Log::debug ("Exception during scavenger of table %s.%s: %s\n",
						table->schemaName, table->name, exception.getText());
				}
			}
			
		int newTotal = overflowSize;
		
		for (int n = 0; n < AGE_GROUPS; ++n)
			newTotal += ageGroupSizes [n];
			
		Log::log(LogScavenge, " %d bytes reclaimed\n", total - newTotal);
		total = newTotal;
		}
	else if ((total - lastRecordMemory) < (recordMemoryUpper - recordMemoryLower) / 20)
		{
		cleanupRecords (-1, oldestActiveTransaction);
		
		return;
		}
	else
		{
		cleanupRecords (-1, oldestActiveTransaction);
		printRecordMemory (threshold, total);
		}

	lastRecordMemory = total;
	INTERLOCKED_INCREMENT (currentAgeGroup);
	overflowSize += ageGroupSizes [AGE_GROUPS - 1];

	for (n = AGE_GROUPS - 1; n > 0; --n)
		ageGroupSizes [n] = ageGroupSizes [n - 1];

	ageGroupSizes [0] = 0;
	transactionManager->scavengeRecords(threshold);
}

void Database::printRecordMemory(int64 threshold, int64 total)
{
	Log::debug ("Record Memory usage for %s:\n", (const char*) name);
	int max;

	for (max = AGE_GROUPS - 1; max > 0; --max)
		if (ageGroupSizes [max])
			break;

	for (int n = 0; n <= max; ++n)
		if (ageGroupSizes [n])
			Log::debug ("  %d. %d\n", currentAgeGroup - n, ageGroupSizes [n]);

	Log::log(LogScavenge, " total: " I64FORMAT ", threshold " I64FORMAT "%s\n", total, threshold,
				(total > recordMemoryUpper) ? " -- scavenge" : "");
}


bool Database::deleteIndexEntry(int32 indexId, int indexVersion, IndexKey *key, int32 recordNumber, Transaction *transaction)
{
	return dbb->deleteIndexEntry (indexId, indexVersion, key, recordNumber, TRANSACTION_ID(transaction));
}

void Database::ticker(void * database)
{
	((Database*) database)->ticker();
}

void Database::ticker()
{
	Thread *thread = Thread::getThread("Database::ticker");

	while (!thread->shutdownInProgress)
		{
		timestamp = time (NULL);
		thread->sleep (1000);
		}
}

int Database::createSequence(QUAD initialValue)
{
	Transaction *transaction = systemConnection->getTransaction();

	return dbb->createSequence (initialValue, TRANSACTION_ID(transaction));
}

int64 Database::updateSequence(int sequenceId, int64 delta, Transaction *transaction)
{
	return dbb->updateSequence (sequenceId, delta, TRANSACTION_ID(transaction));
}


Transaction* Database::getSystemTransaction()
{
	return systemConnection->getTransaction();
}


void Database::rebuildIndexes()
{
	Transaction *transaction = getSystemTransaction();

	for (Table *table = tableList; table; table = table->next)
		table->reIndex(transaction);

	commitSystemTransaction();
}

const char* Database::getSymbol(const char *string)
{
	return symbolManager->getSymbol (string);
}

bool Database::isSymbol(const char *string)
{
	return symbolManager->isSymbol (string);
}

const char* Database::getSymbol(const WCString *string)
{
	return symbolManager->getSymbol (string);
}

const char* Database::getString(const char *string)
{
	return symbolManager->getString (string);
}

void Database::upgradeSystemTables()
{
	Table *table = findTable ("SYSTEM", "SCHEMAS");

	if (!table)
		{
		Statement *statement = createStatement();
		
		for (const char **sql = ods2UpgradeStatements; *sql; ++sql)
			statement->execute (*sql);
			
		statement->close();
		}

	table = findTable ("SYSTEM", "FIELDS");
	table->refreshFields();
	table->loadIndexes();

	//table = findTable ("SYSTEM", "FIELDS");
	
	if (!table->findField ("PRECISION"))
		{
		Statement *statement = createStatement();
		
		for (const char **sql = ods3Upgrade; *sql; ++sql)
			statement->execute (*sql);
			
		statement->close();
		}

	table = findTable ("SYSTEM", "INDEXFIELDS");
	table->refreshFields();

	fieldExtensions = true;
}

JString Database::analyze(int mask)
{
	Stream stream;
	stream.setMalloc (true);
	Sync syncSystemTransaction(&syncSysConnection, "Database::analyze");

	if (mask & analyzeMemory)
		MemMgrAnalyze (mask, &stream);

#ifndef STORAGE_ENGINE
	if (mask & analyzeMemory)
		java->analyzeMemory (mask, &stream);

	if (mask & analyzeClasses)
		{
		stream.putSegment ("\nClasses\n");
		java->analyze (mask, &stream);
		stream.putCharacter ('\n');
		}

	if (mask & analyzeObjects)
		{
		stream.putSegment ("\nObject allocations\n");
		java->analyzeObjects (mask, &stream);
		stream.putCharacter ('\n');
		}
#endif

	if (mask & analyzeRecords)
		{
		stream.putSegment ("\nRecords\n");
		
		for (Table *table = tableList; table; table = table->next)
			{
			int count = table->countActiveRecords();
			
			if (count)
				stream.format ("%s.%s\t%d\n", table->schemaName, table->name, count);
			}
			
		stream.putCharacter ('\n');
		}

	if (mask & analyzeStatements)
		{
		stream.putSegment ("\nStatements\n");
		Sync sync (&syncStatements, "Database::analyze");
		sync.lock (Shared);

		for (CompiledStatement *statement = compiledStatements; statement;
			 statement = statement->next)
			{
			stream.putSegment (statement->sqlString);
			stream.format ("\t(%d)\n", statement->countInstances());
			}
		stream.putCharacter ('\n');
		sync.unlock();
		}

	if (mask & analyzeTables)
		{
		syncSystemTransaction.lock(Shared);
		PreparedStatement *statement = prepareStatement (
			"select schema,tableName,dataSection,blobSection from tables order by schema, tableName");
		PreparedStatement *indexQuery = prepareStatement(
			"select indexName, indexId from system.indexes where schema = ? and tableName = ?");
		ResultSet *resultSet = statement->executeQuery();

		while (resultSet->next())
			{
			int n = 1;
			const char *schema = resultSet->getString (n++);
			const char *tableName = resultSet->getString (n++);
			int dataSection = resultSet->getInt (n++);
			int blobSection = resultSet->getInt (n++);
			stream.format ("Table %s.%s\n", schema, tableName);
			dbb->analyzeSection (dataSection, "Data section", 3, &stream);
			dbb->analyzeSection (blobSection, "Blob section", 3, &stream);
			
			indexQuery->setString(1, schema);
			indexQuery->setString(2, tableName);
			ResultSet *indexes = indexQuery->executeQuery();
			
			while (indexes->next())
				{
				const char *indexName = indexes->getString(1);
				int combinedId = indexes->getInt(2);
				int indexId = INDEX_ID(combinedId);
				int indexVersion = INDEX_VERSION(combinedId);
				dbb->analyseIndex(indexId, indexVersion, indexName, 3, &stream);
				}
			
			indexes->close();
			}

		statement->close();
		indexQuery->close();
		}

	if (mask & analyzeSpace)
		dbb->analyzeSpace(0, &stream);
			
	if (mask & analyzeCache)
		dbb->cache->analyze (&stream);

	return stream.getJString();
}

int Database::getMemorySize(const char *string)
{
	int n = 0;

	for (const char *p = string; *p;)
		{
		char c = *p++;
		if (c >= '0' && c <= '9')
			n = n * 10 + c - '0';
		else if (c == 'm' || c == 'M')
			n *= 1000000;
		else if (c == 'k' || c == 'K')
			n *= 1000;
		}

	return n;
}


void Database::cleanupRecords(int threshold, TransId oldestActiveTransaction)
{
	Sync sync (&syncTables, "Database::cleanupRecords");
	sync.lock (Shared);

	for (Table *table = tableList; table; table = table->next)
		{
		try
			{
			table->cleanupRecords (threshold, oldestActiveTransaction);
			}
		catch (SQLException &exception)
			{
			Log::debug ("Exception during cleanupRecords of table %s.%s: %s\n",
					table->schemaName, table->name, exception.getText());
			}
		}
}

void Database::licenseCheck()
{
#ifdef LICENSE
	licensed = false;
	LicenseProduct *product = licenseManager->getProduct (SERVER_PRODUCT);

	if (!(licensed = product->isLicensed()))
		{
		DateTime expiration = DateTime::convert (BUILD_DATE);
		expiration.add (EXPIRATION_DAYS * 24 * 60 * 60);
		DateTime now;
		now.setNow();
		if (now.after (expiration))
			throw SQLError (LICENSE_ERROR, "Unlicensed server usage period has expired");
		}	
#endif
}

Repository* Database::findRepository(const char *schema, const char *name)
{
	return repositoryManager->findRepository (schema, name);
}

Repository* Database::getRepository(const char *schema, const char *name)
{
	return repositoryManager->getRepository (schema, name);
}

Repository* Database::createRepository(const char *name, const char *schema, Sequence *sequence, const char *fileName, int volume, const char *rolloverString)
{
	return repositoryManager->createRepository (name, schema, sequence, fileName, volume, rolloverString);
}

Schema* Database::getSchema(const char *name)
{
	int slot = HASH (name, TABLE_HASH_SIZE);
	Schema *schema;

	for (schema = schemas [slot]; schema; schema = schema->collision)
		if (schema->name == name)
			return schema;

	schema = new Schema (this, name);
	schema->collision = schemas [slot];
	schemas [slot] = schema;

	return schema;
}

void Database::deleteRepository(Repository *repository)
{
	repositoryManager->deleteRepository (repository);
}

void Database::deleteRepositoryBlob(const char *schema, const char *repositoryName, int volume, QUAD blobId, Transaction *transaction)
{
	Repository *repository = getRepository (getSymbol (schema), getSymbol (repositoryName));
	repository->deleteBlob (volume, blobId, transaction);	
}

void Database::commit(Transaction *transaction)
{
	dbb->commit(transaction);
}

void Database::rollback(Transaction *transaction)
{
	dbb->rollback(transaction->transactionId, transaction->hasUpdates);
}

/***
void Database::prepare(Transaction *transaction)
{
	dbb->prepareTransaction(transaction->transactionId);
}
***/

void Database::renameTable(Table* table, const char* newSchema, const char* newName)
{
	newSchema = getSymbol(newSchema);
	newName = getSymbol(newName);
	Sync sync (&syncTables, "Database::renameTable");
	sync.lock (Exclusive);
	roleModel->renameTable(table, newSchema, newName);

	// Remove table from name hash table

	for (Table **ptr = tables + HASH (table->name, TABLE_HASH_SIZE); *ptr;
		 ptr = &((*ptr)->collision))
		if (*ptr == table)
			{
			*ptr = table->collision;
			break;
			}

	// Add table back to name hash table

	table->name = newName;
	table->schemaName = newSchema;
	int slot = HASH (table->name, TABLE_HASH_SIZE);
	table->collision = tables[slot];
	tables[slot] = table;

	invalidateCompiledStatements(table);
}

void Database::dropDatabase()
{
	shutdown();
	
	if (serialLog)
		serialLog->dropDatabase();

	dbb->dropDatabase();
}

void Database::shutdownNow()
{
	panicShutdown = true;

	if (cache)
		cache->shutdownNow();

	if (serialLog)
		serialLog->shutdownNow();
}

/***
Transaction* Database::findTransaction(TransId transactionId)
{
	Sync sync (&activeTransactions.syncObject, "Database::findTransaction");
	sync.lock (Shared);
	Transaction *transaction;

	for (transaction = activeTransactions.first; transaction; transaction = transaction->next)
		if (transaction->transactionId == transactionId)
			return transaction;
	
	sync.unlock();
	Sync syncCommitted (&committedTransactions.syncObject, "Database::findTransaction");
	syncCommitted.lock(Shared);
	
	for (transaction = committedTransactions.first; transaction; transaction = transaction->next)
		if (transaction->transactionId == transactionId)
			return transaction;
	
			
	return NULL;
}
***/

void Database::validateCache(void)
{
	dbb->validateCache();
}

bool Database::hasUncommittedRecords(Table* table, Transaction *transaction)
{
	return transactionManager->hasUncommittedRecords(table, transaction);
}

void Database::commitByXid(int xidLength, const UCHAR* xid)
{
	serialLog->commitByXid(xidLength, xid);
	transactionManager->commitByXid(xidLength, xid);
}

void Database::rollbackByXid(int xidLength, const UCHAR* xid)
{
	serialLog->rollbackByXid(xidLength, xid);
	transactionManager->rollbackByXid(xidLength, xid);
}
