/* Copyright (C) 2006 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

// BigInt.cpp: implementation of the BigInt class.
//
//////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include "Engine.h"
#include "BigInt.h"

static const int powersOfTen [] = {
	1,
	10,
	100,
	1000,
	10000,
	100000,
	1000000,
	10000000,
	100000000,
	1000000000
	};

static int byteShifts [] = { 24, 16, 8, 0 };

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

BigInt::BigInt()
{
	clear();
}

BigInt::~BigInt()
{

}

BigInt::BigInt(BigInt *bigInt)
{
	set(bigInt);
}

void BigInt::set(BigInt *bigInt)
{
	neg = bigInt->neg;
	length = bigInt->length;
	scale = bigInt->scale;

	for (int n = 0; n < length; ++n)
		words[n] = bigInt->words[n];
}

void BigInt::set(int64 value, int valueScale)
{
	scale = valueScale;
	neg = value < 0;
	length = 0;

	if (value)
		{
		if (neg)
			value = -value;

		words[length++] = LOW_WORD(value);
		BigWord highWord = HIGH_WORD(value);

		if (highWord)
			words[length++] = highWord;
		}
}


void BigInt::subtract(int index, BigWord value)
{
	while (value)
		{
		BigDWord acc = FETCH_WORD(index) - value;
		words[index++] = LOW_WORD(acc);
		value = HIGH_WORD(acc);
		}

	if (index > length)
		length = index;
}

int64 BigInt::getInt()
{
	int64 value = ((int64) FETCH_WORD(1) << bigWordBits) + FETCH_WORD(0);

	return (neg) ? -value : value;
}

void BigInt::print(const char *msg)
{
	char temp[128];
	getString(sizeof(temp), temp);
	printf ("%s(%d): %s\n", msg, length, temp);
}

void BigInt::multiply(BigWord value)
{
	BigInt org(this);
	length = 0;

	for (int n = 0; n < org.length; ++n)
		{
		BigDWord acc = (BigDWord) value * org.words[n];
		add(n, LOW_WORD(acc));
		add(n + 1, HIGH_WORD(acc));
		}

}

void BigInt::add(int64 value)
{
	add(0, LOW_WORD(value));
	add(1, HIGH_WORD(value));
}

BigWord BigInt::divide(BigWord value)
{
	BigWord rem = 0;

	for (int n = length; n; --n)
		{
		BigDWord acc = GET_INT64(rem, words[n - 1]);
		words[n - 1] = (BigWord) (acc / value);
		rem = (BigWord) (acc % value);
		}

	normalize();
	
	return rem;
}

void BigInt::set(BigWord value)
{
	scale = 0;
	neg = false;
	
	if (value)
		{
		words[0] = value;
		length = 1;
		}
	else
		length = 0;
		
}

int BigInt::getByteLength(void)
{
	if (length == 0)
		return 0;
	
	BigWord highWord = words[length - 1];

	for (int n = 0; n < 4; ++n)
		{
		unsigned char byte = (unsigned char) (highWord >> byteShifts[n]);
		
		if (byte)
			return length * sizeof(BigWord) - n + ((byte & 0x80) ? 1 : 0);
		}
		
	return length * sizeof(BigWord);
}

void BigInt::getBytes(char* bytes)
{
	if (length == 0)
		return;

	char *p = bytes;
	int position = length;
	BigWord highWord = words[--position];
	bool sig = false;
	
	// Handle first word -- skip zero high order bytes, handle sign, etc
	
	for (int n = 0; n < 4; ++n)
		{
		char byte = (char) (highWord >> byteShifts[n]);
		
		if (sig)
			*p++ = byte;
		else if (byte)
			{
			if (byte & 0x80)
				*p++ = 0;
			
			*p++ = byte;
			sig = true;
			}
		}
	
	// The rest of the words are boring -- just squirt out the byte in the right order
	
	while (position > 0)
		{
		BigWord word = words[--position];
		
		for (int m = 0; m < 4; ++m)
			*p++ = (char) (word >> byteShifts[m]);
		}
	
	if (neg)
		*bytes |= 0x80;
}

void BigInt::setBytes(int scl, int byteLength, const char* bytes)
{
	scale = scl;
	neg = (*bytes & 0x80) != 0;
	const UCHAR *p = (const UCHAR*) bytes;
	int position = length = (byteLength + sizeof(BigWord) - 1) / sizeof(BigWord);
	int partial = byteLength % sizeof(BigWord);
	
	if (partial == 0)
		partial = sizeof(BigWord);

	BigWord word = *p++ & 0x7f;
	
	for (int n = 1; n < partial; ++n)
		{
		word <<= 8;
		word |= *p++;
		}

	words[--position] = word;
	
	while (position > 0)
		{
		for (int n = 0; n < sizeof(BigWord); ++n)
			{
			word <<= 8;
			word |= *p++;
			}
			
		words[--position] = word;
		}

	normalize();
}

int BigInt::scaleTo(int toScale)
{
	int delta = toScale - scale;
	scale = toScale;
	
	if (delta == 0)
		return 0;
	
	if (delta > 0)
		{
		while (delta > 9)
			{
			divide(powersOfTen[9]);
			delta -= 9;
			}
			
		return divide(powersOfTen[delta]);
		}
	
	while (delta < -9)
		{
		multiply(powersOfTen[9]);
		delta += 9;
		}

	multiply(powersOfTen[-delta]);
	
	return 0;
}

void BigInt::setString(int stringLength, const char* string)
{
	length = 0;
	neg = false;
	scale = 0;
	bool decimalPoint = false;
	char c;
	
	for (const char *p = string, *end = string + stringLength; p < end && (c = *p++);)
		{
		if (c >= '0' && c <= '9')
			{
			multiply(10);
			add(0, c - '0');
			
			if (decimalPoint)
				--scale;
			}
		else if (c == '-')
			neg = !neg;
		else if (c == '.')
			decimalPoint = true;
		}	
}

void BigInt::getString(int bufferLength, char* buffer)
{
	char temp[128];
	char *p = temp;
	BigInt number(this);
	
	while (number.length)
		{
		int n = number.divide(10);
		*p++ = '0' + n;
		}
	
	char *q = buffer;
	char *decimalPoint = (scale) ? buffer + (p - temp) + scale : buffer + bufferLength;
		
	if (neg)
		*q++ = '-';
	
	while (p > temp)
		{
		if (q == decimalPoint)
			*q++ = '.';
		
		*q++ = *--p;
		}
	
	*q = 0;
}

double BigInt::getDouble(void)
{
	double d = 0;
	
	for (int n = length - 1; n >= 0; --n)
		d = d * 4294967296. + words[n];
	
	if (scale < 0)
		d /= powersOfTen[-scale];
	else if (scale > 0)
		d *= powersOfTen[scale];
	
	return (neg) ? -d : d;
}

// If we're higher than the other guy, return 1; if lower, return -1; if equal, return 0

int BigInt::compare(BigInt* bigInt)
{
	// If we're zero, do a quick look at the other guy
	
	if (length == 0)
		{
		if (bigInt->length == 0)
			return 0;
		
		return (bigInt->neg) ? 1 : -1;
		}
	
	// And if the other guy is zero, look at it
	
	if (bigInt->length == 0)
		return (neg) ? -1 : 1;
	
	// If we have different signs, things are also easy
	
	if (neg != bigInt->neg)
		return (neg) ? -1 : 1;
	
	// If the scales are the same, we get off easily
	
	int ret;
	
	if (scale == bigInt->scale)
		ret = compareRaw(bigInt);
	else if (scale < bigInt->scale)
		{
		BigInt temp(bigInt);
		temp.scaleTo(scale);
		ret = compareRaw(&temp);
		}
	else
		{
		BigInt temp(this);
		temp.scaleTo(bigInt->scale);
		ret = temp.compareRaw(bigInt);
		}
	
	return (neg) ? -ret : ret;
}
