/**
*** Program jdbcMysqlBag.java     
***    in product twz1jdbcForMysql, 
***    Copyright 1997, 1998 by Terrence W. Zellers.
***   
***  All rights explicitly reserved.
***
***  See file "LICENSE" in this package for conditions of use.
**/

package twz1.jdbc.mysql;

import twz1.jdbc.mysql.jdbcMysqlBase;
import twz1.jdbc.mysql.jdbcMysqlDebug;
import twz1.jdbc.mysql.jdbcMysqlDumpBuffer;

import java.sql.*;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.IOException;
import java.net.*;

/** Monty and the gwe/jms drivers call the information encapsulation
*** unit for the MySQL protocol a packet.   The word is overworked.
*** Just to be different I'm calling it a bag.
**/

final class jdbcMysqlBag
{

/** Default buffer size */
static final int DEFAULTBUFFERSIZE = 8192;
static final int HEADERSIZE = 4;

/** Size of the currently allocated buffer including header. */
int capacity;

/** Sequence number */
int seq;

/** Where we are in the bag. */
int at;

/** Maximum legal value for at */
int maxAt; 

/** The goodies in the bag. */
byte[] inside;

/** fixed Head */
byte[] fhead;

/** Input connector. */
BufferedInputStream bis;

/** Output connector. */
BufferedOutputStream bos;

/** dump on read. */
boolean dumpRead;

/** dump on write. */
boolean dumpWrite;

/** Source connection */
jdbcMysqlConnex connex;

/** Boolean teeny - use "small" headers for big numbers */
boolean teeny;

/*-------------------------------------------------------------------+
|                         Error strings                              |
+-------------------------------------------------------------------*/

static final String[] errs = { 
    "E0100 Placeholder.",
    "P0101 Input stream null in jdbcMysqlBag.read().",
    "E0102 Socket stream was closed in while read was incomplete.",
    "E0103 IO exception returned while reading socket -",
    "P0104 Data request beyonds limits of received data in iTob().",
    "P0105 Data request to read beyond max in bToI();",
    "E0106 A getZTbytes() requested beyond end of data.",
    "E0107 Error writing bag to socket.",
    "E0108 putZT(String) - " ,
    "E0109 putZT(byte[]) - ",
    "E0110 getZTstring() - ",
    "E0111 newCapacity(int) - ",
    "E0112 read() - ",
    "E0113 getZTbytes() - ",
    "E0114 Data request beyond end of data in viToI().",
    "E0115 Error in viToI() -",
    "E0116 Error in nbToI() -",
    "E0117 In getNbytes() the requested size would exceed data.",
    "E0118 Error in getNbytes() - ", 
    "E0119 Error in getNstring() - ",
    "E0120 putByteArray() - ",
    "E0121 Error in getRbytes() -",
    "E0122 Error in getRstring() -",
    "E0123 Numeric overflow() -",
                     };


/*===================================================================+
||                       Main constructor                           ||
+===================================================================*/

/** Default constructor */
jdbcMysqlBag()
    {
    this.capacity = HEADERSIZE + DEFAULTBUFFERSIZE;
    init();
    }

jdbcMysqlBag(int newSize)
    {
    this.capacity = newSize;
    init();
    }

void init()
    {
    this.seq = 0;
    this.at = 0;
    this.maxAt = 0; 
    this.bos = null;
    this.bis = null;
    this.dumpRead  = false;
    this.dumpWrite = false;
    this.inside = null;
    if(this.capacity > 0)this.inside = new byte[this.capacity];
    this.fhead = new byte[HEADERSIZE];
    this.teeny = false;
    }

/*===================================================================+
||                   simple set and get methods                     ||
+===================================================================*/

jdbcMysqlBag dataCopy()
    {
    jdbcMysqlBag nb = new jdbcMysqlBag(maxAt);
    nb.maxAt = maxAt;
    nb.at = HEADERSIZE;
    System.arraycopy(inside, 0, nb.inside, 0, maxAt);
    return nb;
    }

/** Set the input */
void setInput(BufferedInputStream is) { this.bis = is; }

/** Set the output */
void setOutput(BufferedOutputStream os) {this.bos = os; }

/** Set this bag for write. */
void newWrite() { at = maxAt = 4; }
    
/** Set this bag for write.
*** @param s The new sequence number for this bag.
**/
void newWrite(int s) { at = maxAt = 4; seq = s; }

/** Set the packet sequence number. */
void setSequence(int s) { seq = s; }

/** Retrieve the packet sequence number */
int getSequence() { return seq; }

/** Get the current position */
int getPosition() { return at;}

/** Get the size of the bag (less header) */
int getSize() { return maxAt - HEADERSIZE; }

/** Set the current postion */
void setPosition(int a) { at = a; }

/** set the read dump */
void setReadDump(boolean onoff) { dumpRead = onoff; }

/** set the write dump */
void setWriteDump(boolean onoff) { dumpWrite = onoff; }

/** Look at current byte without changing ref */
int peekAbyte()
    {
    int w = inside[at];
    if(w < 0) w += 256;
    return w;
    }


/*-------------------------------------------------------------------+
|                     A test read to read bags                       |
+-------------------------------------------------------------------*/
void testread()
    {
    Thread me = Thread.currentThread();
    boolean done = false;
    while(!done)
        {
        try {
            if(bis.available() > 0) read();
            else me.sleep(2000);
            if(bis.available() == 0) done = true;
            }
        catch(Exception e) { e.printStackTrace(); }
        }
    }

/*===================================================================+
||                  read a "packet" from mysql                      ||
+===================================================================*/

/** The read method for this bag. */
void read() throws SQLException
    {
    if(bis == null) throw new SQLException(errs[1]);
    try {
        int amount;
        int rc;
        inside = fhead;
        capacity = HEADERSIZE;
        at = 0;
        readSome(4);
        maxAt = 4;
        at = 0;  
        amount = bToI(3);
        seq    = bToI(1);
        newCapacity(amount);
        readSome(amount);
        maxAt = amount + HEADERSIZE;
        if(dumpRead)dump("data read ...");
        at = 4;     
        }
    catch(Exception e) { errHandler(errs[12] + e); }
    }


/*-------------------------------------------------------------------+
|                     fully read a "packet"                          |
+-------------------------------------------------------------------*/

/** Method to fully read a specified number of bytes.
*** @param howmuch How many bytes to be read.
**/

private void readSome(int howmuch) throws SQLException
    {
    int left;
    int rc; 
    left = howmuch;
    try {
        while(left > 0)
            {
	    rc = bis.read(inside, at, left);
            if(rc < 0) throw new SQLException(errs[2]);
            at += rc;
            left -= rc;
            }
        }
    catch(IOException e) 
        { 
        if(connex != null) connex.broken = true;
        errHandlerD(errs[3] + e);
        }
    }


/*-------------------------------------------------------------------+
|                      enlarge capacity                              |
+-------------------------------------------------------------------*/

/** Change the capacity of the inside of the bag. */
private void newCapacity(int newSize) throws SQLException
    {
    try {
        byte[] nb;
        nb = new byte[newSize + HEADERSIZE];
        System.arraycopy(inside, 0, nb, 0, at);
        inside = nb;
        capacity = inside.length;
        }
    catch(Exception e) { errHandler(errs[11] + e); }
    }   


/*===================================================================+
||                 write a "packet" to the server                   ||
+===================================================================*/

/** Write this bag */
void write() throws SQLException
    {
    try {
        at = 0;
        iToB(maxAt - HEADERSIZE, 3);
        iToB(seq, 1);
        if(dumpWrite) dump("writing...");
        bos.write(inside, 0, maxAt); 
        bos.flush();
        }
    catch(Exception e)
        {
        if(connex != null) connex.broken = true;
        errHandlerD(errs[7] + e); 
        }

    }    


/*===================================================================+
||                     put a single byte in the bag                 ||
+===================================================================*/

/** Put a single byte into the bag.
*** @param b the  byte to be put.
**/
void putByte(byte b) throws SQLException
    {
    try {
        while(maxAt + 1 > capacity) 
               newCapacity(capacity + DEFAULTBUFFERSIZE);
        inside[at++] = b;
        if(at > maxAt)maxAt = at;
        }
    catch(Exception e) { errHandler(errs[20] + e); }
    }


/*===================================================================+
||                  put a an array of bytes in the bag              ||
+===================================================================*/

/** Put an  array of bytes into the bag.
*** @param b the array of bytes to be put.
*** @param s the size of the part of the array to put.
**/
void putByteArray(byte[] b, int s) throws SQLException
    {
    int min;
    if(s < 0) min = b.length;
    else min =  b.length < s ? b.length : s;
    if(min == 0) return;
    try {
        while(maxAt + min > capacity) 
               newCapacity(capacity + min + DEFAULTBUFFERSIZE);
        System.arraycopy(b, 0, inside, at, min);
        at += min;
        if(at > maxAt)maxAt = at;
        }
    catch(Exception e) { errHandler(errs[20] + e); }
    }


/*===================================================================+
||              put a an array of bytes in the bag with \0          ||
+===================================================================*/

/** Put out a null terminated array of bytes.
*** @param b the array of bytes to be put.
**/
void putZT(byte[] b) throws SQLException
    {
    try {
        while(maxAt + b.length + 1 > capacity) 
               newCapacity(capacity + b.length + DEFAULTBUFFERSIZE);
        System.arraycopy(b, 0, inside, at, b.length);
        at += b.length;
        inside[at++] = 0;
        if(at > maxAt)maxAt = at;
        }
    catch(Exception e) { errHandler(errs[9] + e); }
    }


/*===================================================================+
||                 put a string in the bag with \0                  ||
+===================================================================*/

/** Put out a null terminated string.  Does the platform
*** dependent (read unsafe) getbytes() on the parm and then
*** calls putZT(byte[]).
*** @param s The string to put in the database.
**/
void putZT(String s) throws SQLException
    {
    try {
        byte[] b = s.getBytes();
        putZT(b);
        }
    catch(Exception e) { errHandler(errs[8] + e); }
    }
 


/*===================================================================+
||                   Get a string of specified size.                ||
+===================================================================*/

String getNstring(int n) throws SQLException
    {
    try {
        byte[] b = getNbytes(n);
        return new String(b);
        }
    catch(Exception e) { errHandlerM(errs[19], e); }
    return null;
    }        

/*===================================================================+
||                   Get a specified number of bytes.               ||
+===================================================================*/

byte[] getNbytes(int n) throws SQLException
    {
    int t = at + n;
    if(t > maxAt) errHandlerD(errs[17]);
    try {
        byte[] b = new byte[n];
        System.arraycopy(inside, at, b, 0, n);
        at = t;
        return b;
        }
    catch(Exception e) { errHandlerM(errs[18], e); }
    return null;
    }

/*===================================================================+
||                 Get the rest of the bag as a byte[]              ||
+===================================================================*/

/** Get a byte array which is the rest of the bag */
byte[] getRbytes() throws SQLException
    {
    int d = maxAt - at;
    if(d < 1) return null;
    byte[] b = null;
    try { b = getNbytes(d); }
    catch(Exception e) { errHandlerM(errs[21], e); }
    return b;
    }


/*===================================================================+
||                 Get the rest of the bag as a String              ||
+===================================================================*/

/** Get a byte array which is the rest of the bag */
String getRstring() throws SQLException
    {
    int d = maxAt - at;
    if(d < 1) return null;
    String b = null;
    try { b = getNstring(d); }
    catch(Exception e) { errHandlerM(errs[22], e); }
    return b;
    }

/*===================================================================+
||                  Get a zt string from the bag.                   ||
+===================================================================*/

/** Get a string terminated by zero or up to the length of
*** the buffer.  Does a getZTbytes() then does the unsafe
*** conversion to string.
**/
String getZTstring() throws SQLException
    {
    try {
        byte b[] = getZTbytes();
        if(b == null) return "";
        return new String(b);
        }
    catch(Exception e) { errHandler(errs[10] + e); }
    return null;
    }


/*===================================================================+
||                  Get a zt byte array from the bag                ||
+===================================================================*/

/** Get a list of bytes which is zero terminated or up to
*** the length in the buffer.
**/
byte[] getZTbytes() throws SQLException
    {
    if(at >= maxAt) errHandlerD(errs[6]);
    try {
        int count = 0;
        boolean end = false;
        boolean maxxed = false;
        for(int i = at; !end; i++)
            {
            if(i >= maxAt) maxxed = end = true; 
            else if(inside[i] == 0) end = true;
            if(!end)count++;
            }
        if(count == 0)
            { 
            if(!maxxed)at++;
            return null;
            }
        byte[] rb = new byte[count];
        System.arraycopy(inside, at, rb, 0, count);
        at += count + 1;
        if(maxxed) at--;
        return rb;
        }
    catch(Exception e) { errHandler(errs[13] + e); }
    return null;
    }


/*-------------------------------------------------------------------+
|               Get a variable sized integer from the bag            |
+-------------------------------------------------------------------*/

int viToI()  throws SQLException // from net_field_length in libmysql.c
    {
    int j = bToI(1);
    return xnToI(j);
    }

int xToI() throws SQLException // Same as above except -1 for nulls.
    {
    int j = bToI(1);
    if(j == 251) return -1;
    return xnToI(j);
    }

private int xnToI(int j) throws SQLException
    {
    try {
        switch(j)
            {
            case 251: return 0;
            case 252: return bToI(2);
            case 253: return bToI(3);

	    
// Monty says! (I'm gonna crap out with a numeric if this goes over 2G!
            case 254: 
                long x;
                if(teeny)
                    {
                    x = bToL(4); 
                    if(at > maxAt) errHandlerD(errs[14]);
                    }
                else x = bToL(8);
                if(x > Integer.MAX_VALUE) errHandlerD(errs[23]);
		return (int)x;
              
            default: return j;
            }
        }
    catch(Exception e) { errHandlerM(errs[15], e); } 
    return j;
    }


/*-------------------------------------------------------------------+
|               Get a fixed sized integer from the bag               |
+-------------------------------------------------------------------*/

int nbToI()  throws SQLException 
    {
    try {
        int j = bToI(1);
        return bToI(j);
        }
    catch(Exception e) { errHandlerM(errs[16], e); }
    return 0;
    }

/*===================================================================+
||                            bToI                                  ||
+===================================================================*/

/** Return the following bytes as an integer 
*** @param size how many bytes to interpret as an integer.
*** @return the integer value.
**/
int bToI(int size) throws SQLException
    {
    int t, v, w;
    w = at + size - 1;
    v = 0;
    while(w >= at)
        { 
        t = inside[w];
        t &= 255;
        v <<= 8;
        v |= t;
        w--;
        }
    at += size;
    if(at > maxAt) throw new SQLException(errs[5]);
    return v;
    }

/*===================================================================+
||                       LONG bToI                                  ||
+===================================================================*/

  // Forrestal's Law : Being paranoid doesn't mean they're not
  //                   out to get you.

/** Return the following bytes as an integer 
*** @param size how many bytes to interpret as an integer.
*** @return the integer value.
**/
long bToL(int size) throws SQLException
    {
    int t, w;
    long v;
    w = at + size - 1;
    v = 0;
    while(w >= at)
        { 
        t = inside[w];
        t &= 255;;
        v <<= 8;
        v |= t;
        w--;
        }
    at += size;
    if(at > maxAt) throw new SQLException(errs[5]);
    return v;
    }


/*-------------------------------------------------------------------+
|               Get a variable sized long  from the bag              |
+-------------------------------------------------------------------*/

long viToL()  throws SQLException // from net_field_length in libmysql.c
    {
    int j = bToI(1);
    return xnToL(j);
    }

long xToI_L() throws SQLException // Same as above except -1 for nulls.
    {
    int j = bToI(1);
    if(j == 251) return -1L;
    return xnToL(j);
    }

private long xnToL(int j) throws SQLException
    {
    try {
        switch(j)
            {
            case 251: return 0L;
            case 252: return bToL(2);
            case 253: return bToL(3);
            case 254: 
                long x;
                if(teeny)
                    {
                    x = bToL(4); 
                    if(at > maxAt) errHandlerD(errs[14]);
                    }
                else x = bToL(8);
		return x;
            default: return (long)j;
            }
        }
    catch(Exception e) { errHandlerM(errs[15], e); } 
    return (long)j;
    }


/*===================================================================+
||                             iToB                                 ||
+===================================================================*/

/** Store an int into the bag for the specified length
*** @param value The value to store;
*** @size  the number of bytes to write.
**/
void iToB(int value, int size) throws SQLException
    {
    int v, s, t;
    t = at + size;
    if(t > capacity) newCapacity(t + DEFAULTBUFFERSIZE);
    v = value;
    s = size;
    while(s > 0)
       {
       t = v & 255;
       v >>= 8;
       if(t > 127)t -= 256;  
       inside[at++] = (byte)t;
       s--;
       }
    if(at > maxAt) maxAt = at;
    }


/*-------------------------------------------------------------------+
|                    Error handler with dump                         |
+-------------------------------------------------------------------*/

private void errHandlerM(String s, Exception e) throws SQLException
    {
    String o = s + "\n" + e.getMessage();
    errHandlerD(o);
    }

/*-------------------------------------------------------------------+
|                    Error handler with dump                         |
+-------------------------------------------------------------------*/

private void errHandlerD(String s) throws SQLException
    {
    dump(s);
    throw new SQLException(s);
    }


/*-------------------------------------------------------------------+
|                          Error handler                             |
+-------------------------------------------------------------------*/

private void errHandler(String s) throws SQLException
    {
    debugMessage(s);
    throw new SQLException(s);
    }


/*-------------------------------------------------------------------+
|                      Send a message to debug                       |
+-------------------------------------------------------------------*/

/** Debug Message
**/
void debugMessage(String s)
    {
    jdbcMysqlDebug DEBUG = jdbcMysqlBase.getDebug();
    if(DEBUG == null) return;
    DEBUG.put(s);
    DEBUG.flush();
    }


/*-------------------------------------------------------------------+
|                   dump the bag all over debug                      |
+-------------------------------------------------------------------*/

/** Dump the current bag to the debug output with a message
*** @param message preceeding message.
**/
void dump(String message)
    {
    jdbcMysqlDebug DEBUG = jdbcMysqlBase.getDebug();
    if(DEBUG == null) return;
    jdbcMysqlDumpBuffer db = new jdbcMysqlDumpBuffer();
    db.put(" ");
    db.put(message);
    db.put("Capacity " + capacity + "   maxAt " + maxAt + "   at " 
                       + at +  "   seq " + seq);  
    DEBUG.dump(inside, 0, maxAt, db);
    DEBUG.put(db);
    }

}



