%{
 #include <mysql.h>
 #include <stdarg.h>

 #include "httpd.h"
 #include "http_config.h"
 #include "http_core.h"
 #include "http_log.h"
 #include "http_protocol.h"

 #ifndef CR_SERVER_GONE_ERROR
 #define CR_SERVER_GONE_ERROR 	2006
 #endif


module MODULE_VAR_EXPORT mysql_include_module;


static  request_rec* glob_r = NULL;
static char* glob_mysql_include_tag = "mysql-include";
static char* glob_default_mysql_include = NULL; 
static char* glob_warn_failed_query = NULL; 

static char  glob_last_tag[64], glob_last_tag_param[64];
static server_rec* glob_s = NULL;
 
static void send_parsed_content(FILE *f, request_rec *r, void* m);
static void send_mysql_include(const char* query, const char* name, 
 const char* print_cols, const char* ttl_str);
static void run_query(const char* query);

static void print_var(const char* name);
static void set_var(const char* name, const char* value, int val_len, const char* ttl_str);
int my_strcpy(char * buf, int buf_size, const char* str);


typedef struct 
  {
    char* host;
    char* db;
    char* user;
    char* pass;
    char* default_include;
    char* warn_failed_query;
  }
 mysql_include_config_rec;

 // var_rec is for the linked list of module variables
 // they can be set or used in the mod_mysql_include tags
  
 struct var_rec;
 struct var_rec_node
  {
   struct var_rec *data;
  } ;

 struct var_rec
  {
    char* name;
    char* val;
    int val_len;
    time_t expires;
    struct var_rec_node next;
  } ;

static const struct var_rec* tmp_var_rec = NULL;


 // memory is allocated with malloc
 static struct var_rec* var_rec_init(const char* name, const char* val, int val_len, int ttl);
 
 // assumes v was allocated with var_rec_init
 // if the record with the same name is found in the linked list,
 // it will be freed with pfree and the new one will be inserted 
 // otherwise, the record is appended to the end of the list
 // returns 1 if the record was found, and 0 if it had to be inserted
 static int var_rec_set(struct var_rec* v);

 // returns the matching record, or NULL
 static const struct var_rec* var_rec_get(const char* name); 

 static MYSQL glob_dbh, *glob_dbh_ptr = NULL;
 static struct var_rec_node glob_mod_vars = {NULL};
  
 static int glob_bad_conn = 1;
 static table* glob_tag_params = NULL;
 static table* glob_vars = NULL;
 static char glob_last_tag_param_val[8192];
 static int glob_val_offset = 0;


 #ifdef MYSQL_INCLUDE_DEBUG
  #define DEBUG(args) debug_print args
  void debug_print (const char *fmt, ...)
   {
     va_list args;
     char buf[1024];
     va_start(args, fmt);
     
     vsnprintf(buf, sizeof(buf), fmt, args);
     ap_log_error(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, glob_s,buf);
   }
 #else
  #define DEBUG(m)
 #endif


%}

%x SCAN_TAG_NAME
%x SCAN_TAG_PARAMS
%x SCAN_TAG_PARAM_NAME
%x SCAN_TAG_PARAM_EQ
%x SCAN_TAG_PARAM_VAL
%x SCAN_TAG_PARAM_ESC
%x SCAN_TAG_PARAM_QUOTED
%x SCAN_VAR_NAME

TAG_NAME_SYM 	[a-zA-Z\-]
VAR_NAME_SYM 	[0-9a-zA-Z\-\.\[\]]
TAG_PARAM_SYM 	[a-zA-Z\-]
OPEN_TAG        "<+"
CLOSE_TAG 	"+>"

%%
<INITIAL>{OPEN_TAG} {
  BEGIN SCAN_TAG_NAME;
}

<INITIAL>.|\n {
 ap_rputs(yytext, glob_r );
}

<SCAN_TAG_NAME>{TAG_NAME_SYM}+ {
  strncpy(glob_last_tag, yytext, sizeof(glob_last_tag));
  ap_clear_table(glob_tag_params);
  BEGIN SCAN_TAG_PARAMS;
}

<SCAN_TAG_NAME>.|\n {
}

<SCAN_TAG_PARAMS>{TAG_PARAM_SYM}+ {
  strncpy(glob_last_tag_param, yytext, sizeof(glob_last_tag_param));
  ap_str_tolower(glob_last_tag_param);
  BEGIN SCAN_TAG_PARAM_EQ;
}

<SCAN_TAG_PARAMS>{CLOSE_TAG} {
  if(!strcasecmp(glob_last_tag, glob_mysql_include_tag))
   {
     send_mysql_include(ap_table_get(glob_tag_params, "query"),
       ap_table_get(glob_tag_params,"name"), 
       ap_table_get(glob_tag_params,"print-cols"), 
       ap_table_get(glob_tag_params,"ttl") 
      );
   }
  else if(!strcasecmp(glob_last_tag, "mysql-run"))
   {
     run_query(ap_table_get(glob_tag_params, "query"));
   }
  else if (!strcasecmp(glob_last_tag, "var-echo"))
   {
     print_var(ap_table_get(glob_tag_params, "name"));
   }
  else if (!strcasecmp(glob_last_tag, "var-set"))
   {
     set_var(ap_table_get(glob_tag_params, "name"),
      ap_table_get(glob_tag_params, "value"),
      0,
      ap_table_get(glob_tag_params, "ttl")
      );
   }

  BEGIN 0;
}

<SCAN_TAG_PARAMS>.|\n {
}

<SCAN_TAG_PARAM_EQ>"=" {
 BEGIN SCAN_TAG_PARAM_VAL;
}

<SCAN_TAG_PARAM_EQ>.|\n {
}

<SCAN_TAG_PARAM_VAL>"\"" {
  glob_val_offset = 0;
  BEGIN SCAN_TAG_PARAM_QUOTED;
}

<SCAN_TAG_PARAM_VAL>.|\n {
}

<SCAN_TAG_PARAM_QUOTED>"\\" {
  BEGIN SCAN_TAG_PARAM_ESC;
}

<SCAN_TAG_PARAM_QUOTED>"\"" {
  ap_table_set(glob_tag_params, glob_last_tag_param, glob_last_tag_param_val);
  BEGIN SCAN_TAG_PARAMS;
}

<SCAN_TAG_PARAM_QUOTED>"$" {
  BEGIN SCAN_VAR_NAME;
}

<SCAN_TAG_PARAM_QUOTED>.|\n {
  glob_val_offset += my_strcpy(glob_last_tag_param_val + glob_val_offset,
    sizeof(glob_last_tag_param_val) - glob_val_offset, yytext);
}

<SCAN_TAG_PARAM_ESC>"\""|"\\"|"$" {
  if(glob_val_offset < sizeof(glob_last_tag_param_val) - 1)
    {
      glob_last_tag_param_val[glob_val_offset++] = yytext[0];
      glob_last_tag_param_val[glob_val_offset] = 0;
    }

  BEGIN SCAN_TAG_PARAM_QUOTED;
}

<SCAN_VAR_NAME>{VAR_NAME_SYM}+ {
  tmp_var_rec = var_rec_get(yytext);
  if(tmp_var_rec)
    {
      if(tmp_var_rec->val && 
        tmp_var_rec->val_len  < sizeof(glob_last_tag_param_val) - glob_val_offset + 1)
      {
        memcpy(glob_last_tag_param_val + glob_val_offset,tmp_var_rec->val,
         tmp_var_rec->val_len);
        glob_val_offset += tmp_var_rec->val_len;
       *(glob_last_tag_param_val + glob_val_offset) = 0;
      }
    }

  
  BEGIN SCAN_TAG_PARAM_QUOTED;
}

<SCAN_VAR_NAME,SCAN_TAG_PARAMS,SCAN_TAG_PARAM_ESC>.|\n {
}

%%

int my_strcpy(char * buf, int buf_size, const char* str)
 {
   int i = 0;
   for(i = 0; i < buf_size - 1; i++)
    {
      if(!str[i]) break;
      buf[i] = str[i];
    }
    buf[i] = 0;
    return i;
 }


// var rec stuff begin
 static struct var_rec* var_rec_init(const char* name, const char* val, int val_len, int ttl)
  {
    int name_len;
    int record_len;
    struct var_rec* v;
    name_len = strlen(name);
    record_len = name_len + 1 + val_len + sizeof(struct var_rec);
    v = (struct var_rec*)malloc(record_len);
    if(!v) return NULL;

    v->name = (char*)v + sizeof(*v);
    memcpy(v->name, name, name_len);
    
    if(val)
     {
      v->val =  v->name + name_len + 1;
      v->name[name_len] = 0;
      memcpy(v->val, val, val_len);
      v->val_len = val_len;
     }
    else
     {
       v->val = NULL;
       v->val_len = 0;
     }

    v->expires = time(NULL) + ttl;
    v->next.data = NULL;

    DEBUG(("var_rec_init, v->name = %s", v->name));
    return v;
  }

 static int var_rec_set(struct var_rec* v)
  {
   struct var_rec* cur_v = glob_mod_vars.data;
   struct var_rec* last_v = NULL;
   while(cur_v)
    {
      if(!strcmp(cur_v->name, v->name))
       {
         DEBUG(("replacing %s", cur_v->name));
	 if(last_v)
	  last_v->next.data = v;
	 else
	  glob_mod_vars.data = v;

	 v->next.data = cur_v->next.data;
	 free(cur_v);
         return 1;
       }
      last_v = cur_v;
      cur_v = cur_v->next.data;
    }
  
    if(last_v)
     {
       last_v->next.data = v;
     }
    else
     {
      glob_mod_vars.data = v;
     }
  
    return 0;
  }

 static const struct var_rec* var_rec_get(const char* name)
  {
   struct var_rec* cur_v = glob_mod_vars.data;
   struct var_rec* last_v = NULL;
   while(cur_v)
    {
      DEBUG(("var_rec_get(): cur_v->name=%s name=%s", cur_v->name,name));
      if(!strcmp(cur_v->name, name))
       {
         return cur_v;
       }
     cur_v = cur_v->next.data;
    }
  
  
    return NULL;
  } 

// var_rec stuff end

static void db_connect(request_rec *r, void* m)
 {
   mysql_include_config_rec* cfg = (mysql_include_config_rec*)m;
   
   if(glob_dbh_ptr && glob_bad_conn)
     return;
  
   DEBUG(("calling mysql_connect"));

   glob_dbh_ptr = mysql_connect(&glob_dbh, cfg->host, cfg->user, cfg->pass);
   if(!glob_dbh_ptr)
    {
      ap_log_rerror(APLOG_MARK, APLOG_ERR|APLOG_NOERRNO, r, 
       "Error connecting to MySQL: %s", mysql_error(&glob_dbh));
      glob_bad_conn = 1;
      return;
    }
   
   DEBUG(("mysql_connect successful"));

   if(mysql_select_db(&glob_dbh, cfg->db))
    {
      ap_log_rerror(APLOG_MARK, APLOG_ERR|APLOG_NOERRNO, r, 
       "Error selecting database: %s", mysql_error(&glob_dbh));
      glob_bad_conn = 1;
      return;
    }
   
   DEBUG(("mysql_select_db successful"));

   glob_bad_conn = 0;
 }

static void mysql_include_child_exit(server_rec* s, pool* p)
 {
   if(glob_bad_conn) return;
   if(!glob_dbh_ptr) return;
   mysql_close(&glob_dbh);
 }

static void result_cleanup(void* result)
 {
   DEBUG(("result cleanup"));
   mysql_free_result((MYSQL_RES*) result);
 }

static MYSQL_RES *safe_store_result(pool *p)
{
  MYSQL_RES *result;
  if(glob_bad_conn) return NULL;

  result = mysql_store_result(&glob_dbh);

  if (result) 
  {
    ap_block_alarms();
    ap_register_cleanup(p, (void*)result, result_cleanup, result_cleanup);
    ap_unblock_alarms();
  }
  return result;
}


static int safe_query(request_rec *r, void* m, const char *query)
{
  int error = 1;
  int num_tries = 2; // in the future, make this configurable in httpd.conf
  int i;

  void (*sigpipe_handler)();

  #ifdef MYSQL_INCLUDE_DEBUG

  ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r, "running query %s", 
   query);
  
  #endif

  if(glob_bad_conn) return;

  
  for(i = 0; i < num_tries; i++)
   {
    sigpipe_handler = signal(SIGPIPE, SIG_IGN);
    error = mysql_query(&glob_dbh, query);
    if(error)
     if(mysql_errno(&glob_dbh) == CR_SERVER_GONE_ERROR)
      {
        glob_bad_conn = 1;
	ap_log_rerror(APLOG_MARK, APLOG_ERR|APLOG_NOERRNO, r, 
	 "MySQL server has gone away, trying to reconnect");
	db_connect(r, m);
	if(glob_bad_conn)
	  {
           ap_log_rerror(APLOG_MARK, APLOG_ERR|APLOG_NOERRNO, r, 
	    "could not re-establish connection to MySQL");
	   return error;
	  }
      }
      else
       {
         ap_log_rerror(APLOG_MARK, APLOG_ERR|APLOG_NOERRNO, r, 
	    "error running query: %s", mysql_error(&glob_dbh));
	 return error;
       }
     else
       return error;
   }

  return error;
}

static void mysql_include_init(server_rec* s, pool* p)
 {
   glob_s = s;
#if MODULE_MAGIC_NUMBER >= 19980527
    ap_add_version_component("mod_mysql_include/1.0");
#endif
 
 }

static void mysql_child_init(server_rec*s, pool* p)
 {
   glob_tag_params = ap_make_table(p, 16);
 }

static void* mysql_include_create_dir_config(pool *p, char* dirspec)
 {
  mysql_include_config_rec * cfg;
  cfg = (mysql_include_config_rec*)ap_pcalloc(p, 
   sizeof(mysql_include_config_rec));
  cfg->host = "localhost";
  cfg->user = "www";
  cfg->pass = "";
  cfg->db = "test";
  cfg->default_include = 
    "<center><A href=\"https://www.mysql.com/license.htmy\">\
<img border=0 src=\"http://www.mysql.com/images/powered_by_mysql.gif\" width=\"88\"\
height=\"46\" alt=\"Powered by MySQL\"></a></center>";
  glob_default_mysql_include = cfg->default_include;
  cfg->warn_failed_query = "<font color=red>mysql query failed -- details in \
   the error log</font>";
  glob_warn_failed_query = cfg->warn_failed_query;
  return (void*)cfg;
 } 

static void set_var(const char* name, const char* value, int val_len,
 const char* ttl_str)
 {
  struct var_rec* v;
  int ttl = 84600 * 1000; 
  //void (*sigpipe_handler)();
  
  DEBUG(("pid %d entered set_var()", getpid()));

  if(value)
   {
     if(!val_len)
      val_len = strlen(value);
   }
  else
   val_len = 0;

  if(ttl_str)
   ttl = atoi(ttl_str);
  ap_block_alarms();
  //sigpipe_handler = signal(SIGPIPE, SIG_IGN);

  v = var_rec_init(name, value, val_len, ttl);
  if(v)
   {
     var_rec_set(v);
   }
  else
   ap_log_rerror(APLOG_MARK, APLOG_CRIT|APLOG_NOERRNO, glob_r, 
    "Out of memory in var_rec_init()");
 
   ap_unblock_alarms();
  DEBUG(("pid %d exiting set_var()", getpid()));
 }

static void print_var(const char* name)
 {
   const struct var_rec* v;
   v = var_rec_get(name);
   if(!v)
    ap_rprintf(glob_r, "NULL");
   else
    ap_rwrite(v->val, v->val_len, glob_r);

 }

static void run_query(const char* query)
 {
  int error;
  void* m = ap_get_module_config(glob_r->per_dir_config, 
    &mysql_include_module);
 
  if(query)
    { 
      error = safe_query(glob_r, m, query);
      if(error)
       {
         ap_rputs(glob_warn_failed_query, glob_r);
       }
    }
 }

static void send_mysql_include(const char* query, const char* name,
  const char* print_cols, const char* ttl_str)
 {
   int error = 1;
   MYSQL_RES* res;
   MYSQL_ROW row;
   MYSQL_FIELD* fields;
   int num_fields = 0, row_i, col_i;
   char cur_field_name[1024];
   char print_col_mask[1024];
   register int i, tmp, c;
   register const char *p;
   //int ttl = 0; 

   unsigned long *lens = NULL;
   char* mysql_include = glob_default_mysql_include;
   void* m = ap_get_module_config(glob_r->per_dir_config, 
    &mysql_include_module);

   if(!query) goto fail;


   if(!glob_dbh_ptr)
     { 
      DEBUG(("estabislishing connection")) ;
      db_connect(glob_r, m);
     }

   if(!glob_bad_conn)
    {
      DEBUG(("last connection was good")) ;
      //if(ttl_str)
      // ttl = atoi(ttl_str);

      if(print_cols)
       {
         memset(print_col_mask, 0, sizeof(print_col_mask));
         p = print_cols;
	 tmp = 0;
	 
	 do {
	    if(c >= '0' && c <= '9')
	     tmp += c - '0' + tmp * 10;
	    else
	     {
	       if(tmp < sizeof(print_col_mask))
	         print_col_mask[tmp] = 1;
	       tmp = 0;
	     }

	    p++;
	  } while((c = *p));

       }
      else
        memset(print_col_mask, 1, sizeof(print_col_mask));

      error = safe_query(glob_r, m, query);
      if(error) goto fail;
      res = safe_store_result(glob_r->pool);
      if(!res) goto fail;
      num_fields = mysql_num_fields(res);
      if(num_fields < 1) goto fail;
      
      // there is something wrong with the guy who would select
      // that many fields, but a precaution never hurts :-)
      if(num_fields > sizeof(print_col_mask))
       num_fields = sizeof(print_col_mask);
      fields = mysql_fetch_fields(res);
      row_i = 0;

      while((row = mysql_fetch_row(res)))
       {
	if(name)
	  lens = (unsigned long*)mysql_fetch_lengths(res);

	for(col_i = 0; col_i < num_fields; col_i++)
         if(row[col_i]) 
	  {
	    if(print_col_mask[col_i]) 
	     ap_rputs(row[col_i], glob_r);
	    
	    if(name && lens && fields)
	      {
	        ap_snprintf(cur_field_name, sizeof(cur_field_name),
		  "%s.%s[%d]", name, fields[col_i].name, row_i);
		set_var(cur_field_name, row[col_i], lens[col_i], ttl_str);
	      }
	  }

	 row_i++;
       }

      return;
    }

fail:  
    ap_rputs(mysql_include, glob_r );
 }

int yywrap(void)
 {
   return 1;
 }
 
static void send_parsed_content(FILE *f, request_rec *r, void* m)
 {
  mysql_include_config_rec *cfg = (mysql_include_config_rec*)m;
  yy_switch_to_buffer(yy_create_buffer(f, YY_BUF_SIZE));
  yyout = stderr;
  glob_r = r;
  glob_default_mysql_include = cfg->default_include;
  glob_warn_failed_query = cfg->warn_failed_query;
  BEGIN 0;
  yylex();
  yy_delete_buffer(YY_CURRENT_BUFFER);
 }

 static int mysql_include_handler(request_rec* r)
  {
   const char* hostname;
   FILE *f;
   mysql_include_config_rec *cfg = ap_get_module_config(r->per_dir_config,
    &mysql_include_module);

    if (!(ap_allow_options(r) & OPT_INCLUDES)) {
        return DECLINED;
    }
    r->allowed |= (1 << M_GET);
    if (r->method_number != M_GET) {
        return DECLINED;
    }
    if (r->finfo.st_mode == 0) {
        ap_log_rerror(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r,
		    "File does not exist: %s",
                    (r->path_info
                     ? ap_pstrcat(r->pool, r->filename, r->path_info, NULL)
                     : r->filename));
        return HTTP_NOT_FOUND;
    }

    if (!(f = ap_pfopen(r->pool, r->filename, "r"))) {
        ap_log_rerror(APLOG_MARK, APLOG_ERR, r,
                    "file permissions deny server access: %s", r->filename);
        return HTTP_FORBIDDEN;
    }


   r->content_type = "text/html";
   ap_send_http_header(r);
  
   if (r->header_only) {
        ap_pfclose(r->pool, f);
        return OK;
    }

  
  
   send_parsed_content(f, r, cfg);

   
   ap_pfclose(r->pool, f);
   return OK;
  }

 static const command_rec mysql_include_cmds[] =
  {
    {
     "MySQLIncludeDefaultHTML",
     ap_set_string_slot,
     (void*)XtOffsetOf(mysql_include_config_rec, default_include),
     OR_ALL,
     TAKE1,
     "HTML to show if the mysql-include query fails"
    },
    {
     "MySQLIncludeWarnFailedQueryHTML",
     ap_set_string_slot,
     (void*)XtOffsetOf(mysql_include_config_rec, warn_failed_query),
     OR_ALL,
     TAKE1,
     "HTML to show if the mysql-run query fails"
    },
    {
     "MySQLIncludeHost",
     ap_set_string_slot,
     (void*)XtOffsetOf(mysql_include_config_rec, host),
     OR_ALL,
     TAKE1,
     "MySQL host"
    },
    {
     "MySQLIncludeDB",
     ap_set_string_slot,
     (void*)XtOffsetOf(mysql_include_config_rec, db),
     OR_ALL,
     TAKE1,
     "MySQL database"
    },
    {
     "MySQLIncludeUser",
     ap_set_string_slot,
     (void*)XtOffsetOf(mysql_include_config_rec, user),
     OR_ALL,
     TAKE1,
     "MySQL user"
    },
    {
     "MySQLIncludePass",
     ap_set_string_slot,
     (void*)XtOffsetOf(mysql_include_config_rec, pass),
     OR_ALL,
     TAKE1,
     "MySQL password"
    },
    {
     NULL
    }
  };

 static handler_rec mysql_include_handlers[] = 
  {
   {"mysql-include-parsed", mysql_include_handler},
   {NULL}
  };

 module MODULE_VAR_EXPORT mysql_include_module =
  {
   STANDARD_MODULE_STUFF,
   mysql_include_init, /* module initializer*/
   mysql_include_create_dir_config, /* per-directory config creator*/
   NULL, /* dir config merger */
   NULL, /* server config creator*/
   NULL, /* server config merger */
   mysql_include_cmds, /* command table */
   mysql_include_handlers, /* content handlers */
   NULL, /* URI-to-file name translation */
   NULL, /* check/validate user ID */
   NULL, /* check user_id valid here */
   NULL, /* check access by host address */
   NULL, /* MIME type checker/setter */
   NULL, /* fixups */
   NULL, /* logger */
   NULL, /* header parser */
   mysql_child_init, /* process init */
   mysql_include_child_exit, /* process exit/cleanup */
   NULL /* post read request handling */
  };
