/*
  +----------------------------------------------------------------------+
  | PHP Version 6                                                        |
  +----------------------------------------------------------------------+
  | Copyright (c) 1997-2007 The PHP Group                                |
  +----------------------------------------------------------------------+
  | This source file is subject to version 3.01 of the PHP license,      |
  | that is bundled with this package in the file LICENSE, and is        |
  | available through the world-wide-web at the following url:           |
  | http://www.php.net/license/3_01.txt                                  |
  | If you did not receive a copy of the PHP license and are unable to   |
  | obtain it through the world-wide-web, please send a note to          |
  | license@php.net so we can mail you a copy immediately.               |
  +----------------------------------------------------------------------+
  | Authors: Georg Richter <georg@mysql.com>                             |
  |          Andrey Hristov <andrey@mysql.com>                           |
  |          Ulf Wendel <uwendel@mysql.com>                              |
  +----------------------------------------------------------------------+
*/

/* $Id: header,v 1.17 2006/01/01 13:09:48 sniper Exp $ */
#include "php.h"
#include "mysqlnd.h"
#include "mysqlnd_priv.h"


#define MYSQLND_SILENT



MYSQLND_RES *mysqlnd_stmt_store_result(MYSQLND_STMT *stmt TSRMLS_DC);
MYSQLND_RES *mysqlnd_stmt_use_result(MYSQLND_STMT *stmt TSRMLS_DC);

enum_func_status mysqlnd_fetch_stmt_row_buffered(MYSQLND_RES *result, void *param,
												unsigned int flags,
												zend_bool *fetched_anything TSRMLS_DC);

enum_func_status mysqlnd_fetch_stmt_row_cursor(MYSQLND_RES *result, void *param,
											   unsigned int flags,
											   zend_bool *fetched_anything TSRMLS_DC);

void mysqlnd_stmt_separate_result_bind(MYSQLND_STMT *stmt);


typedef int8  my_int8;
typedef uint8  my_uint8;

typedef int16 my_int16;
typedef uint16 my_uint16;

typedef int32   my_int32;
typedef uint32 my_uint32;


enum mysqlnd_timestamp_type
{
  MYSQLND_TIMESTAMP_NONE= -2,
  MYSQLND_TIMESTAMP_ERROR= -1,
  MYSQLND_TIMESTAMP_DATE= 0,
  MYSQLND_TIMESTAMP_DATETIME= 1,
  MYSQLND_TIMESTAMP_TIME= 2
};


struct st_mysqlnd_time
{
  unsigned int  year, month, day, hour, minute, second;
  unsigned long second_part;
  zend_bool     neg;
  enum mysqlnd_timestamp_type time_type;
};



struct mysqlnd_perm_bind fetch_functions[MYSQL_TYPE_LAST + 1];

#define MYSQLND_PS_SKIP_RESULT_W_LEN	-1
#define MYSQLND_PS_SKIP_RESULT_STR		-2




static
void ps_store_string(char *value, unsigned int len, zend_uchar **row)
{
	char *to= (char *) php_mysqlnd_net_store_length(*row, len);
	memcpy(to, value, len);
	*row = (zend_uchar*) to + len;
}



static
void ps_fetch_null(zval *zv, const MYSQLND_FIELD * const field,
				   uint pack_len, zend_uchar **row TSRMLS_DC)
{
	ZVAL_NULL(zv);
}


static
void ps_fetch_int8(zval *zv, const MYSQLND_FIELD * const field,
				   uint pack_len, zend_uchar **row TSRMLS_DC)
{

	if (field->flags & UNSIGNED_FLAG) {
		ZVAL_LONG(zv, *(my_uint8*)*row);
	} else {
		php_printf("%.2X \n", *((int8*)*row));
		ZVAL_LONG(zv, *(my_int8*)*row);
	}
	(*row)++;
}


static
void ps_fetch_int16(zval *zv, const MYSQLND_FIELD * const field,
					uint pack_len, zend_uchar **row TSRMLS_DC)
{
	if (field->flags & UNSIGNED_FLAG) {
		ZVAL_LONG(zv, (my_uint16) sint2korr(*row));
	} else {
		ZVAL_LONG(zv, (my_int16) sint2korr(*row));	
	}
	(*row)+= 2;
}


static void ps_fetch_int32(zval *zv, const MYSQLND_FIELD * const field,
						  uint pack_len, zend_uchar **row TSRMLS_DC)
{
	if (field->flags & UNSIGNED_FLAG) {
		my_uint32 uval;

		/* unsigned int (11) */
		uval= (my_uint32) sint4korr(*row);
		if (uval > INT_MAX) {
			char *tmp, *p;
			int j=10;
			tmp= emalloc(11);
			php_printf("int32 converted to string %p\n", tmp);
			p= &tmp[9];
			do { 
				*p-- = (uval % 10) + 48;
				uval = uval / 10;							
			} while (--j > 0);
			tmp[10]= '\0';
			/* unsigned int > INT_MAX is 10 digits - ALWAYS */
			ZVAL_UTF8_STRINGL(zv, tmp, 10, 0);
			if (UG(unicode)) {
				efree(tmp);
			}
		} else {
			ZVAL_LONG(zv, uval);
		}
	} else {
		ZVAL_LONG(zv, (my_int32) sint4korr(*row));
	}
	(*row)+= 4;
}


static
void ps_fetch_int64(zval *zv, const MYSQLND_FIELD * const field,
					uint pack_len, zend_uchar **row TSRMLS_DC)
{
	my_uint64 llval = (my_uint64) sint8korr(*row);
	zend_bool uns = field->flags & UNSIGNED_FLAG? TRUE:FALSE;
	
#if SIZEOF_LONG==8  
	if (uns == TRUE && llval > 9223372036854775807L) {
#elif SIZEOF_LONG==4
	if ((uns == TRUE && llval > L64(2147483647)) || 
	    (uns == FALSE && (( L64(2147483647) < (my_int64) llval) ||
		(L64(-2147483648) > (my_int64) llval))))
	{
#endif
		char tmp[22];
		/* even though lval is declared as unsigned, the value
		 * may be negative. Therefor we cannot use MYSQLND_LLU_SPEC and must
		 * use MYSQLND_LL_SPEC.
		 */
		sprintf((char *)&tmp, uns == TRUE? MYSQLND_LLU_SPEC : MYSQLND_LL_SPEC, llval);
		ZVAL_UTF8_STRING(zv, tmp, ZSTR_DUPLICATE);
	} else {
		ZVAL_LONG(zv, llval);
	}
  	(*row)+= 8;
}


static
void ps_fetch_float(zval *zv, const MYSQLND_FIELD * const field,
					uint pack_len, zend_uchar **row TSRMLS_DC)
{
	float value;
	float4get(value, *row);
	ZVAL_DOUBLE(zv, value);
  	(*row)+= 4;
}


static
void ps_fetch_double(zval *zv, const MYSQLND_FIELD * const field,
					 uint pack_len, zend_uchar **row TSRMLS_DC)
{
	double value;
	float8get(value, *row);
	ZVAL_DOUBLE(zv, value);
	(*row)+= 8;
}



static
void ps_fetch_time(zval *zv, const MYSQLND_FIELD * const field,
					uint pack_len, zend_uchar **row TSRMLS_DC)
{
	struct st_mysqlnd_time t;
	unsigned int length; /* First byte encodes the length*/
	char *to;

	if ((length = php_mysqlnd_net_field_length(row))) {
		zend_uchar *to= *row;

		t.time_type = MYSQLND_TIMESTAMP_TIME;
		t.neg			= (zend_bool) to[0];

		t.day			= (unsigned long) sint4korr(to+1);
		t.hour			= (unsigned int) to[5];
		t.minute		= (unsigned int) to[6];
		t.second		= (unsigned int) to[7];
		t.second_part	= (length > 8) ? (unsigned long) sint4korr(to+8) : 0;
		t.year			= t.month= 0;
		if (t.day) {
			/* Convert days to hours at once */
			t.hour += t.day*24;
			t.day	= 0;
		}

		(*row) += length;
	} else {
		memset(&t, 0, sizeof(t));
		t.time_type = MYSQLND_TIMESTAMP_TIME;
 	}

	/*
	  QQ : How to make this unicode without copying two times the buffer -
	  Unicode equivalent of spprintf?
	*/
	length = spprintf(&to, 0, "%s%02u:%02u:%02u",
					 (t.neg ? "-" : ""), t.hour, t.minute, t.second);
	ZVAL_STRINGL(zv, to, length, 1);
	efree(to);
}


static
void ps_fetch_date(zval *zv, const MYSQLND_FIELD * const field,
					uint pack_len, zend_uchar **row TSRMLS_DC)
{
	struct st_mysqlnd_time t = {0};
	unsigned int length; /* First byte encodes the length*/
	char *to;

	if ((length = php_mysqlnd_net_field_length(row))) {
		zend_uchar *to= *row;

		t.time_type= MYSQLND_TIMESTAMP_DATE;
		t.neg= 0;

		t.second_part = t.hour = t.minute = t.second = 0;

		t.year	= (unsigned int) sint2korr(to);
		t.month = (unsigned int) to[2];
		t.day	= (unsigned int) to[3];

		(*row)+= length;
	} else {
		memset(&t, 0, sizeof(t));
		t.time_type = MYSQLND_TIMESTAMP_DATE;
	}

	/*
	  QQ : How to make this unicode without copying two times the buffer -
	  Unicode equivalent of spprintf?
	*/
	length = spprintf(&to, 0, "%04u-%02u-%02u", t.year, t.month, t.day);
	ZVAL_STRINGL(zv, to, length, 1);
	efree(to);
}


static
void ps_fetch_datetime(zval *zv, const MYSQLND_FIELD * const field,
						uint pack_len, zend_uchar **row TSRMLS_DC)
{
	struct st_mysqlnd_time t;
	unsigned int length; /* First byte encodes the length*/
	char *to;

	if ((length = php_mysqlnd_net_field_length(row))) {
		zend_uchar *to= *row;

		t.time_type = MYSQLND_TIMESTAMP_DATETIME;
		t.neg	 = 0;

		t.year	 = (unsigned int) sint2korr(to);
		t.month = (unsigned int) to[2];
		t.day	 = (unsigned int) to[3];

		if (length > 4) {
			t.hour	 = (unsigned int) to[4];
			t.minute = (unsigned int) to[5];
			t.second = (unsigned int) to[6];
		} else {
      		t.hour = t.minute = t.second= 0;
		}
		t.second_part = (length > 7) ? (unsigned long) sint4korr(to+7) : 0;

		(*row)+= length;
	} else {
		memset(&t, 0, sizeof(t));
		t.time_type = MYSQLND_TIMESTAMP_DATETIME;
	}

	/*
	  QQ : How to make this unicode without copying two times the buffer -
	  Unicode equivalent of spprintf?
	*/
	length = spprintf(&to, 0, "%04u-%02u-%02u %02u:%02u:%02u",
					  t.year, t.month, t.day, t.hour, t.minute, t.second);
	ZVAL_STRINGL(zv, to, length, 1);
	efree(to);
}


static
void ps_fetch_string(zval *zv, const MYSQLND_FIELD * const field,
					 uint pack_len, zend_uchar **row TSRMLS_DC)
{
	/*
	  For now just copy, before we make it possible
	  to write \0 to the row buffer
	*/
	unsigned long length= php_mysqlnd_net_field_length(row);
	ZVAL_STRINGL(zv, *row, length, 1);
	(*row) += length;
}



/* {{{ mysqlnd_init_ps */
/* Call this function at MINIT() */
void mysqlnd_init_ps()
{
	memset(fetch_functions, 0, sizeof(fetch_functions));
	fetch_functions[MYSQL_TYPE_NULL].func		= ps_fetch_null;
	fetch_functions[MYSQL_TYPE_NULL].pack_len 	= 0;
	fetch_functions[MYSQL_TYPE_NULL].php_type 	= IS_NULL;

	fetch_functions[MYSQL_TYPE_TINY].func		= ps_fetch_int8;
	fetch_functions[MYSQL_TYPE_TINY].pack_len	= 1;
	fetch_functions[MYSQL_TYPE_TINY].php_type	= IS_LONG;

	fetch_functions[MYSQL_TYPE_SHORT].func		= ps_fetch_int16;
	fetch_functions[MYSQL_TYPE_SHORT].pack_len	= 2;
	fetch_functions[MYSQL_TYPE_SHORT].php_type	= IS_LONG;

	fetch_functions[MYSQL_TYPE_YEAR].func		= ps_fetch_int16;
	fetch_functions[MYSQL_TYPE_YEAR].pack_len	= 2;
	fetch_functions[MYSQL_TYPE_YEAR].php_type 	= IS_LONG;

	fetch_functions[MYSQL_TYPE_INT24].func		= ps_fetch_int32;
	fetch_functions[MYSQL_TYPE_INT24].pack_len	= 4;
	fetch_functions[MYSQL_TYPE_INT24].php_type	= IS_LONG;

	fetch_functions[MYSQL_TYPE_LONG].func		= ps_fetch_int32;
	fetch_functions[MYSQL_TYPE_LONG].pack_len	= 4;
	fetch_functions[MYSQL_TYPE_LONG].php_type 	= IS_LONG;

	fetch_functions[MYSQL_TYPE_LONGLONG].func	= ps_fetch_int64;
	fetch_functions[MYSQL_TYPE_LONGLONG].pack_len= 8;
	fetch_functions[MYSQL_TYPE_LONGLONG].php_type = IS_LONG;

	fetch_functions[MYSQL_TYPE_FLOAT].func		= ps_fetch_float;
	fetch_functions[MYSQL_TYPE_FLOAT].pack_len	= 4;
	fetch_functions[MYSQL_TYPE_FLOAT].php_type	= IS_DOUBLE;

	fetch_functions[MYSQL_TYPE_DOUBLE].func		= ps_fetch_double;
	fetch_functions[MYSQL_TYPE_DOUBLE].pack_len	= 8;
	fetch_functions[MYSQL_TYPE_DOUBLE].php_type = IS_DOUBLE;
	
	fetch_functions[MYSQL_TYPE_TIME].func		= ps_fetch_time;
	fetch_functions[MYSQL_TYPE_TIME].pack_len	= MYSQLND_PS_SKIP_RESULT_W_LEN;
	fetch_functions[MYSQL_TYPE_TIME].php_type	= IS_STRING;

	fetch_functions[MYSQL_TYPE_DATE].func		= ps_fetch_date;
	fetch_functions[MYSQL_TYPE_DATE].pack_len	= MYSQLND_PS_SKIP_RESULT_W_LEN;
	fetch_functions[MYSQL_TYPE_DATE].php_type	= IS_STRING;
	
	fetch_functions[MYSQL_TYPE_DATETIME].func		= ps_fetch_datetime;
	fetch_functions[MYSQL_TYPE_DATETIME].pack_len	= MYSQLND_PS_SKIP_RESULT_W_LEN;
	fetch_functions[MYSQL_TYPE_DATETIME].php_type	= IS_STRING;

	fetch_functions[MYSQL_TYPE_TIMESTAMP].func		= ps_fetch_datetime;
	fetch_functions[MYSQL_TYPE_TIMESTAMP].pack_len	= MYSQLND_PS_SKIP_RESULT_W_LEN;
	fetch_functions[MYSQL_TYPE_TIMESTAMP].php_type	= IS_STRING;
	
	fetch_functions[MYSQL_TYPE_TINY_BLOB].func		= ps_fetch_string;
	fetch_functions[MYSQL_TYPE_TINY_BLOB].pack_len	= MYSQLND_PS_SKIP_RESULT_STR;
	fetch_functions[MYSQL_TYPE_TINY_BLOB].php_type = IS_STRING;

	fetch_functions[MYSQL_TYPE_BLOB].func			= ps_fetch_string;
	fetch_functions[MYSQL_TYPE_BLOB].pack_len		= MYSQLND_PS_SKIP_RESULT_STR;
	fetch_functions[MYSQL_TYPE_BLOB].php_type		= IS_STRING;
	
	fetch_functions[MYSQL_TYPE_MEDIUM_BLOB].func	= ps_fetch_string;
	fetch_functions[MYSQL_TYPE_MEDIUM_BLOB].pack_len= MYSQLND_PS_SKIP_RESULT_STR;
	fetch_functions[MYSQL_TYPE_MEDIUM_BLOB].php_type= IS_STRING;

	fetch_functions[MYSQL_TYPE_LONG_BLOB].func		= ps_fetch_string;
	fetch_functions[MYSQL_TYPE_LONG_BLOB].pack_len	= MYSQLND_PS_SKIP_RESULT_STR;
	fetch_functions[MYSQL_TYPE_LONG_BLOB].php_type	= IS_STRING;

	fetch_functions[MYSQL_TYPE_BIT].func			= ps_fetch_string;
	fetch_functions[MYSQL_TYPE_BIT].pack_len		= MYSQLND_PS_SKIP_RESULT_STR;
	fetch_functions[MYSQL_TYPE_BIT].php_type		= IS_STRING;

	fetch_functions[MYSQL_TYPE_VAR_STRING].func		= ps_fetch_string;
	fetch_functions[MYSQL_TYPE_VAR_STRING].pack_len	= MYSQLND_PS_SKIP_RESULT_STR;
	fetch_functions[MYSQL_TYPE_VAR_STRING].php_type = IS_STRING;

	fetch_functions[MYSQL_TYPE_STRING].func			= ps_fetch_string;
	fetch_functions[MYSQL_TYPE_STRING].pack_len		= MYSQLND_PS_SKIP_RESULT_STR;
	fetch_functions[MYSQL_TYPE_STRING].php_type 	= IS_STRING;

	fetch_functions[MYSQL_TYPE_DECIMAL].func		= ps_fetch_string;
	fetch_functions[MYSQL_TYPE_DECIMAL].pack_len	= MYSQLND_PS_SKIP_RESULT_STR;
	fetch_functions[MYSQL_TYPE_DECIMAL].php_type	= IS_STRING;

	fetch_functions[MYSQL_TYPE_NEWDECIMAL].func		= ps_fetch_string;
	fetch_functions[MYSQL_TYPE_NEWDECIMAL].pack_len	= MYSQLND_PS_SKIP_RESULT_STR;
	fetch_functions[MYSQL_TYPE_NEWDECIMAL].php_type	= IS_STRING;

	fetch_functions[MYSQL_TYPE_ENUM].func			= ps_fetch_string;
	fetch_functions[MYSQL_TYPE_ENUM].pack_len		= MYSQLND_PS_SKIP_RESULT_STR;
	fetch_functions[MYSQL_TYPE_ENUM].php_type		= IS_STRING;

	fetch_functions[MYSQL_TYPE_SET].func			= ps_fetch_string;
	fetch_functions[MYSQL_TYPE_SET].pack_len		= MYSQLND_PS_SKIP_RESULT_STR;
	fetch_functions[MYSQL_TYPE_SET].php_type		= IS_STRING;

	fetch_functions[MYSQL_TYPE_GEOMETRY].func		= ps_fetch_string;
	fetch_functions[MYSQL_TYPE_GEOMETRY].pack_len	= MYSQLND_PS_SKIP_RESULT_STR;
	fetch_functions[MYSQL_TYPE_GEOMETRY].php_type	= IS_STRING;
}
/* }}} */


/* {{{ mysqlnd_stmt_init */
MYSQLND_STMT *mysqlnd_stmt_init(MYSQLND *conn)
{
	MYSQLND_STMT *stmt = ecalloc(1, sizeof(MYSQLND_STMT));

	stmt->state = MYSQLND_STMT_INITTED;
	stmt->conn = conn;
	stmt->cmd_buffer.length = 4096;
	stmt->cmd_buffer.buffer = emalloc(stmt->cmd_buffer.length);
	/*
	  Mark that we reference the connection, thus it won't be
	  be destructed till there is open statements. The last statement
	  or normal query result will close it then.
	*/
	conn->references++;

	return stmt;
}
/* }}} */


/* {{{ mysqlnd_stmt_prepare */
enum_func_status
mysqlnd_stmt_prepare(MYSQLND_STMT *stmt, char *query, unsigned int query_len TSRMLS_DC)
{
	enum_func_status ret;
	MYSQLND_RES *result = NULL;
	php_mysql_packet_prepare_response prepare_resp;
	php_mysql_packet_eof fields_eof;

	if (stmt->state > MYSQLND_STMT_INITTED) {
		result = NULL;
		/* close previously alloced resources and prepare the new statement */
		/* This should free param_bind and result_bind and set to NULL !!! */
		/* CLEAN BUFFERS !!!!!!!!!!!!!!!!!!*/
	}

	if (FAIL == mysqlnd_simple_command(stmt->conn, COM_STMT_PREPARE, query, query_len,
									   PROT_LAST /* we will handle the OK packet*/,
									   FALSE TSRMLS_CC)) {
		return FAIL;		
	}

	PACKET_INIT_ALLOCA(prepare_resp, PROT_PREPARE_RESP_PACKET);
	if (FAIL == PACKET_READ_ALLOCA(prepare_resp, stmt->conn)) {
		PACKET_FREE_ALLOCA(prepare_resp);
		return FAIL;
	}

	if (0xFF == prepare_resp.error_code) {
		stmt->error_info = stmt->conn->error_info = prepare_resp.error_info;
		return FAIL;
	}

	stmt->stmt_id = prepare_resp.stmt_id;
	stmt->warning_count = stmt->conn->upsert_status.warning_count = prepare_resp.warning_count;
	stmt->field_count = stmt->conn->field_count = prepare_resp.field_count;
	stmt->param_count = prepare_resp.param_count;
	PACKET_FREE_ALLOCA(prepare_resp);

	if (stmt->param_count) {
		/* Follows parameter metadata, we have just to skip it, as libmysql does */
		unsigned int i = 0;
		php_mysql_packet_res_field field_packet;

		PACKET_INIT_ALLOCA(field_packet, PROT_RSET_FLD_PACKET);
		field_packet.skip_parsing = TRUE;
		for (;i < stmt->param_count; i++) {
			if (FAIL == PACKET_READ_ALLOCA(field_packet, stmt->conn)) {
				PACKET_FREE_ALLOCA(field_packet);
				return FAIL;
			}
		}
		PACKET_FREE_ALLOCA(field_packet);
	}

	/* Read metadata only if there is actual result set */
	if (stmt->field_count) {
		/* Because results reference the connection */
		stmt->conn->references++;
		/* Allocate the result now as it is needed for the reading of metadata */
		result = ecalloc(1, sizeof(MYSQLND_RES));
		result->type = MYSQLND_RES_PS;

		if (FAIL == mysqlnd_read_result_metadata(stmt->conn, result TSRMLS_CC)) {
			/* Hence if this fails, our statement is in bad shape. */
			efree(result);
			memset(stmt, 0, sizeof(MYSQLND_STMT));
			stmt->state = MYSQLND_STMT_INITTED;
			return FAIL;
		}
		result->conn = stmt->conn;

		stmt->result = result;
	}
	ret = PASS;
	PACKET_INIT_ALLOCA(fields_eof, PROT_EOF_PACKET);
	if (FAIL == PACKET_READ_ALLOCA(fields_eof, stmt->conn)) {
		if (stmt->result) {
			mysqlnd_internal_free_result_contents(stmt->result);
			efree(stmt->result);
		}
		ret= FAIL;
	} else {
		stmt->upsert_status.server_status = fields_eof.server_status;
		stmt->upsert_status.warning_count = fields_eof.warning_count;
		stmt->state = MYSQLND_STMT_PREPARED;
	}
	PACKET_FREE_ALLOCA(fields_eof);

	return PASS;
}
/* }}} */


static enum_func_status
mysqlnd_stmt_bind_set_original_type(MYSQLND_STMT *stmt)
{
	unsigned int i;
	if (!stmt->result) {
		return FAIL;
	}
	if (stmt->field_count) {
		if (!stmt->result_bind) {
			stmt->result_bind = ecalloc(stmt->field_count, sizeof(MYSQLND_RESULT_BIND));
		}
		for (i = 0; i < stmt->field_count; i++) {
			stmt->result_bind[i].original_type = fetch_functions[stmt->result->fields[i].type].php_type;
		}
	}
	return PASS;
}




static void
mysqlnd_stmt_execute_store_params(MYSQLND_STMT *stmt, zend_uchar **buf, zend_uchar **p,
								  unsigned int *buf_len, unsigned int null_byte_offset)
{
	unsigned int i = 0;
	unsigned left = (*buf_len - (*p - *buf));
	unsigned int data_size = 0;

/* 1. Store type information */
	if (stmt->send_types_to_server) {

		/* 2 bytes per type, and leave 20 bytes for future use */
		if (left < ((stmt->param_count * 2) + 20)) {
			unsigned int offset = p - buf;
			*buf_len = offset + stmt->param_count * 2 + 20;
			*buf = erealloc(*buf, *buf_len);
			/* Update our pos pointer */
			*p = *buf + offset;
		}
		for (i = 0; i < stmt->param_count; i++) {
			/* our types are not unsigned */
#if SIZEOF_LONG==8  
			if (stmt->param_bind[i].type == MYSQL_TYPE_LONG) {
				stmt->param_bind[i].type = MYSQL_TYPE_LONGLONG;
			}
#endif
			int2store(*p, stmt->param_bind[i].type);
			*p+= 2;
		}
	}

/* 2. Store data */
	/* 2.1 Calculate how much space we need */
	for (i = 0; i < stmt->param_count; i++) {
		if (Z_TYPE_P(stmt->param_bind[i].zv) == IS_NULL) {
			continue;
		}

		switch (stmt->param_bind[i].type) {
			case MYSQL_TYPE_DOUBLE:
				data_size += 8;
				break;
#if SIZEOF_LONG==8  
			case MYSQL_TYPE_LONGLONG:
				data_size += 8;
				break;
#elif SIZEOF_LONG==4
			case MYSQL_TYPE_LONG:
				data_size += 4;
				break;
#else
#error "Should not happen"
#endif
			case MYSQL_TYPE_BLOB:
				data_size += 0;
				break;
			case MYSQL_TYPE_VAR_STRING:
				data_size += 8; /* max 8 bytes for size */
				convert_to_string_ex(&stmt->param_bind[i].zv);
				data_size += Z_STRLEN_P(stmt->param_bind[i].zv);
				break;
		}

	}

	/* 2.2 Enlarge the buffer, if needed */
	left = (*buf_len - (*p - *buf));
	if (left < data_size) {
		unsigned int offset = p - buf;
		*buf_len = offset + data_size + 10; /* Allocate + 10 for safety */
		*buf = erealloc(*buf, *buf_len);
		/* Update our pos pointer */
		*p = *buf + offset;	
	}

	/* 2.3 Store the actual data */
	for (i = 0; i < stmt->param_count; i++) {
		/* Handle long data */
		if (Z_TYPE_P(stmt->param_bind[i].zv) == IS_NULL) {
			(*buf + null_byte_offset)[i/8] |= (zend_uchar) (1 << (i & 7));
		} else {
			zval *data = stmt->param_bind[i].zv;
			switch (stmt->param_bind[i].type) {
				case MYSQL_TYPE_DOUBLE:
					convert_to_double_ex(&data);
					float8store(*p, Z_DVAL_P(data));
					(*p) += 8;
					break;
#if SIZEOF_LONG==8  
				case MYSQL_TYPE_LONGLONG:
					convert_to_long_ex(&data);
					int8store(*p, Z_LVAL_P(data));
					(*p) += 8;
					break;
#elif SIZEOF_LONG==4
				case MYSQL_TYPE_LONG:
					convert_to_long_ex(&data);
					int4store(*p, Z_LVAL_P(data));
					(*p) += 4;
					break;
#else
#error "Should not happen"
#endif
				case MYSQL_TYPE_BLOB:
					break;
				case MYSQL_TYPE_VAR_STRING:
					/* We have already converted it to string */
					{
						unsigned int len = Z_STRLEN_P(data);
						/* to is after p. The latter hasn't been moved */
						*p = php_mysqlnd_net_store_length(*p, len);
						memcpy(p, Z_STRVAL_P(data), len);
						(*p) += len;
					}

					ps_store_string(Z_STRVAL_P(data), Z_STRLEN_P(data), buf);
					break;
				default:
					/* Won't happen, but set to NULL */
					(*buf + null_byte_offset)[i/8] |= (zend_uchar) (1 << (i & 7));
					break;
			}
		}
	}
}


/* {{{ mysqlnd_stmt_execute */
enum_func_status
mysqlnd_stmt_execute(MYSQLND_STMT *stmt TSRMLS_DC)
{
	zend_uchar *p = stmt->cmd_buffer.buffer;
	unsigned int null_byte_offset;
	unsigned int null_count= (stmt->param_count+7) /8;
	php_mysql_packet_eof fields_eof;
	php_mysql_packet_rset_header rset_header;
	enum_func_status ret;
	MYSQLND *conn = stmt->conn;

	if (stmt->state > MYSQLND_STMT_PREPARED) {
		/*
		  Re-execute the statement
		  
		  CLEAN BUFFERS!!!!
		*/
	} else if (stmt->state < MYSQLND_STMT_PREPARED) {
		/* Only initted - error */
		SET_CLIENT_ERROR(conn->error_info, CR_COMMANDS_OUT_OF_SYNC, UNKNOWN_SQLSTATE,
						 "Commands out of sync; you can't run this command now");
		SET_STMT_ERROR(stmt, CR_COMMANDS_OUT_OF_SYNC, UNKNOWN_SQLSTATE,
						"Commands out of sync; you can't run this command now");
		return FAIL;
	}

	if (stmt->param_count && !stmt->param_bind) {
		SET_STMT_ERROR(stmt, CR_PARAMS_NOT_BOUND, UNKNOWN_SQLSTATE,
						 "No data supplied for parameters in prepared statement");
		return FAIL;
	}

	int4store(p, stmt->stmt_id);
	p += 4;

	int1store(p, stmt->flags);
	p++;

	int1store(p, 1); /* iteration count */
	p++;

	null_byte_offset = p - stmt->cmd_buffer.buffer;
	memset(p, 1, null_count); /* send always as NULL for now */
	p += null_count;


	int1store(p, stmt->send_types_to_server); 
	p++;
	mysqlnd_stmt_execute_store_params(stmt, &stmt->cmd_buffer.buffer, &p,
									  &stmt->cmd_buffer.length, null_byte_offset);

	
	/* support for buffer types should be added here ! */

	if (FAIL == mysqlnd_simple_command(stmt->conn, COM_STMT_EXECUTE, stmt->cmd_buffer.buffer,
									   p - stmt->cmd_buffer.buffer,
									   PROT_LAST /* we will handle the response packet*/,
									   FALSE TSRMLS_CC)) {
		stmt->upsert_status = stmt->conn->upsert_status;
		return FAIL;		
	}

	ret = FAIL;
	PACKET_INIT_ALLOCA(rset_header, PROT_RSET_HEADER_PACKET);
	do {
		if (FAIL == (ret = PACKET_READ_ALLOCA(rset_header, conn))) {
			php_error_docref(NULL TSRMLS_CC, E_WARNING, "Error reading result set's header");
			break;
		}

		if (rset_header.error_info.error_no) {
			stmt->error_info = conn->error_info = rset_header.error_info;
			stmt->upsert_status.affected_rows =
					conn->upsert_status.affected_rows = -1;
			ret = FAIL;
			break;
		}
		conn->error_info.error_no = 0;

		switch (rset_header.field_count) {
			case MYSQLND_NULL_LENGTH:	/* LOAD DATA LOCAL INFILE */
				conn->last_query_type = QUERY_LOAD_LOCAL;
				php_error_docref(NULL TSRMLS_CC, E_WARNING, "LOAD DATA LOCAL still not supported");
				ret = FAIL;
				break;
			case 0:				/* UPSERT		*/
				conn->last_query_type = QUERY_UPSERT;
				conn->state = CONN_READY;
				conn->field_count = rset_header.field_count;
				conn->upsert_status.warning_count = rset_header.warning_count;
				conn->upsert_status.server_status = rset_header.server_status;
				conn->upsert_status.affected_rows = rset_header.affected_rows;
				conn->upsert_status.last_insert_id = rset_header.last_insert_id;
				SET_NEW_MESSAGE(conn->last_message, conn->last_message_len,
								rset_header.info_or_local_file, rset_header.info_or_local_file_len);
				stmt->upsert_status = conn->upsert_status;
				stmt->state = MYSQLND_STMT_EXECUTED;

				PACKET_FREE_ALLOCA(rset_header);
				return PASS;
			default:{			/* Result set	*/
				memset(&conn->upsert_status, 0, sizeof(conn->upsert_status));
				conn->last_query_type = QUERY_SELECT;
				stmt->result->field_count = conn->field_count = rset_header.field_count;
				conn->state = CONN_QUERY_SENT;

				stmt->result->lengths = emalloc(stmt->result->field_count * sizeof(unsigned long));

				ret = PASS;
				break;
			}
		}
	} while (0);
	PACKET_FREE_ALLOCA(rset_header);
	if (ret == FAIL) {
		return FAIL;
	}

	stmt->send_types_to_server = 0;
	/* !! Here we are only there is a result set !! */

    if (stmt->field_count == 0) {
		return FAIL;
		/*
		  This is 'SHOW'/'EXPLAIN'-like query. Current implementation of
		  prepared statements can't send result set metadata for these queries
		  on prepare stage. Read it now.
		*/
    } else {
		/*
		  Update result set metadata if it for some reason changed between
		  prepare and execute, i.e.:
		  - in case of 'SELECT ?' we don't know column type unless data was
		    supplied to mysql_stmt_execute, so updated column type is sent
			now.
		  - if data dictionary changed between prepare and execute, for
			example a table used in the query was altered.
		  Note, that now (4.1.3) we always send metadata in reply to
		  COM_STMT_EXECUTE (even if it is not necessary), so either this or
		  previous branch always works.
		*/
		if (FAIL == mysqlnd_read_result_metadata(stmt->conn, stmt->result TSRMLS_CC)) {
			/*
			  mysqlnd_read_result_metadata() will clean stmt->result
			  Therefore we need just to free it's memory. This MEANS that we can have
			  empty 
			*/
			memset(stmt, 0, sizeof(MYSQLND_STMT));
			stmt->state = MYSQLND_STMT_INITTED;
			return FAIL;
		}
		mysqlnd_stmt_bind_set_original_type(stmt);
    }
	stmt->state = MYSQLND_STMT_EXECUTED;

	/* Check for
	  - SERVER_STATUS_MORE_RESULTS
	  - SERVER_STATUS_LAST_ROW_SENT (cursors)
	  - SERVER_STATUS_CURSOR_EXISTS (cursors)
	  if needed
	*/
	PACKET_INIT_ALLOCA(fields_eof, PROT_EOF_PACKET);
	if (FAIL == PACKET_READ_ALLOCA(fields_eof, conn)) {
		mysqlnd_internal_free_result_contents(stmt->result);
		efree(stmt->result);
		PACKET_FREE_ALLOCA(fields_eof);
		return FAIL;
	}
	stmt->upsert_status.server_status = fields_eof.server_status;
	stmt->upsert_status.warning_count = fields_eof.warning_count;
	PACKET_FREE_ALLOCA(fields_eof);

	if (stmt->field_count) {
		if (stmt->upsert_status.server_status & SERVER_STATUS_CURSOR_EXISTS) {
      		conn->state = CONN_READY;
//			stmt->read_row_func= stmt_read_row_from_cursor;
    	} else if (stmt->flags & CURSOR_TYPE_READ_ONLY) {
			/*
			  This is a single-row result set, a result set with no rows, EXPLAIN,
			  SHOW VARIABLES, or some other command which either a) bypasses the
			  cursors framework in the server and writes rows directly to the
			  network or b) is more efficient if all (few) result set rows are
			  precached on client and server's resources are freed.
			*/
      		mysqlnd_stmt_store_result(stmt TSRMLS_CC);
      		conn->state = CONN_READY;
		} else {
			/* unbuffered read */
			conn->state = CONN_QUERY_SENT;
      		mysqlnd_stmt_use_result(stmt TSRMLS_CC);
      		conn->state = CONN_READY;
		}
	}

	return ret;
}
/* }}} */


/* {{{ mysqlnd_fetch_lengths_stmt */
static
unsigned long * mysqlnd_fetch_lengths_stmt(MYSQLND_RES * result)
{
	/* Non-sense */
	return NULL;
}
/* }}} */


/* {{{ mysqlnd_fetch_stmt_row_buffered */
enum_func_status
mysqlnd_fetch_stmt_row_buffered(MYSQLND_RES *result, void *param, unsigned int flags,
								zend_bool *fetched_anything TSRMLS_DC)
{
	unsigned int i;
	MYSQLND_STMT *stmt = (MYSQLND_STMT *) param;

	if (stmt->state == MYSQLND_STMT_FETCHED) {
		/* Execute only once. We have to free the previous contents of user's bound vars */
		for (i = 0; i < result->field_count; i++) {
			if (stmt->result_bind[i].bound == TRUE) {
				zval_dtor(stmt->result_bind[i].zv);
				ZVAL_NULL(stmt->result_bind[i].zv);
			}
		}
		stmt->state = MYSQLND_STMT_USER_FETCHING;
	}

	/* If we haven't read everything */
	if (result->data_cursor && (result->data_cursor - result->data) < result->row_count) {
		zval **current_row = *result->data_cursor;
		for (i = 0; i < result->field_count; i++) {
			/* copy the type */
			if (stmt->result_bind[i].bound == TRUE) {
				if (Z_TYPE_P(current_row[i]) != IS_NULL) {
					/*
					  Copy the value.
					  Pre-condition is that the zvals in the result_bind buffer
					  have been  ZVAL_NULL()-ed or to another simple type
					  (int, double, bool but not string). Because of the reference
					  counting the user can't delete the strings the variables point to.
					*/

					Z_TYPE_P(stmt->result_bind[i].zv) = Z_TYPE_P(current_row[i]);
					stmt->result_bind[i].zv->value = current_row[i]->value;
				} else {
					ZVAL_NULL(stmt->result_bind[i].zv);
				}
			}
		}
		result->data_cursor++;
		*fetched_anything = TRUE;
	} else {
		result->data_cursor = NULL;
		*fetched_anything = FALSE;
#ifndef MYSQLND_SILENT
		php_printf("NO MORE DATA\n ");
#endif
	}
	return PASS;
}
/* }}} */


/* {{{ mysqlnd_store_result */
MYSQLND_RES *mysqlnd_stmt_store_result(MYSQLND_STMT *stmt TSRMLS_DC)
{
	enum_func_status ret;
	MYSQLND *conn = stmt->conn;
	int next_extend = 32, free_rows;
	php_mysql_packet_row row_packet;
	MYSQLND_RES *result;

	/* Nothing to store for UPSERT/LOAD DATA*/
	if (conn->last_query_type != QUERY_SELECT) {
		return NULL;
	}

	if (conn->state != CONN_QUERY_SENT) {
		SET_CLIENT_ERROR(conn->error_info, CR_COMMANDS_OUT_OF_SYNC, UNKNOWN_SQLSTATE,
						 "Commands out of sync; you can't run this command now"); 
		return NULL;
	}

	result = stmt->result;
	result->fetch_row		= mysqlnd_fetch_stmt_row_buffered;
	result->fetch_lengths	= mysqlnd_fetch_lengths_stmt;
//	result->zval_cache		= mysqlnd_alloc_get_cache_reference(conn->zval_cache);
	result->zval_cache		= NULL;

	conn->state = CONN_FETCHING_DATA;

	/* Create room for 'next_extend' rows */

	result->conn->references--; /* increased in prepare */
	result->conn		= NULL;	/* store result does not reference  the connection */
	result->data 		= emalloc(next_extend * sizeof(zval **));
	result->row_buffers = emalloc(next_extend * sizeof(zend_uchar *));

	free_rows = next_extend;

	PACKET_INIT_ALLOCA(row_packet, PROT_ROW_PACKET);
	row_packet.field_count = stmt->field_count;
	row_packet.binary_protocol = TRUE;
	row_packet.fields_metadata = stmt->result->fields;
	/* Let the row packet fill our buffer and skip additional malloc + memcpy */
	while (FAIL != (ret = PACKET_READ_ALLOCA(row_packet, conn)) && !row_packet.eof) {
		int i;
		zval **current_row;

		if (!free_rows) {
			unsigned long total_rows = free_rows = next_extend = next_extend * 5 / 3; /* extend with 33% */
			total_rows += result->row_count;
			result->data = erealloc(result->data, total_rows * sizeof(zval **));
			result->row_buffers = erealloc(result->row_buffers, total_rows * sizeof(zend_uchar *));
		}
		free_rows--;
		current_row = result->data[result->row_count] = row_packet.fields;
		result->row_buffers[result->row_count] = row_packet.row_buffer;
		result->row_count++;
		
		/* So row_packet's destructor function won't efree() it */
		row_packet.fields = NULL;
		row_packet.row_buffer = NULL;

		for (i = 0; i < row_packet.field_count; i++) {
			if (Z_TYPE_P(current_row[i]) >= IS_STRING) {
				unsigned long len = Z_STRLEN_P(current_row[i]);
				if (result->fields[i].max_length < len) {
					result->fields[i].max_length = len;
				}			
			}
		}
		/*
		  No need to FREE_ALLOCA as we can reuse the
		  'lengths' and 'fields' arrays. For lengths its absolutely safe.
		  'fields' is reused because the ownership of the strings has been
		  transfered above. 
		*/
	}
	/* Finally clean */
	if (row_packet.eof) { 
		conn->upsert_status.warning_count = row_packet.warning_count;
		conn->upsert_status.server_status = row_packet.server_status;
	}
	PACKET_FREE_ALLOCA(row_packet);
	/* We can realloc here to save some memory, if free_rows > 0 ?*/

	if (conn->upsert_status.server_status & SERVER_MORE_RESULTS_EXISTS) {
		conn->state = CONN_NEXT_RESULT_PENDING;
	} else {
		conn->state = CONN_READY;
	}

	if (PASS == ret) {
		/* Position at the first row */
		result->data_cursor = result->data;
		stmt->state = MYSQLND_STMT_FETCHED;
		/* No multithreading issues as we don't share the connection :) */
	} else {
		mysqlnd_internal_free_result_contents(stmt->result);
		efree(stmt->result);
		stmt->result = NULL;
		php_error_docref(NULL TSRMLS_CC, E_WARNING, "Pretty serious error");
		stmt->state = MYSQLND_STMT_PREPARED;
	}
	return result;
}
/* }}} */


/* {{{ mysqlnd_stmt_use_result */
MYSQLND_RES *mysqlnd_stmt_use_result(MYSQLND_STMT *stmt TSRMLS_DC)
{
	return NULL;
}
/* }}} */


#define STMT_ID_LENGTH 4

/* {{{ mysqlnd_fetch_row_cursor */
enum_func_status
mysqlnd_fetch_stmt_row_cursor(MYSQLND_RES *result, void *param, unsigned int flags,
							  zend_bool *fetched_anything TSRMLS_DC)
{
	MYSQLND_STMT *stmt = (MYSQLND_STMT *) param;
    char buf[STMT_ID_LENGTH /* statement id */ + 4 /* number of rows to fetch */];

	if (stmt->state < MYSQLND_STMT_EXECUTED) {
		/* Only initted - error */
		SET_CLIENT_ERROR(stmt->conn->error_info, CR_COMMANDS_OUT_OF_SYNC, UNKNOWN_SQLSTATE,
						 "Commands out of sync; you can't run this command now");
		return FAIL;
	}

	int4store(buf, stmt->stmt_id);
	int4store(buf + STMT_ID_LENGTH, 1); /* for now fetch only one row */
	if (FAIL == mysqlnd_simple_command(stmt->conn, COM_STMT_FETCH, buf, sizeof(buf),
									   PROT_LAST /* we will handle the response packet*/,
									   FALSE TSRMLS_CC)) {
		return FAIL;		
	}

	{
		php_mysql_packet_row row_packet;
		PACKET_INIT_ALLOCA(row_packet, PROT_ROW_PACKET);
		row_packet.field_count = stmt->field_count;
		row_packet.binary_protocol = TRUE;
		row_packet.fields_metadata = stmt->result->fields;
		if (FAIL == PACKET_READ_ALLOCA(row_packet, stmt->conn)) {
			PACKET_FREE_ALLOCA(row_packet);
			return FAIL;
		} else if (!row_packet.eof) {
			unsigned int i;
			MYSQLND_RES *result = stmt->result;
			zval **current_row = row_packet.fields;

			mysqlnd_unbuffered_free_last_data(result);

			result->last_row_data = row_packet.fields;
			result->last_row_buffer = row_packet.row_buffer;
			row_packet.fields = NULL;
			row_packet.row_buffer = NULL;
	
			*fetched_anything = TRUE;
			for (i = 0; i < result->field_count; i++) {
				/* copy the type */
				if (stmt->result_bind[i].bound == TRUE &&
					(IS_NULL != (Z_TYPE_P(stmt->result_bind[i].zv) = Z_TYPE_P(current_row[i])))) {
					/*
					  Copy the value.
					  Pre-condition is that the zvals in the result_bind buffer
					  have been  ZVAL_NULL()-ed or to another simple type
					  (int, double, bool but not string). Because of the reference
					  counting the user can't delete the strings the variables point to.
					*/

					stmt->result_bind[i].zv->value = current_row[i]->value;
				}
			}
		} else {
			*fetched_anything = FALSE;

			stmt->upsert_status.warning_count =
				stmt->conn->upsert_status.warning_count =
					row_packet.warning_count;

			stmt->upsert_status.server_status = 
				stmt->conn->upsert_status.server_status =
					row_packet.server_status;
		}
		
		PACKET_FREE_ALLOCA(row_packet);
	}

	return PASS;
}
/* }}} */


/* {{{ mysqlnd_stmt_fetch */
enum_func_status
mysqlnd_stmt_fetch(MYSQLND_STMT *stmt, zend_bool *fetched_anything TSRMLS_DC)
{
	return !stmt->result? FAIL:stmt->result->fetch_row(stmt->result, (void*)stmt, 0,
													   fetched_anything TSRMLS_CC);

}
/* }}} */


/* {{{ mysqlnd_stmt_close */
enum_func_status
mysqlnd_stmt_close(MYSQLND_STMT *stmt TSRMLS_DC)
{
	mysqlnd_internal_free_stmt_content(stmt TSRMLS_CC);
	/* Don't free it, the resource destructor will do it */
	return PASS;
}
/* }}} */


/* {{{ mysqlnd_stmt_bind_param */
enum_func_status
mysqlnd_stmt_bind_param(MYSQLND_STMT *stmt, MYSQLND_PARAM_BIND *param_bind)
{
	int i = 0;

	if (stmt->state < MYSQLND_STMT_PREPARED) {
		SET_STMT_ERROR(stmt, CR_NO_PREPARE_STMT, UNKNOWN_SQLSTATE,
						 "Statement not prepared");
		return FAIL;	
	}

	if (stmt->param_count) {
		if (!param_bind) {
			SET_STMT_ERROR(stmt, CR_COMMANDS_OUT_OF_SYNC, UNKNOWN_SQLSTATE,
							 "Re-binding (still) not supported");
			return FAIL;
		} else if (stmt->param_bind) {
			/*
			  There is already result bound.
			  Forbid for now re-binding!!
			*/
			for (i = 0; i < stmt->param_count; i++) {
				if (stmt->param_bind[i].zv) {
					/*
					  We may have the last reference, then call zval_ptr_dtor()
					  or we may leak memory.
					*/
					zval_ptr_dtor(&stmt->param_bind[i].zv);
					stmt->param_bind[i].zv = NULL;
				}
			}
			efree(stmt->param_bind);
		}

		stmt->param_bind = param_bind;	
		for (i = 0; i < stmt->param_count; i++) {
			/* Prevent from freeing */
			ZVAL_ADDREF(stmt->param_bind[i].zv);
			/* Don't update is_ref, or we will leak during conversion */
		}
		stmt->send_types_to_server = 1;
	}
	return PASS;
}
/* }}} */



/* {{{ mysqlnd_stmt_bind_result */
enum_func_status
mysqlnd_stmt_bind_result(MYSQLND_STMT *stmt, MYSQLND_RESULT_BIND *result_bind)
{
	int i = 0;
	if (stmt->state < MYSQLND_STMT_PREPARED) {
		SET_STMT_ERROR(stmt, CR_NO_PREPARE_STMT, UNKNOWN_SQLSTATE,
						 "Statement not prepared");
		return FAIL;	
	}

	if (stmt->field_count) {
		if (!result_bind) {
			return FAIL;
		} else if (stmt->result_bind) {
			mysqlnd_stmt_separate_result_bind(stmt);	
			efree(stmt->result_bind);
		}
		
		stmt->result_bind = result_bind;
		for (i = 0; i < stmt->field_count; i++) {
			/* Prevent from freeing */
			ZVAL_ADDREF(stmt->result_bind[i].zv);
			/* Update as ZVAL_ADDREF does not do it */
			stmt->result_bind[i].zv->is_ref = 1;
			stmt->result_bind[i].bound = TRUE;
		}
	}
	return PASS;
}
/* }}} */


/* {{{ mysqlnd_stmt_insert_id */
unsigned long long mysqlnd_stmt_insert_id(const MYSQLND_STMT * const stmt)
{
	return stmt->upsert_status.last_insert_id;
}
/* }}} */


/* {{{ mysqlnd_stmt_affected_rows */
unsigned long long mysqlnd_stmt_affected_rows(const MYSQLND_STMT * const stmt)
{
	return stmt->upsert_status.affected_rows;
}
/* }}} */


/* {{{ mysqlnd_stmt_warning_count */
unsigned int mysqlnd_stmt_warning_count(const MYSQLND_STMT * const stmt)
{
	return stmt->upsert_status.warning_count;
}
/* }}} */


/* {{{ mysqlnd_stmt_field_count */
unsigned int mysqlnd_stmt_field_count(const MYSQLND_STMT * const stmt)
{
	return stmt->field_count;
}
/* }}} */


/* {{{ mysqlnd_stmt_param_count */
unsigned int mysqlnd_stmt_param_count(const MYSQLND_STMT * const stmt)
{
	return stmt->param_count;
}
/* }}} */


/* {{{ mysqlnd_stmt_errno */
unsigned int mysqlnd_stmt_errno(const MYSQLND_STMT * const stmt)
{
	return stmt->error_info.error_no;
}
/* }}} */


/* {{{ mysqlnd_stmt_error */
const char *mysqlnd_stmt_error(const MYSQLND_STMT * const stmt)
{
	return stmt->error_info.error;
}
/* }}} */


/* {{{ mysqlnd_stmt_sqlstate */
const char *mysqlnd_stmt_sqlstate(const MYSQLND_STMT * const stmt)
{
	return stmt->error_info.sqlstate[0] ? stmt->error_info.sqlstate:MYSQLND_SQLSTATE_NULL;
}
/* }}} */


/* {{{ mysqlnd_stmt_attr_set */
enum_func_status
mysqlnd_stmt_attr_set(MYSQLND_STMT *stmt, enum mysqlnd_stmt_attr attr_type, unsigned long value)
{
	switch (attr_type) {
		case STMT_ATTR_UPDATE_MAX_LENGTH:
			stmt->update_max_length = value? TRUE:FALSE;
			break;
		case STMT_ATTR_CURSOR_TYPE: {
			if (value > (unsigned long) CURSOR_TYPE_READ_ONLY) {
				SET_STMT_ERROR(stmt, CR_NOT_IMPLEMENTED, UNKNOWN_SQLSTATE, "Not implemented");
				return FAIL;
			}
			stmt->flags = value;
			break;
		}
		case STMT_ATTR_PREFETCH_ROWS: {
			if (value == 0) {
				value = DEFAULT_PREFETCH_ROWS;
			}
			stmt->prefetch_rows = value;
			break;
		}
		default:
			SET_STMT_ERROR(stmt, CR_NOT_IMPLEMENTED, UNKNOWN_SQLSTATE, "Not implemented");
			return FAIL;
	}
  	return PASS;
}
/* }}} */


/* {{{ mysqlnd_stmt_attr_get */
enum_func_status
mysqlnd_stmt_attr_get(MYSQLND_STMT *stmt, enum mysqlnd_stmt_attr attr_type, void *value)
{
	switch (attr_type) {
		case STMT_ATTR_UPDATE_MAX_LENGTH:
			*(zend_bool *) value= stmt->update_max_length;
			break;
		case STMT_ATTR_CURSOR_TYPE:
			*(unsigned long *) value= stmt->flags;
			break;
/* Unsupported for now			
		case STMT_ATTR_PREFETCH_ROWS:
			*(unsigned long *) value= stmt->prefetch_rows;
			break;
*/
		default:
 			return FAIL;
	}
	return PASS;
}
/* }}} */


/* {{{ mysqlnd_stmt_separate_result_bind */
void mysqlnd_stmt_separate_result_bind(MYSQLND_STMT *stmt)
{
	int i;
	if (!stmt->result_bind) {
		return;
	}
	/*
	  Because only the bound variables can point to our internal buffers, then
	  separate or free only them. Free is possible because the user could have
	  lost reference.
	*/
	for (i = 0; i < stmt->field_count; i++) {
		/* Let's try with no cache */
		if (stmt->result_bind[i].bound == TRUE) {
			if (stmt->state < MYSQLND_STMT_USER_FETCHING) {
				/*
				  There was no fetch from the user. Just call zval_ptr_dtor()
				  to remove the add_ref
				*/
				zval_ptr_dtor(&stmt->result_bind[i].zv);
			} else {
				/*
				  We have to separate the actual zval value of the bound
				  variable from our allocated zvals or we will face double-free
				*/
				if (ZVAL_REFCOUNT(stmt->result_bind[i].zv) > 1) {
					zval_copy_ctor(stmt->result_bind[i].zv);
				}
				/*
				  Maybe use the zval cache??? If we have copied, it won't destruct
				  the copy, as the refcount > 1, it will decrease the counter.
				*/
				zval_ptr_dtor(&stmt->result_bind[i].zv);
			}
		}
	}
}
/* }}} */


/* {{{ mysqlnd_internal_free_stmt_content */
void mysqlnd_internal_free_stmt_content(MYSQLND_STMT *stmt TSRMLS_DC)
{
	/* Destroy the input bind */
	if (stmt->param_bind) {
		int i;
		/*
		  Because only the bound variables can point to our internal buffers, then
		  separate or free only them. Free is possible because the user could have
		  lost reference.
		*/
		for (i = 0; i < stmt->param_count; i++) {
			if (stmt->param_bind[i].zv) {
				zval_ptr_dtor(&stmt->param_bind[i].zv);
			}
		}

		efree(stmt->param_bind);
		stmt->param_bind = NULL;
	}

	/*
	  First separate the bound variables, which point to the result set, then
	  destroy the set.
	*/
	if (stmt->result_bind) {
		mysqlnd_stmt_separate_result_bind(stmt);

		efree(stmt->result_bind);
		stmt->result_bind = NULL;
	}
	/* Not every statement has a result set attached */
	if (stmt->result) {
		mysqlnd_internal_free_result(stmt->result TSRMLS_CC);
		stmt->result = NULL;
	}
	if (stmt->cmd_buffer.buffer) {
		efree(stmt->cmd_buffer.buffer);
		stmt->cmd_buffer.buffer = NULL;
	}
	
	/*
	  result->conn is an address if this is an unbuffered query.
	  In this case, decrement the reference counter in the connection
	  object and if needed free the latter. If quit_sent is no
	*/
	if (stmt->conn && (!(--stmt->conn->references)) && stmt->conn->state == CONN_QUIT_SENT) {
		/*
		  No multithreading issues as we don't share the connection :)
		  This will free the object too, of course because references has
		  reached zero.
		*/
		mysqlnd_conn_free(stmt->conn TSRMLS_CC);
		stmt->conn = NULL;
	}
}
/* }}} */
