/*
  +----------------------------------------------------------------------+
  | PHP Version 6                                                        |
  +----------------------------------------------------------------------+
  | Copyright (c) 1997-2007 The PHP Group                                |
  +----------------------------------------------------------------------+
  | This source file is subject to version 3.01 of the PHP license,      |
  | that is bundled with this package in the file LICENSE, and is        |
  | available through the world-wide-web at the following url:           |
  | http://www.php.net/license/3_01.txt                                  |
  | If you did not receive a copy of the PHP license and are unable to   |
  | obtain it through the world-wide-web, please send a note to          |
  | license@php.net so we can mail you a copy immediately.               |
  +----------------------------------------------------------------------+
  | Authors: Georg Richter <georg@mysql.com>                             |
  |          Andrey Hristov <andrey@mysql.com>                           |
  |          Ulf Wendel <uwendel@mysql.com>                              |
  +----------------------------------------------------------------------+
*/

/* $Id: header,v 1.17 2006/01/01 13:09:48 sniper Exp $ */

#ifndef MYSQLND_PRIV_H
#define MYSQLND_PRIV_H


#ifdef ZTS
#include "TSRM.h"
#endif

typedef zend_uchar	mysqlnd_1b;
typedef zend_ushort mysqlnd_2b;
typedef zend_uint	mysqlnd_4b;


#define SERVER_STATUS_IN_TRANS     1	/* Transaction has started */
#define SERVER_STATUS_AUTOCOMMIT   2	/* Server in auto_commit mode */
#define SERVER_MORE_RESULTS_EXISTS 8    /* Multi query - next query exists */
/*
  The server was able to fulfill the clients request and opened a
  read-only non-scrollable cursor for a query. This flag comes
  in reply to COM_STMT_EXECUTE and COM_STMT_FETCH commands.
*/
#define SERVER_STATUS_CURSOR_EXISTS 64
/*
  This flag is sent when a read-only cursor is exhausted, in reply to
  COM_STMT_FETCH command.
*/
#define SERVER_STATUS_LAST_ROW_SENT 128
#define SERVER_STATUS_DB_DROPPED        256 /* A database was dropped */
#define SERVER_STATUS_NO_BACKSLASH_ESCAPES 512



/* Client Error codes */
#define CR_UNKNOWN_ERROR		2000
#define CR_CONNECTION_ERROR		2002
#define CR_SERVER_GONE_ERROR	2006
#define CR_OUT_OF_MEMORY		2008
#define CR_SERVER_LOST			2013
#define CR_COMMANDS_OUT_OF_SYNC 2014
#define CR_CANT_FIND_CHARSET	2019
#define CR_MALFORMED_PACKET     2027
#define CR_NOT_IMPLEMENTED		2054
#define CR_NO_PREPARE_STMT		2030
#define CR_PARAMS_NOT_BOUND		2031

#define MYSQLND_EE_READ			 2
#define MYSQLND_EE_FILENOTFOUND	29

#define UNKNOWN_SQLSTATE		"HY000"

#define MAX_CHARSET_LEN			32

#define MYSQLND_HEADER_SIZE 4

#define int1store(T,A)		 do { *((mysqlnd_1b*) (T))=A; } while(0)
#define uint1korr(A)	((mysqlnd_1b)(A)[0])


#define MYSQLND_NULL_LENGTH	(unsigned long) ~0


/* Packet handling */
#define PACKET_INIT(packet, enum_type, c_type)  \
	{ \
		packet = (c_type) ecalloc(1, packet_methods[enum_type].struct_size); \
		((c_type) (packet))->header.methods = &packet_methods[enum_type]; \
	}
#define PACKET_WRITE(packet, conn)	((packet)->header.methods->write_to_net((packet), (conn) TSRMLS_CC))
#define PACKET_READ(packet, conn)	((packet)->header.methods->read_from_net((packet), (conn) TSRMLS_CC))
#define PACKET_FREE(packet) \
	do { \
		((packet)->header.methods->free_mem((packet), FALSE)); \
	} while (0);

#define PACKET_INIT_ALLOCA(packet, enum_type)  \
	{ \
		memset(&(packet), 0, packet_methods[enum_type].struct_size); \
		(packet).header.methods = &packet_methods[enum_type]; \
	}
#define PACKET_WRITE_ALLOCA(packet, conn)	PACKET_WRITE(&(packet), (conn))
#define PACKET_READ_ALLOCA(packet, conn)	PACKET_READ(&(packet), (conn))
#define PACKET_FREE_ALLOCA(packet)			(packet.header.methods->free_mem(&(packet), TRUE))


/* Error handling */
#define SET_NEW_MESSAGE(buf, buf_len, message, len) \
	{\
		if ((buf)) { \
			efree(buf); \
		} \
		(buf) = (message); \
		(buf_len) = (len); \
		/* Transfer ownership*/ \
		(message) = NULL; \
	}

#define SET_EMPTY_ERROR(error_info) \
	{ \
		error_info.error_no = 0; \
		error_info.error[0] = '\0'; \
		strcpy(error_info.sqlstate, "00000"); \
	}

#define SET_CLIENT_ERROR(error_info, a, b, c) \
	{ \
		error_info.error_no = a; \
		strncpy(error_info.sqlstate, b, sizeof(error_info.sqlstate)); \
		strncpy(error_info.error, c, sizeof(error_info.error)); \
	}

#define SET_STMT_ERROR(stmt, a, b, c)	SET_CLIENT_ERROR(stmt->error_info, a, b, c)



enum php_mysql_packet_type
{
	PROT_GREET_PACKET= 0,
	PROT_AUTH_PACKET,
	PROT_OK_PACKET,
	PROT_EOF_PACKET,
	PROT_CMD_PACKET,
	PROT_RSET_HEADER_PACKET,
	PROT_RSET_FLD_PACKET,
	PROT_ROW_PACKET,
	PROT_STATS_PACKET,
	PROT_PREPARE_RESP_PACKET,
	PROT_CHG_USER_PACKET,
	PROT_BINROW_PACKET,
	PROT_LAST, /* should always be last */
};


enum php_mysqlnd_server_command
{
	COM_SLEEP = 0,
	COM_QUIT,
	COM_INIT_DB,
	COM_QUERY,
	COM_FIELD_LIST,
	COM_CREATE_DB,
	COM_DROP_DB,
	COM_REFRESH,
	COM_SHUTDOWN,
	COM_STATISTICS,
	COM_PROCESS_INFO,
	COM_CONNECT,
	COM_PROCESS_KILL,
	COM_DEBUG,
	COM_PING,
	COM_TIME,
	COM_DELAYED_INSERT,
	COM_CHANGE_USER,
	COM_BINLOG_DUMP,
	COM_TABLE_DUMP,
	COM_CONNECT_OUT,
	COM_REGISTER_SLAVE,
	COM_STMT_PREPARE,
	COM_STMT_EXECUTE,
	COM_STMT_SEND_LONG_DATA,
	COM_STMT_CLOSE,
	COM_STMT_RESET,
	COM_SET_OPTION,
	COM_STMT_FETCH,
	COM_DAEMON,
	COM_END
};


/* Session caching allocator */
struct mysqlnd_zval_list {
	zval 	**ptr_line;
	zval	**last_added;
};


struct _mysqlnd_zval_cache {
	struct mysqlnd_zval_list	*free_list;
	unsigned int		free_items;
	unsigned int		max_items;
	unsigned int		references;
	unsigned long		get_hits;
	unsigned long		get_misses;
	unsigned long		put_hits;
	unsigned long		put_full_misses;
	unsigned long		put_refcount_misses;
};


/* Persistent caching allocator */
struct _mysqlnd_zval {
	/* Should be first */
	zval		zv;
	zend_bool	ze_alloced;
};


typedef struct _mysqlnd_ndzval_list {
	mysqlnd_zval 	**ptr_line;
	mysqlnd_zval	**last_added;
} mysqlnd_ndzval_list;


struct _mysqlnd_zval_pcache {
	mysqlnd_zval		*block;
	mysqlnd_zval		*last_in_block;
	mysqlnd_ndzval_list	*free_list;
	unsigned int		free_items;
	unsigned int		max_items;
	unsigned int		references;

	unsigned long		get_hits;
	unsigned long		get_misses;
	unsigned long		put_hits;
	unsigned long		put_misses;
};



/* PS stuff */
typedef void (*ps_field_fetch_func)(zval *zv, const MYSQLND_FIELD * const field,
									uint pack_len, zend_uchar **row TSRMLS_DC);
struct mysqlnd_perm_bind {
	ps_field_fetch_func func;
	/* should be signed int */
	int					pack_len;
	unsigned int		php_type;
};

extern struct mysqlnd_perm_bind fetch_functions[MYSQL_TYPE_LAST + 1];


/* Low-level extraction functionality */
typedef struct _mysqlnd_packet_methods {
	size_t				struct_size;
	enum_func_status	(*read_from_net)(void *packet, MYSQLND *conn TSRMLS_DC);
	size_t				(*write_to_net)(void *packet, MYSQLND *conn TSRMLS_DC);
	void				(*free_mem)(void *packet, zend_bool alloca);
} mysqlnd_packet_methods;

extern mysqlnd_packet_methods packet_methods[];


typedef struct _mysqlnd_packet_header {
	mysqlnd_4b	size;
	mysqlnd_1b	packet_no;
	struct _mysqlnd_packet_methods *methods;
} mysqlnd_packet_header;


/* Server greets the client */
typedef struct _php_mysql_packet_greet {
	mysqlnd_packet_header		header;
	mysqlnd_1b	protocol_version;
	char		*server_version;
	mysqlnd_4b	thread_id;
	mysqlnd_1b	scramble_buf[SCRAMBLE_LENGTH];
	/* 1 byte pad */
	mysqlnd_2b	server_capabilities;
	mysqlnd_1b	charset_no;
	mysqlnd_2b	server_status;
	/* 13 byte pad*/
	zend_bool	pre41;
} php_mysql_packet_greet;


/* Client authenticates */
typedef struct _php_mysql_packet_auth {
	mysqlnd_packet_header		header;
	mysqlnd_4b	client_flags;
	uint32		max_packet_size;
	mysqlnd_1b	charset_no;
	/* 23 byte pad */
	char		*user;
	/* 8 byte scramble */
	char		*db;
	/* 12 byte scramble */

	/* Here the packet ends. This is user supplied data */
	char		*password;
	/* +1 for \0 because of scramble() */
	unsigned char	*server_scramble_buf;
	size_t			db_len;
} php_mysql_packet_auth;



/* OK packet */
typedef struct _php_mysql_packet_ok {
	mysqlnd_packet_header		header;
	mysqlnd_1b		field_count; /* always 0x0 */
	mynd_ulonglong	affected_rows;
	mynd_ulonglong	last_insert_id;
	mysqlnd_2b		server_status;
	mysqlnd_2b		warning_count;
	char			*message;
	size_t			message_len;
	/* If error packet, we use these */
	char 			error[MYSQLND_ERRMSG_SIZE+1];
	char 			sqlstate[MYSQLND_SQLSTATE_LENGTH + 1];
	unsigned int 	error_no;
} php_mysql_packet_ok;


/* Command packet */
typedef struct _php_mysql_packet_command {
	mysqlnd_packet_header			header;
	enum php_mysqlnd_server_command	command;
	const char						*argument;
	size_t							arg_len;
} php_mysql_packet_command;


/* EOF packet */
typedef struct _php_mysql_packet_eof {
	mysqlnd_packet_header		header;
	mysqlnd_1b		field_count; /* 0xFE */
	mysqlnd_2b		warning_count;
	mysqlnd_2b		server_status;
} php_mysql_packet_eof;
/* EOF packet */


/* Result Set header*/
typedef struct _php_mysql_packet_rset_header {
	mysqlnd_packet_header		header;
	/*
	  0x00 => ok
	  ~0   => LOAD DATA LOCAL
	  error_no != 0 => error
	  others => result set -> Read res_field packets up to field_count
	*/
	unsigned long		field_count;
	/*
	  These are filled if no SELECT query. For SELECT warning_count
	  and server status are in the last row packet, the EOF packet.
	*/
	mysqlnd_2b			warning_count;
	mysqlnd_2b			server_status;
	unsigned long long	affected_rows;
	unsigned long long	last_insert_id;
	/* This is for both LOAD DATA or info, when no result set */
	char				*info_or_local_file;
	size_t				info_or_local_file_len;
	/* If error packet, we use these */
	mysqlnd_error_info	error_info;
} php_mysql_packet_rset_header;


/* Result set field packet */
typedef struct _php_mysql_packet_res_field {
	mysqlnd_packet_header		header;
	MYSQLND_FIELD				*metadata;
	/* For table definitions, empty for result sets */
	zend_bool					skip_parsing;
} php_mysql_packet_res_field;


/* Row packet */
typedef struct _php_mysql_packet_row {
	mysqlnd_packet_header	header;
	zval			**fields;
	mysqlnd_4b		field_count;
	zend_bool		eof;
	/*
	  These are, of course, only for SELECT in the EOF packet,
	  which is detected by this packet
	*/
	mysqlnd_2b		warning_count;
	mysqlnd_2b		server_status;

	zend_uchar		*row_buffer;

	zend_bool			skip_extraction;
	zend_bool			binary_protocol;
	MYSQLND_FIELD		*fields_metadata;
} php_mysql_packet_row;


/* Binary Row packet */
typedef struct _php_mysql_packet_binrow {
	mysqlnd_packet_header	header;
	zval			**fields;
	mysqlnd_4b		field_count;
	zend_bool		eof;
	/*
	  These are, of course, only for SELECT in the EOF packet,
	  which is detected by this packet
	*/
	mysqlnd_2b		warning_count;
	mysqlnd_2b		server_status;

	zend_uchar		*row_buffer;

	zend_bool		skip_extraction;
} php_mysql_packet_binrow;



/* Statistics packet */
typedef struct _php_mysql_packet_stats {
	mysqlnd_packet_header	header;
	char *message;
	/* message_len is not part of the packet*/
	size_t message_len;
} php_mysql_packet_stats;


/* COM_PREPARE response packet */
typedef struct _php_mysql_packet_prepare_response {
	mysqlnd_packet_header	header;
	/* also known as field_count 0x00=OK , 0xFF=error */
	unsigned char	error_code;
	unsigned long	stmt_id;
	unsigned int	field_count;
	unsigned int	param_count;
	unsigned int	warning_count;

	/* present in case of error */
	mysqlnd_error_info	error_info;
} php_mysql_packet_prepare_response;


/* Statistics packet */
typedef struct _php_mysql_packet_chg_user_resp {
	mysqlnd_packet_header	header;
	mysqlnd_4b		field_count;
	
	/* message_len is not part of the packet*/
	mysqlnd_2b	server_capabilities;
	/* If error packet, we use these */
	mysqlnd_error_info	error_info;
} php_mysql_packet_chg_user_resp;

 
void php_mysqlnd_free_field_metadata(MYSQLND_FIELD *meta);
void php_mysqlnd_scramble(unsigned char *buffer, const unsigned char *scramble,
						  const unsigned char *password);

unsigned long php_mysqlnd_net_field_length(zend_uchar **packet);
zend_uchar *php_mysqlnd_net_store_length(zend_uchar *packet, mynd_ulonglong length);


enum_func_status mysqlnd_read_result_metadata(MYSQLND *conn, MYSQLND_RES *result TSRMLS_DC);
enum_func_status mysqlnd_simple_command(MYSQLND *conn, enum php_mysqlnd_server_command command,
										const char * const arg, size_t arg_len,
										enum php_mysql_packet_type ok_packet,
										zend_bool silent TSRMLS_DC);
enum_func_status mysqlnd_simple_command_handle_response(MYSQLND *conn,
										enum php_mysql_packet_type ok_packet,
										zend_bool silent, enum php_mysqlnd_server_command command
									  	TSRMLS_DC);
										
void mysqlnd_unbuffered_free_last_data(MYSQLND_RES *result);
void mysqlnd_internal_free_result_contents(MYSQLND_RES *result);
void mysqlnd_internal_free_stmt_content(MYSQLND_STMT *stmt TSRMLS_DC);
void mysqlnd_internal_free_result(MYSQLND_RES *result TSRMLS_DC);
void mysqlnd_conn_free(MYSQLND *conn TSRMLS_DC);
size_t mysqlnd_stream_write(MYSQLND *conn, char * const buf, size_t count TSRMLS_DC);


enum_func_status mysqlnd_handle_local_infile(MYSQLND *conn, const char *filename TSRMLS_DC);



#endif	/* MYSQLND_PRIV_H */


/*
 * Local variables:
 * tab-width: 4
 * c-basic-offset: 4
 * End:
 * vim600: noet sw=4 ts=4 fdm=marker
 * vim<600: noet sw=4 ts=4
 */
