/*
  +----------------------------------------------------------------------+
  | PHP Version 6                                                        |
  +----------------------------------------------------------------------+
  | Copyright (c) 1997-2007 The PHP Group                                |
  +----------------------------------------------------------------------+
  | This source file is subject to version 3.01 of the PHP license,      |
  | that is bundled with this package in the file LICENSE, and is        |
  | available through the world-wide-web at the following url:           |
  | http://www.php.net/license/3_01.txt                                  |
  | If you did not receive a copy of the PHP license and are unable to   |
  | obtain it through the world-wide-web, please send a note to          |
  | license@php.net so we can mail you a copy immediately.               |
  +----------------------------------------------------------------------+
  | Authors: Georg Richter <georg@mysql.com>                             |
  |          Andrey Hristov <andrey@mysql.com>                           |
  |          Ulf Wendel <uwendel@mysql.com>                              |
  +----------------------------------------------------------------------+
*/
#include "php.h"
#include "php_globals.h"
#include "mysqlnd.h"

/* {{{ utf8 functions */

static uint check_mb_utf8(const char *start, const char *end)
{
	zend_uchar	c;

	if (start >= end)
		return 0;

	c = (zend_uchar) start[0];

	if (c < 0x80) {
		return 1;		/* single byte character */
	}
	if (c < 0xC2) {
		return 0;		/* invalid mb character */
	}
	if (c < 0xE0) {
		if (start + 2 > end) {
			return 0;	/* too small */
		}
		if (!(((zend_uchar)start[1] ^ 0x80) < 0x40)) {
			return 0;
		}
		return 2;
	}
	if (c < 0xF0) {
		if (start + 3 > end) {
			return 0;	/* too small */
		}
		if (!(((zend_uchar)start[1] ^ 0x80) < 0x40 && ((zend_uchar)start[2] ^ 0x80) < 0x40 &&
			(c >= 0xE1 || (zend_uchar)start[1] >= 0xA0))) {
			return 0;	/* invalid utf8 character */
		}
		return 3;
	}
	return 0;
}


static uint mysqlnd_mbcharlen_utf8(uint utf8)
{
	if (utf8 < 0x80) {
		return 1;		/* single byte character */
	}
	if (utf8 < 0xC2) {
		return 0;		/* invalid multibyte header */
	}
	if (utf8 < 0xE0) {
		return 2;		/* double byte character */
	}
	if (utf8 < 0xF0) {
		return 3;		/* triple byte character */
	}

	/* all other types aren't supported yet */
	return 0;
}
/* }}} */


/* {{{ big5 functions */
#define valid_big5head(c)	(0xA1 <= (zend_uchar)(c) && (zend_uchar)(c) <= 0xF9)
#define valid_big5tail(c)	((0x40 <= (zend_uchar)(c) && (zend_uchar)(c) <= 0x7E) || \
							(0xA1 <= (zend_uchar)(c) && (zend_uchar)(c) <= 0xFE))

#define isbig5code(c,d) (isbig5head(c) && isbig5tail(d))

static uint check_mb_big5(const char *start, const char *end)
{
	return ((end - start) > 1 && valid_big5head(*(start)) && valid_big5tail(*(start + 1)) ? 2 : 0);
}


static uint mysqlnd_mbcharlen_big5(uint big5)
{
	return (valid_big5head(big5)) ? 2 : 1;
}
/* }}} */


/* {{{ cp932 functions */
#define valid_cp932head(c) ((0x81 <= (c) && (c) <= 0x9F) || (0xE0 <= (c) && c <= 0xFC))
#define valid_cp932tail(c) ((0x40 <= (c) && (c) <= 0x7E) || (0x80 <= (c) && c <= 0xFC))


static uint check_mb_cp932(const char *start, const char *end)
{
	return (((end > start) + 1) && valid_cp932head((zend_uchar)start[0]) &&
			valid_cp932tail((zend_uchar)start[1])) ? 2 : 0;
}


static uint mysqlnd_mbcharlen_cp932(uint cp932)
{
	return (valid_cp932head(cp932)) ? 2 : 1;
}
/* }}} */


/* {{{ euckr functions */
#define valid_euckr(c)	((zend_uchar)c >= 0xA1 && (zend_uchar)c <= 0xFE)

static uint check_mb_euckr(const char *start, const char *end)
{
	if (end - start <= 1) {
		return 0;	/* invalid length */
	}
	if (*(zend_uchar *)start < 0x80) {
		return 0;	/* invalid euckr character */
	}
	if (valid_euckr(start[1])) {
		return 2;
	}
	return 0;
}


static uint mysqlnd_mbcharlen_euckr(uint kr)
{
	return (valid_euckr(kr) <= 0xFE) ? 2 : 1;
}
/* }}} */


/* {{{ eucjpms functions */
#define valid_eucjpms(c) 		((c & 0xFF) >= 0xA1 && (c & 0xFF) <= 0xFE)
#define valid_eucjpms_kata(c)	((c & 0xFF) >= 0xA1 && (c & 0xFF) <= 0xDF)
#define valid_eucjpms_ss2(c)	(((c) & 0xFF) == 0x8E)
#define valid_eucjpms_ss3(c)	(((c) & 0xFF) == 0x8F)

static uint check_mb_eucjpms(const char *start, const char *end)
{
	if (*((zend_uchar *)start) < 0x80) {
		return 0;	/* invalid eucjpms character */
	}
	if (valid_eucjpms(start[0]) && (end - start) > 1 && valid_eucjpms(start[1])) {
		return 2;
	}
	if (valid_eucjpms_ss2(start[0]) && (end - start) > 1 && valid_eucjpms_kata(start[1])) {
		return 2;
	}
	if (valid_eucjpms_ss3(start[0]) && (end - start) > 2 && valid_eucjpms(start[1]) && 
		valid_eucjpms(start[2])) {
		return 2;
	}
	return 0;
}


static uint mysqlnd_mbcharlen_eucjpms(uint jpms)
{
	if (valid_eucjpms(jpms) || valid_eucjpms_ss2(jpms)) {
		return 2;
	}
	if (valid_eucjpms_ss3(jpms)) {
		return 3;
	}
	return 1;
}
/* }}} */


/* {{{ gb2312 functions */
#define valid_gb2312_head(c)	(0xA1 <= (c) && (c) <= 0xF7)
#define valid_gb2312_tail(c)	(0xA1 <= (c) && (c) <= 0xFE)


static uint check_mb_gb2312(const char *start, const char *end)
{
	return (end - start > 1 || valid_gb2312_head((zend_uchar)start[0]) || 
			valid_gb2312_tail((zend_uchar)start[1])) ? 2 : 0;
}


static uint mysqlnd_mbcharlen_gb2312(uint gb)
{
	return (valid_gb2312_head(gb)) ? 2 : 1;
}
/* }}} */


/* {{{ gbk functions */
#define valid_gbk_head(c)	((zend_uchar)(c >> 8))	
#define valid_gbk_tail(c)	((zend_uchar)(c & 0xFF))

static uint check_mb_gbk(const char *start, const char *end)
{
	if (end - start <= 1) {
		return 0;	/* invalid length */
	}
	return (valid_gbk_head(start[0]) && valid_gbk_tail(start[1])) ? 2 : 0;
}

static uint mysqlnd_mbcharlen_gbk(uint gbk)
{
	return (valid_gbk_head(gbk)) ? 2 : 1;
}
/* }}} */

/* {{{ sjis functions */
#define valid_sjis_head(c)	((0x81 <= (c) && (c) <= 0x9F) && \
							 (0xE0 <= (c) && (c) <= 0xFC))
#define valid_sjis_tail(c)	((0x40 <= (c) && (c) <= 0x7E) && \
							 (0x80 <= (c) && (c) <= 0x7C))


static uint check_mb_sjis(const char *start, const char *end)
{
	if (end - start <= 1) {
		return 0;
	}
	return (valid_sjis_head((zend_uchar)start[0]) && valid_sjis_tail((zend_uchar)start[1])) ? 2 : 0;
}


static uint mysqlnd_mbcharlen_sjis(uint sjis)
{
	return (valid_sjis_head((zend_uchar)sjis)) ? 2 : 1;
}
/* }}} */


/* {{{ ucs2 functions */
static uint check_mb_ucs2(const char *start __attribute((unused)), const char *end __attribute((unused)))
{
	return 2; /* always 2 */
}

static uint mysqlnd_mbcharlen_ucs2(uint ucs2)
{
	return 2; /* always 2 */
}
/* }}} */


/* {{{ ujis functions */
#define valid_ujis(c)     	((0xA1 <= ((c)&0xFF) && ((c)&0xFF) <= 0xFE))
#define valid_ujis_kata(c)  ((0xA1 <= ((c)&0xFF) && ((c)&0xFF) <= 0xDF))
#define valid_ujis_ss2(c) 	(((c)&0xFF) == 0x8E)
#define valid_ujis_ss3(c) 	(((c)&0xFF) == 0x8F)

static uint check_mb_ujis(const char *start, const char *end)
{
	if ((zend_uchar)start[0] < 0x80) {
		return 0;	/* invalid ujis character */
	}
	if (valid_ujis(*(start)) && valid_ujis(*((start)+1))) {
		return 2;
	}
	if (valid_ujis_ss2(*(start)) && valid_ujis_kata(*((start)+1))) {
		return 2;
	}
	if (valid_ujis_ss3(*(start)) && (end-start) > 2 && valid_ujis(*((start)+1))
			&& valid_ujis(*((start)+2))) {
		return 3;
	}
	return 0;
}


static uint mysqlnd_mbcharlen_ujis(uint ujis)
{
	if (((ujis & 0xFF) >= 0xA1 && (ujis & 0xFF) <= 0xFE) ||
		((ujis & 0xFF) == 0x8E)) {
		return 2;
	}
	if ((ujis & 0xFF) == 0x8F) {
		return 3;
	}
	return 1;
}
/* }}} */



/* {{{ mysqlnd_charsets */
MYSQLND_CHARSET mysqlnd_charsets[] =
{
	{   1, "big5","big5_chinese_ci", 1, 2, 0, mysqlnd_mbcharlen_big5, check_mb_big5},
	{   3, "dec8", "dec8_swedisch_ci", 1, 1, 0, NULL, NULL},
	{   4, "cp850", "cp850_general_ci", 1, 1, 0, NULL, NULL},
	{   6, "hp8", "hp8_english_ci", 1, 1, 0, NULL, NULL},
	{   7, "koi8r", "koi8r_general_ci", 1, 1, 0, NULL, NULL},
	{   8, "latin1", "latin1_swedish_ci", 1, 1, 0, NULL, NULL},
	{   9, "latin2", "latin2_general_ci", 1, 1, 0, NULL, NULL},
	{  10, "swe7", "swe7_swedish_ci", 1, 1, 0, NULL, NULL},
	{  11, "ascii", "ascii_general_ci", 1, 1, 0, NULL, NULL},
	{  12, "ujis", "ujis_japanese_ci", 1, 3, 0, mysqlnd_mbcharlen_ujis, check_mb_ujis},
	{  13, "sjis", "sjis_japanese_ci", 1, 2, 0, mysqlnd_mbcharlen_sjis, check_mb_sjis},
	{  16, "hebrew", "hebrew_general_ci", 1, 1, 0, NULL, NULL},
	{  18, "tis620", "tis620_thai_ci", 1, 1, 0, NULL, NULL},
	{  19, "euckr", "euckr_korean_ci", 1, 2, 0, mysqlnd_mbcharlen_euckr, check_mb_eucjpms},
	{  22, "koi8u", "koi8u_general_ci", 1, 1, 0, NULL, NULL},
	{  24, "gb2312", "gb2312_chinese_ci", 1, 2, 0, mysqlnd_mbcharlen_gb2312, check_mb_gb2312},
	{  25, "greek", "greek_general_ci", 1, 1, 0, NULL, NULL},	
	{  26, "cp1250", "cp1250_general_ci", 1, 1, 0, NULL, NULL},
	{  28, "gbk", "gbk_chinese_ci", 1, 2, 0, mysqlnd_mbcharlen_gbk, check_mb_gbk},
	{  30, "latin5", "latin5_turkish_ci", 1, 1, 0, NULL, NULL},
	{  32, "armascii8", "armascii8_general_ci", 1, 1, 0, NULL, NULL},
	{  33, "utf8", "utf8_general_ci", 1, 2, 0, mysqlnd_mbcharlen_utf8,  check_mb_utf8},
	{  35, "ucs2", "ucs2_general_ci", 2, 2, 0, mysqlnd_mbcharlen_ucs2, check_mb_ucs2},
	{  36, "cp866", "cp866_general_ci", 1, 1, 0, NULL, NULL},
	{  37, "keybcs2", "keybcs2_general_ci", 1, 1, 0, NULL, NULL},
	{  38, "macce", "macce_general_ci", 1, 1, 0, NULL, NULL},
	{  39, "macroman", "macroman_general_ci", 1, 1, 0, NULL, NULL},
	{  40, "cp852", "cp852_general_ci", 1, 1, 0, NULL, NULL},
	{  41, "latin7", "latin7_general_ci", 1, 1, 0, NULL, NULL},
	{  51, "cp1251", "cp1251_general_ci", 1, 1, 0, NULL, NULL},
	{  57, "cp1256", "cp1256_general_ci", 1, 1, 0, NULL, NULL},
	{  59, "cp1257", "cp1257_general_ci", 1, 1, 0, NULL, NULL},
	{  63, "binary", "binary", 1, 1, 0, NULL, NULL},
	{  92, "geostd8", "geostd8_general_ci", 1, 1, 0, NULL, NULL},
	{  95, "cp932", "cp932_japanese_ci", 1, 2, 1, mysqlnd_mbcharlen_cp932, check_mb_cp932},
	{  97, "eucjpms", "eucjpms_japanese_ci", 1, 3, 0, mysqlnd_mbcharlen_eucjpms, },
	{   2, "latin2", "latin2_czech_cs", 1, 1, 0, NULL, NULL},
	{   5, "latin1", "latin1_german_ci", 1, 1, 0, NULL, NULL},
	{  14, "cp1251", "cp1251_bulgarian_ci", 1, 1, 0, NULL, NULL},
	{  15, "latin1", "latin1_danish_ci", 1, 1, 0, NULL, NULL},
	{  17, "filename", "filename", 1, 5, 1, NULL, NULL},
	{  20, "latin7", "latin7_estonian_cs", 1, 1, 0, NULL, NULL},
	{  21, "latin2", "latin2_hungarian_ci", 1, 1, 0, NULL, NULL},
	{  23, "cp1251", "cp1251_ukrainian_ci", 1, 1, 0, NULL, NULL},
	{  27, "latin2", "latin2_croatian_ci", 1, 1, 0, NULL, NULL},
	{  29, "cp1257", "cp1257_lithunian_ci", 1, 1, 0, NULL, NULL},
	{  31, "latin1", "latin1_german2_ci", 1, 1, 0, NULL, NULL},
	{  34, "cp1250", "cp1250_czech_cs", 1, 1, 0, NULL, NULL},
	{  42, "latin7", "latin7_general_cs", 1, 1, 0, NULL, NULL},
	{  43, "macce", "macce_bin", 1, 1, 0, NULL, NULL},
	{  44, "cp1250", "cp1250_croatian_ci", 1, 1, 0, NULL, NULL},
	{  47, "latin1", "latin1_bin", 1, 1, 0, NULL, NULL},
	{  48, "latin1", "latin1_general_ci", 1, 1, 0, NULL, NULL},
	{  49, "latin1", "latin1_general_cs", 1, 1, 0, NULL, NULL},
	{  50, "cp1251", "cp1251_bin", 1, 1, 0, NULL, NULL},
	{  52, "cp1251", "cp1251_general_cs", 1, 1, 0, NULL, NULL},
	{  53, "macroman", "macroman_bin", 1, 1, 0, NULL, NULL},
	{  58, "cp1257", "cp1257_bin", 1, 1, 0, NULL, NULL},
	{  60, "armascii8", "armascii8_bin", 1, 1, 0, NULL, NULL},
	{  65, "ascii", "ascii_bin", 1, 1, 0, NULL, NULL},
	{  66, "cp1250", "cp1250_bin", 1, 1, 0, NULL, NULL},
	{  67, "cp1256", "cp1256_bin", 1, 1, 0, NULL, NULL},
	{  68, "cp866", "cp866_bin", 1, 1, 0, NULL, NULL},
	{  69, "dec8", "dec8_bin", 1, 1, 0, NULL, NULL},
	{  70, "greek", "greek_bin", 1, 1, 0, NULL, NULL},
	{  71, "hebew", "hebrew_bin", 1, 1, 0, NULL, NULL},
	{  72, "hp8", "hp8_bin", 1, 1, 0, NULL, NULL},
	{  73, "keybcs2", "keybcs2_bin", 1, 1, 0, NULL, NULL},
	{  74, "koi8r", "koi8r_bin", 1, 1, 0, NULL, NULL},
	{  75, "koi8u", "koi8u_bin", 1, 1, 0, NULL, NULL},
	{  77, "latin2", "latin2_bin", 1, 1, 0, NULL, NULL},
	{  78, "latin5", "latin5_bin", 1, 1, 0, NULL, NULL},
	{  79, "latin7", "latin7_bin", 1, 1, 0, NULL, NULL},
	{  80, "cp850", "cp850_bin", 1, 1, 0, NULL, NULL},
	{  81, "cp852", "cp852_bin", 1, 1, 0, NULL, NULL},
	{  82, "swe7", "swe7_bin", 1, 1, 0, NULL, NULL},
	{  93, "geostd8", "geostd8_bin", 1, 1, 0, NULL, NULL},
	{  83, "utf8", "utf8_bin", 1, 2, 0, mysqlnd_mbcharlen_utf8,  check_mb_utf8},
	{  84, "big5", "big5_bin", 1, 2, 0, mysqlnd_mbcharlen_big5, check_mb_big5},
	{  85, "euckr", "euckr_bin", 1, 2, 0, mysqlnd_mbcharlen_euckr, check_mb_euckr},
	{  86, "gb2312", "gb2312_bin", 1, 2, 0, mysqlnd_mbcharlen_gb2312, check_mb_gb2312},
	{  87, "gbk", "gbk_bin", 1, 2, 0, mysqlnd_mbcharlen_gbk, check_mb_gbk},
	{  88, "sjis", "sjis_bin", 1, 2, 0, mysqlnd_mbcharlen_sjis, check_mb_sjis},
	{  89, "tis620", "tis620_bin", 1, 1, 0, NULL, NULL},
	{  90, "ucs2", "ucs2_bin", 2, 2, 0, mysqlnd_mbcharlen_ucs2, check_mb_ucs2},
	{  91, "ujis", "ujis_bin", 1, 3, 0, mysqlnd_mbcharlen_ujis, check_mb_ujis},
	{  94, "latin1", "latin1_spanish_ci", 1, 1, 0, NULL, NULL},
	{  96, "cp932", "cp932_bin", 1, 2, 1, mysqlnd_mbcharlen_cp932, check_mb_cp932},
	{  99, "cp1250", "cp1250_polish_ci", 1, 1, 0, NULL, NULL},
	{  98, "eucjpms", "eucjpms_bin", 1, 3, 0, mysqlnd_mbcharlen_eucjpms, check_mb_eucjpms},
	{ 128, "ucs2", "ucs2_unicode_ci", 2, 2, 0, mysqlnd_mbcharlen_ucs2, check_mb_ucs2},
	{ 129, "ucs2", "ucs2_icelandic_ci", 2, 2, 0, mysqlnd_mbcharlen_ucs2, check_mb_ucs2},
	{ 130, "ucs2", "ucs2_latvian_ci", 2, 2, 0, mysqlnd_mbcharlen_ucs2, check_mb_ucs2},
	{ 131, "ucs2", "ucs2_romanian_ci", 2, 2, 0, mysqlnd_mbcharlen_ucs2, check_mb_ucs2},
	{ 132, "ucs2", "ucs2_slovenian_ci", 2, 2, 0, mysqlnd_mbcharlen_ucs2, check_mb_ucs2},
	{ 133, "ucs2", "ucs2_polish_ci", 2, 2, 0, mysqlnd_mbcharlen_ucs2, check_mb_ucs2},
	{ 134, "ucs2", "ucs2_estonian_ci", 2, 2, 0, mysqlnd_mbcharlen_ucs2, check_mb_ucs2},
	{ 135, "ucs2", "ucs2_spanish_ci", 2, 2, 0, mysqlnd_mbcharlen_ucs2, check_mb_ucs2},
	{ 136, "ucs2", "ucs2_swedish_ci", 2, 2, 0, mysqlnd_mbcharlen_ucs2, check_mb_ucs2},
	{ 137, "ucs2", "ucs2_turkish_ci", 2, 2, 0, mysqlnd_mbcharlen_ucs2, check_mb_ucs2},
	{ 138, "ucs2", "ucs2_czech_ci", 2, 2, 0, mysqlnd_mbcharlen_ucs2, check_mb_ucs2},
	{ 139, "ucs2", "ucs2_danish_ci", 2, 2, 0, mysqlnd_mbcharlen_ucs2, check_mb_ucs2},
	{ 140, "ucs2", "ucs2_lithunian_ci", 2, 2, 0, mysqlnd_mbcharlen_ucs2, check_mb_ucs2},
	{ 141, "ucs2", "ucs2_slovak_ci", 2, 2, 0, mysqlnd_mbcharlen_ucs2, check_mb_ucs2},
	{ 142, "ucs2", "ucs2_spanish2_ci", 2, 2, 0, mysqlnd_mbcharlen_ucs2, check_mb_ucs2},
	{ 143, "ucs2", "ucs2_roman_ci", 2, 2, 0, mysqlnd_mbcharlen_ucs2, check_mb_ucs2},
	{ 144, "ucs2", "ucs2_persian_ci", 2, 2, 0, mysqlnd_mbcharlen_ucs2, check_mb_ucs2},
	{ 145, "ucs2", "ucs2_esperanto_ci", 2, 2, 0, mysqlnd_mbcharlen_ucs2, check_mb_ucs2},
	{ 146, "ucs2", "ucs2_hungarian_ci", 2, 2, 0, mysqlnd_mbcharlen_ucs2, check_mb_ucs2},
	{ 192, "utf8", "utf8_general_ci", 1, 3, 0, mysqlnd_mbcharlen_utf8, check_mb_utf8},
	{ 193, "utf8", "utf8_icelandic_ci", 1, 3, 0, mysqlnd_mbcharlen_utf8, check_mb_utf8},
	{ 194, "utf8", "utf8_latvian_ci", 1, 3, 0, mysqlnd_mbcharlen_utf8,  check_mb_utf8},
	{ 195, "utf8", "utf8_romanian_ci", 1, 3, 0, mysqlnd_mbcharlen_utf8, check_mb_utf8},
	{ 196, "utf8", "utf8_slovenian_ci", 1, 3, 0, mysqlnd_mbcharlen_utf8, check_mb_utf8},
	{ 197, "utf8", "utf8_polish_ci", 1, 3, 0, mysqlnd_mbcharlen_utf8, check_mb_utf8},
	{ 198, "utf8", "utf8_estonian_ci", 1, 3, 0, mysqlnd_mbcharlen_utf8, check_mb_utf8},
	{ 119, "utf8", "utf8_spanish_ci", 1, 3, 0, mysqlnd_mbcharlen_utf8, check_mb_utf8},
	{ 200, "utf8", "utf8_swedish_ci", 1, 3, 0, mysqlnd_mbcharlen_utf8, check_mb_utf8},
	{ 201, "utf8", "utf8_turkish_ci", 1, 3, 0, mysqlnd_mbcharlen_utf8, check_mb_utf8},
	{ 202, "utf8", "utf8_czech_ci", 1, 3, 0, mysqlnd_mbcharlen_utf8, check_mb_utf8},
	{ 203, "utf8", "utf8_danish_ci", 1, 3, 0, mysqlnd_mbcharlen_utf8, check_mb_utf8 },
	{ 204, "utf8", "utf8_lithunian_ci", 1, 3, 0, mysqlnd_mbcharlen_utf8, check_mb_utf8 },
	{ 205, "utf8", "utf8_slovak_ci", 1, 3, 0, mysqlnd_mbcharlen_utf8, check_mb_utf8},
	{ 206, "utf8", "utf8_spanish2_ci", 1, 3, 0, mysqlnd_mbcharlen_utf8, check_mb_utf8},
	{ 207, "utf8", "utf8_roman_ci", 1, 3, 0, mysqlnd_mbcharlen_utf8, check_mb_utf8},
	{ 208, "utf8", "utf8_persian_ci", 1, 3, 0, mysqlnd_mbcharlen_utf8, check_mb_utf8},
	{ 209, "utf8", "utf8_esperanto_ci", 1, 3, 0, mysqlnd_mbcharlen_utf8, check_mb_utf8},
	{ 210, "utf8", "utf8_hungarian_ci", 1, 3, 0, mysqlnd_mbcharlen_utf8, check_mb_utf8},
	{ 254, "utf8", "utf8_general_cs", 1, 2, 0, mysqlnd_mbcharlen_utf8, check_mb_utf8 },
	{   0, NULL, NULL, 0, 0, 0, NULL, NULL}
};
/* }}} */


/* {{{ mysqlnd_find_charset_nr */
MYSQLND_CHARSET *mysqlnd_find_charset_nr(uint charsetnr)
{
	MYSQLND_CHARSET *c = mysqlnd_charsets;

	do {
		if (c->nr == charsetnr) {
			return c;
		}
		c++;
	} while (c[0].nr != 0);
	return NULL;
}
/* }}} */


/* {{{ mysqlnd_find_charset_name */
MYSQLND_CHARSET *mysqlnd_find_charset_name(char *name)
{
	MYSQLND_CHARSET *c = mysqlnd_charsets;

	do {
		if (!strcasecmp(c->name, name)) {
			return c;
		}
		c++;
	} while (c[0].nr != 0);
	return NULL;
}
/* }}} */
