/*
 *  A Z-Machine
 *  Copyright (C) 2000 Andrew Hunter
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * Main interpreter loop
 */

#include <stdio.h>
#include <stdlib.h>

#include "../config.h"

/* #define inline */

#include "zmachine.h"
#include "op.h"
#include "zscii.h"

#ifdef INLINE_OPS
#define INLINING
#include "op.c"
#endif

#include "varop.h"

#if defined(SUPPORT_VERSION_4) || defined(SUPPORT_VERSION_5) || defined(SUPPORT_VERSION_6) || defined(SUPPORT_VERSION_7) || defined(SUPPORT_VERSION_8)
/* #include "doubleop.h" -- This gets far too large */

/*
 * (non-specialised, slow, version)
 */
int zmachine_decode_doubleop(ZStack* stack,
			     ZByte* param,
			     ZArgblock* argblock)
{
  ZUWord params;
  int padding = 2;
  int x, type;

  params = (param[0]<<8)|param[1];
  x=0;
  argblock->n_args = 0;
  do
    {
      type = (params>>(14-x*2))&3;

      switch (type)
	{
	case 0:
	  argblock->arg[x] = (param[padding]<<8)|param[padding+1];
	  padding+=2;
	  argblock->n_args++;
	  break;
	  
	case 1:
	  argblock->arg[x] = param[padding];
	  padding+=1;
	  argblock->n_args++;
	  break;
	  
	case 2:
	  argblock->arg[x] = GetVar(param[padding]);
	  padding+=1;
	  argblock->n_args++;
	  break;
	}

      x++;
    }
  while (type != 3 && x<8);

  return padding-2;
}
#endif

void zmachine_run(const int version)
{
#ifdef GLOBAL_PC
# define pc machine.pc
#else
  ZDWord         pc;
#endif
  int            padding;
  int            store;
  int            tmp;
  int            negate;
  ZDWord         branch;
  ZWord          arg1 = 0;
  ZWord          arg2 = 0;
  ZArgblock      argblock;
  ZByte*         code;
  ZStack*        stack;
  char*          string;
  register ZByte instr;

  code  = machine.memory;
  pc    = GetWord(machine.header, ZH_initpc);
  stack = &machine.stack;

  zmachine_setup_header();
  machine.undo = NULL;

#if defined(SUPPORT_VERSION_6)
  if (version == 6)
    {
      call_routine(&pc, stack,
		   (4*GetWord(machine.header, ZH_initpc)) +
		   machine.routine_offset);
      zcode_v6_initialise();
    }
#endif

  switch (version)
    {
#if defined(SUPPORT_VERSION_4) || defined(SUPPORT_VERSION_5)
    case 4:
    case 5:
      machine.packtype = packed_v4;
      break;
#ifdef SUPPORT_VERSION_5
    case 8:
      machine.packtype = packed_v8;
      break;
#endif
#endif

#if defined(SUPPORT_VERSION_5) || defined(SUPPORT_VERSION_6)
    case 6:
    case 7:
      machine.packtype = packed_v6;
      machine.routine_offset = 8*Word(ZH_routines);
      machine.string_offset = 8*Word(ZH_staticstrings);
      break;
#endif

#if defined(SUPPORT_VERSION_3)
    case 3:
      machine.packtype = packed_v3;
      break;
#endif

    default:
      zmachine_fatal("Version %i not supported", version);
    }
  
  while (pc != -1)
    {
      instr = GetCode(pc);

#ifdef DEBUG
      printf("PC = %x\n", pc);
#endif

#ifdef SAFE
      if (pc < 0 || pc > machine.story_length)
	zmachine_fatal("PC set to a value outside the story file");
#endif

      /*
       * This bit is a tad confusing :-) What's going on here is that
       * first the interpreter checks for a 'general' instruction,
       * (using a specialised interpreter generated by builder)
       * common to all interpreters. If it finds one, it executes it
       * and loops. If it doesn't find one, it then checks for a
       * version-specific instruction.
       *
       * The reason this is all done with gotos rather than functions
       * is one of speed - the cost of function calls is enough to
       * negate the benefit of specialisation!
       *
       * Note that this can be specialised further; this is a
       * compromise on size and compile time. First, the specialiser
       * currently does not output specialised code for decoding 2OPs
       * in their variable form, prefering to call the decoder
       * function (this is not a big performance hit, though, as the
       * decoder itself is specialised). Second, the 'general' and
       * 'version-specific' interpreters could be combined into
       * one. Unfortunately, this rather increases code size, and only 
       * provides benefits on operations that change between versions.
       * Third, we could use different interpreters for versions 5, 7, 
       * & 8 - this would speed up manipulation of packed addresses
       * (though once again for a size penalty)
       */
      
#include "interp_gen.h"
      goto loop;
      
    version:
      switch(version)
	{
#ifdef SUPPORT_VERSION_3
	case 3:
	  goto version3;
#endif
#ifdef SUPPORT_VERSION_4
	case 4:
	  goto version4;
#endif
#ifdef SUPPORT_VERSION_5
	case 5:
	case 7:
	case 8:
	  goto version5;
#endif
#ifdef SUPPORT_VERSION_6
	case 6:
	  goto version6;
#endif
	}
      zmachine_fatal("Unsupported version");
      
#ifdef SUPPORT_VERSION_3
    version3:
#include "interp_z3.h"
      goto loop;
#endif
      
#ifdef SUPPORT_VERSION_4
    version4:
#include "interp_z4.h"
      goto loop;
#endif
      
#ifdef SUPPORT_VERSION_5
    version5:
#include "interp_z5.h"
      goto loop;
#endif
      
#ifdef SUPPORT_VERSION_6
    version6:
#include "interp_z6.h"
      goto loop;
#endif

    loop:
    }
}
