/*            Copyright (C) 2001, 2002, 2003 Stijn van Dongen
 *
 * This file is part of Zoem. You can redistribute and/or modify Zoem under the
 * terms of the GNU General Public License;  either version 2 of the License or
 * (at your option) any later  version.  You should have received a copy of the
 * GPL along with Zoem, in the file COPYING.
*/

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <sys/wait.h>

#include <limits.h>
#include <string.h>
#include <ctype.h>

#include "filter.h"
#include "read.h"
#include "file.h"
#include "parse.h"
#include "iface.h"
#include "segment.h"
#include "curly.h"
#include "util.h"
#include "key.h"

#include "util/io.h"
#include "util/hash.h"
#include "util/types.h"
#include "util/err.h"
#include "util/ding.h"

static   mcxHash*    mskTable_g        =  NULL;    /* masked input files*/

/*
 * receives mcxIO object open for reading.  The function's code is somewhat
 * ugly. Not yet sure how it should be refactored.
 *
 * This code also finds and stores inline files.
 * It is not error/memory clean and contains hard exits.
*/

mcxTing*  readFile
(  mcxIO          *xf
,  mcxTing        *filetxt
,  mcxbits        flags
,  int            *linecount
)
#define BUFLEN 1024
   {  char     cbuf[BUFLEN]
   ;  short    c
   ;  int      i           =  0
   ;  int      esc         =  0
   ;  int      lc          =  0
   ;  mcxbool  masking     =  flags & READ_MASKING
   ;  struct stat mystat
   ;  size_t   sz          =  1024

   ;  if (!xf->stdio)
      {  if (stat(xf->fn->str, &mystat))
         mcxIOerr(xf, "readFile", "can not stat file")
      ;  else
         sz = mystat.st_size
   ;  }

      if (!xf->fp)   /* fixme; shy so harsh ? */
         mcxIOerr(xf, "readFile PBD", "is not open")
      ,  mcxExit(1)

   ;  filetxt = mcxTingEmpty(filetxt, sz)

   ;  if (0)
      fprintf(stderr, "allocing %d bytes\n", (int) filetxt->mxl)

   ;  if (!mskTable_g)         
      mskTable_g = yamHashNew(5)

   ;  while
      (  c = mcxIOstep(xf)
      ,  c != EOF
      )
      {  int discardnl = 0

      ;  if (c == '\n')
         lc++

      ;  if (esc)
         {  esc = 0

         ;  if (c == ':')
            {  i--            /* remove backslash just written */

            ;  c = mcxIOstep(xf)  

            ;  if (c == '!')
               c = '\\'
            ;  else
               {  if (c == '/')
                  discardnl = 1
               ;  while (c != '\n' && c != EOF)
                  c = mcxIOstep(xf)
            ;  }

               if (c == EOF)
               break
              /* else write the newline or the bs below  */
         ;  }

            else if (c == '=' && !masking)
            {  mcxKV    *kv
            ;  mcxTing  *masked, *fname = NULL
            ;  mcxTing  *line    =  mcxIOreadLine(xf, NULL, MCX_READLINE_CHOMP)
            ;  mcxTing  *jumplc  =  mcxTingEmpty(NULL, 30)
            ;  int      l        =  yamClosingCurly(line,0,NULL,RETURN_ON_FAIL)
            ;  int      sublc    =  0

            ;  if (l<0)
               {  yamErr("readFile", "format err in \\={fname} specification")
               ;  mcxExit(1)
            ;  }
               else
               fname = mcxTingNNew(line->str+1, l-1)

            ;  mcxTingFree(&line)
            ;  kv = mcxHashSearch(fname, mskTable_g, MCX_DATUM_INSERT)
            ;  if (kv->key != fname)
               {  yamErr("readFile", "file <%s> already masked", fname->str)
               ;  mcxExit(1)
            ;  }
               mcxTell("zoem", "reading inline file <%s>", fname->str)
            ;  masked   =  readFile(xf, NULL, 1, &sublc)
            ;  kv->val  =  masked

            ;  mcxTingPrint(jumplc, "\\___jump_lc___{%d}\n", sublc+1)
            ;  mcxTingAppend(filetxt, jumplc->str)
            ;  if (0) fprintf(stderr, "<%s>\n", jumplc->str)

            ;  i--            /* remove the backslash */
            ;  mcxTingFree(&jumplc)
            ;  continue       /* do not write the '=' */
         ;  }

            else if (c == '=' && masking)
            {  mcxTing* line = mcxIOreadLine(xf, NULL, MCX_READLINE_CHOMP)
            ;  mcxTell(NULL, "=== done reading inline file")
            ;  i--            /* get rid of backslash          */

            ;  if (line->str[0] != '=')
               {  yamErr("zoem","expecting closing '\\==' at line <%d>",xf->lc)
               ;  mcxExit(1)
            ;  }
               mcxTingFree(&line)

            ;  if (i)
               {  cbuf[i]  =  '\0'
               ;  mcxTingAppend(filetxt, cbuf)
            ;  }

               if (linecount)
               *linecount = lc
            ;  else
               {  yamErr("zoem PBD", "should have linecount pointer")
               ;  mcxExit(1)
            ;  }

               return filetxt
         ;  }

         /* else if c == x    write it below                   */
      ;  }

         else if (c == '\\')
         {  esc = 1           /* write the backslash below     */
      ;  }

         if (discardnl && c == '\n')
         continue

      ;  cbuf[i] =  (char) c

      ;  if (i == BUFLEN-2)
         {  if (esc)
            {  mcxIOstepback(c, xf)
            ;  esc = 0
            ;  i--
         ;  }
            cbuf[i+1] =  '\0'
         ;  mcxTingNAppend(filetxt, cbuf, i+1)
         ;  i =  0
      ;  }
         else
         i++
   ;  }

      if (0)
      fprintf(stderr, "read <%d> bytes\n", xf->bc)

   ;  if (masking)
      {  yamErr("zoem", "masking scope not closed!\n")
      ;  mcxExit(1)
   ;  }

      if (i)
      {  cbuf[i]  =  '\0'
      ;  mcxTingAppend(filetxt, cbuf)
   ;  }

      if (linecount)
      *linecount = lc

   ;  return filetxt

#undef BUFLEN
;  }


void yamReadExit
(  void
)
   {  if (mskTable_g)
      mcxHashFree(&mskTable_g, mcxTingFree_v, mcxTingFree_v)
;  }


mcxbool yamInlineFile
(  mcxTing* fname
,  mcxTing* filetxt
)
   {  mcxKV* kv =    mskTable_g
                  ?  mcxHashSearch(fname, mskTable_g, MCX_DATUM_FIND)
                  :  NULL
   ;  if (kv)
      {  mcxTingWrite(filetxt, ((mcxTing*)kv->val)->str)
      ;  return TRUE
   ;  }
      return FALSE
;  }


void yamUnprotect
(  mcxTing* txt
)
   {  char* p  =  txt->str
   ;  char* q  =  txt->str
   ;  int   esc=  0  
   
   ;  while(p<txt->str+txt->len)
      {
         if (esc)
         {  if (*p == '\\' || *p == '{' || *p == '}')
            *(q++) = *p
         ;  else
            {  *(q++) = '\\'
            ;  *(q++) = *p
         ;  }
            esc = 0
      ;  }
         else if (*p == '\\')
         esc = 1
      ;  else
         *(q++) = *p

      ;  p++
   ;  }

      *q = '\0'
   ;  txt->len = q-txt->str
;  }


mcxTing*  yamProtect
(  mcxTing* data
)
#define BUFLEN 1024
   {  int      i     =   0
   ;  int      j     =   0
   ;  int      n     =   data->len + (data->len/8)

   ;  mcxTing* protected   =  mcxTingEmpty(NULL, n)

   ;  for(i=0;i<data->len;i++)
      {  char c      =  *(data->str+i)
      ;  int esc     =  (c == '\\' || c == '{' || c == '}') ? 1 : 0

      ;  if (j+esc >= protected->mxl)
         mcxTingEnsure(protected, protected->mxl + (protected->mxl/3) + 2)

      ;  if (esc)
         *(protected->str+j++) = '\\'

      ;  *(protected->str+j++) = c
   ;  }

      protected->len =  j
   ;  *(protected->str+j) = '\0'
   ;  return protected
#undef BUFLEN
;  }


mcxTing*  yamReadData
(  mcxIO          *xf
,  mcxTing        *filetxt
)
#define BUFLEN 1024
   {  char     cbuf[BUFLEN]
   ;  short    c
   ;  int      i        =  0

   ;  if (!xf->fp)
         mcxIOerr(xf, "yamReadData PBD", "is not open")
      ,  mcxExit(1)

   ;  filetxt           =   mcxTingEmpty(filetxt, 1)

   ;  while
      (  c = mcxIOstep(xf)
      ,  c != EOF
      )
      {  if (c == '\\' || c == '{' || c == '}')
         cbuf[i++] =  (char) '\\'

      ;  cbuf[i] = (char) c

      ;  if (i+2 >= BUFLEN)      /* if not, no room for double bs */
         {  mcxTingNAppend(filetxt, cbuf, i+1)
         ;  i = 0
         ;  continue
      ;  }

         i++
   ;  }

      if (i)               /* nothing written at this position */
      mcxTingNAppend(filetxt, cbuf, i)

   ;  xf->ateof   =  1

   ;  return filetxt
#undef BUFLEN
;  }


mcxIO* yamRead__open_file__
(  const char* name
)
   {  mcxIO *xf = mcxIOnew(name, "r")
   ;  if (mcxIOopen(xf, RETURN_ON_FAIL) == STATUS_OK)
      return xf
   ;  else
      mcxIOfree(&xf)
   ;  return NULL
;  }


/*
 *  Receives file name and tries to open it, possibly by searching
 *  in different locations.
 *
 *  NOTE. may change fname by prepending the location in which it was found.
*/

mcxstatus yamReadFile
(  mcxTing*  fname
,  mcxTing*  filetxt
,  mcxbool   useSearchPath
)
#define LDB (tracing_g & ZOEM_TRACE_IO)
   {  mcxIO *xf = NULL
   ;  char* envp

   ;  if (fname->len > 1024)
      {  yamErr("yamReadFile", "input file name expansion too long (>1024)\n")
      ;  return STATUS_FAIL
   ;  }

   /* .  first check inline file
    * .  then check $ZOEMSEARCHPATH
    * .  then check __searchpath__
    * .  then check yamInputGetPath (last file openened)
   */

   ;  if (yamInlineFile(fname, filetxt))
      {  if (LDB)
         mcxTell(NULL, "=== using inline file <%s>", fname->str)
      ;  return STATUS_OK
   ;  }

      if (LDB)
      printf("<IO>| trying (cwd) <%s>\n", fname->str)
   ;  xf = yamRead__open_file__(fname->str)

   /*
    * Try to find a readable file in a few more places.
   */

   ;  if (!xf && (envp = getenv("ZOEMSEARCHPATH")))
      {  mcxTing* zsp = mcxTingNew(envp)
      ;  mcxTing* file  = mcxTingEmpty(NULL, 30)
      ;  char *q, *p = zsp->str
      ;  mcxTingTr(zsp, ": \n\t\f\r", "      ", TR_SQUASH)
      ;  while (p)
         {  while (isspace(*p))
            p++
         ;  q = mcxStrChrIs(p, isspace, -1)
         ;  if (q)
            mcxTingNWrite(file, p, q-p)
         ;  else
            mcxTingWrite(file, p)

         ;  mcxTingAppend(file, fname->str)
         ;  if (LDB)
            printf("<IO>| trying (env) <%s>\n", file->str)
         ;  if ((xf = yamRead__open_file__(file->str)))
            {  mcxTingWrite(fname, file->str)
            ;  break
         ;  }
            p = q
      ;  }
         mcxTingFree(&zsp)
      ;  mcxTingFree(&file)
   ;  }

      if (!xf && useSearchPath)
      {  
         mcxTing* tmp = mcxTingNew("__searchpath__")
      ;  mcxTing* file  = mcxTingEmpty(NULL, 30)
      ;  mcxTing* sp  = yamKeyGet(tmp)
      ;  yamSeg*  spseg = sp ? yamSegPush(NULL, sp) : NULL

      ;  while (sp && yamParseScopes(spseg, 1, 0) == 1)
         {  mcxTingPrint(file, "%s/%s", arg1_g->str, fname->str)
         ;  if (LDB)
            printf("<IO>| trying (var) <%s>\n", file->str)
         ;  if ((xf = yamRead__open_file__(file->str)))
            {  mcxTingWrite(fname, file->str)
            ;  break
         ;  }
         }

         mcxTingFree(&file)
      ;  mcxTingFree(&tmp)
      ;  yamSegFree(&spseg)
   ;  }

      if (!xf)
      {  mcxTing* file = yamInputGetPath()
      ;  if (file)
         {  if (LDB)
            printf
            ("<IO>| trying (his) <%s> path stripped from last include\n"
            ,  file->str
            )
         ;  mcxTingAppend(file, fname->str)
         ;  xf = yamRead__open_file__(file->str)
         ;  mcxTingFree(&file)
      ;  }
      }

      if (xf)
      {  readFile(xf, filetxt, 0, NULL)
      ;  if (LDB)
         printf("<IO>| found <%s>\n", xf->fn->str)
      ;  mcxIOfree(&xf)
      ;  return STATUS_OK
   ;  }

      return STATUS_FAIL
#undef LDB
;  }


/*
 *    I had some trouble getting the details right, and perhaps the stuff below
 *    is stupid in places. I had problems getting rid of duplicate output; The
 *    fflush(NULL) solves that.  These are the different cases that once caused
 *    trouble: Assume a file t.azm containing
 *
 *    \write{bar}{txt}{1 2 3 4 hoedje van papier}
 *    \system{date}{}{}
 *    \system{date}{}{}
 *    \system{date}{}{}
 *    \system{date}{}{}
 *
 *    and consider the command lines
 *
 *    zoem -i t -o -
 *    zoem -i t -o foo
 *    zoem -i t -o - > foo
 *
 *    All these cases were buggy under different circumstances; pipe ends need
 *    to be closed, data needs to be flushed, and open file descriptors
 *    need to be closed.
*/

mcxTing* yamSystem
(  char* cmd
,  char* argv[]
,  mcxTing* data
)
#define LDB (tracing_g & ZOEM_TRACE_SYSTEM)
#define  RB_SIZE 2048
   {  int   stat1, stat2, xstat, k
   ;  int   fx[2]
   ;  int   fy[2]
   ;  pid_t cp1, cp2
   ;  char  rb[RB_SIZE]

   ;  int statusx =  pipe(fx)
   ;  int statusy =  pipe(fy)

   ;  if (statusx && statusy)
      ;  /* elude compiler warning */

   ;  yamUnprotect(data)
   ;  if (fflush(NULL))
      perror("flush")

   ;  cp1 = fork()

   ;  if (cp1 == -1)       /* error */
      {  yamErr("\\system#3", "fork error")
      ;  perror("\\system#3")
      ;  return NULL
   ;  }
      else if (cp1 == 0)  /* child */
      {  if (LDB)
         printf
         (" sys| CHILD closing x1=%d and y0=%d\n", (int) fx[1], (int) fy[0])

      ;  if (close(fx[1])<0)
         perror("close fx[1]")  /* close parent output */
      ;  if (close(fy[0])<0)
         perror("close fy[0]")  /* close parent input  */

      ;  if (LDB)
         {  printf
            (" sys| CHILD mapping %d to %d\n",(int)fx[0], (int) fileno(stdin))
         ;  printf
            (" sys| CHILD mapping %d to %d\n",(int)fy[1], (int) fileno(stdout))
      ;  }

         dup2(fx[0], fileno(stdin))
      ;  dup2(fy[1], fileno(stdout))

      ;  if (close(fx[0])<0)
         perror("close fx[0]")
      ;  if (close(fy[1])<0)
         perror("close fy[1]")

      ;  switch ((cp2 = fork()))
         {  case -1
         :  perror("fork")
         ;  exit(1)
         ;

            case 0               /* grandchild */
         :  execvp(cmd, argv)    /* do the actual stuff */
         ;  yamErr("\\system#3", "exec <%s> failed (bummer)", cmd)
         ;  if (0)
            perror("exec failed (bummer)")
         ;  exit(1)
         ;

            default              /* child */
         :  if (waitpid(cp2, &stat2, 0)< 0 || !WIFEXITED(stat2))
            {  if (0)
               perror("child")
         ;  }

            while (1 && (k = read(fileno(stdin), rb, RB_SIZE)))
            {  if (LDB)
               fprintf
               (  stderr
               ," sys| CHILD discarding %d bytes\n", (int) k
               )
         ;  }
           /* The discard loop is for this:
            * \system{cat}{{-#}}{abcdef} (i.e. failing command)
            * we need someone to read the stdin, lest we get a broken pipe.
            * (and we want \system never to fail in zoem).
           */

           /*  Do I need the close below? e.g. wrt 'zoem -i t -o - > foo'.  I'm
            *  not too clear on these issues, but it certainly seems best
            *  practice to close all pipe ends when done.
           */
         ;  if (close(fileno(stdin))<0)
            perror("close fileno(stdin)")
         ;  if (close(fileno(stdout))<0)
            perror("close fileno(stdout)")
         ;  if (stat2)
            yamErr("\\system#3", "child <%s> failed", cmd)
         ;  exit(stat2 ? 1 : 0)
      ;  }
      }
      else /* parent */
      {  mcxTing* buf   =  mcxTingEmpty(NULL, 80)
      ;  mcxTing* outp
      ;  static int ct = 0

      ;  if (LDB)
         printf
         (" sys| PARENT closing x0=%d and y1=%d\n", (int) fx[0], (int) fy[1])

      ;  close(fy[1])                  /* close child output     */
      ;  close(fx[0])                  /* close child input      */
      ;  if (data->len)                /* mq, must not exceed buffer size */
         {  fprintf(stderr, "<%d> bytes written\n", (int) data->len)
         ;  write(fx[1], data->str, data->len)
      ;  }
         close(fx[1])

      ;  ct++

      ;  while ((k = read(fy[0], rb, RB_SIZE)))
         {  if (k<0)
            {  perror("read")
            ;  break
         ;  }
            if (buf->len + k > buf->mxl)
            mcxTingEnsure(buf, buf->mxl  + buf->mxl/2)
         ;  mcxTingNAppend(buf, rb, k)
      ;  }
         close(fy[0])
      ;  if ((waitpid(cp1, &stat1, 0)) < 0 || !WIFEXITED(stat1))
         perror("waiting parent")
      ;  xstat = WIFEXITED(stat1) ? WEXITSTATUS(stat1) : 1
      ;  yamKeySet("__sysval__", xstat ? "1" : "0")
      ;  outp = yamProtect(buf)
      ;  mcxTingFree(&buf)
      ;  return outp
   ;  }
      return NULL
;  }
#undef LDB
#undef RB_SIZE

