/*
 * zap -- interactive process killer 
 * Based on an earlier program of the same name
 * by Kernighan & Pike, in their classic book
 * The UNIX Programming Environment, 1984.
 *
 * $Id: zap.c,v 1.8 2002/03/06 00:44:03 ian Exp $
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <ctype.h>

#include "zap.h"

char	*progname;	/* program name for error message */
const char	*ps = PS_ALL;
int		progpid;
static int debug = 0;		/* print debugging? */

/* Short for sending the signal and printing an error message if needed. */
void
do_kill(pid_t pid, int signum) {
		if (debug) {
			fprintf(stderr, "%s: kill(%u, %u)\n", progname, pid, signum);
		}
		if (kill(pid, signum)<0) {
			char msgbuf[128];
			sprintf(msgbuf, "kill(%d, %d)", pid, signum);
			perror(msgbuf);
		}
}

int
main(int argc, char **argv)
{
	FILE *fin, *popen();
	char buf[BUFSIZ];	/* a line to read */
	int errors = 0;		/* for getopt */
	int c = 0;			/* from getopt */
	extern int optind;	/* from getopt */
	extern char* optarg;/* from getopt */
	int pid;			/* target PID */
	int interact = 1;	/* ? ask : just do it */
	int signum = SIGTERM;	/* how shall I tell thee? */

	progname = argv[0];
	progpid  = getpid();

	while ((c = getopt(argc, argv, "dfs:y")) != EOF) {
		switch (c) {
		case 's':
			if (!isdigit(*optarg)) {
				if ((signum = signame(optarg)) < 0) {
					fprintf(stderr, "%s: signame %s unknown, using TERM\n",
						progname, optarg);
					signum = SIGTERM;
				}
			} else {
				signum = atoi(optarg);
			}
			break;
		case 'd':
			++debug;
			break;
		case 'f':
		case 'y':
			interact = 0;
			break;
		case '?':
		default:
			errors++;
			break;
		}
	}
	if (errors || (argc-optind<1)) {
		(void) fprintf(stderr,
			"usage: %s [-dfy] [-s signum] processname ...\n", progname);
		exit(2);
	}

	if (debug) {
		fprintf(stderr, "signum = %d\n", signum);
	}

	if ((fin = popen(ps, "r")) == NULL) {
		fprintf(stderr, "%s: can't run %s\n", progname, ps);
		exit(1);
	}
	fgets(buf, sizeof buf, fin);	/* get header line */
	fprintf(stderr, "%s", buf);

	/* Read a line from ps, process it. */
	while (fgets(buf, sizeof buf, fin) != NULL) {
		buf[strlen(buf)-1] = '\0'; /* suppress \n */
		if (strstr(buf, argv[optind]) != NULL && 
				(pid = atoi(buf)) != progpid) {

			fprintf(stderr, "%s", buf);

			if (interact) {
				fprintf(stderr, "? ");
				if (ttyin() == 'y') {
					do_kill(pid, signum);
				}
			} else {
				fprintf(stderr, "\n");
				do_kill(pid, signum);
			}
		}
	}
	exit(0);
}
