/*
 * xvkbd - Virtual Keyboard for X Window System
 * (Version 2.4, 2002-10-02)
 *
 * Copyright (C) 2000-2002 by Tom Sato <VEF00200@nifty.ne.jp>
 * http://member.nifty.ne.jp/tsato/
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 */
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>
#include <string.h>
#include <signal.h>
#include <errno.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/keysym.h>
#include <X11/cursorfont.h>
#include <X11/Xproto.h>  /* to get request code */
#include <X11/Xaw/Box.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Repeater.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/MenuButton.h>
#include <X11/Xaw/SimpleMenu.h>
#include <X11/Xaw/SmeBSB.h>
#include <X11/Xaw/SmeLine.h>
#include <X11/Xaw/AsciiText.h>
#include <X11/Xaw/Viewport.h>
#include <X11/Xaw/List.h>
#include <X11/Xmu/WinUtil.h>

#ifdef USE_I18N
# include <X11/Xlocale.h>
#endif

#ifdef USE_XTEST
# include <X11/extensions/XTest.h>
#endif

#include "resources.h"
#define PROGRAM_NAME_WITH_VERSION "xvkbd (v2.4)"

/*
 * Default keyboard layout is hardcoded here.
 * Layout of the main keyboard can be redefined by resources.
 */
#define NUM_KEY_ROWS     6
#define NUM_KEY_COLS    20

static char *keys_normal[NUM_KEY_ROWS][NUM_KEY_COLS] = {
  { "F1", "F2", "F3", "F4", "F5", "F6", "F7", "F8", "F9", "F10", "F11", "F12", "BackSpace" },
  { "Escape", "1", "2", "3", "4", "5", "6", "7", "8", "9", "0", "-", "=", "\\", "`" },
  { "Tab", "q", "w", "e", "r", "t", "y", "u", "i", "o", "p", "[", "]", "Delete" },
  { "Control_L", "a", "s", "d", "f", "g", "h", "j", "k", "l", ";", "'", "Return" },
  { "Shift_L", "z", "x", "c", "v", "b", "n", "m", ",", ".", "/", "Multi_key", "Shift_R" },
  { "MainMenu", "Caps_Lock", "Alt_L", "Meta_L", "space", "Meta_R", "Alt_R",
    "Left", "Right", "Up", "Down", "Focus" },
};
static char *keys_shift[NUM_KEY_ROWS][NUM_KEY_COLS] = {
  { "F1", "F2", "F3", "F4", "F5", "F6", "F7", "F8", "F9", "F10", "F11", "F12", "BackSpace" },
  { "Escape", "!", "@", "#", "$", "%", "^", "&", "*", "(", ")", "_", "+", "|", "~" },
  { "Tab", "Q", "W", "E", "R", "T", "Y", "U", "I", "O", "P", "{", "}", "Delete" },
  { "Control_L", "A", "S", "D", "F", "G", "H", "J", "K", "L", ":", "\"", "Return" },
  { "Shift_L", "Z", "X", "C", "V", "B", "N", "M", "<", ">", "?", "Multi_key", "Shift_R" },
  { "MainMenu", "Caps_Lock", "Alt_L", "Meta_L", "space", "Meta_R", "Alt_R",
    "Left", "Right", "Up", "Down", "Focus" },
};
static char *keys_altgr[NUM_KEY_ROWS][NUM_KEY_COLS] = { { NULL } };
static char *keys_shift_altgr[NUM_KEY_ROWS][NUM_KEY_COLS] = { { NULL } };

static char *key_labels[NUM_KEY_ROWS][NUM_KEY_COLS] = {
  { "F1", "F2", "F3", "F4", "F5", "F6", "F7", "F8", "F9", "F10", "F11", "F12", "Backspace" },
  { "Esc", "!\n1", "@\n2", "#\n3", "$\n4", "%\n5", "^\n6",
    "&\n7", "*\n8", "(\n9", ")\n0", "_\n-", "+\n=", "|\n\\", "~\n`" },
  { "Tab", "Q", "W", "E", "R", "T", "Y", "U", "I", "O", "P", "{\n[", "}\n]", "Del" },
  { "Control", "A", "S", "D", "F", "G", "H", "J", "K", "L", ":\n;", "\"\n'", "Return" },
  { "Shift", "Z", "X", "C", "V", "B", "N", "M", "<\n,", ">\n.", "?\n/", "Com\npose", "Shift" },
  { "MainMenu", "Caps\nLock", "Alt", "Meta", "", "Meta", "Alt",
    "left", "right", "up", "down", "Focus" },
};
static char *normal_key_labels[NUM_KEY_ROWS][NUM_KEY_COLS] = {
  { "F1", "F2", "F3", "F4", "F5", "F6", "F7", "F8", "F9", "F10", "F11", "F12", "Backspace" },
  { "Esc", "1", "2", "3", "4", "5", "6", "7", "8", "9", "0", "-", "=", "\\", "`" },
  { "Tab", "q", "w", "e", "r", "t", "y", "u", "i", "o", "p", "[", "]", "Del" },
  { "Ctrl", "a", "s", "d", "f", "g", "h", "j", "k", "l", ";", "'", "Return" },
  { "Shift", "z", "x", "c", "v", "b", "n", "m", ",", ".", "/", "Comp", "Shift" },
  { "MainMenu", "Caps", "Alt", "Meta", "", "Meta", "Alt",
    "left", "right", "up", "down", "Focus" },
};
static char *shift_key_labels[NUM_KEY_ROWS][NUM_KEY_COLS] = {
  { "F1", "F2", "F3", "F4", "F5", "F6", "F7", "F8", "F9", "F10", "F11", "F12", "Backspace" },
  { "Esc", "!", "@", "#", "$", "%", "^", "&", "*", "(", ")", "_", "+", "|", "~" },
  { "Tab", "Q", "W", "E", "R", "T", "Y", "U", "I", "O", "P", "{", "}", "Del" },
  { "Ctrl", "A", "S", "D", "F", "G", "H", "J", "K", "L", ":", "\"", "Return" },
  { "Shift", "Z", "X", "C", "V", "B", "N", "M", "<", ">", "?", "Comp", "Shift" },
  { "MainMenu", "Caps", "Alt", "Meta", "", "Meta", "Alt",
    "left", "right", "up", "down", "Focus" },
};
static char *altgr_key_labels[NUM_KEY_ROWS][NUM_KEY_COLS] = { { NULL } };
static char *shift_altgr_key_labels[NUM_KEY_ROWS][NUM_KEY_COLS] = { { NULL } };


#define NUM_KEYPAD_ROWS  5
#define NUM_KEYPAD_COLS  5

static char *keypad[NUM_KEYPAD_ROWS][NUM_KEYPAD_COLS] = {
  { "Num_Lock",  "KP_Divide",   "KP_Multiply", "Focus"       },
  { "Home",      "Up",          "Page_Up",     "KP_Add"      },
  { "Left",      "5",           "Right",       "KP_Subtract" },
  { "End",       "Down",        "Page_Down",   "KP_Enter"    },
  { "Insert",                   "Delete"                     },
};
static char *keypad_shift[NUM_KEYPAD_ROWS][NUM_KEYPAD_COLS] = {
  { "Num_Lock",  "KP_Divide",   "KP_Multiply", "Focus"       },
  { "KP_7",      "KP_8",        "KP_9",        "KP_Add"      },
  { "KP_4",      "KP_5",        "KP_6",        "KP_Subtract" },
  { "KP_1",      "KP_2",        "KP_3",        "KP_Enter"    },
  { "KP_0",                     "."                          },
};
static char *keypad_label[NUM_KEYPAD_ROWS][NUM_KEYPAD_COLS] = {
  { "Num\nLock", "/",           "*",           "Focus"       },
  { "7\nHome",   "8\nUp  ",     "9\nPgUp",     "+"           },
  { "4\nLeft",   "5\n    ",     "6\nRight",    "-"           },
  { "1\nEnd ",   "2\nDown",     "3\nPgDn",     "Enter"       },
  { "0\nIns ",                  ".\nDel "                    },
};

#define NUM_SUN_FKEY_ROWS 6
#define NUM_SUN_FKEY_COLS 3

static char *sun_fkey[NUM_SUN_FKEY_ROWS][NUM_SUN_FKEY_COLS] = {
  { "L1", "L2"  },
  { "L3", "L4"  },
  { "L5", "L6"  },
  { "L7", "L8"  },
  { "L9", "L10" },
  { "Help"      },
};
static char *sun_fkey_label[NUM_SUN_FKEY_ROWS][NUM_SUN_FKEY_COLS] = {
  { "Stop \nL1", "Again\nL2"  },
  { "Props\nL3", "Undo \nL4"  },
  { "Front\nL5", "Copy \nL6"  },
  { "Open \nL7", "Paste\nL8"  },
  { "Find \nL9", "Cut  \nL10" },
  { "Help"                    },
};

/*
 * Image for arrow keys
 */
#define up_width 7
#define up_height 13
static unsigned char up_bits[] = {
   0x08, 0x1c, 0x1c, 0x3e, 0x2a, 0x49, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08,
   0x08};

#define down_width 7
#define down_height 13
static unsigned char down_bits[] = {
   0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x49, 0x2a, 0x3e, 0x1c, 0x1c,
   0x08};

#define left_width 13
#define left_height 13
static unsigned char left_bits[] = {
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x18, 0x00, 0x0e, 0x00,
   0xff, 0x1f, 0x0e, 0x00, 0x18, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00};

#define right_width 13
#define right_height 13
static unsigned char right_bits[] = {
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x03, 0x00, 0x0e,
   0xff, 0x1f, 0x00, 0x0e, 0x00, 0x03, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00};

#define check_width 16
#define check_height 16
static unsigned char check_bits[] = {
  0x00, 0x00, 0x00, 0x00, 0x00, 0x70, 0x00, 0x38, 0x00, 0x1e, 0x08, 0x0f,
  0x8c, 0x07, 0xde, 0x03, 0xfe, 0x03, 0xfc, 0x01, 0xf8, 0x00, 0xf0, 0x00,
  0x70, 0x00, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00};

#define back_width 18
#define back_height 13
static unsigned char back_bits[] = {
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0x00, 0x00,
   0x78, 0x00, 0x00, 0xfe, 0xff, 0x03, 0xff, 0xff, 0x03, 0xfe, 0xff, 0x03,
   0x78, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00};

/*
 * Resources and options
 */
#define Offset(entry) XtOffset(struct appres_struct *, entry)
static XtResource application_resources[] = {
  { "description", "Description", XtRString, sizeof(char *),
    Offset(description), XtRImmediate,
    PROGRAM_NAME_WITH_VERSION " - virtual keyboard for X window system\n\n"
    "Copyright (C) 2000-2002 by Tom Sato <VEF00200@nifty.ne.jp>\n"
    "http://member.nifty.ne.jp/tsato/\n\n"
    "This program is free software with ABSOLUTELY NO WARRANTY,\n"
    "distributed under the terms of the GNU General Public License.\n" },
  { "showManualCommand", "ShowManualCommand", XtRString, sizeof(char *),
    Offset(show_manual_command), XtRImmediate, "xterm -e man xvkbd &" },

  { "windowGeometry", "Geometry", XtRString, sizeof(char *),
    Offset(geometry), XtRImmediate, "" },
  { "debug", "Debug", XtRBoolean, sizeof(Boolean),
     Offset(debug), XtRImmediate, (XtPointer)FALSE },
  { "xtest", "XTest", XtRBoolean, sizeof(Boolean),
     Offset(xtest), XtRImmediate, (XtPointer)TRUE },
  { "jumpPointer", "JumpPointer", XtRBoolean, sizeof(Boolean),
     Offset(jump_pointer), XtRImmediate, (XtPointer)TRUE },
  { "jumpPointerAlways", "JumpPointer", XtRBoolean, sizeof(Boolean),
     Offset(jump_pointer_always), XtRImmediate, (XtPointer)TRUE },
  { "altgrLock", "ModifiersLock", XtRBoolean, sizeof(Boolean),
     Offset(altgr_lock), XtRImmediate, (XtPointer)TRUE },
  { "shiftLock", "ModifiersLock", XtRBoolean, sizeof(Boolean),
     Offset(shift_lock), XtRImmediate, (XtPointer)FALSE },
  { "modifiersLock", "ModifiersLock", XtRBoolean, sizeof(Boolean),
     Offset(modifiers_lock), XtRImmediate, (XtPointer)FALSE },
  { "autoRepeat", "AutoRepeat", XtRBoolean, sizeof(Boolean),
     Offset(auto_repeat), XtRImmediate, (XtPointer)TRUE },
  { "modalKeytop", "ModalKeytop", XtRBoolean, sizeof(Boolean),
     Offset(modal_keytop), XtRImmediate, (XtPointer)FALSE },
  { "modalKeytop", "ModalKeytop", XtRBoolean, sizeof(Boolean),
     Offset(modal_keytop), XtRImmediate, (XtPointer)FALSE },
  { "modalThreshold", "ModalThreshold", XtRInt, sizeof(int),
     Offset(modal_threshold), XtRImmediate, (XtPointer)150 },
  { "keypad", "Keypad", XtRBoolean, sizeof(Boolean),
     Offset(keypad), XtRImmediate, (XtPointer)TRUE },
  { "functionkey", "FunctionKey", XtRBoolean, sizeof(Boolean),
     Offset(function_key), XtRImmediate, (XtPointer)TRUE },
  { "compact", "Compact", XtRBoolean, sizeof(Boolean),
     Offset(compact), XtRImmediate, (XtPointer)FALSE },
  { "keypadOnly", "KeypadOnly", XtRBoolean, sizeof(Boolean),
     Offset(keypad_only), XtRImmediate, (XtPointer)FALSE },
  { "keypadKeysym", "KeypadKeysym", XtRBoolean, sizeof(Boolean),
     Offset(keypad_keysym), XtRImmediate, (XtPointer)FALSE },
  { "autoAddKeysym", "AutoAddKeysym", XtRBoolean, sizeof(Boolean),
     Offset(auto_add_keysym), XtRImmediate, (XtPointer)TRUE },
  { "listWidgets", "Debug", XtRBoolean, sizeof(Boolean),
     Offset(list_widgets), XtRImmediate, (XtPointer)FALSE },
  { "text", "Text", XtRString, sizeof(char *),
    Offset(text), XtRImmediate, "" },
  { "file", "File", XtRString, sizeof(char *),
    Offset(file), XtRImmediate, "" },
  { "window", "Window", XtRString, sizeof(char *),
    Offset(window), XtRImmediate, "" },
  { "widget", "Widget", XtRString, sizeof(char *),
    Offset(widget), XtRImmediate, "" },
  { "generalFont", XtCFont, XtRFontStruct, sizeof(XFontStruct *),
      Offset(general_font), XtRString, XtDefaultFont},
  { "letterFont", XtCFont, XtRFontStruct, sizeof(XFontStruct *),
      Offset(letter_font), XtRString, XtDefaultFont},
  { "specialFont", XtCFont, XtRFontStruct, sizeof(XFontStruct *),
      Offset(special_font), XtRString, XtDefaultFont},
  { "keypadFont", XtCFont, XtRFontStruct, sizeof(XFontStruct *),
      Offset(keypad_font), XtRString, XtDefaultFont},
  { "generalBackground", XtCBackground, XtRPixel, sizeof(Pixel),
     Offset(general_background), XtRString, "gray" },
  { "specialBackground", XtCBackground, XtRPixel, sizeof(Pixel),
     Offset(special_background), XtRString, "gray" },
  { "specialForeground", XtCForeground, XtRPixel, sizeof(Pixel),
     Offset(special_foreground), XtRString, "black" },
#ifdef USE_I18N
  { "specialFontSet", XtCFontSet, XtRFontSet, sizeof(XFontSet),
      Offset(special_fontset), XtRString, XtDefaultFontSet},
#endif
  { "highlightBackground", XtCBackground, XtRPixel, sizeof(Pixel),
     Offset(highlight_background), XtRString, "gray" },
  { "highlightForeground", XtCForeground, XtRPixel, sizeof(Pixel),
     Offset(highlight_foreground), XtRString, "forestgreen" },
  { "focusBackground", XtCBackground, XtRPixel, sizeof(Pixel),
     Offset(focus_background), XtRString, "gray" },
  { "remoteFocusBackground", XtCBackground, XtRPixel, sizeof(Pixel),
     Offset(remote_focus_background), XtRString, "cyan" },
  { "balloonBackground", XtCBackground, XtRPixel, sizeof(Pixel),
     Offset(balloon_background), XtRString, "LightYellow1" },
  { "launchBalloonBackground", XtCBackground, XtRPixel, sizeof(Pixel),
     Offset(launch_balloon_background), XtRString, "SkyBlue1" },

  { "normalkeys", "NormalKeys", XtRString, sizeof(char *),
    Offset(keys_normal), XtRImmediate, "" },
  { "shiftkeys", "ShiftKeys", XtRString, sizeof(char *),
    Offset(keys_shift), XtRImmediate, "" },
  { "altgrkeys", "AltgrKeys", XtRString, sizeof(char *),
    Offset(keys_altgr), XtRImmediate, "" },
  { "shiftaltgrkeys", "ShiftAltgrKeys", XtRString, sizeof(char *),
    Offset(keys_shift_altgr), XtRImmediate, "" },
  { "keylabels", "KeyLabels", XtRString, sizeof(char *),
    Offset(key_labels), XtRImmediate, "" },
  { "normalkeylabels", "NormalKeyLabels", XtRString, sizeof(char *),
    Offset(normal_key_labels), XtRImmediate, "" },
  { "shiftkeylabels", "ShiftKeyLabels", XtRString, sizeof(char *),
    Offset(shift_key_labels), XtRImmediate, "" },
  { "altgrkeylabels", "AltgrKeyLabels", XtRString, sizeof(char *),
    Offset(altgr_key_labels), XtRImmediate, "" },
  { "shiftaltgrkeylabels", "ShiftAltgrKeyLabels", XtRString, sizeof(char *),
    Offset(shift_altgr_key_labels), XtRImmediate, "" },
  { "deadkeys", "DeadKeys", XtRString, sizeof(char *),
    Offset(deadkeys), XtRImmediate, "" },

  { "keyFile", "KeyFile", XtRString, sizeof(char *),
    Offset(key_file), XtRImmediate, ".xvkbd" },
  { "dictFile", "DictFile", XtRString, sizeof(char *),
    Offset(dict_file), XtRImmediate, "/usr/dict/words" },
  { "customizations", "Customizations", XtRString, sizeof(char *),
    Offset(customizations), XtRImmediate, "default" },
  { "editableFunctionKeys", "EditableFunctionKeys", XtRInt, sizeof(int),
     Offset(editable_function_keys), XtRImmediate, (XtPointer)12 },

  { "maxWidthRatio", "MaxRatio", XtRFloat, sizeof(float),
     Offset(max_width_ratio), XtRString, "0.9" },
  { "maxHeightRatio", "MaxRatio", XtRFloat, sizeof(float),
     Offset(max_height_ratio), XtRString, "0.5" },
};
#undef Offset

static XrmOptionDescRec options[] = {
  { "-geometry", ".windowGeometry", XrmoptionSepArg, NULL },
  { "-windowgeometry", ".windowGeometry", XrmoptionSepArg, NULL },
  { "-debug", ".debug", XrmoptionNoArg, "True" },
#ifdef USE_XTEST
  { "-xtest", ".xtest", XrmoptionNoArg, "True" },
  { "-xsendevent", ".xtest", XrmoptionNoArg, "False" },
  { "-no-jump-pointer", ".jumpPointer", XrmoptionNoArg, "False" },
#endif
  { "-text", ".text", XrmoptionSepArg, NULL },
  { "-file", ".file", XrmoptionSepArg, NULL },
  { "-window", ".window", XrmoptionSepArg, NULL },
  { "-widget", ".widget", XrmoptionSepArg, NULL },
  { "-altgr-lock", ".altgrLock", XrmoptionNoArg, "True" },
  { "-no-altgr-lock", ".altgrLock", XrmoptionNoArg, "False" },
  { "-no-repeat", ".autoRepeat", XrmoptionNoArg, "False" },
  { "-norepeat", ".autoRepeat", XrmoptionNoArg, "False" },
  { "-no-keypad", ".keypad", XrmoptionNoArg, "False" },
  { "-nokeypad", ".keypad", XrmoptionNoArg, "False" },
  { "-no-functionkey", ".functionkey", XrmoptionNoArg, "False" },
  { "-nofunctionkey", ".functionkey", XrmoptionNoArg, "False" },
  { "-highlight", ".highlightForeground", XrmoptionSepArg, NULL },
  { "-compact", ".compact", XrmoptionNoArg, "True" },
  { "-keypad", ".keypadOnly", XrmoptionNoArg, "True" },
  { "-true-keypad", ".keypadKeysym", XrmoptionNoArg, "True" },
  { "-truekeypad", ".keypadKeysym", XrmoptionNoArg, "True" },
  { "-no-add-keysym", ".autoAddKeysym", XrmoptionNoArg, "False" },
  { "-list", ".listWidgets", XrmoptionNoArg, "True" },
  { "-modal", ".modalKeytop", XrmoptionNoArg, "True" },
  { "-dict", ".dictFile", XrmoptionSepArg, NULL },
  { "-keyfile", ".keyFile", XrmoptionSepArg, NULL },
  { "-customizations", ".customizations", XrmoptionSepArg, NULL },
};

/*
 * Global variables
 */
static int argc1;
static char **argv1;

static XtAppContext app_con;
static Widget toplevel;
static Widget key_widgets[NUM_KEY_ROWS][NUM_KEY_COLS];
static Widget main_menu = None;

static Dimension toplevel_height = 1000;

static Display *dpy;
static Atom wm_delete_window = None;

static KeySym *keysym_table = NULL;
static int min_keycode, max_keycode;
static int keysym_per_keycode;
static Boolean error_detected;

static int alt_mask = 0;
static int meta_mask = 0;
static int altgr_mask = 0;

static int shift_state = 0;
static int mouse_shift = 0;

static Boolean num_lock_state = FALSE;

static Display *target_dpy = NULL;

static Window toplevel_parent = None;
static Window focused_window = None;
static Window focused_subwindow = None;

static int AddKeysym(KeySym keysym, Boolean top);  /* forward */
static void SendString(const unsigned char *str);
static void MakeKeyboard(Boolean remake);
static void MakeKeypad(Widget form, Widget from_vert, Widget from_horiz);
static void MakeSunFunctionKey(Widget form, Widget from_vert, Widget from_horiz);
static void MakeDeadkeyPanel(Widget form);
static void RefreshMainMenu(void);
static void PopupFunctionKeyEditor(void);

/*
 * Search for window which has specified instance name (WM_NAME)
 * or class name (WM_CLASS).
 */
static Window FindWindow(Window top, char *name)
{
  Window w;
  Window *children, dummy;
  unsigned int nchildren;
  int i;
  XClassHint hint;

  w = None;

  if (appres.debug) fprintf(stderr, "FindWindow: id=0x%lX", (long)top);

  if (XGetClassHint(target_dpy, top, &hint)) {
    if (hint.res_name) {
      if (appres.debug) fprintf(stderr, " instance=\"%s\"", hint.res_name);
      if (strcmp(hint.res_name, name) == 0) w = top;
      XFree(hint.res_name);
    }
    if (hint.res_class) {
      if (appres.debug) fprintf(stderr, " class=\"%s\"", hint.res_class);
      if (strcmp(hint.res_class, name) == 0) w = top;
      XFree(hint.res_class);
    }
  }
  if (appres.debug) fprintf(stderr, "\n");

  if (w == None &&
      XQueryTree(target_dpy, top, &dummy, &dummy, &children, &nchildren)) {
    for (i = 0; i < nchildren; i++) {
      w = FindWindow(children[i], name);
      if (w != None) break;
    }
    if (children) XFree((char *)children);
  }

  return(w);
}

/*
 * This will be called to get window to set input focus,
 * when user pressed the "Focus" button.
 */
static void GetFocusedWindow(void)
{
  Cursor cursor;
  XEvent event;
  Window target_root, child;
  int junk_i;
  unsigned junk_u;
  Window junk_w;
  int scrn;
  int cur_x, cur_y, last_x, last_y;
  double x_ratio, y_ratio;

  XFlush(target_dpy);
  target_root = RootWindow(target_dpy, DefaultScreen(target_dpy));

  cursor = XCreateFontCursor(dpy, (target_dpy == dpy) ? XC_crosshair : XC_dot);
  if (XGrabPointer(dpy, RootWindow(dpy, DefaultScreen(dpy)), False, ButtonPressMask,
                   GrabModeSync, GrabModeAsync, None,
                   cursor, CurrentTime) == 0) {
    if (appres.debug)
      fprintf(stderr, "Grab pointer - waiting for button press\n");
    last_x = -1;
    last_y = -1;
    x_ratio = ((double)WidthOfScreen(DefaultScreenOfDisplay(target_dpy))
	       / WidthOfScreen(XtScreen(toplevel)));
    y_ratio = ((double)HeightOfScreen(DefaultScreenOfDisplay(target_dpy))
	       / HeightOfScreen(XtScreen(toplevel)));
    do {
      XAllowEvents(dpy, SyncPointer, CurrentTime);
      if (target_dpy == dpy) {
	XNextEvent(dpy, &event);
      } else {
	XCheckTypedEvent(dpy, ButtonPress, &event);
	if (XQueryPointer(dpy, RootWindow(dpy, DefaultScreen(dpy)), &junk_w, &junk_w,
			  &cur_x, &cur_y, &junk_i, &junk_i, &junk_u)) {
	  cur_x = cur_x * x_ratio;
	  cur_y = cur_y * y_ratio;
	}
	if (cur_x != last_x || cur_y != last_y) {
	  if (appres.debug) fprintf(stderr, "Moving pointer to (%d, %d) on %s\n",
				    cur_x, cur_y, XDisplayString(target_dpy));
	  XWarpPointer(target_dpy, None, target_root, 0, 0, 0, 0, cur_x, cur_y);
	  XFlush(target_dpy);
	  last_x = cur_x;
	  last_y = cur_y;
	  XQueryPointer(target_dpy, target_root, &junk_w, &child,
			&cur_x, &cur_y, &junk_i, &junk_i, &junk_u);
	  usleep(10000);
	} else {
	  usleep(100000);
	}
      }
    } while (event.type != ButtonPress);
    XUngrabPointer(dpy, CurrentTime);

    focused_window = None;
    if (target_dpy == dpy) focused_window = event.xbutton.subwindow;
    if (focused_window == None) {
      XFlush(target_dpy);
      for (scrn = 0; scrn < ScreenCount(target_dpy); scrn++) {
	if (XQueryPointer(target_dpy, RootWindow(target_dpy, scrn), &junk_w, &child,
			  &junk_i, &junk_i, &junk_i, &junk_i, &junk_u)) {
	  if (appres.debug)
	    fprintf(stderr, "Window on the other display/screen (screen #%d of %s) focused\n",
		    scrn, XDisplayString(target_dpy));
	  target_root = RootWindow(target_dpy, scrn);
	  focused_window = child;
	  break;
	}
      }
    }
    if (focused_window == None) focused_window = target_root;
    else focused_window = XmuClientWindow(target_dpy, focused_window);
    if (appres.debug) fprintf(stderr, "Selected window is: 0x%lX on %s\n",
			      focused_window, XDisplayString(target_dpy));

    if (target_dpy == dpy && XtWindow(toplevel) == focused_window) {
      focused_window = None;
      focused_subwindow = focused_window;
      return;
    }

    focused_subwindow = focused_window;
    do {  /* search the child window */
      XQueryPointer(target_dpy, focused_subwindow, &junk_w, &child,
                    &junk_i, &junk_i, &junk_i, &junk_i, &junk_u);
      if (child != None) {
        focused_subwindow = child;
        if (appres.debug) fprintf(stderr, "  going down: 0x%lX\n", focused_subwindow);
      }
    } while (child != None);
    if (appres.list_widgets || strlen(appres.widget) != 0) {
      child = FindWidget(toplevel, focused_window, appres.widget);
      if (child != None) focused_subwindow = child;
    }
  } else {
    fprintf(stderr, "%s: cannot grab pointer\n", PROGRAM_NAME);
  }
}

/*
 * Read keyboard mapping and modifier mapping.
 * Keyboard mapping is used to know what keys are in shifted position.
 * Modifier mapping is required because we should know Alt and Meta
 * key are used as which modifier.
 */
static void Highlight(char *name, Boolean state);

static void ReadKeymap(void)
{
  int i;
  int keycode, inx;
  KeySym keysym;
  XModifierKeymap *modifiers;
  Widget w;
  int last_altgr_mask;

  XDisplayKeycodes(target_dpy, &min_keycode, &max_keycode);
  if (keysym_table != NULL) XFree(keysym_table);
  keysym_table = XGetKeyboardMapping(target_dpy,
                             min_keycode, max_keycode - min_keycode + 1,
                             &keysym_per_keycode);
  for (keycode = min_keycode; keycode <= max_keycode; keycode++) {
    /* if the first keysym is alphabet and the second keysym is NoSymbol,
       it is equivalent to pair of lowercase and uppercase alphabet */
    inx = (keycode - min_keycode) * keysym_per_keycode;
    if (keysym_table[inx + 1] == NoSymbol
	&& ((XK_A <= keysym_table[inx] && keysym_table[inx] <= XK_Z)
	    || (XK_a <= keysym_table[inx] && keysym_table[inx] <= XK_z))) {
      if (XK_A <= keysym_table[inx] && keysym_table[inx] <= XK_Z)
	keysym_table[inx] = keysym_table[inx] - XK_A + XK_a;
      keysym_table[inx + 1] = keysym_table[inx] - XK_a + XK_A;
    }
  }

  last_altgr_mask = altgr_mask;
  alt_mask = 0;
  meta_mask = 0;
  altgr_mask = 0;
  modifiers = XGetModifierMapping(target_dpy);
  for (i = 0; i < 8; i++) {
    keycode = modifiers->modifiermap[i * modifiers->max_keypermod];
    keysym = keysym_table[(keycode - min_keycode) * keysym_per_keycode];
    if (keysym == XK_Alt_L || keysym == XK_Alt_R) {
      alt_mask = 1 << i;
    } else if (keysym == XK_Meta_L || keysym == XK_Meta_R) {
      meta_mask = 1 << i;
    } else if (keysym == XK_Mode_switch) {
      altgr_mask = 0x0101 << i;
      /* I don't know why, but 0x2000 was required for mod3 on my Linux box */
    }
  }
  XFreeModifiermap(modifiers);

  w = XtNameToWidget(toplevel, "*Multi_key");
  if (w != None) {
    if (XKeysymToKeycode(target_dpy, XK_Multi_key) == NoSymbol) {
      if (!appres.auto_add_keysym || AddKeysym(XK_Multi_key, FALSE) == NoSymbol)
	XtSetSensitive(w, FALSE);
    }
  }
  w = XtNameToWidget(toplevel, "*Mode_switch");
  if (w != None) {
    if (altgr_mask) {
      XtSetSensitive(w, TRUE);
    } else {
      XtSetSensitive(w, FALSE);
      if (shift_state & last_altgr_mask) {
	shift_state &= ~last_altgr_mask;
	Highlight("Mode_switch", FALSE);
      }
    }
  }
}

/*
 * This will called when X error is detected when attempting to
 * send a event to a client window;  this will normally caused
 * when the client window is destroyed.
 */
static int MyErrorHandler(Display *my_dpy, XErrorEvent *event)
{
  char msg[200];

  error_detected = TRUE;
  if (event->error_code == BadWindow) {
    if (appres.debug)
      fprintf(stderr, "%s: BadWindow - couldn't find target window 0x%lX (destroyed?)\n",
	      PROGRAM_NAME, (long)focused_window);
    return 0;
  }
  XGetErrorText(my_dpy, event->error_code, msg, sizeof(msg) - 1);
  fprintf(stderr, "X error trapped: %s, request-code=%d\n", msg, event->request_code);
  return 0;
}

/*
 * Send event to the focused window.
 * If input focus is specified explicitly, select the window
 * before send event to the window.
 */
static void SendEvent(XKeyEvent *event)
{
  XSync(event->display, FALSE);
  XSetErrorHandler(MyErrorHandler);

  error_detected = FALSE;
  if (focused_window != None) {
    /* set input focus if input focus is set explicitly */
    if (appres.debug)
      fprintf(stderr, "Set input focus to window 0x%lX (0x%lX)\n",
              (long)focused_window, (long)event->window);
    XSetInputFocus(event->display, focused_window, RevertToParent, CurrentTime);
    XSync(event->display, FALSE);
  }
  if (!error_detected) {
    if (appres.xtest) {
#ifdef USE_XTEST
      if (appres.debug)
	fprintf(stderr, "XTestFakeKeyEvent(0x%lx, 0x%lx, %d)\n",
		(long)event->display, (long)event->keycode, event->type == KeyPress);
      if (appres.jump_pointer) {
	Window root, child, w;
	int root_x, root_y, x, y;
	unsigned int mask;
	int revert_to;

	w = focused_subwindow;
	if (w == None && appres.jump_pointer_always)
	  XGetInputFocus(event->display, &w, &revert_to);

	if (w != None) {
	  if (appres.debug)
	    fprintf(stderr, "SendEvent: jump pointer to window 0x%lx\n", (long)w);
	    
	  XQueryPointer(event->display, w,
			&root, &child, &root_x, &root_y, &x, &y, &mask);
	  XWarpPointer(event->display, None, w, 0, 0, 0, 0, 1, 1);
	  XFlush(event->display);
	}
	XTestFakeKeyEvent(event->display, event->keycode, event->type == KeyPress, 0);
	XFlush(event->display);
	if (w != None) {
	  XWarpPointer(event->display, None, root, 0, 0, 0, 0, root_x, root_y);
	  XFlush(event->display);
	}
      } else {
	XTestFakeKeyEvent(event->display, event->keycode, event->type == KeyPress, 0);
      }
#else
      fprintf(stderr, "%s: this binary is compiled without XTEST support\n",
	      PROGRAM_NAME);
#endif
    } else {
      XSendEvent(event->display, event->window, TRUE, KeyPressMask, (XEvent *)event);
      XSync(event->display, FALSE);

      if (error_detected
	  && (focused_subwindow != None) && (focused_subwindow != event->window)) {
	error_detected = FALSE;
	event->window = focused_subwindow;
	if (appres.debug)
	  fprintf(stderr, "   retry: send event to window 0x%lX (0x%lX)\n",
		  (long)focused_window, (long)event->window);
	XSendEvent(event->display, event->window, TRUE, KeyPressMask, (XEvent *)event);
	XSync(event->display, FALSE);
      }
    }
  }

  if (error_detected) {
    /* reset focus because focused window is (probably) no longer exist */
    XBell(dpy, 0);
    focused_window = None;
    focused_subwindow = None;
  }

  XSetErrorHandler(NULL);
}

/*
 * Insert a specified keysym to unused position in the keymap table.
 * This will be called to add required keysyms on-the-fly.
 * if the second parameter is TRUE, the keysym will be added to the
 * non-shifted position - this may be required for modifier keys
 * (e.g. Mode_switch) and some special keys (e.g. F20).
 */
static int AddKeysym(KeySym keysym, Boolean top)
{
  int keycode, pos, max_pos, inx;

  if (top) {
    max_pos = 0;
  } else {
    max_pos = keysym_per_keycode - 1;
    if (3 < max_pos) max_pos = 3;
  }

  for (pos = max_pos; 0 <= pos; pos--) {
    for (keycode = max_keycode; min_keycode <= keycode; keycode--) {
      inx = (keycode - min_keycode) * keysym_per_keycode;
      if (keysym_table[inx] < 0xFF00) {
	/* if the key is not modifier key or other special key */
	if (keysym_table[inx + pos] == NoSymbol) {
	  if (appres.debug)
	    fprintf(stderr, "Adding keysym \"%s\" at keycode %d position %d/%d\n",
		    XKeysymToString(keysym), keycode, pos, keysym_per_keycode);
	  keysym_table[inx + pos] = keysym;
	  XChangeKeyboardMapping(target_dpy, min_keycode, keysym_per_keycode,
				 keysym_table, max_keycode - min_keycode + 1);
	  XFlush(target_dpy);
	  return keycode;
	}
      }
    }
  }
  fprintf(stderr, "%s: couldn't add \"%s\" to keymap\n",
	  PROGRAM_NAME, XKeysymToString(keysym));
  XBell(dpy, 0);
  return NoSymbol;
}

/*
 * Add the specified key as a new modifier.
 * This is used to use Mode_switch (AltGr) as a modifier.
 */
static void AddModifier(KeySym keysym)
{
  XModifierKeymap *modifiers;
  int keycode, i;

  keycode = XKeysymToKeycode(target_dpy, keysym);
  if (keycode == NoSymbol) keycode = AddKeysym(keysym, TRUE);
  
  modifiers = XGetModifierMapping(target_dpy);
  for (i = 3; i < 8; i++) {
    if (modifiers->modifiermap[i * modifiers->max_keypermod] == NoSymbol) {
      if (appres.debug)
	fprintf(stderr, "Adding modifier \"%s\" as %dth modifier\n",
		XKeysymToString(keysym), i);
      modifiers->modifiermap[i * modifiers->max_keypermod] = keycode;
      XSetModifierMapping(target_dpy, modifiers);
      return;
    }
  }
  fprintf(stderr, "%s: couldn't add \"%s\" as modifier\n",
	  PROGRAM_NAME, XKeysymToString(keysym));
  XBell(dpy, 0);
}

/*
 * Send sequence of KeyPressed/KeyReleased events to the focused
 * window to simulate keyboard.  If modifiers (shift, control, etc)
 * are set ON, many events will be sent.
 */
static void SendKeyPressedEvent(KeySym keysym, unsigned int shift)
{
  Window cur_focus;
  int revert_to;
  XKeyEvent event;
  int keycode;
  Window root, *children;
  unsigned int n_children;
  int phase, inx;
  Boolean found;

  if (focused_subwindow != None)
    cur_focus = focused_subwindow;
  else
    XGetInputFocus(target_dpy, &cur_focus, &revert_to);

  if (appres.debug) {
    char ch = '?';
    if ((keysym & ~0x7f) == 0 && isprint(keysym)) ch = keysym;
    fprintf(stderr, "SendKeyPressedEvent: focus=0x%lX, key=0x%lX (%c), shift=0x%lX\n",
            (long)cur_focus, (long)keysym, ch, (long)shift);
  }

  if (XtWindow(toplevel) != None) {
    if (toplevel_parent == None) {
      XQueryTree(target_dpy, RootWindow(target_dpy, DefaultScreen(target_dpy)),
                 &root, &toplevel_parent, &children, &n_children);
      XFree(children);
    }
    if (cur_focus == None || cur_focus == PointerRoot
	|| cur_focus == XtWindow(toplevel) || cur_focus == toplevel_parent) {
      /* notice user when no window focused or the xvkbd window is focused */
      XBell(dpy, 0);
      return;
    }
  }

  found = FALSE;
  keycode = 0;
  if (keysym != NoSymbol) {
    for (phase = 0; phase < 2; phase++) {
      for (keycode = min_keycode; !found && (keycode <= max_keycode); keycode++) {
	/* Determine keycode for the keysym:  we use this instead
	   of XKeysymToKeycode() because we must know shift_state, too */
	inx = (keycode - min_keycode) * keysym_per_keycode;
	if (keysym_table[inx] == keysym) {
	  shift &= ~altgr_mask;
	  if (keysym_table[inx + 1] != NoSymbol) shift &= ~ShiftMask;
	  found = TRUE;
	  break;
	} else if (keysym_table[inx + 1] == keysym) {
	  shift &= ~altgr_mask;
	  shift |= ShiftMask;
	  found = TRUE;
	  break;
	}
      }
      if (!found && altgr_mask && 3 <= keysym_per_keycode) {
	for (keycode = min_keycode; !found && (keycode <= max_keycode); keycode++) {
	  inx = (keycode - min_keycode) * keysym_per_keycode;
	  if (keysym_table[inx + 2] == keysym) {
	    shift &= ~ShiftMask;
	    shift |= altgr_mask;
	    found = TRUE;
	    break;
	  } else if (4 <= keysym_per_keycode && keysym_table[inx + 3] == keysym) {
	    shift |= ShiftMask | altgr_mask;
	    found = TRUE;
	    break;
	  }
	}
      }
      if (found || !appres.auto_add_keysym) break;

      if (0xF000 <= keysym) {
	/* for special keys such as function keys,
	   first try to add it in the non-shifted position of the keymap */
	if (AddKeysym(keysym, TRUE) == NoSymbol) AddKeysym(keysym, FALSE);
      } else {
	AddKeysym(keysym, FALSE);
      }
    }
    if (appres.debug) {
      if (found)
	fprintf(stderr, "SendKeyPressedEvent: keysym=0x%lx, keycode=%ld, shift=0x%lX\n",
		(long)keysym, (long)keycode, (long)shift);
      else
	fprintf(stderr, "SendKeyPressedEvent: keysym=0x%lx - keycode not found\n",
		(long)keysym);
    }
  }

  event.display = target_dpy;
  event.window = cur_focus;
  event.root = RootWindow(event.display, DefaultScreen(event.display));
  event.subwindow = None;
  event.time = CurrentTime;
  event.x = 1;
  event.y = 1;
  event.x_root = 1;
  event.y_root = 1;
  event.same_screen = TRUE;

#ifdef USE_XTEST
  if (appres.xtest) {
    Window root, child;
    int root_x, root_y, x, y;
    unsigned int mask;

    XQueryPointer(target_dpy, event.root, &root, &child, &root_x, &root_y, &x, &y, &mask);

    event.type = KeyRelease;
    event.state = 0;
    if (mask & ControlMask) {
      event.keycode = XKeysymToKeycode(target_dpy, XK_Control_L);
      SendEvent(&event);
    }
    if (mask & alt_mask) {
      event.keycode = XKeysymToKeycode(target_dpy, XK_Alt_L);
      SendEvent(&event);
    }
    if (mask & meta_mask) {
      event.keycode = XKeysymToKeycode(target_dpy, XK_Meta_L);
      SendEvent(&event);
    }
    if (mask & altgr_mask) {
      event.keycode = XKeysymToKeycode(target_dpy, XK_Mode_switch);
      SendEvent(&event);
    }
    if (mask & ShiftMask) {
      event.keycode = XKeysymToKeycode(target_dpy, XK_Shift_L);
      SendEvent(&event);
    }
    if (mask & LockMask) {
      event.keycode = XKeysymToKeycode(target_dpy, XK_Caps_Lock);
      SendEvent(&event);
    }
  }
#endif

  event.type = KeyPress;
  event.state = 0;
  if (shift & ControlMask) {
    event.keycode = XKeysymToKeycode(target_dpy, XK_Control_L);
    SendEvent(&event);
    event.state |= ControlMask;
  }
  if (shift & alt_mask) {
    event.keycode = XKeysymToKeycode(target_dpy, XK_Alt_L);
    SendEvent(&event);
    event.state |= alt_mask;
  }
  if (shift & meta_mask) {
    event.keycode = XKeysymToKeycode(target_dpy, XK_Meta_L);
    SendEvent(&event);
    event.state |= meta_mask;
  }
  if (shift & altgr_mask) {
    event.keycode = XKeysymToKeycode(target_dpy, XK_Mode_switch);
    SendEvent(&event);
    event.state |= altgr_mask;
  }
  if (shift & ShiftMask) {
    event.keycode = XKeysymToKeycode(target_dpy, XK_Shift_L);
    SendEvent(&event);
    event.state |= ShiftMask;
  }

  if (keysym != NoSymbol) {  /* send event for the key itself */
    event.keycode = found ? keycode : XKeysymToKeycode(target_dpy, keysym);
    if (event.keycode == NoSymbol) {
      if ((keysym & ~0x7f) == 0 && isprint(keysym))
        fprintf(stderr, "%s: no such key: %c\n",
                PROGRAM_NAME, (char)keysym);
      else if (XKeysymToString(keysym) != NULL)
        fprintf(stderr, "%s: no such key: keysym=%s (0x%lX)\n",
                PROGRAM_NAME, XKeysymToString(keysym), (long)keysym);
      else
        fprintf(stderr, "%s: no such key: keysym=0x%lX\n",
                PROGRAM_NAME, (long)keysym);
      XBell(dpy, 0);
    } else {
      SendEvent(&event);
      event.type = KeyRelease;
      SendEvent(&event);
    }
  }

  event.type = KeyRelease;
  if (shift & ShiftMask) {
    event.keycode = XKeysymToKeycode(target_dpy, XK_Shift_L);
    SendEvent(&event);
    event.state &= ~ShiftMask;
  }
  if (shift & altgr_mask) {
    event.keycode = XKeysymToKeycode(target_dpy, XK_Mode_switch);
    SendEvent(&event);
    event.state &= ~altgr_mask;
  }
  if (shift & meta_mask) {
    event.keycode = XKeysymToKeycode(target_dpy, XK_Meta_L);
    SendEvent(&event);
    event.state &= ~meta_mask;
  }
  if (shift & alt_mask) {
    event.keycode = XKeysymToKeycode(target_dpy, XK_Alt_L);
    SendEvent(&event);
    event.state &= ~alt_mask;
  }
  if (shift & ControlMask) {
    event.keycode = XKeysymToKeycode(target_dpy, XK_Control_L);
    SendEvent(&event);
    event.state &= ~ControlMask;
  }
}

/*
 * Word completion - list of words which match the prefix entered
 * via xvkbd can be listed, and choosing one of them will send the
 * suffix to the clients.
 * Words for completion will be read from dictionary file specified
 * with xvkbd.dictFile resource, such as /usr/dict/words.
 */
static Widget completion_panel = None;
static Widget completion_entry = None;
static Widget completion_list = None;

static char completion_text[100] = "";

#define HASH_SIZE 100

#define Hash(str)   ((toupper(str[0]) * 26 + toupper(str[1])) % HASH_SIZE)

static struct WORDLIST {
  struct WORDLIST *next;
  char *word;
} completion_words[HASH_SIZE];

#define MAX_WORDS 200

static String word_list[MAX_WORDS + 1];
static int n_word_list = 0;

static void ReadCompletionDictionary(void)
{
  FILE *fp;
  struct WORDLIST *node_ptr;
  char str[50];
  int i;
  
  for (i = 0; i < HASH_SIZE; i++) {
    completion_words[i].next = NULL;
    completion_words[i].word = NULL;
  }

  fp = fopen(appres.dict_file, "r");
  if (fp == NULL) {
    fprintf(stderr, "%s: can't read dictionary file %s: %s\n",
	    PROGRAM_NAME, appres.dict_file, strerror(errno));
  } else {
    while (fgets(str, sizeof(str) - 1, fp)) {
      if (2 < strlen(str)) {
	str[strlen(str) - 1] = '\0';
	node_ptr = &completion_words[Hash(str)];
	while (node_ptr->word != NULL) node_ptr = node_ptr->next;

	node_ptr->word = XtNewString(str);
	node_ptr->next = malloc(sizeof(struct WORDLIST));
	node_ptr->next->next = NULL;
	node_ptr->next->word = NULL;
      }
    }
    fclose(fp);
  }
}

static void AddToCompletionText(KeySym keysym)
{
  int len;
  struct WORDLIST *node_ptr;

  if (completion_entry != None) {
    len = strlen(completion_text);
    if (keysym == XK_BackSpace || keysym == XK_Delete) {
      if (0 < len) completion_text[len - 1] = '\0';
    } else if (keysym != NoSymbol && keysym < 0x100
	       && !ispunct((char)keysym) && !isspace((char)keysym)) {
      if (len < sizeof(completion_text) - 2) {
	completion_text[len + 1] = '\0';
	completion_text[len] = keysym;
      }
    } else {
      completion_text[0] = '\0';
    }
    XtVaSetValues(completion_entry, XtNlabel, completion_text, NULL);

    n_word_list = 0;
    if (2 <= strlen(completion_text)) {
      node_ptr = &completion_words[Hash(completion_text)];
      while (node_ptr->word != NULL && n_word_list < MAX_WORDS) {
	if (strncasecmp(node_ptr->word, completion_text, strlen(completion_text)) == 0) {
	  word_list[n_word_list] = node_ptr->word;
	  n_word_list = n_word_list + 1;
	}
	node_ptr = node_ptr->next;
      }
    }
    word_list[n_word_list] = NULL;
    XawListChange(completion_list, word_list, 0, 0, TRUE);
  }
}

static void CompletionWordSelected(Widget w, XtPointer client_data, XtPointer call_data)
{
  Boolean capitalize;
  unsigned char ch;
  int n, i;

  n = ((XawListReturnStruct *)call_data)->list_index;
  if (0 <= n && n < n_word_list) {
    capitalize = TRUE;
    for (i = 0; completion_text[i] != '\0'; i++) {
      if (islower(completion_text[i])) capitalize = FALSE;
    }
    for (i = strlen(completion_text); word_list[n][i] != '\0'; i++) {
      ch = word_list[n][i];
      if (capitalize) ch = toupper(ch);
      SendKeyPressedEvent(ch, 0);
    }
  }
  AddToCompletionText(NoSymbol);
}

static void PopupCompletionPanel(void)
{
  Widget form, label, view;

  if (completion_panel == None) {
    completion_panel = XtVaCreatePopupShell("completion_panel", transientShellWidgetClass,
					toplevel, NULL);
    form = XtVaCreateManagedWidget("form", formWidgetClass, completion_panel, NULL);
    label = XtVaCreateManagedWidget("label", labelWidgetClass, form, NULL);
    completion_entry = XtVaCreateManagedWidget("entry", labelWidgetClass, form,
				    XtNfromHoriz, label, NULL);
    view = XtVaCreateManagedWidget("view", viewportWidgetClass, form,
				    XtNfromVert, label, NULL);
    completion_list = XtVaCreateManagedWidget("list", listWidgetClass, view, NULL);
    XtAddCallback(completion_list, XtNcallback, CompletionWordSelected, NULL);
    XtRealizeWidget(completion_panel);
    XSetWMProtocols(dpy, XtWindow(completion_panel), &wm_delete_window, 1);

    XtPopup(completion_panel, XtGrabNone);
    AddToCompletionText(NoSymbol);
    XFlush(dpy);

    ReadCompletionDictionary();
  } else {
    XtPopup(completion_panel, XtGrabNone);
  }
}

/*
 * Send given string to the focused window as if the string
 * is typed from a keyboard.
 */
static void KeyPressed(Widget w, char *key, char *data);

static void SendString(const unsigned char *str)
{
  const unsigned char *cp, *cp2;
  char key[50];
  int len;

  shift_state = 0;
  for (cp = str; *cp != '\0'; cp++) {
    if (*cp == '\\') {
      cp++;
      switch (*cp) {
      case '\0':
        fprintf(stderr, "%s: missing character after \"\\\"\n",
                PROGRAM_NAME);
        return;
      case '[':  /* we can write any keysym as "\[the-keysym]" here */
        cp2 = strchr(cp, ']');
        if (cp2 == NULL) {
          fprintf(stderr, "%s: no closing \"]\" after \"\\[\"\n",
                  PROGRAM_NAME);
        } else {
          len = cp2 - cp - 1;
          if (sizeof(key) <= len) len = sizeof(key) - 1;
          strncpy(key, cp + 1, len);
          key[len] = '\0';
          KeyPressed(None, key, NULL);
          cp = cp2;
        }
        break;
      case 'S': shift_state |= ShiftMask; break;
      case 'C': shift_state |= ControlMask; break;
      case 'A': shift_state |= alt_mask; break;
      case 'M': shift_state |= meta_mask; break;
      case 'b': SendKeyPressedEvent(XK_BackSpace, 0); break;
      case 't': SendKeyPressedEvent(XK_Tab, 0); break;
      case 'n': SendKeyPressedEvent(XK_Linefeed, 0); break;
      case 'r': SendKeyPressedEvent(XK_Return, 0); break;
      case 'e': SendKeyPressedEvent(XK_Escape, 0); break;
      case 'd': SendKeyPressedEvent(XK_Delete, 0); break;
      default: SendKeyPressedEvent(*cp, 0); break;
      }
    } else {
      SendKeyPressedEvent(*cp, shift_state);
      shift_state = 0;
    }
  }
}

/*
 * Send content of the file as if the it is typed from a keyboard.
 */
static void SendFileContent(const char *file)
{
  FILE *fp;
  int ch;

  fp = stdin;
  if (strcmp(file, "-") != 0) fp = fopen(file, "r");

  if (fp == NULL) {
    fprintf(stderr, "%s: can't read the file: %s\n", PROGRAM_NAME, file);
    exit(1);
  }
  while ((ch = fgetc(fp)) != EOF) {
    if (ch == '\n') {  /* newline - send Return instead */
      SendKeyPressedEvent(XK_Return, 0);
    } else if (ch == '\033') {  /* ESC */
      SendKeyPressedEvent(XK_Escape, 0);
    } else if (ch == '\177') {  /* DEL */
      SendKeyPressedEvent(XK_Delete, 0);
    } else if (1 <= ch && ch <= 26) {  /* Ctrl-x */
      SendKeyPressedEvent('a' + ch - 1, ControlMask);
    } else {  /* normal characters */
      SendKeyPressedEvent(ch, 0);
    }
  }
  if (strcmp(file, "-") != 0) fclose(fp);
}

/*
 * Highlight/unhighligh spcified modifier key on the screen.
 */
static void Highlight(char *name, Boolean state)
{
  char name1[50];
  Widget w;

  sprintf(name1, "*%s", name);
  w = XtNameToWidget(toplevel, name1);
  if (w != None) {
    if (strstr(name, "Focus") != NULL) {
      if (target_dpy == dpy)
        XtVaSetValues(w, XtNbackground, appres.focus_background, NULL);
      else
        XtVaSetValues(w, XtNbackground, appres.remote_focus_background, NULL);
      if (state)
        XtVaSetValues(w, XtNforeground, appres.highlight_foreground, NULL);
      else
        XtVaSetValues(w, XtNforeground, appres.special_foreground, NULL);
    } else {
      if (state)
        XtVaSetValues(w, XtNbackground, appres.highlight_background,
                      XtNforeground, appres.highlight_foreground, NULL);
      else
        XtVaSetValues(w, XtNbackground, appres.special_background,
                      XtNforeground, appres.special_foreground, NULL);
    }
  }
}

/*
 * Highlight/unhighligh keys on the screen to reflect the state.
 */
static void RefreshShiftState(Boolean force)
{
  static Boolean first = TRUE;
  static int last_shift_state = 0;
  static int last_mouse_shift = 0;
  static int last_num_lock_state = FALSE;
  static Display *last_target_dpy = NULL;
  static long last_focus = 0;
  int cur_shift;
  int changed;
  int first_row, row, col;
  Boolean shifted;
  char *label;
  int mask;

  cur_shift = shift_state | mouse_shift;
  changed = cur_shift ^ (last_shift_state | last_mouse_shift);
  if (first || force) changed = 0xffff;

  if (changed & ShiftMask) {
    Highlight("Shift_L", cur_shift & ShiftMask);
    Highlight("Shift_R", cur_shift & ShiftMask);
  }
  if (changed & ControlMask) {
    Highlight("Control_L", cur_shift & ControlMask);
    Highlight("Control_R", cur_shift & ControlMask);
  }
  if (changed & alt_mask) {
    Highlight("Alt_L", cur_shift & alt_mask);
    Highlight("Alt_R", cur_shift & alt_mask);
  }
  if (changed & meta_mask) {
    Highlight("Meta_L", cur_shift & meta_mask);
    Highlight("Meta_R", cur_shift & meta_mask);
  }
  if (changed & LockMask) {
    Highlight("Caps_Lock", cur_shift & LockMask);
  }
  if (changed & altgr_mask) {
    Highlight("Mode_switch", cur_shift & altgr_mask);
  }
  if (last_num_lock_state != num_lock_state) {
    Highlight("Num_Lock", num_lock_state);
    Highlight("keypad_panel*Num_Lock", num_lock_state);
  }
  if (last_target_dpy != target_dpy || last_focus != focused_window) {
    Highlight("Focus", focused_window != 0);
    Highlight("keypad*Focus", focused_window != 0);
    Highlight("keypad_panel*Focus", focused_window != 0);
    last_target_dpy = target_dpy;
    last_focus = focused_window;
  }

  mask = ShiftMask | LockMask | altgr_mask;
  changed = (shift_state & mask) ^ (last_shift_state & mask);
  if (first || force) changed = TRUE;
  if (changed && !appres.keypad_only
      && (appres.modal_keytop || toplevel_height < appres.modal_threshold)) {
    first_row = appres.function_key ? 0 : 1;
    for (row = first_row; row < NUM_KEY_ROWS; row++) {
      for (col = 0; col < NUM_KEY_COLS; col++) {
	shifted = (shift_state & ShiftMask);
	if (key_widgets[row][col] != None) {
	  if ((shift_state & altgr_mask) && altgr_key_labels[row][col] != NULL) {
	    if (shifted && shift_altgr_key_labels[row][col] != NULL)
	      label = shift_altgr_key_labels[row][col];
	    else
	      label = altgr_key_labels[row][col];
	  } else {
	    if ((shift_state & LockMask)
		&& isalpha(keys_normal[row][col][0]) && keys_normal[row][col][1] == '\0')
	      shifted = !shifted;
	    if (shifted && shift_key_labels[row][col] != NULL)
	      label = shift_key_labels[row][col];
	    else
	      label = normal_key_labels[row][col];
	  }
	  if (label == NULL) {
	    fprintf(stderr, "%s: no label for key %d,%d\n", PROGRAM_NAME, row, col);
	    label = "";
	  }
	  if (strcmp(label, "space") == 0) label = "";
	  XtVaSetValues(key_widgets[row][col], XtNlabel, label, NULL);
	}
      }
    }
  }

  last_shift_state = shift_state;
  last_mouse_shift = mouse_shift;
  last_num_lock_state = num_lock_state;
  first = FALSE;
}

/*
 * This function will be called when mouse button is pressed on a key
 * on the screen.  Most operation will be performed in KeyPressed()
 * which will be called as callback for the Command widgets, and we
 * only need to check which mouse button is pressed here.
 */
static void ButtonDownAction(Widget w, XEvent *event, String *pars, Cardinal *n_pars)
{
  switch (event->xbutton.button) {
  case Button2:
    mouse_shift |= ControlMask;
    break;
  case Button3:
  case Button4:
    mouse_shift |= ShiftMask;
    break;
  }
  RefreshShiftState(FALSE);
}

/*
 * This function will be called when mouse button is released on a key
 * on the screen, after callback is called.
 */
static void ButtonUpAction(Widget w, XEvent *event, String *pars, Cardinal *n_pars)
{
  mouse_shift = 0;
  RefreshShiftState(FALSE);
}

/*
 * Get the geometry of the base window.
 */
static char *GetWindowGeometry(Widget w)
{
  static char geom[50];

  Position x0, y0;
  Window root;
  int x1, y1;
  unsigned int wd, ht, bd, dp;

  XtVaGetValues(w, XtNx, &x0, XtNy, &y0, NULL);
  XGetGeometry(dpy, XtWindow(w), &root, &x1, &y1, &wd, &ht, &bd, &dp);
  sprintf(geom, "%dx%d+%d+%d", wd, ht, (int)(x0 - x1), (int)(y0 - y1));

  return geom;
}

/*
 * Restart the program to (possibly) change the keyboard layout,
 * by loading the app-default file for the selected "customization".
 */
static void LayoutSelected(Widget w, char *key, char *data)
{
  static char *env_lang, *env_xenv;
  char name[50];
  char customization[30] = "", lang[30] = "C";
  char *xenv = NULL;

  int i;

  if (strcmp(key, "default") != 0) {
    sscanf(key, "%29[^/]/%29s", customization, lang);
    sprintf(name, "XVkbd-%s", customization);
    xenv = XtResolvePathname(dpy, "app-defaults", name, NULL, NULL, NULL, 0, NULL);
    if (xenv == NULL) {
      fprintf(stderr, "%s: app-default file \"%s\" not installed\n",
	      PROGRAM_NAME, name);
    }
  }

  env_lang = malloc(strlen("LC_ALL=") + strlen(lang) + 1);
  sprintf(env_lang, "LC_ALL=%s", lang);
  putenv(env_lang);
  if (xenv != NULL) {
    env_xenv = malloc(strlen("XENVIRONMENT=") + strlen(xenv) + 1);
    sprintf(env_xenv, "XENVIRONMENT=%s", xenv);
    putenv(env_xenv);
  } else if (getenv("XENVIRONMENT") != NULL) {
    putenv("XENVIRONMENT=");
  }

  for (i = 1; i < argc1; i++) {
    if (strncmp(argv1[i], "-geom", strlen("-geom")) == 0) {
      argv1[i + 1] = GetWindowGeometry(toplevel);
      break;
    }
  }
  if (i == argc1) {
    argv1[argc1++] = "-geometry";
    argv1[argc1++] = GetWindowGeometry(toplevel);
    argv1[argc1] = NULL;
  }

  if (appres.debug) {
    fprintf(stderr, "XENVIRONMENT=%s, LC_ALL=%s\n", (xenv != NULL) ? xenv : "", lang);
    fprintf(stderr, "Exec:");
    for (i = 0; i < argc1; i++) fprintf(stderr, " %s", argv1[i]);
    fprintf(stderr, "\n");
  }

  execvp(argv1[0], argv1);
}

/*
 * Popup a window to select the (possibly) keyboard layout.
 * The "XVkbd.customizations" resource will define the list,
 * such as "default,german,swissgerman,french,latin1,jisx6004/ja".
 * For example, "german" here will make this program to load
 * "XVkbd-german" app-default file.  Locale for each configuration
 * can be specified by putting the locale name after "/".
 */
static void PopupLayoutPanel(void)
{
  static Widget layout_panel = None;

  char *customizations;
  char *cp, *cp2;
  Widget box, button;

  if (layout_panel == None) {
    layout_panel = XtVaCreatePopupShell("layout_panel", transientShellWidgetClass,
					toplevel, NULL);
    box = XtVaCreateManagedWidget("box", boxWidgetClass, layout_panel, NULL);

    customizations = XtNewString(appres.customizations);
    cp = strtok(customizations, " \t,");
    while (cp != NULL) {
      cp2 = strchr(cp, '/');
      if (cp2 != NULL) *cp2 = '\0';  /* temporary remove '/' */
      button = XtVaCreateManagedWidget(cp, commandWidgetClass, box, NULL);
      if (cp2 != NULL) *cp2 = '/';
      XtAddCallback(button, XtNcallback, (XtCallbackProc)LayoutSelected, XtNewString(cp));
      cp = strtok(NULL, " \t,");
    }
    XtRealizeWidget(layout_panel);
    XSetWMProtocols(dpy, XtWindow(layout_panel), &wm_delete_window, 1);

    XtFree(customizations);
  }

  XtPopup(layout_panel, XtGrabNone);
}

/*
 * Callback for main menu (activated from "xvkbd" logo).
 */
static Widget about_panel = None;
static Widget keypad_panel = None;
static Widget sun_fkey_panel = None;
static Widget deadkey_panel = None;
static Widget display_panel = None;
static Widget display_status = None;

#define DISPLAY_NAME_LENGTH 50

static void OpenRemoteDisplay(Widget w, char *display_name, char *data)
{
  static char name[DISPLAY_NAME_LENGTH + 10];
  char *cp;

  focused_window = None;
  focused_subwindow = None;
  if (target_dpy != NULL && target_dpy != dpy) XCloseDisplay(target_dpy);

  strcpy(name, (display_name == NULL) ? "" : display_name);
  for (cp = name; isascii(*cp) && isprint(*cp); cp++) ;
  *cp = '\0';

  if (strlen(name) == 0) {
    target_dpy = dpy;
    XtVaSetValues(display_status, XtNlabel, "Disconnected - local display selected", NULL);
    XtPopdown(display_panel);
  } else {
    if (strchr(name, ':') == NULL) strcat(name, ":0");
    target_dpy = XOpenDisplay(name);
    if (target_dpy == NULL) {
      XtVaSetValues(display_status, XtNlabel, "Couldn't connect to the display", NULL);
      target_dpy = dpy;
      XBell(dpy, 0);
    } else {
      XtVaSetValues(display_status, XtNlabel, "Connected", NULL);
      XtPopdown(display_panel);
    }
  }

  ReadKeymap();
  if (!altgr_mask && appres.auto_add_keysym) AddModifier(XK_Mode_switch);

  RefreshMainMenu();
  RefreshShiftState(FALSE);
}

static void MenuSelected(Widget w, char *key)
{
  Widget form;
  
  if (strcmp(key, "man") == 0) {
    system(appres.show_manual_command);
  } else if (strcmp(key, "about") == 0) {
    if (about_panel == None) {
      about_panel = XtVaCreatePopupShell("about_panel", transientShellWidgetClass,
					  toplevel, NULL);
      XtVaCreateManagedWidget("message", labelWidgetClass, about_panel,
			      XtNlabel, appres.description, NULL);
      XtRealizeWidget(about_panel);
      XSetWMProtocols(dpy, XtWindow(about_panel), &wm_delete_window, 1);
    }
    XtPopup(about_panel, XtGrabNone);
  } else if (strcmp(key, "keypad") == 0) {
    if (keypad_panel == None) {
      keypad_panel = XtVaCreatePopupShell("keypad_panel", transientShellWidgetClass,
					  toplevel, NULL);
      form = XtVaCreateManagedWidget("form", formWidgetClass, keypad_panel, NULL);
      MakeKeypad(form, None, None);
      XtRealizeWidget(keypad_panel);
      XSetWMProtocols(dpy, XtWindow(keypad_panel), &wm_delete_window, 1);
    }
    XtPopup(keypad_panel, XtGrabNone);
  } else if (strcmp(key, "sun_fkey") == 0) {
    if (sun_fkey_panel == None) {
      sun_fkey_panel = XtVaCreatePopupShell("sun_fkey_panel", transientShellWidgetClass,
					  toplevel, NULL);
      form = XtVaCreateManagedWidget("form", formWidgetClass, sun_fkey_panel, NULL);
      MakeSunFunctionKey(form, None, None);
      XtRealizeWidget(sun_fkey_panel);
      XSetWMProtocols(dpy, XtWindow(sun_fkey_panel), &wm_delete_window, 1);
    }
    XtPopup(sun_fkey_panel, XtGrabNone);
  } else if (strcmp(key, "deadkey") == 0) {
    if (deadkey_panel == None) {
      deadkey_panel = XtVaCreatePopupShell("deadkey_panel", transientShellWidgetClass,
					  toplevel, NULL);
      form = XtVaCreateManagedWidget("form", formWidgetClass, deadkey_panel, NULL);
      MakeDeadkeyPanel(form);
      XtRealizeWidget(deadkey_panel);
      XSetWMProtocols(dpy, XtWindow(deadkey_panel), &wm_delete_window, 1);
    }
    XtPopup(deadkey_panel, XtGrabNone);
  } else if (strcmp(key, "completion") == 0) {
    PopupCompletionPanel();
  } else if (strcmp(key, "select_layout") == 0) {
    PopupLayoutPanel();
  } else if (strcmp(key, "edit_fkey") == 0) {
    PopupFunctionKeyEditor();
  } else if (strcmp(key, "show_keypad") == 0
	     || strcmp(key, "show_functionkey") == 0) {
    if (strcmp(key, "show_keypad") == 0) appres.keypad = !appres.keypad;
    else appres.function_key = !appres.function_key;
    MakeKeyboard(TRUE);
  } else if (strcmp(key, "use_xtest") == 0) {
    appres.xtest = !appres.xtest;
    RefreshMainMenu();
  } else if (strcmp(key, "shift_lock") == 0) {
    appres.shift_lock = !appres.shift_lock;
    RefreshMainMenu();
  } else if (strcmp(key, "modifiers_lock") == 0) {
    appres.modifiers_lock = !appres.modifiers_lock;
    RefreshMainMenu();
  } else if (strcmp(key, "open_display") == 0) {
    if (display_panel == None) {
      Widget label, entry, button;
      static char display_name[DISPLAY_NAME_LENGTH] = "";
      display_panel = XtVaCreatePopupShell("display_panel", transientShellWidgetClass,
					  toplevel, NULL);
      form = XtVaCreateManagedWidget("form", formWidgetClass, display_panel, NULL);
      label = XtVaCreateManagedWidget("label", labelWidgetClass, form, NULL);
      entry = XtVaCreateManagedWidget("entry", asciiTextWidgetClass, form,
				      XtNfromHoriz, label,
				      XtNuseStringInPlace, True,
				      XtNeditType, XawtextEdit,
				      XtNstring, display_name,
				      XtNlength, sizeof(display_name) - 1,
				      NULL);

      button = XtVaCreateManagedWidget("ok", commandWidgetClass, form,
				       XtNfromHoriz, entry, NULL);
      XtAddCallback(button, XtNcallback, (XtCallbackProc)OpenRemoteDisplay, (XtPointer)display_name);

      display_status = XtVaCreateManagedWidget("status", labelWidgetClass, form,
					       XtNfromVert, label,
					       XtNlabel, "", NULL);
      XtRealizeWidget(display_panel);
      XSetWMProtocols(dpy, XtWindow(display_panel), &wm_delete_window, 1);

      XtSetKeyboardFocus(display_panel, entry);
    }
    XtPopup(display_panel, XtGrabNone);
  } else if (strcmp(key, "close_display") == 0) {
    OpenRemoteDisplay(None, NULL, NULL);
  } else if (strcmp(key, "quit") == 0) {
    XtDestroyApplicationContext(XtWidgetToApplicationContext(toplevel));
    exit(0);
  }
}

static void ClosePopupPanel(Widget w)
{
  if (w == keypad_panel) {
    XtDestroyWidget(w);
    keypad_panel = None;
  } else {
    XtPopdown(w);
  }
}

/*
 * This will be called when user pressed a key on the screen.
 */
static const char *FindFunctionKeyValue(const char *key, Boolean shiftable);
static void ShowBalloon(Widget w, XEvent *event, String *pars, Cardinal *n_pars);

static void KeyPressed(Widget w, char *key, char *data)
{
  int row, col;
  int cur_shift;
  char *key1;
  KeySym keysym;
  Boolean shifted;
  const char *value;
  Boolean found;

  if (appres.debug) fprintf(stderr, "KeyPressed: key=%s, widget=%lx\n", key, (long)w);

  value = FindFunctionKeyValue(key, TRUE);
  if (value != NULL) {
    if (appres.debug) fprintf(stderr, "Assigned string: %s\n", value);
    if (value[0] == '!') {
      if (appres.debug) fprintf(stderr, "Launching: %s\n", value + 1);
      system(value + 1);
    } else {
      if (value[0] == '\\') value = value + 1;
      if (appres.debug) fprintf(stderr, "Sending: %s\n", value);
      SendString(value);
    }
    ShowBalloon(w, NULL, NULL, NULL);
    return;
  }

  if (strncmp(key, "Shift", strlen("Shift")) == 0) {
    if (shift_state & ShiftMask) SendKeyPressedEvent(NoSymbol, shift_state);
    shift_state ^= ShiftMask;
  } else if (strncmp(key, "Control", strlen("Control")) == 0) {
    if (shift_state & ControlMask) SendKeyPressedEvent(NoSymbol, shift_state);
    shift_state ^= ControlMask;
  } else if (strncmp(key, "Alt", strlen("Alt")) == 0) {
    if (shift_state & alt_mask) SendKeyPressedEvent(NoSymbol, shift_state);
    shift_state ^= alt_mask;
  } else if (strncmp(key, "Meta", strlen("Meta")) == 0) {
    if (shift_state & meta_mask) SendKeyPressedEvent(NoSymbol, shift_state);
    shift_state ^= meta_mask;
  } else if (strcmp(key, "Caps_Lock") == 0) {
    if (shift_state & LockMask) SendKeyPressedEvent(NoSymbol, shift_state);
    shift_state ^= LockMask;
  } else if (strcmp(key, "Mode_switch") == 0) {
    if (shift_state & altgr_mask) SendKeyPressedEvent(NoSymbol, shift_state);
    shift_state ^= altgr_mask;
  } else if (strcmp(key, "Num_Lock") == 0) {
    num_lock_state = !num_lock_state;
  } else if (strcmp(key, "Focus") == 0) {
    cur_shift = shift_state | mouse_shift;
    if (cur_shift & ShiftMask) {
      focused_window = None;
      focused_subwindow = None;
    } else {
      GetFocusedWindow();
    }
  } else {
    cur_shift = shift_state | mouse_shift;
    shifted = (cur_shift & ShiftMask);
    key1 = key;
    if (w != None) {
      if (sscanf(key, "pad%d,%d", &row, &col) == 2) {
	if (num_lock_state) shifted = !shifted;
	key1 = shifted ? keypad_shift[row][col]: keypad[row][col];
      } else {
	found = FALSE;
	if (sscanf(key, "%d,%d", &row, &col) == 2) {
	  found = TRUE;
	} else if (w != None) {
	  int first_row = appres.function_key ? 0 : 1;
	  for (row = first_row; row < NUM_KEY_ROWS; row++) {
	    for (col = 0; col < NUM_KEY_COLS; col++) {
	      if (key_widgets[row][col] == w) {
		found = TRUE;
		break;
	      }
	    }
	    if (col < NUM_KEY_COLS) break;
	  }
	}
	if (found) {
	  if ((cur_shift & LockMask)
	      && isalpha(keys_normal[row][col][0]) && keys_normal[row][col][1] == '\0')
	    shifted = !shifted;
	  if ((cur_shift & altgr_mask) && keys_altgr[row][col] != NULL) {
	    if (shifted && keys_shift_altgr[row][col] != NULL) {
	      key1 = keys_shift_altgr[row][col];
	      if (strcmp(keys_altgr[row][col], keys_shift_altgr[row][col]) != 0)
		cur_shift &= ~ShiftMask;
	    } else {
	      key1 = keys_altgr[row][col];
	    }
	  } else {
	    if (shifted && keys_shift[row][col] != NULL) {
	      key1 = keys_shift[row][col];
	      if (strcmp(keys_normal[row][col], keys_shift[row][col]) != 0)
		cur_shift &= ~ShiftMask;
	    } else {
	      key1 = keys_normal[row][col];
	    }
	  }
	}  /* if (found) ... */
      }  /* if (sscanf(key, "pad%d,%d", ... */
    }  /* if (w != None) ... */
    if (strlen(key1) == 1) {
      SendKeyPressedEvent((KeySym)*key1 & 0xff, cur_shift);
      AddToCompletionText((KeySym)*key1);
    } else {
      if (islower(key1[0]) && key1[1] == ':') {
	switch (key1[0]) {
	case 's': cur_shift |= ShiftMask; break;
	case 'c': cur_shift |= ControlMask; break;
	case 'a': cur_shift |= alt_mask; break;
	case 'm': cur_shift |= meta_mask; break;
	default: fprintf(stderr, "%s: unknown modidier: %s\n",
			 PROGRAM_NAME, key1); break;
	}
	key1 = key1 + 2;
      }
      keysym = XStringToKeysym(key1);
      if ((!appres.keypad_keysym && strncmp(key1, "KP_", 3) == 0)
	  || XKeysymToKeycode(target_dpy, keysym) == NoSymbol) {
	switch ((unsigned)keysym) {
	case XK_KP_Equal: keysym = XK_equal; break;
	case XK_KP_Divide: keysym = XK_slash; break;
	case XK_KP_Multiply: keysym = XK_asterisk; break;
	case XK_KP_Add: keysym = XK_plus; break;
	case XK_KP_Subtract: keysym = XK_minus; break;
	case XK_KP_Enter: keysym = XK_Return; break;
	case XK_KP_1: keysym = XK_1; break;
	case XK_KP_2: keysym = XK_2; break;
	case XK_KP_3: keysym = XK_3; break;
	case XK_KP_4: keysym = XK_4; break;
	case XK_KP_5: keysym = XK_5; break;
	case XK_KP_6: keysym = XK_6; break;
	case XK_KP_7: keysym = XK_7; break;
	case XK_KP_8: keysym = XK_8; break;
	case XK_KP_9: keysym = XK_9; break;
	case XK_Shift_L: keysym = XK_Shift_R; break;
	case XK_Shift_R: keysym = XK_Shift_L; break;
	case XK_Control_L: keysym = XK_Control_R; break;
	case XK_Control_R: keysym = XK_Control_L; break;
	case XK_Alt_L: keysym = XK_Alt_R; break;
	case XK_Alt_R: keysym = XK_Alt_L; break;
	case XK_Meta_L: keysym = XK_Meta_R; break;
	case XK_Meta_R: keysym = XK_Meta_L; break;
	default:
	  if (keysym == NoSymbol || !appres.auto_add_keysym)
	    fprintf(stderr, "%s: no such key: %s\n",
		    PROGRAM_NAME, key1); break;
	}
      }
      SendKeyPressedEvent(keysym, cur_shift);
      AddToCompletionText(keysym);

      if ((cur_shift & ControlMask) && (cur_shift & alt_mask)) {
        if (strstr(XServerVendor(dpy), "XFree86") != NULL) {
          if (strcmp(key1, "KP_Add") == 0) {
            system("xvidtune -next");
          } else if (strcmp(key1, "KP_Subtract") == 0) {
            system("xvidtune -prev");
          }
        }
      }
    }
    if (!appres.shift_lock)
      shift_state &= ~ShiftMask;
    if (!appres.modifiers_lock)
      shift_state &= ~(ControlMask | alt_mask | meta_mask);
    if (!appres.altgr_lock)
      shift_state &= ~altgr_mask;
  }
  RefreshShiftState(FALSE);
}

/*
 * Redefine keyboard layout.
 * "spec" is a sequence of words separated with spaces, and it is
 * usally specified in app-defaults file, as:
 * 
 *   xvkbd.AltGrKeys: \
 *      F1 F2 F3 F4 F5 F6 F7 F8 F9 F10 F11 F12 BackSpace \n\
 *      Escape \271 \262 \263 \243 \254 \251 { [ ] } \\ ' ^ ' \n\
 *      ...
 *
 * White spaces separate the keys, and " \n" (note that white space
 * before the \n) separate the rows of keys.
 */
static void RedefineKeys(char *array[NUM_KEY_ROWS][NUM_KEY_COLS], const char *spec)
{
  char *s = XtNewString(spec);
  char *cp;
  int row, col;

  for (row = 0; row < NUM_KEY_ROWS; row++) {
    for (col = 0; col < NUM_KEY_COLS; col++) array[row][col] = NULL;
  }
  row = 0;
  col = 0;
  cp = strtok(s, " ");
  while (cp != NULL) {
    if (*cp == '\n') {
      row = row + 1;
      col = 0;
      cp = cp + 1;
    }
    if (*cp != '\0') {
      if (NUM_KEY_ROWS <= row) {
        fprintf(stderr, "%s: too many key rows: \"%s\" ignored\n",
                PROGRAM_NAME, cp);
      } else if (NUM_KEY_COLS <= col) {
        fprintf(stderr, "%s: too many keys in a row: \"%s\" ignored\n",
                PROGRAM_NAME, cp);
      } else {
        array[row][col] = XtNewString(cp);
        col = col + 1;
      }
    }
    cp = strtok(NULL, " ");
  }
  XtFree(s);
}

/*
 * Create keyboard on the screen.
 */
static Widget MakeKey(Widget parent, const char *name, const char *label, Pixel color)
{
  static Pixmap up_pixmap = None;
  static Pixmap down_pixmap = None;
  static Pixmap left_pixmap = None;
  static Pixmap right_pixmap = None;
  static Pixmap back_pixmap = None;
  Widget w;
  Window scr = RootWindow(dpy, DefaultScreen(dpy));
  char str[50];
  int len;

  if (!appres.auto_repeat
      || strncmp(name, "Shift", strlen("Shift")) == 0
      || strncmp(name, "Control", strlen("Control")) == 0
      || strncmp(name, "Alt", strlen("Alt")) == 0
      || strncmp(name, "Meta", strlen("Meta")) == 0
      || strcmp(name, "Caps_Lock") == 0
      || strcmp(name, "Mode_switch") == 0
      || strcmp(name, "Num_Lock") == 0
      || strcmp(name, "Focus") == 0) {
    w = XtVaCreateManagedWidget(name, commandWidgetClass, parent,
                                XtNbackground, color, NULL);
  } else {
    w = XtVaCreateManagedWidget(name, repeaterWidgetClass, parent,
                                XtNbackground, color, NULL);
  }
  XtAddCallback(w, XtNcallback, (XtCallbackProc)KeyPressed, (XtPointer)name);

  if (label != NULL) {
    strncpy(str, label, sizeof(str) - 1);
    if (strcmp(str, "space") == 0) strcpy(str, "");
    len = strlen(str);
    if (3 <= len) {
      if (str[1] == '_') str[1] = ' ';
      if (str[len - 2] == '_') str[len - 2] = ' ';
    }
    XtVaSetValues(w, XtNlabel, str, NULL);

    if (strcmp(label, "up") == 0) {
      if (up_pixmap == None)
        up_pixmap = XCreateBitmapFromData(dpy, scr,
                                (char *)up_bits, up_width, up_height);
      XtVaSetValues(w, XtNbitmap, up_pixmap, NULL);
    } else if (strcmp(label, "down") == 0) {
      if (down_pixmap == None)
        down_pixmap = XCreateBitmapFromData(dpy, scr,
                                (char *)down_bits, down_width, down_height);
      XtVaSetValues(w, XtNbitmap, down_pixmap, NULL);
    } else if (strcmp(label, "left") == 0) {
      if (left_pixmap == None)
        left_pixmap = XCreateBitmapFromData(dpy, scr,
                                (char *)left_bits, left_width, left_height);
      XtVaSetValues(w, XtNbitmap, left_pixmap, NULL);
    } else if (strcmp(label, "right") == 0) {
      if (right_pixmap == None)
        right_pixmap = XCreateBitmapFromData(dpy, scr,
                                (char *)right_bits, right_width, right_height);
      XtVaSetValues(w, XtNbitmap, right_pixmap, NULL);
    } else if (strcmp(label, "back") == 0) {
      if (back_pixmap == None)
        back_pixmap = XCreateBitmapFromData(dpy, scr,
                                (char *)back_bits, back_width, back_height);
      XtVaSetValues(w, XtNbitmap, back_pixmap, NULL);
    }
  }

  return w;
}

static void MakeKeypad(Widget form, Widget from_vert, Widget from_horiz)
{
  Widget key, left, upper;
  Pixel color;
  XFontStruct *font;
  int row, col;
  Widget keypad_box;
  Widget keypad_row[NUM_KEYPAD_ROWS];
  char name[50];

  keypad_box = XtVaCreateManagedWidget("keypad", formWidgetClass, form, NULL);
  if (from_horiz != None)
    XtVaSetValues(keypad_box, XtNfromHoriz, from_horiz, NULL);
  else
    XtVaSetValues(keypad_box, XtNhorizDistance, 0, NULL);
  if (from_vert != None)
    XtVaSetValues(keypad_box, XtNfromVert, from_vert, NULL);
  else
    XtVaSetValues(keypad_box, XtNvertDistance, 0, NULL);
  upper = None;
  for (row = 0; row < NUM_KEYPAD_ROWS; row++) {
    left = None;
    for (col = 0; keypad[row][col] != NULL; col++) {
      font = appres.keypad_font;
      if (strlen(keypad_label[row][col]) == 1) font = appres.letter_font;
      color = appres.special_background;
      if (strcmp(keypad[row][col], "Focus") == 0)
	color = appres.focus_background;
      else if (strcmp(keypad_shift[row][col], ".") == 0
	       || (strncmp(keypad_shift[row][col], "KP_", 3) == 0
		   && isdigit(keypad_shift[row][col][3])))
	color = appres.general_background;
      strcpy(name, keypad[row][col]);
      if (strcmp(name, "Focus") != 0 && strcmp(name, "Num_Lock") != 0)
	sprintf(name, "pad%d,%d", row, col);
      key = MakeKey(keypad_box, XtNewString(name),
		    keypad_label[row][col], color);
      XtVaSetValues(key, XtNfont, font, NULL);
      if (row != 0)
	XtVaSetValues(key, XtNfromVert, keypad_row[row - 1], NULL);
      if (left != None)
	XtVaSetValues(key, XtNfromHoriz, left, NULL);
      if (col == 0)
	keypad_row[row] = key;
      left = key;
    }
  }
}

static void MakeSunFunctionKey(Widget form, Widget from_vert, Widget from_horiz)
{
  Widget key, left, upper;
  int row, col;
  Widget fkey_box;
  Widget fkey_row[NUM_SUN_FKEY_ROWS];

  fkey_box = XtVaCreateManagedWidget("fkey", formWidgetClass, form, NULL);
  if (from_horiz != None)
    XtVaSetValues(fkey_box, XtNfromHoriz, from_horiz, NULL);
  else
    XtVaSetValues(fkey_box, XtNhorizDistance, 0, NULL);
  if (from_vert != None)
    XtVaSetValues(fkey_box, XtNfromVert, from_vert, NULL);
  else
    XtVaSetValues(fkey_box, XtNvertDistance, 0, NULL);
  upper = None;
  for (row = 0; row < NUM_SUN_FKEY_ROWS; row++) {
    left = None;
    for (col = 0; sun_fkey[row][col] != NULL; col++) {
      key = MakeKey(fkey_box, sun_fkey[row][col],
		    sun_fkey_label[row][col], appres.special_background);
      XtVaSetValues(key, XtNfont, appres.keypad_font, NULL);
      if (row != 0)
	XtVaSetValues(key, XtNfromVert, fkey_row[row - 1], NULL);
      if (left != None)
	XtVaSetValues(key, XtNfromHoriz, left, NULL);
      if (col == 0)
	fkey_row[row] = key;
      left = key;
    }
  }
}

static void MakeDeadkeyPanel(Widget form)
{
  int i;
  Widget deadkey_box, left, key;
  char *deadkeys, *cp, *cp2;

  deadkeys = XtNewString(appres.deadkeys);

  deadkey_box = XtVaCreateManagedWidget("deadkey", formWidgetClass, form, NULL);
  left = None;
  cp = strtok(deadkeys, " \t,");
  while (cp != NULL) {
    cp2 = XtNewString(cp);
    key = MakeKey(deadkey_box, cp2, NULL, appres.general_background);
    if (left != None) XtVaSetValues(key, XtNfromHoriz, left, NULL);
    left = key;
    cp = strtok(NULL, " \t,");
  }
  XtFree(deadkeys);
}

static void RefreshMainMenu(void)
{
  static Pixmap check_pixmap = None;

  if (check_pixmap == None) {
    check_pixmap = XCreateBitmapFromData(dpy, RootWindow(dpy, DefaultScreen(dpy)),
				 (char *)check_bits, check_width, check_height);
  }
  XtVaSetValues(XtNameToWidget(main_menu, "*show_keypad"),
		XtNrightBitmap, appres.keypad ? check_pixmap : None, NULL);
  XtVaSetValues(XtNameToWidget(main_menu, "*show_functionkey"),
		XtNrightBitmap, appres.function_key ? check_pixmap : None, NULL);
#ifdef USE_XTEST
  XtVaSetValues(XtNameToWidget(main_menu, "*use_xtest"),
		XtNrightBitmap, appres.xtest ? check_pixmap : None, NULL);
#endif
  XtVaSetValues(XtNameToWidget(main_menu, "*shift_lock"),
		XtNrightBitmap, appres.shift_lock ? check_pixmap : None, NULL);
  XtVaSetValues(XtNameToWidget(main_menu, "*modifiers_lock"),
		XtNrightBitmap, appres.modifiers_lock ? check_pixmap : None, NULL);

  XtSetSensitive(XtNameToWidget(main_menu, "*edit_fkey"), appres.function_key);
  XtSetSensitive(XtNameToWidget(main_menu, "*close_display"), target_dpy != dpy);
}

static void MakeKeyboard(Boolean remake)
{
  static char *main_menu_items[] = {
    "about", "man", "keypad", "sun_fkey", "deadkey", "completion", "",
    "select_layout",
    "edit_fkey",
    "show_keypad",
    "show_functionkey",
#ifdef USE_XTEST
    "use_xtest",
#endif
    "shift_lock", "modifiers_lock", "",
    "open_display", "close_display", "",
    "quit" };

  Widget form, key, left;
  Pixel color;
  XFontStruct *font;
  Dimension wd, max_wd;
  int row, col, first_row;
  char name[50], *label;
  Widget key_box[NUM_KEY_ROWS];
  Widget menu_entry;
  int i;

#include "xvkbd.xbm"
  Pixmap xvkbd_pixmap;

  if (remake) {
    appres.geometry = GetWindowGeometry(toplevel);
    XtUnrealizeWidget(toplevel);
    XtDestroyWidget(XtNameToWidget(toplevel, "form"));
  }

  form = XtVaCreateManagedWidget("form", formWidgetClass, toplevel, NULL);

  key_box[0] = None;
  key_box[1] = None;
  first_row = appres.function_key ? 0 : 1;
  if (!appres.keypad_only) {
    for (row = first_row; row < NUM_KEY_ROWS; row++) {
      sprintf(name, "row%d", row);
      key_box[row] = XtVaCreateManagedWidget(name, formWidgetClass, form, NULL);
      if (row != first_row)
        XtVaSetValues(key_box[row], XtNfromVert, key_box[row - 1], NULL);
      else if (!appres.function_key)
        XtVaSetValues(key_box[row], XtNvertDistance, 0, NULL);
        
      left = None;
      for (col = 0; keys_normal[row][col] != NULL; col++) {
        strcpy(name, keys_normal[row][col]);
	if (strcmp(name, "MainMenu") == 0) {
	  xvkbd_pixmap = XCreateBitmapFromData(dpy, RootWindow(dpy, DefaultScreen(dpy)),
			       (char *)xvkbd_bits, xvkbd_width, xvkbd_height);
	  key = XtVaCreateManagedWidget("MainMenu", menuButtonWidgetClass, key_box[row],
					XtNbitmap, xvkbd_pixmap, NULL);
	  main_menu = XtVaCreatePopupShell("menu", simpleMenuWidgetClass, key, NULL);
	  for (i = 0; i < XtNumber(main_menu_items); i++) {
	    if (strlen(main_menu_items[i]) == 0) {
	      XtVaCreateManagedWidget("separator", smeLineObjectClass, main_menu, NULL);
	    } else {
	      menu_entry = XtVaCreateManagedWidget(main_menu_items[i], smeBSBObjectClass,
						   main_menu, NULL);
	      XtAddCallback(menu_entry, XtNcallback, (XtCallbackProc)MenuSelected,
			    (XtPointer)main_menu_items[i]);
	    }
	  }
	} else {
	  label = appres.modal_keytop ? normal_key_labels[row][col] : key_labels[row][col];
	  if (isascii(name[0]) && isupper(name[0])) {
	    if (strcmp(name, "Focus") == 0) {
	      color = appres.focus_background;
	      font = appres.keypad_font;
	    } else {
	      color = appres.special_background;
	      if (label != NULL && strchr(label, '\n') != NULL) font = appres.keypad_font;
	      else font = appres.special_font;
	    }
	  } else {
	    color = appres.general_background;
	    font = appres.general_font;
	    if (isalpha(name[0])) font = appres.letter_font;
	    if (strcmp(name, "space") != 0) sprintf(name, "%d,%d", row, col);
	  }
	  key = MakeKey(key_box[row], XtNewString(name), label, color);
	  XtVaGetValues(key, XtNwidth, &wd, NULL);
	  if (wd <= 1) {
	    /* keys can be removed by setting its width to 1 */
	    XtDestroyWidget(key);
	    key = None;
	  } else {
	    XtVaSetValues(key, XtNfont, font, NULL);
#ifdef USE_I18N
	    if (font == appres.special_font || font == appres.keypad_font)
	      XtVaSetValues(key, XtNfontSet, appres.special_fontset, NULL);
#endif
	  }
	}
	if (key != None) {
	  if (left != None) XtVaSetValues(key, XtNfromHoriz, left, NULL);
	  left = key;
	}
	key_widgets[row][col] = key;
      }
    }
  }

  if (appres.keypad) MakeKeypad(form, key_box[0], key_box[1]);

  if (!appres.keypad_only && appres.function_key && appres.keypad) {
    XtVaCreateManagedWidget("banner", labelWidgetClass, form,
                            XtNfromHoriz, key_box[1],
                            XtNlabel, PROGRAM_NAME_WITH_VERSION, NULL);
  }

  XtRealizeWidget(toplevel);

  if (!remake && strlen(appres.geometry) == 0) {
    Window root;
    int x1, y1;
    unsigned int wd, ht, bd, dp;
    int max_wd, max_ht;

    XGetGeometry(dpy, XtWindow(toplevel), &root, &x1, &y1, &wd, &ht, &bd, &dp);
    max_wd = XtScreen(toplevel)->width * appres.max_width_ratio;
    max_ht = XtScreen(toplevel)->height * appres.max_height_ratio;
    if (appres.debug)
      fprintf(stderr, "window size: %dx%d, max size: %dx%d\n", wd, ht, max_wd, max_ht);
    if (max_wd < wd || max_ht < ht) {
      if (max_wd < wd) wd = max_wd;
      if (max_ht < ht) ht = max_ht;
      if (appres.debug)
	fprintf(stderr, "setting window size: %dx%d\n", wd, ht);
      XResizeWindow(dpy, XtWindow(toplevel), wd, ht);
    }
  }

  if (!appres.debug && key_box[first_row] != None) {
    if (appres.keypad) {
      XtVaGetValues(key_box[1], XtNwidth, &max_wd, NULL);
    } else {
      max_wd = 0;
      for (row = first_row; row < NUM_KEY_ROWS; row++) {
        XtVaGetValues(key_box[row], XtNwidth, &wd, NULL);
        if (max_wd < wd) max_wd = wd;
      }
    }
    for (row = first_row; row < NUM_KEY_ROWS; row++) {
      XtVaSetValues(key_box[row], XtNwidth, max_wd, NULL);
    }
  }
  if (0 < strlen(appres.geometry)) {
    if (appres.debug)
      fprintf(stderr, "setting window geometry: %s\n", appres.geometry);
    XtVaSetValues(toplevel, XtNgeometry, appres.geometry, NULL);
    XtUnrealizeWidget(toplevel);
    XtRealizeWidget(toplevel);
  }

  ReadKeymap();
  if (main_menu != None) RefreshMainMenu();
  RefreshShiftState(FALSE);

  XtMapWidget(toplevel);

  if (wm_delete_window == None)
    wm_delete_window = XInternAtom(dpy, "WM_DELETE_WINDOW", FALSE);
  XSetWMProtocols(dpy, XtWindow(toplevel), &wm_delete_window, 1);

  XtVaGetValues(toplevel, XtNheight, &toplevel_height, NULL);
}

/*
 * WM_DELETE_WINDOW has been sent - terminate the program.
 */
static void DeleteWindowProc(Widget w, XEvent *event,
                             String *pars, Cardinal *n_pars)
{
  XtDestroyApplicationContext(XtWidgetToApplicationContext(toplevel));
  exit(0);
}

/*
 * Callback for ConfigureNotify event, which will be invoked when
 * the toplevel window is resized.
 * We may need to switch the keytop labels when window becomes
 * smaller than appres.modal_threshold, and vice versa.
 */
static void WindowResized(Widget w, XEvent *event,
			  String *pars, Cardinal *n_pars)
{
  Dimension ht;

  XtVaGetValues(toplevel, XtNheight, &ht, NULL);
  if (appres.modal_threshold <= ht) {
    if (toplevel_height < appres.modal_threshold) MakeKeyboard(TRUE);
  } else {
    toplevel_height = ht;
  }
  RefreshShiftState(TRUE);
}

/*
 * Load list of text to be assigned to function keys.
 * Each line contains name of the key (with optional modifier)
 * and the text to be assigned to the key, as:
 *
 *   F1 text for F1
 *   s:F2 text for Shift-F2
 */
#ifndef PATH_MAX
# define PATH_MAX 300
#endif

static char fkey_filename[PATH_MAX] = "";

static struct fkey_struct {
  struct fkey_struct *next;
  char *value;
} *fkey_list = NULL;

static void ReadFuncionKeys(void)
{
  FILE *fp;
  char str[200];
  struct fkey_struct *sp, *new_node;
  char len;

  if (appres.key_file[0] != '/' && getenv("HOME") != NULL)
    sprintf(fkey_filename, "%s/%s", getenv("HOME"), appres.key_file);
  else
    strcpy(fkey_filename, appres.key_file);

  fp = fopen(fkey_filename, "r");
  if (fp == NULL) return;

  while (fgets(str, sizeof(str) - 1, fp)) {
    if (isalpha(str[0])) {
      len = strlen(str);
      if (str[len - 1] == '\n') str[len - 1] = '\0';

      new_node = malloc(sizeof(struct fkey_struct));
      if (fkey_list == NULL) fkey_list = new_node;
      else sp->next = new_node;
      sp = new_node;

      sp->next = NULL;
      sp->value = XtNewString(str);
    }
  }
  fclose(fp);
}

/*
 * Edit string assigned for function keys.
 * Modifiers (Shift, Ctrl, etc.) can't be handled here.
 */
static Widget edit_fkey_panel = None;
static Widget fkey_menu_button = None;
static Widget fkey_value_menu_button = None;
static Widget fkey_value_entry = None;
static char fkey_value[100] = "";
static char cur_fkey[20] = "";
static char *cur_fkey_value_mode = "";

static void FKeyValueMenuSelected(Widget w, char *key)
{
  char *key1, *cp;

  if (key[0] == 'c') {
    cur_fkey_value_mode = "command";
    key1 = "*command";
  } else {
    cur_fkey_value_mode = "string";
    key1 = "*string";
  }
  XtVaGetValues(XtNameToWidget(fkey_value_menu_button, key1), XtNlabel, &cp, NULL);
  XtVaSetValues(fkey_value_menu_button, XtNlabel, cp, NULL);
}

static void FKeyMenuSelected(Widget w, char *key)
{
  struct fkey_struct *sp, *sp2;
  int len;
  const char *value, *prefix;
  char key2[20];

  if (key == NULL)
    strcpy(key2, "");
  else if (strncmp(key, "Shift-", strlen("Shift-")) == 0)
    sprintf(key2, "s:%s", &key[strlen("Shift-")]);
  else
    strcpy(key2, key);

  if (strcmp(cur_fkey, key2) != 0) {
    if (strlen(cur_fkey) != 0) {
      len = strlen(cur_fkey);
      sp2 = NULL;
      for (sp = fkey_list; sp != NULL; sp = sp->next) {
	if (strncmp(sp->value, cur_fkey, len) == 0 && isspace(sp->value[len]))
	  break;
	sp2 = sp;
      }
      if (strlen(fkey_value) != 0) {  /* assign new string for the function key */
	if (sp == NULL) {  /* it was not defined before now */
	  sp = malloc(sizeof(struct fkey_struct));
	  if (fkey_list == NULL) fkey_list = sp;
	  else sp2->next = sp;
	  sp->next = NULL;
	  sp->value = NULL;
	}
	sp->value = realloc(sp->value, len + strlen(fkey_value) + 5);
	prefix = "";
	if (cur_fkey_value_mode[0] == 'c') prefix = "!";
	else if (fkey_value[0] == '!') prefix = "\\";
	sprintf(sp->value, "%s %s%s", cur_fkey, prefix, fkey_value);
      } else {  /* empty string - remove the entry for the function key */
	if (sp != NULL) {
	  if (sp2 != NULL) sp2->next = sp->next;
	  else fkey_list = sp->next;
	  free(sp->value);
	  free(sp);
	}
      }
    }

    if (key != NULL) {
      XtVaSetValues(fkey_menu_button, XtNlabel, key, NULL);
  
      value = FindFunctionKeyValue(key2, FALSE);
      if (value == NULL) value = "";

      FKeyValueMenuSelected(None, (value[0] == '!') ? "command" : "string");

      if (value[0] == '!' || value[0] == '\\') value = value + 1;
      strncpy(fkey_value, value, sizeof(fkey_value) - 1);
      XtVaSetValues(fkey_value_entry, XtNstring, fkey_value, NULL);

      strcpy(cur_fkey, key2);
    }
  }
}

static void CloseFunctionKeyPanel(void)
{
  XtPopdown(edit_fkey_panel);
}

static void SaveFunctionKey(void)
{
  struct fkey_struct *sp;
  FILE *fp;

  FKeyMenuSelected(None, NULL);

  fp = fopen(fkey_filename, "w");
  if (fp == NULL) {
    fprintf(stderr, "%s: can't open \"%s\": %s\n",
	    PROGRAM_NAME, fkey_filename, strerror(errno));
    return;
  }
  for (sp = fkey_list; sp != NULL; sp = sp->next) {
    fprintf(fp, "%s\n", sp->value);
  }
  fclose(fp);

  CloseFunctionKeyPanel();
}

static void PopupFunctionKeyEditor(void)
{
  Widget form, form2, menu, menu_entry, button;
  char label[20];
  char *key;
  int i, j;

  if (edit_fkey_panel == None) {
    edit_fkey_panel = XtVaCreatePopupShell("edit_fkey_panel", transientShellWidgetClass,
					   toplevel, NULL);
    form = XtVaCreateManagedWidget("form", formWidgetClass, edit_fkey_panel, NULL);

    form2 = XtVaCreateManagedWidget("form2", formWidgetClass, form, NULL);
    XtVaCreateManagedWidget("fkey_label", labelWidgetClass, form2, NULL);
    fkey_menu_button = XtVaCreateManagedWidget("fkey_menu", menuButtonWidgetClass,
					       form2, NULL);
    menu = XtVaCreatePopupShell("menu", simpleMenuWidgetClass, fkey_menu_button, NULL);
    for (j = 0; j <= 1; j++) {
      for (i = 1; i <= appres.editable_function_keys; i++) {
	if (j == 0)
	  sprintf(label, "F%d", i);
	else 
	  sprintf(label, "Shift-F%d", i);
	key = XtNewString(label);
	menu_entry = XtVaCreateManagedWidget(key, smeBSBObjectClass, menu, NULL);
	XtAddCallback(menu_entry, XtNcallback, (XtCallbackProc)FKeyMenuSelected,
		      (XtPointer)key);
      }
    }

    fkey_value_menu_button = XtVaCreateManagedWidget("fkey_value_menu", menuButtonWidgetClass,
						     form2, NULL);
    menu = XtVaCreatePopupShell("menu", simpleMenuWidgetClass, fkey_value_menu_button, NULL);
    menu_entry = XtVaCreateManagedWidget("string", smeBSBObjectClass, menu, NULL);
    XtAddCallback(menu_entry, XtNcallback, (XtCallbackProc)FKeyValueMenuSelected,
		  (XtPointer)"string");
    menu_entry = XtVaCreateManagedWidget("command", smeBSBObjectClass, menu, NULL);
    XtAddCallback(menu_entry, XtNcallback, (XtCallbackProc)FKeyValueMenuSelected,
		  (XtPointer)"command");

    XtVaCreateManagedWidget("fkey_value_sep", labelWidgetClass, form2, NULL);

    fkey_value_entry = XtVaCreateManagedWidget("fkey_value", asciiTextWidgetClass, form2,
					       XtNuseStringInPlace, True,
					       XtNeditType, XawtextEdit,
					       XtNstring, fkey_value,
					       XtNlength, sizeof(fkey_value) - 1,
					       NULL);

    button = XtVaCreateManagedWidget("save_button", commandWidgetClass, form, NULL);
    XtAddCallback(button, XtNcallback, (XtCallbackProc)SaveFunctionKey, NULL);

    button = XtVaCreateManagedWidget("close_button", commandWidgetClass, form, NULL);
    XtAddCallback(button, XtNcallback, (XtCallbackProc)CloseFunctionKeyPanel, NULL);

    XtRealizeWidget(edit_fkey_panel);
    XSetWMProtocols(dpy, XtWindow(edit_fkey_panel), &wm_delete_window, 1);

    XtSetKeyboardFocus(edit_fkey_panel, fkey_value_entry);

    FKeyMenuSelected(None, "F1");
  }

  XtPopup(edit_fkey_panel, XtGrabNone);
}

/*
 * If text is assigned to the specified function key,
 * return the text.  Otherwise, return NULL.
 */
static const char *FindFunctionKeyValue(const char *key, Boolean shiftable)
{
  char label[50];
  char prefix;
  struct fkey_struct *sp;
  int len;

  prefix = '\0';
  if (shiftable) {
    if (shift_state & meta_mask) prefix = 'm';
    else if (shift_state & alt_mask) prefix = 'a';
    else if (shift_state & ControlMask) prefix = 'c';
    else if (shift_state & ShiftMask) prefix = 's';
  }
  if (prefix == '\0') sprintf(label, "%s", key);
  else sprintf(label, "%c:%s", prefix, key);
  len = strlen(label);
  
  for (sp = fkey_list; sp != NULL; sp = sp->next) {
    if (strncmp(sp->value, label, len) == 0 && isspace(sp->value[len]))
      return &(sp->value[len + 1]);
  }
  return NULL;
}

/*
 * Display balloon message for the function keys,
 * if text is assigned to the key.
 */
static Boolean balloon_panel_open = FALSE;
static Widget balloon_panel = None;

static void ShowBalloon(Widget w, XEvent *event, String *pars, Cardinal *n_pars)
{
  static Widget message;
  Position x, y;
  Dimension ht;
  const char *value;

  if (strcmp(XtName(w), "MainMenu") == 0) value = "Main menu";
  else value = FindFunctionKeyValue(XtName(w), TRUE);
  if (value == NULL) return;

  if (balloon_panel == None) {
    balloon_panel = XtVaCreatePopupShell("balloon_panel", transientShellWidgetClass, toplevel,
					 XtNoverrideRedirect, TRUE, NULL);
    message = XtVaCreateManagedWidget("message", labelWidgetClass, balloon_panel, NULL);
  }
  XtVaGetValues(w, XtNheight, &ht, NULL);
  XtTranslateCoords(w, 6, ht + 2, &x, &y);
  XtVaSetValues(balloon_panel, XtNx, x, XtNy, y, NULL);
  if (value[0] == '!') {
    XtVaSetValues(message, XtNlabel, value + 1,
		  XtNbackground, appres.launch_balloon_background, NULL);
  } else {
    if (value[0] == '\\') value = value + 1;
    XtVaSetValues(message, XtNlabel, value,
		  XtNbackground, appres.balloon_background, NULL);
  }
  XtPopup(balloon_panel, XtGrabNone);

  balloon_panel_open = TRUE;
}

static void CloseBalloon(Widget w, XEvent *event, String *pars, Cardinal *n_pars)
{
  if (balloon_panel_open) {
    XtPopdown(balloon_panel);
    balloon_panel_open = FALSE;
  }
}

/*
 * Set icon image.
 */
static void SetIconBitmap(Widget w)
{
#include "xvkbd_icon.xbm"
#include "xvkbd_iconmask.xbm"

  Pixmap icon_pixmap, icon_mask;

  icon_pixmap = XCreateBitmapFromData(XtDisplay(w), XtWindow(w),
				      (char *)xvkbd_icon_bits,
				      xvkbd_icon_width, xvkbd_icon_height);;
  icon_mask = XCreateBitmapFromData(XtDisplay(w), XtWindow(w),
				    (char *)xvkbd_iconmask_bits,
				    xvkbd_iconmask_width, xvkbd_iconmask_height);
  XtVaSetValues(w, XtNiconPixmap, icon_pixmap, XtNiconMask, icon_mask, NULL);
}

/*
 * The main program.
 */
int main(int argc, char *argv[])
{
  static XtActionsRec actions[] = {
    { "DeleteWindowProc", DeleteWindowProc },
    { "WindowResized", WindowResized },
    { "ReadKeymap", (XtActionProc)ReadKeymap },
    { "ButtonDownAction", ButtonDownAction },
    { "ButtonUpAction", ButtonUpAction },
    { "ShowBalloon", ShowBalloon },
    { "CloseBalloon", CloseBalloon },
    { "ClosePopupPanel", (XtActionProc)ClosePopupPanel },
  };
  static String fallback_resources[] = {
#include "XVkbd-common.h"
    NULL,
  };

  char ch;
  Window child;
  int op, ev, err;

  argc1 = argc;
  argv1 = malloc(sizeof(char *) * (argc1 + 5));
  memcpy(argv1, argv, sizeof(char *) * argc1);
  argv1[argc1] = NULL;

#ifdef USE_I18N
  XtSetLanguageProc(NULL, NULL, NULL);
#endif

  toplevel = XtVaAppInitialize(NULL, "XVkbd",
                               options, XtNumber(options),
                               &argc, argv, fallback_resources, NULL);
  dpy = XtDisplay(toplevel);
  app_con = XtWidgetToApplicationContext(toplevel);
  XtAppAddActions(app_con, actions, XtNumber(actions));

  target_dpy = dpy;

  if (1 < argc) {
    fprintf(stderr, "%s: illegal option: %s\n\n", PROGRAM_NAME, argv[1]);
  }

  XtGetApplicationResources(toplevel, &appres,
            application_resources, XtNumber(application_resources),
            NULL, 0);
  if (appres.compact) {
    appres.keypad = FALSE;
    appres.function_key = FALSE;
  }
  if (appres.keypad_only && !appres.keypad) {
    appres.keypad_only = FALSE;
    fprintf(stderr, "%s: warning: keypad_only ignored\n", PROGRAM_NAME);
  }
  
  if (0 < strlen(appres.window)) {
    if (sscanf(appres.window, "0x%lX%c", &focused_window, &ch) != 1) {
      if (sscanf(appres.window, "%ld%c", &focused_window, &ch) != 1) {
        focused_window = FindWindow(RootWindow(dpy, DefaultScreen(dpy)),
                                    appres.window);
        if (focused_window == None) {
          fprintf(stderr, "%s: no such window: %s\n", PROGRAM_NAME, appres.window);
        }
      }
    }
  }
  focused_subwindow = focused_window;

  ReadKeymap();
  if (!altgr_mask && appres.auto_add_keysym) AddModifier(XK_Mode_switch);

  if (strlen(appres.text) != 0 || strlen(appres.file) != 0) {
    appres.keypad_keysym = TRUE;
    if (focused_window != None &&
        (appres.list_widgets || strlen(appres.widget) != 0)) {
      XtVaSetValues(toplevel, XtNwidth, 1, XtNheight, 1, NULL);
      XtRealizeWidget(toplevel);
      child = FindWidget(toplevel, focused_window, appres.widget);
      if (child != None) focused_subwindow = child;
    }
    if (strlen(appres.text) != 0)
      SendString(appres.text);
    else
      SendFileContent(appres.file);
    exit(0);
 } else {
    if (0 < strlen(appres.keys_normal))
      RedefineKeys(keys_normal, appres.keys_normal);
    if (0 < strlen(appres.keys_shift))
      RedefineKeys(keys_shift, appres.keys_shift);
    if (0 < strlen(appres.keys_altgr))
      RedefineKeys(keys_altgr, appres.keys_altgr);
    if (0 < strlen(appres.keys_shift_altgr))
      RedefineKeys(keys_shift_altgr, appres.keys_shift_altgr);

    if (0 < strlen(appres.key_labels))
      RedefineKeys(key_labels, appres.key_labels);
    if (0 < strlen(appres.normal_key_labels))
      RedefineKeys(normal_key_labels, appres.normal_key_labels);
    if (0 < strlen(appres.shift_key_labels))
      RedefineKeys(shift_key_labels, appres.shift_key_labels);
    if (0 < strlen(appres.altgr_key_labels))
      RedefineKeys(altgr_key_labels, appres.altgr_key_labels);
    if (0 < strlen(appres.shift_altgr_key_labels))
      RedefineKeys(shift_altgr_key_labels, appres.shift_altgr_key_labels);

    MakeKeyboard(FALSE);

    if (focused_window != None &&
        (appres.list_widgets || strlen(appres.widget) != 0)) {
      child = FindWidget(toplevel, focused_window, appres.widget);
      if (child != None) focused_subwindow = child;
    }

    if (main_menu != None) {
      if (strlen(appres.dict_file) == 0)
	XtSetSensitive(XtNameToWidget(main_menu, "*completion"), FALSE);
      if (strlen(appres.customizations) == 0)
	XtSetSensitive(XtNameToWidget(main_menu, "*select_layout"), FALSE);
    }

#ifdef USE_XTEST
    if (!XQueryExtension(dpy, "XTEST", &op, &ev, &err)) {
      if (appres.xtest) {
	fprintf(stderr, "%s: XTEST extension is not supported by the X server\n",
		PROGRAM_NAME);
	fprintf(stderr, "%s: XSendEvent will be used instead\n",
		PROGRAM_NAME);
	appres.xtest = FALSE;
      }
      if (main_menu != None) {
	XtSetSensitive(XtNameToWidget(main_menu, "*use_xtest"), FALSE);
	RefreshMainMenu();
      }
    }
#endif

    if (!appres.debug) {
#ifdef SYSV
      signal(SIGINT, SIG_IGN);
      signal(SIGQUIT, SIG_IGN);
#else
      struct sigaction sigact;
      sigact.sa_handler = SIG_IGN;
      sigemptyset(&sigact.sa_mask);
      sigact.sa_flags = 0;
      sigaction(SIGINT, &sigact, NULL);
      sigaction(SIGQUIT, &sigact, NULL);
#endif
    }

    ReadFuncionKeys();

    SetIconBitmap(toplevel);

    XtAppMainLoop(app_con);
  }
  exit(0);
}

/*
 * Replace setlocale() in the standard library here, because
 * it may not support some locales used for localized keyboards.
 */
#if defined(USE_I18N) && !defined(HAVE_SETLOCALE)

char *setlocale(int category, const char *locale)
{
  static char old_locale[100] = "C";
  static char cur_locale[100] = "C";
  const char *s;
  if (locale == NULL) {
    return cur_locale;
  } else if (category == LC_ALL) {
    strcpy(old_locale, cur_locale);
    if (locale[0] == '\0') {
      s = getenv("LC_ALL");
      if (s == NULL) s = "C";  /* LC_ALL not defined */
    } else {
      s = locale;
    }
    strncpy(cur_locale, s, sizeof(cur_locale) - 1);
    return old_locale;
  } else {
    return cur_locale;
  }
}
#endif  /* HAVE_SETLOCALE */
