/*
 * Copyright (c) 1997, 2000, 2001, Mark Buser.
 * Copyright (c) 2001, 2003, 2004, Danny Backx.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the names the authors (see above), nor the names of other
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $Header: /pack/anoncvs/xinvest/src/fileXq.c,v 1.20 2004/11/05 18:31:47 danny Exp $
 */
#include <errno.h>
#include <sys/stat.h>

#include <Xm/XmAll.h>

#include "opttick.h"
#include "optupdate.h"
#include "session.h"
#include "status.h"
#include "util.h"
#include "xutil.h"

#ifndef XQUOTE
#include "xinvest.h"
#endif

/* 
** File app menu support 
*/
Widget Fmenu[3];               /* File menu items           */
Widget OFmenu[3];              /* popup File menu items     */

static void loadfileselect( Widget, XtPointer, XtPointer );

static Widget Fdialog = NULL;  /* File selection dialog     */
static Widget Cdialog = NULL;  /* File clear confirm dialog */
static Widget Tbutton[2];      /* Log option select         */
static Widget Cbutton;         /* Log option ok/cancl       */

static int xformat=True;
static char *dir_and_file = NULL;
static FILE *LogFile = 0;

/* File pointer access routine */
FILE *getLogFilePointer ()
{
  return (LogFile);
}

/* File log format access routine */
int getLogFileFormat ()
{
  return (xformat);
}


static void clearlogfile( Widget w, XtPointer client_data, 
		          XtPointer call_data)
{
  if (LogFile) fclose (LogFile);

  /* Open for write, truncate to 0 length */
  LogFile = fopen (dir_and_file, "w");
  if (LogFile == NULL) {
    sprintf ( errmsg, 
              "File \"%.40s\" could not be reopened:\n%.40s", 
              dir_and_file, strerror(errno) );
    write_status ( errmsg, ERR );
  }

  XtUnmanageChild(Cdialog);
  XtPopdown( XtParent(Cdialog) );
}


/* ARGSUSED */
/* When File menu posted and activated */
void fileCB(Widget w, XtPointer client, XtPointer call)
{
	int	menu_pos = (int)client;
	char	*filename;

	switch (menu_pos) {
	case 0: /* Update from network now */
		triggerUpdate (TRIGGER_REFRESH, (XtIntervalId)NULL);
		break;

	case 1: /* Stop Update */
		triggerUpdate (TRIGGER_STOP, (XtIntervalId)NULL);
		break;

	case 2: /* Manual Save State */
		filename = sessSaveFile(True);
		saveState(filename);
		XtFree(filename);
		break;

	case 3:	/* User-initiated Live Update to Xinvest */
		workNet(7);
		/* netTellXinvest(); */
		break;

	case 4: /* Log to File */
	/* Make a file selection dialog if we don't have one */
		if (!Fdialog) {
			XmString pattern;

			Fdialog = XmCreateFileSelectionDialog (GetTopShell(w), 
				"Filesel", NULL, 0);
			XtSetSensitive (XmFileSelectionBoxGetChild(Fdialog, XmDIALOG_HELP_BUTTON),
				False);

			/* Set a pattern based on logging format */
			pattern = XmStringCreateSimple ((xformat)?"*.inv":"*.csv");
			XtVaSetValues (Fdialog, XmNpattern, pattern, NULL);
			XmStringFree (pattern);

			XtAddCallback (Fdialog, 
				XmNcancelCallback,
				(XtCallbackProc) XtUnmanageChild, 
				(XtPointer) NULL);
			XtAddCallback (Fdialog, 
				XmNokCallback,
				(XtCallbackProc)loadfileselect, 
				(XtPointer) NULL); 
		}
		XtManageChild(Fdialog);
		XtPopup(XtParent(Fdialog), XtGrabNone);
		break;

	case 5: /* End Log to File */
		if (LogFile) 
			fclose (LogFile);
		LogFile = 0;

		if (Fmenu[0])                        /* Enable start log */
			XtSetSensitive (Fmenu[0], True);
		if (OFmenu[0])
			XtSetSensitive (OFmenu[0], True);
		if (Fmenu[1])                        /* Disable end log, clear log */
			XtSetSensitive (Fmenu[1], False);
		if (OFmenu[1])
			XtSetSensitive (OFmenu[1], False);
		if (Fmenu[2])
			XtSetSensitive (Fmenu[2], False);
		if (OFmenu[2])
			XtSetSensitive (OFmenu[2], False);
		break;

	case 6: /* Clear Log File */
		if (dir_and_file[0] != '\0') {
			if (Cdialog == NULL) {
				XmString message;

				Cdialog = XmCreateQuestionDialog(GetTopShell(w), "Cdialog", 
					NULL, 0);
				XtSetSensitive(XmMessageBoxGetChild(Cdialog, XmDIALOG_HELP_BUTTON),
					False);

				sprintf (errmsg,"Ok to clear file \"%.80s\"?", dir_and_file);
				message = XmStringCreateLocalized (errmsg);
				XtVaSetValues (Cdialog, XmNmessageString, message, NULL);
				XmStringFree (message);

				XtAddCallback (Cdialog, 
					XmNcancelCallback,
					(XtCallbackProc) XtUnmanageChild, 
					(XtPointer) NULL);
				XtAddCallback (Cdialog, 
					XmNokCallback,
					(XtCallbackProc)clearlogfile, 
					(XtPointer) NULL); 
			}

			XtManageChild(Cdialog);
			XtPopup(XtParent(Cdialog), XtGrabNone);

		}
	break;

	case 7: /* Quit */
#if !defined(CDE) && XtSpecificationRelease > 5
		/* close session connection before quitting */
		XtVaSetValues ( GetTopShell(w), XtNjoinSession, False, NULL);
		XtCallCallbacks ( GetTopShell(w), XtNdieCallback, NULL);
#endif
		exit (0);
		break;

	default:
		break;
	}
}

/*
** Callbacks 
*/

/* When file selection  dialog ok or cancel activated */
static void loadfileselect ( Widget w,
                XtPointer client_data,
                XtPointer call_data )
{
  char *filename;
  char *directory;
  char *last;
  struct stat statb;

  XmFileSelectionBoxCallbackStruct *cbf =
     (XmFileSelectionBoxCallbackStruct *) call_data;

  /* Determine max number of characters in a path from root and allocate
  ** enough space to hold it. */
  int num = path_length();

  dir_and_file = XtRealloc ( dir_and_file, num*sizeof(char) );
  if (dir_and_file == NULL) {
    write_status ("Could not allocate memory for path.\n"
                  "File not opened.", ERR);
    return;
  }

  /* Break into directory and filename components */
  XmStringGetLtoR ( cbf->dir, XmFONTLIST_DEFAULT_TAG, &directory );
  if ( strlen(directory) )
    strcpy (dir_and_file, directory );
  else
    dir_and_file[0] = '\0'; 
  XtFree ( directory );

  XmStringGetLtoR ( cbf->value, XmFONTLIST_DEFAULT_TAG, &filename );
  if (strlen(filename)) {
    /* strip path components if any */
    last = strrchr ( filename, '/');
    if ( last && strlen(last+1) )
      strcat( dir_and_file, last+1 );
    else
      strcat( dir_and_file, filename );
  }
  XtFree ( filename );

  /* Check filename for logging */
  if (dir_and_file[0] == '\0') {
   write_status ( "No file specified!", ERR );
   return;
  }

  /* Some quick checks, does file exist? */
  if (stat (dir_and_file, &statb) == 0)
  {
    /* is directory? */
    if (S_ISDIR( statb.st_mode ) ) {
      sprintf ( errmsg, "File is a directory:\n\"%.80s\"", 
		dir_and_file);
      write_status (errmsg, ERR);
      return;
    } else {
      /* is writable? */
      if ( !(statb.st_mode & S_IWUSR) ) {
        sprintf ( errmsg, "File \"%.80s\" is read only.", 
		  dir_and_file);
        write_status ( errmsg, ERR );
        return;
      }
    }
  }

  /* Open the file for appending */
  LogFile = fopen (dir_and_file, "a");
  if (LogFile == NULL) {
    sprintf ( errmsg, "File \"%.80s\" could not be opened:\n%.40s", 
              dir_and_file, strerror(errno));
    write_status ( errmsg, ERR );
    return;
  }

  if (Fmenu[0])                        /* Disable another open */
    XtSetSensitive (Fmenu[0], False);
  if (OFmenu[0])
    XtSetSensitive (OFmenu[0], False);
  if (Fmenu[1])                        /* Enable stop/clear log function */
    XtSetSensitive (Fmenu[1], True);
  if (OFmenu[1])
    XtSetSensitive (OFmenu[1], True);
  if (Fmenu[2])
    XtSetSensitive (Fmenu[2], True);
  if (OFmenu[2])
    XtSetSensitive (OFmenu[2], True);

  XtUnmanageChild ( w );
}

/* When option dialog ok/cancel activated */
static void fileDialogResponse ( Widget widget,
                XtPointer client_data,
                XtPointer call_data )
{
  Widget dialog = (Widget)client_data;
  XmString pattern;

  if (widget == Cbutton) {
    if (Tbutton[0] && Tbutton[1]) {       /* Cancel, revert to old settings */
      XmToggleButtonSetState (Tbutton[0], (xformat==True)?True:False, False);
      XmToggleButtonSetState (Tbutton[1], (xformat==True)?False:True, False);
    }
  } else {
    if (Tbutton[0])        /* Ok, make any changes */
      xformat = XmToggleButtonGetState (Tbutton[0]);
  }

  /* Change file selection pattern to match format */
  if (Fdialog) {
    pattern = XmStringCreateSimple ((xformat)?"*.inv":"*.csv");
    XtVaSetValues (Fdialog, XmNpattern, pattern, NULL);
    XmStringFree (pattern);
  }

  XtUnmanageChild (dialog);
  XtPopdown (dialog);
}


/* Create the file logging option dialog */
Widget createLogDialog ( )
{
  Widget File_dialog = (Widget)NULL;

  Dimension height, width, border;
  char *buttons[] = { "button_0","button_1","button_2"};
  int num;

  /* Create option dialog */
  Widget pane, frame, form, row;

  File_dialog = XtVaCreatePopupShell( "OptionFile",
                              xmDialogShellWidgetClass,
                              GetTopShell(per->Toplevel),
			      XmNmappedWhenManaged, False,
                              NULL );
  /*
  ** This allows the frame to work and the buttons to not cause
  ** the dialog to auto unmanage.  Where is this documented?
  */
  pane = XtVaCreateWidget ( "FilePane", xmPanedWindowWidgetClass,
                            File_dialog,
                            XmNsashWidth, 1,
                            XmNsashHeight, 1,
                            NULL );

  frame =   XtVaCreateManagedWidget( "Fileframe",
                            xmFrameWidgetClass, pane,
                            NULL);

  form = XtVaCreateWidget ( "Fileform", xmFormWidgetClass, frame, NULL );
  row = XmVaCreateSimpleRadioBox ( form, "FileRow", 0, NULL,
                                   XmNtopAttachment, XmATTACH_FORM,
                                   XmNleftAttachment, XmATTACH_FORM,
                                   XmVaRADIOBUTTON, NULL, NULL, NULL, NULL,
                                   XmVaRADIOBUTTON, NULL, NULL, NULL, NULL,
                                   NULL);
  Tbutton[0] = XtNameToWidget (row, "button_0");
  Tbutton[1] = XtNameToWidget (row, "button_1");

  XtManageChild (row);
  XtManageChild (form);

  /* Buttons to ok, cancel */
  form = XtVaCreateWidget ("ButForm", xmFormWidgetClass, pane, NULL);

  for (num=0; num < XtNumber(buttons)-1; num++) {
    Cbutton = XtVaCreateManagedWidget (buttons[num],
                            xmPushButtonWidgetClass, form,
                            XmNtopAttachment, XmATTACH_FORM,
                            XmNbottomAttachment, XmATTACH_FORM,
                            XmNleftAttachment, XmATTACH_POSITION,
                            XmNrightAttachment, XmATTACH_POSITION,
                            XmNshowAsDefault, (num==0)?True:False,
                            XmNdefaultButtonShadowThickness, 1,
                            NULL);
    XtAddCallback (Cbutton, XmNactivateCallback,
                   (XtCallbackProc) fileDialogResponse, (XtPointer)File_dialog);
  }
  XtManageChild (form);
  XtManageChild (pane);

  /* Prevent bulletin board from changing size */
  XtVaGetValues ( File_dialog,
                  XmNwidth, &width,
                  XmNheight, &height,
                  XmNborderWidth, &border,
                  NULL );

  XtVaSetValues ( File_dialog,
                  XmNminWidth,  width +  border,
                  XmNmaxWidth,  width +  border,
                  XmNminHeight, height + border,
                  XmNmaxHeight, height + border,
                  NULL );
  return (File_dialog);
}
