/*
 * png.c - load and write routines for 'PNG' format pictures
 *
 * callable functions
 *
 *    LoadPNG(fname, pinfo)
 */

/* (c) 1995 by Alexander Lehmann <lehmann@mathematik.th-darmstadt.de>
 *   this file is a suplement to xv and is supplied under the same copying
 *   conditions (except the shareware part)
 * Modified by Andreas Dilger <adilger@enel.ucalgary.ca> to fix
 *   error handling for bad PNGs, add dialogs for interlacing and
 *   compression selection, and upgrade to libpng-0.89
 * The copyright will be passed on to JB at some future point if he
 * so desires.
 */
 /* Modified for use in ImagEngine for X Northern Captain
  * by Leo Khramov <leo@xnc.dubna.su>  1998
  */

#define JPEG_IMAGES
#include "picinfo.h"

#define HAVE_PNG
#ifdef HAVE_PNG

#include "png.h"

/*** Stuff for PNG Dialog box ***/
#define PWIDE 318
#define PHIGH 215

#define DISPLAY_GAMMA 2.20  /* Default display gamma */
/* Default zlib compression level
#define COMPRESSION   Z_BEST_COMPRESSION
*/
#define COMPRESSION   6

#define DWIDE     86
#define DHIGH    104
#define PFX PWIDE-93
#define PFY       44
#define PFH       20

#define P_BOK    0
#define P_BCANC  1
#define P_NBUTTS 2

#define BUTTH    24

#define LF       10   /* a.k.a. '\n' on ASCII machines */
#define CR       13   /* a.k.a. '\r' on ASCII machines */
/*** local functions ***/
static    void png_xv_error   (png_structp png_ptr,
                                    png_const_charp message);
static    void png_xv_warning (png_structp png_ptr,
                                    png_const_charp message);

static void FatalError(char *mes);

/*** local variables ***/
static char *filename;
static char *fbasename;
static int   colorType;
static int   read_anything;
static double Display_Gamma = DISPLAY_GAMMA;


/*******************************************/
int LoadPNG(fname, pinfo)
     char    *fname;
     PICINFO *pinfo;
/*******************************************/
{
  /* returns '1' on success */

  FILE  *fp;
  png_struct *png_ptr;
  png_info *info_ptr;
  png_color_16 my_background;
  int i,j;
  int linesize;
  int filesize;
  int pass;
  size_t commentsize;

  fbasename = fname;

  pinfo->pic     = (byte *) NULL;
  pinfo->comment = (char *) NULL;

  read_anything=0;

  /* open the file */
  fp = fopen(fname,"r");
  if (!fp)
  {
    return 0;
  }

  /* find the size of the file */
  fseek(fp, 0L, 2);
  filesize = ftell(fp);
  fseek(fp, 0L, 0);
  
  png_ptr = png_create_read_struct(PNG_LIBPNG_VER_STRING, NULL,
                                   png_xv_error, png_xv_warning);
  if(!png_ptr) {
    fclose(fp);
    FatalError("malloc failure in LoadPNG");
        return 0;
  }

  info_ptr = png_create_info_struct(png_ptr);

  if(!info_ptr) {
    fclose(fp);
    png_destroy_read_struct(&png_ptr, (png_infopp)NULL, (png_infopp)NULL);
    FatalError("malloc failure in LoadPNG");
        return 0;
  }

  if(setjmp(png_ptr->jmpbuf)) {
    fclose(fp);
    png_destroy_read_struct(&png_ptr, &info_ptr, (png_infopp)NULL);
    if(!read_anything) {
      if(pinfo->pic) {
        free(pinfo->pic);
        pinfo->pic = NULL;
      }
      if(pinfo->comment) {
        free(pinfo->comment);
        pinfo->comment = NULL;
      }
    }
    return read_anything;
  }

  png_init_io(png_ptr, fp);
  png_read_info(png_ptr, info_ptr);

  pinfo->w = info_ptr->width;
  pinfo->h = info_ptr->height;

  pinfo->frmType = F_PNG;

  sprintf(pinfo->fullInfo, "PNG, %d bit %dx%d ",
          info_ptr->bit_depth * info_ptr->channels,pinfo->w,pinfo->h);

  switch(info_ptr->color_type) {
    case PNG_COLOR_TYPE_PALETTE:
      strcat(pinfo->fullInfo, "palette color");
      break;

    case PNG_COLOR_TYPE_GRAY:
      strcat(pinfo->fullInfo, "grayscale");
      break;

    case PNG_COLOR_TYPE_GRAY_ALPHA:
      strcat(pinfo->fullInfo, "grayscale+alpha");
      break;

    case PNG_COLOR_TYPE_RGB:
      strcat(pinfo->fullInfo, "truecolor");
      break;

    case PNG_COLOR_TYPE_RGB_ALPHA:
      strcat(pinfo->fullInfo, "truecolor+alpha");
      break;
  }

  sprintf(pinfo->fullInfo + strlen(pinfo->fullInfo),
          ", %sinterlaced. (%d bytes)",
          info_ptr->interlace_type ? "" : "non-", filesize);

  sprintf(pinfo->shrtInfo, "%dx%d PNG", info_ptr->width, info_ptr->height);

  if (info_ptr->bit_depth < 8)
      png_set_packing(png_ptr);

  if (info_ptr->valid & PNG_INFO_gAMA)
    png_set_gamma(png_ptr, Display_Gamma, info_ptr->gamma);
  else
    png_set_gamma(png_ptr, Display_Gamma, 0.45);

  if (info_ptr->valid & PNG_INFO_bKGD)
    png_set_background(png_ptr, &info_ptr->background,
                       PNG_BACKGROUND_GAMMA_FILE, 1, 1.0);
  else {
    my_background.red = my_background.green = my_background.blue =
      my_background.gray = 0;
    png_set_background(png_ptr, &my_background, PNG_BACKGROUND_GAMMA_SCREEN,
                       0, Display_Gamma);
  }

  if (info_ptr->bit_depth == 16)
    png_set_strip_16(png_ptr);

  if (info_ptr->color_type == PNG_COLOR_TYPE_GRAY ||
      info_ptr->color_type == PNG_COLOR_TYPE_GRAY_ALPHA)
  {
    if (info_ptr->bit_depth == 1)
      pinfo->colType = F_BWDITHER;
    else
      pinfo->colType = F_GREYSCALE;
    png_set_expand(png_ptr);
  }

  pass=png_set_interlace_handling(png_ptr);

  png_read_update_info(png_ptr, info_ptr);

  if(info_ptr->color_type == PNG_COLOR_TYPE_RGB ||
     info_ptr->color_type == PNG_COLOR_TYPE_RGB_ALPHA) {
    linesize = pinfo->w * 3;
    pinfo->colType = F_FULLCOLOR;
    pinfo->type = PIC24;
  } else {
    linesize = pinfo->w;
    pinfo->type = PIC8;
    if(info_ptr->color_type == PNG_COLOR_TYPE_GRAY ||
       info_ptr->color_type == PNG_COLOR_TYPE_GRAY_ALPHA) {
      for(i = 0; i < 768; i+=3)
        pinfo->pal[i] = pinfo->pal[i+1] = pinfo->pal[i+2] = i;
    } else {
      pinfo->colType = F_FULLCOLOR;
      for(i = 0; i < info_ptr->num_palette; i++) {
        pinfo->pal[i*3] = info_ptr->palette[i].red;
        pinfo->pal[i*3+1] = info_ptr->palette[i].green;
        pinfo->pal[i*3+2] = info_ptr->palette[i].blue;
      }
    }
  }
  pinfo->pic = calloc((size_t)(linesize*pinfo->h), (size_t)1);

  if(!pinfo->pic) {
    png_error(png_ptr, "can't allocate space for PNG image");
  }

  png_start_read_image(png_ptr);

  for(i = 0; i < pass; i++) {
    byte *p = pinfo->pic;
    for(j = 0; j < pinfo->h; j++) {
      png_read_row(png_ptr, p, NULL);
      read_anything = 1;
/*      if((j & 0x1f) == 0) WaitCursor();*/
      p += linesize;
    }
  }

  png_read_end(png_ptr, info_ptr);

  if(info_ptr->num_text > 0) {
    commentsize = 1;

    for(i = 0; i < info_ptr->num_text; i++)
      commentsize += strlen(info_ptr->text[i].key) + 1 +
                     info_ptr->text[i].text_length + 2;

    if((pinfo->comment = malloc(commentsize)) == NULL) {
      png_warning(png_ptr,"can't allocate comment string");
    }
    else {
      pinfo->comment[0] = '\0';
      for(i = 0; i < info_ptr->num_text; i++) {
        strcat(pinfo->comment, info_ptr->text[i].key);
        strcat(pinfo->comment, "::");
        strcat(pinfo->comment, info_ptr->text[i].text);
        strcat(pinfo->comment, "\n");
      }
    }
  }

  png_destroy_read_struct(&png_ptr, &info_ptr, (png_infopp)NULL);

  fclose(fp);

  return 1;
}


/*******************************************/
static void
png_xv_error(png_ptr, message)
/*
     png_struct *png_ptr;
     char *message;
 */
     png_structp png_ptr;
     png_const_charp message;
{
  fprintf(stderr,"%s:  libpng error: %s", fbasename, message);

  longjmp(png_ptr->jmpbuf, 1);
}


/*******************************************/
static void
png_xv_warning(png_ptr, message)
/*
     png_struct *png_ptr;
     char *message;
 */
     png_structp png_ptr;
     png_const_charp message;
{
  if (!png_ptr)
    return;

  fprintf(stderr,"%s:  libpng warning: %s", fbasename, message);
}

static void FatalError(char *mes)
{
        fprintf(stderr,"PNG_Error: %s\n",mes);
}

#endif
