/*
 * xvbmp.c - i/o routines for .BMP files (MS Windows 3.x)
 *
 * LoadBMP(fname, numcols)
 */

#define JPEG_IMAGES
#include "picinfo.h"
#include "accel.h"
#include "image.h"

/* comments on error handling:
   a truncated file is not considered a Major Error.  The file is loaded, the
   rest of the pic is filled with 0's.

   a file with garbage characters in it is an unloadable file.  All allocated
   stuff is tossed, and LoadPBM returns non-zero

   not being able to malloc is a Fatal Error.  The program is aborted. */


#define BI_RGB  0
#define BI_RLE8 1
#define BI_RLE4 2

#define WIN_OS2_OLD 12
#define WIN_NEW     40
#define OS2_NEW     64

extern long filesize;

extern int bdep, b_pad;

int   loadBMP1   (FILE *, byte *, u_int, u_int, u_int);
int   loadBMP4   (FILE *, byte *, u_int, u_int, u_int, u_int);
int   loadBMP8  (FILE *, byte *, u_int, u_int, u_int, u_int);
int   loadBMP24 (FILE *, byte *, u_int, u_int, u_int);
u_int getshort   (FILE *);
u_int getint    (FILE *);
int   bmpError   (char *, char *);


#define FERROR(fp) (ferror(fp) || feof(fp))

/*******************************************/
int LoadBMP(char *fname,PICINFO* pinfo)
/*******************************************/
{
  FILE         *fp;
  int          i, c, c1, rv;
  unsigned int bfSize, bfOffBits, biSize, biWidth, biHeight, biPlanes;
  unsigned int biBitCount, biCompression, biSizeImage, biXPelsPerMeter;
  unsigned int biYPelsPerMeter, biClrUsed, biClrImportant;
  unsigned int L_Width;
  int bPad;
  char         *cmpstr;
  byte         *pic24, *pic8;
  char          buf[512], *bname;

  /* returns '1' on success */

  pic8 = pic24 = (byte *) NULL;
  bname = fname;

  fp = fopen(fname,"r");
  if (!fp) return (bmpError(bname, "couldn't open file"));
  
  fseek(fp, 0L, 2);      /* figure out the file size */
  filesize = ftell(fp);
  fseek(fp, 0L, 0);


  /* read the file type (first two bytes) */
  c = getc(fp);  c1 = getc(fp);
  if (c!='B' || c1!='M') { bmpError(bname,"file type != 'BM'"); goto ERROR; }

  bfSize = getint(fp);
  getshort(fp);         /* reserved and ignored */
  getshort(fp);
  bfOffBits = getint(fp);

  biSize          = getint(fp);

  if (biSize == WIN_NEW || biSize == OS2_NEW) {
    biWidth         = getint(fp);
    biHeight        = getint(fp);
    biPlanes        = getshort(fp);
    biBitCount      = getshort(fp);
    biCompression   = getint(fp);
    biSizeImage     = getint(fp);
    biXPelsPerMeter = getint(fp);
    biYPelsPerMeter = getint(fp);
    biClrUsed       = getint(fp);
    biClrImportant  = getint(fp);
  }

  else {    /* old bitmap format */
    biWidth         = getshort(fp);          /* Types have changed ! */
    biHeight        = getshort(fp);
    biPlanes        = getshort(fp);
    biBitCount      = getshort(fp);
    
    /* Not in old versions so have to compute them*/
    biSizeImage = (((biPlanes * biBitCount*biWidth)+31)/32)*4*biHeight;
    
    biCompression   = BI_RGB; 
    biXPelsPerMeter = biYPelsPerMeter = 0;
    biClrUsed       = biClrImportant  = 0;
  }


  if (FERROR(fp)) { bmpError(bname,"EOF reached in file header"); goto ERROR; }


  /* error checking */
  if ((biBitCount!=1 && biBitCount!=4 && biBitCount!=8 && biBitCount!=24) || 
      biPlanes!=1 || biCompression>BI_RLE4) {

    sprintf(buf,"Bogus BMP File!  (bitCount=%d, Planes=%d, Compression=%d)",
            biBitCount, biPlanes, biCompression);

    bmpError(bname, buf);
    goto ERROR;
  }

  if (((biBitCount==1 || biBitCount==24) && biCompression != BI_RGB) ||
      (biBitCount==4 && biCompression==BI_RLE8) ||
      (biBitCount==8 && biCompression==BI_RLE4)) {

    sprintf(buf,"Bogus BMP File!  (bitCount=%d, Compression=%d)",
            biBitCount, biCompression);

    bmpError(bname, buf);
    goto ERROR;
  }


  bPad = 0;
  if (biSize != WIN_OS2_OLD) {
    /* skip ahead to colormap, using biSize */
    c = biSize - 40;    /* 40 bytes read from biSize to biClrImportant */
    for (i=0; i<c; i++) getc(fp);
    
    bPad = bfOffBits - (biSize + 14);
  }

  /* load up colormap, if any */
  if (biBitCount!=24) {
    int i, cmaplen;

    cmaplen = (biClrUsed) ? biClrUsed : 1 << biBitCount;
    for (i=0; i<cmaplen; i++) {
      pinfo->pal[i*3+2] = getc(fp);
      pinfo->pal[i*3+1] = getc(fp);
      pinfo->pal[i*3] = getc(fp);
      if (biSize != WIN_OS2_OLD) {
        getc(fp);
        bPad -= 4;
      }
    }

    if (FERROR(fp)) 
      { bmpError(bname,"EOF reached in BMP colormap"); goto ERROR; }

  }

  if (biSize != WIN_OS2_OLD) {
    /* Waste any unused bytes between the colour map (if present)
       and the start of the actual bitmap data. */
    
    while (bPad > 0) {
      (void) getc(fp);
      bPad--;
    }
  }

  /* create pic8 or pic24 */

  if (biBitCount==24) {
    pinfo->w=biWidth;
    pinfo->h=biHeight;
    pic24 = (byte *) im_alloc_true(pinfo->w,pinfo->h);
    L_Width = im_get_linew_true(biWidth);
    pinfo->pic=pic24;
    pinfo->type=PIC24;
    if (!pic24) return (bmpError(bname, "couldn't malloc 'pic24'"));
  }
  else {
    im_init_translation();
    if(translation==0) {
       L_Width = biWidth;
    } else {
       L_Width = im_get_linew_true(biWidth);
    }
    pic8 = (byte *) malloc(biHeight*L_Width);
    if (!pic8) return(bmpError(bname, "couldn't malloc 'pic8'"));
  }


  /* load up the image */
  if      (biBitCount == 1) rv = loadBMP1(fp,pic8,biWidth,biHeight,L_Width);
  else if (biBitCount == 4) rv = loadBMP4(fp,pic8,biWidth,biHeight,L_Width,
                                          biCompression);
  else if (biBitCount == 8) rv = loadBMP8(fp,pic8,biWidth,biHeight,L_Width,
                                          biCompression);
  else                      rv = loadBMP24(fp,pic24,biWidth,biHeight,L_Width);

  if (rv) bmpError(bname, "File appears truncated.  Winging it.\n");

  fclose(fp);


  if (biBitCount == 24) {
    pinfo->pic  = pic24;
    pinfo->type = PIC24;
  }
  else {
    pinfo->pic  = pic8;
    pinfo->type = PIC8;
  }

  cmpstr = "";
  if      (biCompression == BI_RLE4) cmpstr = ", RLE4 compressed";
  else if (biCompression == BI_RLE8) cmpstr = ", RLE8 compressed";

  pinfo->w = biWidth;  pinfo->h = biHeight;
  pinfo->frmType = F_BMP;

  sprintf(pinfo->fullInfo, "%sBMP, %d bit%s per pixel%s.  (%ld bytes)",
          ((biSize==WIN_OS2_OLD) ? "Old OS/2 " :
           (biSize==WIN_NEW)     ? "Windows "  : ""),
          biBitCount,  (biBitCount == 1) ? "" : "s",
          cmpstr, filesize);

  sprintf(pinfo->shrtInfo, "%dx%d BMP.", biWidth, biHeight);
  pinfo->comment = (char *) NULL;

  return 1;


 ERROR:
  fclose(fp);
  return 0;
}  


/*******************************************/
int loadBMP1(FILE* fp,byte* pic8,u_int w,u_int h,u_int lw)
{
  int   i,j,c,bitnum,padw;
  byte *pp;

  c = 0;
  padw = ((w + 31)/32) * 32;  /* 'w', padded to be a multiple of 32 */

  for (i=h-1; i>=0; i--) {
    pp = pic8 + i * lw;
    for (j=bitnum=0; j<padw; j++,bitnum++) {
      if ((bitnum&7) == 0) { /* read the next byte */
        c = getc(fp);
        bitnum = 0;
      }
      
      if (j<w) {
        im_coding256((char**) &pp, (c & 0x80) ? 1 : 0);
        c <<= 1;
      }
    }
    if (FERROR(fp)) break;
  }

  return (FERROR(fp));
}  



/*******************************************/
int loadBMP4(FILE* fp,byte* pic8,u_int w,u_int h,u_int lw,u_int comp)
{
  int   i,j,c,c1,x,y,nybnum,padw,rv;
  byte *pp;
  
  
  rv = 0;
  c = c1 = 0;
  
  if (comp == BI_RGB) {   /* read uncompressed data */
    padw = ((w + 7)/8) * 8; /* 'w' padded to a multiple of 8pix (32 bits) */
    
    for (i=h-1; i>=0; i--) {
      pp = pic8 + i * lw;
      
      for (j=nybnum=0; j<padw; j++,nybnum++) {
        if ((nybnum & 1) == 0) { /* read next byte */
          c = getc(fp);
          nybnum = 0;
        }
        
        if (j<w) {
          im_coding256((char**) &pp, (c & 0xf0) >> 4);
          c <<= 4;
        }
      }
      if (FERROR(fp)) break;
    }
  }
  
  else if (comp == BI_RLE4) {  /* read RLE4 compressed data */
    x = y = 0;  
    pp = pic8  + (h-1)*lw;
    
    while (y<h) {
      c = getc(fp);  if (c == EOF) { rv = 1;  break; }
      
      if (c) {                                   /* encoded mode */
        c1 = getc(fp);
        for (i=0; i<c; i++,x++) 
          im_coding256((char**) &pp, (i&1) ? (c1 & 0x0f) : ((c1>>4)&0x0f));
      }
      
      else {    /* c==0x00  :  escape codes */
        c = getc(fp);  if (c == EOF) { rv = 1;  break; }
        
        if      (c == 0x00) {                    /* end of line */
          x=0;  y++;  pp = pic8  + (h-y-1)*lw;
        } 
        
        else if (c == 0x01) break;               /* end of pic8 */
        
        else if (c == 0x02) {                    /* delta */
          c = getc(fp);  x += c;
          c = getc(fp);  y += c;
          pp = pic8 + x * bdep + (h-y-1) * lw;
        }
        
        else {                                   /* absolute mode */
          for (i=0; i<c; i++, x++) {
            if ((i&1) == 0) c1 = getc(fp);
            im_coding256((char**) &pp, (i&1) ? (c1 & 0x0f) : ((c1>>4)&0x0f));
          }
          
          if (((c&3)==1) || ((c&3)==2)) getc(fp);  /* read pad byte */
        }
      }  /* escape processing */
      if (FERROR(fp)) break;
    }  /* while */
  }
  
  else {
    fprintf(stderr,"unknown BMP compression type 0x%0x\n", comp);
  }
  
  if (FERROR(fp)) rv = 1;
  return rv;
}  



/*******************************************/
int loadBMP8(FILE* fp,byte* pic8,u_int w,u_int h,u_int lw,u_int comp)
{
  int   i,j,c,c1,padw,x,y,rv;
  byte *pp;
  
  rv = 0;

  if (comp == BI_RGB) {   /* read uncompressed data */
    padw = ((w + 3)/4) * 4; /* 'w' padded to a multiple of 4pix (32 bits) */

    for (i=h-1; i>=0; i--) {
      pp = pic8 + i * lw;

      for (j=0; j<padw; j++) {
        c = getc(fp);  if (c==EOF) rv = 1;
        if (j<w) im_coding256((char**) &pp,c);
      }
      if (FERROR(fp)) break;
    }
  }

  else if (comp == BI_RLE8) {  /* read RLE8 compressed data */
    x = y = 0;  
    pp = pic8 + (h-1)*lw;

    while (y<h) {
      c = getc(fp);  if (c == EOF) { rv = 1;  break; }

      if (c) {                                   /* encoded mode */
        c1 = getc(fp);
        for (i=0; i<c; i++,x++) im_coding256((char**) &pp,c1);
      }

      else {    /* c==0x00  :  escape codes */
        c = getc(fp);  if (c == EOF) { rv = 1;  break; }

        if      (c == 0x00) {                    /* end of line */
          x=0;  y++;  pp = pic8 + (h-y-1)*lw;
        } 

        else if (c == 0x01) break;               /* end of pic8 */

        else if (c == 0x02) {                    /* delta */
          c = getc(fp);  x += c;
          c = getc(fp);  y += c;
          pp = pic8 + x * bdep + (h-y-1)*lw;
        }

        else {                                   /* absolute mode */
          for (i=0; i<c; i++, x++) {
            c1 = getc(fp);
            im_coding256((char**) &pp,c1);
          }
          
          if (c & 1) getc(fp);  /* odd length run: read an extra pad byte */
        }
      }  /* escape processing */
      if (FERROR(fp)) break;
    }  /* while */
  }
  
  else {
    fprintf(stderr,"unknown BMP compression type 0x%0x\n", comp);
  }

  if (FERROR(fp)) rv = 1;
  return rv;
}  



/*******************************************/
int loadBMP24(FILE* fp,byte* pic24,u_int w,u_int h,u_int lw)
{
  int   i,j,padb,rv;
  unsigned rbit,gbit,bbit;
  byte *pp;

  rv = 0;

  padb = (4 - ((w*3) % 4)) & 0x03;  /* # of pad bytes to read at EOscanline */
  pp=pic24;
  for (i=h-1; i>=0; i--) {
    pp=pic24+i*lw;
    for (j=0; j<w; j++) {
      bbit = getc(fp);   /* blue */
      gbit = getc(fp);   /* green */
      rbit = getc(fp);   /* red */
      im_coding(rbit,gbit,bbit,(char**) &pp);
    }

    for (j=0; j<padb; j++) getc(fp);

    rv = (FERROR(fp));
    if (rv) break;
  }

  return rv;
}  



/*******************************************/
unsigned int getshort(FILE* fp)
{
  int c, c1;
  c = getc(fp);  c1 = getc(fp);
  return ((unsigned int) c) + (((unsigned int) c1) << 8);
}


/*******************************************/
unsigned int getint(FILE* fp)
{
  int c, c1, c2, c3;
  c = getc(fp);  c1 = getc(fp);  c2 = getc(fp);  c3 = getc(fp);
  return ((unsigned int) c) +
         (((unsigned int) c1) << 8) + 
         (((unsigned int) c2) << 16) +
         (((unsigned int) c3) << 24);
}



/*******************************************/
int bmpError(char* fname,char* st)
{
  fprintf(stderr,"Image Error %s:  %s", fname, st);
  return 0;
}



