/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * filename: m-xtract.c                                                    *
 *                                                                         *
 * UTIL C-source: Medical Image Conversion Utility                         *
 *                                                                         *
 * purpose      : extract specified images                                 *
 *                                                                         *
 * project      : (X)MedCon by Erik Nolf                                   *
 *                                                                         *
 * Functions    : MdcGetImagesToExtract() - Ask images to extract          * 
 *                MdcExtractImages()      - Extract/Reorder the images     *
 *                                                                         *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* $Id: m-xtract.c,v 1.1.1.1 2000/10/28 16:51:09 enlf Exp $
 */

/*
   Copyright (C) 1997-2000 by Erik Nolf

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 2, or (at your option) any later
   version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
   Public License for more details.

   You should have received a copy of the GNU General Public License along
   with this program; if not, write to the Free Software Foundation, Inc.,
   59 Place - Suite 330, Boston, MA 02111-1307, USA.  */

/****************************************************************************
                              H E A D E R S
****************************************************************************/

#include "medcon.h"

/****************************************************************************
                              D E F I N E S 
****************************************************************************/

MdcExtractInputStruct mdcextractinput;

/****************************************************************************
                            F U N C T I O N S
****************************************************************************/

int MdcGetImagesToExtract(FILEINFO *fi, MdcExtractInputStruct *input)
{
   Uint32 images=1;
   Uint32 *frames, *planes, *gates, *beds;
   Uint32 p,f,g,b;
   Uint32 it, bt;
   int error;

   /* initialize the extract input structure */
   input->style = MDC_INPUT_NORM_STYLE;
   input->inrs = NULL;
   input->num_p = 1;
   input->num_f = 1;
   input->num_g = 1;
   input->num_b = 1;

   MdcPrintLine('-',MDC_FULL_LENGTH);
   printf("\tEXTRACT IMAGES\t\tFILE: %s\n",fi->ifname);
   MdcPrintLine('-',MDC_FULL_LENGTH);

   input->style = MdcGetSelectionType();

   if ( input->style == MDC_INPUT_ECAT_STYLE ) { /* ecat input type */

     printf("\n");
     printf("\n\tInput notes: a) Finish the list with a negative number");
     printf("\n\t             b) Any number must be one-based (0 = All)");
     printf("\n\t             c) The syntax of a range: X...Y\n");
     printf("\n\tExample    : 1 3 5...10 12 -1\n");
     if ( (planes=(Uint32 *)malloc((fi->dim[3]+1)*sizeof(Uint32)))==NULL ) {
       MdcPrntWarn("Couldn't allocate planes buffer");
       return(MDC_NO);
     }
     memset(planes,0,(fi->dim[3]+1)*sizeof(Uint32));
     if ( (frames=(Uint32 *)malloc((fi->dim[4]+1)*sizeof(Uint32)))==NULL ) {
       MdcPrntWarn("Couldn't allocate frames buffer");
       MdcFree(planes);
       return(MDC_NO);
     }
     memset(frames,0,(fi->dim[4]+1)*sizeof(Uint32));
     if ( (gates=(Uint32 *)malloc((fi->dim[5]+1)*sizeof(Uint32)))==NULL ) {
       MdcPrntWarn("Couldn't allocate gates buffer");
        MdcFree(planes); MdcFree(frames);
        return(MDC_NO);
     }
     memset(gates,0,(fi->dim[5]+1)*sizeof(Uint32));
     if ( (beds=(Uint32 *)malloc((fi->dim[6]+1)*sizeof(Uint32)))==NULL ) {
       MdcPrntWarn("Couldn't allocate beds buffer");
       MdcFree(planes); MdcFree(frames); MdcFree(gates);
       return(MDC_NO);
     }
     memset(beds,0,(fi->dim[6]+1)*sizeof(Uint32));

     printf("\n\tGive planes list (1...%u): ",fi->dim[3]);
     MdcGetStrInput(mdcbufr,MDC_2KB_OFFSET);

     if (MdcHandleEcatList(mdcbufr,&planes,fi->dim[3]) != MDC_OK) {
       MdcFree(planes); MdcFree(frames); MdcFree(gates); MdcFree(beds);
       return(MDC_BAD_CODE);
     }

     printf("\n\tGive frames list (1...%u): ",fi->dim[4]);
     MdcGetStrInput(mdcbufr,MDC_2KB_OFFSET);
     
     if (MdcHandleEcatList(mdcbufr,&frames,fi->dim[4]) != MDC_OK) {
       MdcFree(planes); MdcFree(frames); MdcFree(gates); MdcFree(beds);
       return(MDC_BAD_CODE);
     }

     printf("\n\tGive gates  list (1...%u): ",fi->dim[5]);
     MdcGetStrInput(mdcbufr,MDC_2KB_OFFSET);

     if (MdcHandleEcatList(mdcbufr,&gates,fi->dim[5]) != MDC_OK) {
       MdcFree(planes); MdcFree(frames); MdcFree(gates); MdcFree(beds);
       return(MDC_BAD_CODE);
     }

     printf("\n\tGive beds  list (1...%u): ",fi->dim[6]);
     MdcGetStrInput(mdcbufr,MDC_2KB_OFFSET);
      
     if (MdcHandleEcatList(mdcbufr,&beds,fi->dim[6]) != MDC_OK) {
       MdcFree(planes); MdcFree(frames); MdcFree(gates); MdcFree(beds);
       return(MDC_BAD_CODE);
     }

     images*=planes[0]*frames[0]*gates[0]*beds[0];

     input->num_p = planes[0];
     input->num_f = frames[0];
     input->num_g = gates[0];
     input->num_b = beds[0];


     if ((input->inrs=(Uint32 *)malloc((images+1)*sizeof(Uint32)))==NULL) {
       MdcFree(planes); MdcFree(frames); MdcFree(gates); MdcFree(beds);
       MdcPrntWarn("Couldn't malloc images number buffer");
       return(MDC_BAD_ALLOC);
     }

     /* get sequential image numbers (like normal selection) */
     it = 1;
     for (b=1; b<=fi->dim[6];b++) if (beds[b]) 
     for (g=1; g<=fi->dim[5];g++) if (gates[g])
     for (f=1; f<=fi->dim[4];f++) if (frames[f])
     for (p=1; p<=fi->dim[3];p++) if (planes[p]) {

        images = p +                     /* the image number */
                 fi->dim[3]*( (f-1) +
                 fi->dim[4]*( (g-1) +
                 fi->dim[5]*( (b-1)   ) ) );
       
        input->inrs[it++]=images;  

     }

     MdcFree(planes); MdcFree(frames); MdcFree(gates); MdcFree(beds);
 
   }else{  /* normal input type */

     if ((input->inrs=(Uint32 *)malloc(MDC_BUF_ITMS*sizeof(Uint32)))==NULL) {
       MdcPrntWarn("Couldn't allocate images number buffer");
       return(MDC_BAD_ALLOC);
     }

     printf("\n\tInput notes: a) Finish the list with a negative number");
     printf("\n\t             b) Any number must be one-based ");
     printf(" (0 = All reversed)");
     printf("\n\t             c) The syntax of a range: X...Y");
     printf("\n\t             d) The list is sequence sensitive!\n");
     printf("\n\tExample    : 1 3 12...6 84 31 -1\n");
     printf("\n\tYour input [1...%u]: ",fi->number);
     MdcGetStrInput(mdcbufr,MDC_2KB_OFFSET);

     it = 1; bt = 2;
     error = MdcHandleNormList(mdcbufr,&input->inrs,&it,&bt,fi->number);
     if (error != MDC_OK) {
       MdcFree(input->inrs);
       return(error);
     } 

   }

   input->inrs[0] = it - 1;

   MdcPrintLine('-',MDC_FULL_LENGTH);

   if (input->inrs[0] == 0) {
     MdcPrntWarn("No images specified!");
     MdcFree(input->inrs);
     return(MDC_BAD_CODE);
   }

   return(MDC_YES);

}

int MdcExtractImages(FILEINFO *fi) 
{  
   MdcExtractInputStruct *input = &mdcextractinput;
   Uint32 i, j, bytes;
   char *msg=NULL;
   IMG_DATA id_tmp, *new_image;
   IMG_DATA *id_src, *id_dest;
   int error;

   /* skip command-line asking for XMedCon ... already handled */
   if (XMDC_MEDCON == MDC_NO) {
     error=MdcGetImagesToExtract(fi,input);
     if (error != MDC_YES)  return(error);
   }

   if (input->inrs[1] == 0) { /* reverse images */
     for (i=0; i < (fi->number/2); i++) {
        memcpy(&id_tmp,&fi->image[i],sizeof(IMG_DATA));
        memcpy(&fi->image[i],&fi->image[fi->number-1-i],sizeof(IMG_DATA));
        memcpy(&fi->image[fi->number-1-i],&id_tmp,sizeof(IMG_DATA));
     }
   }else{              /* extract/reorder images */ 

     new_image = (IMG_DATA *)malloc(input->inrs[0]*sizeof(IMG_DATA));
     if (new_image == NULL) {
#if XSUPPORTED
       if (XMDC_MEDCON) {
         XMdcDisplayErr("Couldn't alloc new IMG_DATA array");
       }else
#endif
       MdcPrntWarn("Couldn't alloc new IMG_DATA array");
       MdcFree(input->inrs);
       return(MDC_BAD_ALLOC);
     }

     for (i=1; i<=input->inrs[0]; i++) {
#if XSUPPORTED
       if (XMDC_MEDCON) {
         pvalue += 1./(float)input->inrs[0];
         XMdcUpdateProgressBar(NULL);
       }
#endif 
       id_dest = &new_image[i-1];
       id_src  = &fi->image[input->inrs[i]-1];
       bytes = id_src->width * id_src->height * MdcType2Bytes(id_src->type);

       /* copy IMG_DATA struct    */
       memcpy((Uint8 *)id_dest,(Uint8 *)id_src,sizeof(IMG_DATA));

       /* copy fresh image buffer */
       id_dest->buf = MdcGetImgBuffer(bytes);  
       if (id_dest->buf == NULL) {
#if XSUPPORTED
         if (XMDC_MEDCON) {
           XMdcDisplayErr("Couldn't alloc new image buffer");
         }else
#endif
           MdcPrntWarn("Couldn't alloc new image buffer");
         /* free up previous allocated resources */
         for (j=0; j < i-1; j++) MdcFree(new_image[j].buf);
         MdcFree(new_image);
         MdcFree(input->inrs);
 
         return(MDC_BAD_ALLOC);

       }else{

         memcpy(id_dest->buf,id_src->buf,bytes);

       }

     }

     /* remove all previous images */
     for (i=0; i<fi->number; i++) MdcFree(fi->image[i].buf);

     /* re-init FI */
     MdcFree(fi->image);
     fi->number = input->inrs[0];
     if (input->style == MDC_INPUT_ECAT_STYLE) {
       /* preserve ECAT style image */
         fi->dim[0] = 6;
         fi->dim[3] = input->num_p;
         fi->dim[4] = input->num_f;
         fi->dim[5] = input->num_g;
         fi->dim[6] = input->num_b;
     }else{
         fi->dim[0] = 3; 
         fi->dim[3] = fi->number;
         for (i=4; i < 8; i++) fi->dim[i] = 1;
     }
     fi->image = new_image;
     if (fi->acquisition_type == MDC_ACQUISITION_DYNAMIC) {
       if (fi->dim[4] > 1) { 
         fi->acquisition_type = MDC_ACQUISITION_DYNAMIC;
       }else{
         fi->acquisition_type = MDC_ACQUISITION_TOMO;
       }
     }
     fi->endian = MDC_FILE_ENDIAN = MDC_HOST_ENDIAN;
     msg=MdcImagesPixelFiddle(fi);
   }

   MdcFree(input->inrs);

   if (msg != NULL) {
#if XSUPPORTED
     if (XMDC_MEDCON) {
       XMdcDisplayFatalErr(MDC_BAD_CODE,msg);
     }else
#endif
     MdcPrntWarn(msg);
     return(MDC_BAD_CODE);
   }

   return(MDC_OK);
}

