/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * filename: m-intf.c                                                      *
 *                                                                         *
 * UTIL C-source: Medical Image Conversion Utility                         *
 *                                                                         *
 * purpose      : read and write InterFile 3.3                             *
 *                                                                         *
 * project      : (X)MedCon by Erik Nolf                                   *
 *                                                                         *
 * Functions    : MdcCheckINTF()        - Check for InterFile 3.3 format   *
 *                MdcGetIntfKey()       - Get InterFile key                *
 *                MdcInitIntf()         - Init InterFile struct (defaults) *
 *                MdcThisString()       - String occurency test            *
 *                MdcGetIntKey()        - Get key with integer             *
 *                MdcGetYesNoKey()      - Get key Y or N                   *
 *                MdcGetFloatKey()      - Get key with float               *
 *                MdcGetStrKey()        - Get key with string              *
 *                MdcGetDataType()      - Get data type of pixels          *
 *                MdcGetProcessStatus() - Get process status               *
 *                MdcGetPatientRot()    - Get patient rotation             *
 *                MdcGetPatientOrient() - Get patient orientation          *
 *                MdcGetSliceOrient()   - Get slice orient                 *
 *                MdcGetPatSlOrient()   - Get patient slice orientation    *
 *                MdcGetPixelType()     - Get pixel data type              *
 *                MdcGetRotation()      - Get rotation direction           *
 *                MdcGetMotion()        - Get detector motion              *
 *                MdcSpecifyPixelType() - Specify pixel data type (bytes)  *
 *                MdcReadIntfHeader()   - Read InterFile header            *
 *                MdcReadIntfImages()   - Read InterFile images            *
 *                MdcReadINTF()         - Read InterFile file              *
 *                MdcType2Intf()        - Translate data type to InterFile *
 *                MdcGetProgramDate()   - Get date in correct format       *
 *                MdcWriteGenImgData()  - Write general image data         *
 *                MdcWriteWindows()     - Write energy windows             *
 *                MdcWriteMatrixInfo()  - Write matrix info                *
 *                MdcWriteIntfStatic()  - Write a Static header            *
 *                MdcWriteIntfDynamic() - Write a Dynamic header           *
 *                MdcWriteIntfTomo()    - Write a Tomographic header       *
 *                MdcWriteIntfHeader()  - Write InterFile header           *
 *                MdcWriteIntfImages()  - Write InterFile images           *
 *                MdcWriteINTF()        - Write InterFile file             * 
 *                                                                         *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* $Id: m-intf.c,v 1.1.1.1 2000/10/28 16:51:33 enlf Exp $
 */

/*
   Copyright (C) 1997-2000 by Erik Nolf

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 2, or (at your option) any later
   version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
   Public License for more details.

   You should have received a copy of the GNU General Public License along
   with this program; if not, write to the Free Software Foundation, Inc.,
   59 Place - Suite 330, Boston, MA 02111-1307, USA.  */

/****************************************************************************
                              H E A D E R S
****************************************************************************/

#include "medcon.h"

/****************************************************************************
                              D E F I N E S 
****************************************************************************/

static char keystr[MDC_INTF_MAXKEYCHARS+1];
static char keystr_case[MDC_INTF_MAXKEYCHARS+1];

static Uint32 ACQI = 0;

/****************************************************************************
                            F U N C T I O N S
****************************************************************************/

int MdcCheckINTF(FILEINFO *fi)
{
  if (MdcGetIntfKey(fi->ifp) != MDC_OK) return(MDC_BAD_READ);

  if (strstr(keystr,MDC_INTF_SIG) == NULL)  return(MDC_FRMT_BAD);

  return(MDC_FRMT_INTF);
}


int MdcGetIntfKey(FILE *fp)
{
  char *pkeyval = NULL;

  /* assure string termination */
  memset(keystr,'\0',MDC_INTF_MAXKEYCHARS+1);

  fgets(keystr,MDC_INTF_MAXKEYCHARS,fp);
  if (ferror(fp)) return(MDC_BAD_READ);

  memcpy(keystr_case,keystr,MDC_INTF_MAXKEYCHARS+1);

  pkeyval = strstr(keystr,":=");
  if (pkeyval != NULL) {
    pkeyval += 2;
    MdcKillSpaces(pkeyval);
  }

  MdcLowStr(keystr);

  return MDC_OK;
  
}


void MdcInitIntf(MDC_INTERFILE *intf)
{
  intf->data_type = MDC_INTF_STATIC;
  intf->process_status = MDC_INTF_UNKNOWN;
  intf->pixel_type = BIT8_U;
  intf->data_offset = intf->data_blocks = 0;
  intf->imagesize = intf->number_images = 0;
  intf->images_per_dimension = 1;
  intf->energy_windows = intf->frame_groups = 1;
  intf->time_windows = intf->detector_heads = 1;
  intf->slice_thickness=1.;
  intf->centre_centre_separation=1.;
  intf->study_duration=0.;
  intf->image_duration = 0.;
  intf->image_pause = 0.;
  intf->group_pause = 0.;
  intf->ext_rot = 0.;
  intf->patient_rot = MDC_SUPINE;
  intf->patient_orient = MDC_HEADFIRST;
  intf->slice_orient =  MDC_TRANSVERSAL;
}

int MdcIsEmptyKeyValue(void)
{
   char *pkeyval = NULL;
 
   pkeyval = strstr(keystr,":=") + 2;

   if (pkeyval[0] == '\0') return(MDC_YES);

   return(MDC_NO);
}

int MdcThisString(char *string)
{
  if( strstr(keystr,string) != NULL) return MDC_YES;
  return MDC_NO;
}

int MdcGetIntKey(void)
{
  return(atoi(strstr(keystr,":=") + 2));
}

int MdcGetYesNoKey(void)
{
  strcpy(mdcbufr,(strstr(keystr,":=") + 2));
  MdcKillSpaces(mdcbufr);
  if (mdcbufr[0] == 'y') return MDC_YES;
  if (mdcbufr[0] == 'n') return MDC_NO;
  return(MDC_NO);
}
  

double MdcGetFloatKey(void)
{
  double d;

  d = (double)atof(strstr(keystr,":=") + 2);
  return(d);
}

void MdcGetStrKey(char *str)
{
  memcpy(str,(strstr(keystr_case,":=") + 2),MDC_MAXSTR-1);
  str[MDC_MAXSTR-1] = '\0';
  MdcKillSpaces(str);
}

int MdcGetDataType(void)
{
  if (MdcThisString("static"))
    return MDC_INTF_STATIC;
  if (MdcThisString("dynamic"))
    return MDC_INTF_DYNAMIC;
  if (MdcThisString("gated"))
    return MDC_INTF_GATED;
  if (MdcThisString("tomographic"))
    return MDC_INTF_TOMOGRAPH;
  if (MdcThisString("curve"))
    return MDC_INTF_CURVE;
  if (MdcThisString("roi"))
    return MDC_INTF_ROI;

  return MDC_INTF_UNKNOWN;
}

int MdcGetProcessStatus(void)
{
  if (MdcThisString("acquired"))
    return MDC_INTF_ACQUIRED;
  if (MdcThisString("reconstructed"))
    return MDC_INTF_RECONSTRUCTED;
 
  return MDC_INTF_UNKNOWN;
}

int MdcGetPatientRot(void)
{
  if (MdcThisString("supine")) return MDC_SUPINE;
  if (MdcThisString("prone"))  return MDC_PRONE;
  
  return MDC_UNKNOWN;
}

int MdcGetPatientOrient(void)
{
  if (MdcThisString("head")) return MDC_HEADFIRST;
  if (MdcThisString("feet")) return MDC_FEETFIRST;
 
  return MDC_UNKNOWN;
}

int MdcGetSliceOrient(void)
{
  if (MdcThisString("transverse")) return MDC_TRANSVERSAL;
  if (MdcThisString("sagittal"))   return MDC_SAGITTAL;
  if (MdcThisString("coronal"))    return MDC_CORONAL;

  return MDC_UNKNOWN;
}  

int MdcGetPatSlOrient(MDC_INTERFILE *intf)
{
  switch (intf->patient_rot) {
    case MDC_SUPINE: 
        switch (intf->patient_orient) {
               case MDC_HEADFIRST: 
                   switch (intf->slice_orient) {
                         case MDC_TRANSVERSAL:
                             return(MDC_SUPINE_HEADFIRST_TRANSVERSAL); break;
                         case MDC_SAGITTAL   :
                             return(MDC_SUPINE_HEADFIRST_SAGITTAL);    break;
                         case MDC_CORONAL    :
                             return(MDC_SUPINE_HEADFIRST_CORONAL);     break; 
                   }
                   break;

               case MDC_FEETFIRST: 
                   switch(intf->slice_orient) {
                         case MDC_TRANSVERSAL: 
                             return(MDC_SUPINE_FEETFIRST_TRANSVERSAL); break;
                         case MDC_SAGITTAL   :
                             return(MDC_SUPINE_FEETFIRST_SAGITTAL);    break;
                         case MDC_CORONAL    :
                             return(MDC_SUPINE_FEETFIRST_CORONAL);     break;
                   }
                   break;
        }
        break;
    case MDC_PRONE : 
        switch (intf->patient_orient) {
              case MDC_HEADFIRST:  
                  switch (intf->slice_orient) {
                         case MDC_TRANSVERSAL: 
                             return(MDC_PRONE_HEADFIRST_TRANSVERSAL);  break;
                         case MDC_SAGITTAL   :
                             return(MDC_PRONE_HEADFIRST_SAGITTAL);     break;
                         case MDC_CORONAL    :
                             return(MDC_PRONE_HEADFIRST_CORONAL);      break;
                  }
                  break;
              case MDC_FEETFIRST:
                  switch (intf->slice_orient) {
                         case MDC_TRANSVERSAL: 
                             return(MDC_PRONE_FEETFIRST_TRANSVERSAL);  break;
                         case MDC_SAGITTAL   :
                             return(MDC_PRONE_FEETFIRST_SAGITTAL);     break;
                         case MDC_CORONAL    :
                             return(MDC_PRONE_FEETFIRST_CORONAL);      break;
                  }
                  break;
        }
        break;
  }

  return(MDC_SUPINE_HEADFIRST_TRANSVERSAL); /* default for InterFile (!) */

}

int MdcGetPixelType(void)
{
  if (MdcThisString("unsigned integer")) return BIT8_U;
  if (MdcThisString("signed integer"))   return BIT8_S;
  if (MdcThisString("long float"))       return FLT64;
  if (MdcThisString("short float"))      return FLT32;
  if (MdcThisString("bit"))              return BIT1;
  if (MdcThisString("ascii"))            return ASCII;

  return BIT8_U;
}

int MdcGetRotation(void)
{
  if (MdcThisString("ccw")) return(MDC_ROTATION_CC);
  if (MdcThisString("cw"))  return(MDC_ROTATION_CW);

  return(MDC_UNKNOWN);
}

int MdcGetMotion(void)
{
  if (MdcThisString("step"))       return(MDC_MOTION_STEP);
  if (MdcThisString("continuous")) return(MDC_MOTION_CONT);

  return(MDC_UNKNOWN);
}

int MdcSpecifyPixelType(MDC_INTERFILE *intf)
{
   int bytes;

   bytes = MdcGetIntKey();

   if (intf->pixel_type == BIT8_S) 
     switch (bytes) {
       case 1: break;
       case 2: intf->pixel_type = BIT16_S; break;
       case 4: intf->pixel_type = BIT32_S; break;
       case 8: intf->pixel_type = BIT64_S; break;
      default: intf->pixel_type = 0;
   }else if (intf->pixel_type == BIT8_U)
     switch (bytes) {
       case 1: break;
       case 2: intf->pixel_type = BIT16_U; break;
       case 4: intf->pixel_type = BIT32_U; break;
       case 8: intf->pixel_type = BIT64_U; break;
      default: intf->pixel_type = 0;
  }

  return intf->pixel_type;

}
        

char *MdcReadIntfHeader(FILEINFO *fi, MDC_INTERFILE *intf)
{
  ACQ_DATA *acq=NULL;
  IMG_DATA *id, *id0;
  FILE *fp = fi->ifp;
  Uint32 i, counter=0, img = 0;
  char *err=NULL, *pfname=NULL;

  if (MDC_INFO) {
    MdcPrintLine('-',MDC_HALF_LENGTH);
    printf("InterFile Header\n");
    MdcPrintLine('-',MDC_HALF_LENGTH);
  }

  while (!feof(fp)) {
    if (MdcGetIntfKey(fp) != MDC_OK) return("INTF Bad read of key");
    if (err != NULL) return(err);
    if (MDC_INFO) {
      printf("%s",keystr_case);
    }
    if (ferror(fp)) return("INTF Bad read header file");
    if (MdcThisString(";")) continue;
    if (MdcThisString("version of keys")) {
      intf->version = MdcGetFloatKey(); continue;
    }
    if (MdcThisString("data compression")) {
      if (! (MdcThisString("none") || MdcIsEmptyKeyValue()) ) 
        return("INTF Don't handle compressed images");
    }
    if (MdcThisString("data encode")) {
      if (! (MdcThisString("none") || MdcIsEmptyKeyValue()) )
        return("INTF Don't handle encoded images");
    }
    if (MdcThisString("data starting block")) {
      intf->data_blocks = MdcGetIntKey();
      continue;
    }
    if (MdcThisString("data offset in bytes")) {
      intf->data_offset = MdcGetIntKey();
      continue;
    }
    if (MdcThisString("name of data file")) {
      pfname = strstr(keystr_case,":=") + 2;
      MdcKillSpaces(pfname);
      if ( strlen(pfname) > 0 ) {
        /* protect against empty key */
        fi->ifname = pfname;
        if (MdcThisString("/") || MdcThisString("\\")) { 
          /* absolute path */ 
          strcpy(fi->ipath,fi->ifname);
        }else{
          /* no absolute path */
          if (fi->idir != NULL) {
            strcpy(fi->ipath,fi->idir);
            strcat(fi->ipath,"/");
            strcat(fi->ipath,fi->ifname);
          }else{ 
            strcpy(fi->ipath,fi->ifname); 
          }
        } 
        MdcSplitPath(fi->ipath,fi->idir,fi->ifname);
      }
      continue;
    } 
    if (MdcThisString("patient name")) {
      MdcGetStrKey(fi->patient_name); continue;
    }
    if (MdcThisString("patient id")) {
      MdcGetStrKey(fi->patient_id); continue;
    }  
    if (MdcThisString("patient sex")) {
      MdcGetStrKey(fi->patient_sex); continue;
    }

    if (MdcThisString("study id")) {
      MdcGetStrKey(fi->study_name); continue;
    }
    
    if (MdcThisString("total number of images")) {
      fi->number = MdcGetIntKey();
      if (fi->number == 0) return("INTF No valid images specified");
      if (!MdcGetStructID(fi)) return("INTF Bad malloc IMG_DATA structs"); 
      id0 = &fi->image[0];
      continue;
    }
    if (MdcThisString("imagedata byte order")) {
      if (MdcThisString("bigendian")) 
        MDC_FILE_ENDIAN = MDC_BIG_ENDIAN;
      else if (MdcThisString("littleendian")) 
        MDC_FILE_ENDIAN = MDC_LITTLE_ENDIAN;
      else 
        MDC_FILE_ENDIAN = MDC_BIG_ENDIAN;
      fi->endian = MDC_FILE_ENDIAN; 
      continue;
    }
    if (MdcThisString("type of data")) {
      intf->data_type = MdcGetDataType(); 
      if (intf->data_type == MDC_INTF_UNKNOWN)
        intf->data_type = MDC_INTF_STATIC;  /* take this as default */
      switch (intf->data_type) {
        case MDC_INTF_CURVE      : /* we do not use this yet */
        case MDC_INTF_GATED      : /* we do not use this yet */
        case MDC_INTF_ROI        : /* we do not use this yet */
        case MDC_INTF_STATIC     : 
                          fi->acquisition_type = MDC_ACQUISITION_STATIC; 
                          break;
        case MDC_INTF_DYNAMIC    : 
                          fi->acquisition_type = MDC_ACQUISITION_DYNAMIC;
                          break;
        case MDC_INTF_TOMOGRAPH  : 
                          fi->acquisition_type = MDC_ACQUISITION_TOMO;
                          break;
        default         : fi->acquisition_type = MDC_ACQUISITION_STATIC;
      }
      continue;
    }
    if (MdcThisString("study date")) {
      sscanf((char *)(strstr(keystr,":=")+2),"%4hd:%2hd:%2hd"
                                            ,&fi->study_date_year
                                            ,&fi->study_date_month
                                            ,&fi->study_date_day);
      continue;
    }

    if (MdcThisString("study time")) {
      sscanf((char *)(strstr(keystr,":=")+2),"%2hd:%2hd:%2hd"
                                            ,&fi->study_time_hour
                                            ,&fi->study_time_minute
                                            ,&fi->study_time_second);
      continue;
    } 
    
    if (MdcThisString("number of energy windows")) {
      intf->energy_windows = MdcGetIntKey(); continue;
    }

    if (MdcThisString("flood corrected")) {
      fi->flood_corrected = MdcGetYesNoKey();
      continue;
    }  
    if (MdcThisString("decay corrected")) {
      fi->decay_corrected = MdcGetYesNoKey();
      continue;
    } 
    switch (intf->data_type) {
      case MDC_INTF_STATIC:
      case MDC_INTF_ROI:
          if (MdcThisString("image number")) {
            img = MdcGetIntKey() - 1;
            continue;
          }
          if (MdcThisString("number of images/energy window")) {
            intf->number_images = MdcGetIntKey();
            intf->images_per_dimension = intf->number_images;
            continue;
          }
          if (MdcThisString("matrix size [1]")) {
            intf->width = MdcGetIntKey();
            for (i=img; i<fi->number; i++) { /* fill the rest too */
               fi->image[i].width = intf->width;
            }
            continue;
          }
          if (MdcThisString("matrix size [2]")) {
            intf->height = MdcGetIntKey();
            for (i=img; i<fi->number; i++) { /* fill the rest too */ 
               fi->image[i].height = intf->height;
            }
            continue;
          }
          if (MdcThisString("number format")) {
            intf->pixel_type = MdcGetPixelType();
            for (i=img; i<fi->number; i++) { /* fill the rest too */
               fi->image[i].type = intf->pixel_type;
               fi->image[i].bits = MdcType2Bits(fi->image[i].type);
            }
            continue;
          }
          if (MdcThisString("number of bytes per pixel")) {
            intf->pixel_type = MdcSpecifyPixelType(intf); 
            for (i=img; i<fi->number; i++) { /* fill the rest too */
               fi->image[i].type = intf->pixel_type;
               fi->image[i].bits = MdcType2Bits(fi->image[i].type);
            }
            continue;
          }
          if (MdcThisString("scaling factor (mm/pixel) [1]")) {
            fi->image[img].pixel_xsize = (float)MdcGetFloatKey();
            continue;
          }
          if (MdcThisString("scaling factor (mm/pixel) [2]")) {
            fi->image[img].pixel_ysize = (float)MdcGetFloatKey();
            continue;
          }
        break;

      case MDC_INTF_DYNAMIC:
          if (MdcThisString("number of frame groups")) {
            intf->frame_groups = MdcGetIntKey(); 
            continue;
          }
          if (MdcThisString("frame group number")) {
            counter = MdcGetIntKey();
          }else{
           if (MdcThisString("matrix size [1]")) {
             intf->width = MdcGetIntKey();
             continue;
           }
           if (MdcThisString("matrix size [2]")) {
             intf->height = MdcGetIntKey();
             continue;
           }
           if (MdcThisString("number format")) {
             intf->pixel_type = MdcGetPixelType();
             continue;
           }
           if (MdcThisString("number of bytes per pixel")) {
             intf->pixel_type = MdcSpecifyPixelType(intf);
             continue;
           }
           if (MdcThisString("scaling factor (mm/pixel) [1]")) {
             intf->pixel_xsize = (float)MdcGetFloatKey();
             continue;
           }
           if (MdcThisString("scaling factor (mm/pixel) [2]")) {
             intf->pixel_ysize = (float)MdcGetFloatKey();
             continue;
           }
           if (MdcThisString("number of images this frame group")) {
             intf->number_images = MdcGetIntKey();
             intf->images_per_dimension = intf->number_images;
             continue;
           }
           if (MdcThisString("image duration (sec)")) {
             intf->image_duration=MdcGetFloatKey();
             continue;
           }
           if (MdcThisString("pause between images (sec)")) {
             intf->image_pause=MdcGetFloatKey();
             continue;
           }
           if (MdcThisString("pause between frame groups (sec)")) {
             intf->group_pause=MdcGetFloatKey();
             continue;
           }
          }
          /* write info to all images in frame group # */
          if ( MdcThisString("frame group number") 
            || MdcThisString("end of interfile"))
          if (counter > 0 && counter <= intf->frame_groups
                          && img < fi->number) { 
            for (i=0; i<intf->number_images; i++, img++) {
               if (i == fi->number) break;
               id = &fi->image[img];
               id->type = intf->pixel_type;
               id->bits = MdcType2Bits(id->type);
               id->width = intf->width;
               id->height = intf->height;
               id->pixel_xsize = intf->pixel_xsize;
               id->pixel_ysize = intf->pixel_ysize;
               id->frame_duration =((intf->image_duration+intf->image_pause)
                                    * 1000.0) * intf->number_images;
               id->frame_start    = intf->group_pause * 1000.0;
            }

            intf->number_images=0;
            /* fill the rest too */
            for (i=img; i<fi->number; i++) {
               id = &fi->image[i];
               id->type = intf->pixel_type;
               id->bits = MdcType2Bits(id->type);
               id->width = intf->width;
               id->height = intf->height;
               id->pixel_xsize = intf->pixel_xsize;
               id->pixel_ysize = intf->pixel_ysize;
               id->frame_duration =((intf->image_duration+intf->image_pause)
                                    * 1000.0) * intf->number_images;
               id->frame_start    = intf->group_pause * 1000.0;

            }
          }
        break;

      case MDC_INTF_GATED:
          if (MdcThisString("matrix size [1]")) {
            intf->width = MdcGetIntKey();
            continue;
          }
          if (MdcThisString("matrix size [2]")) {
            intf->height = MdcGetIntKey();
            continue;
          }
          if (MdcThisString("number format")) {
            intf->pixel_type = MdcGetPixelType();
            continue;
          }
          if (MdcThisString("number of bytes per pixel")) {
            intf->pixel_type = MdcSpecifyPixelType(intf);
            continue;
          }
          if (MdcThisString("scaling factor (mm/pixel) [1]")) {
            intf->pixel_xsize = (float)MdcGetFloatKey();
            continue;
          }
          if (MdcThisString("scaling factor (mm/pixel) [2]")) {
            intf->pixel_ysize = (float)MdcGetFloatKey();
            continue;
          }
          if (MdcThisString("number of time windows")) {
            intf->time_windows = MdcGetIntKey();
            continue;
          }
          if (MdcThisString("time window number")) {
            counter = MdcGetIntKey();
          }else{ 
           if (MdcThisString("number of images in time window")) {
             intf->number_images = MdcGetIntKey();
             intf->images_per_dimension = intf->number_images;
             continue;
           }
          }
          if ( MdcThisString("time window number")
            || MdcThisString("end of interfile"))
          if (counter > 0  && counter <= intf->time_windows
                          && img < fi->number) {
            for (i=0; i<intf->number_images; i++, img++) {
               if (i == fi->number) break;
               id = &fi->image[img];
               id->type = intf->pixel_type;
               id->bits = MdcType2Bits(id->type);
               id->width = intf->width;
               id->height = intf->height;
               id->pixel_xsize = intf->pixel_xsize;
               id->pixel_ysize = intf->pixel_ysize;
            }

            intf->number_images=0;
            /* fill in the rest too */
            for (i=img; i<fi->number; i++) {
               id = &fi->image[i];
               id->type = intf->pixel_type;
               id->bits = MdcType2Bits(id->type);
               id->width = intf->width;
               id->height = intf->height;
               id->pixel_xsize = intf->pixel_xsize;
               id->pixel_ysize = intf->pixel_ysize;
            }

          }
        break;

      case MDC_INTF_TOMOGRAPH:
          if (MdcThisString("number of detector heads")) {
            intf->detector_heads = MdcGetIntKey();
            continue;
          }
          if (MdcThisString("process status")) {
            intf->process_status = MdcGetProcessStatus();
            switch (intf->process_status) {
              case MDC_INTF_ACQUIRED      : 
                  fi->reconstructed = MDC_NO;
                  fi->acqnr = intf->detector_heads * intf->energy_windows;
                  if (!MdcGetStructAD(fi))
                    return("INTF Couldn't malloc ACQ_DATA structs");
                  break;
              case MDC_INTF_RECONSTRUCTED : 
                  fi->reconstructed = MDC_YES;
                  break;
              default            : fi->reconstructed = MDC_YES;
            }
            continue;
          }
          if (MdcThisString("matrix size [1]")) {
            intf->width = MdcGetIntKey();
            continue;
          }
          if (MdcThisString("matrix size [2]")) {
            intf->height = MdcGetIntKey();
            continue;
          }
          if (MdcThisString("number format")) {
            intf->pixel_type = MdcGetPixelType();
            continue;
          }
          if (MdcThisString("number of bytes per pixel")) {
            intf->pixel_type = MdcSpecifyPixelType(intf);
            continue;
          }
          if (MdcThisString("scaling factor (mm/pixel) [1]")) {
            intf->pixel_xsize = (float)MdcGetFloatKey();
            continue;
          }
          if (MdcThisString("scaling factor (mm/pixel) [2]")) {
            intf->pixel_ysize = (float)MdcGetFloatKey();
            continue;
          }
          if (MdcThisString("number of projections")) {
            intf->number_images = MdcGetIntKey();
            intf->images_per_dimension = intf->number_images;
            continue;
          }
          if (MdcThisString("extent of rotation")) {
            intf->ext_rot = (float)MdcGetFloatKey();
            continue;
          } 
          if (MdcThisString("patient rotation")) {
            intf->patient_rot = MdcGetPatientRot();
            continue;
          }
          if (MdcThisString("patient orientation")) {
            intf->patient_orient = MdcGetPatientOrient();
            continue;
          } 
          switch (intf->process_status) {
            case MDC_INTF_ACQUIRED:
                if (MdcThisString("spect study (acquired data)")) {
                  if (fi->acqnr > counter && fi->acqdata != NULL) {
                    acq = &fi->acqdata[counter];
                    acq->scan_arc = intf->ext_rot;
                    if (intf->number_images > 0)
                      acq->angle_step=intf->ext_rot/(float)intf->number_images;
                  }else{
                    acq = NULL;
                  }
                  counter += 1;
                  continue;
                }
                if (MdcThisString("direction of rotation")) {
                  if (acq != NULL) {
                    acq->rotation_direction = (Int16)MdcGetRotation();
                  }
                  continue;
                }
                if (MdcThisString("acquisition mode")) {
                  if (acq != NULL) {
                    acq->detector_motion = (Int16)MdcGetMotion();
                  }
                  continue;
                }
                if (MdcThisString("start angle")) {
                  if (acq != NULL) {
                    acq->angle_start = (float)MdcGetFloatKey();
                  }
                  continue;
                }
                break;
            case MDC_INTF_RECONSTRUCTED:
                if (MdcThisString("method of reconstruction")) {
                  MdcGetStrKey(fi->recon_method);
                  continue;
                }
                if (MdcThisString("number of slices")) {
                  intf->number_images = MdcGetIntKey();
                  intf->images_per_dimension = intf->number_images;
                  continue;
                }
                if (MdcThisString("slice orientation")) {
                  intf->slice_orient = MdcGetSliceOrient();
                  continue;
                }
                if (MdcThisString("slice thickness (pixels)")) {
                  intf->slice_thickness = MdcGetFloatKey();
                  continue;
                }
                if (MdcThisString("centre-centre slice separation (pixels)"))
                {
                  intf->centre_centre_separation = MdcGetFloatKey();
                  continue;
                }
                if (MdcThisString("center-center slice separation (pixels)"))
                {
                  intf->centre_centre_separation = MdcGetFloatKey();
                  continue;
                }
                if (MdcThisString("filter name")) {
                  MdcGetStrKey(fi->filter_type);
                  continue;
                }
                if (MdcThisString("spect study (reconstructed data)")) {
                  counter += 1;
                }
                break;
          }
          if ((MdcThisString("spect study (acquired data)") 
                         && (counter>1))  ||
              (MdcThisString("spect study (reconstructed data)") 
                         && (counter>1))  ||
              (MdcThisString("end of interfile")))
          if (counter <= intf->detector_heads  && img < fi->number) {
            for (i=0; i<intf->number_images; i++, img++) {
               if (i == fi->number) break;
               id = &fi->image[img];
               id->type = intf->pixel_type;
               id->bits = MdcType2Bits(id->type);
               id->width = intf->width;
               id->height = intf->height;
               id->pixel_xsize = intf->pixel_xsize;
               id->pixel_ysize = intf->pixel_ysize;

               id->slice_width  = ((id->pixel_xsize + id->pixel_ysize)/2.)
                                  * intf->slice_thickness;

               id->pat_slice_orient = MdcGetPatSlOrient(intf);
               id->slice_spacing= ((id->pixel_xsize + id->pixel_ysize)/2.)
                                  * intf->centre_centre_separation;

               MdcFillImgPos(fi,i,i,0.0);
               MdcFillImgOrient(fi,i);

               id->frame_duration = intf->study_duration * 1000.0;
            }
            intf->number_images = 0;
            /* fill in the rest too */
            for (i=img; i<fi->number; i++) {
               id = &fi->image[i];
               id->type = intf->pixel_type;
               id->bits = MdcType2Bits(id->type);
               id->width = intf->width;
               id->height = intf->height;
               id->pixel_xsize = intf->pixel_xsize;
               id->pixel_ysize = intf->pixel_ysize;

               id->slice_width = ((id->pixel_xsize + id->pixel_ysize)/2.)
                                 * (float)intf->slice_thickness;

               id->pat_slice_orient = MdcGetPatSlOrient(intf);
               id->slice_spacing= ((id->pixel_xsize + id->pixel_ysize)/2.)
                                  * intf->centre_centre_separation;

               MdcFillImgPos(fi,i,i,0.0);
               MdcFillImgOrient(fi,i);

               id->frame_duration = intf->study_duration * 1000.0;

            } 

          }
        break;

      case MDC_INTF_CURVE:
          MdcFileClose(fi->ifp);
          return("INTF Curve data not supported");
          break;

    }

    if (MdcThisString("end of interfile")) break;
  }

  if (MDC_INFO) {
    printf("\n");
    MdcPrintLine('-',MDC_HALF_LENGTH);
  }
    
  return NULL;
}

char *MdcReadIntfImages(FILEINFO *fi, MDC_INTERFILE *intf)
{
  IMG_DATA *id;
  Uint32 i, p, bytes, nbr;
  char *err;

  /* set FILE pointer to begin of data */
  if (intf->data_offset > 0L) 
    fseek(fi->ifp,intf->data_offset,SEEK_SET);
  if (intf->data_blocks > 0L) 
    fseek(fi->ifp,intf->data_blocks * 2048L,SEEK_SET);

  for (i=0; i<fi->number; i++) {
#if XSUPPORTED
     if (XMDC_MEDCON) {
       pvalue += 1./(float)fi->number;
       XMdcUpdateProgressBar(NULL);
     }
#endif

     id = &fi->image[i];
     bytes = id->width * id->height * MdcType2Bytes(id->type);
     if ( (id->buf = MdcGetImgBuffer(bytes)) == NULL)
       return("INTF Bad malloc image buffer");
     switch(id->type) {
      case  BIT1: /* convert directly to BIT8_U */
          {
            bytes = MdcPixels2Bytes(id->width * id->height);
            if (fread(id->buf,1,bytes,fi->ifp) != bytes) {
              err=MdcHandleTruncated(fi,i+1,MDC_YES);
              if (err != NULL) return(err);
            }
            MdcMakeBIT8_U(id->buf, fi, i);
            id->type = BIT8_U; 
          }
          break;

      case ASCII: 
          {
            double *pix = (double *)id->buf;

            for (p=0; p<(id->width*id->height); p++) {
               fscanf(fi->ifp,"%le",&pix[p]);
               if (ferror(fi->ifp)) {
                 err=MdcHandleTruncated(fi,i+1,MDC_YES);
                 if (err != NULL) return(err);
                 break;
               } 
            } 
            id->type = FLT64; MDC_FILE_ENDIAN = MDC_HOST_ENDIAN;
          }
          break;

        default: 
          if ((nbr=fread(id->buf,1,bytes,fi->ifp)) != bytes) { 
              if (nbr > 0) err=MdcHandleTruncated(fi,i+1,MDC_YES);
              else err=MdcHandleTruncated(fi,i,MDC_YES);
              if (err != NULL) return(err);
          } 
     } 
     if (fi->truncated) break;
  }

  return NULL;

}


char *MdcReadINTF(FILEINFO *fi)
{
  MDC_INTERFILE intf;
  char *err, *origpath=NULL;
  Uint32 i, check=1;
  Int8 WAS_COMPRESSED=MDC_NO;

  /* put in some defaults */
  fi->endian = MDC_FILE_ENDIAN = MDC_BIG_ENDIAN;
  fi->flood_corrected = MDC_YES;
  fi->decay_corrected = MDC_NO;
  fi->reconstructed = MDC_YES;
 
#if XSUPPORTED
  if (XMDC_MEDCON) XMdcBeginProgressBar("Reading InterFile:");
#endif

  if (MDC_VERBOSE) MdcPrntMesg("INTF Reading <%s> ...",fi->ifname);

  /* preserve original input path - before header read fills fi->ipath */
  MdcMergePath(fi->ipath,fi->idir,fi->ifname);
  if ((origpath=malloc(strlen(fi->ipath) + 1)) == NULL) {
    return("INTF Couldn't allocate original path");
  }
  strcpy(origpath,fi->ipath);
  MdcSplitPath(fi->ipath,fi->idir,fi->ifname);

  /* initialize intf struct */
  MdcInitIntf(&intf);

  /* read the header */
  err=MdcReadIntfHeader(fi, &intf);
  if (err != NULL) { MdcFree(origpath); return(err); }

  MdcFileClose(fi->ifp);

  /* complete FILEINFO stuct */
  fi->type  = fi->image[0].type;
  fi->bits  = fi->image[0].bits;

  fi->dim[3] = intf.images_per_dimension;
  fi->dim[7] = intf.energy_windows;

  switch (intf.data_type) {
    case MDC_INTF_DYNAMIC:
        fi->dim[4] = intf.frame_groups;
        break;
    case MDC_INTF_TOMOGRAPH:
        fi->dim[6] = intf.detector_heads;
        break;
    case MDC_INTF_GATED:
        fi->dim[5] = intf.time_windows;
        break;
  }

  for (i=7; i>3; i--) if (fi->dim[i] > 1) break;
  fi->dim[0] = i;

  /* check fi->dim[] integrity */
  for (i=7; i>2; i--) check*=fi->dim[i];
  if (check != fi->number) { 
    if (fi->dim[0] == 3) {
      /* bad total images defined          */
      /* sometimes for reconstructed TOMO  */
      /* the <number of slices> key was not*/
      /* filled in properly                */

      /* we don't issue a warning ...      */
      /*
#if XSUPPORTED
       if (XMDC_MEDCON) {
          XMdcDisplayWarn("INTF Wrong number of images defined");
       }else
#endif
       {
          MdcPrntWarn("INTF Wrong number of images defined");
       }
      */ 
    }else{ 
      /* damn, an asymmetric amount of images per  */
      /* dimension is really unsupported           */
      /* => no frames, no time_windows, no nothing */
      /*    keeping only one dimension = TOMO      */
#if XSUPPORTED
       if (XMDC_MEDCON) {
         XMdcDisplayWarn("INTF Garbled or unsupported images/dimension:\n" \
                         "** using one dimensional array **\n" \
                         "** image position values might be corrupted **" );
       }else
#endif
       {
         MdcPrntWarn("INTF Garbled or unsupported images/dimension:\n" \
                     "\t\t      - using one dimensional array\n"       \
                     "\t\t      - image position values might be corrupted");
       }

       intf.data_type = MDC_INTF_TOMOGRAPH; /* disable dynamic etc ... */
 
    }

    /* fix the dimensions */
    fi->dim[0] = 3;
    fi->dim[3] = fi->number;
    for (i=4; i<8; i++) fi->dim[i] = 1;

  }

  fi->pixdim[0] = 3.;
  if (fi->image[0].pixel_xsize == 0. ) fi->pixdim[1]=1.;
  else fi->pixdim[1] = fi->image[0].pixel_xsize;
  if (fi->image[0].pixel_ysize == 0. ) fi->pixdim[2]=1.;
  else fi->pixdim[2] = fi->image[0].pixel_ysize;
  if (fi->image[0].slice_width == 0)
    fi->pixdim[3]= (fi->pixdim[1] + fi->pixdim[2]) / 2. ;
  else 
    fi->pixdim[3]=fi->image[0].slice_width;


#ifndef HAVE_8BYTE_INT
  /* check some unsupported things */
  for (i=0; i<fi->number; i++)
     if (fi->image[i].type == BIT64_S || fi->image[i].type == BIT64_U) {
       MdcFree(origpath); return("INTF Unsupported data type BIT64");
     }
#endif

  
  MdcMergePath(fi->ipath,fi->idir,fi->ifname);

  /* check for compression */
  if (MdcWhichCompression(fi->ipath) != MDC_NO) {
    /* "name of data file" with proper .Z or .gz extension */
    if (MdcDecompressFile(fi->ipath) != MDC_OK) {
      MdcFree(origpath); return("INTF Decompression image file failed");
    }

    WAS_COMPRESSED = MDC_YES;

  }else{
    if (MdcFileExists(fi->ipath) == MDC_NO) {
      /* no uncompressed image file found           */
      /* so we look for the compressed image file   */
      /* depending on the compression of the header */
      /* => result from doing `gzip basename.*` ;-) */
      MdcAddCompressionExt(fi->compression, fi->ipath);
    
      if (MdcDecompressFile(fi->ipath) != MDC_OK) {
        MdcFree(origpath); return("INTF Decompression image file failed");
      }

      /* Yep, you loose if you're using a different */
      /* compression for header and image files     */

      WAS_COMPRESSED = MDC_YES;

    }
  }


  /* open the (decompressed) image file */
  if ( (fi->ifp=fopen(fi->ipath,"rb")) == NULL) {
    MdcFree(origpath); return("INTF Couldn't open image file");
  }

  if (WAS_COMPRESSED == MDC_YES) { 
    unlink(fi->ipath); /* delete after use */
#if XSUPPORTED
    if (XMDC_MEDCON) XMdcBeginProgressBar("Reading InterFile:");
#endif
  }

  MdcSplitPath(fi->ipath,fi->idir,fi->ifname);

  err=MdcReadIntfImages(fi, &intf);
  if (err != NULL) { MdcFree(origpath); return(err); }

  MdcFileClose(fi->ifp);

  /* restore original filename */
  strcpy(fi->ipath,origpath); 
  MdcSplitPath(fi->ipath,fi->idir,fi->ifname);
  MdcFree(origpath);

  if (fi->truncated) return("INTF Truncated image file");

  return NULL; 

}

char *MdcType2Intf(int type)
{
  switch (type) {
   case BIT1   : return("bit"); break;
   case BIT8_U :
   case BIT16_U:
   case BIT32_U:
   case BIT64_U: return("unsigned integer"); break;
   case BIT8_S :
   case BIT16_S:
   case BIT32_S:
   case BIT64_S: return("signed integer"); break; 
   case FLT32  : return("short float"); break;
   case FLT64  : return("long float"); break;
   case ASCII  : return("ASCII"); break;
  }
  return("unsigned integer");
}


char *MdcGetProgramDate(void)
{
  int date, month=0, year;

  sscanf(MDC_DATE,"%2d-%3s-%4d",&date,keystr,&year);

  if (     MdcThisString("Jan")) month=1;
  else if (MdcThisString("Feb")) month=2;
  else if (MdcThisString("Mar")) month=3;
  else if (MdcThisString("Apr")) month=4;
  else if (MdcThisString("May")) month=5;
  else if (MdcThisString("Jun")) month=6;
  else if (MdcThisString("Jul")) month=7;
  else if (MdcThisString("Aug")) month=8;
  else if (MdcThisString("Sep")) month=9;
  else if (MdcThisString("Oct")) month=10;
  else if (MdcThisString("Nov")) month=11;
  else if (MdcThisString("Dec")) month=12;

  sprintf(keystr,"%04d:%02d:%02d",year,month,date);

  return(keystr);

}

char *MdcWriteGenImgData(FILEINFO *fi)
{
  FILE *fp = fi->ofp;

  fprintf(fp,";\r\n!GENERAL IMAGE DATA :=\r\n");
  fprintf(fp,"!type of data := ");
  switch (fi->acquisition_type) {
   case MDC_ACQUISITION_UNKNOWN: fprintf(fp,"Static\r\n");      break;
   case MDC_ACQUISITION_STATIC : fprintf(fp,"Static\r\n");      break;
   case MDC_ACQUISITION_DYNAMIC: fprintf(fp,"Dynamic\r\n");     break;
   case MDC_ACQUISITION_TOMO   : fprintf(fp,"Tomographic\r\n"); break;
   default                     : fprintf(fp,"Static\r\n");      break;
  }
  fprintf(fp,"!total number of images := %u\r\n",fi->number);
  fprintf(fp,"study date := %04d:%02d:%02d\r\n",fi->study_date_year
                                             ,fi->study_date_month
                                             ,fi->study_date_day);
  fprintf(fp,"study time := %02d:%02d:%02d\r\n",fi->study_time_hour
                                             ,fi->study_time_minute
                                             ,fi->study_time_second);
  fprintf(fp,"imagedata byte order := ");  
  if (MDC_FILE_ENDIAN == MDC_LITTLE_ENDIAN)
    fprintf(fp,"LITTLEENDIAN\r\n");
  else 
    fprintf(fp,"BIGENDIAN\r\n");

  return(NULL);

}

char *MdcWriteMatrixInfo(FILEINFO *fi, Uint32 img)
{
  IMG_DATA *id = &fi->image[img];
  FILE *fp = fi->ofp;

  fprintf(fp,"!matrix size [1] := %u\r\n",id->width);
  fprintf(fp,"!matrix size [2] := %u\r\n",id->height);

  if (MDC_FORCE_INT != MDC_NO) {
    switch (MDC_FORCE_INT) {
      case BIT8_U :
          fprintf(fp,"!number format := %s\r\n",MdcType2Intf(BIT8_U));
          fprintf(fp,"!number of bytes per pixel := %u\r\n"
                                               ,MdcType2Bytes(BIT8_U));
          break;
      case BIT16_S:
          fprintf(fp,"!number format := %s\r\n",MdcType2Intf(BIT16_S));
          fprintf(fp,"!number of bytes per pixel := %u\r\n"
                                               ,MdcType2Bytes(BIT16_S));
          break;
      default     :
          fprintf(fp,"!number format := %s\r\n",MdcType2Intf(BIT16_S));
          fprintf(fp,"!number of bytes per pixel := %u\r\n",
                                                MdcType2Bytes(BIT16_S));
    }
  }else if (MDC_QUANTIFY || MDC_CALIBRATE) {
          fprintf(fp,"!number format := short float\r\n");
          fprintf(fp,"!number of bytes per pixel := 4\r\n");
  }else{
          fprintf(fp,"!number format := %s\r\n",MdcType2Intf(id->type));
          fprintf(fp,"!number of bytes per pixel := %u\r\n",
                                             MdcType2Bytes(id->type));
  }

  fprintf(fp,"scaling factor (mm/pixel) [1] := %+e\r\n",id->pixel_xsize);
  fprintf(fp,"scaling factor (mm/pixel) [2] := %+e\r\n",id->pixel_ysize);

  return (NULL);

}


char *MdcWriteWindows(FILEINFO *fi)
{ 
  Uint32 window, total_energy_windows = fi->dim[7];
  FILE *fp = fi->ofp; 
  char *msg = NULL;

  if (total_energy_windows == 0) return("INTF Bad total number of windows");

  fprintf(fp,"number of energy windows := %u\r\n",total_energy_windows);

  for (window=1; window <= total_energy_windows; window++) {

     fprintf(fp,";\r\n");
     fprintf(fp,"energy window [%u] :=\r\n",window);
     fprintf(fp,"energy window lower level [%u] :=\r\n",window);
     fprintf(fp,"energy window upper level [%u] :=\r\n",window);
     fprintf(fp,"flood corrected := ");
     if (fi->flood_corrected == MDC_YES)
       fprintf(fp,"Y\r\n");
     else
       fprintf(fp,"N\r\n");
     fprintf(fp,"decay corrected := ");
     if (fi->decay_corrected == MDC_YES)
       fprintf(fp,"Y\r\n");
     else
       fprintf(fp,"N\r\n");

     switch (fi->acquisition_type) {
       case MDC_ACQUISITION_UNKNOWN:  msg=MdcWriteIntfStatic(fi);  break;
       case MDC_ACQUISITION_STATIC :  msg=MdcWriteIntfStatic(fi);  break;
       case MDC_ACQUISITION_DYNAMIC:  msg=MdcWriteIntfDynamic(fi); break;
       case MDC_ACQUISITION_TOMO   :  msg=MdcWriteIntfTomo(fi);    break;
       default                     :  msg=MdcWriteIntfStatic(fi);
     }

     if (msg != NULL) return(msg);

  } 

  return(NULL);
}


char *MdcWriteIntfStatic(FILEINFO *fi) 
{
  Uint32 i, total_energy_windows=fi->dim[7];
  Uint32 images_per_window = fi->number/total_energy_windows;
  IMG_DATA *id = NULL;
  FILE *fp  = fi->ofp;
  char *msg = NULL;

  fprintf(fp,";\r\n!STATIC STUDY (General) :=\r\n");
  fprintf(fp,"number of images/energy window := %u\r\n",images_per_window);

  for (i=0; i<images_per_window; i++) {
     id = &fi->image[i];
     fprintf(fp,"!Static Study (each frame) :=\r\n");
     fprintf(fp,"!image number := %u\r\n",i+1);

     msg = MdcWriteMatrixInfo(fi, i);
     if (msg != NULL) return(msg);

     fprintf(fp,"image duration (sec) :=\r\n");
     fprintf(fp,"image start time :=\r\n");
     fprintf(fp,"label :=\r\n");

     if (id->rescaled) {
       fprintf(fp,"maximum pixel count := %+e\r\n",id->rescaled_max);
       fprintf(fp,"minimun pixel count := %+e\r\n",id->rescaled_min);
     }else{
       fprintf(fp,"maximum pixel count := %+e\r\n",id->max);
       fprintf(fp,"minimum pixel count := %+e\r\n",id->min);
     }

     fprintf(fp,"total counts :=\r\n");
 
  }

  if (ferror(fp)) return("INTF Error writing Static Header");

  return(NULL);

}

char *MdcWriteIntfDynamic(FILEINFO *fi)
{
  Uint32 plane, frame, first_plane, img=0;
  Uint32 total_frames=1, planes_per_frame=fi->dim[3];
  double max_pixel_count; 
  IMG_DATA *id = NULL; 
  FILE *fp  = fi->ofp;
  char *msg = NULL;

  if (fi->dim[0] >= 4) total_frames = fi->dim[4];

  if (fi->diff_size == MDC_YES) 
    return("INTF Dynamic different sizes unsupported");
  if (fi->diff_type == MDC_YES)
    return("INTF Dynamic different types unsupported");

  fprintf(fp,";\r\n!DYNAMIC STUDY (general) :=\r\n");
  fprintf(fp,"!number of frame groups := %u\r\n",total_frames);
  for (frame=0; frame < total_frames; frame++) {
     first_plane = frame * planes_per_frame;
     id = &fi->image[first_plane]; /* first image of frame */
     fprintf(fp,"!Dynamic Study (each frame group) :=\r\n");
     fprintf(fp,"!frame group number := %u\r\n",frame+1);

     msg = MdcWriteMatrixInfo(fi, img);
     if (msg != NULL) return(msg);

     fprintf(fp,"!number of images this frame group := %u\r\n"
               ,planes_per_frame);
     fprintf(fp,"!image duration (sec) := %+e\r\n"
               ,(id->frame_duration/1000.0)/(float)planes_per_frame);
     fprintf(fp,"pause between images (sec) := 0\r\n");
     fprintf(fp,"pause between frame groups (sec) := %+e\r\n"
               ,(id->frame_start/1000.0));
     /* time between 'start of study' and this frame group */

     if (id->rescaled || MDC_CALIBRATE || MDC_QUANTIFY) {
       max_pixel_count = id->rescaled_max;
     }else{
       max_pixel_count = id->max;
     }

     for (plane=1; plane < planes_per_frame; plane++) {
        id = &fi->image[first_plane + plane];
        if (id->rescaled) {
          if (id->rescaled_max > max_pixel_count) 
            max_pixel_count = id->rescaled_max;
        }else{
          if (id->max          > max_pixel_count) 
            max_pixel_count = id->max;
        }
     }
     fprintf(fp,"!maximum pixel count in group := %+e\r\n",max_pixel_count);
  }

  if (ferror(fp)) return("INTF Error writing Dynamic Header");

  return(NULL);

}

char *MdcWriteIntfTomo(FILEINFO *fi)
{
  Uint32 total_energy_windows=fi->dim[7], total_detector_heads=fi->dim[6];
  Uint32 head, img=0, planes = fi->dim[3], images_per_window;
  float slice_thickness, slice_separation;
  ACQ_DATA *acq = NULL;
  IMG_DATA *id = &fi->image[0];
  FILE *fp  = fi->ofp;
  char *msg = NULL;

  images_per_window = fi->number / total_energy_windows;

  if (fi->diff_size == MDC_YES)
    return("INTF Dynamic different sizes unsupported");
  if (fi->diff_type == MDC_YES)
    return("INTF Dynamic different types unsupported");

  /* in pixels instead of mm */
  slice_thickness=id->slice_width/((id->pixel_xsize+id->pixel_ysize)/2.);
  slice_separation=id->slice_spacing/((id->pixel_xsize+id->pixel_ysize)/2.);

  fprintf(fp,";\r\n!SPECT STUDY (general) :=\r\n");
  fprintf(fp,"number of detector heads := %u\r\n",total_detector_heads);

  for (head=0; head < total_detector_heads; head++, ACQI++) {
     if (ACQI < fi->acqnr && fi->acqdata != NULL) {
       acq = &fi->acqdata[ACQI];
     }else{
       acq = NULL;
     }
     fprintf(fp,";\r\n");
     fprintf(fp,"!number of images/energy window := %u\r\n",images_per_window);
     fprintf(fp,"!process status := ");
     if (fi->reconstructed == MDC_NO) {
       fprintf(fp,"Acquired\r\n");
     }else{
       fprintf(fp,"Reconstructed\r\n");
     }

     msg = MdcWriteMatrixInfo(fi, img);
     if (msg != NULL) return(msg);

     fprintf(fp,"!number of projections := %u\r\n",planes);
     fprintf(fp,"!extent of rotation := ");
     if (acq != NULL) fprintf(fp,"%g",acq->angle_step*(float)planes);
     fprintf(fp,"\r\n");
     fprintf(fp,"!time per projection (sec) := %+e\r\n"
               ,(id->frame_duration/1000.0)/planes);
     fprintf(fp,"study duration (sec) := %+e\r\n",id->frame_duration/1000.0);
     if (MDC_FORCE_INT != MDC_NO) {
       switch (MDC_FORCE_INT) {
         case BIT8_U: 
          fprintf(fp,"!maximum pixel count := %+e\r\n",(float)MDC_MAX_BIT8_U);
          break;
         case BIT16_S:
          fprintf(fp,"!maximum pixel count := %+e\r\n",(float)MDC_MAX_BIT16_S);
          break;
         default:
          fprintf(fp,"!maximum pixel count := %+e\r\n",(float)MDC_MAX_BIT16_S);
       } 
     }else if (MDC_QUANTIFY || MDC_CALIBRATE) {
          fprintf(fp,"!maximum pixel count := %+e\r\n",fi->qglmax);
     }else{
          fprintf(fp,"!maximum pixel count := %+e\r\n",fi->glmax);
     }
     fprintf(fp,"patient orientation := %s\r\n"
               ,MdcGetStrPatientRot(id->pat_slice_orient));
     fprintf(fp,"patient rotation := %s\r\n"
               ,MdcGetStrPatientPos(id->pat_slice_orient));
     if (fi->reconstructed == MDC_NO) {
          fprintf(fp,";\r\n!SPECT STUDY (acquired data) :=\r\n");
          fprintf(fp,"!direction of rotation := ");
          if (acq != NULL) {
            switch (acq->rotation_direction) {
              case MDC_ROTATION_CW: fprintf(fp,"CW");  break;
              case MDC_ROTATION_CC: fprintf(fp,"CCW"); break;
            }
          }
          fprintf(fp,"\r\n");
          fprintf(fp,"start angle := ");
          if (acq != NULL) {
            fprintf(fp,"%g",acq->angle_start);
          }
          fprintf(fp,"\r\n");
          fprintf(fp,"first projection angle in data set :=\r\n");
          fprintf(fp,"acquisition mode := ");
          if (acq != NULL) {
            switch (acq->detector_motion) {
              case MDC_MOTION_STEP: fprintf(fp,"stepped");    break;
              case MDC_MOTION_CONT: fprintf(fp,"continuous"); break;
              default             : fprintf(fp,"unknown");
            }
          }
          fprintf(fp,"\r\n");
          fprintf(fp,"Centre_of_rotation := Corrected\r\n"); 
          fprintf(fp,"orbit := circular\r\n");
          fprintf(fp,"preprocessed :=\r\n");
     }else{
          fprintf(fp,";\r\n!SPECT STUDY (reconstructed data) :=\r\n");
          fprintf(fp,"method of reconstruction := %s\r\n",fi->recon_method);
          fprintf(fp,"!number of slices := %u\r\n",planes);
          fprintf(fp,"number of reference frame := 0\r\n");
          fprintf(fp,"slice orientation := %s\r\n",
                        MdcGetStrSliceOrient(fi->image[0].pat_slice_orient));
          fprintf(fp,"slice thickness (pixels) := %+e\r\n",slice_thickness);
          fprintf(fp,"centre-centre slice separation (pixels) := %+e\r\n",
                                                         slice_separation);
          fprintf(fp,"filter name := %s\r\n",fi->filter_type);
          fprintf(fp,"filter parameters := Cutoff\r\n");
          /*fprintf(fp,"z-axis filter :=\r\n");*/
          /*fprintf(fp,"attenuation correction coefficient/cm :=\r\n");*/
          fprintf(fp,"method of attenuation correction := measured\r\n");
          fprintf(fp,"scatter corrected := N\r\n"); 
          /*fprintf(fp,"method of scatter correction :=\r\n");*/
          fprintf(fp,"oblique reconstruction := N\r\n");
          /*fprintf(fp,"oblique orientation :=\r\n");*/
     }

  } 

  if (ferror(fp)) return("INTF Error writing Tomographic Header");

  return(NULL);

}

char *MdcWriteIntfHeader(FILEINFO *fi)
{
  FILE *fp = fi->ofp;
  char *msg=NULL;

  fprintf(fp,"!INTERFILE :=\r\n");
  fprintf(fp,"!imaging modality := nucmed\r\n");
  fprintf(fp,"!originating system := greetings\r\n");
  fprintf(fp,"!version of keys := %g\r\n",MDC_INTF_SUPP_VERS);
  fprintf(fp,"date of keys := %s\r\n",MDC_INTF_SUPP_DATE);
  fprintf(fp,"conversion program := %s\r\n",MDC_PRGR);
  fprintf(fp,"program author := Erik Nolf\r\n");
  fprintf(fp,"program version := %s\r\n",MDC_VERSION);
  fprintf(fp,"program date := %s\r\n",MdcGetProgramDate());
  fprintf(fp,";\r\n!GENERAL DATA :=\r\n");
  fprintf(fp,"!data offset in bytes := 0\r\n");
  if (XMDC_MEDCON) MdcSplitPath(fi->opath,fi->odir,fi->ofname);
  MdcNewExt(fi->ofname,'\0',"i33");
  fprintf(fp,"!name of data file := %s\r\n",fi->ofname);
  MdcNewExt(fi->ofname,'\0',FrmtExt[MDC_FRMT_INTF]);
  if (XMDC_MEDCON) MdcMergePath(fi->opath,fi->odir,fi->ofname);
  fprintf(fp,"patient name := %s\r\n",fi->patient_name);
  fprintf(fp,"!patient ID := %s\r\n",fi->patient_id);
  fprintf(fp,"patient sex := %s\r\n",fi->patient_sex);
  fprintf(fp,"!study ID := %s\r\n",fi->study_name);
  fprintf(fp,"exam type := %s\r\n",fi->study_descr);
  fprintf(fp,"data compression := none\r\n");
  fprintf(fp,"data encode := none\r\n");

  msg = MdcWriteGenImgData(fi);
  if (msg != NULL) return(msg);

  msg = MdcWriteWindows(fi);
  if (msg != NULL) return(msg);

  fprintf(fp,"!END OF INTERFILE :=\r\n%c",MDC_CNTRL_Z);

  if (ferror(fp)) return("INTF Bad write header file");

  return(NULL);

}


char *MdcWriteIntfImages(FILEINFO *fi)
{
  IMG_DATA *id;
  FILE *fp = fi->ofp;
  Uint32 i, size;
  Uint8 *buf;
  
  for (i=0; i<fi->number; i++) {
#if XSUPPORTED
     if (XMDC_MEDCON) {
       pvalue += 1./(float)fi->number;
       XMdcUpdateProgressBar(NULL);
     }
#endif

     id = &fi->image[i];
     size = id->width * id->height;
     if (MDC_FORCE_INT != MDC_NO) {
       switch (MDC_FORCE_INT) {
         case BIT8_U: 
             buf = MdcGetImgBIT8_U(fi, i);
             if (buf == NULL) return("INTF Bad malloc Uint8 buffer"); 
             /* no endian swap necessary */
             if (fwrite(buf,MdcType2Bytes(BIT8_U),size,fp) != size) {
               MdcFree(buf);
               return("INTF Bad write Uint8 image");
             }
             break;
         case BIT16_S: 
             buf = MdcGetImgBIT16_S(fi, i);
             if (buf == NULL) return("INTF Bad malloc Int16 buffer");
             if (MDC_FILE_ENDIAN != MDC_HOST_ENDIAN)
               MdcMakeImgSwapped(buf,fi,i,id->width,id->height,BIT16_S);
             if (fwrite(buf,MdcType2Bytes(BIT16_S),size,fp) != size) {
               MdcFree(buf);
               return("INTF Bad write Int16 image");
             }
             break;
         default:
             buf = MdcGetImgBIT16_S(fi, i);
             if (buf == NULL) return("INTF Bad malloc Int16 buffer");
             if (MDC_FILE_ENDIAN != MDC_HOST_ENDIAN)
               MdcMakeImgSwapped(buf,fi,i,id->width,id->height,BIT16_S);
             if (fwrite(buf,MdcType2Bytes(BIT16_S),size,fp) != size) {
               MdcFree(buf);
               return("INTF Bad write Int16 image");
             }
             
       }
       MdcFree(buf); 
     }else if (!(MDC_QUANTIFY || MDC_CALIBRATE)) {
       switch ( id->type ) {
         case  BIT1: return("INTF 1-Bit format unsupported"); break;
         case ASCII: return("INTF Ascii format unsupported"); break;
         default:
             if (MDC_FILE_ENDIAN != MDC_HOST_ENDIAN && 
                  id->type != BIT8_U && id->type != BIT8_S) {
               buf = MdcGetImgSwapped(fi,i);
               if (buf == NULL) return("INTF Couldn't malloc swapped image");
               if (fwrite(buf,MdcType2Bytes(id->type),size,fp) != size) {
                 MdcFree(buf);
                 return("INTF Bad write swapped image"); 
               }
               MdcFree(buf);
             }else{
               if (fwrite(id->buf,MdcType2Bytes(id->type),size,fp) != size)
                 return("INTF Bad write image");
             }
       }
     }else{
       buf = MdcGetImgFLT32( fi, i);
       if (buf == NULL) return("INTF Bad malloc buf");
       if (MDC_FILE_ENDIAN != MDC_HOST_ENDIAN) 
         MdcMakeImgSwapped(buf,fi,i,id->width,id->height,FLT32);
       if (fwrite(buf,MdcType2Bytes(FLT32),size,fp) != size) {
         MdcFree(buf);
         return("INTF Bad write quantified image");
       }
       MdcFree(buf);
     } 
  }

  return NULL;

}


char *MdcWriteINTF(FILEINFO *fi)
{
  char *err, *tmpfname = mdcbufr;

  MDC_FILE_ENDIAN = MDC_WRITE_ENDIAN;

  /* get filename and print message */
#if XSUPPORTED
  if (XMDC_MEDCON) {
    strcpy(tmpfname,fi->ofname);
    XMdcBeginProgressBar("Writing InterFile:");
  }else{
#endif
    strcpy(tmpfname,fi->ifname);
    MdcDefaultName(MDC_FRMT_INTF,fi->ofname,tmpfname);
#if XSUPPORTED
  }
#endif 
  if (MDC_VERBOSE) MdcPrntMesg("INTF Writing <%s> & <.i33> ...",fi->ofname);

  /* first we write the image file */
  if (XMDC_MEDCON) {
    fi->ofname[0]='\0'; MdcNewExt(fi->ofname,tmpfname,"i33");
  }else{ 
    MdcNewName(fi->ofname,tmpfname,"i33");  
  }
  if (MdcFileExists(fi->ofname))
    return("INTF Image file exists!!");
  if ( (fi->ofp=fopen(fi->ofname,"wb")) == NULL)
    return("INTF Couldn't open image file");

  err = MdcWriteIntfImages(fi);
  if (err != NULL) return(err);

  MdcFileClose(fi->ofp);

  /* now we write the header file */
  /* got rescaled stuff info      */

  if (XMDC_MEDCON) {
    strcpy(fi->ofname,tmpfname);
  }else{
    MdcDefaultName(MDC_FRMT_INTF,fi->ofname,tmpfname);
  }
  if (MdcFileExists(fi->ofname))
    return("INTF Header file exists!!");
  if ( (fi->ofp=fopen(fi->ofname,"wb")) == NULL)
    return("INTF Couldn't open header file");

  err = MdcWriteIntfHeader(fi);
  if (err != NULL) return(err);

  MdcFileClose(fi->ofp);

  return NULL;

}
