/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * filename: m-dicm.c                                                      *
 *                                                                         *
 * UTIL C-source: Medical Image Conversion Utility                         *
 *                                                                         *
 * purpose      : Read DICOM files                                         *
 *                                                                         * 
 * project      : (X)MedCon by Erik Nolf                                   *
 *                                                                         *
 * Functions    : MdcCheckDICM()       - Check DICOM format                *
 *                MdcReadDICM()        - Read  DICOM format                * 
 *                MdcWriteDICM()       - Write DICOM format                *
 *                                                                         *
 * Notes        : Source needs VT-DICOM-package written by Tony Voet       *
 *                                                                         *
 *                                                                         *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* $Id: m-dicm.c,v 1.1.1.1 2000/10/28 16:51:37 enlf Exp $
 */

/*
   Copyright (C) 1997-2000 by Erik Nolf

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 2, or (at your option) any later
   version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
   Public License for more details.

   You should have received a copy of the GNU General Public License along
   with this program; if not, write to the Free Software Foundation, Inc.,
   59 Place - Suite 330, Boston, MA 02111-1307, USA.  */

/****************************************************************************
                              H E A D E R S
****************************************************************************/

#include <time.h>

#include "dicom.h"
#include "medcon.h"

/****************************************************************************
                       G L O B A L  D E F I N E S
****************************************************************************/

#define MDC_DICOM_FAKED_UID   "777.777.0.0.0.0.0.%u.%u.%u"

#define UNDEFINED_LENGTH      0xffffffff

#define IROW 2
#define ICOL 1

/* extra stuff for reading DICOM */
static char *MdcDicomRescale(IMG_DATA *id, double slope, double intercept);
static char *MdcDicomContrast(IMG_DATA *id, double center, double width);
void MdcDicomInvert(IMG_DATA *id);

/* extra stuff for writing DICOM */
static void MdcDicomMakeUID(char *str, time_t *psec_utc);

static void MdcDicomWriteInfoSeq(FILE *fp, Uint16 group, Uint16 element);
static void MdcDicomWriteItem(FILE *fp);
static void MdcDicomWriteItemDelItem(FILE *fp);
static void MdcDicomWriteInfoSeqDelItem(FILE *fp);

/* including the addapted dicom lib functions */
static int  mdc_dicom_read(FILEINFO *fi, IMAGE **image, int *images);
static void mdc_dicom_dumpinfo(FILEINFO *fi);
static void mdc_dicom_printinfo(const ELEMENT *e,const char *description);
static void mdc_dicom_getinfo(FILEINFO *fi);

static void mdc_dicom_get_vr(ELEMENT *e);
static Uint8 *mdc_dicom_handle_vr(ELEMENT *e, Uint8 *tdata);
static int  mdc_dicom_write_element(FILE *fp, Uint16 group, Uint16 element, Uint32 length, Uint8 *data);

static int verbose = MDC_YES;

static Uint32 majoruid = 0; /* extra increment for UID's in batch conversions */
static Uint32 minoruid = 0; /* increment for each UID in same file            */

extern MDC_DICOM_STUFF_T mdc_dicom_stuff;

/****************************************************************************
                            F U N C T I O N S
****************************************************************************/

int MdcCheckDICM(FILEINFO *fi)
{
   char sig[5];

   fseek(fi->ifp,128,SEEK_SET);
   fread(sig,1,4,fi->ifp);
   fseek(fi->ifp,0,SEEK_SET);

   if (ferror(fi->ifp)) return(MDC_BAD_READ);

   sig[4]='\0';
   MdcLowStr(sig);

   if (strstr(sig,MDC_SIG_DICM) == NULL) return(MDC_FRMT_BAD);

   return(MDC_FRMT_DICM);
}

char *MdcDicomRescale(IMG_DATA *id, double slope, double intercept)
{
   double max=0., min=0., MAXINT;
   double pixvalue, scale;
   Uint32 n, i;
   Uint8 *pixel;

   MAXINT = MdcTypeIntMax(id->type);
   if (MAXINT == 0.0)
     return("DICM Rescaling unsupported for float pixel types");
  
   n = id->width * id->height;
   /* retrieve the dicom rescaled max/min values */
   for (pixel=id->buf, i=0; i<n; i++, pixel+=MdcType2Bytes(id->type)) {
      pixvalue = MdcGetDoublePixel(pixel,id->type);
      if (i==0) {
        max = pixvalue; min = pixvalue;
      }else{
        if ( pixvalue > max ) max = pixvalue;
        else if ( pixvalue < min ) min = pixvalue;
      } 
   }
   
   /* prepare scale factor to original pixeltype */
   max = max * slope + intercept;
   min = min * slope + intercept;
   scale = ( max == min) ? 1. : MAXINT/(max - min);

   /* rescale now the pixel values and scale back to original pixel type */
   for (pixel=id->buf, i=0; i<n; i++, pixel+=MdcType2Bytes(id->type)) {
      /* get pixel value */
      pixvalue = MdcGetDoublePixel(pixel,id->type);

      /* apply rescale slope/intercept */
      pixvalue = (pixvalue * slope) + intercept;

      /* rescale to original pixel type while preserving scale factor */
      pixvalue = (pixvalue - min) * scale;

      MdcPutDoublePixel(pixel,pixvalue,id->type);
      
   } 

   /* finally preserve quantification scale factor */
   id->quant_scale = 1./scale;

   return(NULL);
}

char *MdcDicomContrast(IMG_DATA *id, double center, double width) 
{
   double pixvalue, MAXINT;
   double max, min;
   Uint8 *pixel;
   Uint32 i, n;

   /* prepare range values */
   min = center - (width/2.); 
   max = center + (width/2.);
   
   MAXINT = MdcTypeIntMax(id->type);
   if (MAXINT == 0.0) 
     return("DICM Contrast scaling unsupported for float pixel types");

   n = id->width * id->height;
   for (pixel=id->buf, i=0; i<n; i++, pixel+=MdcType2Bytes(id->type)) {

      /* get pixel value */
      pixvalue = MdcGetDoublePixel(pixel,id->type);

      /* apply window center/width */
      if ( pixvalue > max ) pixvalue = MAXINT;
      else if ( pixvalue < min ) pixvalue = 0.; 
      else { 
        pixvalue = ((pixvalue - min) * MAXINT) / (max - min);
        if ( pixvalue < 0. ) pixvalue = 0.;
        else if (pixvalue > MAXINT) pixvalue = MAXINT;
      }

      /* put value back in buffer, already fit for original pixel type */
      MdcPutDoublePixel(pixel,pixvalue,id->type);

   }

   return(NULL);

}

void MdcDicomInvert(IMG_DATA *id)
{
   double pixvalue;
   double max=0., min=0.;
   Uint8 *pixel;
   Uint32 i, n;

   n = id->width * id->height;
   /* retrieve present max/min values */
   for (pixel=id->buf, i=0; i<n; i++, pixel+=MdcType2Bytes(id->type)) {
      pixvalue = MdcGetDoublePixel(pixel,id->type);
      if (i==0) {
        max = pixvalue; min = pixvalue;
      }else{
        if ( pixvalue > max ) max = pixvalue;
        else if ( pixvalue < min ) min = pixvalue;
      }
   }

   /* invert pixel values */
   for (pixel=id->buf, i=0; i<n; i++, pixel+=MdcType2Bytes(id->type)) {
      pixvalue = MdcGetDoublePixel(pixel,id->type);
      pixvalue = max - pixvalue + min;
      MdcPutDoublePixel(pixel,pixvalue,id->type);
   } 

}

char *MdcReadDICM(FILEINFO *fi)
{
  IMAGE *image, *pimg;
  IMG_DATA *id;
  MDC_DICOM_STUFF_T *dicom=&mdc_dicom_stuff;
  Uint32 img=0, i, f, images, bytes, t;
  Uint8 *pdata=NULL;
  char *msg=NULL;

  MDC_FILE_ENDIAN = MDC_HOST_ENDIAN;

  /* init dicom struct */
  MdcDicomInitStuff(dicom);

#if XSUPPORTED
  if (XMDC_MEDCON) XMdcBeginProgressBar("Reading DICOM:");
#endif

  if (MDC_VERBOSE) MdcPrntMesg("DICM Reading <%s> ...",fi->ifname);

  MdcMergePath(fi->ipath,fi->idir,fi->ifname);

  /* reading file 1st time for info printout */
  if (MDC_INFO) {
    MdcPrintLine('*',MDC_HALF_LENGTH);
    printf("Pass #1: through DICOM reader\n");
    MdcPrintLine('*',MDC_HALF_LENGTH); 
    mdc_dicom_dumpinfo(fi);
  }

#if XSUPPORTED
     if (XMDC_MEDCON) {
       pvalue += 0.1;
       XMdcUpdateProgressBar(NULL);
     }
#endif

  /* reading file 2nd time for images                                    */
  if (mdc_dicom_read(fi,&image,(Int32 *)&images)) {
    MdcSplitPath(fi->ipath,fi->idir,fi->ifname);
    dicom_free(image,images);
    return("DICM Error reading file");
  }

#if XSUPPORTED
     if (XMDC_MEDCON) {
       pvalue += 0.1;
       XMdcUpdateProgressBar(NULL);
     }
#endif

  MdcSplitPath(fi->ipath,fi->idir,fi->ifname);

  /* temporary fill in the FILEINFO struct */
  fi->type   = MDC_DICM_PIXEL_TYPE;
  fi->bits   = MdcType2Bits(MDC_DICM_PIXEL_TYPE);
  for (fi->number=1, i=0; i<images; i++) {
     pimg = &image[i];
     fi->number *= pimg->frames;
     if (fi->number  == 0 ) {
       dicom_free(image,images);
       return("DICM Bad number of images");
     }
     if (pimg->rgb) dicom_gray(pimg);
  }
  fi->endian = MDC_HOST_ENDIAN;
  fi->dim[0] = 3;
  fi->dim[1] = fi->mwidth;
  fi->dim[2] = fi->mheight;
/*fi->dim[3] = fi->number;*/
  fi->pixdim[0] = 0.;
  fi->reconstructed = MDC_YES;
  fi->acquisition_type = MDC_ACQUISITION_TOMO;

  if (!MdcGetStructID(fi)) {
    dicom_free(image,images);
    return("DICM Bad malloc IMG_DATA structs");
  }

  /* reading file 3rd time for info retrieving (through Acr/Nema reader) */
  if (MDC_INFO) {
    printf("\n\n");
    MdcPrintLine('*',MDC_HALF_LENGTH);
    printf("Pass #2: through Acr/Nema reader\n");
    MdcPrintLine('*',MDC_HALF_LENGTH);
  } 


  MdcMergePath(fi->ipath,fi->idir,fi->ifname);
  mdc_dicom_getinfo(fi); 
  MdcSplitPath(fi->ipath,fi->idir,fi->ifname);

  /* fill in the FILEINFO structs */
  if (dicom->sign == 1) fi->type = BIT16_S;
  else fi->type = BIT16_U;
  fi->bits = MdcType2Bits(fi->type);
  for (t=7; t > 3; t--) if (fi->dim[t] > 1) break;
  fi->dim[0] = t; fi->pixdim[0] = t;
  fi->pixdim[1] = fi->image[0].pixel_xsize;
  fi->pixdim[2] = fi->image[0].pixel_ysize;
  fi->pixdim[3] = fi->image[0].slice_width;
  /* MARK : add better and more info later */

  /* put images and info in IMG_DATA structs */
  for (img=0, i=0; i<images; i++) {
     pimg = &image[i];
     for (f=0; f<image[i].frames; f++) {
#if XSUPPORTED
        if (XMDC_MEDCON) {
          pvalue += 0.80/(float)fi->number;
          XMdcUpdateProgressBar(NULL);
        }
#endif
        pdata = (Uint8 *)(pimg->data.gray + (f*pimg->w*pimg->h));
        id = &fi->image[img];
        id->width  = (Uint32)pimg->w;
        id->height = (Uint32)pimg->h;
        id->type   = fi->type;
        bytes     = id->width*id->height*MdcType2Bytes(id->type);
        id->bits  = MdcType2Bits(id->type);
        id->buf   = MdcGetImgBuffer(bytes);
        if (id->buf == NULL) {
          dicom_free(image,images);
          return("DICM Couldn't allocate image buffer");
        }
        memcpy(id->buf,pdata,bytes);

        if (!((img == 0) && (f == 0))) {
          /* copy voxel size and orient values from the first image */
          id->pixel_xsize = fi->image[0].pixel_xsize;
          id->pixel_ysize = fi->image[0].pixel_ysize;
          id->slice_width = fi->image[0].slice_width;
          id->slice_spacing = fi->image[0].slice_spacing;
          id->pat_slice_orient = fi->image[0].pat_slice_orient;

          strncpy(id->pat_pos,fi->image[0].pat_pos,MDC_MAXSTR);
          strncpy(id->pat_orient,fi->image[0].pat_orient,MDC_MAXSTR);

        }

        /* image orient/position according to patient coordinate system */
        if (id->image_orient_pat[0]==0.0 && id->image_orient_pat[1]==0.0 &&
            id->image_orient_pat[4]==0.0 && id->image_orient_pat[5]==0.0 ) {
            /* no patient coordinate system defines, try pat_orient */
          if ((img == 0) && (f == 0)) 
            id->pat_slice_orient = MdcTryPatSliceOrient(id->pat_orient);
          if (id->pat_slice_orient != MDC_UNKNOWN) {
            MdcFillImgPos(fi,img,img%fi->dim[3],0.0);
            MdcFillImgOrient(fi,img); 
          }
        }

        /* image orient/position according to device (RETIRED)*/

        if (id->image_orient_dev[0]==0.0 && id->image_orient_dev[1]==0.0 &&
            id->image_orient_dev[4]==0.0 && id->image_orient_dev[5]==0.0 ) {
            /* no patient coordinate system defines */
          switch (id->pat_slice_orient) {
            case MDC_SUPINE_HEADFIRST_TRANSVERSAL:
            case MDC_SUPINE_HEADFIRST_SAGITTAL   :
            case MDC_SUPINE_HEADFIRST_CORONAL    : /* same coordinate system */
                for (t=0; t<6; t++) 
                   id->image_orient_dev[t] = id->image_orient_pat[t];
                break;
          }
        }

        if (id->image_pos_dev[0] == 0.0 && id->image_pos_dev[1] == 0.0 &&
            id->image_pos_dev[2] == 0.0 ) {

          switch (id->pat_slice_orient) {
            case MDC_SUPINE_HEADFIRST_TRANSVERSAL:
            case MDC_SUPINE_HEADFIRST_SAGITTAL   :
            case MDC_SUPINE_HEADFIRST_CORONAL    : /* same coordinate system */
                for (t=0; t<3; t++)
                   id->image_pos_dev[t] = id->image_pos_pat[t];
                break;
          }
        } 

        if (MDC_QUANTIFY == MDC_YES || MDC_CALIBRATE == MDC_YES) {
          if (dicom->si_slope != 1.0 || dicom->si_intercept != 0.0) {
            /* apply rescale slope / intercept */
            msg = MdcDicomRescale(id,dicom->si_slope,dicom->si_intercept);
            if (msg != NULL) {
#if XSUPPORTED
              if (XMDC_MEDCON) {
                XMdcDisplayWarn(msg);
              }else
#endif
              {
                MdcPrntWarn(msg);
              }
            }
          }else if (dicom->cw_width != 0.0 ) {
            /* apply contrast window / center */
            msg = MdcDicomContrast(id,dicom->cw_center,dicom->cw_width);
            if (msg != NULL) {
#if XSUPPORTED
              if (XMDC_MEDCON) {
                XMdcDisplayWarn(msg);
              }else
#endif
              {
                MdcPrntWarn(msg);
              }
            } 
          } 
        }

        if (dicom->INVERT == MDC_YES) MdcDicomInvert(id);

        img+=1;
     }
  }

  dicom_free(image,images);

  MdcFileClose(fi->ifp);

  return(NULL);

}

void MdcDicomMakeUID(char *str, time_t *psec)
{
   minoruid = minoruid <= 0 ? 1 : minoruid+1;

   if (psec != NULL) {
     /* succeeded time() */
     sprintf(str,MDC_DICOM_FAKED_UID,majoruid,minoruid,(Uint32)*psec);
   }else{
     /* failed    time() */
     sprintf(str,MDC_DICOM_FAKED_UID,majoruid,minoruid,777U);
   }
}

void MdcDicomWriteInfoSeq(FILE *fp, Uint16 group, Uint16 element)
{
  mdc_dicom_write_element(fp,group,element,UNDEFINED_LENGTH,NULL);
} 
void MdcDicomWriteItem(FILE *fp)
{
  mdc_dicom_write_element(fp,0xfffe,0xe000,UNDEFINED_LENGTH,NULL);
} 
void MdcDicomWriteItemDelItem(FILE *fp)
{
  mdc_dicom_write_element(fp,0xfffe,0xe00d,0,NULL);
} 
void MdcDicomWriteInfoSeqDelItem(FILE *fp)
{
  mdc_dicom_write_element(fp,0xfffe,0xe0dd,0,NULL);
}

char *MdcWriteDICM(FILEINFO *fi)
{

  MDC_DICOM_STUFF_T *dicom=&mdc_dicom_stuff;
  Int16 type=BIT16_S; /* for now only 16-bit */
  Uint16 ui16, *pui16, bits_allocated, bits_stored;
  Int32 i32;
  Uint32 i, bytes, pixels, vect;
  Uint32 REWRF, BEGIN, END;
  Uint8 *newbuff, *buff;
  float rescale_intercept=0., rescale_slope=1.;
  char *pdata, dummy1[]="1";
  time_t sec, *psec=NULL;

  MDC_FILE_ENDIAN = MDC_WRITE_ENDIAN;

  /* get universal time (seconds) */
  if ( time(&sec) == ((time_t)-1) ) {
    strcpy(mdcbufr,"DICM Generating unique UID failed");
#if XSUPPORTED
    if (XMDC_MEDCON) {
      XMdcDisplayWarn(mdcbufr);
    }else
#endif
      MdcPrntMesg(mdcbufr);
  }else{
    psec = &sec;
  }
 
  /* init dicom struct */
  MdcDicomInitStuff(dicom);

  /* increment  majoruid => for each conversion  */
  majoruid+=1;
  /* initialize minoruid => for each local UID   */
  minoruid=0;

  /* set the modality to write */
  if (strncmp(fi->image[0].image_mod,"PT",2) == 0)
    dicom->PET_MOD = MDC_YES;
  else 
    dicom->PET_MOD = MDC_NO;

  if (XMDC_MEDCON == MDC_NO) 
    MdcDefaultName(MDC_FRMT_DICM,fi->ofname,fi->ifname);

#if XSUPPORTED
  if (XMDC_MEDCON) XMdcBeginProgressBar("Writing DICOM:");
#endif

  if (MDC_VERBOSE) MdcPrntMesg("DICM Writing <%s> ...",fi->ofname);

  if (MdcFileExists(fi->ofname))
    return("DICM File exists!!");
  if ( (fi->ofp=fopen(fi->ofname,"wb")) == NULL)
    return("DICM Couldn't open file");

  /* check for requested pixel type */
  if (MDC_FORCE_INT != MDC_NO) type = MDC_FORCE_INT;

  /* only Int16 or Uint8 type supported */
  if ( (type != BIT16_S) && (type != BIT8_U) ) {
    type = BIT16_S;
    strcpy(mdcbufr,"DICM Only Int16 or Uint8 pixels supported");
#if XSUPPORTED
    if (XMDC_MEDCON) {
      XMdcDisplayWarn(mdcbufr);
    }else
#endif
      MdcPrntWarn(mdcbufr);
  }

  /* write the empty preamable */
  memset(mdcbufr,0,128);
  if (fwrite(mdcbufr,1,128,fi->ofp) != 128)
    return("DICM Error writing preamable");

  /* write the signature */
  strcpy(mdcbufr,"DICM");
  if (fwrite(mdcbufr,1,4,fi->ofp) != 4)
    return("DICM Error writing signature");
 
  /* group 0x0002 */
  REWRF = ftell(fi->ofp); 
  i32=0;
  mdc_dicom_write_element(fi->ofp,0x0002,0x0000,4,(Uint8 *)&i32);

  BEGIN = ftell(fi->ofp);

  mdcbufr[0]=0x00; mdcbufr[1]=0x01;
  mdc_dicom_write_element(fi->ofp,0x0002,0x0001,2,mdcbufr);

  strcpy(mdcbufr,"1.2.840.10008.5.1.4.1.1.20");
  mdc_dicom_write_element(fi->ofp,0x0002,0x0002,strlen(mdcbufr),mdcbufr);

  /* strcpy(mdcbufr,"1.2.840.10008.Media.Storage.SOP.Instance"); */
  MdcDicomMakeUID(mdcbufr,psec);
  mdc_dicom_write_element(fi->ofp,0x0002,0x0003,strlen(mdcbufr),mdcbufr);

  if (MDC_FILE_ENDIAN == MDC_LITTLE_ENDIAN) {
    strcpy(mdcbufr,"1.2.840.10008.1.2.1");
  }else{
    strcpy(mdcbufr,"1.2.840.10008.1.2.2");
  } 
  mdc_dicom_write_element(fi->ofp,0x0002,0x0010,strlen(mdcbufr),mdcbufr);

  strcpy(mdcbufr,"0.0.0.0");
  mdc_dicom_write_element(fi->ofp,0x0002,0x0012,strlen(mdcbufr),mdcbufr);

  strcpy(mdcbufr,"NOTSPECIFIED");
  mdc_dicom_write_element(fi->ofp,0x0002,0x0013,strlen(mdcbufr),mdcbufr);

  strcpy(mdcbufr,"NOTSPECIFIED");
  mdc_dicom_write_element(fi->ofp,0x0002,0x0016,strlen(mdcbufr),mdcbufr);

  END = ftell(fi->ofp);


  /* rewrite group length */
  fseek(fi->ofp,REWRF,SEEK_SET);
  i32 = END - BEGIN;
  mdc_dicom_write_element(fi->ofp,0x0002,0x0000,4,(Uint8 *)&i32);
  fseek(fi->ofp,0,SEEK_END); 

  /* DEBUG  */
  /* printf("MARK: group length = END - BEGIN = %u - %u = %u\n",END,BEGIN,END-BEGIN); */

  /* group 0x0008 */
  strcpy(mdcbufr,"DERIVED\\PRIMARY");
  if (dicom->PET_MOD != MDC_YES) {
    switch (fi->acquisition_type) {
       case MDC_ACQUISITION_TOMO: if (fi->reconstructed == MDC_YES) 
                                    strcat(mdcbufr,"\\RECON TOMO");
                                  else 
                                    strcat(mdcbufr,"\\TOMO");
           break; 
       case MDC_ACQUISITION_DYNAMIC: strcat(mdcbufr,"\\DYNAMIC");
           break;
       case MDC_ACQUISITION_UNKNOWN: /* fake as static */
       case MDC_ACQUISITION_STATIC: strcat(mdcbufr,"\\STATIC");
           break;
       default: strcat(mdcbufr,"\\UNSPECIFIED");
    }
    strcat(mdcbufr,"\\EMISSION");  /* MARK: no flag for transmission yet */
  }
  mdc_dicom_write_element(fi->ofp,0x0008,0x0008,strlen(mdcbufr),mdcbufr);


  strftime(mdcbufr,35,"%Y%m%d",localtime(psec));
  mdc_dicom_write_element(fi->ofp,0x0008,0x0012,strlen(mdcbufr),mdcbufr);

  strftime(mdcbufr,35,"%H%M%S",localtime(psec));
  mdc_dicom_write_element(fi->ofp,0x0008,0x0013,strlen(mdcbufr),mdcbufr);

  /* strcpy(mdcbufr,"1.2.840.10008.Instance.Creator"); */
  MdcDicomMakeUID(mdcbufr,psec);
  mdc_dicom_write_element(fi->ofp,0x0008,0x0014,strlen(mdcbufr),mdcbufr);

  strcpy(mdcbufr,"1.2.840.10008.5.1.4.1.1.20");
  mdc_dicom_write_element(fi->ofp,0x0008,0x0016,strlen(mdcbufr),mdcbufr);

  /* strcpy(mdcbufr,"1.2.840.10008.Media.Storage.SOP.Instance"); */
  MdcDicomMakeUID(mdcbufr,psec);
  mdc_dicom_write_element(fi->ofp,0x0008,0x0018,strlen(mdcbufr),mdcbufr);

  /* date settings, make sure it is conform */
  if (fi->study_date_year == 0) {
    pdata = NULL; bytes=0;
  }else{
    sprintf(mdcbufr,"%04d%02d%02d",fi->study_date_year
                                  ,fi->study_date_month
                                  ,fi->study_date_day);
    pdata = mdcbufr; bytes = strlen(mdcbufr);
  }
  mdc_dicom_write_element(fi->ofp,0x0008,0x0020,bytes,pdata);
  mdc_dicom_write_element(fi->ofp,0x0008,0x0021,bytes,pdata);
  mdc_dicom_write_element(fi->ofp,0x0008,0x0022,bytes,pdata);
  mdc_dicom_write_element(fi->ofp,0x0008,0x0023,bytes,pdata);

  /* time settings, can be full of zero's ... */ 
  sprintf(mdcbufr,"%02d%02d%02d",fi->study_time_hour
                                ,fi->study_time_minute
                                ,fi->study_time_second);
  mdc_dicom_write_element(fi->ofp,0x0008,0x0030,strlen(mdcbufr),mdcbufr);
  mdc_dicom_write_element(fi->ofp,0x0008,0x0031,strlen(mdcbufr),mdcbufr);
  mdc_dicom_write_element(fi->ofp,0x0008,0x0032,strlen(mdcbufr),mdcbufr);
  mdc_dicom_write_element(fi->ofp,0x0008,0x0033,strlen(mdcbufr),mdcbufr);

  mdc_dicom_write_element(fi->ofp,0x0008,0x0050,strlen(dummy1),dummy1);


  strcpy(mdcbufr,fi->image[0].image_mod);
  mdc_dicom_write_element(fi->ofp,0x0008,0x0060,strlen(mdcbufr),mdcbufr);

  strcpy(mdcbufr,MDC_LIBVERS);
  mdc_dicom_write_element(fi->ofp,0x0008,0x0070,strlen(mdcbufr),mdcbufr);

  strcpy(mdcbufr,"Unknown^^^^");
  mdc_dicom_write_element(fi->ofp,0x0008,0x0090,strlen(mdcbufr),mdcbufr);

  strcpy(mdcbufr,fi->study_descr);
  mdc_dicom_write_element(fi->ofp,0x0008,0x1030,strlen(mdcbufr),mdcbufr);
 
  /* group 0x0010 */ 
  sprintf(mdcbufr,"%.64s^^^^",fi->patient_name);
  mdc_dicom_write_element(fi->ofp,0x0010,0x0010,strlen(mdcbufr),mdcbufr);

  pdata = fi->patient_id;
  mdc_dicom_write_element(fi->ofp,0x0010,0x0020,strlen(pdata),pdata);

  mdc_dicom_write_element(fi->ofp,0x0010,0x0030,0,NULL); /* Pat Birth Date */
  mdc_dicom_write_element(fi->ofp,0x0010,0x0032,0,NULL); /* Pat Birth Time */

  strcpy(mdcbufr,fi->patient_sex); MdcLowStr(mdcbufr);
  if (strchr(mdcbufr,'f') != NULL) {
    /* first check for fe-male */
    strcpy(mdcbufr,"F");
  }else if (strchr(mdcbufr,'m') != NULL) {
    /* now   check for    male */
    strcpy(mdcbufr,"M");
  }else {
    /* guess what? */
    strcpy(mdcbufr,"O");
  }
  mdc_dicom_write_element(fi->ofp,0x0010,0x0040,strlen(mdcbufr),mdcbufr);

  /* 0x0018 */
  sprintf(mdcbufr,"%+e",fi->image[0].slice_width);
  mdc_dicom_write_element(fi->ofp,0x0018,0x0050,strlen(mdcbufr),mdcbufr);

  /* MARK */
  strcpy(mdcbufr,"0");
  mdc_dicom_write_element(fi->ofp,0x0018,0x0070,strlen(mdcbufr),mdcbufr);
  /* replaced mdc_dicom_write_element(fi->ofp,0x0018,0x0070,0,NULL); */

  sprintf(mdcbufr,"%+e",fi->image[0].slice_spacing);
  mdc_dicom_write_element(fi->ofp,0x0018,0x0088,strlen(mdcbufr),mdcbufr);

  pdata = NULL;
  switch (fi->image[0].pat_slice_orient) {

      case MDC_SUPINE_HEADFIRST_TRANSVERSAL:
      case MDC_SUPINE_HEADFIRST_SAGITTAL   :
      case MDC_SUPINE_HEADFIRST_CORONAL    :
          strcpy(mdcbufr,"HFS"); pdata = mdcbufr; break;
      case MDC_PRONE_HEADFIRST_TRANSVERSAL :
      case MDC_PRONE_HEADFIRST_SAGITTAL    :
      case MDC_PRONE_HEADFIRST_CORONAL     :
          strcpy(mdcbufr,"HFP"); pdata = mdcbufr; break;
      case MDC_SUPINE_FEETFIRST_TRANSVERSAL:
      case MDC_SUPINE_FEETFIRST_SAGITTAL   :
      case MDC_SUPINE_FEETFIRST_CORONAL    :
          strcpy(mdcbufr,"FFS"); pdata = mdcbufr; break;
      case MDC_PRONE_FEETFIRST_TRANSVERSAL :
      case MDC_PRONE_FEETFIRST_SAGITTAL    :
      case MDC_PRONE_FEETFIRST_CORONAL     :
          strcpy(mdcbufr,"FFP"); pdata = mdcbufr; break;
  }

  if (dicom->PET_MOD != MDC_YES) {
    switch (fi->acquisition_type) {
      case MDC_ACQUISITION_UNKNOWN: /* fake as static */
      case MDC_ACQUISITION_STATIC:  /*MARK: normally whole body too ... */
        sprintf(mdcbufr,"%12.0f",fi->image[0].frame_duration);
        mdc_dicom_write_element(fi->ofp,0x0018,0x1242,strlen(mdcbufr),mdcbufr);
    }
  }

  /* MARK: "Patient Position" removed - only for MR module               */
  /*
  if (pdata != NULL)
    mdc_dicom_write_element(fi->ofp,0x0018,0x5100,strlen(pdata),pdata);
  else
    mdc_dicom_write_element(fi->ofp,0x0018,0x5100,0,NULL);
  */

  /* group 0x0020 */
  /* strcpy(mdcbufr,"1.2.840.10008.Study.Instance"); */ 
  MdcDicomMakeUID(mdcbufr,psec);
  mdc_dicom_write_element(fi->ofp,0x0020,0x000D,strlen(mdcbufr),mdcbufr);

  /* strcpy(mdcbufr,"1.2.840.10008.Series.Instance"); */
  MdcDicomMakeUID(mdcbufr,psec);
  mdc_dicom_write_element(fi->ofp,0x0020,0x000E,strlen(mdcbufr),mdcbufr);

  pdata = fi->study_name;
  mdc_dicom_write_element(fi->ofp,0x0020,0x0010,strlen(pdata),pdata);

  /* MARK */
  mdc_dicom_write_element(fi->ofp,0x0020,0x0011,strlen(dummy1),dummy1);
  mdc_dicom_write_element(fi->ofp,0x0020,0x0013,strlen(dummy1),dummy1);
  /* replaced mdc_dicom_write_element(fi->ofp,0x0020,0x0011,0,NULL); */
  /* replaced mdc_dicom_write_element(fi->ofp,0x0020,0x0013,0,NULL); */

  /* strcpy(mdcbufr,"1.2.840.10008.Frame.of.Reference"); */
  MdcDicomMakeUID(mdcbufr,psec);
  mdc_dicom_write_element(fi->ofp,0x0020,0x0052,strlen(mdcbufr),mdcbufr);

  /* mdc_dicom_write_element(fi->ofp,0x0020,0x0060,0,NULL); Laterality */

  sprintf(mdcbufr,"%u",fi->number);
  mdc_dicom_write_element(fi->ofp,0x0020,0x1002,strlen(mdcbufr),mdcbufr);

  strcpy(mdcbufr,"Unknown"); 
  mdc_dicom_write_element(fi->ofp,0x0020,0x1040,strlen(mdcbufr),mdcbufr);
 
  strcpy(mdcbufr,"*** NOT APPROVED ***");
  mdc_dicom_write_element(fi->ofp,0x0020,0x4000,strlen(mdcbufr),mdcbufr);

  /* group 0x0028 */
  ui16 = 1;
  mdc_dicom_write_element(fi->ofp,0x0028,0x0002,sizeof(ui16),(Uint8 *)&ui16);

  strcpy(mdcbufr,"MONOCHROME2");
  mdc_dicom_write_element(fi->ofp,0x0028,0x0004,strlen(mdcbufr),mdcbufr);

  sprintf(mdcbufr,"%u",fi->number);
  mdc_dicom_write_element(fi->ofp,0x0028,0x0008,strlen(mdcbufr),mdcbufr);

  bytes = 0; pui16 = NULL;
  if (dicom->PET_MOD == MDC_YES) {
    switch (fi->acquisition_type) {
     case MDC_ACQUISITION_DYNAMIC:
         bytes = 4 * sizeof(Uint16); pui16 = (Uint16 *)malloc(bytes);
         if (pui16 == NULL) 
           return("DICM Couldn't malloc (PT/DYNAMIC) FrameIncrPointer");
         pui16[0]=0x0054; pui16[1]=0x0080; /* slices */
         pui16[2]=0x0054; pui16[3]=0x0100; /* frames */
         dicom->VectDO[MDC_VECT_SLICE]     = MDC_YES;
         dicom->VectDO[MDC_VECT_TIMESLICE] = MDC_YES;
         break;
     case MDC_ACQUISITION_TOMO:
     case MDC_ACQUISITION_STATIC:
     case MDC_ACQUISITION_UNKNOWN:
     default:
         bytes = 2 * sizeof(Uint16); pui16 = (Uint16 *)malloc(bytes);
         if (pui16 == NULL)
           return("DICM Couldn't malloc (PT/STATIC) FrameIncrPointer");
         pui16[0]=0x0054; pui16[1]=0x0080; /* slices */
         dicom->VectDO[MDC_VECT_SLICE]     = MDC_YES;
    }
    mdc_dicom_write_element(fi->ofp,0x0028,0x0009,bytes,(Uint8 *)pui16);
    MdcFree(pui16);
  }else{
    switch (fi->acquisition_type) {
     case MDC_ACQUISITION_TOMO:
         bytes = 2 * sizeof(Uint16); pui16 = (Uint16 *)malloc(bytes);
         if (pui16 == NULL) 
           return("DICM Couldn't malloc (NM/TOMO) FrameIncrPointer");
         pui16[0]=0x0054; pui16[1]=0x0080; /* slices */
         dicom->VectDO[MDC_VECT_SLICE]     = MDC_YES;
         break;
     case MDC_ACQUISITION_DYNAMIC:
         bytes = 8 * sizeof(Uint16); pui16 = (Uint16 *)malloc(bytes);
         if (pui16 == NULL)
           return("DICM Couldn't malloc (NM/DYNAMIC) FrameIncrPointer");
         pui16[0]=0x0054; pui16[1]=0x0010; /* energy windows */
         pui16[2]=0x0054; pui16[3]=0x0020; /* detectors      */
         pui16[4]=0x0054; pui16[5]=0x0030; /* phases         */
         pui16[6]=0x0054; pui16[7]=0x0100; /* time slices    */
         dicom->VectDO[MDC_VECT_ENERGYWINDOW] = MDC_YES;
         dicom->VectDO[MDC_VECT_DETECTOR]     = MDC_YES;
         dicom->VectDO[MDC_VECT_PHASE]        = MDC_YES;
         dicom->VectDO[MDC_VECT_TIMESLICE]    = MDC_YES;
         break;
     case MDC_ACQUISITION_UNKNOWN: /* fake as static */
     case MDC_ACQUISITION_STATIC:
         bytes = 4 * sizeof(Uint16); pui16 = (Uint16 *)malloc(bytes);
         if (pui16 == NULL)
           return("DICM Couldn't malloc (NM/STATIC) FrameIncrPointer");
         pui16[0]=0x0054; pui16[1]=0x0010; /* energy windows */
         pui16[2]=0x0054; pui16[3]=0x0020; /* detectors      */
         dicom->VectDO[MDC_VECT_ENERGYWINDOW] = MDC_YES;
         dicom->VectDO[MDC_VECT_DETECTOR]     = MDC_YES;
         break; 
    }
    mdc_dicom_write_element(fi->ofp,0x0028,0x0009,bytes,(Uint8 *)pui16);
    MdcFree(pui16);
  }

  ui16 = (Uint16) fi->mheight;
  mdc_dicom_write_element(fi->ofp,0x0028,0x0010,sizeof(ui16),(Uint8 *)&ui16);

  ui16 = (Uint16) fi->mwidth;
  mdc_dicom_write_element(fi->ofp,0x0028,0x0011,sizeof(ui16),(Uint8 *)&ui16);

  sprintf(mdcbufr,"%+e\\%+e",fi->pixdim[IROW],fi->pixdim[ICOL]);
  mdc_dicom_write_element(fi->ofp,0x0028,0x0030,strlen(mdcbufr),mdcbufr);

  if (fi->decay_corrected) {
    strcpy(mdcbufr,"DECY");
    mdc_dicom_write_element(fi->ofp,0x0028,0x0051,strlen(mdcbufr),mdcbufr);
  }else{
    mdc_dicom_write_element(fi->ofp,0x0028,0x0051,0,NULL);
  }

  bits_allocated = (Uint16)MdcType2Bits(type);
  if (MDC_FORCE_INT == BIT16_S) {
    bits_stored = MDC_INT16_BITS_USED;
  }else{
    bits_stored = MdcType2Bits(type);
  }
  ui16 = bits_allocated; /* bits allocated */
  mdc_dicom_write_element(fi->ofp,0x0028,0x0100,sizeof(ui16),(Uint8 *)&ui16);
  
  ui16 = bits_stored;      /* bits stored */
  mdc_dicom_write_element(fi->ofp,0x0028,0x0101,sizeof(ui16),(Uint8 *)&ui16);

  ui16 = bits_stored - 1;  /* high bit */
  mdc_dicom_write_element(fi->ofp,0x0028,0x0102,sizeof(ui16),(Uint8 *)&ui16);

  switch (type) {
   case  BIT8_U:
   case BIT16_U: 
   case BIT32_U:
   case BIT64_U:  ui16 = 0; break;

   case  BIT8_S:
   case BIT16_S:
   case BIT32_S:
   case BIT64_S:  ui16 = 1; break;

   default: ui16 = 0;
  } 
  if (type == BIT16_S && MDC_INT16_BITS_USED < 16) ui16 = 0; /* unsigned */
  mdc_dicom_write_element(fi->ofp,0x0028,0x0103,sizeof(ui16),(Uint8 *)&ui16);

  sprintf(mdcbufr,"%+e",rescale_intercept);
  mdc_dicom_write_element(fi->ofp,0x0028,0x1052,strlen(mdcbufr),mdcbufr);
 
  /* need to rewrite the following tag after rescaling images */
  REWRF = ftell(fi->ofp);
  sprintf(mdcbufr,"%+e",rescale_slope);
  mdc_dicom_write_element(fi->ofp,0x0028,0x1053,strlen(mdcbufr),mdcbufr);

  if (dicom->VectDO[MDC_VECT_ENERGYWINDOW] == MDC_YES) {
    /* MARK: window vectors */
      if (fi->dim[7] == 0)
        return("DICM Bad zero value for fi->dim[7]");
      if (fi->number % fi->dim[7])
        return("DICM Garbled value for fi->dim[7]");
      vect = fi->number / fi->dim[7];
      bytes = fi->number * sizeof(Uint16);
      pui16 = (Uint16 *)malloc(bytes);
      if (pui16 == NULL)
        return("DICM Couldn't malloc EnergyWindowVector");
      for (i=0; i<fi->number; i++) pui16[i] = (Uint16)((i/vect)+1);
      mdc_dicom_write_element(fi->ofp,0x0054,0x0010,bytes,(Uint8 *)pui16);
      MdcFree(pui16);
  }

  /* number of energy windows */
  ui16 = fi->dim[7];
  mdc_dicom_write_element(fi->ofp,0x0054,0x0011,sizeof(ui16),(Uint8 *)&ui16);

  /* window information sequence */
  MdcDicomWriteInfoSeq(fi->ofp,0x0054,0x0012);

  /* item */
  MdcDicomWriteItem(fi->ofp);
    /* window range sequence */
    MdcDicomWriteInfoSeq(fi->ofp,0x0054,0x0013);
     /* item */
     MdcDicomWriteItem(fi->ofp);
     /* item delimitation item */
     MdcDicomWriteItemDelItem(fi->ofp);
     /* sequence delimitation item*/
    MdcDicomWriteInfoSeqDelItem(fi->ofp);
  /* item delimitation item */
  MdcDicomWriteItemDelItem(fi->ofp);
  /* sequence delimitation item */
  MdcDicomWriteInfoSeqDelItem(fi->ofp);

  /* radiopharmaceutical info sequence */
  MdcDicomWriteInfoSeq(fi->ofp,0x0054,0x0016);
  /* sequence delimiter */
  MdcDicomWriteInfoSeqDelItem(fi->ofp);
  

  if (dicom->VectDO[MDC_VECT_DETECTOR] == MDC_YES) { 
    /* MARK: detector vectors */
    if (fi->dim[6] == 0)
      return("DICM Bad zero value for fi->dim[6]");
    if (fi->number % fi->dim[6])
      return("DICM Garbled value for fi->dim[6]");
    vect = fi->number / fi->dim[6];
    bytes = fi->number * sizeof(Uint16);
    pui16 = (Uint16 *)malloc(bytes);
    if (pui16 == NULL)
      return("DICM Couldn't malloc DetectorVector");
    for (i=0; i<fi->number; i++) pui16[i] = (Uint16)((i/vect)+1);
    mdc_dicom_write_element(fi->ofp,0x0054,0x0020,bytes,(Uint8 *)pui16);
    MdcFree(pui16);
  }
  /* number of detector heads */
  ui16 = fi->dim[6];
  mdc_dicom_write_element(fi->ofp,0x0054,0x0021,sizeof(ui16),(Uint8 *)&ui16);
  
  /* detector info sequence */
  MdcDicomWriteInfoSeq(fi->ofp,0x0054,0x0022);
     
   /* item */
   MdcDicomWriteItem(fi->ofp);
      
    /* colimator type */
    mdc_dicom_write_element(fi->ofp,0x0018,0x1181,0,NULL);
    
    /* focal distance */
    mdc_dicom_write_element(fi->ofp,0x0018,0x1182,0,NULL);
    
    /* image pos patient */        
    sprintf(mdcbufr,"%+e\\%+e\\%+e",fi->image[0].image_pos_pat[0]
                                   ,fi->image[0].image_pos_pat[1]
                                   ,fi->image[0].image_pos_pat[2]);
    mdc_dicom_write_element(fi->ofp,0x0020,0x0032,strlen(mdcbufr),mdcbufr);
    /* image orient patient */     
    sprintf(mdcbufr,"%+e\\%+e\\%+e\\%+e\\%+e\\%+e"
                                   ,fi->image[0].image_orient_pat[0]
                                   ,fi->image[0].image_orient_pat[1]
                                   ,fi->image[0].image_orient_pat[2]
                                   ,fi->image[0].image_orient_pat[3]
                                   ,fi->image[0].image_orient_pat[4]
                                   ,fi->image[0].image_orient_pat[5]);
    mdc_dicom_write_element(fi->ofp,0x0020,0x0037,strlen(mdcbufr),mdcbufr);
    
    /* start angle */
    mdc_dicom_write_element(fi->ofp,0x0054,0x0200,0,NULL);
   
   /* item delimitation item */
   MdcDicomWriteItemDelItem(fi->ofp);
  
  /* sequence delimitation item */
  MdcDicomWriteInfoSeqDelItem(fi->ofp);
 
  if (dicom->PET_MOD == MDC_YES) {

   switch (fi->acquisition_type) {

     /* PET MODALITY */
     case MDC_ACQUISITION_DYNAMIC:
      /* MARK: slice vector */
      if (dicom->VectDO[MDC_VECT_SLICE] == MDC_YES) {
        if (fi->dim[3] == 0)
          return("DICM Bad zero value for fi->dim[3] (PT/DYNAMIC)");
        vect = fi->dim[3];          
        bytes = fi->number*sizeof(Uint16); 
        pui16=(Uint16 *)malloc(bytes);
        if (pui16 == NULL)
          return("DICM Couldn't malloc SliceVector (PT/DYNAMIC)");
        for (i=0; i<fi->number; i++) pui16[i]=(Uint16)((i%vect)+1);
        mdc_dicom_write_element(fi->ofp,0x0054,0x0080,bytes,(Uint8 *)pui16);
        MdcFree(pui16);
      }

      ui16 = (Uint16) fi->dim[3];
      mdc_dicom_write_element(fi->ofp,0x0054,0x0081,sizeof(Uint16)
                                                   ,(Uint8 *)&ui16);

      /* MARK: time slice vector */
      if (dicom->VectDO[MDC_VECT_TIMESLICE] == MDC_YES) {
        if (fi->dim[4] == 0)
          return("DICM Bad zero value for fi->dim[4] (PT/DYNAMIC)");
        if (fi->number % fi->dim[4])
          return("DICM Garbled value for fi->dim[4] (PT/DYNAMIC)");
        vect = fi->number / fi->dim[4];
        bytes = fi->number * sizeof(Uint16);
        pui16 = (Uint16 *)malloc(bytes);
        if (pui16 == NULL)
          return("DICM Couldn't malloc TimeSlotVector (PT/DYNAMIC)");
        for (i=0; i<fi->number; i++) pui16[i] = (Uint16)((i/vect)+1);
        mdc_dicom_write_element(fi->ofp,0x0054,0x0100,bytes,(Uint8 *)pui16);
        MdcFree(pui16);
      }

      ui16 = (Uint16) fi->dim[4];
      mdc_dicom_write_element(fi->ofp,0x0054,0x0101,sizeof(Uint16)
                                                   ,(Uint8 *)&ui16);
      break;
     case MDC_ACQUISITION_STATIC :
     case MDC_ACQUISITION_TOMO   :
     case MDC_ACQUISITION_UNKNOWN:
     default:
      if (fi->dim[4] > 1) return("DICM Unsupported dim[]-values (PT/TOMO)");

      /* MARK: slice vector */
      if (dicom->VectDO[MDC_VECT_SLICE] == MDC_YES) {
        bytes = fi->number*sizeof(Uint16); pui16=(Uint16 *)malloc(bytes);
        if (pui16 == NULL)
          return("DICM Couldn't malloc slice vector buffer (PT/TOMO)");
        for (i=0; i<fi->number; i++) pui16[i]=(Uint16)i+1;
        mdc_dicom_write_element(fi->ofp,0x0054,0x0080,bytes,(Uint8 *)pui16);
        MdcFree(pui16);
      }

      ui16 = (Uint16) fi->dim[3];
      mdc_dicom_write_element(fi->ofp,0x0054,0x0081,sizeof(Uint16)
                                                   ,(Uint8 *)&ui16);
    }

    switch (fi->acquisition_type) {
     case MDC_ACQUISITION_TOMO: strcpy(mdcbufr,"STATIC"); break;
     case MDC_ACQUISITION_DYNAMIC: strcpy(mdcbufr,"DYNAMIC"); break;
     case MDC_ACQUISITION_UNKNOWN: /* fake as static */
     case MDC_ACQUISITION_STATIC : strcpy(mdcbufr,"STATIC"); break;
     default: strcpy(mdcbufr,"UNSPECIFIED");
    }

    /* type of detector motion */
    /* strcpy(mdcbufr,"UNDEFINED"); */
    /* mdc_dicom_write_element(fi->ofp,0x0054,0x0202,strlen(mdcbufr),mdcbufr);*/

    /* patient orientation code sequence */
    MdcDicomWriteInfoSeq(fi->ofp,0x0054,0x0410);
    /* sequence delimitation item */
    MdcDicomWriteInfoSeqDelItem(fi->ofp);

    /* patient gantry relationship code sequence */
    MdcDicomWriteInfoSeq(fi->ofp,0x0054,0x0414);
    /* sequence delimitation item */
    MdcDicomWriteInfoSeqDelItem(fi->ofp);

    if (fi->reconstructed == MDC_YES)
      strcat(mdcbufr,"\\IMAGE");
    else
      strcat(mdcbufr,"\\REPROJECTION");
    mdc_dicom_write_element(fi->ofp,0x0054,0x1000,strlen(mdcbufr),mdcbufr); 

 
    strcpy(mdcbufr,"UNKNOWN"); /* units */
    mdc_dicom_write_element(fi->ofp,0x0054,0x1001,strlen(mdcbufr),mdcbufr);

    strcpy(mdcbufr,"UNKNOWN"); /* counts source */
    mdc_dicom_write_element(fi->ofp,0x0054,0x1002,strlen(mdcbufr),mdcbufr);

    if (fi->decay_corrected) {
      strcpy(mdcbufr,"ADMIN");
      mdc_dicom_write_element(fi->ofp,0x0054,0x1102,strlen(mdcbufr),mdcbufr);
    }

  }else{ /* NM-MODALITY */

    /* group 0x0054 */ /* MARK: SHIT STARTS HERE <=> InterFile */
                       /* will fail for unreconstructed images */

    if (fi->acquisition_type == MDC_ACQUISITION_DYNAMIC) {

     /* MARK: phases vector*/
     if (dicom->VectDO[MDC_VECT_PHASE] == MDC_YES) {
       if (fi->dim[4] == 0)
         return("DICM Bad zero value for fi->dim[4] (NM/DYNAMIC)");
       if (fi->number % fi->dim[4])
         return("DICM Garbled value for fi->dim[4] (NM/DYNAMIC)");
       vect = fi->number / fi->dim[4];
       bytes = fi->number * sizeof(Uint16);
       pui16 = (Uint16 *)malloc(bytes);
       if (pui16 == NULL)
         return("DICM Couldn't malloc PhaseVector (NM/DYNAMIC)");
       for (i=0; i<fi->number; i++) pui16[i] = (Uint16)((i/vect)+1);
       mdc_dicom_write_element(fi->ofp,0x0054,0x0030,bytes,(Uint8 *)pui16);
       MdcFree(pui16);
     }

     /* number of phases */
     ui16 = fi->dim[4];
     mdc_dicom_write_element(fi->ofp,0x0054,0x0031,sizeof(ui16),(Uint8 *)&ui16);

     MdcDicomWriteInfoSeq(fi->ofp,0x0054,0x0032);

     for (i=0; i<fi->dim[4]; i++) {
        MdcDicomWriteItem(fi->ofp);
        /* actual frame duration */
        sprintf(mdcbufr,"%12.0f",fi->image[i*fi->dim[3]].frame_duration);
        mdc_dicom_write_element(fi->ofp,0x0018,0x1242,strlen(mdcbufr)
                                                     ,(Uint8 *)mdcbufr);
        /* number of frames in phase */
        ui16 = (Uint16)fi->dim[3];
        mdc_dicom_write_element(fi->ofp,0x0054,0x0033,sizeof(ui16)
                                                     ,(Uint8 *)&ui16);

        strcpy(mdcbufr,"0");
        /* phase delay */
        mdc_dicom_write_element(fi->ofp,0x0054,0x0036,strlen(mdcbufr),mdcbufr);
        /* pause between frames */
        mdc_dicom_write_element(fi->ofp,0x0054,0x0038,strlen(mdcbufr),mdcbufr);
        /* number of triggers in phase */
        /* ui16 = 0;
        mdc_dicom_write_element(fi->ofp,0x0054,0x0211,sizeof(ui16)
                                                     ,(Uint8 *)&ui16);
        */
        MdcDicomWriteItemDelItem(fi->ofp);
     }
     MdcDicomWriteInfoSeqDelItem(fi->ofp);

     /* MARK: time slot vector*/
     if (dicom->VectDO[MDC_VECT_TIMESLICE] == MDC_YES) {
       if (fi->dim[3] == 0)
         return("DICM Bad zero value for fi->dim[3] (NM/DYNAMIC)");
       vect = fi->dim[3];
       bytes = fi->number * sizeof(Uint16);
       pui16 = (Uint16 *)malloc(bytes);
       if (pui16 == NULL)
         return("DICM Couldn't malloc TimeSlotVector (NM/DYNAMIC)");
       for (i=0; i<fi->number; i++) pui16[i] = (Uint16)((i%vect)+1);
       mdc_dicom_write_element(fi->ofp,0x0054,0x0100,bytes,(Uint8 *)pui16);
       MdcFree(pui16);
     }
     
    } 

    if (fi->acquisition_type == MDC_ACQUISITION_TOMO) {

      /* number of rotations */
      ui16 = (Uint16)fi->acqnr;
      mdc_dicom_write_element(fi->ofp,0x0054,0x0051,sizeof(ui16)
                                                   ,(Uint8 *)&ui16);
      
      /* rotation info sequence */
      MdcDicomWriteInfoSeq(fi->ofp,0x0054,0x0052);

       if (fi->acqdata != NULL) for (i=0; i<fi->acqnr; i++) {
         ACQ_DATA *acq = &fi->acqdata[i];
         /* item */
         MdcDicomWriteItem(fi->ofp);

         /* MARK: distance source to detector: only transmission   */
         /* mdc_dicom_write_element(fi->ofp,0x0018,0x1110,0,NULL); */

         switch (acq->rotation_direction) {
           case MDC_ROTATION_CW: strcpy(mdcbufr,"CW"); break;
           case MDC_ROTATION_CC: strcpy(mdcbufr,"CC"); break;
           default             : mdcbufr[0] = '\0';
         } 
         mdc_dicom_write_element(fi->ofp,0x0018,0x1140,strlen(mdcbufr),mdcbufr);
         sprintf(mdcbufr,"%g",acq->scan_arc);
         mdc_dicom_write_element(fi->ofp,0x0018,0x1143,strlen(mdcbufr),mdcbufr);
         sprintf(mdcbufr,"%g",acq->angle_step);
         mdc_dicom_write_element(fi->ofp,0x0018,0x1144,strlen(mdcbufr),mdcbufr);
         strcpy(mdcbufr,"0");
         mdc_dicom_write_element(fi->ofp,0x0018,0x1242,strlen(mdcbufr),mdcbufr);

         ui16 = (Uint16)fi->dim[3];
         mdc_dicom_write_element(fi->ofp,0x0054,0x0053,sizeof(ui16)
                                                      ,(Uint8 *)&ui16);
         sprintf(mdcbufr,"%g",acq->angle_start);
         mdc_dicom_write_element(fi->ofp,0x0054,0x0200,strlen(mdcbufr),mdcbufr);

         switch (acq->detector_motion) {
           case MDC_MOTION_STEP: strcpy(mdcbufr,"STEP AND SHOOT");  break;
           case MDC_MOTION_CONT: strcpy(mdcbufr,"CONTINUOUS");      break;
           case MDC_MOTION_DRNG: strcpy(mdcbufr,"ACQ DURING STEP"); break;
           default             : strcpy(mdcbufr,"UNDEFINED");
         }
         mdc_dicom_write_element(fi->ofp,0x0054,0x0202,strlen(mdcbufr),mdcbufr);

         MdcDicomWriteItemDelItem(fi->ofp);
       }

      /* sequence delimitation item */
      MdcDicomWriteInfoSeqDelItem(fi->ofp);
  
      if (dicom->VectDO[MDC_VECT_SLICE] == MDC_YES) { 
        bytes = fi->number*sizeof(Uint16); pui16=(Uint16 *)malloc(bytes);
        if (pui16 == NULL)
          return("DICM Couldn't malloc slice vector buffer (NM/TOMO)");
        vect = fi->dim[3];
        for (i=0; i<fi->number; i++) pui16[i]=(Uint16)((i%vect)+1);
        mdc_dicom_write_element(fi->ofp,0x0054,0x0080,bytes,(Uint8 *)pui16);
        MdcFree(pui16);
      }
 
      /* number of slices */
      ui16 = (Uint16) fi->dim[3];
      mdc_dicom_write_element(fi->ofp,0x0054,0x0081,sizeof(Uint16)
                                                   ,(Uint8 *)&ui16);

    } 

    /* patient orientation code sequence */
    MdcDicomWriteInfoSeq(fi->ofp,0x0054,0x0410);
    /* sequence delimitation item */
    MdcDicomWriteInfoSeqDelItem(fi->ofp);

    /* patient gantry relationship code sequence */
    MdcDicomWriteInfoSeq(fi->ofp,0x0054,0x0414);
    /* sequence delimitation item */
    MdcDicomWriteInfoSeqDelItem(fi->ofp);
  }

  /* dump the images */
  bytes = fi->number * fi->mwidth * fi->mheight * MdcType2Bytes(type);
  mdc_dicom_write_element(fi->ofp,0x7fe0,0x0010,bytes,(Uint8 *)&type);

  for (i=0; i<fi->number; i++) {
#if XSUPPORTED
     if (XMDC_MEDCON) {
       pvalue += 1./(float)fi->number;
       XMdcUpdateProgressBar(NULL);
     }
#endif 

     switch (type) {
       case BIT8_U : newbuff = MdcGetImgBIT8_U(fi,i);   break;
       case BIT16_S: newbuff = MdcGetImgBIT16_S(fi,i);  break;
       default: newbuff = NULL; /* bad pixel type */
     } 
     if (newbuff == NULL) return("DICM Couldn't malloc newbuff image");

     if (fi->diff_size == MDC_YES) {
       buff = MdcGetResizedImage(fi,newbuff,type,i);
       if (buff == NULL) return("DICM Couldn't malloc resized image");
       MdcFree(newbuff);
     }else buff = newbuff;

     if (MDC_FILE_ENDIAN != MDC_HOST_ENDIAN) 
       MdcMakeImgSwapped(buff, fi, i, fi->mwidth, fi->mheight, type);

     pixels = fi->mwidth * fi->mheight; 
     bytes = MdcType2Bytes(type);
     if (fwrite(buff,bytes,pixels,fi->ofp) != pixels)
       return("DICM Bad writing of image");

     MdcFree(buff);
  }

  if (MDC_QUANTIFY == MDC_YES || MDC_CALIBRATE == MDC_YES) {
    /* rewrite the true rescale_slope value */
    fseek(fi->ofp,REWRF,SEEK_SET);
    rescale_slope = fi->image[0].rescaled_fctr;
    sprintf(mdcbufr,"%+e",rescale_slope);
    mdc_dicom_write_element(fi->ofp,0x0028,0x1053,strlen(mdcbufr),mdcbufr);
  } 
 
  MdcFileClose(fi->ofp);

  return NULL;

}


/************
 * MDC open *
 ************/

/**********
 * images *
 **********/

static int mdc_dicom_read(FILEINFO *fi, IMAGE **image, int *images)
{
  int err;

  dicom_init(fi->ifp);

  /* last argument: parametric(=1) (see more in vtdicom file transform.c)*/
  /* we need original values, preventing the rescaling to max/inverting  */
  /* for doing the things ourselves (width/center or slope/intercept)    */

  err = dicom_read(fi->ipath,image,images,1);

  return(err);

}

/********
 * info *
 ********/

static void mdc_dicom_getinfo(FILEINFO *fi)
{
  ELEMENT       *e;
  DICTIONARY    *d;
  MDC_ACR_TAG   acrtag;
  /* MARK Int8 saved_file_endian = MDC_FILE_ENDIAN; */

  dicom_log(INFO,"dump_open()");

  dicom_init(fi->ifp);

  if (dicom_open(fi->ipath))
    return;

  for (;;)
  {
    e=dicom_element();
    if (!e)
      return;

    d=dicom_query(e);

    if (e->vr==UN)
      e->vr=d->vr;

    if (mdc_dicom_load(e->vr))
      return;

    acrtag.group  = e->group;
    acrtag.element= e->element;
    acrtag.length = e->length;
    acrtag.data   = (Uint8 *)e->value.UN;
    MdcDoTag(&acrtag,fi,0);

    MdcFree(e->value.UN);
  }
}


/********
 * open *
 ********/
static void mdc_dicom_dumpinfo(FILEINFO *fi)
{
  ELEMENT       *e;
  DICTIONARY    *d;
 
  dicom_log(INFO,"dump_open()");

  dicom_init(fi->ifp);
 
  if (dicom_open(fi->ipath))
    return;
 
  for (;;)
  {
    e=dicom_element();
    if (!e)
      return;

    d=dicom_query(e);
 
    if (e->vr==UN && d->vr!=ox) { /* replace, except for special tags */
      e->vr=d->vr;
    }
 
    if (dicom_load(e->vr))
      return;
 
    mdc_dicom_printinfo(e,d->description);

    MdcFree(e->value.UN);
  }
}


/*********
 * print *
 *********/
 
static void mdc_dicom_printinfo(const ELEMENT *e,const char *description)
{
  U32 i;

  dicom_log(INFO,"dump_print()");
 
  for (i=e->sequence; i; i--)
    printf("  ");
 
  if (verbose)
    printf("(%.4X,%.4X) %c%c[%u] ",e->group,e->element,e->vr>>8,e->vr&0xFF,
    e->vm);
 
  printf("%s%s: ",e->encapsulated?"Encapsulated ":"",description);
 
  if (!e->vm)
  {
    puts("(no value)");
    return;
  }
 
  if (e->length == UNDEFINED_LENGTH)
  {
    puts("(undefined length)");
    return;
  }
 
  for (i=0; i<e->vm; i++)
    switch(e->vr)
    {
    case US :
      printf("%u ",e->value.US[i]);
      break;
 
    case SS :
      printf("%d ",e->value.SS[i]);
      break;
 
    case UL :
      printf("%u ",e->value.UL[i]);
      break;
 
    case SL :
      printf("%d ",e->value.SL[i]);
      break;
 
    case AT :
      printf("(%.4X,%.4X) ",e->value.AT[i].group,e->value.AT[i].element);
      break;
 
    case FL :
      printf("%f ",e->value.FL[i]);
      break;
 
    case FD :
      printf("%f ",e->value.FD[i]);
      break;
 
    case LT :
    case ST :
      MdcGetSafeString(mdcbufr,e->value.LT,e->length,MDC_2KB_OFFSET);
      printf("[%s] ",mdcbufr);
      break;
 
    case AE :
    case AS :
    case CS :
    case DA :
    case DS :
    case DT :
    case IS :
    case LO :
    case PN :
    case SH :
    case TM :
    case UI :
      MdcGetSafeString(mdcbufr,e->value.AE[i],strlen(e->value.AE[i]),MDC_2KB_OFFSET);
      printf("[%s] ",mdcbufr);
      break;
 
    default :
      printf("(%u bytes)\n",e->length);
      return;
    }
 
  if (verbose)
    printf("(%u bytes)",e->length);
 
  puts("");
}


void mdc_dicom_get_vr(ELEMENT *e)
{
   DICTIONARY *d;

   d = dicom_query(e);

   e->vr = d->vr;
}

Uint8 *mdc_dicom_handle_vr(ELEMENT *e, Uint8 *tdata)
{
  switch (e->vr) {
    case ox:
        if ((e->group == 0x7fe0) && (e->element == 0x0010)) {
          Int16 type; memcpy(&type,tdata,2);
          switch (type) {
            case BIT8_U : e->vr = OB; return(NULL);
            case BIT16_S: e->vr = OW; return(NULL);
          }
        }
        /* else other handle code */
        break;
    default: return(tdata); /* no special VR */

  }

  /* error exit = unhandled special VR */
  MdcPrntErr(MDC_BAD_CODE,"Internal ## Extra code required for tag %x:%x"
                         ,e->group,e->element);

  return(tdata);

}


/*********
 * write *
 *********/

int mdc_dicom_write_element(FILE *fp, Uint16 group, Uint16 element, Uint32 length, Uint8 *data)
{
  ELEMENT element_t, *e;
  Uint32 i, vr_w, length32_w;
  Uint16 length16_w;
  Int8 file_endian_saved=MDC_FILE_ENDIAN, MAKE_EVEN=0;
  Uint8 *ndata=NULL;

  /* make even tags */
  if ((length%2) && (length != UNDEFINED_LENGTH)) {
    MAKE_EVEN = 1;
  }

  /* fill in the values */
  e = &element_t;
  e->group = group;
  e->element = element;
  e->length  = length + MAKE_EVEN;
  length32_w = e->length; length16_w = (Uint16)e->length;

  if (e->group == 0x0002) MDC_FILE_ENDIAN=MDC_LITTLE_ENDIAN; /* force little */

  /* fix endian of tag items to write */
  MdcSWAP(group); MdcSWAP(element);
  MdcSWAP(length16_w); MdcSWAP(length32_w);

  /* write group */
  fwrite((Uint8 *)&group,1,sizeof(e->group),fp);
  /* write element */
  fwrite((Uint8 *)&element,1,sizeof(e->element),fp);
  /* write value representation & length */
  mdc_dicom_get_vr(e);
  /* handle special VR values */
  ndata = mdc_dicom_handle_vr(e,data);
  vr_w = e->vr; 
  if (MdcHostBig()) vr_w = (vr_w << 16);
  else MdcForceSwap((Uint8 *)&vr_w,2); 
  switch (e->vr) { 
    case OB :
    case OW :
    case SQ :
    case UN :
    case UT : 
        if (e->group != 0xfffe) fwrite((Uint8 *)&vr_w,1,4,fp); 
        fwrite((Uint8 *)&length32_w,1,4,fp);
        break;
    case AT : /* 2 bytes endian sensitive data */
    case SS :
    case US :
        fwrite((Uint8 *)&vr_w,1,2,fp);
        fwrite((Uint8 *)&length16_w,1,2,fp);
        e->vm = length >> 1;
        for (i=0; i<e->vm; i++) MdcSwapBytes(ndata+(i<<1),2);
        break;
    case FL : /* 4 bytes endian sensitive data */
    case SL :
    case UL : 
        fwrite((Uint8 *)&vr_w,1,2,fp);
        fwrite((Uint8 *)&length16_w,1,2,fp);
        e->vm = length >> 2;
        for (i=0; i<e->vm; i++) MdcSwapBytes(ndata+(i<<2),4);
        break;
    case FD : /* 8 bytes endian sensitive data */
        fwrite((Uint8 *)&vr_w,1,2,fp);
        fwrite((Uint8 *)&length16_w,1,2,fp);
        e->vm = length >> 3;
        for (i=0; i<e->vm; i++) MdcSwapBytes(ndata+(i<<3),8);
        break;

    default :
        fwrite((Uint8 *)&vr_w,1,2,fp);
        fwrite((Uint8 *)&length16_w,1,2,fp); 
  }
  /* write value data */
  if ((ndata != NULL) && (length != 0) && (length != UNDEFINED_LENGTH)) {
    fwrite((Uint8 *)ndata,1,length,fp);
    if (MAKE_EVEN) fputc('\0',fp);
  }
  /* restore original output file endian */
  MDC_FILE_ENDIAN = file_endian_saved;

  if (ferror(fp)) return(MDC_NO);

  return(MDC_YES);

}

