/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * filename: m-anlz.c                                                      *
 *                                                                         *
 * UTIL C-source: Medical Image Conversion Utility                         *
 *                                                                         *
 * purpose      : read and write ANALYZE files                             *
 *                                                                         *
 * project      : (X)MedCon by Erik Nolf                                   *
 *                                                                         *
 * Functions    : MdcCheckANLZ()           - Check ANALYZE format          * 
 *                MdcReadANLZ()            - Read ANALYZE file             *
 *                MdcWriteANLZ()           - Write ANALYZE file            *
 *                MdcWriteHeaderKey()      - Write Header Key to file      *
 *                MdcWriteImageDimension() - Write Image Dimension to file *
 *                MdcWriteDataHistory()    - Write Data History to file    * 
 *                MdcWriteImagesData()     - Write the images to file      *
 *                MdcGetSpmOpt()           - Get specific SPM options      *
 *                                                                         *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* $Id: m-anlz.c,v 1.1.1.1 2000/10/28 16:51:23 enlf Exp $
 */

/*
   Copyright (C) 1997-2000 by Erik Nolf

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 2, or (at your option) any later
   version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
   Public License for more details.

   You should have received a copy of the GNU General Public License along
   with this program; if not, write to the Free Software Foundation, Inc.,
   59 Place - Suite 330, Boston, MA 02111-1307, USA.  */

/****************************************************************************
                              H E A D E R S
****************************************************************************/

#include "medcon.h"


/****************************************************************************
                            F U N C T I O N S
****************************************************************************/

int MdcCheckANLZ(FILEINFO *fi)
{
  MDC_ANLZ_HEADER_KEY hk;
  int check=2, FORMAT=MDC_FRMT_BAD;

  if (fread((char *)&hk,1,MDC_ANLZ_HK_SIZE,fi->ifp) != MDC_ANLZ_HK_SIZE) 
    return(MDC_BAD_READ);

  MDC_FILE_ENDIAN = MDC_HOST_ENDIAN;

  while (check--) {

    if ( (hk.sizeof_hdr==348 || hk.sizeof_hdr==148 || hk.sizeof_hdr==228) 
      && (hk.regular == MDC_SIG_ANLZ ) ) {
        FORMAT = MDC_FRMT_ANLZ;
        break;
    }
    MDC_FILE_ENDIAN = !MDC_HOST_ENDIAN;

    MdcSWAP(hk.sizeof_hdr);

  }

  return(FORMAT);

}

char *MdcReadANLZ(FILEINFO *fi)
{
  MDC_SPMOPT opt;
  FILE *fp=fi->ifp;
  MDC_ANLZ_HEADER_KEY hk;
  MDC_ANLZ_IMAGE_DIMS imd;
  MDC_ANLZ_DATA_HIST dh;
  IMG_DATA *id;
  Uint32 pixoffset, bytes, d, w, i;
  Uint8 *pbuf;
  Int8 WAS_COMPRESSED = MDC_NO;

  char *err=NULL, *origpath=NULL;
#if XSUPPORTED
  if (XMDC_MEDCON) XMdcBeginProgressBar("Reading Analyze:");
#endif

  if (MDC_VERBOSE) MdcPrntMesg("ANLZ Reading <%s> ...",fi->ifname);

  /* get endian of the file in MDC_FILE_ENDIAN */
  i=MdcCheckANLZ(fi); fseek(fp,0,SEEK_SET);
  if (i != MDC_FRMT_ANLZ) {
    if (MDC_FALLBACK_FRMT == MDC_FRMT_ANLZ) {
      /* set host endian to try analyze */
      MDC_FILE_ENDIAN = MDC_HOST_ENDIAN;
    }else{
      /* bail out */
      return("ANLZ Endian check failed");
    }
  }
 
  memset(&hk,0,MDC_ANLZ_HK_SIZE);
  memset(&imd,0,MDC_ANLZ_IMD_SIZE);
  memset(&dh,0,MDC_ANLZ_DH_SIZE);

  /* put some default we use */
  fi->reconstructed = MDC_YES;
  fi->acquisition_type = MDC_ACQUISITION_TOMO;

  dh.orient=(char)0xff;

  if (fread((char *)&hk,1,MDC_ANLZ_HK_SIZE,fp) != MDC_ANLZ_HK_SIZE) 
    return("ANLZ Bad read HeadKey struct");


  fi->endian=MDC_FILE_ENDIAN;

  MdcSWAP(hk.sizeof_hdr); MdcSWAP(hk.extents); MdcSWAP(hk.session_error);

  if (MDC_INFO) {
    printf("\nMDC_ANLZ_HEADER_KEY (%d bytes)\n",MDC_ANLZ_HK_SIZE);
    MdcPrintLine('-',MDC_HALF_LENGTH);
    printf("sizeof_hdr   : %d\n",hk.sizeof_hdr);
    printf("data_type    : "); MdcPrintStr(hk.data_type);
    printf("db_name      : "); MdcPrintStr(hk.db_name);
    printf("extents      : %d\n",hk.extents);
    printf("session_error: %hd\n",hk.session_error);
    printf("regular      : "); MdcPrintChar(hk.regular);
    printf("\n");
    printf("hkey_un0     : "); MdcPrintChar(hk.hkey_un0);
    printf("\n");
  }

  if (MDC_INFO) {
    printf("\nIMAGE_DIMENSION (%d bytes)\n",MDC_ANLZ_IMD_SIZE);
    MdcPrintLine('-',MDC_HALF_LENGTH);
  }
  if (fread((char *)&imd,1,MDC_ANLZ_IMD_SIZE,fp)!=MDC_ANLZ_IMD_SIZE) 
    return("ANLZ Bad read ImageDimensions struct");

  for (i=0; i<8; i++) { MdcSWAP(imd.dim[i]); MdcSWAP(imd.pixdim[i]); }
  for (i=0; i<7; i++) { MdcSWAP(imd.unused[i]); }
  for (i=0; i<6; i++) { MdcSWAP(imd.funused[i]); }
  
  MdcSWAP(imd.datatype);    MdcSWAP(imd.bitpix);       MdcSWAP(imd.dim_un0);
  MdcSWAP(imd.compressed);  MdcSWAP(imd.verified); 
  MdcSWAP(imd.glmax);       MdcSWAP(imd.glmin);

  if ((imd.dim[0] < 3) || (imd.dim[0] > 7))
    return("ANLZ Bad header information");

  if (MDC_INFO) {
    for (i=0; i<8; i++) 
       printf("dim[%d]       : %hd\n",i,imd.dim[i]);
    for (i=0; i<7; i++)
       printf("unused[%d]    : %hd\n",i,imd.unused[i]);
    printf("datatype     : ");
    switch (imd.datatype) {
          case MDC_ANLZ_DT_UNKNOWN      : printf("Unknown");              break;
          case MDC_ANLZ_DT_BINARY       : printf("Binary");               break;
          case MDC_ANLZ_DT_UNSIGNED_CHAR: printf("Unsigned character");   break;
          case MDC_ANLZ_DT_SIGNED_SHORT : printf("Signed short integer"); break;
          case MDC_ANLZ_DT_SIGNED_INT   : printf("Signed integer");       break;
          case MDC_ANLZ_DT_FLOAT        : printf("Floating point");       break;
          case MDC_ANLZ_DT_COMPLEX      : printf("Complex");              break;
          case MDC_ANLZ_DT_DOUBLE       : printf("Double precision");     break;
          case MDC_ANLZ_DT_RGB          : printf("Coloured RGB");         break;
          case MDC_ANLZ_DT_ALL          : printf("All");                  break;
          default: printf("Unknown");
    }
    printf("\n");
    printf("bitpix       : %hd\n",imd.bitpix);
    for (i=0; i<8; i++) {
       printf("pixdim[%d]    : %-6e ",i,imd.pixdim[i]);
       if ( i<4 ) printf("[mm]\n");
       else printf("[ms]\n");
    }
    for (i=0; i<6; i++) printf("funused[%d]   : %-6e\n",i,imd.funused[i]);

    printf("compressed   : %6f\n",imd.compressed);
    printf("verified     : %6f\n",imd.verified);
    printf("glmax        : %d\n",imd.glmax);
    printf("glmin        : %d\n",imd.glmin);

  }

  if (MDC_INFO) {
    printf("\nMDC_ANLZ_DATA_HISTORY (%d bytes)\n", MDC_ANLZ_DH_SIZE);
    MdcPrintLine('-',MDC_HALF_LENGTH);
  }
  if (fread((char *)&dh,1,MDC_ANLZ_DH_SIZE,fp) != MDC_ANLZ_DH_SIZE) {
#if XSUPPORTED
    if (XMDC_MEDCON) {
      XMdcDisplayWarn("ANLZ Truncated header");
    }else
#endif
    { 
    MdcPrntWarn("Reading <%s> - ANLZ Truncated header",fi->ifname);
    }
  }

  memcpy(&opt.origin_x,&dh.originator[0],2); MdcSWAP(opt.origin_x);
  memcpy(&opt.origin_y,&dh.originator[2],2); MdcSWAP(opt.origin_y);
  memcpy(&opt.origin_z,&dh.originator[4],2); MdcSWAP(opt.origin_z);
  
  
  MdcSWAP(dh.views);       MdcSWAP(dh.vols_added);   
  MdcSWAP(dh.start_field); MdcSWAP(dh.field_skip);
  MdcSWAP(dh.omax);        MdcSWAP(dh.omin);
  MdcSWAP(dh.smax);        MdcSWAP(dh.smin);

  if (MDC_INFO) { 
    printf("description    : "); MdcPrintStr(dh.descrip);
    printf("aux_file       : "); MdcPrintStr(dh.aux_file);
    printf("orient         : ");
    switch (dh.orient) {
      case MDC_ANLZ_TRANS_UNFLIPPED: printf("transverse unflipped");    break;
      case MDC_ANLZ_CORON_UNFLIPPED: printf("coronal unflipped");       break;
      case MDC_ANLZ_SAGIT_UNFLIPPED: printf("sagittal unflipped");      break;
      case MDC_ANLZ_TRANS_FLIPPED  : printf("transverse flipped");      break;
      case MDC_ANLZ_CORON_FLIPPED  : printf("coronal flipped");         break;
      case MDC_ANLZ_SAGIT_FLIPPED  : printf("sagittal flipped");        break;
     default: printf("Unknown");
    }
    printf("\n");
    printf("originator     : "); MdcPrintStr(dh.originator);
    printf("generated      : "); MdcPrintStr(dh.generated);
    printf("scannum        : "); MdcPrintStr(dh.scannum);
    printf("patient_id     : "); MdcPrintStr(dh.patient_id);
    printf("exp_date       : "); MdcPrintStr(dh.exp_date);
    printf("exp_time       : "); MdcPrintStr(dh.exp_time);
    printf("views          : %d\n",dh.views);
    printf("vols_added     : %d\n",dh.vols_added);
    printf("start_field    : %d\n",dh.start_field); 
    printf("omax           : %d\n",dh.omax);
    printf("omin           : %d\n",dh.omin);
    printf("smax           : %d\n",dh.smax);
    printf("smin           : %d\n",dh.smin);
  }


  if (MDC_INFO) {
    printf("\n");
    MdcPrintLine('=',MDC_FULL_LENGTH);
    printf("SPM - HEADER INTERPRETATION\n");
    MdcPrintLine('-',MDC_HALF_LENGTH);
    printf("image {x}   : %hd\n",imd.dim[1]);
    printf("image {y}   : %hd\n",imd.dim[2]);
    printf("image {z}   : %hd\n",imd.dim[3]);
    printf("voxel {x}   : %+e\n",imd.pixdim[1]);
    printf("voxel {y}   : %+e\n",imd.pixdim[2]);
    printf("voxel {z}   : %+e\n",imd.pixdim[3]);
    printf("scaling     : %+e\n",imd.funused[1]);
    printf("data type   : %hd\n",imd.datatype);
    printf("offset      : %+e\n",imd.funused[0]);
    printf("origin      : %hd  %hd  %hd\n",opt.origin_x
                                          ,opt.origin_y
                                          ,opt.origin_z);
    printf("description : "); MdcPrintStr(dh.descrip);
    MdcPrintLine('=',MDC_FULL_LENGTH);
  } 

  /* save the offset, valid for SPM / MRIcro Analyze files */
  opt.offset = imd.funused[0];

  /* update our FILEINFO structure */
  if (strlen(dh.patient_id)) {
    sprintf(fi->patient_id,"%.10s",dh.patient_id);
  }

  memcpy(fi->dim,imd.dim,sizeof(imd.dim));
  memcpy(fi->pixdim,imd.pixdim,sizeof(imd.pixdim));

  fi->mwidth = (Uint32) imd.dim[1]; fi->mheight = (Uint32) imd.dim[2];
  for ( fi->number=1, i=3; i<=imd.dim[0]; i++)
     fi->number*=imd.dim[i];

  if (fi->number == 0) return("ANLZ No valid images specified");

  fi->bits = imd.bitpix; 

  switch (imd.datatype) {
    case MDC_ANLZ_DT_BINARY       : fi->type=BIT1;    fi->bits=8;  break;
    case MDC_ANLZ_DT_UNSIGNED_CHAR: fi->type=BIT8_U;  fi->bits=8;  break;
    case MDC_ANLZ_DT_SIGNED_SHORT : fi->type=BIT16_S; fi->bits=16; break;
    case MDC_ANLZ_DT_SIGNED_INT   : fi->type=BIT32_S; fi->bits=32; break;
    case MDC_ANLZ_DT_FLOAT        : fi->type=FLT32;   fi->bits=32; break;
    case MDC_ANLZ_DT_COMPLEX      : 
        return("ANLZ Datatype `complex' unsupported");             break;
    case MDC_ANLZ_DT_DOUBLE       : fi->type=FLT64;   fi->bits=64; break;
    case MDC_ANLZ_DT_RGB          :
        return("ANLZ Datatype `RGB' unsupported");                 break;
    case MDC_ANLZ_DT_ALL          :
        return("ANLZ Datatype `All' unsupported");                 break;
   default : 
    switch (fi->bits) {
      case  1: fi->type=BIT1;    break;
      case  8: fi->type=BIT8_U;  break;
      case 16: fi->type=BIT16_S; break;
      case 32: fi->type=BIT32_S; break; /* could be FLT32 as well */
      default: 
#if XSUPPORTED
         if (XMDC_MEDCON) {
           XMdcDisplayErr("ANLZ Unknown datatype");
         }else
#endif
         {
         MdcPrntWarn("ANLZ Unknown datatype");
         }
    }
  }


  /* preserve original path */
  MdcMergePath(fi->ipath,fi->idir,fi->ifname);
  if ((origpath=malloc(strlen(fi->ipath) + 1)) == NULL)
    return("ANLZ Couldn't allocate original path");

  strcpy(origpath,fi->ipath);
  MdcSplitPath(fi->ipath,fi->idir,fi->ifname);


  /* read the image file */
  MdcFileClose(fi->ifp);
  MdcMergePath(fi->ipath,fi->idir,fi->ifname);
  MdcNewExt(fi->ipath,NULL,"img");

  /* check for compressed image file */
  if (MdcFileExists(fi->ipath) == MDC_NO) {
    MdcAddCompressionExt(fi->compression,fi->ipath);
    if (MdcDecompressFile(fi->ipath) != MDC_OK) {
      MdcFree(origpath); return("ANLZ Decompression image file failed");
    }
    WAS_COMPRESSED = MDC_YES;
  }
 
  if ( (fi->ifp=fopen(fi->ipath,"rb")) == NULL ) {
    MdcFree(origpath); return("ANLZ Couldn't open image file");
  }

  if (WAS_COMPRESSED == MDC_YES) {
    unlink(fi->ipath);
#if XSUPPORTED
    if (XMDC_MEDCON) XMdcBeginProgressBar("Reading Analyze:");
#endif
  }

  MdcSplitPath(fi->ipath,fi->idir,fi->ifname);

  if (MDC_ANLZ_SPM == MDC_YES) { 
    /* interpret offset value from the header but we keep */
    /* our precautions ... badly initialized headers      */
    Uint32 offsetu = (Uint32)opt.offset;
    if ((float)offsetu == opt.offset) fseek(fi->ifp,offsetu,SEEK_SET);
  }

  if (!MdcGetStructID(fi)) {
    MdcFree(origpath); return("ANLZ Bad malloc IMG_DATA structs");
  }

  for ( i=fi->number; i>0; i--) { /* reversed planes */
     Uint32 nr, plane;

#if XSUPPORTED
     if (XMDC_MEDCON) {
       pvalue += 1./(float)fi->number;
       XMdcUpdateProgressBar(NULL);
     }
#endif
     if (MDC_ANLZ_REV) 
       nr = i-1;            /* reversed planes */
     else 
       nr = fi->number - i; /* normal   planes */

     plane = nr % fi->dim[3];

     id = &fi->image[nr];
  
     id->width = fi->mwidth;
     id->height = fi->mheight;
     id->bits  = fi->bits;
     id->type  = fi->type;
     if (MDC_ANLZ_SPM) {
       /* consider the scaling factor */
       if (imd.funused[1] > 0.0 ) id->quant_scale = imd.funused[1];
     }
     if (fi->pixdim[0] == 3.0 ) {  
       id->pixel_xsize = fi->pixdim[1];
       id->pixel_ysize = fi->pixdim[2]; 
       id->slice_width = fi->pixdim[3];
     }else if (fi->pixdim[0] == 4.0 ) {
       id->pixel_xsize = fi->pixdim[1];
       id->pixel_ysize = fi->pixdim[2];
       id->slice_width = fi->pixdim[3];
       id->frame_duration = fi->pixdim[4];
     }else if ( (fi->pixdim[1] > 0.0) &&  
                (fi->pixdim[2] > 0.0) && 
                (fi->pixdim[3] > 0.0) ) { /* we will try it anyway          */
                                          /* some don't fill in pixdim[0]   */
                                          /* for example PMOD (11-Apr-2000) */
       id->pixel_xsize = fi->pixdim[1];
       id->pixel_ysize = fi->pixdim[2];
       id->slice_width = fi->pixdim[3];
       fi->pixdim[0] = 3.0;
     }else {
       id->pixel_xsize = 1.0;
       id->pixel_ysize = 1.0;
       id->slice_width = 1.0;
     }
   
     /* attempt to fill in the Acr/Nema variables */
     switch (dh.orient) {
      /* flipped, what's the meaning of flipped here ? */
      case MDC_ANLZ_TRANS_UNFLIPPED: 
          id->pat_slice_orient=MDC_SUPINE_HEADFIRST_TRANSVERSAL; break;
      case MDC_ANLZ_CORON_UNFLIPPED: 
          id->pat_slice_orient=MDC_SUPINE_HEADFIRST_CORONAL;     break;
      case MDC_ANLZ_SAGIT_UNFLIPPED: 
          id->pat_slice_orient=MDC_SUPINE_HEADFIRST_SAGITTAL;    break;
      case MDC_ANLZ_TRANS_FLIPPED: 
          id->pat_slice_orient=MDC_SUPINE_HEADFIRST_TRANSVERSAL; break;
      case MDC_ANLZ_CORON_FLIPPED: 
          id->pat_slice_orient=MDC_SUPINE_HEADFIRST_CORONAL;     break;
      case MDC_ANLZ_SAGIT_FLIPPED: 
          id->pat_slice_orient=MDC_SUPINE_HEADFIRST_SAGITTAL;    break;
     }
     id->slice_spacing = id->slice_width;
     strcpy(id->pat_pos,MdcGetStrPatientPos(id->pat_slice_orient));
     strcpy(id->pat_orient,MdcGetStrPatientOrient(id->pat_slice_orient));
     MdcFillImgPos(fi,nr,plane,0.0);
     MdcFillImgOrient(fi,nr);
 
     bytes = MdcPixels2Bytes(fi->mwidth*fi->mheight*fi->bits);
     if ( (id->buf=MdcGetImgBuffer(bytes)) == NULL ) {
       MdcFree(origpath); return("ANLZ Bad malloc image buffer");
     }

     if (MDC_ANLZ_REV) { /* bottom/top  & right/left */
       for (d = id->height; d > 0; d--) { 
          pixoffset = (d-1) * id->width * MdcType2Bytes(id->type);
          for (w = id->width; w > 0; w--) {
             pbuf = (id->buf + pixoffset + ((w-1)*MdcType2Bytes(id->type)));   
             if ( fread(pbuf,MdcType2Bytes(id->type),1,fi->ifp)  != 1 ) {
               int j, n=1;
               if (nr > 1) { /* shift images to begin */
                 for (j=nr; j<fi->number; j++, n++) {
                    memcpy(&fi->image[n],&fi->image[j],sizeof(IMG_DATA));
                 }
               }
               err=MdcHandleTruncated(fi,n,MDC_YES);
               if (err != NULL) { MdcFree(origpath); return(err); }
               break;
             }

             if (fi->truncated) break;
          }

          if (fi->truncated) break;
       }

       if (fi->truncated) break; 

     }else{ /* normal */

       if (fread(id->buf,1,bytes,fi->ifp) != bytes ) {
         err=MdcHandleTruncated(fi, nr+1,MDC_YES);
         if (err != NULL) { MdcFree(origpath); return(err); }
       }

       if (fi->truncated) break;
     }

  }

  MdcFileClose(fi->ifp);

  /* check some finale FILEINFO entries */
  if (fi->dim[4] > 1) fi->acquisition_type = MDC_ACQUISITION_DYNAMIC;

  /* restore original filename */
  strcpy(fi->ipath,origpath);
  MdcSplitPath(fi->ipath,fi->idir,fi->ifname);
  MdcFree(origpath);

  if (fi->truncated) return("ANLZ Truncated image file");

  return NULL;
}

int MdcWriteHeaderKey(FILEINFO *fi)
{
  MDC_ANLZ_HEADER_KEY hk;
  char *p = NULL;
  
  memset(&hk,0,MDC_ANLZ_HK_SIZE);

  hk.sizeof_hdr = MDC_ANLZ_HK_SIZE + MDC_ANLZ_IMD_SIZE + MDC_ANLZ_DH_SIZE;
  sprintf(hk.data_type,"dsr");
  MdcSplitPath(fi->opath,fi->odir,fi->ofname);
  p = strrchr(fi->ofname,'.');
  if (p != NULL) *p = '\0'; /* remove extension */
  sprintf(hk.db_name,"%.18s",fi->ofname);
  if (p != NULL) *p = '.';  /* add    extension */
  MdcMergePath(fi->opath,fi->odir,fi->ofname);
  hk.extents=16384;
  hk.session_error=0;
  hk.regular='r';

  MdcSWAP(hk.sizeof_hdr); MdcSWAP(hk.extents); MdcSWAP(hk.session_error);

  fwrite((char *)&hk,1,MDC_ANLZ_HK_SIZE,fi->ofp);

  if (ferror(fi->ofp)) return(MDC_NO);


  return(MDC_YES);

}

int MdcWriteImageDimension(FILEINFO *fi, MDC_SPMOPT *opt)
{
  MDC_ANLZ_IMAGE_DIMS imd;
  int i;

  memset(&imd,0,MDC_ANLZ_IMD_SIZE);

  for (i=0; i <= fi->dim[0]; i++) imd.dim[i] = fi->dim[i]; 
  for (i=0; i <= fi->pixdim[0]; i++) imd.pixdim[i] = fi->pixdim[i];

#ifdef MDC_USE_SLICE_SPACING
  if (fi->number > 1) imd.pixdim[3] = fi->image[0].slice_spacing;
#endif

  imd.dim[1] = (Int16) fi->mwidth;
  imd.dim[2] = (Int16) fi->mheight;

  if (MDC_FORCE_INT != MDC_NO) {
    switch (MDC_FORCE_INT) {
     case BIT8_U : imd.datatype = MDC_ANLZ_DT_UNSIGNED_CHAR;
                   imd.bitpix   = 8;
                   break;
     case BIT16_S: imd.datatype = MDC_ANLZ_DT_SIGNED_SHORT;
                   imd.bitpix   = 16;
                   break;
     default     : imd.datatype = MDC_ANLZ_DT_SIGNED_SHORT;
                   imd.bitpix   = 16;
    }
  }else if (!(MDC_QUANTIFY || MDC_CALIBRATE)) {
    if ( fi->diff_type )  {  
      imd.datatype = MDC_ANLZ_DT_SIGNED_SHORT; 
      imd.bitpix   = 16;
    }else{ 
        switch ( fi->type ) { 
          case  BIT8_U:
          case  BIT8_S: imd.datatype = MDC_ANLZ_DT_UNSIGNED_CHAR; 
                        imd.bitpix   = 8; 
                        break;
          case BIT16_U:
          case BIT16_S: imd.datatype = MDC_ANLZ_DT_SIGNED_SHORT; 
                        imd.bitpix   = 16; 
                        break;
#ifdef HAVE_8BYTE_INT
          case BIT64_U:
          case BIT64_S:
#endif
          case BIT32_U:
          case BIT32_S: imd.datatype = MDC_ANLZ_DT_SIGNED_INT; 
                        imd.bitpix   = 32; 
                        break;
          case   FLT32: imd.datatype = MDC_ANLZ_DT_FLOAT;  
                        imd.bitpix   = 32; 
                        break;
          case   FLT64: imd.datatype = MDC_ANLZ_DT_DOUBLE; 
                        imd.bitpix   = 64;
                        break;
        }
    }
 }else{ 
  if (MDC_ANLZ_SPM) { /* BIT16_S with scaling factor! */
     imd.datatype = MDC_ANLZ_DT_SIGNED_SHORT; 
     imd.bitpix   = 16;
  }else{
     imd.datatype = MDC_ANLZ_DT_FLOAT; 
     imd.bitpix   = 32; 
  }
 }

 if (MDC_FORCE_INT != MDC_NO) {
   switch (MDC_FORCE_INT) { 
    case BIT8_U : imd.glmin = 0; imd.glmax = MDC_MAX_BIT8_U;  break;
    case BIT16_S: imd.glmin = 0; imd.glmax = MDC_MAX_BIT16_S; break;
    default     : imd.glmin = 0; imd.glmax = MDC_MAX_BIT16_S;
   }
 }else if (!(MDC_QUANTIFY || MDC_CALIBRATE)) {
   imd.glmin = (Int32) fi->glmin;
   imd.glmax = (Int32) fi->glmax;
 }else{
   if (MDC_ANLZ_SPM) {
     imd.glmin = 0; 
     imd.glmax = MDC_MAX_BIT16_S;
   }else{  /* a little useless to cast a float value */
     imd.glmin = (Int32) fi->qglmin;
     imd.glmax = (Int32) fi->qglmax;
   }
 }

 /* thinking about SPM */
 if (imd.pixdim[0] <= 0.0  || imd.pixdim[0] >= 8. ) {
     imd.pixdim[0]=3.; 
     imd.pixdim[1]=1.;
     imd.pixdim[2]=1.;
     imd.pixdim[3]=1.;
 }

 if (opt != NULL) imd.funused[0] = opt->offset;

 if (MDC_ANLZ_SPM) { /* the scaling factor */
   if (fi->image[0].rescaled) 
   imd.funused[1]=(float)fi->image[0].rescaled_fctr; 
   /* did rescale over all images -> all images same factor */
 }else{   
   imd.funused[1]=1.; 
 }

 /* swap the data if necessary */ 
 for(i=0; i<8; i++) { MdcSWAP(imd.dim[i]); MdcSWAP(imd.pixdim[i]); }
 for(i=0; i<7; i++) { MdcSWAP(imd.unused[i]); }
 for(i=0; i<6; i++) { MdcSWAP(imd.funused[i]); }
 MdcSWAP(imd.datatype);   MdcSWAP(imd.bitpix);  
 MdcSWAP(imd.compressed); MdcSWAP(imd.verified);
 MdcSWAP(imd.glmax);      MdcSWAP(imd.glmin);

 fwrite((char *)&imd,1,MDC_ANLZ_IMD_SIZE,fi->ofp);

 if (ferror(fi->ofp)) return(MDC_NO);

 return(MDC_YES);

}

int MdcWriteDataHistory(FILEINFO *fi, MDC_SPMOPT *opt)
{
  MDC_ANLZ_DATA_HIST dh;

  memset(&dh,0,MDC_ANLZ_DH_SIZE);

  sprintf(dh.descrip,"%.35s",fi->study_descr);
  sprintf(dh.scannum,"%.10s",fi->study_name);
  sprintf(dh.patient_id,"%.10s",fi->patient_id);
  sprintf(dh.generated,"%.10s",MDC_PRGR);

  switch (fi->image[0].pat_slice_orient) {
   case MDC_SUPINE_HEADFIRST_TRANSVERSAL: 
   case MDC_PRONE_HEADFIRST_TRANSVERSAL :
   case MDC_SUPINE_FEETFIRST_TRANSVERSAL:
   case MDC_PRONE_FEETFIRST_TRANSVERSAL : dh.orient = MDC_ANLZ_TRANS_UNFLIPPED;
                                          break;
   case MDC_SUPINE_HEADFIRST_CORONAL    :
   case MDC_PRONE_HEADFIRST_CORONAL     :
   case MDC_SUPINE_FEETFIRST_CORONAL    :
   case MDC_PRONE_FEETFIRST_CORONAL     : dh.orient = MDC_ANLZ_CORON_UNFLIPPED;
                                          break;

   case MDC_SUPINE_HEADFIRST_SAGITTAL   :
   case MDC_PRONE_HEADFIRST_SAGITTAL    :
   case MDC_SUPINE_FEETFIRST_SAGITTAL   :
   case MDC_PRONE_FEETFIRST_SAGITTAL    : dh.orient = MDC_ANLZ_SAGIT_UNFLIPPED;
                                          break;
  }

  if (opt != NULL) {
    MdcSWAP(opt->origin_x); memcpy(&dh.originator[0],&opt->origin_x,2);
    MdcSWAP(opt->origin_y); memcpy(&dh.originator[2],&opt->origin_y,2);
    MdcSWAP(opt->origin_z); memcpy(&dh.originator[4],&opt->origin_z,2);
  }

  fwrite((char *)&dh,1,MDC_ANLZ_DH_SIZE,fi->ofp);

  if (ferror(fi->ofp)) return(MDC_NO);

  return(MDC_YES);

}

char *MdcWriteImagesData(FILEINFO *fi)
{
  double pval;
  Uint32 i, d, w, FREE;
  Uint32 size, n, nr;
  Uint16 type;
  Uint8 *p, *buf, *pbuf, *maxbuf;
  Int8 saved_norm_over_frames=MDC_NORM_OVER_FRAMES;
  IMG_DATA *id;

  for (i=fi->number; i>0; i-- ) { 
#if XSUPPORTED
     if (XMDC_MEDCON) {
       pvalue += 1./(float)fi->number;
       XMdcUpdateProgressBar(NULL);
     }
#endif

     if (MDC_ANLZ_REV) 
       nr = i-1;             /* reversed planes */
     else
       nr = fi->number - i;  /* normal   planes */

     id = &fi->image[nr];

     buf = id->buf;  FREE = MDC_NO;
     type = id->type;

     if (MDC_FORCE_INT != MDC_NO) {
       if (MDC_ANLZ_SPM) MDC_NORM_OVER_FRAMES = MDC_NO;
       switch (MDC_FORCE_INT) {
        case BIT8_U : buf = MdcGetImgBIT8_U(fi,nr);
                      type=BIT8_U;  FREE=MDC_YES;
                      break;
        case BIT16_S: buf = MdcGetImgBIT16_S(fi,nr);
                      type=BIT16_S; FREE=MDC_YES;
                      break;
        default     : buf = MdcGetImgBIT16_S(fi,nr);
                      type=BIT16_S; FREE=MDC_YES;
       }
       if (MDC_ANLZ_SPM) MDC_NORM_OVER_FRAMES = saved_norm_over_frames;
      
     }else if (!(MDC_QUANTIFY || MDC_CALIBRATE)) { 
       if ( fi->diff_type ) {
         switch (id->type) {
           case BIT16_S: buf = id->buf; 
                          type = BIT16_S; FREE=MDC_NO; 
                          break;
           default     : buf = MdcGetImgBIT16_S(fi,nr);
                          type = BIT16_S; FREE=MDC_YES;
                          break;
         }
       }else switch (id->type) {
           case  BIT8_S: buf = MdcGetImgBIT8_U(fi,nr); 
                         type=BIT8_U ;   FREE=MDC_YES;
                         break;
           case BIT16_U: buf = MdcGetImgBIT16_S(fi,nr);
                         type=BIT16_S;   FREE=MDC_YES;
                         break;
           case BIT32_U: buf = MdcGetImgBIT32_S(fi,nr); 
                         type=BIT32_S;   FREE=MDC_YES;
                         break;
           case BIT64_S:
           case BIT64_U: buf = MdcGetImgBIT32_S(fi,nr);
                         type=BIT32_S;   FREE=MDC_YES;
                         break;
       }
     }else{
          if (MDC_ANLZ_SPM) {
            /* using the global scale factor <=> normalize over ALL images! */
            /* so all images have the same scale factor                     */
            MDC_NORM_OVER_FRAMES=MDC_NO;
            buf = MdcGetImgBIT16_S(fi,nr);
            type = BIT16_S;  FREE=MDC_YES;
            MDC_NORM_OVER_FRAMES=saved_norm_over_frames;
          }else{
            buf = MdcGetImgFLT32(fi,nr); 
            type=FLT32;    FREE=MDC_YES;
          }
     } 

     if (buf == NULL) return("ANLZ Bad malloc image buffer");


     if (fi->diff_size) {

       maxbuf = MdcGetResizedImage(fi, buf, type, nr);

       if (maxbuf == NULL) return("ANLZ Bad malloc maxbuf");

       if (FREE) MdcFree(buf);

       FREE = MDC_YES;

     }else{

       maxbuf = buf;

     }
    
     if (MDC_ANLZ_REV) {  /* bottom/top & right/left with uniform sizes */
       for (d=fi->mheight ; d > 0; d--) {
          pbuf= maxbuf + ((d-1) * fi->mwidth * MdcType2Bytes(type));
          /* write line */
          for (w=fi->mwidth; w > 0; w--) {
             p = (Uint8 *)&pbuf[(w-1)*MdcType2Bytes(type)];
             pval=MdcGetDoublePixel(p,type);
             if (!MdcWriteDoublePixel(pval,type,fi->ofp))
               return("ANLZ Bad write image pixel");
          } 
       }
     }else{ /* normal with uniform sizes */

       size = fi->mwidth * fi->mheight * MdcType2Bytes(type);
 
       for (n=0; n < size; n += MdcType2Bytes(type)) {
          pval = MdcGetDoublePixel((Uint8 *)&maxbuf[n],type);
          if (!MdcWriteDoublePixel(pval,type,fi->ofp))
            return("ANLZ Bad write image pixel");
       }
     }

     if (FREE) MdcFree(maxbuf);

     if (ferror(fi->ofp)) return("ANLZ Bad writing of images");

  }

  return NULL;

}

void MdcGetSpmOpt(FILEINFO *fi, MDC_SPMOPT *opt)
{
  opt->origin_x = 0;
  opt->origin_y = 0;
  opt->origin_z = 0;
  opt->offset   = 0.;

  MdcPrintLine('-',MDC_FULL_LENGTH);
  printf("\tSPM  OPTIONS\t\tORIG FILE: %s\n",fi->ifname);
  MdcPrintLine('-',MDC_FULL_LENGTH);
  printf("\n\tThe origin values must be an Int16 value");
  printf("\n\tThere is NO check performed on the input!\n");
  printf("\n\tOrigin X [%d]? ",opt->origin_x);
  if (!MdcPutDefault(mdcbufr)) opt->origin_x = (Int16)atoi(mdcbufr);
  printf("\n\tOrigin Y [%d]? ",opt->origin_y);
  if (!MdcPutDefault(mdcbufr)) opt->origin_y = (Int16)atoi(mdcbufr);
  printf("\n\tOrigin Z [%d]? ",opt->origin_z);
  if (!MdcPutDefault(mdcbufr)) opt->origin_z = (Int16)atoi(mdcbufr);
  /* MARK:  skip asking about offset                                    */
  /* printf("\n\tOffset   [%+e]? ",opt->offset);                        */
  /* if (!MdcPutDefault(mdcbufr)) opt->offset   = (float)atof(mdcbufr); */
  printf("\n");
  MdcPrintLine('-',MDC_FULL_LENGTH);
}



char *MdcWriteANLZ(FILEINFO *fi)
{
  MDC_SPMOPT opt;
  char *msg, *tmpfname = mdcbufr;
  
  MDC_FILE_ENDIAN = MDC_WRITE_ENDIAN;

  /* user wanted to supply some parameters */
  if (MDC_ANLZ_OPTIONS == MDC_YES && XMDC_MEDCON == MDC_NO) {
    MdcGetSpmOpt(fi,&opt);
  }else {
    opt.origin_x = 0;
    opt.origin_y = 0;
    opt.origin_z = 0;
    opt.offset   = 0.;
  }

  /* header and image separate, rescaled stuff very important */
  /* so we will write the images first ! */

  /* checking filename length */
  /* We don't truncate anymore (SPM, PMOD etc don't rely on db_name[18]) */

#ifdef ANLZ_SHORT_FILENAME 
/*********** filename truncation: ENABLED */

#if XSUPPORTED
  if (XMDC_MEDCON) {
    char *pext=NULL;
    /* here we get the full path name (dir+prefix+fname+extension!!) */
    MdcSplitPath(fi->opath,fi->odir,fi->ofname);
    pext = (char *)strrchr(fi->ofname,'.');
    if (pext != NULL) *pext = '\0';  /* temporarily remove extension */
    if (strlen(fi->ofname) > 17 ) {
      XMdcDisplayWarn("ANLZ Filename too big ... truncated!");
      fi->ofname[17] = '\0';         /* prefix was already added     */
      if (pext != NULL) {            /* add the extension back on    */
        *pext = '.';
        strcat(fi->ofname,pext);     /* filename was shortened       */
      }
    }else{
      if (pext != NULL) *pext = '.'; /* file was not shortened ...   */
    }                                /* strcat could cause problem if*/
                                     /* '.' was on place 18 ;-)      */

    MdcMergePath(fi->opath,fi->odir,fi->ofname);
    strcpy(tmpfname,fi->opath);

  }else{
#endif
    strcpy(tmpfname,fi->ifname); 
    if (strlen(tmpfname) > 17 ) {
      MdcPrntWarn("ANLZ Filename too big ... truncated!");
      tmpfname[13] = '\0';
    }
#if XSUPPORTED
  }
#endif

#else 
/********** filename truncation: DISABLED */
  if (XMDC_MEDCON) {
    strcpy(tmpfname,fi->opath);
  }else
    strcpy(tmpfname,fi->ifname);

/******************************************/
#endif /* ANLZ_SHORT_FILENAME */

  /* printing message; create MedCon filename */
#if XSUPPORTED
  if (XMDC_MEDCON) {
    XMdcBeginProgressBar("Writing Analyze:");
  }else 
#endif
    {
    MdcDefaultName(MDC_FRMT_ANLZ,fi->ofname,tmpfname);
    }
  if (MDC_VERBOSE) MdcPrntMesg("ANLZ Writing <%s> & <.img> ...",fi->ofname);

  /* writing images */
  if (XMDC_MEDCON) {
    fi->ofname[0]='\0'; MdcNewExt(fi->ofname,tmpfname,"img");
  }else{
    MdcNewName(fi->ofname,tmpfname,"img");
  }
  if (MdcFileExists(fi->ofname)) 
    return("Image file exists!!");
  if ( (fi->ofp=fopen(fi->ofname,"wb")) == NULL )
    return ("ANLZ Couldn't open image file");

  msg = MdcWriteImagesData(fi);
  if (msg != NULL) return(msg);

  MdcFileClose(fi->ofp);

  /* writing header with rescaled stuff */
  if (XMDC_MEDCON) {
    strcpy(fi->ofname,tmpfname);
  }else{
    MdcDefaultName(MDC_FRMT_ANLZ,fi->ofname,tmpfname);
  }
  if (MdcFileExists(fi->ofname))
    return("Header file exists!!");
  if ( (fi->ofp=fopen(fi->ofname,"wb")) == NULL )
    return("ANLZ Couldn't open header file");

  if ( !MdcWriteHeaderKey(fi) ) 
    return("ANLZ Bad write HeaderKey struct");

  if ( !MdcWriteImageDimension(fi, &opt) ) 
    return("ANLZ Bad write ImageDimension struct");

  if ( !MdcWriteDataHistory(fi, &opt) )
    return("ANLZ Bad write DataHistory struct");

  MdcFileClose(fi->ofp);

  return(NULL);

}
