/* ======================================================================== */
/* =============================== COPYRIGHT ============================== */
/* ======================================================================== */
/*

G65C816 CPU Emulator V0.91

Copyright (c) 2000 Karl Stenerud
All rights reserved.

Permission is granted to use this source code for non-commercial purposes.
To use this code for commercial purposes, you must get permission from the
author (Karl Stenerud) at karl@higashiyama-unet.ocn.ne.jp.


*/
/* ======================================================================== */
/* ================================= NOTES ================================ */
/* ======================================================================== */
/*

Changes:
	0.92 (2000-05-28):
			Lee Hammerton <lee-hammerton@hammerhead.ltd.uk>
			- Fixed debugger bug that caused D to be misrepresented.
			- Fixed MVN and MVP (they were reversed)

	0.91 (2000-05-22):
			Lee Hammerton <lee-hammerton@hammerhead.ltd.uk>
			- Fixed reset vector fetch to be little endian
			- Fixed disassembler call bug
			- Fixed C flag in SBC (should be inverted before operation)
			- Fixed JSR to stack PC-1 and RTS to pull PC and add 1

			Karl Stenerud <karl@higashiyama-unet.ocn.ne.jp>
			- Added correct timing for absolute indexed operations
			- SBC: fixed corruption of interim values

	0.90 (2000-05-17):
			Karl Stenerud <karl@higashiyama-unet.ocn.ne.jp>
			- first public release


Note on timings:
	- For instructions that write to memory (ASL, ASR, LSL, ROL, ROR, DEC,
	  INC, STA, STZ), the absolute indexed addressing mode takes 1 extra
	  cycle to complete.
	- The spec says fc (JMP axi) is 6 cyles, but elsewhere says 8 cycles
	  (which is what it should be)


TODO general:
	- Fix the SBC table.  I think there's a bug in there somewhere...

	- WAI will not stop if RDY is held high.

	- RDY internally held low when WAI executed and returned to hi when RES,
	  ABORT, NMI, or IRQ asserted.

	- ABORT will terminate WAI instruction but wil not restart the processor

	- If interrupt occurs after ABORT of WAI, processor returns to WAI
	  instruction.

	- Add one cycle when indexing across page boundary and E=1 except for STA
	  and STZ instructions.

	- Add 1 cycle if branch is taken. In Emulation (E= 1 ) mode only --add 1
	  cycle if the branch is taken and crosses a page boundary.

	- Add 1 cycle in Emulation mode (E=1) for (dir),y; abs,x; and abs,y
	  addressing modes.

*/
/* ======================================================================== */
/* ================================= DATA ================================= */
/* ======================================================================== */

#include "g65816cm.h"

/* Our CPU structure */
g65816i_cpu_struct g65816i_cpu = {0};

int g65816_ICount = 0;

/* Temporary Variables */
uint g65816i_source;
uint g65816i_destination;

/* Layout of the registers in the MAME debugger */
static unsigned char g65816i_register_layout[] =
{
	G65816_PB, G65816_PC, G65816_S, G65816_DB, G65816_D, G65816_P, G65816_E, -1,
	G65816_A, G65816_X, G65816_Y, G65816_NMI_STATE, G65816_IRQ_STATE, 0
};

/* Layout of the MAME debugger windows x,y,w,h */
static unsigned char g65816i_window_layout[] = {
	25, 0,55, 2, /* register window (top, right rows) */
	 0, 0,24,22, /* disassembler window (left colums) */
	25, 3,55, 9, /* memory #1 window (right, upper middle) */
	25,13,55, 9, /* memory #2 window (right, lower middle) */
	 0,23,80, 1, /* command line window (bottom rows) */
};


uint g65816i_adc_tbl[1024] =
{
	0x000,0x001,0x002,0x003,0x004,0x005,0x006,0x007,0x008,0x009,0x000,0x001,0x002,0x003,0x004,0x005,
	0x010,0x011,0x012,0x013,0x014,0x015,0x016,0x017,0x018,0x019,0x010,0x011,0x012,0x013,0x014,0x015,
	0x020,0x021,0x022,0x023,0x024,0x025,0x026,0x027,0x028,0x029,0x020,0x021,0x022,0x023,0x024,0x025,
	0x030,0x031,0x032,0x033,0x034,0x035,0x036,0x037,0x038,0x039,0x030,0x031,0x032,0x033,0x034,0x035,
	0x040,0x041,0x042,0x043,0x044,0x045,0x046,0x047,0x048,0x049,0x040,0x041,0x042,0x043,0x044,0x045,
	0x050,0x051,0x052,0x053,0x054,0x055,0x056,0x057,0x058,0x059,0x050,0x051,0x052,0x053,0x054,0x055,
	0x060,0x061,0x062,0x063,0x064,0x065,0x066,0x067,0x068,0x069,0x060,0x061,0x062,0x063,0x064,0x065,
	0x070,0x071,0x072,0x073,0x074,0x075,0x076,0x077,0x078,0x079,0x070,0x071,0x072,0x073,0x074,0x075,
	0x080,0x081,0x082,0x083,0x084,0x085,0x086,0x087,0x088,0x089,0x080,0x081,0x082,0x083,0x084,0x085,
	0x090,0x091,0x092,0x093,0x094,0x095,0x096,0x097,0x098,0x099,0x090,0x091,0x092,0x093,0x094,0x095,
	0x100,0x101,0x102,0x103,0x104,0x105,0x106,0x107,0x108,0x109,0x100,0x101,0x102,0x103,0x104,0x105,
	0x110,0x111,0x112,0x113,0x114,0x115,0x116,0x117,0x118,0x119,0x110,0x111,0x112,0x113,0x114,0x115,
	0x120,0x121,0x122,0x123,0x124,0x125,0x126,0x127,0x128,0x129,0x120,0x121,0x122,0x123,0x124,0x125,
	0x130,0x131,0x132,0x133,0x134,0x135,0x136,0x137,0x138,0x139,0x130,0x131,0x132,0x133,0x134,0x135,
	0x140,0x141,0x142,0x143,0x144,0x145,0x146,0x147,0x148,0x149,0x140,0x141,0x142,0x143,0x144,0x145,
	0x150,0x151,0x152,0x153,0x154,0x155,0x156,0x157,0x158,0x159,0x150,0x151,0x152,0x153,0x154,0x155,
	0x160,0x161,0x162,0x163,0x164,0x165,0x166,0x167,0x168,0x169,0x160,0x161,0x162,0x163,0x164,0x165,
	0x170,0x171,0x172,0x173,0x174,0x175,0x176,0x177,0x178,0x179,0x170,0x171,0x172,0x173,0x174,0x175,
	0x180,0x181,0x182,0x183,0x184,0x185,0x186,0x187,0x188,0x189,0x180,0x181,0x182,0x183,0x184,0x185,
	0x190,0x191,0x192,0x193,0x194,0x195,0x196,0x197,0x198,0x199,0x190,0x191,0x192,0x193,0x194,0x195,
	0x1a0,0x1a1,0x1a2,0x1a3,0x1a4,0x1a5,0x1a6,0x1a7,0x1a8,0x1a9,0x1a0,0x1a1,0x1a2,0x1a3,0x1a4,0x1a5,
	0x1b0,0x1b1,0x1b2,0x1b3,0x1b4,0x1b5,0x1b6,0x1b7,0x1b8,0x1b9,0x1b0,0x1b1,0x1b2,0x1b3,0x1b4,0x1b5,
	0x1c0,0x1c1,0x1c2,0x1c3,0x1c4,0x1c5,0x1c6,0x1c7,0x1c8,0x1c9,0x1c0,0x1c1,0x1c2,0x1c3,0x1c4,0x1c5,
	0x1d0,0x1d1,0x1d2,0x1d3,0x1d4,0x1d5,0x1d6,0x1d7,0x1d8,0x1d9,0x1d0,0x1d1,0x1d2,0x1d3,0x1d4,0x1d5,
	0x1e0,0x1e1,0x1e2,0x1e3,0x1e4,0x1e5,0x1e6,0x1e7,0x1e8,0x1e9,0x1e0,0x1e1,0x1e2,0x1e3,0x1e4,0x1e5,
	0x1f0,0x1f1,0x1f2,0x1f3,0x1f4,0x1f5,0x1f6,0x1f7,0x1f8,0x1f9,0x1f0,0x1f1,0x1f2,0x1f3,0x1f4,0x1f5,
	0x100,0x101,0x102,0x103,0x104,0x105,0x106,0x107,0x108,0x109,0x100,0x101,0x102,0x103,0x104,0x105,
	0x110,0x111,0x112,0x113,0x114,0x115,0x116,0x117,0x118,0x119,0x110,0x111,0x112,0x113,0x114,0x115,
	0x120,0x121,0x122,0x123,0x124,0x125,0x126,0x127,0x128,0x129,0x120,0x121,0x122,0x123,0x124,0x125,
	0x130,0x131,0x132,0x133,0x134,0x135,0x136,0x137,0x138,0x139,0x130,0x131,0x132,0x133,0x134,0x135,
	0x140,0x141,0x142,0x143,0x144,0x145,0x146,0x147,0x148,0x149,0x140,0x141,0x142,0x143,0x144,0x145,
	0x150,0x151,0x152,0x153,0x154,0x155,0x156,0x157,0x158,0x159,0x150,0x151,0x152,0x153,0x154,0x155,
	0x006,0x007,0x008,0x009,0x00a,0x00b,0x00c,0x00d,0x00e,0x00f,0x000,0x001,0x002,0x003,0x004,0x005,
	0x016,0x017,0x018,0x019,0x01a,0x01b,0x01c,0x01d,0x01e,0x01f,0x010,0x011,0x012,0x013,0x014,0x015,
	0x026,0x027,0x028,0x029,0x02a,0x02b,0x02c,0x02d,0x02e,0x02f,0x020,0x021,0x022,0x023,0x024,0x025,
	0x036,0x037,0x038,0x039,0x03a,0x03b,0x03c,0x03d,0x03e,0x03f,0x030,0x031,0x032,0x033,0x034,0x035,
	0x046,0x047,0x048,0x049,0x04a,0x04b,0x04c,0x04d,0x04e,0x04f,0x040,0x041,0x042,0x043,0x044,0x045,
	0x056,0x057,0x058,0x059,0x05a,0x05b,0x05c,0x05d,0x05e,0x05f,0x050,0x051,0x052,0x053,0x054,0x055,
	0x066,0x067,0x068,0x069,0x06a,0x06b,0x06c,0x06d,0x06e,0x06f,0x060,0x061,0x062,0x063,0x064,0x065,
	0x076,0x077,0x078,0x079,0x07a,0x07b,0x07c,0x07d,0x07e,0x07f,0x070,0x071,0x072,0x073,0x074,0x075,
	0x086,0x087,0x088,0x089,0x08a,0x08b,0x08c,0x08d,0x08e,0x08f,0x080,0x081,0x082,0x083,0x084,0x085,
	0x096,0x097,0x098,0x099,0x09a,0x09b,0x09c,0x09d,0x09e,0x09f,0x090,0x091,0x092,0x093,0x094,0x095,
	0x106,0x107,0x108,0x109,0x10a,0x10b,0x10c,0x10d,0x10e,0x10f,0x100,0x101,0x102,0x103,0x104,0x105,
	0x116,0x117,0x118,0x119,0x11a,0x11b,0x11c,0x11d,0x11e,0x11f,0x110,0x111,0x112,0x113,0x114,0x115,
	0x126,0x127,0x128,0x129,0x12a,0x12b,0x12c,0x12d,0x12e,0x12f,0x120,0x121,0x122,0x123,0x124,0x125,
	0x136,0x137,0x138,0x139,0x13a,0x13b,0x13c,0x13d,0x13e,0x13f,0x130,0x131,0x132,0x133,0x134,0x135,
	0x146,0x147,0x148,0x149,0x14a,0x14b,0x14c,0x14d,0x14e,0x14f,0x140,0x141,0x142,0x143,0x144,0x145,
	0x156,0x157,0x158,0x159,0x15a,0x15b,0x15c,0x15d,0x15e,0x15f,0x150,0x151,0x152,0x153,0x154,0x155,
	0x166,0x167,0x168,0x169,0x16a,0x16b,0x16c,0x16d,0x16e,0x16f,0x160,0x161,0x162,0x163,0x164,0x165,
	0x176,0x177,0x178,0x179,0x17a,0x17b,0x17c,0x17d,0x17e,0x17f,0x170,0x171,0x172,0x173,0x174,0x175,
	0x186,0x187,0x188,0x189,0x18a,0x18b,0x18c,0x18d,0x18e,0x18f,0x180,0x181,0x182,0x183,0x184,0x185,
	0x196,0x197,0x198,0x199,0x19a,0x19b,0x19c,0x19d,0x19e,0x19f,0x190,0x191,0x192,0x193,0x194,0x195,
	0x1a6,0x1a7,0x1a8,0x1a9,0x1aa,0x1ab,0x1ac,0x1ad,0x1ae,0x1af,0x1a0,0x1a1,0x1a2,0x1a3,0x1a4,0x1a5,
	0x1b6,0x1b7,0x1b8,0x1b9,0x1ba,0x1bb,0x1bc,0x1bd,0x1be,0x1bf,0x1b0,0x1b1,0x1b2,0x1b3,0x1b4,0x1b5,
	0x1c6,0x1c7,0x1c8,0x1c9,0x1ca,0x1cb,0x1cc,0x1cd,0x1ce,0x1cf,0x1c0,0x1c1,0x1c2,0x1c3,0x1c4,0x1c5,
	0x1d6,0x1d7,0x1d8,0x1d9,0x1da,0x1db,0x1dc,0x1dd,0x1de,0x1df,0x1d0,0x1d1,0x1d2,0x1d3,0x1d4,0x1d5,
	0x1e6,0x1e7,0x1e8,0x1e9,0x1ea,0x1eb,0x1ec,0x1ed,0x1ee,0x1ef,0x1e0,0x1e1,0x1e2,0x1e3,0x1e4,0x1e5,
	0x1f6,0x1f7,0x1f8,0x1f9,0x1fa,0x1fb,0x1fc,0x1fd,0x1fe,0x1ff,0x1f0,0x1f1,0x1f2,0x1f3,0x1f4,0x1f5,
	0x106,0x107,0x108,0x109,0x10a,0x10b,0x10c,0x10d,0x10e,0x10f,0x100,0x101,0x102,0x103,0x104,0x105,
	0x116,0x117,0x118,0x119,0x11a,0x11b,0x11c,0x11d,0x11e,0x11f,0x110,0x111,0x112,0x113,0x114,0x115,
	0x126,0x127,0x128,0x129,0x12a,0x12b,0x12c,0x12d,0x12e,0x12f,0x120,0x121,0x122,0x123,0x124,0x125,
	0x136,0x137,0x138,0x139,0x13a,0x13b,0x13c,0x13d,0x13e,0x13f,0x130,0x131,0x132,0x133,0x134,0x135,
	0x146,0x147,0x148,0x149,0x14a,0x14b,0x14c,0x14d,0x14e,0x14f,0x140,0x141,0x142,0x143,0x144,0x145,
	0x156,0x157,0x158,0x159,0x15a,0x15b,0x15c,0x15d,0x15e,0x15f,0x150,0x151,0x152,0x153,0x154,0x155
};

uint g65816i_sbc_tbl[1024] =
{
	0x000,0x101,0x002,0x103,0x004,0x105,0x006,0x107,0x008,0x109,0x00a,0x10b,0x00c,0x10d,0x00e,0x10f,
	0x010,0x111,0x012,0x113,0x014,0x115,0x016,0x117,0x018,0x119,0x01a,0x11b,0x01c,0x11d,0x01e,0x11f,
	0x020,0x121,0x022,0x123,0x024,0x125,0x026,0x127,0x028,0x129,0x02a,0x12b,0x02c,0x12d,0x02e,0x12f,
	0x030,0x131,0x032,0x133,0x034,0x135,0x036,0x137,0x038,0x139,0x03a,0x13b,0x03c,0x13d,0x03e,0x13f,
	0x040,0x141,0x042,0x143,0x044,0x145,0x046,0x147,0x048,0x149,0x04a,0x14b,0x04c,0x14d,0x04e,0x14f,
	0x050,0x151,0x052,0x153,0x054,0x155,0x056,0x157,0x058,0x159,0x05a,0x15b,0x05c,0x15d,0x05e,0x15f,
	0x060,0x161,0x062,0x163,0x064,0x165,0x066,0x167,0x068,0x169,0x06a,0x16b,0x06c,0x16d,0x06e,0x16f,
	0x070,0x171,0x072,0x173,0x074,0x175,0x076,0x177,0x078,0x179,0x07a,0x17b,0x07c,0x17d,0x07e,0x17f,
	0x080,0x181,0x082,0x183,0x084,0x185,0x086,0x187,0x088,0x189,0x08a,0x18b,0x08c,0x18d,0x08e,0x18f,
	0x090,0x191,0x092,0x193,0x094,0x195,0x096,0x197,0x098,0x199,0x09a,0x19b,0x09c,0x19d,0x09e,0x19f,
	0x0a0,0x1a1,0x0a2,0x1a3,0x0a4,0x1a5,0x0a6,0x1a7,0x0a8,0x1a9,0x0aa,0x1ab,0x0ac,0x1ad,0x0ae,0x1af,
	0x0b0,0x1b1,0x0b2,0x1b3,0x0b4,0x1b5,0x0b6,0x1b7,0x0b8,0x1b9,0x0ba,0x1bb,0x0bc,0x1bd,0x0be,0x1bf,
	0x0c0,0x1c1,0x0c2,0x1c3,0x0c4,0x1c5,0x0c6,0x1c7,0x0c8,0x1c9,0x0ca,0x1cb,0x0cc,0x1cd,0x0ce,0x1cf,
	0x0d0,0x1d1,0x0d2,0x1d3,0x0d4,0x1d5,0x0d6,0x1d7,0x0d8,0x1d9,0x0da,0x1db,0x0dc,0x1dd,0x0de,0x1df,
	0x0e0,0x1e1,0x0e2,0x1e3,0x0e4,0x1e5,0x0e6,0x1e7,0x0e8,0x1e9,0x0ea,0x1eb,0x0ec,0x1ed,0x0ee,0x1ef,
	0x0f0,0x1f1,0x0f2,0x1f3,0x0f4,0x1f5,0x0f6,0x1f7,0x0f8,0x1f9,0x0fa,0x1fb,0x0fc,0x1fd,0x0fe,0x1ff,
	0x0a0,0x1a1,0x0a2,0x1a3,0x0a4,0x1a5,0x0a6,0x1a7,0x0a8,0x1a9,0x0aa,0x1ab,0x0ac,0x1ad,0x0ae,0x1af,
	0x0b0,0x1b1,0x0b2,0x1b3,0x0b4,0x1b5,0x0b6,0x1b7,0x0b8,0x1b9,0x0ba,0x1bb,0x0bc,0x1bd,0x0be,0x1bf,
	0x0c0,0x1c1,0x0c2,0x1c3,0x0c4,0x1c5,0x0c6,0x1c7,0x0c8,0x1c9,0x0ca,0x1cb,0x0cc,0x1cd,0x0ce,0x1cf,
	0x0d0,0x1d1,0x0d2,0x1d3,0x0d4,0x1d5,0x0d6,0x1d7,0x0d8,0x1d9,0x0da,0x1db,0x0dc,0x1dd,0x0de,0x1df,
	0x0e0,0x1e1,0x0e2,0x1e3,0x0e4,0x1e5,0x0e6,0x1e7,0x0e8,0x1e9,0x0ea,0x1eb,0x0ec,0x1ed,0x0ee,0x1ef,
	0x0f0,0x1f1,0x0f2,0x1f3,0x0f4,0x1f5,0x0f6,0x1f7,0x0f8,0x1f9,0x0fa,0x1fb,0x0fc,0x1fd,0x0fe,0x1ff,
	0x000,0x101,0x002,0x103,0x004,0x105,0x006,0x107,0x008,0x109,0x00a,0x10b,0x00c,0x10d,0x00e,0x10f,
	0x010,0x111,0x012,0x113,0x014,0x115,0x016,0x117,0x018,0x119,0x01a,0x11b,0x01c,0x11d,0x01e,0x11f,
	0x020,0x121,0x022,0x123,0x024,0x125,0x026,0x127,0x028,0x129,0x02a,0x12b,0x02c,0x12d,0x02e,0x12f,
	0x030,0x131,0x032,0x133,0x034,0x135,0x036,0x137,0x038,0x139,0x03a,0x13b,0x03c,0x13d,0x03e,0x13f,
	0x040,0x141,0x042,0x143,0x044,0x145,0x046,0x147,0x048,0x149,0x04a,0x14b,0x04c,0x14d,0x04e,0x14f,
	0x050,0x151,0x052,0x153,0x054,0x155,0x056,0x157,0x058,0x159,0x05a,0x15b,0x05c,0x15d,0x05e,0x15f,
	0x060,0x161,0x062,0x163,0x064,0x165,0x066,0x167,0x068,0x169,0x06a,0x16b,0x06c,0x16d,0x06e,0x16f,
	0x070,0x171,0x072,0x173,0x074,0x175,0x076,0x177,0x078,0x179,0x07a,0x17b,0x07c,0x17d,0x07e,0x17f,
	0x080,0x181,0x082,0x183,0x084,0x185,0x086,0x187,0x088,0x189,0x08a,0x18b,0x08c,0x18d,0x08e,0x18f,
	0x090,0x191,0x092,0x193,0x094,0x195,0x096,0x197,0x098,0x199,0x09a,0x19b,0x09c,0x19d,0x09e,0x19f,
	0x01a,0x11b,0x01c,0x11d,0x01e,0x11f,0x0b0,0x1b1,0x0b2,0x1b3,0x0b4,0x1b5,0x0b6,0x1b7,0x0b8,0x1b9,
	0x02a,0x12b,0x02c,0x12d,0x02e,0x12f,0x0c0,0x1c1,0x0c2,0x1c3,0x0c4,0x1c5,0x0c6,0x1c7,0x0c8,0x1c9,
	0x03a,0x13b,0x03c,0x13d,0x03e,0x13f,0x0d0,0x1d1,0x0d2,0x1d3,0x0d4,0x1d5,0x0d6,0x1d7,0x0d8,0x1d9,
	0x04a,0x14b,0x04c,0x14d,0x04e,0x14f,0x0e0,0x1e1,0x0e2,0x1e3,0x0e4,0x1e5,0x0e6,0x1e7,0x0e8,0x1e9,
	0x05a,0x15b,0x05c,0x15d,0x05e,0x15f,0x0f0,0x1f1,0x0f2,0x1f3,0x0f4,0x1f5,0x0f6,0x1f7,0x0f8,0x1f9,
	0x06a,0x16b,0x06c,0x16d,0x06e,0x16f,0x000,0x101,0x002,0x103,0x004,0x105,0x006,0x107,0x008,0x109,
	0x07a,0x17b,0x07c,0x17d,0x07e,0x17f,0x010,0x111,0x012,0x113,0x014,0x115,0x016,0x117,0x018,0x119,
	0x08a,0x18b,0x08c,0x18d,0x08e,0x18f,0x020,0x121,0x022,0x123,0x024,0x125,0x026,0x127,0x028,0x129,
	0x09a,0x19b,0x09c,0x19d,0x09e,0x19f,0x030,0x131,0x032,0x133,0x034,0x135,0x036,0x137,0x038,0x139,
	0x0aa,0x1ab,0x0ac,0x1ad,0x0ae,0x1af,0x040,0x141,0x042,0x143,0x044,0x145,0x046,0x147,0x048,0x149,
	0x0ba,0x1bb,0x0bc,0x1bd,0x0be,0x1bf,0x050,0x151,0x052,0x153,0x054,0x155,0x056,0x157,0x058,0x159,
	0x0ca,0x1cb,0x0cc,0x1cd,0x0ce,0x1cf,0x060,0x161,0x062,0x163,0x064,0x165,0x066,0x167,0x068,0x169,
	0x0da,0x1db,0x0dc,0x1dd,0x0de,0x1df,0x070,0x171,0x072,0x173,0x074,0x175,0x076,0x177,0x078,0x179,
	0x0ea,0x1eb,0x0ec,0x1ed,0x0ee,0x1ef,0x080,0x181,0x082,0x183,0x084,0x185,0x086,0x187,0x088,0x189,
	0x0fa,0x1fb,0x0fc,0x1fd,0x0fe,0x1ff,0x090,0x191,0x092,0x193,0x094,0x195,0x096,0x197,0x098,0x199,
	0x0aa,0x1ab,0x0ac,0x1ad,0x0ae,0x1af,0x0a0,0x1a1,0x0a2,0x1a3,0x0a4,0x1a5,0x0a6,0x1a7,0x0a8,0x1a9,
	0x0ba,0x1bb,0x0bc,0x1bd,0x0be,0x1bf,0x010,0x111,0x012,0x113,0x014,0x115,0x016,0x117,0x018,0x119,
	0x0ca,0x1cb,0x0cc,0x1cd,0x0ce,0x1cf,0x020,0x121,0x022,0x123,0x024,0x125,0x026,0x127,0x028,0x129,
	0x0da,0x1db,0x0dc,0x1dd,0x0de,0x1df,0x030,0x131,0x032,0x133,0x034,0x135,0x036,0x137,0x038,0x139,
	0x0ea,0x1eb,0x0ec,0x1ed,0x0ee,0x1ef,0x040,0x141,0x042,0x143,0x044,0x145,0x046,0x147,0x048,0x149,
	0x0fa,0x1fb,0x0fc,0x1fd,0x0fe,0x1ff,0x050,0x151,0x052,0x153,0x054,0x155,0x056,0x157,0x058,0x159,
	0x00a,0x10b,0x00c,0x10d,0x00e,0x10f,0x060,0x161,0x062,0x163,0x064,0x165,0x066,0x167,0x068,0x169,
	0x01a,0x11b,0x01c,0x11d,0x01e,0x11f,0x070,0x171,0x072,0x173,0x074,0x175,0x076,0x177,0x078,0x179,
	0x02a,0x12b,0x02c,0x12d,0x02e,0x12f,0x080,0x181,0x082,0x183,0x084,0x185,0x086,0x187,0x088,0x189,
	0x03a,0x13b,0x03c,0x13d,0x03e,0x13f,0x090,0x191,0x092,0x193,0x094,0x195,0x096,0x197,0x098,0x199,
	0x04a,0x14b,0x04c,0x14d,0x04e,0x14f,0x0a0,0x1a1,0x0a2,0x1a3,0x0a4,0x1a5,0x0a6,0x1a7,0x0a8,0x1a9,
	0x05a,0x15b,0x05c,0x15d,0x05e,0x15f,0x0b0,0x1b1,0x0b2,0x1b3,0x0b4,0x1b5,0x0b6,0x1b7,0x0b8,0x1b9,
	0x06a,0x16b,0x06c,0x16d,0x06e,0x16f,0x0c0,0x1c1,0x0c2,0x1c3,0x0c4,0x1c5,0x0c6,0x1c7,0x0c8,0x1c9,
	0x07a,0x17b,0x07c,0x17d,0x07e,0x17f,0x0d0,0x1d1,0x0d2,0x1d3,0x0d4,0x1d5,0x0d6,0x1d7,0x0d8,0x1d9,
	0x08a,0x18b,0x08c,0x18d,0x08e,0x18f,0x0e0,0x1e1,0x0e2,0x1e3,0x0e4,0x1e5,0x0e6,0x1e7,0x0e8,0x1e9,
	0x09a,0x19b,0x09c,0x19d,0x09e,0x19f,0x0f0,0x1f1,0x0f2,0x1f3,0x0f4,0x1f5,0x0f6,0x1f7,0x0f8,0x1f9,
	0x00a,0x10b,0x00c,0x10d,0x00e,0x10f,0x0a0,0x1a1,0x0a2,0x1a3,0x0a4,0x1a5,0x0a6,0x1a7,0x0a8,0x1a9
};


extern void (*g65816i_opcodes_M0X0[])(void);
extern uint g65816i_get_reg_M0X0(int regnum);
extern void g65816i_set_reg_M0X0(int regnum, uint val);
extern void g65816i_set_line_M0X0(int line, int state);
extern int  g65816i_execute_M0X0(int cycles);

extern void (*g65816i_opcodes_M0X1[])(void);
extern uint g65816i_get_reg_M0X1(int regnum);
extern void g65816i_set_reg_M0X1(int regnum, uint val);
extern void g65816i_set_line_M0X1(int line, int state);
extern int  g65816i_execute_M0X1(int cycles);

extern void (*g65816i_opcodes_M1X0[])(void);
extern uint g65816i_get_reg_M1X0(int regnum);
extern void g65816i_set_reg_M1X0(int regnum, uint val);
extern void g65816i_set_line_M1X0(int line, int state);
extern int  g65816i_execute_M1X0(int cycles);

extern void (*g65816i_opcodes_M1X1[])(void);
extern uint g65816i_get_reg_M1X1(int regnum);
extern void g65816i_set_reg_M1X1(int regnum, uint val);
extern void g65816i_set_line_M1X1(int line, int state);
extern int  g65816i_execute_M1X1(int cycles);

extern void (*g65816i_opcodes_E[])(void);
extern uint g65816i_get_reg_E(int regnum);
extern void g65816i_set_reg_E(int regnum, uint val);
extern void g65816i_set_line_E(int line, int state);
extern int  g65816i_execute_E(int cycles);

void (**g65816i_opcodes[5])(void) =
{
	g65816i_opcodes_M0X0,
	g65816i_opcodes_M0X1,
	g65816i_opcodes_M1X0,
	g65816i_opcodes_M1X1,
	g65816i_opcodes_E
};

uint (*g65816i_get_reg[5])(int regnum) =
{
	g65816i_get_reg_M0X0,
	g65816i_get_reg_M0X1,
	g65816i_get_reg_M1X0,
	g65816i_get_reg_M1X1,
	g65816i_get_reg_E
};

void (*g65816i_set_reg[5])(int regnum, uint val) =
{
	g65816i_set_reg_M0X0,
	g65816i_set_reg_M0X1,
	g65816i_set_reg_M1X0,
	g65816i_set_reg_M1X1,
	g65816i_set_reg_E
};

void (*g65816i_set_line[5])(int line, int state) =
{
	g65816i_set_line_M0X0,
	g65816i_set_line_M0X1,
	g65816i_set_line_M1X0,
	g65816i_set_line_M1X1,
	g65816i_set_line_E
};

int (*g65816i_execute[5])(int cycles) =
{
	g65816i_execute_M0X0,
	g65816i_execute_M0X1,
	g65816i_execute_M1X0,
	g65816i_execute_M1X1,
	g65816i_execute_E
};

/* ======================================================================== */
/* ================================= API ================================== */
/* ======================================================================== */


void g65816_reset(void* param)
{
		/* Start the CPU */
		CPU_STOPPED = 0;

		/* Put into emulation mode */
		REG_D = 0;
		REG_PB = 0;
		REG_DB = 0;
		REG_S = (REG_S & 0xff) | 0x100;
		REG_X &= 0xff;
		REG_Y &= 0xff;
		if(!FLAG_M)
		{
			REG_B = REG_A & 0xff00;
			REG_A &= 0xff;
		}
		FLAG_E = EFLAG_SET;
		FLAG_M = MFLAG_SET;
		FLAG_X = XFLAG_SET;

		/* Clear D and set I */
		FLAG_D = DFLAG_CLEAR;
		FLAG_I = IFLAG_SET;

		/* Clear all pending interrupts (should we really do this?) */
		LINE_IRQ = 0;
		LINE_NMI = 0;
		IRQ_DELAY = 0;

		/* Set the function tables to emulation mode */
		g65816i_set_execution_mode(EXECUTION_MODE_E);

		/* 6502 expects these, but its not in the 65816 spec */
		FLAG_Z = ZFLAG_CLEAR;
		REG_S = 0x1ff;

		/* Fetch the reset vector */
		REG_PC = g65816_read_8(VECTOR_RESET) | (g65816_read_8(VECTOR_RESET+1)<<8);
		g65816i_jumping(REG_PB | REG_PC);
}

/* Exit and clean up */
void g65816_exit(void)
{
	/* nothing to do yet */
}

/* Execute some instructions */
int g65816_execute(int cycles)
{
	return FTABLE_EXECUTE(cycles);
}


/* Get the current CPU context */
unsigned g65816_get_context(void *dst_context)
{
	if(dst_context)
		*(g65816i_cpu_struct*)dst_context = g65816i_cpu;
	return sizeof(g65816i_cpu);
}

/* Set the current CPU context */
void g65816_set_context(void *src_context)
{
	if(src_context)
	{
		g65816i_cpu = *(g65816i_cpu_struct*)src_context;
		g65816i_jumping(REG_PB | REG_PC);
	}
}

/* Get the current Program Counter */
unsigned g65816_get_pc(void)
{
	return REG_PC;
}

/* Set the Program Counter */
void g65816_set_pc(unsigned val)
{
	REG_PC = MAKE_UINT_16(val);
	g65816_jumping(REG_PB | REG_PC);
}

/* Get the current Stack Pointer */
unsigned g65816_get_sp(void)
{
	return REG_S;
}

/* Set the Stack Pointer */
void g65816_set_sp(unsigned val)
{
	REG_S = FLAG_E ? MAKE_UINT_8(val) | 0x100 : MAKE_UINT_16(val);
}

/* Get a register */
unsigned g65816_get_reg(int regnum)
{
	return FTABLE_GET_REG(regnum);
}

/* Set a register */
void g65816_set_reg(int regnum, unsigned value)
{
	FTABLE_SET_REG(regnum, value);
}

/* Load a CPU state */
void g65816_state_load(void *file)
{
}

/* Save the current CPU state */
void g65816_state_save(void *file)
{
}

/* Set the non-maskable interrupt line */
void g65816_set_nmi_line(int state)
{
	FTABLE_SET_LINE(G65816_LINE_NMI, state);
}

/* Set an interrupt line */
void g65816_set_irq_line(int line, int state)
{
	FTABLE_SET_LINE(line, state);
}

/* Set the callback that is called when servicing an interrupt */
void g65816_set_irq_callback(int (*callback)(int))
{
	INT_ACK = callback;
}


/* Get a formatted string representing a register and its contents */
const char *g65816_info(void *context, int regnum)
{
	static char buffer[16][47+1];
	static int which = 0;
	g65816i_cpu_struct* r = context;

	which = ++which % 16;
	buffer[which][0] = '\0';
	if(!context)
		r = &g65816i_cpu;

	switch(regnum)
	{
		case CPU_INFO_REG+G65816_PC:		sprintf(buffer[which], "PC:%04X", r->pc); break;
		case CPU_INFO_REG+G65816_PB:		sprintf(buffer[which], "PB:%02X", r->pb>>16); break;
		case CPU_INFO_REG+G65816_DB:		sprintf(buffer[which], "DB:%02X", r->db>>16); break;
		case CPU_INFO_REG+G65816_D:			sprintf(buffer[which], "D:%04X", r->d); break;
		case CPU_INFO_REG+G65816_S:			sprintf(buffer[which], "S:%04X", r->s); break;
		case CPU_INFO_REG+G65816_P:			sprintf(buffer[which], "P:%02X",
											 (r->flag_n&0x80)		|
											((r->flag_v>>1)&0x40)	|
											r->flag_m				|
											r->flag_x				|
											r->flag_d				|
											r->flag_i				|
											((!r->flag_z)<<1)		|
											((r->flag_c>>8)&1)); break;
		case CPU_INFO_REG+G65816_E:			sprintf(buffer[which], "E:%d", r->flag_e); break;
		case CPU_INFO_REG+G65816_A:			sprintf(buffer[which], "A:%04X", r->a | r->b); break;
		case CPU_INFO_REG+G65816_X:			sprintf(buffer[which], "X:%04X", r->x); break;
		case CPU_INFO_REG+G65816_Y:			sprintf(buffer[which], "Y:%04X", r->y); break;
		case CPU_INFO_REG+G65816_NMI_STATE:	sprintf(buffer[which], "NMI:%X", r->line_nmi); break;
		case CPU_INFO_REG+G65816_IRQ_STATE:	sprintf(buffer[which], "IRQ:%X", r->line_irq); break;
		case CPU_INFO_FLAGS:
			sprintf(buffer[which], "%c%c%c%c%c%c%c%c",
				r->flag_n & NFLAG_SET ? 'N':'.',
				r->flag_v & VFLAG_SET ? 'V':'.',
				r->flag_m & MFLAG_SET ? 'M':'.',
				r->flag_x & XFLAG_SET ? 'X':'.',
				r->flag_d & DFLAG_SET ? 'D':'.',
				r->flag_i & IFLAG_SET ? 'I':'.',
				r->flag_z == 0        ? 'Z':'.',
				r->flag_c & CFLAG_SET ? 'C':'.');
			break;
		case CPU_INFO_NAME: return "G65C816";
		case CPU_INFO_FAMILY: return "6500";
		case CPU_INFO_VERSION: return "0.90";
		case CPU_INFO_FILE: return __FILE__;
		case CPU_INFO_CREDITS: return "Copyright (c) 2000 Karl Stenerud, all rights reserved.";
		case CPU_INFO_REG_LAYOUT: return (const char*)g65816i_register_layout;
		case CPU_INFO_WIN_LAYOUT: return (const char*)g65816i_window_layout;
	}
	return buffer[which];
}


/* Disassemble an instruction */
#ifdef MAME_DEBUG
#include "g65816ds.h"
#endif
unsigned g65816_dasm(char *buffer, unsigned pc)
{
#ifdef MAME_DEBUG
	return g65816_disassemble(buffer, (pc&0xffff), REG_PB>>16, FLAG_M, FLAG_X);
#else
	sprintf(buffer, "$%02X", g65816_read_8_immediate(REG_PB | (pc&0xffff)));
	return 1;
#endif
}


void g65816_init(void){ return; }


/* ======================================================================== */
/* ============================== END OF FILE ============================= */
/* ======================================================================== */
