/*
 * XLife Copyright 1989 Jon Bennett jb7m+@andrew.cmu.edu, jcrb@cs.cmu.edu
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of the copyright holders not be used in
 * advertising or publicity pertaining to distribution of the software without
 * specific, written prior permission.  The copyright holders make no
 * representations about the suitability of this software for any purpose.  It
 * is provided "as is" without express or implied warranty.
 *
 * THE COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL THE COPYRIGHT HOLDERS BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
 * DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
 * TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 *
 * CMU SUCKS
 */

/*****************************************************************************

This module provides setscale(), paintcell(), drawbox(), boxpattern(),
post(), and redraw().  It implements drawing from the Xlife data
structures to an X window used as a canvas.  The idea is to separate
these things from the command processing, event handling, and window
management.

For this code to work, the variable lifew must be initialized to a
descriptor for a mapped X window, with the cellgc array filled with
valid graphics contexts.  The setscale() function must have been
called at least once before the first redraw().

The dispmesh flag controls whether the redraw code superimposes a
mesh over the canvas.  The wireframe flag controls whether tentative
patterns are displayed normally but with a bounding box (wireframe off)
or with no bounding box but the tentative-pattern cells as open
rectangles).

*****************************************************************************/

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>
#include <X11/Xatom.h>
#include <X11/cursorfont.h>
#include <X11/keysym.h>
#include <X11/keysymdef.h>

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <ctype.h>
#include <time.h>
#include <limits.h>

#include "defs.h"
#include "data.h"
#include "tile.h"

#undef GLOBAL
#define GLOBAL
#include "guidata.h"
#include "xcanvas.h"

/* mouse coordinates to cell coords (relative to top left of screen) */
#define MOUSE2CELL(n)	((n) & SCALEUP(0xffffffff))

void pan(int xoff, int yoff)
/* Pan the viewport */
{
    view_x += xoff;
    view_y += yoff;
}

void boxpattern(pattern *context, const int color)
/* Draw a bounding box in a given color around the live cells of a context */
{
    bounding_box(context);

    drawbox(context->box.min.x, context->box.min.y,
	    context->box.max.x, context->box.max.y,
	    color);
}

#define NO_BOX	-1
static rect current_box;
static int boxcolor = NO_BOX;

void drawbox(coord_t x1, coord_t y1, coord_t x2, coord_t y2, int color)
/* box the area defined by two corners (life-world coordinates) */
{
    if (scale > 1)
    {
	current_box.min.x = x1; current_box.max.x = x2; 
	current_box.min.y = y1; current_box.max.y = y2; 
	boxcolor = color;
    }
}

#ifdef DIRECT_DRAW
void drawlegend(coord_t x, coord_t y, char *incl)
/* draw a text legend on the screen */
{
    XDrawString(disp, lifew, btextgc,
		LIFE2MOUSE(xorigin + lx, view_x),
		LIFE2MOUSE(yorigin + ly, view_y),
		incl, strlen(incl));
}
#endif /* DIRECT_DRAW */

#define MAXCHANGE	8192
static cellcount_t chgpoints[MAXCOLORS], chgrects[MAXCOLORS];
static XPoint points[MAXCOLORS][MAXCHANGE];
static XRectangle rects[MAXCOLORS][MAXCHANGE];

static int changed_boxes = 0;

void paintcell(coord_t x, coord_t y, cell_t c)
/* change a single cell's visual state (user coordinates) */
{
    if (scale == 0)
	XDrawPoint(disp,lifew,cellgc[c],x,y);
    else if (scale > 0)
	XFillRectangle(disp,lifew,cellgc[c],
		       MOUSE2CELL(x),MOUSE2CELL(y),
		       rects[0][0].width, rects[0][0].height);
}

static void flush_update()
{
    int color;

    if (scale < 1)
    {
	for (color = 0; color < MAXCOLORS; color++)
	    if (chgpoints[color])
	    {
		XDrawPoints(disp,lifew,cellgc[color],
			    points[color],chgpoints[color],
			    CoordModeOrigin);
		chgpoints[color] = 0;
	    }
    }
    else
    {
	for (color = 0; color < MAXCOLORS; color++)
	    if (chgrects[color])
	    {
		if (color == LOAD_BOX)
		    XDrawRectangles(disp,lifew,cellgc[color],
				rects[color],chgrects[color]);
		else
		    XFillRectangles(disp,lifew,cellgc[color],
				    rects[color],chgrects[color]);
		chgrects[color] = 0;
	    }
    }
    XFlush(disp);
}

void post(cell_t color, coord_t x, coord_t y, int mode)
/* post a change in cell color to the X structure arrays */
{
    if (mode == POST_TENTATIVE && wireframe)
	color = LOAD_BOX;
    else if (displaymode == DISPLAY_CHANGED)
	color = 1;

    if (mode == POST_TENTATIVE)
    {
	x -= view_x;
	y -= view_y;
    }

    if (scale < 1)
    {
	points[color][chgpoints[color]].x = SCALEUP(x);
	points[color][chgpoints[color]].y = SCALEUP(y);
	chgpoints[color]++;
    }
    else
    {
	rects[color][chgrects[color]].x = SCALEUP(x);
	rects[color][chgrects[color]].y = SCALEUP(y);
	chgrects[color]++;
    }

    if (changed_boxes++ >= MAXCHANGE)
    {
	flush_update();
	changed_boxes = 0;
    }
}


static void redraw_box(rect argbox, int color)
/* draw specified box */
{
    if (boxcolor != NO_BOX)
    {
	int	gcind;
	int	small = SCALEUP(1);
	rect	box;

	box.min.x    = (argbox.min.x > argbox.max.x) ? argbox.max.x : argbox.min.x;
	box.min.y    = (argbox.min.y > argbox.max.y) ? argbox.max.y : argbox.min.y;
	box.max.x    = (argbox.min.x <= argbox.max.x) ? argbox.max.x : argbox.min.x;
	box.max.y    = (argbox.min.y <= argbox.max.y) ? argbox.max.y : argbox.min.y;

        box.min.x -= view_x; box.max.x -= view_x;
        box.min.y -= view_y; box.max.y -= view_y;
	box.max.x++; box.max.y++;
        box.min.x *= small; box.max.x *= small;
        box.min.y *= small; box.max.y *= small;

	/* left-hand vertical line (doubled to cope with pixel aspect ratio) */
	if (color == ERASE_BOX && dispmesh)
	    gcind = (box.min.x % grid_interval) ? MESH_SMALL : MESH_BIG;
	else
	    gcind = color;
	XDrawLine(disp,lifew,cellgc[gcind],
		  box.min.x - 1, box.min.y - 1,
		  box.min.x - 1, box.max.y - 1);
	XDrawLine(disp,lifew,cellgc[gcind],
		  box.min.x - 2, box.min.y - 1,
		  box.min.x - 2, box.max.y - 1);

	/* right-hand vertical line (doubled to cope with pixel aspect ratio)*/
	if (color == ERASE_BOX && dispmesh)
	    gcind = (box.max.x % grid_interval) ? MESH_SMALL : MESH_BIG;
	else
	    gcind = color;
	XDrawLine(disp,lifew,cellgc[gcind],
		  box.max.x - 1, box.min.y - 1,
		  box.max.x - 1, box.max.y - 1);
	XDrawLine(disp,lifew,cellgc[gcind],
		  box.max.x - 2, box.min.y - 1,
		  box.max.x - 2, box.max.y - 1);

	/* top horizontal line */
	if (color == ERASE_BOX && dispmesh)
	    gcind = (box.min.y % grid_interval) ? MESH_SMALL : MESH_BIG;
	else
	    gcind = color;
	XDrawLine(disp,lifew,cellgc[gcind],
		  box.min.x - 1, box.min.y - 1,
		  box.max.x - 1, box.min.y - 1);

	/* bottom horizontal line */
	if (color == ERASE_BOX && dispmesh)
	    gcind = (box.max.y % grid_interval) ? MESH_SMALL : MESH_BIG;
	else
	    gcind = color;
	XDrawLine(disp,lifew,cellgc[gcind],
		  box.min.x - 1, box.max.y - 1,
		  box.max.x - 1, box.max.y - 1);

	/* mark the box erased so we don't have to refresh it */
	if (color == ERASE_BOX)
	    color = NO_BOX;
    }
}

void erasebox(void)
{
    if (scale > 1)
	redraw_box(current_box, ERASE_BOX);
}

void redraw(const pattern *actp, const pattern *tentp, int force)
/* re-display the visible part of the board */
{
#define ONSCREEN(x, y)	(((x > view_x-BOXSIZE) && (x < (view_x+BOXSIZE+cx))) && ((y > view_y-BOXSIZE) && (y < (view_y+BOXSIZE+cy))))

    tile *ptr; 
    XWindowAttributes attrs;
    coord_t x,y, tentx, tenty, cx, cy;
    int gcind, small;

    memset(chgrects,  '\0', sizeof(chgrects));
    memset(chgpoints, '\0', sizeof(chgpoints));
    changed_boxes = 0;
    
    if (force || displaymode == DISPLAY_CHANGED) 
	XClearWindow(disp, lifew);

    /*
     * Canvas may have been resized, or the scale changed, since the
     * last redraw.  The other advantage of doing it this way is
     * that it narrows the canvas interface.
     */
    XGetWindowAttributes(disp, lifew, &attrs);
    cx = SCALEDOWN(attrs.width); cy = SCALEDOWN(attrs.height);

    /* post active-pattern changes */
    for (ptr = actp->tiles; ptr != NULL; ptr = ptr->next)
    {
	x = ptr->x; y = ptr->y;
	if (ONSCREEN(x, y))
	    displaybox(actp, x-view_x, y-view_y, &ptr->cells, force);
    }
    /* post tentative-pattern changes with appropriate transformation */
    for (ptr = tentp->tiles; ptr != NULL; ptr = ptr->next)
    {
	tentx = ptr->x; tenty = ptr->y;
	if (ONSCREEN(TX(tentp->at, tentx,tenty), TY(tentp->at, tentx,tenty)))
	    trdisplaybox(tentp, tentx, tenty, &ptr->cells);
    }
    flush_update();

    /*
     * Everything after this point is decoration, to be done 
     * at larger scales only.
     */
    if (scale <= 1)
    {
	XFlush(disp);
	return;
    }

    small = SCALEUP(1);

    /* we may need to redo the mesh */
    if (dispmesh && force)
    {
	for (x = 0; x <= cx; x++) 
	{
	    gcind = ((x + view_x) % grid_interval) ? MESH_SMALL : MESH_BIG;

	    /*
	     * This line is doubled in order to cope with the nearly 2:1
	     * aspect ratio of typical pixels.
	     */
	    XDrawLine(disp,lifew,cellgc[gcind], x*small-2,0,x*small-2,attrs.height);
	    XDrawLine(disp,lifew,cellgc[gcind], x*small-1,0,x*small-1,attrs.height);
	}
	for (y = 0; y <= cy; y++) 
	{
	    gcind = ((y + view_y) % grid_interval) ? MESH_SMALL : MESH_BIG;

	    XDrawLine(disp,lifew,cellgc[gcind], 0,y*small-1,attrs.width,y*small-1);
	}
    }

    if (!tentp->tiles)
	redraw_box(current_box, boxcolor);
    else
    {
	legend	*ap;

	/* draw pivot of any tentp points */
	int x = SCALEUP(tentp->at.xtrans - view_x);
	int y = SCALEUP(tentp->at.ytrans - view_y);
	int sc = SCALEUP(1) - 1;

	/* draw the pattern pivot */
	XDrawLine(disp,lifew,cellgc[LOAD_BOX], 
		  x+1,     y+1,
		  x+sc-1,  y+sc-1);
	XDrawLine(disp,lifew,cellgc[LOAD_BOX],
		  x+1,     y+sc-1,
		  x+sc-1,  y+1);

	if (!wireframe)
	{
	    current_box.min.x=TX(tentp->at,tentp->box.min.x,tentp->box.min.y);
	    current_box.min.y=TY(tentp->at,tentp->box.min.x,tentp->box.min.y);
	    current_box.max.x=TX(tentp->at,tentp->box.max.x,tentp->box.max.y);
	    current_box.max.y=TY(tentp->at,tentp->box.max.x,tentp->box.max.y);

	    redraw_box(current_box, boxcolor = LOAD_BOX);
	}

	/*
	 * We only display legends in the original orientation.
	 * Otherwise they're too likely to collide with pattern
	 * pieces that have moved.
	 */
	if (tentp->legends && tentp->at.rxx==1 && tentp->at.ryy==1)
	    for (ap = tentp->legends; ap; ap = ap->next)
	    {
		char	*sp, *tp;
		char	linefeeds;

		x = TX(tentp->at, tentp->box.min.x, tentp->box.min.y) + ap->x; 
		y = TY(tentp->at, tentp->box.min.x, tentp->box.min.y) + ap->y;

		linefeeds = 0;
		for (sp = ap->text; *sp; sp++)
		    if (*sp == '\n')
			linefeeds++;

		for (sp = tp = ap->text; *sp; sp = tp + 1)
		{
		    if ((tp = strchr(sp, '\n')) == (char *)NULL)
			tp = sp + strlen(sp);

		    XDrawString(disp, lifew, cellgc[1],
				LIFE2MOUSE(x, view_x),
				LIFE2MOUSE(y, view_y) - SCALEUP(1) - (linefeeds-1)*15 - 5,
				sp, tp - sp);
		    --linefeeds;
		    if (!*tp)
			break;
		}
	    }

    }

    XFlush(disp);
}

void setscale(const int sc)
/* preset scale for X rectangles describing the board */
{
    if (sc > 1)
    {
	int	i, j, size = SCALEUP(1) - 1;

	for (i = 0; i < MAXCOLORS; i++)
	    for (j = 0; j < MAXCHANGE; j++)
	    {
		rects[i][j].height = size;

		/*
		 * The "- (sc  > 2)" leaves room for doubled 
		 * vertical mesh lines.
		 */
		rects[i][j].width = size - (sc > 2);
	    }
    }
}

/* xcanvas.c ends here */
