/*
 * XLife Copyright 1989 Jon Bennett jb7m+@andrew.cmu.edu, jcrb@cs.cmu.edu
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of the copyright holders not be used in
 * advertising or publicity pertaining to distribution of the software without
 * specific, written prior permission.  The copyright holders make no
 * representations about the suitability of this software for any purpose.  It
 * is provided "as is" without express or implied warranty.
 *
 * THE COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL THE COPYRIGHT HOLDERS BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
 * DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
 * TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */

#define BOXSIZE         8	/* chunk size for sparse patterns */

#define MAXGENS		2	/* generations to save per box */
#define PRESENT		0
#define PAST		1

typedef union 
{
    struct twostate_t
    {
	u32bits hihalf[MAXGENS], lohalf[MAXGENS];
	u32bits on[BOXSIZE];
    }
    twostate;	/* 48 bytes (on a 32-bit machine) */
#if STATEBITS > 1
    struct nstate_t
    {
	cell_t	cell[MAXGENS][BOXSIZE][BOXSIZE];
    }
    nstate;	/* 128 bytes (assuming char-sized cells) */
#endif /* STATEBITS */
}
cellbox;

typedef struct tile_t
{
    coord_t x,y;
    int dead;
    struct tile_t *up, *dn, *lf, *rt, *fore, *next,*hfore,*hnext;

    /* MUST be last in the structure or our dynamic allocation will fail */
    cellbox cells;
}
tile;

typedef struct legend_t
{
    char		*text;
    coord_t		x, y;
    struct legend_t	*next;
}
legend;

/*
 * A `pattern' is a linked list of tiles with its own context globals.
 * The universe may contain multiple patterns; they are overlayed on
 * the screen, but can be manipulated independently and evolve independently.
 */
typedef struct
{
    char 		*pattern_name;		/* name of this pattern */
    char		*outcome;		/* outcome description */

#if STATEBITS > 1
    int			maxstates;		/* states per cell */
#endif /* STATEBITS > 1 */

    tile		*tiles;			/* head of the cell list */
    int			tilesize;		/* byte size of tile */
    cellcount_t		tilecount;		/* count of boxes */
    cellcount_t		cellcount;		/* count of cells */
    cellcount_t		changecount;		/* count of changes */
    lifetime_t		generations;		/* context generation number */

    tile		**hashlist;		/* hash list */

    int 		ncomments;		/* count of commants */
    char 		**comments;		/* dynamic array of comments */
    legend		*legends;

    rect		box;		/* bounding box of pattern */
    motion		at;		/* transform to apply to pattern */
    coord_t		obasex, obasey;	/* original load point */
}
pattern;

/* file.c */
GLOBAL void fileinit(void);
GLOBAL char *findfile(const char *file);
GLOBAL void freeloadscript(void);
GLOBAL void saveloadscript(pattern *context, 
			   const char *file, coord_t x, coord_t y);
GLOBAL void confirmload(pattern *from, pattern *to);
GLOBAL void loadfile(const char *file, pattern *context, coord_t x, coord_t y);
GLOBAL int use_rules(char *cp, pattern *context);
GLOBAL char *checktilda(const char *);
GLOBAL char *addlifeext(char *);

/* cell.c */
extern void displaybox(const pattern *pat,
		       coord_t x, coord_t y, cellbox *ptr, int force);
extern void trdisplaybox(const pattern *pat, 
			 coord_t x, coord_t y, cellbox *ptr);

/* tile.c */
extern int getcell(const pattern *pp, cellbox *ptr,
		   const int xdx, const int ydx, const gen);
extern void setcell(const pattern *pp, cellbox *ptr,
		    const int xdx, const int ydx, const gen, const int val);
extern void forgetcell(const pattern *pp, cellbox *ptr);
extern void changesize(pattern *pat, const int newsize);
extern void chgcell(pattern *context, coord_t x, coord_t y, int t, cell_t c);
extern int lookcell(const pattern *context, coord_t x, coord_t y, int t);
extern tile *fetchtile(pattern *context, coord_t x, coord_t y);
extern tile *maketile(pattern *context, coord_t x, coord_t y);
extern void vacuum(pattern *context);

/* pattern.c */
extern void initcells(pattern *context);
extern void resetload(pattern *context);
extern void moveload(pattern *context, const int newx, const int newy);
extern void flipload(pattern *context);
extern void turnload(pattern *context);
extern int transplant(pattern *from, pattern *to,
	    coord_t x1, coord_t y1, coord_t x2, coord_t y2);
extern void forget(pattern *);
extern void clear_pattern(pattern *);
extern void center(pattern *context, coord_t *px, coord_t *py);
extern void median(pattern *context, coord_t *px, coord_t *py);
extern cellcount_t bounding_box(pattern *context);
extern void randomize(pattern *context, 
		      coord_t lx, coord_t ly, coord_t dx, coord_t dy,
		      int density);
extern void addcomment(pattern *context, const char *comment);
extern void clearcomments(pattern *pat);
extern void drawlegend(pattern *context, coord_t x, coord_t y, char *msg);

/* save.c */
extern void saveinit(const char *patternfile);
extern void savepattern(FILE *ofp, pattern *pp, char mode);

/* generate.c */
extern void getcheckstats(pattern *context);
extern void generate(pattern *context);
extern void display_rules(char *ptr);
extern void modify_rules(const char *ptr, pattern *context);
extern char *readrules(const char *file, pattern *context);
extern char *parse_rule(char *buf);

/* utils.c */
extern void stamp(char *leader, FILE *ofp);
extern void fatal(const char *s);
extern void heapsort(coord_t *data, int n, int size);
extern void randomseed(unsigned long x2, unsigned long x1);
extern unsigned long random32(void);
