/*
 * XLife Copyright 1998 Eric S. Raymond <esr@thyrsus.com>
 *
 */
#include <stdio.h>
#include <limits.h>
#include <string.h>
#include "defs.h"
#include "data.h"
#include "tile.h"

/*
 * The only interface to the GUI is error();
 */

#undef COALESCE		/* this has bad effects on blob recognition */

#define DEFAULT_LIBRARY	"named-patterns"

#define MARK		'*'
#define SPACE		'.'

#define NO_MATCH	-1	/* must not be 0..8 */

#define FOR_CELLS(pp, pt, dx, dy)	for (pt=(pp)->tiles;pt;pt=pt->next) \
					    if (!pt->dead) \
					        for (dx=0;dx<BOXSIZE;dx++) \
					            for (dy=0;dy<BOXSIZE;dy++)

typedef struct {int xi; int yi;} offsets; 

typedef struct blobref_t
{
    int			xsize, ysize, number;
    dihedral		where;
    pattern		*pattern;
    pattern		*blob;
    struct blobref_t	*ref;
#ifdef COALESCE
    int			refcount;
#endif /* COALESCE */
}
blobref;

static char *matchlibfile;

void saveinit(const char *patternfile)
{
    char	*cp;

    if (!patternfile)
	patternfile = DEFAULT_LIBRARY;

    if ((cp = findfile(patternfile)))
	matchlibfile = strdup(cp);
    else
	error("Pattern library not found.");
}

static void flood_fill(pattern *from, coord_t x, coord_t y, pattern *to)
/* transplant all points in a blob touching (x,y) between contexts */
{
    static offsets neighbors[] =
    {
	{-1, -1},
	{-1,  0},
	{-1,  1},
	{ 0, -1},
	{ 0,  1},
	{ 1, -1},
	{ 1,  0},
	{ 1,  1},
#ifdef DEPTH2
	{-2, -2},
	{-2, -1},
	{-2,  0},
	{-2,  1},
	{-2,  2},
	{-1,  2},
	{ 0,  2},
	{ 1,  2},
	{ 2,  2},
	{ 2,  1},
	{ 2,  0},
	{ 2, -1},
	{ 2, -2},
	{ 1, -2},
	{ 0, -2},
	{-1, -2},
#endif /* DEPTH2 */
    };
    int i;

    /* skip empty cells */
    if (!lookcell(from, x, y, PRESENT))
	return;

    /* skip cells that are already colored */
    if (lookcell(to, x, y, PRESENT))
	return;

    /* transfer this cell from the copy to the new context */
    chgcell(to,   x, y, PRESENT, lookcell(from, x, y, PRESENT));
    chgcell(from, x, y, PRESENT, 0);

    for (i = 0; i < sizeof(neighbors)/sizeof(neighbors[0]); i++)
    {
	int nx = x + neighbors[i].xi;
	int ny = y + neighbors[i].yi;

	/* recursively transfer all its neighbors */
	flood_fill(from, nx, ny, to);
    }
}

static int refcmp(void *r1, void *r2)
/* sort blobs first by least Y coordinate, then by least X coordinate */
{
    pattern	*p1 = (pattern *)r1;
    pattern	*p2 = (pattern *)r2;

    int	cmp = (p1->box.min.y - p2->box.min.y);

    if (!cmp)
	cmp = (p1->box.min.x - p2->box.min.x);
    return(cmp);
}

static pattern *analyze(pattern *context)
/* analyze a pattern into an allocated list of blobs */
{
    cell_t	state;
    tile	*ptr;
    int		i, dx, dy, color = 0;
    pattern	analysand;
    pattern	*parts, *pp;

    memset(&analysand, '\0', sizeof(pattern));
    initcells(&analysand);
    changesize(&analysand, context->maxstates);

    /* copy the pattern */
    FOR_CELLS(context, ptr, dx, dy)
	if ((state = getcell(context, &ptr->cells, dx, dy, PRESENT)))
	    chgcell(&analysand, ptr->x + dx, ptr->y + dy, PRESENT, state);

    /*
     * This loop is deceptively simple-looking.  It relies on flood_fill()
     * not returning until it has transpanted the largest blob it can find
     * connected to a cell.
     */
    parts = (pattern *)calloc(sizeof(pattern), 1);
    FOR_CELLS(&analysand, ptr, dx, dy)
	if (getcell(&analysand, &ptr->cells, dx, dy, PRESENT))
	{
	    parts = (pattern *)realloc(parts, sizeof(pattern) * (color + 1));

	    memset(&parts[color], '\0', sizeof(pattern));
	    initcells(&parts[color]);
	    changesize(&parts[color], context->maxstates);

	    flood_fill(&analysand, ptr->x+dx, ptr->y+dy, &parts[color]);

	    color++;
	}

    /* create an extra, blank pattern entry as a fencepost */
    parts = (pattern *)realloc(parts, sizeof(pattern) * (color + 1));
    memset(&parts[color], '\0', sizeof(pattern));
    initcells(&parts[color]);

    /* this cleans up the blob order -- strictly cosmetic */
    for (pp = parts; pp->tiles; pp++)
	pp->cellcount = bounding_box(pp);
    qsort(parts, color, sizeof(pattern), refcmp);

    /* return the analyzed parts */
    return(parts);
}

static void savesimple(pattern *pp, int xoff, int yoff, FILE *ofp)
/* save a pattern as a simple picture block */
{
    coord_t x, y;
    cellcount_t cellcount = bounding_box(pp);

    if (cellcount == 0)
	return;

    if (!xoff && !yoff)
	fputs("#P\n", ofp);
    else
	fprintf(ofp, "#P %ld %ld\n", xoff, yoff);
    for (y = pp->box.min.y; y <= pp->box.max.y; y++)
    {
	for (x = pp->box.min.x; x <= pp->box.max.x; x++)
	{
	    int val;

	    if ((val = lookcell(pp, x, y, PRESENT)))
	    {
#if STATEBITS > 1
		if (pp->maxstates > 2)
		    (void) fputc(itos(val), ofp);
		else 
#endif
		    (void) fputc(MARK, ofp);
	    }
	    else
		(void) fputc(SPACE, ofp);
	}
	(void) fputc('\n', ofp);
    }
}

static void savecomplex(pattern *context, FILE *ofp)
/* do a pattern analysis and save the pattern as a blob sequence */
{
    pattern	*pp, *parts = analyze(context);
    coord_t	xcenter, ycenter;
    int		blobcount;

    if (parts == NULL)
    {
	error("Pattern analysis failed; too many blobs in picture.\n");
	return;
    }

    xcenter = (context->box.max.x+1+context->box.min.x)/2;
    ycenter = (context->box.max.y+1+context->box.min.y)/2;

    blobcount = 0;
    for (pp = parts; pp->tiles; pp++)
	blobcount++;
    if (blobcount > 1)
	fprintf(ofp, "# %d blobs\n", blobcount);
    for (pp = parts; pp->tiles; pp++)
	savesimple(pp, (int)(pp->box.min.x-xcenter), (int)(pp->box.min.y-ycenter), ofp);
}

static int transform(int flip, int rotation,
		     coord_t j, coord_t i,
		     coord_t xsize, coord_t ysize, 
		     coord_t *newx, coord_t *newy)
{ 
    switch (rotation + 4 * flip)
    {
    case 0:		/* no rotation, no flip */
	*newx = j;
	*newy = i;
	break;

    case 1:		/* one rotation, no flip */
	*newx = i;
	*newy = ysize - j;
	break;

    case 2:		/* two rotations, no flip */
	*newx = xsize - j;
	*newy = ysize - i;
	break;

    case 3:		/* three rotations, no flip */
	*newx = xsize - i;
	*newy = j;
	break;

    case 4:		/* no rotation, one flip */
	*newx = j;
	*newy = ysize - i;
	break;

    case 5:		/* one rotation, one flip */
	*newx = i;
	*newy = j;
	break;

    case 6:		/* two rotations, one flip */
	*newx = xsize - j;
	*newy = i;
	break;

    case 7:		/* three rotations, one flip */
	*newx = xsize - i;
	*newy = ysize - j;
	break;
    }
}

static int blob_equal(pattern *within, 
		      const blobref *tp, const blobref *sp,
		      dihedral *where)
/* check blob tp for equality with blob sp */
{
    static offsets xoffmagic[] =
	{{0, 0}, {0, 1}, {1, 0}, {0, 0}, {0, 0}, {0, 0}, {1, 0}, {0, 1}}; 
    static offsets yoffmagic[] =
	{{0, 0}, {0, 0}, {0, 1}, {1, 0}, {0, 1}, {0, 0}, {0, 0}, {1, 0}};
    int		i, j, rotation, flip;

    /* check for identical population */
    if (tp->blob->cellcount != sp->blob->cellcount)
	return(NO_MATCH);

    /* bounding rectangles must be compatible */
    if (!(
	(tp->xsize == sp->xsize && tp->ysize == sp->ysize)
	||
	(tp->xsize == sp->ysize && tp->ysize == sp->xsize)
	))
	return(NO_MATCH);

    for (rotation = 0; rotation < 4; rotation++)
	for (flip = 0; flip < 2; flip++)
	{
	    int	trans = rotation + 4*flip;
	    coord_t newx, newy, xbcorner, ybcorner;

#ifdef BLOBDEBUG
	    fprintf(stderr, "Flip %d, rotation %d\n", flip ? -1 : 1, rotation);
#endif /* BLOBDEBUG */

	    for (i = 0; i < sp->ysize; i++) 
		for (j = 0; j < sp->xsize; j++)
		{
		    transform(flip, rotation, 
			      j, i, tp->xsize-1, tp->ysize-1,
			      &newx, &newy);

#ifdef BLOBDEBUG
		    fprintf(stderr, "Left: (%ld, %ld)  Right (%ld, %ld)\n",
			   sp->xmin + j,    sp->ymin + i, 
			   tp->xmin + newx, tp->ymin + newy);
#endif /* BLOBDEBUG */
		    if (lookcell(sp->blob, sp->blob->box.min.x+j, sp->blob->box.min.y+i, PRESENT) 
			!= lookcell(tp->blob, tp->blob->box.min.x + newx, tp->blob->box.min.y + newy, PRESENT))
			goto nomatch;
		}

	    /* OK, now we have the corner of the matching blob pinned down */
	    xbcorner = sp->where.xtrans
		+ xoffmagic[trans].xi * (tp->xsize-1)
		+ xoffmagic[trans].yi * (tp->ysize-1);
	    ybcorner = sp->where.ytrans
		+ yoffmagic[trans].xi * (tp->xsize-1)
		+ yoffmagic[trans].yi * (tp->ysize-1);

	    flip = flip ? -1 : 1;

	    /*
	     * OK, the blobs matched.  What if there is a larger composite
	     * pattern associated with tp?  Does it match the rest of the 
	     * context?
	     */
	    if (tp->pattern != tp->blob)
	    {
		coord_t	xsize = tp->pattern->box.max.x - tp->pattern->box.min.x;
		coord_t	ysize = tp->pattern->box.max.y - tp->pattern->box.min.y;
		coord_t xinner, yinner, xouter, youter, xloc, yloc;

#ifndef __ADVANCED__
		printf("# Possible %s with flip %2d rotation %d based on blob at (%ld,%ld)\n",
		       tp->pattern->pattern_name,
		       flip, rotation,
		       sp->where.xtrans, sp->where.ytrans);
#else
		transform(flip, rotation,
			  tp->blob->box.min.x, tp->blob->box.min.y, 
			  0, 0, /*xsize, ysize,*/
			  &xinner, &yinner);
		transform(flip, rotation,
			  tp->pattern->box.min.x, tp->pattern->box.min.y,
			  0, 0, /*xsize, ysize,*/
			  &xouter, &youter);
		printf("# Inner: (%ld,%ld) Outer: (%ld,%ld) Diff: (%ld,%ld)\n",
		       xinner, yinner,
		       xouter, youter,
		       (xinner - xouter), (yinner - youter));

		xloc = sp->where.xtrans - (xinner - xouter);
		yloc = sp->where.ytrans - (yinner - youter);

		printf("# Possible %s at (%ld,%ld) with flip %2d rotation %d based on blob at (%ld,%ld)\n",
		       tp->pattern->pattern_name,
		       xloc, yloc,
		       flip, rotation,
		       sp->where.xtrans, sp->where.ytrans);
#endif /* ADVANCED */

		continue;
	    }

	    if (where)
	    {
		where->flip = flip;
		where->rotation = rotation;
		where->xtrans = xbcorner; 
		where->ytrans = ybcorner;
	    }
	    return(trans);
	nomatch:;
	}

    return(NO_MATCH);
}

void savestructured(pattern *context, FILE *ofp)
/* save with full scene analysis in structured (#I) format */
{
    pattern	*pp, *parts;
    blobref	*scene, *sp, *tp;
    coord_t	xcenter, ycenter;
    int		blobcount, i, j, nonrefcount;
    FILE	*fp;

    static int		matchcount = 0;
    static blobref	*matchlib;

    if ((parts = analyze(context)) == NULL)
    {
	error("Pattern analysis failed; too many blobs in picture.\n");
	return;
    }

    /* if there is only one blob, save in #P format */
    if (!parts[1].tiles)
    {
	bounding_box(context);
	savesimple(context,
		   -((int)(context->box.max.x+1-context->box.min.x)/2),
		   -((int)(context->box.max.y+1-context->box.min.y)/2),
		   ofp);
	free(parts);
	return;
    }

    /*
     * Otherwise it's time to do a full scene analysis.
     */
    blobcount = 0;
    for (pp = parts; pp->tiles; pp++)
	blobcount++;
    scene = (blobref *)calloc(sizeof(blobref), blobcount); 

    /* assemble a list of blob references */
    for (i = 0; i < blobcount; i++)
    {
	pattern	*pp = parts + i;
	sp = scene + i;

	sp->xsize = (pp->box.max.x - pp->box.min.x + 1); 
	sp->ysize = (pp->box.max.y - pp->box.min.y + 1);

	sp->pattern = sp->blob = &parts[i];

	sp->ref = (blobref *)NULL; 	/* defensive programming */
	sp->where.rotation = 0;		 	/* defensive programming */
	sp->where.flip = 1;
	sp->where.xtrans = sp->pattern->box.min.x;
	sp->where.ytrans = sp->pattern->box.min.y;
#ifdef COALESCE
	sp->refcount = 1;
#endif /* COALESCE */
    }

    /* search for duplicate blobs */
    for (sp = scene; sp < scene + blobcount; sp++)
	for (tp = scene; tp < sp; tp++)
	{
	    int	rot, fli, xoff, yoff;

	    if (tp->ref)
		continue;

#ifdef BLOBDEBUG
	    fprintf(stderr, "Blob %d at (%ld,%ld) size (%ld,%ld)\n",
		    sp - scene,
		    sp->xmin, sp->ymin,
		    sp->xsize, sp->ysize);
	    savesimple(sp->pattern, sp->xmin, sp->ymin, stderr);
	    fprintf(stderr, "Blob %d at (%ld,%ld) size (%ld,%ld)\n",
		    tp - scene, 
		    tp->xmin, tp->ymin,
		    tp->xsize, tp->ysize);
	    savesimple(tp->pattern, tp->xmin, tp->ymin, stderr);
#endif /* BLOBDEBUG */

	    if (blob_equal(context, tp, sp, &sp->where) != NO_MATCH)
	    {
		sp->ref = tp;
#ifdef COALESCE
		tp->refcount++;
#endif /* COALESCE */
#ifdef BLOBDEBUG
		fprintf(stderr, "Matched, flip %d and rotation %d\n", 
			sp->where.flip, sp->where.rotation);
#endif /* BLOBDEBUG */
		break;
	    }
	}

    /* initialize the list of named blobs */
    if (!matchcount)
    {
	if ((fp = fopen(matchlibfile, "r")))
	{
	    char	buf[BUFSIZ], name[BUFSIZ], request[BUFSIZ];
	    int		gens, gen;
	    cell_t	state;

	    matchcount = 0;
	    matchlib = (blobref *)calloc(sizeof(blobref), 1); 
	    while (fgets(buf, sizeof(buf)-1, fp) != (char *)NULL)
	    {
		char	*cp;
		static	pattern	matcher;	/* must be initially zeroed */

		if (buf[0] == '#')	/* allow comments */
		    continue;

		memset(&matcher, '\0', sizeof(matcher));
		initcells(&matcher);
		changesize(&matcher, 2);
		
		if (cp = strchr(buf, '#'))
		    while (isspace(*cp) || *cp == '#')
			*cp-- = '\0';

		name[0] = request[0] = '\0';
		gens = 0;
		sscanf(buf, "%s %s %d", name, request, &gens);

		if (name[0] == '\0')
		    continue;
		else if (request[0] == '\0')
		    strcpy(request, name);

		buf[strlen(buf)-1] = '\0';
		loadfile(request, &matcher, 0, 0);

		gen = 0;
		do {
		    tile	*ptr;
		    int		dx, dy;

		    if (gen == 0)
			strcpy(request, name);
		    else
			sprintf(request, "%s%d", name, gen+1);

		    matchlib = (blobref *)realloc(matchlib, sizeof(blobref) * (matchcount+1));
		    tp = matchlib + matchcount;

		    tp->pattern = (pattern *)malloc(sizeof(pattern));
		    memset(tp->pattern, '\0', sizeof(pattern));
		    initcells(tp->pattern);
		    changesize(tp->pattern, 2);
		    FOR_CELLS(&matcher, ptr, dx, dy)
			if ((state = getcell(&matcher, &ptr->cells, dx, dy, PRESENT)))
			    chgcell(tp->pattern, ptr->x + dx, ptr->y + dy, PRESENT, state);
		    bounding_box(tp->pattern); 
		    tp->blob = (pattern *)NULL;
		    tp->pattern->pattern_name = strdup(request);
		    matchcount++;

		    generate(&matcher);
		} while
		      (++gen < gens);
	    }
	    fclose(fp);

	    /* clear the last load message */
	    announce("");
	}

	/* now go through looking for composites */
	for (tp = matchlib; tp < matchlib + matchcount; tp++)
	{
	    pattern	*pp, *parts;
	    int 	blobcount;

	    parts = analyze(tp->pattern);

	    if (!parts[1].tiles)
		tp->blob = tp->pattern;
	    else
	    {
		int	merit = 0;
		pattern	*best;

		for (pp = parts; pp->tiles; pp++)
		{
		    int	m = pp->cellcount 
			* (pp->box.max.x - pp->box.min.x) * (pp->box.max.y - pp->box.min.y);

		    if (m > merit)
		    {
			best = pp;
			merit = m;
		    }
		}

		tp->blob = (pattern *)malloc(sizeof(pattern));
		memcpy(tp->blob, best, sizeof(pattern));
		initcells(best);
	    }

	    for (pp = parts; pp->tiles; pp++)
		clear_pattern(pp);
	    free(parts);

	    bounding_box(tp->blob);
	    tp->xsize = (tp->blob->box.max.x - tp->blob->box.min.x + 1); 
	    tp->ysize = (tp->blob->box.max.y - tp->blob->box.min.y + 1);
	}
    }

    /* check for named blocks in the pattern */
    if (matchcount)
    {
	int	point;

	for (sp = scene; sp < scene + blobcount; sp++)
	    for (tp = matchlib; tp < matchlib + matchcount; tp++)
		if ((point = blob_equal(context, tp, sp, (dihedral *)NULL)) != NO_MATCH)
		{
		    char	*cp;

		    if (sp->ref)
			continue;

		    if ((cp = strchr(tp->pattern->pattern_name, ':')))
			cp++;
		    else
			cp = tp->pattern->pattern_name;
		    if (sp->pattern->pattern_name)
			free(sp->pattern->pattern_name);
		    sp->pattern->pattern_name = strdup(cp);
		}
	fputs("\n", ofp);
    }

#ifdef COALESCE
    /*
     * Coalesce unnamed blobs with bounding rectangles that nearly touch.
     * This way we get an output report with fewer tiny "junk" blobs.
     */
#define NEAR	2
#define LAP(s1, e1, s2, e2)	((((s2)+NEAR <= (e1)-NEAR) && ((e2)+NEAR >= (s1)-NEAR)) \
				 || (((s1)+NEAR <= (e2)-NEAR) && ((e1)+NEAR >= (s2)-NEAR)))
    do {
	i = 0;
	for (sp = scene; sp < scene + blobcount; sp++)
	    if (sp->refcount > 0 && !sp->pattern->pattern_name)
		for (tp = scene; tp < sp; tp++)
		    if (tp->refcount > 0 && !tp->pattern->pattern_name)
			if (LAP(sp->pattern->box.min.x,
				sp->pattern->box.max.x, 
				tp->pattern->box.min.x,
				tp->pattern->box.max.x)
			    && LAP(sp->pattern->box.min.y,
				   sp->pattern->box.max.y, 
				   tp->pattern->box.min.y,
				   tp->pattern->box.max.y))
		    {
			int 	state, dx, dy;
			tile	*ptr;

			FOR_CELLS(sp->pattern, ptr, dx, dy)
			    if ((state=getcell(sp->pattern,&ptr->cells,dx,dy,PRESENT)))
			    {
				chgcell(tp->pattern,
					ptr->x+dx,ptr->y+dy, PRESENT, state);
				chgcell(sp->pattern,
					ptr->x+dx,ptr->y+dy, PRESENT, 0);
			    }
			sp->refcount--;
			i++;
		    }
    } while
	(i);
#undef LAP
#endif /* COALESCE */

    /* name the unnamed blobs */
    nonrefcount = 0;
    for (sp = scene; sp < scene + blobcount; sp++)
	if (!sp->ref && !sp->pattern->pattern_name)
	{
	    char	namebuf[10];

	    sprintf(namebuf, "part%d", ++nonrefcount);
	    sp->pattern->pattern_name = strdup(namebuf);
	}

    /* emit the blob list */
    for (sp = scene; sp < scene + blobcount; sp++)
    {
#ifdef COALESCE
	if (!sp->refcount)
	   continue;
#endif /* COALESCE */ 
	if (!sp->ref)
	{
	    fprintf(ofp, "#B %s\n", sp->pattern->pattern_name);
	    savesimple(sp->pattern, 0, 0, ofp);
	    fputs("#E\n\n", ofp);
	}
    }

    /* issue the inclusion list */
    xcenter = (context->box.max.x+1+context->box.min.x)/2;
    ycenter = (context->box.max.y+1+context->box.min.y)/2;
    fprintf(ofp, "# %d blobs\n", blobcount);
    for (sp = scene; sp < scene + blobcount; sp++)
    {
#ifdef COALESCE
	if (!sp->refcount)
	   continue;
#endif /* COALESCE */ 
	if (sp->ref)
	    tp = sp->ref;
	else
	    tp = sp;
	fprintf(ofp, "#I :%s\t%4d %4d %d %2d 0\n",
		tp->pattern->pattern_name, 
		sp->where.xtrans - xcenter, 
		sp->where.ytrans - ycenter,
		sp->where.rotation,
		sp->where.flip);
    }

    free(parts);
}

static void *get_list_of_sorted_cellbox_ptr(pattern *context)
{
    coord_t  *coordxyptrlist, i;
    cellcount_t	ctr;
    tile *ptr;

    if (!(coordxyptrlist= (coord_t*) malloc(
	((2+1)*sizeof(coord_t)) * (context->tilecount+1) ) ))
	return (tile **)0;
    ctr = 0;
    coordxyptrlist += 3;
    for (ptr = context->tiles; ptr != NULL; ptr = ptr->next){
	if (!ptr->dead) {
	    coordxyptrlist[ctr++] = ptr->y;
	    coordxyptrlist[ctr++] = ptr->x;
	    coordxyptrlist[ctr++] = (coord_t) ptr;
	}
    } 
    ctr /= 3;
    if (ctr>0)
	heapsort(coordxyptrlist,ctr,3);
    coordxyptrlist -= 3;
    for (i=0;i<ctr;i++)
	coordxyptrlist[i]= coordxyptrlist[3+3*i+2];
    coordxyptrlist[ctr] = 0;
    return  (void *) coordxyptrlist;
}

void savepattern(FILE *ofp, pattern *pp, char mode)
/* save the given context */
{
    tile *ptr;
    int dx, dy, val,n;
    char	buf[PATNAMESIZ];
    cellcount_t cellcount = bounding_box(pp);

    fputs(SAVE_VERSION "\n", ofp);
    if (pp->pattern_name)
	fprintf(ofp,"#N %s\n", pp->pattern_name);
	
    display_rules(buf);
#if STATEBITS > 1
    if (pp->maxstates > 2)
	(void) fprintf(ofp, "#U %s\n", buf);
    else
#endif /* STATEBITS > 1 */
	if (strcmp(buf, "Life"))
	    (void) fprintf(ofp, "#T %s\n", buf);

    stamp("#0", ofp);

    for (n = 0; n < pp->ncomments; n++)
	fprintf(ofp,"#C%s\n", pp->comments[n]);
	
    if (pp->outcome)
	fprintf(ofp,"#K%s\n", pp->outcome);
	
    (void) fputs("\n", ofp);
    if (mode == 'A')		/* save in absolute mode */
    {
	fputs("#A\n", ofp);
	for (ptr = pp->tiles; ptr != NULL; ptr = ptr->next)
	    if (!ptr->dead)
		for (dx = 0; dx < BOXSIZE; dx++)
		    for (dy = 0; dy < BOXSIZE; dy++)
			if ((val = getcell(pp, &ptr->cells, dx, dy, PRESENT)))
#if STATEBITS > 1
			    if (pp->maxstates > 2)
				(void) fprintf(ofp, "%ld %ld %d\n",
					       ptr->x+dx, ptr->y+dy, val);
			    else 
#endif
				(void) fprintf(ofp,
					       "%ld %ld\n",
					       ptr->x+dx,
					       ptr->y+dy);
	return;
    }
    else if (mode == 'R')
    {
	fputs("#R\n", ofp);
	for (ptr = pp->tiles; ptr != NULL; ptr = ptr->next)
	    if (!ptr->dead)
		for (dx = 0; dx < BOXSIZE; dx++)
		    for (dy = 0; dy < BOXSIZE; dy++)
			if ((val = getcell(pp, &ptr->cells, dx, dy, PRESENT)))
#if STATEBITS > 1
			    if (pp->maxstates > 2)
				(void) fprintf(ofp, "%ld %ld %d\n",
					       ptr->x+dx-pp->box.min.x, ptr->y+dy-pp->box.min.y,
					       val);
			    else 
#endif
				(void) fprintf(ofp, "%ld %ld\n",
					       ptr->x+dx-pp->box.min.x, ptr->y+dy-pp->box.min.y);
    }
    else if (mode == 'D')
    {
	coord_t xx=0, yy=0;
	fputs("#D\n", ofp);
	for (ptr = pp->tiles; ptr != NULL; ptr = ptr->next)
	    if (!ptr->dead)
		for (dx = 0; dx < BOXSIZE; dx++)
		    for (dy = 0; dy < BOXSIZE; dy++)
			if (val = getcell(pp, &ptr->cells, dx, dy, PRESENT))
			{
			    (void) fprintf(ofp, "%ld %ld\n",
				   ptr->x+dx-pp->box.min.x-xx, ptr->y+dy-pp->box.min.y-yy);
			    xx = ptr->x+dx-pp->box.min.x;
			    yy = ptr->y+dy-pp->box.min.y;
			}
    }
    else if (mode == 'M')
    {
	coord_t xx= -1, yy=0, x, y, i, j, k=0, n=0;
#define  SURE_LINE_LENGTH_LIMIT  (70-10)  /* David Buckingham's definition */
	tile **lex_start;

	if (!(lex_start = (tile **) get_list_of_sorted_cellbox_ptr(pp)))
	{
	    error("Insufficient memory: Can't save configuration.");
	    return;
	}
	fprintf(ofp, "#M  x = %ld , y = %ld\n", pp->box.max.x+1-pp->box.min.x, pp->box.max.y+1-pp->box.min.y );
	for (j=0;lex_start[j] != NULL;j=i)
	    for (dy = 0; dy < BOXSIZE; dy++)
		for (ptr = lex_start[i=j]; ptr != NULL &&
			 (i==j || lex_start[i-1]->y==ptr->y) ; ptr = lex_start[++i])
		    if (!ptr->dead)
			for (dx = 0; dx < BOXSIZE; dx++)
			    if (val = getcell(pp, &ptr->cells, dx, dy, PRESENT))
			    {
				x = ptr->x+dx-pp->box.min.x;
				y = ptr->y+dy-pp->box.min.y;
				if (y == yy  &&  x == xx+1)
				{
				    k++;  xx++;
				    continue;
				}
				if (k>1)
				    n+= fprintf(ofp, "%ldo",k);
				else if (k)
				    n++, fputc('o', ofp);
				k=1;
				if (y > yy)
				{
				    xx= -1;
				    if (y > yy+1)
					    n += fprintf(ofp, "%ld$",y-yy);
				    else
					    n++, fputc('$', ofp);
				    yy = y;
				}
				if (x > xx+2)
				    n += fprintf(ofp, "%ldb",x-xx-1);
				else if (x == xx+2)
				    n++, fputc('b', ofp);
				xx = x;
				if (n > SURE_LINE_LENGTH_LIMIT)
				    n=0, fputc('\n', ofp);
			    }
	if (k>1)
	    n += fprintf(ofp, "%ldo",k);
	else if (k)
	    n++, fputc('o', ofp);
	(void) fputc('!', ofp);
	(void) fputc('\n', ofp);
	free(lex_start);
#undef  SURE_LINE_LENGTH_LIMIT
    }
    else if (mode == 'P')
    {
	savesimple(pp,
		   -((int)(pp->box.max.x+1-pp->box.min.x)/2),
		   -((int)(pp->box.max.y+1-pp->box.min.y)/2),
		   ofp);
    }
    else if (mode == 'S')
	savecomplex(pp, ofp);
    else /* if (mode == 'I' || mode == '\0')) */
	savestructured(pp, ofp);

    if (mode == 'S' || mode == 'P' || mode == 'I' || mode == '\0')
    {
	(void) fputs("\n", ofp);
	(void) fputs("## The following sets edit modes for GNU EMACS\n", ofp);
	(void) fputs("## Local ", ofp); (void) fputs("Variables:\n", ofp);
	(void) fputs("## mode:picture\n", ofp);
	(void) fputs("## truncate-lines:t\n", ofp);
	(void) fputs("## End:\n", ofp);
    }
}

/* save.c ends here */
