/*
 * XLife Copyright 1989 Jon Bennett jb7m+@andrew.cmu.edu, jcrb@cs.cmu.edu
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of the copyright holders not be used in
 * advertising or publicity pertaining to distribution of the software without
 * specific, written prior permission.  The copyright holders make no
 * representations about the suitability of this software for any purpose.  It
 * is provided "as is" without express or implied warranty.
 *
 * THE COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL THE COPYRIGHT HOLDERS BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
 * DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
 * TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */

#include <stdlib.h>
#include <stdio.h>
#include <pwd.h>
#include <ctype.h>
#include <dirent.h>
#include <string.h>
#include <unistd.h>

#include "defs.h"
#include "data.h"

GLOBAL char *addlifeext(char *);
GLOBAL char *checktilda(char *);

static cellcount_t generations;

#define MAXDIRS		64
static char *dirs[MAXDIRS];

static FILE *outfp;
static int filecount = 0;

static void searchinit(void)
/* set up initial search path for file loads */
{
    extern char *getenv();
    char *cp, **dirp = dirs;

    if ((cp = getenv("LIFEPATH")))
    {
	cp = strtok(cp, ":");
	do {
	    if (dirp - dirs >= MAXDIRS)
		break;	/* too many pattern libraries? ignore silently */
	    *dirp++ = cp;
	} while
	    ((cp = strtok((char *)NULL, ":")));
    }
    else
    {
	*dirp++ = ".";
	*dirp++ = "patterns";
	*dirp++ = LIFEDIR;
    }
}

static char *deletelifeext(buf)
char *buf;
{
    int len=strlen(buf);
    if (!strcmp(buf+len-LOADEXTLEN, LOADEXT))
	buf[len-LOADEXTLEN]='\0';
    
    return(buf);
}

static void seppatdir(filename,filefield) 
char *filename,*filefield;
{
char *ffptr,*colptr;
/* Separates path from rest of pattern's name.  */

   /* Make sure "/" in pattern block name isn't confusing by temporarily
      eliminating ":" suffix */
   if ((colptr=strchr(filename,':')))
       (*colptr)='\0';

   ffptr=strrchr(filename,'/');
   if (ffptr) {
      strcpy(filefield,ffptr+1);
      *(ffptr+1)='\0';
   }
   else {
      strcpy(filefield,filename);
      *(filename)='\0';
   } 

   /* restore ":" suffix */
   if (colptr)
       (*colptr)=':';
}

static void add_loadreq(loadqueue, patname, relpath, loadtime)
    LoadReq **loadqueue;
    char *patname;
    int relpath;
    lifetime_t loadtime;
{
  LoadReq *newreq;
  /* Find last entry where request can go while maintaining time order.
     Hopefully, this will cause the resulting file to be ordered in a nice
     way.

     After loop, we have a pointer to the pointer that must be modified
     (Could be pointer to list itself). */ 

  while ((*loadqueue)!=NULL && ((*loadqueue)->loadtime <= loadtime)) 
     loadqueue= &((*loadqueue)->next); 

  /* Create new request block and load it up */
  newreq=(LoadReq *)malloc(sizeof(LoadReq)); 
  newreq->loadtime=loadtime;
  newreq->relpath=relpath;
  /* Silently truncates long file names--should probably tell user */
  strncpy(newreq->patname,patname,PATNAMESIZ);

  /* Insert new request in queue */
  newreq->next=(*loadqueue);
  (*loadqueue)=newreq; 
}

static int seen(seenlist, patname)
    LoadReq **seenlist;
    char *patname;
{
  LoadReq *newpat;

  /* Return whether pattern has been seen.  If not, add to list. */ 

  while ((*seenlist)!=NULL && strcmp((*seenlist)->patname,patname)) 
     seenlist = &((*seenlist)->next); 

  if ((*seenlist)==NULL) {
     /* Create new block and store name */
     newpat=(LoadReq *)malloc(sizeof(LoadReq)); 
     strncpy(newpat->patname,patname,PATNAMESIZ);

     /* Insert new pattern on list */
     newpat->next=(*seenlist);
     (*seenlist)=newpat; 
     return(0);
  } 
  return(1);
}

static void parse_patname(patname,patfield)
char *patname,*patfield;
{
    char *pfptr;
/* Breaks "<filename>:<pattern>" into two strings. */

    pfptr=strchr(patname,':');
    if (pfptr) {
	*pfptr='\0';
	strcpy(patfield,pfptr+1);
    }
    else patfield[0]='\0';
}

static int do_loadfile(seenlist,loadqueue,ismain)
    LoadReq **seenlist;
    LoadReq **loadqueue;
    int ismain;
{
    char patname[PATNAMESIZ],patfield[PATNAMESIZ];
    FILE *loadfl;
    char patfile[BUFSIZ];
    LoadReq *tmpqptr;
    int relpath;

    /* Get load request */
    strcpy(patname,(*loadqueue)->patname);
    relpath=(*loadqueue)->relpath;

    /* Delete request from queue */
    tmpqptr=(*loadqueue)->next; 
    free(*loadqueue);
    (*loadqueue)=tmpqptr;

    /* separate filename from pattern name */
    parse_patname(patname,patfield);

    /* add load extension to filename if needed */
    (void) strcat(patname, addlifeext(patname));

    if (patname[0] == '/')
	(void) strcpy(patfile, patname);
    else
    {
	char	**dirp;

	for (dirp = dirs; *dirp; dirp++)
	{
	    DIR	*dp;
	    struct dirent *entry;

	    if ((dp = opendir(*dirp)))
	    {
		while ((entry = readdir(dp)))
		{
		    if (entry->d_name[0] == '.' && entry->d_name[1] == '.')
			continue;
		    (void) sprintf(patfile, "%s/%s/%s",
				   *dirp, entry->d_name, patname);
		    if (access(patfile, R_OK) == 0)
		    {
			closedir(dp);
			goto foundit;
		    }
		}
		closedir(dp);
	    }
	    else
	    {
		(void) sprintf(patfile, "%s/%s", *dirp, patname);
		if (access(patfile, R_OK) == 0)
		    goto foundit;
	    }
	}

	return 0;
    foundit:;
    }    

    if ((loadfl=fopen(patfile,"r")) == NULL)
	return 0;
    else {
	char	buf[BUFSIZ],patid[PATNAMESIZ],pardir[PATNAMESIZ], *cp;
        int     endpattern=0,found=0,relpathoffset=0;
	bool	lastblank = FALSE;

	if (filecount++ == 0)
	    fputs(SAVE_VERSION "\n", outfp);

	/* this just cleans up the reporting a little */
	for (cp = patfile; *cp; cp++)
	    while ((cp == patfile || cp[-1]=='/') && cp[0]=='.' && cp[1]=='/')
		memmove(cp, cp+2, strlen(cp+2)+1);

        /* print comment telling where pattern was found */ 
        fprintf(outfp, "## FOUND IN %s:\n", patfile);

        /* If we are searching for a specific pattern in the file,
           then we skip lines till we find it.  This is a pretty tacky way
           to handle multiple pattern definitions in the same file,
           but files with #I format probably won't get big enough for
           anyone to notice.  Really, we should implement a dictionary to
           save the location of a pattern definition in a file the first time 
           we see it. 
        */
        deletelifeext(patname+relpath);
        if (patfield[0]!='\0') { 
	    while (!found && fgets(buf, BUFSIZ, loadfl) != (char *)NULL) { 
		if (buf[0]=='#' && buf[1]=='B') {
		    (void) sscanf(buf+2, " %s", patid); 
		    found=!strcmp(patfield,patid);
		} 
	    }
	    if (!found) {
		fclose(loadfl);
		return 0; 
	    } else if (!ismain)
		fprintf(outfp,"#B %s.%s\n",patname+relpath,patfield);
        } else if (!ismain)
	    fprintf(outfp,"#B %s\n",patname+relpath);
	while (fgets(buf, BUFSIZ, loadfl) != (char *)NULL && !endpattern)
	{
	    if (buf[0] == '#')
	    {
		char	incl[BUFSIZ];
		char	tmpstring[PATNAMESIZ];
		coord_t	xoff,yoff;
                int	rotate,flip;
		lifetime_t	loadtime;

		/* skip version stamps: we'll make our own */
		if (strncmp(SAVE_VERSION, buf, sizeof(SAVE_VERSION) - 1) == 0)
		    continue;

		incl[0] = '\0';
		switch(buf[1])
		{
		case 'B':
                    /* Anything between #B and #E is ignored, since it
                       is a pattern definition to be included later.
                       #B and #E cannot be nested, so we just skip till
                       we pass #E */
	            while (fgets(buf, BUFSIZ, loadfl) != (char *)NULL
                           && !(buf[0]=='#' && buf[1]=='E')) {}
                    break;

		case 'E': 
                    /* if we hit a "#E" at this point, then it must be
                       the one that closes the "#B <patfield>" that we
                       are reading (assuming a syntactically correct file) */
                    endpattern=1;
                    if (!ismain)
			fprintf(outfp,"%s",buf);
                    break;

		case 'I':
		    xoff = yoff = rotate = loadtime = 0;
                    flip = 1;
		    (void) sscanf(buf+2, " %s %ld %ld %d %d %ld", incl, 
				  &xoff, &yoff, &rotate, &flip, &loadtime); 
                    /* Silently ignore invalid flips */
                    if ((flip!=1) && (flip!= -1)) flip=1;

                    /* if included pattern begins with ':' then assume
                       it's in active file */
                    if (incl[0]==':') {
			strcpy(tmpstring,patfile);
			strcat(tmpstring,incl);
			strcpy(incl,tmpstring);
                    } else {
			/* if relative path given, assume directory of parent */
			if (!strchr("/~",incl[0])) {
			    strcpy(pardir,patfile);
			    seppatdir(pardir,tmpstring);
			    relpathoffset=strlen(pardir);
			    strcat(pardir,incl);
			    strcpy(incl,pardir);
			}
                    } 
                    if (!seen(seenlist,incl)) 
			add_loadreq(loadqueue, incl, relpathoffset, 
				    generations+loadtime); 
                    parse_patname(incl,tmpstring);
                    deletelifeext(incl);
                    if (tmpstring[0]!='\0') 
			fprintf(outfp,"#I :%s.%s %ld %ld %d %d %ld\n",
			       incl+relpathoffset,
			       tmpstring,
			       xoff, yoff, rotate, flip, loadtime); 
                    else
			fprintf(outfp,"#I :%s %ld %ld %d %d %ld\n",
			       incl+relpathoffset,
			       xoff, yoff, rotate, flip, loadtime); 
		    break;
		    
		default:
                    fprintf(outfp,"%s",buf); 
		    break;
		}
	    }
	    else
	    {
		if (buf[0] == '\n')
		    if (!lastblank) 
			lastblank = TRUE;
		    else
		    {
			lastblank = TRUE;
			continue;
		    }

		fprintf(outfp,"%s",buf);
	    }
	}
	fclose(loadfl);
        if (patfield[0]=='\0' && !ismain)
	    fprintf(outfp,"#E\n"); 
        fprintf(outfp,"\n");
	return 1;
    }
}

static char *collect_loadreq(seenlist,loadqueue)
LoadReq **seenlist;
LoadReq *loadqueue;
{
    static char thispat[PATNAMESIZ];

   strcpy(thispat,loadqueue->patname);
   if (!do_loadfile(seenlist,&loadqueue,1))
          return(thispat);
   while (loadqueue!=NULL) {
      generations=loadqueue->loadtime;
      strcpy(thispat,loadqueue->patname);
      if (!do_loadfile(seenlist,&loadqueue,0)) {
          return(thispat);
      }
   }

   return((char *)NULL);
}

char *collect(char *file, FILE *fpout)
{
    static char outbuf[PATNAMESIZ];
    LoadReq *loadqueue=NULL;
    LoadReq *seenlist=NULL;

    searchinit();
    strcpy(outbuf,checktilda(file));
    generations=0;
    add_loadreq(&loadqueue, outbuf, 0, generations); 
    seen(&seenlist,outbuf);
    outfp = fpout;
    return(collect_loadreq(&seenlist,loadqueue));
}

/* lifecollect.c ends here */
