/* geometry.c -- matrix algebra routines for geometric transformations */

#include "defs.h"

/* matrix for rotations */
static motion rotations[] =
{
    {0,0,  1, 0, 0, 1},
    {0,0,  0,-1, 1, 0},
    {0,0, -1, 0, 0,-1},
    {0,0,  0, 1,-1, 0},
    {0,0,  1, 0, 0,-1},
    {0,0,  0, 1, 1, 0},
    {0,0, -1, 0, 0, 1},
    {0,0,  0,-1,-1, 0},
};

const motion motion_unity = {0, 0, 1,0,0,1};

dihedral motion2dihedral(motion map) 
/* convert motion representation of a transform to dihedral */
{
    dihedral res;
    motion *rotp;

    res.xtrans = map.xtrans;
    res.ytrans = map.ytrans;

    for (rotp = rotations; rotp < rotations + 8; rotp++)
	if (map.rxx == rotp->rxx && map.rxy == rotp->rxy 
		&& map.ryx == rotp->ryx && map.ryy == rotp->ryy)
	    break;

    res.rotation = rotp - rotations;
    if (res.rotation < 4)
	res.flip = 1;
    else
    {
	res.rotation %= 4;
	res.flip = -1;
    }
    return(res);
}

motion apply_rotation(motion transform, int flip, int rotation)
/* apply a dihedral rotation to a motion transformation */
{
    motion	res, *rotp = rotations + mod(rotation, 4) + (flip == -1) * 4;

    res.xtrans = transform.xtrans;
    res.ytrans = transform.ytrans;
    res.rxx = transform.rxx*rotp->rxx + transform.rxy*rotp->ryx;
    res.rxy = transform.rxx*rotp->rxy + transform.rxy*rotp->ryy;
    res.ryx = transform.ryx*rotp->rxx + transform.ryy*rotp->ryx;
    res.ryy = transform.ryx*rotp->rxy + transform.ryy*rotp->ryy;

    return (res);
}


point point_dihedral(point location, dihedral transform)
/* apply a dihedral transformation to a point */
{
    point res;
    motion	*rotp = rotations + mod(transform.rotation, 8);

    res.x = location.x * rotp->rxx + location.y * rotp->rxy;
    res.y = location.x * rotp->ryx + location.y * rotp->ryy;
    res.x += transform.xtrans;
    res.y += transform.ytrans;
    return(res);
}

/* geometry.c ends here */
