/*
 * XLife Copyright 1989 Jon Bennett jb7m+@andrew.cmu.edu, jcrb@cs.cmu.edu
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of the copyright holders not be used in
 * advertising or publicity pertaining to distribution of the software without
 * specific, written prior permission.  The copyright holders make no
 * representations about the suitability of this software for any purpose.  It
 * is provided "as is" without express or implied warranty.
 *
 * THE COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL THE COPYRIGHT HOLDERS BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
 * DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
 * TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */

/*****************************************************************************

This module encapsulates the loading and  saving of patterns.
Its only interface to the GUI is through announce(), error(), 
breakreq() and fatal().

*****************************************************************************/

#include <stdio.h>
#include <pwd.h>
#include <ctype.h>
#include <dirent.h>
#include <errno.h>
#include <stdlib.h>	/* for malloc(3) */
#include <string.h>
#include <unistd.h>

#include "defs.h"
#include "data.h"
#include "tile.h"
#include "guidata.h"

/* global header cell for load script */ 
static LoadReq *loadscript;
 
#define UNKNOWN_ADDRESS	-1

typedef struct entry_t 
{
    char		*request;
    struct entry_t	*next;
    int			refcount;
    char		*name;
    char		*fileref;
    char		*fullpath;
    fpos_t		address;
}
entry;

static entry *dictionary, **freeptr;

#define MAXDIRS		64
static char *dirs[MAXDIRS];

static entry *find_entry(const char *request)
/* find an entry in the pattern dictionary */
{
    entry	*ep;

    for (ep = dictionary; ep; ep = ep->next)
	if (strcmp(request, ep->request) == 0)
	    return(ep);
    return((entry *)NULL);
}

static entry *add_entry(const char *request)
/* add an entry to the pattern dictionary */
{ 
    entry	*res;

    if ((res = find_entry(request)))
	res->refcount++;
    else
    {
	res = (entry *)malloc(sizeof(entry));
	if (!res)
	    fatal("Dictionary entry allocation failed!");
	res->request = (char *)strdup(request);
	res->refcount = 1;
	res->name = (char *)NULL;
	res->fileref = (char *)NULL;
	res->fullpath = (char *)NULL;
	res->address = UNKNOWN_ADDRESS;

	res->next = (entry *)NULL;
	*freeptr = res;
	freeptr = &(res->next);
    }

    return(res);
}

void fileinit(void)
/* set up initial search path for file loads */
{
    extern char *getenv();
    char *cp, **dirp = dirs;

    if ((cp = getenv("LIFEPATH")))
    {
	cp = strtok(cp, ":");
	do {
	    if (dirp - dirs > MAXDIRS)
		fatal("Too many pattern libraries\n");
	    *dirp++ = cp;
	} while
	    ((cp = strtok((char *)NULL, ":")));
    }
    else
    {
	*dirp++ = ".";
	*dirp++ = "patterns";
	*dirp++ = LIFEDIR;
    }

    /* initialize load script to empty (circular list) */
    loadscript = (LoadReq *)malloc(sizeof(LoadReq)); 
    loadscript->next=loadscript;

    dictionary = (entry *)NULL;
    freeptr = &dictionary;

    if (loadirbuf[0] != '/')
    {
	char trunc[DIRBUFLEN];

	strcpy(trunc, loadirbuf);
	if (trunc[strlen(trunc)-1] == '/')
	    trunc[strlen(trunc)-1] = '\0';
	for (dirp = dirs; *dirp; dirp++)
	{
	    DIR	*dp;
	    struct dirent *entry;

	    if ((dp = opendir(*dirp)))
	    {
		 while ((entry = readdir(dp)))
		 {
		     if (!strcmp(entry->d_name, trunc))
		     {
			 closedir(dp);
			 sprintf(loadirbuf, "%s/%s/", *dirp, trunc);
			 goto foundit;
		     }
		 }
		 closedir(dp);
	    }
	}
 foundit:;
    }
}

char *findfile(const char *patfile)
/* find a file from a relative specification */
{
    static char fullpath[PATNAMESIZ];
    char	**dirp;

    if (patfile[0] == '/')
	(void) strcpy(fullpath, patfile);
    else
    {
	for (dirp = dirs; *dirp; dirp++)
	{
	    DIR	*dp;
	    struct dirent *entry;

	    if ((dp = opendir(*dirp)))
	    {
		while ((entry = readdir(dp)))
		{
		    (void) sprintf(fullpath, "%s/%s/%s",
				   *dirp, entry->d_name, patfile);
		    if (access(fullpath, R_OK) == 0)
		    {
			closedir(dp);
			return(fullpath);
		    }
		}
		closedir(dp);
	    }
	    else
	    {
		(void) sprintf(fullpath, "%s/%s", *dirp, patfile);
		if (access(fullpath, R_OK) == 0)
		    return(fullpath);
	    }
	}

	return (char *)NULL;
    }
}

static char *gethome()
{
    return(getpwuid(getuid())->pw_dir);
}


char *checktilda(const char *stng)
{
    static char full[1024];
    struct passwd *pw;
    int i;

    if (stng[0]=='~')
    {
	i=1;
	while (stng[i]!='/' && stng[i]!='\0')
	    full[i-1]=stng[i], i++;
	if (i==1) 
	    strcpy(full,gethome());
	else
	{
	    full[i-1]='\0';
	    if ((pw = getpwnam(full)) == NULL){
		error(strerror(errno));
	    }
	    else{
		strcpy(full,pw->pw_dir);
	    }
	}
	strcat(full,stng+i);
    }
    else
	strcpy(full,stng);
    return(full);
}

char *addlifeext(char *buf)
{
    int len=strlen(buf);
    if (strcmp(buf+len-LOADEXTLEN, LOADEXT))
	return(LOADEXT);
    return("");
}

static void seppatdir(filename,filefield) 
char *filename,*filefield;
{
    char *ffptr,*colptr;
    /* Separates path from rest of pattern's name.  */

    /* Make sure "/" in pattern block name isn't confusing by temporarily
       eliminating ":" suffix */
    if ((colptr=strchr(filename,':')))
	(*colptr)='\0';

    ffptr=strrchr(filename,'/');
    if (ffptr) {
	strcpy(filefield,ffptr+1);
	*(ffptr+1)='\0';
    }
    else
    {
	strcpy(filefield,filename);
	*(filename)='\0';
    } 

    /* restore ":" suffix */
    if (colptr)
	(*colptr)=':';
}

static void parse_patname(patname,patfield)
char *patname,*patfield;
/* Breaks "<filename>:<pattern>" into two strings. */
{
    char *pfptr;

    pfptr=strchr(patname,':');
    if (pfptr)
    {
	*pfptr='\0';
	strcpy(patfield,pfptr+1);
    }
    else patfield[0]='\0';
}

static void add_loadreq(LoadReq **loadqueue, 
			lifetime_t loadtime, char *patname, int relpath,
			const motion *location)
{
    LoadReq *newreq;

    /* Find first entry where request can go while maintaining time order.
       A heap would have better theoretical time complexity, of course, but
       in practice, the list probably won't get too long.   
       
       After loop, we have a pointer to the pointer that must be modified
       (Could be pointer to list itself). */ 

    while ((*loadqueue)!=NULL && ((*loadqueue)->loadtime < loadtime)) 
	loadqueue= &((*loadqueue)->next); 

    /* Create new request block and load it up */
    newreq=(LoadReq *)malloc(sizeof(LoadReq)); 
    newreq->loadtime=loadtime;
    /* Silently truncates long file names--should probably tell user */
    strncpy(newreq->patname,patname,PATNAMESIZ);
    newreq->relpath=relpath;
    newreq->at = *location;

    /* Insert new request in queue */
    newreq->next=(*loadqueue);
    (*loadqueue)=newreq; 
}

static int do_loadfile(loadqueue, context)
LoadReq **loadqueue;
pattern *context;
{
    char patname[PATNAMESIZ],patfield[PATNAMESIZ],tmpstring[PATNAMESIZ];
    int relpath;
    FILE *loadfl;
    char patfile[BUFSIZ], *fullpath;
    LoadReq *tmpqptr;
    entry *ep;
    motion location;

    int x,y, linect = 0, colct = 0;
#define M_ABSOLUTE	0
#define M_RELATIVE	1
#define M_DELTA		2
#define M_RUNLENGTH	3
#define M_PICTURE	4
    int loadmode = M_ABSOLUTE;

    /* Get load request */
    strcpy(patname,(*loadqueue)->patname);
    relpath=(*loadqueue)->relpath;
    location = (*loadqueue)->at;

    /* Delete request from queue */
    tmpqptr=(*loadqueue)->next; 
    free(*loadqueue);
    (*loadqueue)=tmpqptr;

    /* have we seen this request before? */
    ep = add_entry(patname);

    if (ep->fullpath)
    {
	fullpath = ep->fullpath;
	strcpy(patfield, ep->name);
	strcpy(patfile, ep->fileref);
    }
    else
    {
	/* separate filename from pattern name */
	parse_patname(patname,patfield);

	/* add load extension to filename if needed */
	(void) strcat(patname, addlifeext(patname));

	/* open file */
	strcpy(patfile,patname);

	if ((fullpath = findfile(patfile)) == (char *)NULL)
	    return(0);

	ep->name = strdup(patfield);
	ep->fileref = strdup(patname);
	ep->fullpath = strdup(fullpath);
    }

    if ((loadfl = fopen(fullpath,"r")) != NULL)
    {
	char	buf[BUFSIZ],patid[PATNAMESIZ];
	int	state, xoff = 0, yoff = 0, anz = 0, cct = 0;
        bool	endpattern = FALSE, found = FALSE;
	motion	newloc;

	if (ep->address != UNKNOWN_ADDRESS)
	    fseek(loadfl, ep->address, SEEK_SET);
	else
	{
	    if (patfield[0]!='\0')
	    { 
		
		while (!found && fgets(buf, BUFSIZ, loadfl) != (char *)NULL)
		{ 
		    if (buf[0]=='#' && buf[1]=='B')
		    {
			char	*cp;

			for (cp = buf + 2; isspace(*cp) ||  *cp == ':'; cp++)
			    continue;
			(void) sscanf(cp, "%s", patid); 
			found = !strcmp(patfield,patid);
		    }
		}
		if (!found)
		{
		    fclose(loadfl);
		    return 0; 
		}
	    }

	    ep->address = ftell(loadfl);
	}

	while (fgets(buf, BUFSIZ, loadfl) != (char *)NULL && !endpattern)
	{
	    if (buf[0] == '#')
	    {
		char	incl[BUFSIZ];
		char	pardir[PATNAMESIZ], *cp;
		int	lx, ly, rotate, flip, relpathoffset=0;
		lifetime_t	loadtime;

		incl[0] = '\0';
		switch(buf[1])
		{
		case 'A':
		    loadmode = M_ABSOLUTE;
		    break;
		    
		case 'B':
                    /* Anything between #B and #E is ignored, since it
		       is a pattern definition to be included later.
                       #B and #E cannot be nested, so we just skip till
		       we pass #E */
	            while (fgets(buf, BUFSIZ, loadfl) != (char *)NULL
                           && !(buf[0]=='#' && buf[1]=='E'))
			continue;
                    break;

		case 'C':
		    if (ep->refcount <= 1)
		    {
			char	*np = buf + strlen(buf) - 1;

			if (*np == '\n')
			    *np = '\0';

			if (cct++ == 0)
			{
			    char	tbuf[BUFSIZ];

			    sprintf(tbuf, " %s", ep->name);
			    addcomment(context, tbuf);
			}
			addcomment(context, buf+2);
		    }
		    break;

		case 'E': 
                    /* if we hit a "#E" at this point, then it must be
		       the one that closes the "#B <patfield>" that we
		       are reading (assuming a syntactically correct file) */
                    endpattern=1;
                    break;

		case 'I':
		    xoff = yoff = rotate = loadtime = 0;
                    flip = 1;
		    (void) sscanf(buf+2, " %s %ld %ld %d %d %ld", incl, 
				  &xoff, &yoff, &rotate, &flip, &loadtime); 

                    /* if included pattern begins with ':' then assume
		       it's in the file being read */
                    if (incl[0]==':')
		    {
			strcpy(tmpstring,patfile);
			strcat(tmpstring,incl);
			strcpy(incl,tmpstring);
                    }
		    else
		    {
			/* if relative path given, assume directory of parent */
			if (!strchr("/~",incl[0]))
			{
			    strcpy(pardir,patfile);
			    seppatdir(pardir,tmpstring);
			    relpathoffset=strlen(pardir);
			    strcat(pardir,incl);
			    strcpy(incl,pardir);
			}
                    }

		    newloc = apply_rotation(location, flip, rotate);
		    newloc.xtrans = TX(location,xoff,yoff);
		    newloc.ytrans = TY(location,xoff,yoff);

                    add_loadreq(loadqueue,
				context->generations + loadtime,
				incl, relpathoffset, 
				&newloc);
		    break;

		case 'L':
		    drawlegend(context, 
			       location.xtrans + tx(xoff,yoff,location.rxx,location.rxy), 
			       location.ytrans + ty(xoff,yoff,location.ryx,location.ryy),
			       buf + 3);
		    break;

		case 'D':
		    loadmode = M_DELTA;
		    xoff = yoff = 0;
		    break;
		    
		case 'M':
		    loadmode = M_RUNLENGTH;
		    xoff = yoff = 0;
		    colct=0;
		    linect=0;
		    anz=0;
		    break;
		    
		case 'K':
		    buf[strlen(buf)-1] = '\0';
		    context->outcome = strdup(buf+2);
		    break;

		case 'N':
		    buf[strlen(buf)-1] = '\0';
		    context->pattern_name = strdup(buf+2);
		    break;

		case 'p':		/* old P format */
		    loadmode = M_PICTURE;
		    xoff = yoff = 0;
		    (void) sscanf(buf+2, " %ld %ld", &xoff, &yoff); 
                    linect = 0;	 /* this is the difference */
		    break;
		    
		case 'P':
		    loadmode = M_PICTURE;
		    xoff = yoff = 0;
		    (void) sscanf(buf+2, " %ld %ld", &xoff, &yoff); 
                    linect = -1;
		    break;
		    
		case 'R':
		    loadmode = M_RELATIVE;
		    xoff = yoff = 0;
		    (void) sscanf(buf+2, " %ld %ld", &xoff, &yoff);
		    break;

		case 'T':
		    modify_rules(buf + 3, context);
		    break;

		case 'U':
#if STATEBITS > 1
		    for (cp = buf + 2; isspace(*cp); cp++)
			continue;
		    cp[strlen(cp) - 1] = '\0';
		    use_rules(cp, context);
#endif
		    break;
		    
		default:
		    break;
		}
	    }
#if STATEBITS > 1
	    else if (loadmode == M_ABSOLUTE && context->maxstates > 2
		     && sscanf(buf,"%ld %ld %d\n",&x,&y,&state) == 3
		     && state)
	    {
		chgcell(context, 
			xorigin + tx(x,y,location.rxx,location.rxy),
			yorigin + ty(x,y,location.ryx,location.ryy),
			PRESENT,
			state);
	    }
#endif
	    else if (loadmode == M_ABSOLUTE && sscanf(buf,"%ld %ld\n",&x,&y)==2)
	    {
		chgcell(context,
			xorigin + tx(x,y,location.rxx,location.rxy),
			yorigin + ty(x,y,location.ryx,location.ryy),
			PRESENT,
			1);
	    }
#if STATEBITS > 1
	    else if (loadmode == M_DELTA && context->maxstates > 2
		     && sscanf(buf,"%ld %ld %d\n",&x,&y,&state) == 3
		     && state)
	    {
		chgcell(context,
			xorigin + tx(xoff+x,yoff+y,location.rxx,location.rxy), 
                        yorigin + ty(xoff+x,yoff+y,location.ryx,location.ryy),
			PRESENT,
			state);
		xoff += x;
		yoff += y;
	    }
#endif
	    else if (loadmode == M_DELTA && sscanf(buf,"%ld %ld\n",&x,&y)==2)
	    {
		chgcell(context,
			xorigin + tx(xoff+x,yoff+y,location.rxx,location.rxy),
			yorigin + ty(xoff+x,yoff+y,location.ryx,location.ryy),
			PRESENT,
			1);
		xoff += x;
		yoff += y;
	    }
#if STATEBITS > 1
	    else if (loadmode == M_RELATIVE && context->maxstates > 2
		     && sscanf(buf,"%ld %ld %d\n",&x,&y,&state) == 3
		     && state)
	    {
		chgcell(context,
			TX(location,xoff+x,yoff+y), 
                        TY(location,xoff+x,yoff+y),
			PRESENT,
			state);
	    }
#endif
	    else if (loadmode == M_RELATIVE && sscanf(buf,"%ld %ld\n",&x,&y)==2)
	    {
		chgcell(context,
			TX(location,xoff+x,yoff+y), 
                        TY(location,xoff+x,yoff+y),
			PRESENT,
			1);
	    }
	    else if (loadmode == M_RUNLENGTH)
	    {
		char	*cp;
		int  n=anz, i=colct;

		for (cp = buf; *cp && *cp!= '!' ; cp++)
		    switch(*cp)
		    {
		    case 'o':
			if (!n)
			    n=1;
			i+=n;
			for (;n;n--)
			{
			    chgcell(context,
				    TX(location,xoff+i-n, yoff+linect),
				    TY(location,xoff+i-n, yoff+linect),
				    PRESENT,
				    1);
			}
			break;
		    case 'b':
			if (!n)
			    n=1;
			i+=n;
			n=0;
			break;
		    case '$':
			if (!n)
			    n=1;
			linect+=n;
			i = 0;
			n=0;  break;
		    case '0':   case '1':   case '2':   case '3':   case '4':
		    case '5':   case '6':   case '7':   case '8':   case '9':
			n= 10*n + *cp -'0';
			break;
		    case ' ':   case '\n':  case '\t': case '\r':
			break;
		    default:
			*cp = '!';  cp--;
		    } 
		colct = i;
		anz=n;
		if (!*cp)
		    linect--;
	    }
	    else		/* loadmode == M_PICTURE */
	    {
		char	*cp;

		for (cp = buf; *cp; cp++)
		    if (*cp == '*')
		    {
			chgcell(context,
				TX(location,xoff+cp-buf, yoff+linect),
		                TY(location,xoff+cp-buf, yoff+linect),
				PRESENT,
				1);
		    }
#if STATEBITS > 1
		    else if (isdigit(*cp))
		    {
			chgcell(context,
				TX(location,xoff+cp-buf, yoff+linect),
		                TY(location,xoff+cp-buf, yoff+linect),
				PRESENT,
				stoi(*cp));
		    }
#endif /* STATEBITS */
	    }
	    linect++;
	}
	fclose(loadfl);

	return 1;
    } 
    else
	return 0;
}

void freeloadscript(void)
/* reinitialize load script to one self-connected cell */
{
    LoadReq *nptr,*ptr;

    for (ptr=loadscript->next; ptr!=loadscript;)
    {
	nptr=ptr->next;
	free(ptr);
	ptr=nptr;
    }
    loadscript->next=loadscript; 

}

void saveloadscript(pattern *context,
		    const char *file, coord_t savex, coord_t savey)
/*  dump the current loadscript as a sequence of includes */
{
    FILE *savefl;
    char outbuf[100];
    int rot,flip;  
    lifetime_t mingener;
    LoadReq *ptr;

    strcpy(outbuf,checktilda(file));
    strcat(outbuf,addlifeext(outbuf));
    if ((savefl = fopen(outbuf,"w")) != NULL)
    {
	int n;

	fputs(SAVE_VERSION "\n", savefl);
	for (n = 0; n < context->ncomments; n++)
	    fprintf(savefl,"#C %s\n", context->comments[n]);

	/* write load script */ 
	/* first, find minimum generation at which a pattern is
	   to be included so include times can be normalized */
	mingener=context->generations;
	for (ptr=loadscript->next; ptr!=loadscript;)
	{
	    ptr=ptr->next;
	    if (mingener>ptr->loadtime) mingener=ptr->loadtime; 
	}
	/* now output file */
	for (ptr=loadscript->next; ptr!=loadscript;)
	{
	    dihedral rotflip;

	    ptr=ptr->next;
	    rotflip = motion2dihedral(ptr->at);
	    fprintf(savefl,"#I %s	%4ld %4ld %2d %2d %ld\n",
		    ptr->patname, 
		    ptr->at.xtrans-savex, 
		    ptr->at.ytrans-savey, 
		    rotflip.rotation, rotflip.flip, 
		    ptr->loadtime-mingener);
	}

	/* close file */
	fclose(savefl);
    }
    else
	error(strerror(errno));
}

static void add_include_entry(char *patname, lifetime_t loadtime, motion *at)
{
    LoadReq *newreq;

    /* Create new include block and load it up */
    newreq=(LoadReq *)malloc(sizeof(LoadReq)); 
    strncpy(newreq->patname,patname,PATNAMESIZ);
    newreq->loadtime = loadtime;
    newreq->at = *at;

    /* Insert new request at end of list */
    newreq->next = loadscript->next;
    loadscript->next = newreq; 
    loadscript = newreq;
}

void confirmload(pattern *from, pattern *to)
{
    tile *ptr;
    int dx,dy;
    coord_t x,y;

    if (!from->tiles)
	return;

    if (from->pattern_name)
    {
        /* add completed load to load script */
	add_include_entry(from->pattern_name,
			  to->generations - from->generations,
			  &from->at);
    }
    /* officially add loaded cells with given transformations */
    for (ptr = from->tiles; ptr != NULL; ptr = ptr->next)
    {
	int	state;

	if (!ptr->dead)
	    for (dx = 0; dx < BOXSIZE; dx++)
		for (dy = 0; dy < BOXSIZE; dy++)
		    if ((state = getcell(from, &ptr->cells, dx, dy, PRESENT)))
		    {
			x=ptr->x+dx;
			y=ptr->y+dy;
			chgcell(to, TX(from->at,x,y),TY(from->at,x,y), PRESENT, state);
		    }
    }

    clear_pattern(from);
}

void loadfile(const char *file, pattern *context, coord_t x, coord_t y)
{
    char outbuf[100],tmp[PATNAMESIZ];
    int quitload=0;
    char thispat[PATNAMESIZ],badpat[PATNAMESIZ], *cp;
    LoadReq *loadqueue=NULL;
    entry *ep;
  
    strcpy(outbuf,checktilda(file));

    /* redefine user load directory (but not pattern library) */
    strcpy(loadirbuf,outbuf);
    seppatdir(loadirbuf,tmp);

    /* don't need to add life extension now, since it will be added in
       do_loadfile  */

    context->obasex = context->at.xtrans = x;
    context->obasey = context->at.ytrans = y;
    resetload(context);

    /* clear dictionary reference counts (helps with comment display) */
    for (ep = dictionary; ep; ep = ep->next)
	ep->refcount = 0;

    /* put the top level load request on the queue */
    add_loadreq(&loadqueue, 0, outbuf, 0, &motion_unity); 

    /* clear the result context */
    initcells(context);

    /* set the name field of the resulting context */
    if ((cp = strrchr(loadqueue->patname, '/')) == (char *)NULL)
	cp = loadqueue->patname;
    else
	++cp;
    context->pattern_name = strdup(cp);
    cp = context->pattern_name + strlen(context->pattern_name) - LOADEXTLEN;
    if (!strcmp(cp, LOADEXT))
	*cp = '\0';

    /* process the load queue */
    badpat[0]='\0';
    while (loadqueue!=NULL)
    {
	char progress[PATNAMESIZ+30];

	while (loadqueue->loadtime > context->generations)
	{
	    if (divis(loadqueue->loadtime-context->generations,100))
	    {
		sprintf(progress,
			"Generating: %ld steps left",
			loadqueue->loadtime-context->generations);
		announce(progress);
	    }
	    generate(context);
	    if ((quitload = breakreq()))
		break;
	}
	forget(context);	/* keeps incremental display from losing */

	strcpy(thispat,loadqueue->patname+loadqueue->relpath);
	sprintf(progress, "Loading %s", thispat);
	announce(progress);
	if (!do_loadfile(&loadqueue, context))
	    strcpy(badpat,thispat);
	if ((quitload = (quitload || breakreq())))
	    break;
    }

    bounding_box(context);
    context->generations = 0;

    if (badpat[0] == '\0')
	announce("");
    else
    {
	char	errmsg[11 + PATNAMESIZ];
	
	strcpy(errmsg, "Can't load ");
	strcat(errmsg, badpat);
	error(errmsg);
    }
}
			
#if STATEBITS > 1

int use_rules(char *rulefile, pattern *context)
/* load a given ruleset by name */
{
    char	*errmsg, *fullpath, patfile[PATNAMESIZ+80];

    strcpy(patfile, rulefile);
    strcat(patfile, ".r");
    if ((fullpath = findfile(patfile)) == (char *)NULL)
    {
	sprintf(patfile, "Can't find rule file %s\n", rulefile);
	error(patfile);
	return(-1);
    }
    else if ((errmsg = readrules(fullpath, context)) != (char *)NULL)
    {
	error(errmsg);
	return(-1);
    }
    else
	return(0);
}

#endif /* STATEBITS > 1 */
