/*
  Xinvest Copyright (c)1997-2000 
  Mark Buser, All Rights Reserved.

  Permission is hereby granted to copy
  and freely distribute copies of this
  program for non-commercial purposes
  without fee, provided that this notice
  appears in all copies.
  
  All redistributions must be in their
  entirety as originally distributed.
  Feel free to modify Xinvest, but
  modified versions may not be distributed
  without prior consent of the author.
  
  This software is provided 'as-is'
  without any express or implied warranty.
  In no event will the author be held
  liable for any damages resulting from
  the use of this software.

  $Revision: 1.25 $ $Date: 2000/02/04 00:39:02 $
*/

#include <float.h>
#include <math.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include <Xm/XmAll.h>

#include "account.h"
#include "calendar.h"
#include "color.h"
#include "nav.h"
#include "pixmap.h"
#include "pref.h"
#include "rate.h"
#include "report.h"
#include "reportP.h"
#include "resource.h"
#include "status.h"
#include "tooltips.h"
#include "trans.h"
#include "util.h"
#include "xutil.h"

/*
** Globals (all static)
*/
static Widget reportDialog = (Widget)NULL;
static Widget reportDrawing = (Widget)NULL;

static int reportAccount = 0;
static int reportTime = 0;
static int reportValue = 0;

static char datestr[40];

static struct daterange {
  long start;
  long end;
} reportDateRange[REPORT_TIMES];

static Calendar reportCal;        /* Current calendar */
static long Customstart;          /* Custom start and end dates. Copied into */
static long Customend;            /* reportDateRange in reportGenerate() */
static int reportResource;        /* Number used to query report resources */

/* 
** Forwards
*/
static void reportDraw (Widget, XtPointer, XtPointer);


/* Initialize reports */
void reportInit (Widget Toplevel, char *appname)
{
  /* propnames must match names in Xinvest.ad */
  char propnames[][MAXRESCHAR] = { "value", "price", "share", "dValue", 
                                   "dPrice", "dShare", "cost", "debit", "fee",
				   "tr", "irr", "yield", "gain", "dist",
                                   "na", "port"};

  /* Read any non-application/non-widget resources */
  reportResource = readResourceDatabase (Toplevel, appname, "Report", "report", 
                                         propnames, XtNumber(propnames));
}


/* Initialize calendar */
static void reportCustCalInit()
{
  Customstart = convertDate(activeAccount(), FIRST_DATE);
  Customend = convertDate(activeAccount(), CURRENT_DATE);
}

/* Sets/clears time frames to report on */
/* ARGSUSED */
void reportSetTime (Widget w, int which, XmToggleButtonCallbackStruct *calldata)
{
  static Boolean first = True;
  Widget ok;
  int row_num;

  /* Distinguish between short and long term rows of time frames */
  XtVaGetValues (XtParent(w), XmNuserData, &row_num, NULL);
  if (row_num == 1) which += (REPORT_TIMES/2);

  if (calldata->set == XmSET) {
    reportTime |= (1 << which);
    if (which == TIME_CUSTOM) {
      XtVaGetValues (w, XmNuserData, &ok, NULL);
      XtSetSensitive (ok, True);

      if (first) {
        reportCustCalInit();
        first = False;
      }
    }
  } else {
    reportTime &= ~(1 << which);
    if (which == TIME_CUSTOM) {
      XtVaGetValues (w, XmNuserData, &ok, NULL);
      XtSetSensitive (ok, False);
    }
  }

  XtVaGetValues (XtParent(XtParent(w)), XmNuserData, &ok, NULL);
  if (ok && reportAccount && reportTime)
    XtSetSensitive (ok, True);
  else
    XtSetSensitive (ok, False);

}

/* Sets/clears report options */
/* ARGSUSED */
void reportSetValue (Widget w, int which, 
                     XmToggleButtonCallbackStruct *calldata)
{
  if (calldata->set == XmSET)
    reportValue |= (1 << which);
  else
    reportValue &= ~(1 << which);
}

/* Sets/clears accounts to report on */
/* ARGSUSED */
void reportSetAccount (Widget w, int which, 
                       XmToggleButtonCallbackStruct *calldata)
{
  Widget button;
  char name[10];

  if (calldata->set == XmSET) {
    reportAccount |= (1 << which);

    /* Can't have 'this account' AND 'all accounts' set simultaneously */
    if (which == 0 || which == 1) {
      reportAccount &= ~(1 << (1-which));
      sprintf (name, "button_%d", 1-which);
      if ((button = XtNameToWidget (XtParent(w), name)) != (Widget)NULL)
        XmToggleButtonSetState (button, False, False);
    }
  } else {
    reportAccount &= ~(1 << which);
  }

  XtVaGetValues (XtParent(w), XmNuserData, &button, NULL);
  if (button && reportAccount && reportTime)
    XtSetSensitive (button, True);
  else
    XtSetSensitive (button, False);
}


/* Redisplay the custom calendar and/or report.  Useful on account switch */
void displayReport (int account)
{
  void *trans;

  /* If user hasn't touched calendar, change default dates to new account */
  if (!calUserTouched(&reportCal)) {
    getAccount (account, ACCOUNT_TRANS, &trans);
    if (trans) {
      if (Customstart == -1)   /* Displaying start */
        calLtoCal (&reportCal, GetTransLDate (trans, 1));
      else                     /* Displaying end */
        calLtoCal (&reportCal, convertDate( account, CURRENT_DATE));
    }
    reportCustCalInit();
  }
  calSetDate (&reportCal);

  /* Force new drawing if in single window mode */
  if (reportDrawing && reportValue & REPORT_SINGLEWIN_ON)
    reportDraw (reportDrawing, (XtPointer)ALLOW, (XtPointer)NULL);
}

void reportCustomStartEnd (Widget w, XtPointer client_data, 
                           XmAnyCallbackStruct *call_data)
{
  long date = calCaltoL (&reportCal);

  if ((int)client_data == 0) {
    Customend = date;                     /* Save end */
    calLtoCal (&reportCal, Customstart);  /* Display start */
    Customstart = -1L;                    /* Start is in reportCal */
  } else {
    Customstart = date;
    calLtoCal (&reportCal, Customend);
    Customend = -1L;
  }
   
  calSetDate (&reportCal);
}

static void reportCustomWarp (Widget w, XtPointer client_data,
                       XmPushButtonCallbackStruct *call_data)
{
  int account = activeAccount();
  void *trans;
  int numTrans, i;

  long currentDate = calCaltoL (&reportCal);
  long nextDate;

  getAccount (account, ACCOUNT_TRANS, &trans);
  getAccount (account, ACCOUNT_NUM_TRANS, &numTrans);

  if (trans == NULL)
    return;

  switch ((int) client_data) {
  
  case 0: /* previous transaction */
          for (i=numTrans; i >= 1; i--) {
            nextDate = GetTransLDate (trans, i);
            if (nextDate < currentDate)
              break;
          }
          break;
  case 1: /* next transaction */
          for (i=1; i <= numTrans; i++) {
            nextDate = GetTransLDate (trans, i);
            if (nextDate > currentDate)
              break;
          }
          break;
  case 2: /* first transaction */
          nextDate = GetTransLDate (trans, 1);
          break;
  case 3: /* last transaction */
          nextDate = GetTransLDate (trans, numTrans);
          break;
  default: break;
  }

  if (nextDate && nextDate != currentDate) {
    calLtoCal (&reportCal, nextDate);
    calSetDate (&reportCal);
  }
}

void reportCustomTime (Widget w, XtPointer client_data, 
                       XmPushButtonCallbackStruct *call_data)
{
  static Widget form = (Widget)NULL;
  Widget calendar, optmenu, button1, button2;
  Widget reportform = (Widget)client_data;
  Boolean mapped;

  if (form == (Widget)NULL) {

    /* Starting date */
    calLtoCal (&reportCal, convertDate (activeAccount(), 
                                        reportDateRange[TIME_CUSTOM].start));

    /* Same size as parent */
    form = XtVaCreateWidget ("reportform2", 
                       xmFormWidgetClass, XtParent(reportform),
                       XmNmappedWhenManaged, False,
                       XmNtopAttachment, XmATTACH_FORM,
                       XmNleftAttachment, XmATTACH_FORM,
                       XmNrightAttachment, XmATTACH_FORM,
                       XmNbottomAttachment, XmATTACH_FORM,
                       NULL);
    calendar = calCreate (form, &reportCal);
    XtVaSetValues (calendar, 
                       XmNtopAttachment, XmATTACH_FORM,
                       XmNleftAttachment, XmATTACH_FORM,
                   NULL);
    XtManageChild (calendar);
    optmenu = XmVaCreateSimpleOptionMenu (form, "reportoptmenu",
                       NULL, (KeySym) NULL, 0,
                       (XtCallbackProc) reportCustomStartEnd,
                       XmVaPUSHBUTTON, NULL, NULL, NULL, NULL,
                       XmVaPUSHBUTTON, NULL, NULL, NULL, NULL,
                       XmNtopAttachment, XmATTACH_FORM,
                       XmNleftAttachment, XmATTACH_WIDGET,
                       XmNleftWidget, calendar,
                       NULL);
    XtDestroyWidget(XmOptionLabelGadget(optmenu));
    XtManageChild (optmenu);
    /* previous transaction button */
    button1 =  XtVaCreateManagedWidget ("prevTrans",
         xmPushButtonWidgetClass, form,
         XmNlabelType,       XmPIXMAP,
         XmNlabelPixmap,     GetPixmap(PPREVTRANS, NORMAL),
         XmNlabelInsensitivePixmap,   GetPixmap(PPREVTRANS, NORMAL),
         XmNsensitive,       True,
         XmNtopAttachment,   XmATTACH_WIDGET,
         XmNtopWidget,       optmenu,
         XmNleftAttachment,  XmATTACH_WIDGET,
         XmNleftWidget,      calendar,
         NULL);
    XtAddCallback (button1, XmNactivateCallback, 
                   (XtCallbackProc)reportCustomWarp, (XtPointer)0 );
    installToolTips (button1);
    /* next transaction button */
    button2 = XtVaCreateManagedWidget ("nextTrans",
         xmPushButtonWidgetClass,  form,
         XmNlabelType,             XmPIXMAP,
         XmNlabelPixmap,           GetPixmap(PNEXTTRANS, NORMAL),
         XmNlabelInsensitivePixmap,   GetPixmap(PNEXTTRANS, NORMAL),
         XmNsensitive,             True,
         XmNtopAttachment,         XmATTACH_WIDGET,
         XmNtopWidget,             optmenu,
         XmNleftAttachment,        XmATTACH_WIDGET,
         XmNleftWidget,            button1,
         NULL);
    XtAddCallback (button2, XmNactivateCallback, 
                   (XtCallbackProc)reportCustomWarp, (XtPointer)1 );
    installToolTips (button2);
    /* 1st transaction button */ 
    button1 =  XtVaCreateManagedWidget ("firstTrans",
         xmPushButtonWidgetClass, form,
         XmNlabelType,            XmPIXMAP,
         XmNlabelPixmap,          GetPixmap(PFIRST, NORMAL),
         XmNlabelInsensitivePixmap,   GetPixmap(PFIRST, NORMAL),
         XmNsensitive,            True,
         XmNtopAttachment,        XmATTACH_WIDGET,
         XmNtopWidget,            button1,
         XmNleftAttachment,       XmATTACH_WIDGET,
         XmNleftWidget,           calendar,
         NULL);
    XtAddCallback (button1, XmNactivateCallback, 
                   (XtCallbackProc)reportCustomWarp, (XtPointer)2 );
    installToolTips (button1);
    /* last transaction button */
    button2 =  XtVaCreateManagedWidget ("lastTrans",
         xmPushButtonWidgetClass,  form,
         XmNlabelType,             XmPIXMAP,
         XmNlabelPixmap,           GetPixmap(PLAST, NORMAL),
         XmNlabelInsensitivePixmap,   GetPixmap(PLAST, NORMAL),
         XmNsensitive,             True,
         XmNtopAttachment,         XmATTACH_OPPOSITE_WIDGET,
         XmNtopWidget,             button1,
         XmNleftAttachment,        XmATTACH_WIDGET,
         XmNleftWidget,            button1,
         NULL);
    XtAddCallback (button2, XmNactivateCallback, 
                   (XtCallbackProc)reportCustomWarp, (XtPointer)3 );
    installToolTips (button2);

    /* Page return button */
    button1 = XtVaCreateManagedWidget (
                "Screenbut", xmPushButtonWidgetClass, form,
                XmNlabelType,    XmPIXMAP,
                XmNlabelPixmap,  GetPixmap(PPREV,NORMAL),
                XmNlabelInsensitivePixmap, GetPixmap(PPREV,INSENS),
                XmNrightAttachment, XmATTACH_FORM,
                XmNbottomAttachment, XmATTACH_FORM,
                NULL);
    installToolTips (button1);
    XtAddCallback (button1, XmNactivateCallback, 
                   (XtCallbackProc)reportCustomTime, (XtPointer)reportform);

    XtManageChild (form);

    XtVaSetValues (form, XmNnoResize, True, NULL);
    CenterWidget (XtParent(form), form);

  }

  XtVaGetValues (form, XmNmappedWhenManaged, &mapped, NULL);
  if (mapped == False) {
    /* Turn on second page */
    XtSetMappedWhenManaged (reportform, False);
    XtSetMappedWhenManaged (form, True);
  } else {
    /* Return to 1st page */
    XtSetMappedWhenManaged (reportform, True);
    XtSetMappedWhenManaged (form, False);
  }
}

struct report_attrib {
 Display *dpy;
 XFontStruct *font, *sfont;
 GC gc;
 Pixmap pix;
 int x, y;
 Pixel fg, bg, shad, hilite, tshad;
 int shadThick;
 int avgWidth;
} report_attrib;

static int reportDrawArrow (double val, char *caption,
		            int pos_x, int pos_y, int size,
		            struct report_attrib *attrib)
{
  int one_third, one_half, two_third;
  int symbol_border = (int)(0.1*size);
  int symbol_size =   (int)(0.8*size);
  XPoint point[7]; 
  int neg = (val < 0);
 
  one_third = symbol_size/3;
  two_third = one_third *2;
  one_half = symbol_size/2;
 
  /******************
           ^6
         /   \
      0--1   4--5
         |___|
         2   3
  *******************/
  point[0].x = pos_x;
  point[0].y = (neg) ? (pos_y + symbol_border + one_third +1):
                       (pos_y + symbol_border + two_third +1);
  point[1].x = one_third;
  point[1].y = 0;
  point[2].x = 0;
  point[2].y = (neg)?-one_third:one_third;
  point[3].x = one_third;
  point[3].y = 0;
  point[4].x = 0;
  point[4].y = -point[2].y;
  point[5].x = point[1].x;
  point[5].y = 0;
  point[6].x = -one_half;
  point[6].y = (neg)?two_third:-two_third;

  XSetForeground (attrib->dpy, attrib->gc, attrib->shad );
  XFillPolygon (attrib->dpy, attrib->pix, attrib->gc,
                point, 7, Convex, CoordModePrevious );

  point[0].x--;
  point[0].y--;
  if (neg)
    XSetForeground (attrib->dpy, attrib->gc, GetColor(RED) );
  else
    XSetForeground (attrib->dpy, attrib->gc, GetColor(GREEN) );
  if (val == 0.0)
    XSetForeground (attrib->dpy, attrib->gc, GetColor(GREY) );

  XFillPolygon (attrib->dpy, attrib->pix, attrib->gc,
                point, 7, Convex, CoordModePrevious );

  XSetForeground (attrib->dpy, attrib->gc, attrib->fg );

  /* Draw caption */
  if (caption) {
    int adjust_x = XTextWidth (attrib->font, caption, strlen(caption))/2;
    int adjust_y = (neg)?(attrib->font->ascent):(-attrib->font->descent);

    XSetFont (attrib->dpy, attrib->gc, attrib->font->fid);
    XSetForeground (attrib->dpy, attrib->gc, attrib->shad );
    XDrawString (attrib->dpy, attrib->pix, attrib->gc,
               pos_x + symbol_size/2 + attrib->shadThick - adjust_x, 
               point[0].y + adjust_y + attrib->shadThick,
               caption, strlen (caption) );

    XSetForeground ( attrib->dpy, attrib->gc, attrib->fg );
    XDrawString (attrib->dpy, attrib->pix, attrib->gc,
               pos_x + symbol_size/2 - adjust_x, 
	       point[0].y + adjust_y,
               caption, strlen (caption) );
  }
  return (symbol_size);
}


static void reportRenderText (char *valhdr, char *valval, int inc_x, int adjust,
		              struct report_attrib *attrib)
{ 
  int adjust_x = 0, adjust_y = attrib->sfont->ascent;
  int inc_y = attrib->font->ascent + attrib->font->descent
	      + attrib->sfont->ascent + attrib->sfont->descent;

  /* Draw heading */
  if (strlen(valhdr)) {
    if (adjust) { /* Center justify */
      adjust_x = inc_x*attrib->avgWidth - 
	       XTextWidth(attrib->sfont, valhdr, strlen(valhdr));
      if (adjust_x >= 0) 
        adjust_x /= 2;
      else
        adjust_x = 5;
    }
    XSetFont (attrib->dpy, attrib->gc, attrib->sfont->fid);
    XSetForeground (attrib->dpy, attrib->gc, attrib->shad );
    XDrawString (attrib->dpy, attrib->pix, attrib->gc,
                 attrib->x + adjust_x + attrib->shadThick, 
                 attrib->y + adjust_y + attrib->shadThick,
                 valhdr, strlen (valhdr) );

    XSetForeground (attrib->dpy, attrib->gc, attrib->hilite );
    XDrawString (attrib->dpy, attrib->pix, attrib->gc,
                 attrib->x + adjust_x, attrib->y + adjust_y,
                 valhdr, strlen (valhdr) );
  }
  /* Draw value */
  if (strlen(valval)) {
    if (adjust) { /* Right justify */
      adjust_x = XTextWidth (attrib->font, valval, strlen(valval));
      adjust_x -=  inc_x*attrib->avgWidth;
      if (adjust_x >= 0) 
        adjust_x = 0;
    }
    XSetFont (attrib->dpy, attrib->gc, attrib->font->fid);
    XSetForeground (attrib->dpy, attrib->gc, attrib->shad );
    XDrawString (attrib->dpy, attrib->pix, attrib->gc,
                 attrib->x + attrib->shadThick - adjust_x, 
                 attrib->y + adjust_y + attrib->font->ascent +attrib->shadThick,
                 valval, strlen (valval) );
    XSetForeground ( attrib->dpy, attrib->gc, attrib->fg );
    XDrawString (attrib->dpy, attrib->pix, attrib->gc,
                 attrib->x - adjust_x, 
  	         attrib->y + adjust_y + attrib->font->ascent,
                 valval, strlen (valval) );
  }
  /* Draw vertical border to right of text */
  if (adjust) {
    XSetForeground (attrib->dpy, attrib->gc, attrib->tshad);
    XDrawLine (attrib->dpy, attrib->pix, attrib->gc, 
               attrib->x + inc_x*attrib->avgWidth, attrib->y, 
	       attrib->x + inc_x*attrib->avgWidth, attrib->y + inc_y);
  }
}

static void reportSummary (int account, struct report_attrib *attrib)
{
  int k; 
  int inc_x, inc_y;
  char valval[80], valhdr[80];

  double price = 0.0;
  double value = 0.0;
  double shares = 0.0;
  long daymin = LONG_MAX; long daymax = 0L;

  /* Calculate values  */
  if (account == REPORT_PORTFOLIO_ACCOUNT) {
    for (k=1; k <= numAccounts(); k++) {
      if (convertDate (k, LAST_DATE) > daymax)
        daymax = convertDate (k, CURRENT_DATE);
      if (convertDate(k, FIRST_DATE) < daymin)
        daymin = convertDate(k, FIRST_DATE);
      value    += rate_value (k, FIRST_DATE, convertDate(k, CURRENT_DATE) );
    }
  } else {
    daymax   = convertDate (account, CURRENT_DATE);
    daymin   = convertDate (account, FIRST_DATE);
    value    = rate_value (account, FIRST_DATE, daymax);
    shares   = rate_shares (account, FIRST_DATE, daymax);
    price    = rate_price (account, daymax);
  }

  attrib->y += (attrib->font->ascent + attrib->font->descent); 

  /* TITLE */
  {
    char title[40];

    inc_x = 0;
    inc_y = attrib->font->ascent + attrib->font->descent;
    attrib->x = BORDER/2;

    /* Copy logo to front of title if there is enough vertical room */
    if (inc_y >= 16) {
      /* Not the most robust code, 16's will get us into trouble someday.*/
      XSetClipMask (attrib->dpy, attrib->gc, GetPixmap(PTICON, MASK));
      XSetClipOrigin (attrib->dpy, attrib->gc, attrib->x, attrib->y);
      XCopyArea (attrib->dpy, GetPixmap(PTICON,NORMAL), attrib->pix, attrib->gc,
                 0, 0, 16, 16, attrib->x, attrib->y);
      XSetClipMask (attrib->dpy, attrib->gc, None);
      inc_x = 20;
    }

    if (account == REPORT_PORTFOLIO_ACCOUNT) {
      strcpy (title, getResource(reportResource, REPPORT));
    } else {
      char *name = createAccountName (account);
      sprintf (title, "%-.40s", name);
      XtFree (name);
    }
    XSetForeground (attrib->dpy, attrib->gc, attrib->shad);
    XDrawLine (attrib->dpy, attrib->pix, attrib->gc, 
               attrib->x, attrib->y + inc_y, 
	       BORDER/2+REPORT_CHAR_WIDTH*attrib->avgWidth, attrib->y + inc_y);
    XSetForeground (attrib->dpy, attrib->gc, attrib->fg);
    XDrawLine (attrib->dpy, attrib->pix, attrib->gc, 
	       attrib->x, attrib->y + inc_y - attrib->shadThick, 
	       BORDER/2+REPORT_CHAR_WIDTH*attrib->avgWidth, 
	       attrib->y + inc_y - attrib->shadThick);
    XSetFont (attrib->dpy, attrib->gc, attrib->font->fid);
    XSetForeground (attrib->dpy, attrib->gc, attrib->shad );
    XDrawString (attrib->dpy, attrib->pix, attrib->gc,
                attrib->x + inc_x + attrib->shadThick, 
                attrib->y + attrib->font->ascent + attrib->shadThick,
                title, strlen (title) );
    XSetForeground (attrib->dpy, attrib->gc, attrib->hilite);
    XDrawString (attrib->dpy, attrib->pix, attrib->gc,
                attrib->x + inc_x, attrib->y + attrib->font->ascent,
                title, strlen (title) );

    attrib->y += inc_y;
  }

  inc_y = attrib->font->ascent + attrib->font->descent
	  + attrib->sfont->ascent + attrib->sfont->descent;

  /* VALUE */
  strcpy (valhdr, getResource(reportResource, REPVALUE));
  sprintf (valval, "%.20s", utilCurrOut(value, UTILCURR));
  inc_x = 20; 
  attrib->x = BORDER/2 + 20*attrib->avgWidth;
  reportRenderText (valhdr, valval, inc_x, True, attrib);
  attrib->x += inc_x * attrib->avgWidth;

  /* PRICE */
  if (account == REPORT_PORTFOLIO_ACCOUNT) {
    strcpy (valhdr, getResource(reportResource, REPPRICE));
    sprintf (valval, "%.20s", getResource(reportResource, REPNA));
  } else {
    NAV *navp;
    getAccount (account, ACCOUNT_NAV, &navp);
    if (isNavChanged (account, navp))
      sprintf (valhdr, "%.10s (Live Quote)", 
               getResource(reportResource, REPPRICE));
    else
      strcpy (valhdr, getResource(reportResource, REPPRICE));
    sprintf (valval, "%.20s", utilCurrOut(price, UTILCURR));
  }
  inc_x = 20; 
  reportRenderText (valhdr, valval, inc_x, True, attrib);
  attrib->x += inc_x * attrib->avgWidth;

  /* SHARES */
  strcpy (valhdr, getResource(reportResource, REPSHARE));
  if (account == REPORT_PORTFOLIO_ACCOUNT)
    sprintf (valval, "%.16s", getResource(reportResource, REPNA));
  else
    sprintf (valval, "%.16s", utilCurrOut(shares,UTILNUM));
  inc_x = 16; 
  reportRenderText (valhdr, valval, inc_x, True, attrib);
  attrib->x += inc_x * attrib->avgWidth;

  /* Reduce the clutter by drawing a line */
  XSetForeground (attrib->dpy, attrib->gc, attrib->shad);
  XDrawLine (attrib->dpy, attrib->pix, attrib->gc, 
             BORDER/2 + 20*attrib->avgWidth, attrib->y + inc_y, 
	     BORDER/2 + REPORT_CHAR_WIDTH*attrib->avgWidth, attrib->y + inc_y);
  XSetForeground (attrib->dpy, attrib->gc, attrib->fg);
  XDrawLine (attrib->dpy, attrib->pix, attrib->gc, 
	     BORDER/2 + 20*attrib->avgWidth, 
	     attrib->y + inc_y - attrib->shadThick, 
	     BORDER/2 + REPORT_CHAR_WIDTH*attrib->avgWidth, 
	     attrib->y + inc_y - attrib->shadThick);

  attrib->y += inc_y;
}

static void reportTimeVal (int account, struct report_attrib *attrib)
{
  extern Widget Timesform, Timelform;
  Widget w;

  int i, k; 

  double   price, value, shares, cost, debit, fee, distrib;
  double   prevPrice, prevValue, prevShares, prevCost, prevDebit, prevFee, 
           prevDistrib;

  long     daymin, daymax;

  /* Draw selected times */
  for (i=0; i < REPORT_TIMES; i++) {
    if (reportTime & (1 << i)) {
      char timename[20], *timestr;
      XmString timeframe;

      if (i < REPORT_TIMES/2) {  /* Short time frame form */
        sprintf (timename, "button_%d", i);
        w = XtNameToWidget (Timesform, timename);
      } else {                   /* Long time frame form */
        sprintf (timename, "button_%d", i-REPORT_TIMES/2);
        w = XtNameToWidget (Timelform, timename);
      }
      if (w != (Widget)NULL) {

        XtVaGetValues (w, XmNlabelString, &timeframe, NULL);
        XmStringGetLtoR (timeframe, XmFONTLIST_DEFAULT_TAG, &timestr);

        /* Calculate values  */
        price = 0.0;   prevPrice = 0.0;
        value = 0.0;   prevValue = 0.0;
        cost = 0.0;    prevCost = 0.0;
        debit = 0.0;   prevDebit = 0.0;
        fee = 0.0;     prevFee = 0.0;
        distrib = 0.0; prevDistrib = 0.0;
        shares = 0.0;  prevShares = 0.0;
        daymin = LONG_MAX; daymax = 0L;

        if (account == REPORT_PORTFOLIO_ACCOUNT) {
          for (k=1; k <= numAccounts(); k++) {
            if (convertDate (k, reportDateRange[i].end) > daymax)
              daymax = convertDate (k, reportDateRange[i].end);
            if (convertDate(k, reportDateRange[i].start) < daymin)
              daymin = convertDate(k, reportDateRange[i].start);
            value    += rate_value (k, FIRST_DATE, reportDateRange[i].end );
            prevValue += rate_value (k, FIRST_DATE, daymin-1);
            cost     += rate_cost (k, FIRST_DATE, reportDateRange[i].end );
            prevCost += rate_cost (k, FIRST_DATE, daymin-1);
            debit    += rate_withdrawal (k, FIRST_DATE, reportDateRange[i].end);
            prevDebit += rate_withdrawal (k, FIRST_DATE, daymin-1);
            fee      += rate_fee (k, FIRST_DATE, reportDateRange[i].end); 
            prevFee  += rate_fee (k, FIRST_DATE, daymin-1);
            distrib  += rate_distrib (k, FIRST_DATE, reportDateRange[i].end);
            prevDistrib += rate_distrib( k, FIRST_DATE, daymin-1);
          }
        } else {
          daymax   = convertDate (account, reportDateRange[i].end);
          daymin   = convertDate (account, reportDateRange[i].start);
          value    = rate_value (account, FIRST_DATE, reportDateRange[i].end);
          prevValue = rate_value (account, FIRST_DATE, daymin-1);
          cost     = rate_cost (account, FIRST_DATE, reportDateRange[i].end);
          prevCost = rate_cost (account, FIRST_DATE, daymin-1);
          debit = rate_withdrawal (account, FIRST_DATE, reportDateRange[i].end);
          prevDebit = rate_withdrawal (account, FIRST_DATE, daymin-1);
          fee      = rate_fee (account, FIRST_DATE, reportDateRange[i].end); 
          prevFee  = rate_fee (account, FIRST_DATE, daymin-1);
          distrib  = rate_distrib (account, FIRST_DATE, reportDateRange[i].end);
          prevDistrib = rate_distrib (account, FIRST_DATE, daymin-1);
          shares   = rate_shares (account, FIRST_DATE, reportDateRange[i].end);
          prevShares = rate_shares (account, FIRST_DATE, daymin-1);
          price     = rate_price (account, reportDateRange[i].end);
          /* If we precede first trans, use first trans info to calc price */
          if (convertDate (account, FIRST_DATE) > daymin-1) 
            prevPrice = rate_price (account, FIRST_DATE);
	  else
            prevPrice = rate_price (account, daymin-1);
        }

#ifdef DEBUG
        fprintf( stderr, 
                 "daymax -> %ld\n" "daymin -> %ld\n" "value     -> %f\n"
                 "prevalue  -> %f\n" "cost      -> %f\n" "precost   -> %f\n"
                 "debit     -> %f\n" "predebit  -> %f\n" "fee       -> %f\n"
                 "prefee    -> %f\n" "distrib   -> %f\n" "predistrib-> %f\n"
                 "shares    -> %f\n",
                 daymax, daymin, value, prevValue, cost, prevCost, debit,
                 prevDebit, fee, prevFee, distrib, prevDistrib, shares );
#endif
        /* 
        ** Generate strings 
        */
	     
        /* Draw selected values */
	{
          char valval[80], valhdr[80];
          int inc_x, inc_y;
          YIELD result;

	  inc_y = attrib->font->ascent + attrib->font->descent
	          + attrib->sfont->ascent + attrib->sfont->descent;

          /* 
	  ** TIME FRAME 
	  */
          attrib->x = BORDER/2;
          inc_x = 20; 

	  valhdr[0] = '\0';
          if (i == TIME_CUSTOM) {       /* Custom */
	    char *start;  
	    sprintf (valhdr, "%-.20s", timestr);
            /* utilOutDate uses static return, can't call twice below.*/
            start = XtNewString (utilDateOut(reportDateRange[i].start));
            sprintf (valval, "%.10s - %.10s", start,
                     utilDateOut(reportDateRange[i].end) );
	    valval[20] = '\0';
            XtFree (start);
          } else if (i == TIME_INCEP) { /* Since inception */
	    sprintf (valhdr, "%-.20s", timestr);
            sprintf (valval, "%.20s", utilDateOut(daymin));
	    valval[20] = '\0';
          } else if (i == TIME_YTD) {   /* YTD */
 	    sprintf (valhdr, "%-.20s", timestr);
            if (account == REPORT_PORTFOLIO_ACCOUNT) {
              sprintf (valval, "%.20s", 
		     utilDateOut(convertDate(activeAccount(), CURRENT_DATE)));
	    } else {
              sprintf (valval, "%.20s", 
		       utilDateOut(convertDate(account, CURRENT_DATE)));
	    }
	    valval[20] = '\0';
	  } else {                            /* Others */
            sprintf (valval, "%-.20s", timestr);
	  }
          XtFree (timestr);
          reportRenderText (valhdr, valval, inc_x, False, attrib);
          attrib->x += (inc_x*attrib->avgWidth); 

          /* Transaction details line 1 */

          if (reportValue & REPORT_DETAIL_ON) {
            /* 
	    ** delta VALUE 
	    */
            inc_x = 20; 
            attrib->x = BORDER/2 + 20*attrib->avgWidth;
            strcpy (valhdr, getResource(reportResource, REPDVALUE));
	    sprintf (valval, "%.20s", utilCurrOut(value-prevValue, UTILCURR));
            reportRenderText (valhdr, valval, inc_x, True, attrib);
            attrib->x += (inc_x*attrib->avgWidth); 

            /* 
	    ** delta PRICE 
	    */
            strcpy (valhdr, getResource(reportResource, REPDPRICE));
            if (account == REPORT_PORTFOLIO_ACCOUNT) {
	      sprintf (valval, "%.19s", getResource(reportResource, REPNA));
            } else {
	      char *change;  
              /* utilOutDate uses static return, can't call twice below.*/
              change = XtNewString (utilCurrOut(price-prevPrice, UTILCURR));
	      sprintf (valval, "%.9s %8.8s%%", change,
		       utilCurrOut((price-prevPrice)/prevPrice*100.0, UTILNUM));
              XtFree (change);
            }
            inc_x = 20; 
            reportRenderText (valhdr, valval, inc_x, True, attrib);
            attrib->x += (inc_x*attrib->avgWidth); 

            /* 
	    ** delta SHARES 
	    */
            inc_x = 16; 
            strcpy (valhdr, getResource(reportResource, REPDSHARE));
            if (account == REPORT_PORTFOLIO_ACCOUNT) {
	      sprintf (valval, "%.16s", getResource(reportResource, REPNA));
            } else {
	      sprintf (valval, "%.16s", 
		       utilCurrOut(shares-prevShares,UTILNUM));
            }
            reportRenderText (valhdr, valval, inc_x, True, attrib);
            attrib->x += (inc_x*attrib->avgWidth); 
          }

          /* Transaction details line 2 */

          if (reportValue & REPORT_DETAIL_ON) {
            /* 
	    ** delta COST 
	    */
            attrib->x = BORDER/2 + 20*attrib->avgWidth;
	    attrib->y += inc_y;
            inc_x = 20; 
            strcpy (valhdr, getResource(reportResource, REPCOST));
	    sprintf (valval, "%.20s", utilCurrOut(cost-prevCost, UTILCURR));
            reportRenderText (valhdr, valval, inc_x, True, attrib);
            attrib->x += (inc_x*attrib->avgWidth); 

            /* 
	    ** delta DEBIT 
	    */
            inc_x = 20; 
            strcpy (valhdr, getResource(reportResource, REPDEBIT));
	    sprintf (valval, "%.20s", utilCurrOut(debit-prevDebit, UTILCURR));
            reportRenderText (valhdr, valval, inc_x, True, attrib);
            attrib->x += (inc_x*attrib->avgWidth); 

            /* 
	    ** delta FEE 
	    */
            inc_x = 16; 
            strcpy (valhdr, getResource(reportResource, REPFEE));

            if (fee > 0.0)
              fee = -fee;
            if (prevFee > 0.0)
              prevFee = -prevFee;
	    sprintf (valval, "%.16s", utilCurrOut(fee-prevFee, UTILCURR));
            reportRenderText (valhdr, valval, inc_x, True, attrib);
            attrib->x += (inc_x*attrib->avgWidth); 
	  }

          /* Return line */

          /* 
	  ** TR 
	  */
          attrib->x = BORDER/2 + 20*attrib->avgWidth;
          if (reportValue & (REPORT_DETAIL_ON))
	    attrib->y += inc_y;
          inc_x = 20;   
          strcpy (valhdr, getResource(reportResource, REPTR));
          {
            double rate, annRate;
            double years = (double)(daymax - daymin) / 365.0;
            double dCost = cost;
            double dValue = (value-prevValue) + (debit-prevDebit) -
                            (cost-prevCost);

            /* If we're under one less digit than max user preference
               precision, remove any weird significance behavior.     */
            if (fabs(dValue) < DBL_EPSILON*1.0E+07) 
              dValue = 0.0;

            if (dCost == (double)0.0) {
              if (dValue == (double)0.0)
                rate = annRate = 0.0;
              else
                rate = annRate = LONG_MAX; /* trigger inf. format */
            } else {
              rate = dValue / dCost;
              annRate = (pow (1.0+rate, 1.0/years) - 1.0) * 100.0;
              rate *= 100;
            }
	    sprintf (valval, "%.19s%%", utilCurrOut(rate, UTILNUM));

	    /* If we had good results, let them know */
            if (reportValue & (REPORT_DETAIL_ON))
              reportDrawArrow (rate, valval, BORDER/2 + 5*attrib->avgWidth, 
		               attrib->y-inc_y, 6*attrib->avgWidth, attrib);
          }
          reportRenderText (valhdr, valval, inc_x, True, attrib);
          attrib->x += (inc_x*attrib->avgWidth); 

          /* 
	  ** IRR 
	  */
          inc_x = 20; 
          strcpy (valhdr, getResource(reportResource, REPIRR));
          {
            double rate;

            rateClrAccount();

            if (account != REPORT_PORTFOLIO_ACCOUNT) {
              rate_add_account (account, reportDateRange[i].start,
                                reportDateRange[i].end);
            } else {
              for (k=1; k<=numAccounts(); k++) {
                rate_add_account (k, reportDateRange[i].start,
                                  reportDateRange[i].end);
              }
            }
            rate = rate_IRR (value + (distrib-prevDistrib));
	    sprintf (valval, "%.19s%%", utilCurrOut(rate, UTILNUM));
          }
          reportRenderText (valhdr, valval, inc_x, True, attrib);
          attrib->x += (inc_x*attrib->avgWidth); 

          /* 
	  ** YIELD 
	  */
          inc_x = 16; 
          strcpy (valhdr, getResource(reportResource, REPYIELD));
          if (account != REPORT_PORTFOLIO_ACCOUNT) {
            rate_yield (account, reportDateRange[i].start,
                        reportDateRange[i].end, &result);
          } else {
            YIELD acct;
            double years = (double)(daymax - daymin) / 365.0;
	    result.div = result.value = 0.0;
            for (k = 1; k <= numAccounts(); k++) {
              rate_yield (k, reportDateRange[i].start,
                          reportDateRange[i].end, &acct);
              result.div += acct.div;
              result.value += acct.value;
            }
            /* Guard against divide by zero */
            if (result.value == 0.0) result.value = 1.0;
            result.yield = result.div/result.value;
            result.annYield = (pow (1.0+result.yield, 1.0/years) - 1.0) * 100.0;
            result.yield *= 100.0;
          }
          sprintf (valval, "%.15s%%", utilCurrOut(result.yield, UTILNUM));
          reportRenderText (valhdr, valval, inc_x, True, attrib);
          attrib->x += (inc_x*attrib->avgWidth); 

          /* Return details line */

          if (reportValue & REPORT_RTN_DETAIL_ON) {
            inc_x = 20; 
            attrib->x = BORDER/2 + 20*attrib->avgWidth;
	    attrib->y += inc_y;

            /* 
	    ** delta VALUE due to price change
	    */
            inc_x = 20; 
            strcpy (valhdr, getResource(reportResource, REPGAIN));
	    { 
	      double dValue =  (value-prevValue)+(debit-prevDebit)
                              -(cost-prevCost)-result.div;
              if (fabs(dValue) < DBL_EPSILON*1.0E+07) 
                dValue = 0.0;
              sprintf (valval, "%.20s", utilCurrOut(dValue, UTILCURR));
	    }
            reportRenderText (valhdr, valval, inc_x, True, attrib);
            attrib->x += (inc_x*attrib->avgWidth); 

            /* 
	    ** delta VALUE due to distributions
	    */
            inc_x = 20; 
            strcpy (valhdr, getResource(reportResource, REPDIST));
            sprintf (valval, "%.20s", utilCurrOut(result.div, UTILCURR));
            reportRenderText (valhdr, valval, inc_x, True, attrib);
            attrib->x += (inc_x*attrib->avgWidth); 

	    /* Place holder for possible dividend use */
            inc_x = 16; 
            strcpy (valhdr, "");
            strcpy (valval, "");
            reportRenderText (valhdr, valval, inc_x, True, attrib);
            attrib->x += (inc_x*attrib->avgWidth); 
	  }

	  /* Reduce the clutter by drawing a line after each time frame */
          XSetForeground (attrib->dpy, attrib->gc, attrib->shad);
          XDrawLine (attrib->dpy, attrib->pix, attrib->gc, 
                     BORDER/2 + 20*attrib->avgWidth, attrib->y + inc_y, 
	             REPORT_CHAR_WIDTH*attrib->avgWidth, attrib->y + inc_y);
          XSetForeground (attrib->dpy, attrib->gc, attrib->fg);
          XDrawLine (attrib->dpy, attrib->pix, attrib->gc, 
	             BORDER/2 + 20*attrib->avgWidth, 
		     attrib->y + inc_y - attrib->shadThick, 
	             BORDER/2 + REPORT_CHAR_WIDTH*attrib->avgWidth, 
		     attrib->y + inc_y - attrib->shadThick);

        } /* for values */

        attrib->y += (attrib->font->ascent + attrib->font->descent +
		      attrib->sfont->ascent + attrib->sfont->descent); 

      } /* if widget found */
    } /* if time on */
  } /* for time */
}


/* ARGSUSED */
static void reportRedraw (Widget w, XtPointer client_data, XtPointer call_data)
{
  Dimension width, height;
  struct report_attrib *attrib;

  XtVaGetValues ( w, XmNwidth, &width,
                     XmNheight, &height,
                     XmNuserData, &attrib,
                  NULL);
  if (attrib && attrib->pix && attrib->gc)
    XCopyArea (XtDisplay(w), attrib->pix, XtWindow(w), attrib->gc,
               0, 0, width, height, 0, 0 );
}


/* ARGSUSED */
static void reportDraw (Widget w, XtPointer client_data, XtPointer call_data)
{
  int accounts = 0, times = 0;
  unsigned int inner, outer;

  int i;

  Display *dpy = XtDisplay(w);
  GC gc;
  extern XFontStruct *large, *small;

  Pixmap draw;
  Dimension height, width;

  char atoZ[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz";
  char dig[] = "01234567890$% .,";
  struct report_attrib *attrib;
  attrib = (struct report_attrib *) XtCalloc (1, sizeof(struct report_attrib));
  attrib->font = large;
  attrib->sfont = small;
  attrib->avgWidth = XTextWidth (attrib->font, atoZ, strlen(atoZ));
  attrib->avgWidth += XTextWidth (attrib->font, dig, strlen(dig));
  attrib->avgWidth = attrib->avgWidth / (strlen(atoZ) + strlen(dig));

  /* 
  ** How many lines 
  */
  if (reportAccount & REPORT_ALL_ACCOUNTS)
    accounts += numAccounts();
  if (reportAccount & REPORT_ACTIVE_ACCOUNT)
    accounts++;
  if (reportAccount & REPORT_PORTFOLIO)
    accounts++;

  for (i=0; i < REPORT_TIMES; i++) {
   if (reportTime & (1 << i))
     times++;
  }

  if (accounts == 0 || times == 0) {
    write_status ("No account(s) and/or time frame(s)\n"
                  "selected, report not generated.", ERR);
    return;
  }

  inner =   1*(attrib->font->ascent + attrib->font->descent)
          + 1*(attrib->sfont->ascent + attrib->sfont->descent);
  if (reportValue & REPORT_DETAIL_ON) 
    inner += (2*(attrib->font->ascent + attrib->font->descent) +
              2*(attrib->sfont->ascent + attrib->sfont->descent));
  if (reportValue & REPORT_RTN_DETAIL_ON) 
    inner += (1*(attrib->font->ascent + attrib->font->descent) +
              1*(attrib->sfont->ascent + attrib->sfont->descent));
  outer = 3*(attrib->font->ascent + attrib->font->descent) +
	  1*(attrib->sfont->ascent + attrib->sfont->descent);

  setBusyCursor (w, True);

  /* Create pixmap */
  height = accounts*outer + accounts*times*inner + 
	   (attrib->sfont->ascent + attrib->sfont->descent);
  width = REPORT_CHAR_WIDTH*attrib->avgWidth;
  draw = XCreatePixmap (dpy, RootWindowOfScreen (XtScreen(w)),
                        width+BORDER, height+BORDER,
                        DefaultDepthOfScreen (XtScreen(w)));

#ifdef DEBUG
  fprintf( stderr, 
           "accounts -> %d\n" "times    -> %d\n"
           "inner   -> %d  outer   -> %d\n"
           "attrib sfont ascent -> %d\n" "attrib sfont descent-> %d\n"
           "attrib font ascent  -> %d\n" "attrib font descent -> %d\n"
           "height   -> %d\n" "width    -> %d\n",
           accounts, times, inner, outer, 
	   attrib->sfont->ascent, attrib->sfont->descent,
           attrib->font->ascent, attrib->font->descent, height, width);
#endif

  /* Clear background, copy header */
  XtVaGetValues (w, 
                 XmNbackground, &attrib->bg,
                 XmNforeground, &attrib->fg,
                 XmNhighlightColor, &attrib->hilite,
                 NULL);
  if (attrib->shadThick == 0)
    attrib->shadThick = 1;

  if (isColorVisual(dpy)) {
    Pixel top, bot;

    XmGetColors (XtScreen(w), GetColormap(), attrib->bg,
                 NULL, &top, &bot, NULL);
    if (isDarkPixel (dpy, attrib->fg)) {
      attrib->tshad = bot;
      attrib->shad = top;
    } else {
      attrib->tshad = top;
      attrib->shad = bot;
    }
  } else
    attrib->shad = attrib->bg;

  gc = XCreateGC (dpy, RootWindowOfScreen(XtScreen(w)), 0, NULL);
  XSetLineAttributes (dpy, gc, attrib->shadThick, 
                      LineSolid, CapNotLast, JoinBevel);

  XSetTile (dpy, gc, GetPixmap(PSTIPPLE, NORMAL));
  XSetFillStyle (dpy, gc, FillTiled );
  XFillRectangle (dpy, draw, gc, 0, 0, width+BORDER, height+BORDER);
  XSetFillStyle (dpy, gc, FillSolid);

  XSetForeground (dpy, gc, attrib->bg);
  XFillRectangle (dpy, draw, gc, BORDER/2-4, BORDER/2-4, width+7, height+7); 
  XSetForeground (dpy, gc, attrib->shad);
  XDrawRectangle (dpy, draw, gc, BORDER/2-3, BORDER/2-3, width+4, height+4); 
  XSetForeground (dpy, gc, attrib->fg);
  XDrawRectangle (dpy, draw, gc, BORDER/2-3, BORDER/2-3, width+4, height+4); 


  XSetLineAttributes (dpy, gc, attrib->shadThick, 
                      LineSolid, CapNotLast, JoinBevel);
  
  attrib->dpy = dpy;
  attrib->gc = gc;
  attrib->pix = draw;
  attrib->x = BORDER/2;
  attrib->y = BORDER/2;

  DEBUGXY;
  /* Write report */
  if (reportAccount & REPORT_ACTIVE_ACCOUNT) {
    reportSummary (activeAccount(), attrib);
    DEBUGXY;
    reportTimeVal (activeAccount(), attrib);
    DEBUGXY;
  }
  if (reportAccount & REPORT_ALL_ACCOUNTS) {
    for (i=1; i <= numAccounts(); i++) {
      reportSummary (i, attrib);
      DEBUGXY;
      reportTimeVal (i, attrib);
      DEBUGXY;
    }
  }
  if (reportAccount & REPORT_PORTFOLIO) {
    reportSummary (REPORT_PORTFOLIO_ACCOUNT, attrib);
    reportTimeVal (REPORT_PORTFOLIO_ACCOUNT, attrib);
  }

  /* Time stamp the report */
  {
    extern char version[];
    char line[80];
    sprintf (line, "%s: Generated %s", 
	     version, utilDateOut(reportDateRange[0].end));
    XSetFont (attrib->dpy, attrib->gc, attrib->sfont->fid);
    XSetForeground (attrib->dpy, attrib->gc, attrib->shad );
    XDrawString (attrib->dpy, attrib->pix, attrib->gc,
                 BORDER/2 + attrib->shadThick, 
                 height + BORDER/2 - attrib->shadThick - attrib->sfont->descent,
                 line, strlen (line) );
    XSetForeground (attrib->dpy, attrib->gc, attrib->fg);
    XDrawString (attrib->dpy, attrib->pix, attrib->gc,
                 BORDER/2, 
	         height + BORDER/2 - attrib->sfont->descent,
                 line, strlen (line) );
   }
  width += BORDER;  height += BORDER;

  /* Resize drawing area */
  {
    Dimension rheight, rwidth;
    Dimension dheight, dwidth;

    XtVaGetValues (XtParent(reportDrawing), 
		   XmNwidth, &dwidth, XmNheight, &dheight, 
		   NULL);
    XtVaGetValues (reportDialog, XmNwidth, &rwidth, XmNheight, &rheight, NULL);

    XtVaSetValues (reportDrawing, XmNwidth, width, XmNheight, height, NULL);
    /* Resize dialog to make it fit if allowed to */
    if ((int)client_data == ALLOW) {
      XtVaSetValues (reportDialog, 
		     XmNwidth, width + (rwidth-dwidth), 
		     XmNheight, outer + inner + BORDER + (rheight-dheight) +
	                        attrib->sfont->ascent + attrib->sfont->descent,
		     NULL);
    }
  }

  /* Write to screen */
  XCopyArea (dpy, draw, XtWindow(w), gc, 0, 0, width, height, 0, 0 );

  /* Save for redraw */
  XtVaSetValues (w, XmNuserData, attrib, NULL);

  /* Remove initializer, replace with copier */
  XtRemoveCallback (w, XmNexposeCallback, reportDraw, (XtPointer)ALLOW);
  XtAddCallback (w, XmNexposeCallback, 
                 (XtCallbackProc)reportRedraw, (XtPointer)NULL);

  setBusyCursor (w, False);
}


/* ARGSUSED */
void reportDestroy (Widget w, Widget client_data, XtPointer call_data)
{
  struct report_attrib *attrib;

  XtVaGetValues (client_data, XmNuserData, &attrib, NULL);

  if (attrib) {
    if (attrib->pix)
      XFreePixmap (XtDisplay(w), attrib->pix);
    if (attrib && attrib->gc)
      XFreeGC (XtDisplay(w), attrib->gc);
    XtFree ( (char *)attrib);
  }
  XtDestroyWidget (GetTopShell(w));
  reportDialog = (Widget)NULL;
  reportDrawing = (Widget)NULL;
}


/* ARGSUSED */
void reportGenerate (Widget w, XtPointer client_data, XtPointer calldata)
{
  Widget scroll;
  Widget pane, form, button;

  int i, resize = DISALLOW;

  time_t t = time ((time_t *)NULL);
  struct tm *today = localtime (&t);
  char datebuf[40];
  long ldate;

  /*
  ** Set time ranges 
  */
  strftime (datebuf, 20, "%m/%d/%Y", today);
  ldate = calStrtoL (datebuf);
  sprintf (datestr, " - %.30s", utilDateOut (ldate));

  for (i=0; i < REPORT_TIMES; i++) {
    reportDateRange[i].end = ldate;
 
    switch (i) {
      case 0:  reportDateRange[i].start = ldate - 1L; break;
      case 1:  reportDateRange[i].start = ldate - 7L; break;
      case 2:  reportDateRange[i].start = ldate - 30L; break;
      case 3:  reportDateRange[i].start = ldate - (365L/4L); break;
      case 4:  reportDateRange[i].start = ldate - (long) (today->tm_yday); 
               break;
      case 5:  reportDateRange[i].start = ldate - 365L; break;
      case 6:  reportDateRange[i].start = ldate - 2*365L; break;
      case 7:  reportDateRange[i].start = ldate - 3*365L; break;
      case 8:  reportDateRange[i].start = ldate - 5*365L; break;
      case 9:  reportDateRange[i].start = ldate - 10*365L; break;
      case 10: reportDateRange[i].start = FIRST_DATE; break;
      case 11: if (Customstart == -1L) {
                 reportDateRange[i].start = calCaltoL(&reportCal);
                 reportDateRange[i].end = Customend;
               } else if (Customend == -1L) {
                 reportDateRange[i].start = Customstart;
                 reportDateRange[i].end = calCaltoL(&reportCal);
               } else {
                 reportDateRange[i].start = Customstart;
                 reportDateRange[i].end = Customend;
               }
               break;
    }
  }

  /* Check custom time for errors if enabled */
  i = TIME_CUSTOM;
  if (reportTime & (1 << i)) {
    if (reportDateRange[i].start >= reportDateRange[i].end) {
      write_status ("Custom report period end date\n"
                    "must be later than start date.",
                    ERR);
      return;
    }
  }

  /* 
  ** Create scrolled window dialog if in single window mode and none exists,
  ** or multi-window mode. 
  */
  if (((reportValue & REPORT_SINGLEWIN_ON) && reportDialog == (Widget)NULL)
      || !(reportValue & REPORT_SINGLEWIN_ON)) {
    reportDialog = XtVaCreatePopupShell ("Report", xmDialogShellWidgetClass,
                            GetTopShell (w), NULL );
    resize = ALLOW;
    pane = XtVaCreateWidget ("ReportPane", 
                           xmPanedWindowWidgetClass, reportDialog,
                           XmNsashWidth, 1,
                           XmNsashHeight, 1,
                           NULL ); 
    scroll = XtVaCreateManagedWidget ("ReportScroll", 
                           xmScrolledWindowWidgetClass, pane,
                           XmNscrollingPolicy, XmAUTOMATIC,
                           XmNvisualPolicy,    XmCONSTANT,
                           NULL);
    reportDrawing = XtVaCreateManagedWidget ("ReportDrawing", 
                           xmDrawingAreaWidgetClass, scroll,
                           NULL );
    XtAddCallback( reportDrawing, XmNexposeCallback,
                 (XtCallbackProc) reportDraw, (XtPointer) ALLOW);

    /* Dialog control */
    form = XtVaCreateWidget ("form", xmFormWidgetClass, pane,
                           XmNfractionBase, 5,
                           XmNskipAdjust, True,
                           NULL );
    button = XtVaCreateManagedWidget ("Ok", xmPushButtonWidgetClass, form,
                           XmNtopAttachment,    XmATTACH_FORM,
                           XmNbottomAttachment, XmATTACH_FORM,
                           XmNleftAttachment,   XmATTACH_POSITION,
                           XmNleftPosition,     2,
                           XmNrightAttachment,  XmATTACH_POSITION,
                           XmNrightPosition,    3,
                           XmNshowAsDefault,    True,
                           XmNdefaultButtonShadowThickness, 1,
                           NULL );
    XtAddCallback (button, XmNactivateCallback,
                 (XtCallbackProc) reportDestroy, reportDrawing);

    XtManageChild (form);
    XtManageChild (pane);
  }

  XtManageChild (reportDialog);
  XtPopup (reportDialog, XtGrabNone);

  /* Force new drawing if in single window mode */
  if (reportDrawing && reportValue & REPORT_SINGLEWIN_ON)
    reportDraw (reportDrawing, (XtPointer)resize, (XtPointer)NULL);

}
