/*
  Xinvest Copyright (c)1995-96, 2000 
  Mark Buser, All Rights Reserved.

  Permission is hereby granted to copy
  and freely distribute copies of this
  program for non-commercial purposes
  without fee, provided that this notice
  appears in all copies.
  
  All redistributions must be in their
  entirety as originally distributed.
  Feel free to modify Xinvest, but
  modified versions may not be distributed
  without prior consent of the author.
  
  This software is provided 'as-is'
  without any express or implied warranty.
  In no event will the author be held
  liable for any damages resulting from
  the use of this software.

  $Revision: 2.38 $ $Date: 2000/02/04 00:38:16 $
*/

/*  rate of return of a portfolio	*/

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>

#include <Xm/Xm.h>

#include "account.h"
#include "calendar.h"
#include "drawing.h"
#include "nav.h"
#include "pref.h"
#include "rate.h"
#include "trans.h"
#include "status.h"
#include "xutil.h"


/* Convert FIRST_DATE, LAST_DATE, and CURRENT_DATE into real dates */
long convertDate (int account, long date)
{
  void *trans;
  int numTrans;

  getAccount (account, ACCOUNT_TRANS, &trans);
  getAccount (account, ACCOUNT_NUM_TRANS, &numTrans);

  if (trans == NULL)
    return 0L;        /* at least we'll notice this */

  if (date == LAST_DATE) 
    return GetTransLDate (trans, numTrans); 

  else if (date == FIRST_DATE)
    return GetTransLDate (trans, 1); 

  else if (date == CURRENT_DATE) {
    char datebuf[20];
    time_t t = time ((time_t *)NULL);
    struct tm *today = localtime (&t);

    strftime (datebuf, 20, "%m/%d/%Y", today);
    return (calStrtoL (datebuf));

  } else
    return date;
}


/*  
**  strtoday - string into day number 
**  Convert "MM/DD/YY", "MM/DD/YYYY",  or "MMDDYY" into # of days using 
**  Jan 1, 1601 = day 1. REF: CACM 11, 657 (1968) - valid back to 24 Nov -4713
*/
long strtoday( char *dd)
{

#define MONTH (n[0])
#define DAY (n[1])
#define YEAR (n[2])

  static unsigned maxd[] = { 0, 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31};
  char		*ddend;
  int		c = 0, slash = 0;
  long		jdn;
  int		n[3];
  static int	m;

    jdn = strtol(dd, &ddend, 10);		/* convert mmddyy */
    MONTH = jdn;

    if (*ddend == '/') {
	slash = 1;				/* got mm/dd/yy or yyyy  */
	goto middle;
	}

    else if (*ddend) return -1;

    for (; c <= 2; c++) {
	if (slash) {
	    n[c] = strtol(dd, &ddend, 10);	/* convert a number */
	    if (*ddend != "//"[c])		/* check proper terminator */
baddy:		return -1;
middle:	    dd = ddend + 1;
	    }
	else {
	    n[2 - c] = jdn % 100;
	    jdn /= 100;
	    }
	}

    if (YEAR <= 99) YEAR += 1900;	/* correct for c */
    maxd[2] = 28;
    if ((!(YEAR & 3) && (YEAR % 100)) || !(YEAR % 400)) 
	maxd[2] = 29;
    if (MONTH < 1 || MONTH > 12 || DAY < 1 || DAY > maxd[MONTH]) goto baddy;

    m = (MONTH <= 2) ? -1 : 0;
    YEAR += m;
    jdn = (long)DAY - 2337888L;
    jdn += 1461L * ((long)YEAR + 4800L) / 4L;
    jdn += 367L * (MONTH - 2L - (long)m * 12L) / 12L;
    jdn -= 3L * (((long)YEAR + 4900L) / 100L) / 4L;

    return (long)jdn;

#undef MONTH
#undef DAY
#undef YEAR
}

static double *deposits = NULL;   /* value of deposit */
static long *dates = NULL;        /* date of deposit */
static int  ndep = 0;             /* num deposits */
static long IRRmaxday = 0;        /* most recent transaction date */

void rateClrTrans ()
{
  XtFree ( (char *) deposits );
  deposits = NULL;
  XtFree ( (char *) dates );
  dates = NULL;
  ndep = 0;
  IRRmaxday = 0;
}

static void rateAddValue(double val, long day)
{
  deposits = (double *) XtRealloc( (char *)deposits, sizeof(double)*(ndep+1));
  dates  =   (long *) XtRealloc( (char *) dates, sizeof(long)*(ndep+1));
  if ( deposits == NULL || dates == NULL) {
    write_status ( "Out of memory: can't calculate return.", ERR );
    return;
  }
  deposits[ndep] = val;
  dates[ndep] = day;
  ndep++; 
}

void rateAddTrans( TRANS *trans, int transno, double exch)
{
  int type = GetTransType (trans, transno);
  long day = GetTransLDate( trans, transno );

  if ( type==T_BUY || type==T_SELL) {
    deposits = (double *) XtRealloc( (char *)deposits, sizeof(double)*(ndep+1));
    dates  =   (long *) XtRealloc( (char *) dates, sizeof(long)*(ndep+1));
    if ( deposits == NULL || dates == NULL) {
      write_status ( "Out of memory: can't calculate return.", ERR );
      return;
    }

    /* 
    ** Load is added on a buy because this is what it cost, shares+fee.
    */
    if (type == T_BUY) 
      deposits[ndep] = (GetTransNav( trans,transno ) * exch  
    	                * GetTransShares( trans, transno)) 
		       + (GetTransLoad(trans,transno) * exch);
    /* 
    ** Load is NOT subtracted here since this is all we really got out. Load on
    ** sales work into the IRR calculation via current value, which is less
    ** when there is a fee. 
    **/
    else if (type == T_SELL) 
      deposits[ndep] = GetTransNav( trans,transno ) * exch 
	               * GetTransShares( trans, transno);

    dates[ndep] = day;

    ndep++;
  }

  if ( IRRmaxday < day )
    IRRmaxday = day;
}

/* Set up for a fresh round */
void rateClrAccount ()
{
  rateClrTrans();
}


void rate_add_account (int account, long sdate, long edate)
{
  void *trans;
  int  numTrans;
  int transno;

  long start, end;

  getAccount( account, ACCOUNT_NUM_TRANS, &numTrans );
  getAccount( account, ACCOUNT_TRANS, &trans );

  if (sdate == FIRST_DATE)
    start = GetTransLDate (trans, 1);
  else
    start = sdate;

  if (edate == LAST_DATE)
    end = GetTransLDate (trans, numTrans);
  else
    end = edate;

  if (IRRmaxday < end)
    IRRmaxday = end;            /* Calculate IRR to end date */

  if (start-1 > GetTransLDate (trans, 1)) {
    rateAddValue (rate_value (account, FIRST_DATE, start-1), start-1);
#ifdef DEBUG
    fprintf (stderr, "IRR ldate %ld lump: %.2f\n", 
             start-1, rate_value (account, FIRST_DATE, start-1));
#endif
  }

  for (transno = 1; transno <= numTrans; transno++) {
    long date = GetTransLDate (trans, transno);
    if ( date < start )
      continue;
    if ( date > end )
      break;
    rateAddTrans (trans, transno, rate_exch(account, date));
  }

}


/* GCC 2.7.2: bad code at -O2 if this is inside rate_IRR */
static double rate; 

/* 
** Calculate IRR from previously set up input data to curtot. 
*/
double rate_IRR ( double curtot )
{
  /*  Calculate daily rate of return by binary search	*/
  if (ndep) {
    int  i, not_moving = 0;  

    double  testtot, prevtot;
    double  limit[2]; 
    int	    which;

    /* Same total for 'not_moving' tries, we're not converging */
    prevtot = 0.0;

    limit[1] = 1.1;		/* 10 percent every day	*/
    limit[0] = 0.0;		/* throw it all away	*/
    for (;;) {
      rate = (limit[1] + limit[0]) / (double)2.0;
      testtot = 0.0;
      for (i = 0; i < ndep; i++) 
        testtot += (deposits[i] * pow ( rate, (double)(IRRmaxday - dates[i])) );

      /* Bail out if too many tries */
      if ( testtot == prevtot ) {
	if (++not_moving == 20)
          return (double)0.0;
      } else
	not_moving = 0;

      which = (curtot < testtot);

      if (limit[0] >= limit[1] || limit[which] == rate) {
        rate = 100.0 * ( pow ( rate, (double)365.0) - 1.0 );
        return (rate);
      }

      limit[which] = rate;

      prevtot = testtot;
    }

  }
  return (double)0.0;
}


double rate_fee( int account, long sdate, long edate )
{
  int i;
  double load = 0.0;

  int numTrans;
  void *trans;

  long start, end;

  getAccount( account, ACCOUNT_NUM_TRANS, &numTrans );
  getAccount( account, ACCOUNT_TRANS, &trans );

  if (sdate == FIRST_DATE)
    start = GetTransLDate( trans, 1 );
  else
    start = sdate;

  if (edate == LAST_DATE)
    end = GetTransLDate( trans, numTrans );
  else
    end = edate;

  for ( i=1; i <= numTrans; i++) {
    long date = GetTransLDate (trans, i);
    if ( date < start ) 
      continue;
    if ( date > end ) 
      break;

    load += (GetTransLoad (trans, i) * rate_exch(account, date));
  }
  return (load);
}


/* How many dates have transactions with the specified range */
double rate_datenum( int account, long sdate, long edate )
{
  int numTrans;
  void *trans;

  long start, end;
  int i, dateTrans = 0;

  getAccount( account, ACCOUNT_NUM_TRANS, &numTrans );
  getAccount( account, ACCOUNT_TRANS, &trans );

  if (sdate == FIRST_DATE)
    start = GetTransLDate( trans, 1 );
  else
    start = sdate;

  if (edate == LAST_DATE)
    end = GetTransLDate( trans, numTrans );
  else
    end = edate;

  for (i=1; i <= numTrans; i++) {
    if (GetTransLDate(trans, i) < start) 
      continue;
    
    if (GetTransLDate(trans, i) > end) 
      break;

    dateTrans++;
  }

  return (dateTrans);
}


/* Sometimes we need just the exchange rate, not only adjusted price */
double rate_exch( int account, long date )
{
  int i;
  double exchg = 1.0;

  int numTrans;
  void *trans;

  long end;
  int  latestETrans = 0;

  if (prefGetUseCurr()==0)  /* exchange rate use is off */
    return (exchg);

  getAccount( account, ACCOUNT_NUM_TRANS, &numTrans );
  getAccount( account, ACCOUNT_TRANS, &trans );

  if (date == FIRST_DATE)
    end = GetTransLDate( trans, 1 );
  else if (date == LAST_DATE)
    end = GetTransLDate( trans, numTrans );
  else if (date == CURRENT_DATE)
    end = convertDate(account, date);
  else
    end = date;

  /* Find last trans before end date */
  for (i=1; i <= numTrans; i++) {
    long nextDate;
    int type;
	    
    nextDate = GetTransLDate (trans, i);
    type = GetTransType(trans, i);

    /* If we have a exchange rate within range, then done */
    if (nextDate <= end && latestETrans != 0) 
      break;

    /* Otherwise look for the last exchange rate */
    if (type == T_CURR) /* latest exchange rate date */
      latestETrans = i;
  }

  /* Pending exchange quote? If not enabled or not available,   */
  /* use last exchange rate within date range.                  */
  {
    NAV *navp;
    float val;
    getAccount (account, ACCOUNT_CURRENCY, &navp);
    if (latestETrans > 0)
      exchg = GetTransCurrency (trans, latestETrans);
    if (navp && isNavChanged(account, navp)) {
      if ((latestETrans == 0 || (latestETrans > 0 &&
	   navToLDate(navp) > GetTransLDate(trans, latestETrans))) &&
	  navToLDate(navp) <= end ) {
        getNav (navp, NAV_VALUE, &val);
        exchg = (double)val;
      }
    }
  }

  return (exchg);
}



double rate_price( int account, long date )
{
  int i;
  double price = 0.0;
  double exchg = 1.0;
  double split = 1.0;

  int numTrans;
  void *trans;

  long end;
  int  latestPTrans = 0, latestETrans = 0;

  getAccount( account, ACCOUNT_NUM_TRANS, &numTrans );
  getAccount( account, ACCOUNT_TRANS, &trans );

  if (date == FIRST_DATE)
    end = GetTransLDate( trans, 1 );
  else if (date == LAST_DATE)
    end = GetTransLDate( trans, numTrans );
  else if (date == CURRENT_DATE)
    end = convertDate(account, date);
  else
    end = date;

  /* Find last trans before end date */
  for (i=1; i <= numTrans; i++) {
    long nextDate;
    int type;
	    
    nextDate = GetTransLDate (trans, i);
    type = GetTransType(trans, i);

    /* Process prices only within date range or get the next possible price */
    if (nextDate <= end || (nextDate>end && latestPTrans == 0)) 
      if (type != T_CURR) /* latest price date */
        latestPTrans = i;

    /* Process exchange rates within date range or get the next possible */
    if (nextDate <= end || (nextDate>end && latestETrans == 0)) 
      if (type == T_CURR) /* latest exchange rate date */
        latestETrans = i;

    /* We have to check all forward transactions to find any splits */
    if ( nextDate > end && type == T_SPLIT )
      /* Note inversion, price is effected opposite of shares by split */
      split = (split * GetTransSplit(trans,i,TRANS_MINOR)) 
	       / GetTransSplit(trans,i,TRANS_MAJOR);
  }

  /* Pending price point? If not use last NAV within date range. */
  {
    NAV *navp;
    float val = 0.0;
    getAccount (account, ACCOUNT_NAV, &navp);
    /* 
    ** Price point must be later than any existing trans and no later than
    ** end date.
    */
    if (latestPTrans > 0)
      price = GetTransNav (trans, latestPTrans) * split;
    if (navp && isNavChanged (account, navp)) {
      if (latestPTrans > 0 &&
          navToLDate(navp) > GetTransLDate(trans, latestPTrans) &&
	  navToLDate(navp) <= end) { 
        getNav (navp, NAV_VALUE, &val);
        price = (double)val;
      }
    }
  }

  /* Pending exchange quote? If not enabled or not available,   */
  /* use last exchange rate within date range.                  */
  if (prefGetUseCurr()) {
    NAV *navp;
    float val;
    getAccount (account, ACCOUNT_CURRENCY, &navp);
    if (latestETrans > 0)
      exchg = GetTransCurrency (trans, latestETrans);
    if (navp && isNavChanged(account, navp)) {
      if ((latestETrans == 0 || (latestETrans > 0 &&
	   navToLDate(navp) > GetTransLDate(trans, latestETrans))) &&
	  navToLDate(navp) <= end ) {
        getNav (navp, NAV_VALUE, &val);
        exchg = (double)val;
      }
    }
  }

  return (price*exchg);
}


/* How many shares are in the account, taking into consideration dividend
** reinvestment (or not), load on sales, and splits. */
double rate_shares( int account, long sdate, long edate )
{
  int i;
  double shares = 0.0;

  int numTrans;
  void *trans;
  int type;
  int reinvest;
  long start, end;

  getAccount( account, ACCOUNT_NUM_TRANS, &numTrans );
  getAccount( account, ACCOUNT_TRANS, &trans );

  if (sdate == FIRST_DATE)
    start = GetTransLDate( trans, 1 );
  else
    start = sdate;

  if (edate == LAST_DATE)
    end = GetTransLDate( trans, numTrans );
  else if (edate == CURRENT_DATE)
    end = convertDate(account, edate);
  else
    end = edate;

  for ( i=1; i <= numTrans; i++) {

    if ( GetTransLDate(trans, i) < start ) 
      continue;
    
    if ( GetTransLDate(trans, i) > end ) 
      break;

    /* add up all shares owned. */
    type = GetTransType(trans,i);
    reinvest = GetTransFlag ( trans, i, TRANS_REINVEST );

    /* shares are neg for T_SELL. divs only if reinvest on */
    if ( type != T_DIST || reinvest )
      shares += GetTransShares(trans, i);

    /* possible load on sale */
    if ( type == T_SELL )
      shares -= GetTransLoad(trans,i) / GetTransNav(trans,i);

    if ( type == T_SPLIT )
      shares = ( shares * GetTransSplit(trans,i,TRANS_MAJOR)) 
	       / GetTransSplit(trans,i,TRANS_MINOR);
  }

  return ( shares );
}

double rate_esop_cost (int account, long sdate, long edate)
{
  double  curtot = 0.0;
  double  exchg = 1.0;
  int	  i;

  int     numTrans;
  void    *trans;

  long    start, end;

  getAccount( account, ACCOUNT_NUM_TRANS, &numTrans );
  getAccount( account, ACCOUNT_TRANS, &trans );

  if (sdate == FIRST_DATE)
    start = GetTransLDate( trans, 1 );
  else
    start = sdate;

  if (edate == LAST_DATE)
    end = GetTransLDate( trans, numTrans );
  else if (edate == CURRENT_DATE)
    end = convertDate(account, edate);
  else
    end = edate;


  for (i = 1; i <= numTrans; i++) {
    long date = GetTransLDate (trans, i);
    if (date < start)
      continue;

    if (date > end)
      break;

    /* add up cost of ESOP shares only */
    if (GetTransType(trans, i) == T_ESOP) {
      exchg = rate_exch (account, date);
      curtot += ( (GetTransShares(trans, i) * GetTransNav(trans, i) * exchg) + 
                  (GetTransLoad (trans, i) * exchg) ); 
    }
  }
  return (curtot);
}


double rate_value ( int account, long sdate, long edate )
{
  double  curtot = 0.0;
  int     numTrans;
  void    *trans;

  long    start, end;

  getAccount( account, ACCOUNT_NUM_TRANS, &numTrans );
  getAccount( account, ACCOUNT_TRANS, &trans );

  if (sdate == FIRST_DATE)
    start = GetTransLDate( trans, 1 );
  else
    start = sdate;

  if (edate == LAST_DATE)
    end = GetTransLDate( trans, numTrans );
  else
    end = edate;

  curtot = (rate_shares (account, start, end) * rate_price (account, end)) 
	   - rate_esop_cost(account, start, end);

  return (curtot);
}


/* How much have we spent on shares including load */
double rate_cost ( int account, long sdate, long edate )
{
#if 0
  double  sold_val = 0.0;
#endif
  double  curtot = 0.0;
  double  exchg = 1.0;
  int	  i;

  int     numTrans;
  void    *trans;

  long    start, end;

  getAccount( account, ACCOUNT_NUM_TRANS, &numTrans );
  getAccount( account, ACCOUNT_TRANS, &trans );

  if (sdate == FIRST_DATE)
    start = GetTransLDate( trans, 1 );
  else
    start = sdate;

  if (edate == LAST_DATE)
    end = GetTransLDate( trans, numTrans );
  else
    end = edate;


  for (i = 1; i <= numTrans; i++) {
    long date = GetTransLDate (trans, i);
    if (date < start)
      continue;

    if (date > end)
      break;

#if 0
    if (GetTransType(trans, i) == T_BUY) {
      double cur_val = GetTransShares(trans, i) * GetTransNav(trans, i);

      /* If we've bought more than sold, increase cost */
      if (cur_val > sold_val) {
        curtot += (cur_val - sold_val);
	sold_val = 0.0;

      } else 
        sold_val -= cur_val;
    }

    else if ( GetTransType(trans, i) == T_SELL) 
      sold_val -= GetTransShares(trans, i) * GetTransNav(trans, i);

#else
    /* add up all shares owned for current cost, factor in load  */
    if ( GetTransType(trans, i) == T_BUY) {
      exchg = rate_exch (account, date);
      curtot += ( (GetTransShares(trans, i) * GetTransNav(trans, i) * exchg) + 
                  (GetTransLoad (trans, i) * exchg) ); 
    }
#endif
  }
  return (curtot);
}


/* How much money came out of the account not including load which just
** evaporated. */
double rate_withdrawal( int account, long sdate, long edate )
{
#if 0
  double bought_val = 0.0;
#endif
  double withdraw = 0.0;
  double exchg = 1.0;
  int i;

  int numTrans;
  void *trans;
  int reinvest;

  long start, end;

  getAccount( account, ACCOUNT_NUM_TRANS, &numTrans );
  getAccount( account, ACCOUNT_TRANS, &trans );

  if (sdate == FIRST_DATE)
    start = GetTransLDate( trans, 1 );
  else
    start = sdate;

  if (edate == LAST_DATE)
    end = GetTransLDate( trans, numTrans );
  else
    end = edate;

  for ( i=1; i <= numTrans; i++) {
    long date = GetTransLDate (trans, i);

    if (date < start) 
      continue;
    if (date > end) 
      break;

#if 0
    if (GetTransType(trans, i) == T_SELL) {
      /* - since T_SELL shares are negative. */
      double cur_val = -GetTransShares(trans, i) * GetTransNav(trans, i);

      /* If we've sold more than bought, increase withdraw */
      if (cur_val > bought_val) {
        withdraw += (cur_val - bought_val);
	bought_val = 0.0;

      } else 
        bought_val -= cur_val;
    }

    else if ( GetTransType(trans, i) == T_BUY) 
      bought_val += GetTransShares(trans, i) * GetTransNav(trans, i);

#else
    exchg = rate_exch (account, date);

    if ( GetTransType(trans, i) == T_SELL ) 
      /* - since T_SELL shares are negative. */
      withdraw += (-GetTransShares(trans, i) * GetTransNav(trans, i) * exchg);
#endif

    /* divs only if reinvest off. */
    reinvest = GetTransFlag (trans, i, TRANS_REINVEST);
    if ( GetTransType(trans,i) == T_DIST && !reinvest )
      withdraw += (GetTransShares(trans, i) * GetTransNav(trans, i) * exchg);
  }

  /* withdraw is positive for any sales or div distributions */
  return ( withdraw );
}

/* How much gain came out of the account not including load which just
** evaporated. */
double rate_distrib( int account, long sdate, long edate )
{
  int i;
  double withdraw = 0.0;

  int numTrans;
  int reinvest;
  void *trans;

  long start, end;

  getAccount( account, ACCOUNT_NUM_TRANS, &numTrans );
  getAccount( account, ACCOUNT_TRANS, &trans );

  if (sdate == FIRST_DATE)
    start = GetTransLDate( trans, 1 );
  else
    start = sdate;

  if (edate == LAST_DATE)
    end = GetTransLDate( trans, numTrans );
  else
    end = edate;

  for ( i=1; i <= numTrans; i++) {
    long date = GetTransLDate (trans, i);
    if (date < start) 
      continue;
    if (date > end) 
      break;

    /* divs only if reinvest off. */
    reinvest = GetTransFlag (trans, i, TRANS_REINVEST);
    if ( GetTransType(trans,i) == T_DIST && !reinvest )
      withdraw += (GetTransShares(trans, i) * GetTransNav(trans, i) *
		   rate_exch (account, date));
  }

  /* withdraw is positive for any sales or div distributions */
  return ( withdraw );
}

/* 
** Calculate yield between start and end dates.  Return more info if 'result'
** non-NULL. 
*/
double rate_yield( int account, long sdate, long edate, YIELD *result )
{
  int i;

  double totYield = 0.0;
  double yield = 0.0;
  double shares = 0.0;

  double value = 0.0;
  double div = 0.0;
  double exch = 1.0;

  int numTrans;
  void *trans;

  long start, end;

  getAccount( account, ACCOUNT_NUM_TRANS, &numTrans );
  getAccount( account, ACCOUNT_TRANS, &trans );

  if (sdate == FIRST_DATE)
    start = GetTransLDate( trans, 1 );
  else
    start = sdate;

  if (edate == LAST_DATE)
    end = GetTransLDate( trans, numTrans );
  else
    end = edate;

  for ( i=1; i <= numTrans; i++) {
    long date = GetTransLDate (trans, i);
    if (date < start) 
      continue;
    if (date > end) 
      break;

    /* Yield = dividend value/start shares/start price */
    if (GetTransType(trans,i) == T_DIST) {

      /* Dividend value/start price = shares. */
      yield = GetTransShares(trans, i);

      /* / start shares */
      shares = rate_shares (account, FIRST_DATE, GetTransLDate(trans, i-1));
      if (shares > 0.0) {
	exch = rate_exch (account, date);

        /* Portfolio yield */
        value += (shares * GetTransNav (trans, i) * exch);
        div += (yield * GetTransNav (trans, i) * exch);

        /* Account yield */
        yield /= shares;
        totYield += yield;
#ifdef DEBUG
        fprintf (stderr, "YIELD partial %.4f\n", yield);
#endif
      }
    }
  }

  if (result) {
    double years = (double)(end - start) / 365.0;
    result->div = div;
    result->value = value;

    result->yield = totYield * 100.0;
    result->annYield = (pow (1.0+totYield, 1.0/years) - 1.0) * 100.0;
  }

#ifdef DEBUG
  fprintf (stderr, "YIELD final %.4f\n", totYield);
#endif
  return (totYield);
}
