/*
 *  The TextField Widget
 *  --------------------
 
 *  Copyright (C) 1997  by Till Straumann   <strauman@sun6hft.ee.tu-berlin.de>

 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Library Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.

 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Library Public License for more details.

 *  You should have received a copy of the GNU General Library Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 */
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/cursorfont.h>
#include <X11/Xatom.h>
#include <X11/Xos.h>

#ifdef XAW3D
#include <X11/Xaw3d/XawInit.h>
#define SHADOW_WIDTH(tfw) (tfw->threeD.shadow_width)
#else
#include <X11/Xaw/XawInit.h>
#define SHADOW_WIDTH(tfw) 0
#endif

#include <X11/Xmu/Atoms.h>
#include <X11/Xmu/Converters.h>
#include <X11/Xmu/Drawing.h>
#include <X11/Xmu/StdSel.h>

#include "TextFieldP.h"

/* Macro definitions */

#define SetPoint(arg,a1,a2) ((void)((arg).x=(short)(a1), (arg).y=(short)(a2)))
#define TF (tfw->textField)

#ifdef TEXT_COMPAT
#define RESIZE(w) (((w)->textField).resize==XawtextResizeWidth || \
		   ((w)->textField).resize==XawtextResizeBoth)
#else
#define RESIZE(w) ((w)->textField.resize)
#endif

/* is this Dimension probably negative ? */
#define NEGDIMENSION(x) (x & (1<< (8*sizeof(Dimension)-1)))

#define WIDTH			4
#define CURSOR_WIDTH_DIV2	3
#define CURSOR_ASCENT		3
#define CURSOR_HEIGHT		8

#define PRIMARY_BIT		1
#define SECONDARY_BIT		2
#define CLIPBOARD_BIT		4

#define MULTI_TIME		((Time)200)

#define DEF_BUFFER_SIZE		15

/* extern declarations */
extern long strtol();

/* global variables */

/* include the automatically generated strings of resource names */
#include "TextField_res_c"
#include "TextField_class_c"
#include "TextField_type_c"

/* The memory for '*value' allocated in 'ConvertSelection()'
 * must be freed in 'ClipboardXferDone()', for we own it 
 * if a 'done_proc' is registered
 */

String clipboardValue;

/* resources */

static XtResource resources[] = {
#define offset(field) XtOffsetOf(TextFieldRec, textField.field)
 /* {name, class, type, size, offset, default_type, default_addr}, */
    { XtNfont, XtCFont, XtRFontStruct,
	  sizeof(XFontStruct*), offset(font), XtRString, (XtPointer) XtDefaultFont},
    { XtNforeground, XtCForeground, XtRPixel,
	  sizeof(Pixel), offset(foreground), XtRString, (XtPointer) XtDefaultForeground},
    { XtNinsensitiveForeground, XtCInsensitiveForeground, XtRPixel,
	  sizeof(Pixel), offset(insensitive_foreground), XtRString,(XtPointer)  XtDefaultForeground},
    { XtNinternalHeight, XtCHeight, XtRDimension,
	  sizeof(Dimension), offset(internalHeight), XtRImmediate, (XtPointer)2},
    { XtNinternalWidth, XtCWidth, XtRDimension,
	  sizeof(Dimension), offset(internalWidth), XtRImmediate, (XtPointer)2},
    { XtNinsertPosition, XtCTextPosition, XtRInt,
	  sizeof(int), offset(text_cursor), XtRImmediate, (XtPointer)0},
#ifdef TEXT_COMPAT
    { XtNresize, XtCResize, XtRResizeMode,
	  sizeof(XawTextResizeMode), offset(resize), XtRImmediate, (XtPointer)XawtextResizeNever},
#else
    { XtNresize, XtCResize, XtRBoolean,
	  sizeof(Boolean), offset(resize), XtRImmediate, (XtPointer)False},
#endif
    { XtNstring, XtCLabel, XtRString,
	  sizeof(String), offset(string), XtRString, ""},
    { XtNuseStringInPlace, XtCUseStringInPlace, XtRBoolean,
	  sizeof(Boolean), offset(use_string_in_place), XtRImmediate, (XtPointer)False},
    { XtNlength, XtCLength, XtRInt,
	  sizeof(int), offset(maxlen), XtRImmediate, (XtPointer) 0},
    { XtNjustify, XtCJustify, XtRJustify,
	  sizeof(XtJustify), offset(justify), XtRImmediate, (XtPointer)XtJustifyLeft},
    { XtNreadOnly, XtCReadOnly, XtRBoolean,
	  sizeof(Boolean), offset(readonly), XtRImmediate, (XtPointer)False},
    { XtNscrollChars, XtCScrollChars, XtRShort,
	  sizeof(short), offset(scroll_chars), XtRImmediate, (XtPointer)4},
    { XtNcallback, XtCCallback, XtRCallback,
	  sizeof(XtPointer), offset(notify_cbl), XtRCallback, (XtPointer)NULL},
    { XtNmodifiedCallback, XtCCallback, XtRCallback,
	  sizeof(XtPointer), offset(modified_cbl), XtRCallback, (XtPointer)NULL},
    { XtNvalueAddress, XtCValue, XtRPointer,
	  sizeof(caddr_t), offset(value.addr), XtRImmediate, (XtPointer)NULL},
    { XtNvalueSize, XtCValue, XtRInt,
	  sizeof(unsigned int), offset(value.size), XtRImmediate, (XtPointer)NULL},
    { XtNvalueType, XtCValue, XtRString,
	  sizeof(String), offset(value_type), XtRString, (XtPointer)XtRCFormat},
    { XtNinputFormat, XtCValueFormat, XtRString,
	  sizeof(String), offset(input_format), XtRImmediate, (XtPointer)NULL},
    { XtNoutputFormat, XtCValueFormat, XtRString,
	  sizeof(String), offset(output_format), XtRImmediate, (XtPointer)NULL},
    { XtNdisplayCaret, XtCOutput, XtRBoolean,
	  sizeof(Boolean), offset(display_caret), XtRImmediate, (XtPointer)True},
/* modified Simple defaults */
    { XtNcursor, XtCCursor, XtRCursor,
	  sizeof(Cursor), XtOffsetOf(TextFieldRec, simple.cursor), XtRString, "xterm"},
#undef offset
};

/* Private Procedures */
static int str2chr(/*
	char **chpt; */);

static int PointerToCursor(/*
	TextFieldWidget tfw;
	int pos */);

static Boolean GetEventTime(/*
	XEvent		*ev;
	Time		*time;*/);

static Boolean GetSelectionFromArgs(/*
	Widget		w;
	String		*args;
	Cardinal	*nargs;
	Atom		*selection;*/);

static Boolean ConvertSelection(/*
	Widget w;
        Atom *selection, *target, *type;
	XtPointer *value;
	unsigned long *length;
	int *format; */);

static void LoseSelection(/*
	Widget w;
	Atom *selection */);

static void DoPastingCallback(/*
	Widget		w;
	XtPointer	cld;
	Atom		*selection, *type;
	XtPointer	value;
	unsigned long	*length;
	int		*format; */);

static void ClipboardXferDone(/*
	Widget w;
	Atom *selection, *target */);

static int GetTextStart(/*
	TextFieldWidget tfw; */);

static void make_cursor_visible(/*
	TextFieldWidget tfw; */);

static void DrawCursor(/*
	TextFieldWidget tfw;
	int 		txt_x,liney; */);

static void ReDisplay(/*
	TextFieldWidget tfw; Boolean do_clear; */);

static void getGCs(/*
	TextFieldWidget tfw; */);

static void adj_left(/*
	TextFieldWidget tfw; */);

static void adj_right(/*
	TextFieldWidget tfw; */);

static void curs_visible_left(/*
	TextFieldWidget tfw; */);

static void curs_visible_right(/*
	TextFieldWidget tfw; */);

static Boolean resize_width(/*
	TextFieldWidget tfw; */);

static void Unhighlight(/*
	TextFieldWidget tfw; */);

static unsigned long AtomToBit(/*
	Display *d;
	Atom	sel; */);

static void Unselect(/*
	TextFieldWidget tfw;
	Atom *selection;
	Time time */);


/* Actions */

static void MoveCursor(/* Widget, XEvent*, String*, Cardinal* */);
static void InsertChar(/* Widget, XEvent*, String*, Cardinal* */);
static void Delete(/* Widget, XEvent*, String*, Cardinal* */);
static void HighlightStart(/* Widget, XEvent*, String*, Cardinal* */);
static void HighlightExtend(/* Widget, XEvent*, String*, Cardinal* */);
static void Highlight(/* Widget, XEvent*, String*, Cardinal* */);
static void MakeSelection(/* Widget, XEvent*, String*, Cardinal* */);
static void DeleteToClipboard(/* Widget, XEvent*, String*, Cardinal* */);
static void InsertSelection(/* Widget, XEvent*, String*, Cardinal* */);
static void Notify(/* Widget, XEvent*, String*, Cardinal* */);
static void CursorState(/* Widget, XEvent*, String*, Cardinal* */);

static XtActionsRec actions[] =
{
 /* {name, procedure}, */
    {"MoveCursor",	MoveCursor},
    {"InsertChar",	InsertChar},
    {"Delete",		Delete},
    {"HighlightStart",	HighlightStart},
    {"HighlightExtend",	HighlightExtend},
    {"Highlight",	Highlight},
    {"MakeSelection",	MakeSelection},
    {"DeleteToClipboard",DeleteToClipboard},
    {"InsertSelection", InsertSelection},
    {"Notify", 		Notify},
    {"CursorState",	CursorState},
};

/* default translations */

static char translations[] ="\
		<Enter>:	CursorState(Active)\n\
		<Leave>:	CursorState(Inactive)\n\
	None	<Btn1Down>:	MoveCursor()\
				HighlightStart()\n\
		<Btn1Motion>:	MoveCursor()\
				HighlightExtend()\n\
		<Btn1Up>:	HighlightExtend()\
				MakeSelection(PRIMARY)\n\
		<Btn2Up>  :     MoveCursor()\
				InsertSelection(PRIMARY)\n\
	Shift	<Key>Right:	MoveCursor()\
				HighlightExtend()\
				MakeSelection(PRIMARY)\n\
 	Shift	<Key>Home :	MoveCursor(Home)\
				HighlightExtend()\
				MakeSelection(PRIMARY)\n\
 	Shift	<Key>End  :	MoveCursor(End)\
				HighlightExtend()\
				MakeSelection(PRIMARY)\n\
 	Shift	<Key>Left :	MoveCursor(-1)\
				HighlightExtend()\
				MakeSelection(PRIMARY)\n\
 	Shift	<Key>Delete:	Delete(Selection,End)    \n\
 	Shift	<Key>BackSpace:	Delete(Selection,Home)  \n\
 	Alt	<Key>Delete:	Delete(Selection,All)    \n\
 	Alt	<Key>BackSpace:	Delete(Selection,All)  \n\
		<Key>Right:	MoveCursor()	\
				HighlightStart()\n\
 		<Key>Home :	MoveCursor(Home)\
				HighlightStart()\n\
 		<Key>End  :	MoveCursor(End) \
				HighlightStart()\n\
 		<Key>Left :	MoveCursor(-1)  \
				HighlightStart()\n\
 		<Key>Delete:	Delete(Selection,1)    \n\
 		<Key>BackSpace:	Delete(Selection,-1)  \n\
		<Key>SunCopy:	MakeSelection(CLIPBOARD)\n\
		<Key>SunPaste:	InsertSelection(CLIPBOARD)\n\
		<Key>SunCut:	DeleteToClipboard()\n\
		<Key>:		InsertChar()\
";

/* Methods */

static void ClassInitialize();
static void Initialize();
static void Destroy();
static void TfwExpose();
static void Resize();
static Boolean SetValues();
static XtGeometryResult QueryGeometry();
static Boolean AcceptFocus();

/* Converters */
static Boolean CvtByFormat();
#ifdef TEXT_COMPAT
static Boolean CvtStringToResizeMode();
#endif

TextFieldClassRec textFieldClassRec = {
  { /* core fields */
#ifdef XAW3D
    /* superclass		*/	(WidgetClass) &threeDClassRec,
#else
    /* superclass		*/	(WidgetClass) &simpleClassRec,
#endif
    /* class_name		*/	"TextField",
    /* widget_size		*/	sizeof(TextFieldRec),
    /* class_initialize		*/	ClassInitialize,
    /* class_part_initialize	*/	NULL,
    /* class_inited		*/	FALSE,
    /* initialize		*/	Initialize,
    /* initialize_hook		*/	NULL,
    /* realize			*/	XtInheritRealize,
    /* actions			*/	actions,
    /* num_actions		*/	XtNumber(actions),
    /* resources		*/	resources,
    /* num_resources		*/	XtNumber(resources),
    /* xrm_class		*/	NULLQUARK,
    /* compress_motion		*/	TRUE,
    /* compress_exposure	*/	TRUE,
    /* compress_enterleave	*/	TRUE,
    /* visible_interest		*/	FALSE,
    /* destroy			*/	Destroy,
    /* resize			*/	Resize,
    /* expose			*/	TfwExpose,
    /* set_values		*/	SetValues,
    /* set_values_hook		*/	NULL,
    /* set_values_almost	*/	XtInheritSetValuesAlmost,
    /* get_values_hook		*/	NULL,
    /* accept_focus		*/	AcceptFocus,
    /* version			*/	XtVersion,
    /* callback_private		*/	NULL,
    /* tm_table			*/	translations,
    /* query_geometry		*/	QueryGeometry,
    /* display_accelerator	*/	XtInheritDisplayAccelerator,
    /* extension		*/	NULL
  },
  { /* simple    fields */
    /* change_sensitive */		XtInheritChangeSensitive
  },
#ifdef XAW3D
  { /* ThreeD class fields */
    /* shadowdraw	*/		XtInheritXaw3dShadowDraw
  },
#endif
  { /* textField fields */
    /* empty			*/	0
  }
};

WidgetClass textFieldWidgetClass = (WidgetClass)&textFieldClassRec;

/* private Procedures */
static int str2chr(chpt)
char **chpt;
{
char ch=(*(*chpt)++);
int  i,rval;
if (ch != '\\' ) return((int)ch);
ch=(*(*chpt)++);
switch (ch) {
	case '0': return((int)'\0');
	case 'n': return((int)'\n');
	case 't': return((int)'\t');
	case 'b': return((int)'\b');
	case 'f': return((int)'\f');
	case 'r': return((int)'\r');
	case 'v': return((int)'\v');
	case '\'': return((int)'\'');
	case '\\': return((int)'\\');
	default: 
	  rval=0;
	  i=0;
	  do {
	    if ( ch >= '0' && ch <='7' ) {
	      rval=8*rval+(int)(ch-'0');
	      ch=(*(*chpt)++);
	    } else {
	      break;
	    }
	    i++;
          } while (i<=2);
	  return( (i>0 && rval<256) ? rval: -1);
}
}

static unsigned long AtomToBit(d,sel)
 Display *d;
 Atom    sel;
{
 if      (sel==XA_PRIMARY)    		return((unsigned long)PRIMARY_BIT);
 else if (sel==XA_SECONDARY)  		return((unsigned long)SECONDARY_BIT);
 else if (sel==XA_CLIPBOARD(d))		return((unsigned long)CLIPBOARD_BIT);
 else					return((unsigned long)0);
}

 

		

static void Unhighlight(tfw,selectionBits)
TextFieldWidget tfw;
unsigned long   selectionBits;
{
TF.selections &= ~selectionBits;
if (TF.selections==(unsigned long)0) {
  TF.hil_start=TF.hil_end=-1;
}
}


static void Unselect(tfw,selection)
TextFieldWidget tfw;
Atom		*selection;
{
Display *di=XtDisplay((Widget)tfw);
unsigned long bits;

if (selection==NULL) {
  XtDisownSelection((Widget)tfw,XA_CLIPBOARD(di),CurrentTime);
  XtDisownSelection((Widget)tfw,XA_PRIMARY,CurrentTime);
  XtDisownSelection((Widget)tfw,XA_SECONDARY,CurrentTime);
  bits=0;
} else {
  XtDisownSelection((Widget)tfw,*selection,CurrentTime);
  bits=AtomToBit(di,*selection);
}

Unhighlight(tfw,bits);
}

/* adjust the right margin of the visible window */

static void adj_right(tfw)
TextFieldWidget tfw;
{
 int len    =strlen(&TF.buffer[TF.visible_start]);
 int toolong=XTextWidth(TF.font,TF.buffer+TF.visible_start,len)
	     - (int)(tfw->core.width
                     -2*(TF.internalWidth+WIDTH+SHADOW_WIDTH(tfw))
		    );

 TF.visible_end=TF.visible_start+len;

 while(toolong>0 && TF.visible_end>TF.text_cursor) {
	toolong-=XTextWidth(TF.font,&TF.buffer[TF.visible_end-- -1],1);
 }
}

/* adjust the left margin of the visible window */

static void adj_left(tfw)
TextFieldWidget tfw;
{
 int toolong=XTextWidth(TF.font,TF.buffer,(int)TF.visible_end)
	     - (int)(((Widget)tfw)->core.width
                     -2*(TF.internalWidth+WIDTH+SHADOW_WIDTH(tfw))
		    );

 TF.visible_start=0;

 while( toolong>0 && TF.visible_start<TF.text_cursor) {
	toolong-=XTextWidth(TF.font,&TF.buffer[TF.visible_start++],1);
 }
}

/* adjust the visible area so that the cursor is seen
 * at the right margin (preserving 'scroll_chars')
 */
static void curs_visible_right(tfw)
TextFieldWidget tfw;
{
 register int len=strlen(TF.buffer);
 if ( (TF.visible_end=TF.text_cursor+TF.scroll_chars) >len) {
  TF.visible_end=len;
 }
 adj_left(tfw);
}

/* adjust the visible area so that the cursor is seen
 * at the left margin (preserving 'scroll_chars')
 */
static void curs_visible_left(tfw)
TextFieldWidget tfw;
{
 if ( (TF.visible_start=TF.text_cursor-TF.scroll_chars) < 0) {
  TF.visible_start=0;
 }
 adj_right(tfw);
}

/* request the parent to change our width
 * returns false, if this could not be done
 * (might be due to 'resize' resource beeing set to False)
 */

static Boolean resize_width(tfw)
TextFieldWidget tfw;
{
 XtWidgetGeometry request;
 Boolean rval;
 int     len;
 if (!RESIZE(tfw)) return(False);
 request.request_mode=CWWidth;
 request.width= XTextWidth(TF.font,TF.buffer,len=strlen(TF.buffer))
		+ 2*( SHADOW_WIDTH(tfw)+TF.internalWidth+WIDTH);
 rval=(XtMakeGeometryRequest((Widget)tfw,&request,NULL)==XtGeometryYes);
 if (rval) {
  TF.visible_start=0; TF.visible_end=len;
 }
 return(rval);
}

/* release any old GCs before calling this function */

static void getGCs(tfw)
 TextFieldWidget tfw;
{
XGCValues	values;
XtGCMask	mask	=GCForeground|GCBackground|GCFont|GCGraphicsExposures;
XtGCMask	inv_mask=GCFunction|GCPlaneMask|GCGraphicsExposures;

values.foreground=TF.foreground;
values.background=tfw->core.background_pixel;
values.font=TF.font->fid;
values.function=GXinvert;
values.graphics_exposures=False;
values.plane_mask=values.background^values.foreground;

TF.norm=XtGetGC((Widget)tfw,mask,&values);
TF.norm_hil=XtGetGC((Widget)tfw,inv_mask,&values);


  if (TF.foreground==TF.insensitive_foreground) {
    mask		|=GCFillStyle|GCTile;
    values.fill_style=FillTiled;
    values.tile      =XmuCreateStippledPixmap(XtScreen((Widget)tfw),
					  TF.foreground,
					  tfw->core.background_pixel,
					  tfw->core.depth);
  } else {
    values.foreground=TF.insensitive_foreground;
    values.plane_mask=values.background^values.foreground;
  }

TF.insens    =XtGetGC((Widget)tfw,mask,&values);
TF.insens_hil=XtGetGC((Widget)tfw,inv_mask,&values);
}

/* Draw the text cursor at txt_x, liney */

static void DrawCursor(tfw,txt_x,liney)
TextFieldWidget tfw;
int		txt_x;
int		liney;

{
 XPoint		cur_shape[5];
 int		cur_x,cwd2,cheight,h,peak;
 GC		gc;
 Display	*di=XtDisplay((Widget)tfw);
 Window		wi =XtWindow((Widget)tfw);
 int		i;

 cur_x=txt_x + XTextWidth(TF.font,
			 &TF.buffer[TF.visible_start],
		    (int)(TF.text_cursor-TF.visible_start));

 cheight=(TF.font->max_bounds.ascent+TF.font->max_bounds.descent)/3;
 if (cheight<4) cheight=4;

 gc=(!tfw->core.sensitive || 
      TF.readonly       ||
     !TF.active_cursor )?TF.insens:TF.norm;

 peak=liney-CURSOR_ASCENT;
 h=tfw->core.height-SHADOW_WIDTH(tfw) - peak;
 if (h>cheight) h=cheight;
 cwd2=h/2;
 i=0;
 /* !! KEEP i < 6 !! */
 SetPoint(cur_shape[i],cur_x,liney-CURSOR_ASCENT);i++;
 SetPoint(cur_shape[i],cwd2,h);i++;
 SetPoint(cur_shape[i],-(2*cwd2+1),0); i++;

 XFillPolygon(di,wi,gc,cur_shape,i,Convex,CoordModePrevious);
}

/* redisplay everything but 3D borders; erase if 'do_clear' */

static void ReDisplay(tfw,do_clear)
TextFieldWidget tfw;
Boolean 	do_clear;

{
 GC		 gc ;
 Display         *di=XtDisplay((Widget)tfw);
 Window          wi=XtWindow((Widget)tfw);
 int		 liney,txt_x,left,right,len;
 int		 hil1,hil2;

 if (!XtIsRealized((Widget)tfw)) return;

 len  = strlen(TF.buffer);
 left = TF.internalWidth+WIDTH+SHADOW_WIDTH(tfw);
 right= tfw->core.width-left;
 liney= TF.internalHeight+SHADOW_WIDTH(tfw)+TF.font->max_bounds.ascent;

 if (do_clear) {
 /* clear the internal Width and Height area also */
   XClearArea(di,wi,
	   SHADOW_WIDTH(tfw),SHADOW_WIDTH(tfw),
	   tfw->core.width-2*SHADOW_WIDTH(tfw),
	   tfw->core.height-2*SHADOW_WIDTH(tfw),
	   False);
 }

 txt_x=GetTextStart(tfw);


/* Draw String */

 gc=(((Widget)tfw)->core.sensitive && !TF.readonly)?TF.norm:TF.insens;

 XDrawString(di,
		wi,
		gc,
		txt_x,
		liney,
		&TF.buffer[TF.visible_start],
		(int)(TF.visible_end-TF.visible_start));

/* invert highlighted area */

 if ( TF.hil_start<TF.visible_end && 
      TF.hil_end>TF.visible_start &&
      TF.hil_end!=TF.hil_start) {
     hil1=(TF.hil_start>TF.visible_start)?TF.hil_start:TF.visible_start;
     hil2=(TF.hil_end  <TF.visible_end)  ?TF.hil_end  :TF.visible_end;
     XFillRectangle(di,wi,
		(((Widget)tfw)->core.sensitive && !TF.readonly)?TF.norm_hil:TF.insens_hil,
		txt_x+XTextWidth(TF.font,
				TF.buffer+TF.visible_start,
				hil1-TF.visible_start),
		liney-TF.font->max_bounds.ascent,
		XTextWidth(TF.font,
				TF.buffer+hil1,
				hil2-hil1),
		TF.font->max_bounds.ascent+TF.font->max_bounds.descent);
 }
		
/* Draw Cursor */

 if (TF.display_caret) DrawCursor(tfw,txt_x,liney);


/* Draw scroll marks */

 gc=(((Widget)tfw)->core.sensitive)?TF.norm:TF.insens;
 if (TF.visible_start>0) {
  XDrawLine(di,wi,gc,left-1,liney+TF.font->max_bounds.descent,
		     left-1,liney-TF.font->max_bounds.ascent);

  XFillRectangle(di,wi,gc,
		 left-WIDTH,
		 liney-TF.font->max_bounds.ascent+TF.font->max_bounds.descent,
		 WIDTH-1,
		 TF.font->max_bounds.ascent-TF.font->max_bounds.descent);
 }
 if (TF.visible_end < len) {
  XDrawLine(di,wi,gc,right+1,liney+TF.font->max_bounds.descent,
		     right+1,liney-TF.font->max_bounds.ascent);
  XFillRectangle(di,wi,gc,
		 right+2,
		 liney-TF.font->max_bounds.ascent+TF.font->max_bounds.descent,
		 WIDTH-1,
		 TF.font->max_bounds.ascent-TF.font->max_bounds.descent);
 }

}

/* adjust the visible window, if cursor is outside */

static void make_cursor_visible(tfw)
TextFieldWidget tfw;
{
if (TF.visible_end<TF.text_cursor) {
 curs_visible_right(tfw);
}
if (TF.visible_start>TF.text_cursor) {
 curs_visible_left(tfw);
}

}

/* Get X-coordinate of the visible part of the text string
 * (depends on 'justify' etc. )
 */

static int GetTextStart(tfw)
TextFieldWidget tfw;
{
 int left = TF.internalWidth+WIDTH+SHADOW_WIDTH(tfw);
 int right= tfw->core.width-left;
 int txt_x=left;

 int txt_width=XTextWidth(  TF.font,
		&TF.buffer[TF.visible_start],
		(int)(TF.visible_end-TF.visible_start));

 switch (TF.justify) {
	case XtJustifyRight: txt_x= right -txt_width; break;

	case XtJustifyCenter:txt_x= ((int)((Widget)tfw)->core.width-txt_width)/2;

        default:	     break;
 }

 return(txt_x);
}

/* The hightlighted area has been successfully transferred to
 * the CLIPBOARD-owner ('DeleteToClipboard()' action), so
 * it can be deleted now.
 */

/*ARGSUSED*/
static void ClipboardXferDone(w,selection,target)
Widget w;
Atom   *selection,*target;
{
TextFieldWidget tfw=(TextFieldWidget)w;
Cardinal	nargs=1;
static char	*args[]={"S"};

/* free the converted value, because we own it */
XtFree(clipboardValue);
/* Call with NULL event arg. to suppress giving up the clipboard */
Delete((Widget)tfw,(XEvent*)0,args,&nargs);
}


/* A selection has been successfully transferred from another client - it
 * can be inserted into the buffer.
 * This procedure is registered in the 'InsertSelection()' action
 */

/*ARGSUSED*/
static void DoPastingCallback(w,cld,selection,type,vpar,length,format)
	Widget w;
	XtPointer cld;
	Atom *selection, *type;
	XtPointer vpar;
	unsigned long *length;
	int *format;
{
 TextFieldWidget tfw=(TextFieldWidget)w;
 Display	 *di=XtDisplay((Widget)tfw);
 register int	 ncopychars;
 register char   *chpt,*cursp;
 int		 *value=(int*)vpar;

 if ( ! tfw->core.sensitive || TF.readonly ){
	XBell(di,100);
 } else {
   if ( (*type==XT_CONVERT_FAIL)||
        (value==NULL)		||
	(*value==0)		||
	(*length==0)){
	  XBell(di,100);
	  XtWarning("TextField: no selection or selection timed out");
   } else {
	ncopychars=TF.maxlen-strlen(TF.buffer);
	if ((int)*length>ncopychars) {
	   XBell(di,100);
	   XtWarning("TextField: Buffer full");
	} else { 
	   ncopychars=(int)*length;
	}

	if (ncopychars>0) {
	  Unselect(tfw,(Atom*)0);

	  cursp=TF.buffer+TF.text_cursor;
	  for (chpt=TF.buffer+strlen(TF.buffer);
	       chpt>=cursp; chpt-- ) {
		  *(chpt+ncopychars)=*chpt;
	  }

	  (void)strncpy(cursp,(char*)value,ncopychars);
	  TF.text_cursor+=ncopychars;
  
          adj_right(tfw);
          if (TF.text_cursor>=TF.visible_end) {
	    curs_visible_right(tfw);
          }
          ReDisplay(tfw,True);
 	  XtCallCallbackList((Widget)tfw,TF.modified_cbl,(XtPointer)TF.buffer);
	}
   }
 }
 if (value) XtFree((char *)value);
}

/* The ownership of *selection has been lost */

/*ARGSUSED*/
static void LoseSelection(w, selection)
Widget w;
Atom   *selection;
{
TextFieldWidget tfw=(TextFieldWidget)w;
Unhighlight(tfw,AtomToBit(XtDisplay(w),*selection));
if (TF.selections==(unsigned long)0) ReDisplay(tfw,True);
}

static Boolean ConvertSelection(w, selection, target,
				type, valp, length, format)
    Widget w;
    Atom *selection, *target, *type;
    XtPointer *valp;
    unsigned long *length;
    int *format;
{
    char **value=(char**)valp;
    TextFieldWidget tfw=(TextFieldWidget)w;
    Display* d = XtDisplay(w);
    XSelectionRequestEvent* req =
	XtGetSelectionRequest(w, *selection, (XtRequestId)NULL);
    unsigned int tmp;

    if (*target == XA_TARGETS(d)) {
	Atom* targetP;
	Atom* std_targets;
	unsigned long std_length;
	XmuConvertStandardSelection(w, req->time, selection, target, type,
				  (XPointer*)&std_targets, &std_length, format);
	*value = (XtPointer)XtMalloc(sizeof(Atom)*((unsigned)std_length + 5));
	targetP = *(Atom**)value;
	*targetP++ = XA_STRING;
	*targetP++ = XA_TEXT(d);
	*length = std_length + (targetP - (*(Atom **) value));
	(void)memcpy( (void*)targetP, (void*)std_targets,
		      (size_t)(sizeof(Atom)*std_length));
	XtFree((char*)std_targets);
	*type = XA_ATOM;
	*format = 32;
	return True;
    }

    
    
    if (*target == XA_STRING ||
      *target == XA_TEXT(d) ||
      *target == XA_COMPOUND_TEXT(d))
    {
    	if (*target == XA_COMPOUND_TEXT(d))
	    *type = *target;
    	else
	    *type = XA_STRING;
        tmp=      TF.hil_end-TF.hil_start;
	*length=  tmp;
    	*value =  (XtPointer)strncpy(XtMalloc(tmp+1),TF.buffer+TF.hil_start, (int) tmp);
        (*value)[tmp]=(char)0;
    	*format = 8;
        if (*selection==XA_CLIPBOARD(d)) {
	  clipboardValue=*value;

/* Disown the clipboard now; disowning it in
 * 'ClipboardXferDone()' behaves wrong!
 * (Communication with 'xclipboard')
 */
	  XtDisownSelection((Widget)tfw,*selection,CurrentTime);
	}
    	return True;
    }
    
    if (XmuConvertStandardSelection(w, req->time, selection, target, type,
				    (XPointer *)value, length, format))
	return True;

    return False;
}


/* Convert a Pointer position to a cursor position
 *
 * return value -1 indicates a Pointer pos. to the left of the string
 * return value -2 indicates a Pointer pos. to the right of the string
 */
static int PointerToCursor(tfw,pos)
TextFieldWidget tfw;
int		pos;
{
register int c1pos,c2pos,curs,lst_ch;

if (pos<(int)(SHADOW_WIDTH(tfw)+WIDTH+TF.internalWidth))	      return(-1);
if (pos>(int)(((Widget)tfw)->core.width-(SHADOW_WIDTH(tfw)+WIDTH+TF.internalWidth))) return(-2);

pos-=GetTextStart(tfw);
c1pos=0;
curs=TF.visible_start-1;
lst_ch=strlen(TF.buffer)-1;
do {
        curs++;
        c2pos=c1pos;
        c1pos=c2pos+XTextWidth(TF.font,&TF.buffer[curs],1);
} while (c1pos<pos && curs<lst_ch);
if (abs(c2pos-pos)>abs(c1pos-pos)) curs++;
return(curs);
}

/* Get the Time field from an event structure
 * The function returns false, if ev is the pointer to
 * an inappropiate Event, the function prints a warning
 * message and returns 'False'
 */

static Boolean GetEventTime( ev, time )
	XEvent		*ev;
	Time		*time;
{
switch (ev->type) {
	case KeyPress:
	case KeyRelease:	*time=ev->xkey.time; break;

	case ButtonPress:
	case ButtonRelease:	*time=ev->xbutton.time;break;

	case MotionNotify:	*time=ev->xmotion.time;break;

	case EnterNotify:
	case LeaveNotify:	*time=ev->xcrossing.time;break;

	default:
	  XtWarning("TextField: invalid event type (no time field)");
	  return(False);
}
return(True);
}

/* Retrieve a selection name from 
 * an action-procedure's arglist.
 * Prints a warning message and
 * returns 'False' if there are too
 * many arguments, or if the conversion 
 * yields no result.
 * for *nargs==0 *selection defaults to XA_PRIMARY
 *
 * if additional selections will be added to this
 * procedure, don't forget to care about them 
 * in 'Unselect()' 'AtomToBit()' and 'ClipboardXferDone()'
 */

static Boolean GetSelectionFromArgs(w,args, nargs, selection)
	Widget		w;
	String		*args;
	Cardinal	*nargs;
	Atom		*selection;
{
char mess[128];

if (*nargs>1) {
	XtWarning("TextField: Too many arguments");
	return(False);
}

if (*nargs>0) {
	switch( *args[0]){
	  case 'p': case 'P': *selection=XA_PRIMARY; 	  	     break;
	  case 's': case 'S': *selection=XA_SECONDARY;	  	     break;
	  case 'c': case 'C': *selection=XA_CLIPBOARD(XtDisplay(w)); break;

	  default:  
		(void)sprintf(mess,"TextField: %s invalid selection name",args[0]);
		XtWarning(mess);
		return(False);
	}
} else { /* no arguments */
	*selection=XA_PRIMARY;
}
return(True);
}

/* Action Procedures */


/* Move the Cursor
 * a) (ButtonPress, ButtonRelease or MotionNotify event)
 *	to the Pointer position. If this is one of the scroll
 *	marks then scroll the visible window one character.
 * b) (other events)
 *	evaluate 1st arg: 
 *		'E': move to eol
 *		'H': move to beginning of line
 *		<n>: move n chars
 */

static void MoveCursor( w, ev, args, nargs )
 Widget		w;
 XEvent		*ev;
 String		*args;
 Cardinal	*nargs;
{
TextFieldWidget tfw=(TextFieldWidget)w;
int   shift; 
int   len=strlen(TF.buffer);
char *chpt;

switch (ev->type) {
  case ButtonPress:
  case ButtonRelease: shift=PointerToCursor(tfw,ev->xbutton.x); break;

  case MotionNotify:  shift=PointerToCursor(tfw,ev->xmotion.x); break;

 /* retrieve args later */
  default:	      shift=-3; break;
}

switch(shift) {
      default: 			       TF.text_cursor=shift;	break;
      case -1: if (TF.visible_start>0) TF.text_cursor--;	break;
      case -2: if (TF.visible_end<len) TF.text_cursor++;	break;

/* retrieve args */
      case -3:

  	if (*nargs==0) {
   	TF.text_cursor++;
  	} else {
    	switch (*args[0]) {
		case 'e':
		case 'E':   TF.text_cursor=len; break;
		case 'h':
		case 'H':   TF.text_cursor=0; break;
	
        	default:
           	shift=(int) (strtol(args[0],&chpt,10));
	   	if (chpt==args[0]) {
			XtWarning("TextField: MoveCursor() wrong Parameter");
			shift=1;
  	   	}
	   	TF.text_cursor+=shift;
		if (shift==0) return;
	   	break;
   	}
  	}
        break;
}
if (TF.text_cursor<0)   TF.text_cursor=0;
if (TF.text_cursor>len) TF.text_cursor=len;

make_cursor_visible(tfw);
ReDisplay(tfw,True);
}

/* Insert one Character at cursor position (key event)
 * if any argument is present, insert 1st arg
 * a 2nd argument can be specified to suppress parsing of
 * special character sequences in the 1st arg (\t, \xxx expansion)
 */

static void InsertChar( w, ev, args, nargs )
 Widget		w;
 XEvent		*ev;
 String		*args;
 Cardinal	*nargs;
{
 TextFieldWidget tfw=(TextFieldWidget)w;
 register char *chpt=TF.buffer;
 char		ch,*valpt,chbuf[2];
 int		val,end;
 register int   idx;
 register int   curs=TF.text_cursor;
 Boolean	redisp=False, modified=True, parse=True;

 if (TF.readonly){
	XBell(XtDisplay((Widget)tfw),100);
	return;
 }

 if (*nargs>0) {
   valpt=args[0];
   if (*nargs>1) switch(*args[1]) {
	case 'n': case 'N': case 'F': case 'f': parse=False;
	default: break;
   }
 } else {
   if (ev->type!=KeyPress) return;
   if (XLookupString(&(ev->xkey),chbuf,1,NULL,NULL)==0) return;
   parse=False;
   chbuf[1]=(char)0;
   valpt=chbuf;
 }

 end=strlen(TF.buffer)+1;
 while( (val=(parse?str2chr(&valpt):(int)*valpt++)) > 0 ) {
   ch=(char)val;
   if (end>TF.maxlen) {
	XtWarning("TextField: Buffer full");
        XBell(XtDisplay(w),100);
	return;
   }

   Unselect(tfw,(Atom*)0);

   idx=end;
   while (idx>curs) {chpt[idx]=chpt[idx-1]; idx--;}
   chpt[TF.text_cursor++]=ch;
   modified=True; curs++; end++;

   if (!resize_width(tfw)) {
   if (TF.justify!=XtJustifyRight) {
     adj_right(tfw);
     if (TF.text_cursor>=TF.visible_end) {
       curs_visible_right(tfw);
     }
   } else {
     TF.visible_end++;
     adj_left(tfw);
   }
   redisp=True;
   }
 }

 if (val==-1) {
	XtWarning("TextField: InsertChar(): invalid argument");
 }
 
 if (modified) {
   if (redisp) ReDisplay(tfw,True);
   XtCallCallbackList((Widget)tfw,TF.modified_cbl,(XtPointer)TF.buffer);
 }
}

/* Called with an ev=NULL by ClipboardXferDone()
 */

static void Delete( w, ev, args, nargs )
 Widget		w;
 XEvent		*ev;
 String		*args;
 Cardinal	*nargs;
{
 TextFieldWidget tfw=(TextFieldWidget)w;
 register char *chpt=TF.buffer;
 int		len =strlen(TF.buffer);
 register int   curs=TF.text_cursor;
 register int   shift=0, argindx;
 Boolean	moreargs;

 if (TF.readonly) return;
 if (*nargs==0) {
    shift=(len>curs)?1:0;
 } else {
    argindx=0;
    do {
	moreargs=False;
	switch (*args[argindx]) {

	  /* delete to end of line */
  	  case 'e': case 'E':   shift= len-curs; break;

	  /* delete to beginning of line */
	  case 'h': case 'H':   shift= -curs; break;

	  /* delete all */
	  case 'a': case 'A':   shift= len; break;

	  /* delete highlighted area */
	  case 's': case 'S':   if (TF.hil_end>TF.hil_start) {
			        curs=TF.text_cursor=TF.hil_start;
			        shift=TF.hil_end-TF.hil_start;
				make_cursor_visible(tfw);
			        break;
			      } else  if (*nargs <2 ) {
				/* no second parameter that indicates 
				 * what to delete if there's no selection
				 */
				return;
			      }
				/* if there is, get next parameter and 
				 * evaluate it
				 */
			      argindx++;
			      moreargs=True;
			      break;

        default:
           shift=atoi(args[argindx]);
	   if (shift==0) {
		XtWarning("TextField: Delete() wrong Parameter");
		return;
  	   }
	   if (shift>len-curs) shift=len-curs;
	   if (-shift>curs)    shift=-curs;
	   break;
   	}
   } while (moreargs);
  }

 if (shift==0) return;

 if (ev!=NULL) { 
  /* Dont give up the Clipboard now if called
   * by ClipboardXferDone()
   */
  Unselect(tfw,(Atom*)0);
 } else {
  XtDisownSelection((Widget)tfw,XA_PRIMARY,CurrentTime);
  XtDisownSelection((Widget)tfw,XA_SECONDARY,CurrentTime);
  Unhighlight(tfw,(unsigned long)0);
 }

 if (shift==len) {     /* delete everything */
  TF.text_cursor=0;
  TF.visible_start=0;
  TF.visible_end=0;
  TF.buffer[0]=(char)0;
 } else if (shift>0) { /* delete to the right */
  curs--; do {curs++;} while((chpt[curs]=chpt[curs+shift])!=0);
 } else { 		/* delete to the left */
  curs--; do {curs++;} while((chpt[curs+shift]=chpt[curs])!=0);
  TF.text_cursor+=shift;
 }

 if (!resize_width(tfw)) { /* resize not possible */
  if (shift>0) {	   /* deleted to the right */
    TF.visible_end-=shift;
    if (TF.justify==XtJustifyRight) {
	if (TF.text_cursor<TF.visible_end) adj_left(tfw);
    } else {
	adj_right(tfw);
    }
    if (TF.text_cursor>=TF.visible_end &&
        (TF.justify!=XtJustifyLeft || TF.text_cursor<strlen(TF.buffer)) ) {
	curs_visible_right(tfw);
    }
   } else { 		/* delete to the left */
    TF.visible_end+=shift;
    if (TF.justify==XtJustifyLeft) {
	if (TF.text_cursor>TF.visible_start) adj_right(tfw);
    } else {
	adj_left(tfw);
    }
    if (TF.text_cursor <= TF.visible_start &&
        (TF.justify!=XtJustifyRight || TF.text_cursor>0 )) {
	curs_visible_left(tfw);
    }
   }
   ReDisplay(tfw,True);
 }
 XtCallCallbackList((Widget)tfw,TF.modified_cbl,(XtPointer)TF.buffer);
}

/*ARGSUSED*/
static void HighlightStart( w, ev, args, nargs )
 Widget		w;
 XEvent		*ev;
 String		*args;
 Cardinal	*nargs;
{
TextFieldWidget tfw=(TextFieldWidget)w;

if (ev->type==ButtonPress || ev->type==ButtonRelease) {
  TF.hil_start=TF.hil_end=PointerToCursor(tfw,ev->xbutton.x);
} else {
  TF.hil_start=TF.hil_end=TF.text_cursor;
}
ReDisplay(tfw,True);
}

/*ARGSUSED*/
static void HighlightExtend( w, ev, args, nargs )
 Widget		w;
 XEvent		*ev;
 String		*args;
 Cardinal	*nargs;
{
TextFieldWidget tfw=(TextFieldWidget)w;
int		eventpos;
String		word="W";
String		all="A";
Cardinal	n=1;

eventpos=TF.text_cursor;

/* Translations allow no coexistence of
 * Motion Events and multiclicks, so
 * we have to implement them
 */

if (ev->type==ButtonRelease) {
   if (TF.multiclick >= 3)  { /* maximum already reached */
	TF.multiclick=1;
   } else if (ev->xbutton.time-TF.multiclick_time< MULTI_TIME) { /* multiclick */
	TF.multiclick++;
   } else {
	TF.multiclick=1; /* reset */
   }
   TF.multiclick_time=ev->xbutton.time;

   switch (TF.multiclick) {
	case 1: /* 1st click  */ break;
	case 2: /* 2   clicks */ Highlight(w,ev,&word,&n); return;
	case 3: /* 3   clicks */ Highlight(w,ev,&all,&n); return;
   }
}


if (TF.hil_start<0) {
   TF.hil_start=TF.hil_end=eventpos;
   return;
}

if (TF.hil_start==TF.hil_end) {
  if (eventpos>TF.hil_start)	TF.hil_end=eventpos;
  else				TF.hil_start=eventpos;
} else {
 if (abs(eventpos-TF.hil_start)<abs(eventpos-TF.hil_end))
  TF.hil_start=eventpos;
 else 
  TF.hil_end  =eventpos;
}

ReDisplay(tfw,True);
}

static void Highlight( w, ev, args, nargs )
 Widget		w;
 XEvent		*ev;
 String		*args;
 Cardinal	*nargs;
{
TextFieldWidget tfw=(TextFieldWidget)w;
int		eventpos;
char		ch,*chpt;

if (*nargs==0) {
  XtWarning("TextField: Highlight(): no arguments");
  return;
}

switch(ev->type) {
	case ButtonPress:
	case ButtonRelease: 
		eventpos=PointerToCursor(tfw,ev->xbutton.x);
		break;
	case MotionNotify:
		eventpos=PointerToCursor(tfw,ev->xmotion.x);
		break;
	default: 
		eventpos=TF.text_cursor;
		break;
}

switch(*args[0]) {
	case 'w':
	case 'W': /* select word */
		if (eventpos<0) return;
		chpt=strchr(TF.buffer+eventpos,(int)' ');
		TF.hil_end=(chpt)?chpt-TF.buffer:strlen(TF.buffer);

		ch=TF.buffer[eventpos]; TF.buffer[eventpos]=0;
		chpt=strrchr(TF.buffer,(int)' ');
		TF.buffer[eventpos]=ch;
		TF.hil_start=(chpt)?chpt+1-TF.buffer:0;

		break;

	case 'a':
	case 'A': /* select all */
		  TF.hil_start=0; TF.hil_end=strlen(TF.buffer);
		  break;

	case 'e':
	case 'E': /* select to eol */
		  TF.hil_start=eventpos; TF.hil_end=strlen(TF.buffer);
		  break;

	case 'h':
	case 'H': /* select to home */
		  TF.hil_start=0; TF.hil_end=eventpos;
		  break;
	default:
  		XtWarning("TextField: Highlight(): wrong arguments");
		return;
}
ReDisplay(tfw,True);
}

static void MakeSelection( w, ev, args, nargs )
 Widget		w;
 XEvent		*ev;
 String		*args;
 Cardinal	*nargs;
{
TextFieldWidget tfw=(TextFieldWidget)w;
Atom		selection;
Display		*di=XtDisplay((Widget)tfw);
Time		time;

if (TF.hil_start>=TF.hil_end) return; /* nothing highlighted */

if (! (GetEventTime(ev,&time) &&
       GetSelectionFromArgs(w,args,nargs,&selection))) {
	XtWarning("(from MakeSelection())");
	return;
}

if (  !	XtOwnSelection(w,
			selection,
			time,
			ConvertSelection,
			LoseSelection,
			(XtSelectionDoneProc) 0)) {
	XtWarning("TextField: Selection ownership not granted");
	XBell(di,100);
   } else {
    TF.selections|=AtomToBit(di,selection);
   }

}


/*ARGSUSED*/
static void DeleteToClipboard( w, ev, args, nargs )
 Widget		w;
 XEvent		*ev;
 String		*args;
 Cardinal	*nargs;
{
TextFieldWidget tfw=(TextFieldWidget)w;
Time		time;
Display		*di=XtDisplay((Widget)tfw);

if (TF.hil_start>=TF.hil_end) return; /* nothing highlighted */
if (TF.readonly)	      return;

if ( ! GetEventTime(ev,&time)) {
	  XtWarning("(from DeleteToClipboard())");
	  return;
}

if (  !	XtOwnSelection(w,
			XA_CLIPBOARD(di),
			time,
			ConvertSelection,
			LoseSelection,
			ClipboardXferDone)) {
	XtWarning("TextField: Selection ownership not granted");
	XBell(di,100);
   } else {
    TF.selections|=CLIPBOARD_BIT;
   }
}

static void InsertSelection( w, ev, args, nargs )
 Widget		w;
 XEvent		*ev;
 String		*args;
 Cardinal	*nargs;
{
TextFieldWidget tfw=(TextFieldWidget)w;
Time		time;
Atom		selection;

if ( TF.readonly ) {
	XBell(XtDisplay((Widget)tfw),100);
	return;
}

if (! (GetEventTime(ev,&time) &&
       GetSelectionFromArgs(w,args,nargs,&selection))) {
	XtWarning("(from MakeSelection())");
	return;
}

XtGetSelectionValue(
	(Widget)tfw,
	selection,
	XA_STRING,
	DoPastingCallback,
	(XtPointer)(unsigned)TF.text_cursor,
	time);
}

/*ARGSUSED*/
static void Notify( w, ev, args, nargs )
 Widget		w;
 XEvent		*ev;
 String		*args;
 Cardinal	*nargs;
{
TextFieldWidget tfw=(TextFieldWidget)w;

XtCallCallbackList((Widget)tfw,TF.notify_cbl,(XtPointer)((*nargs>0) ? args[0] : 0));
}

/*ARGSUSED*/
static void CursorState( w, ev, args, nargs )
 Widget		w;
 XEvent		*ev;
 String		*args;
 Cardinal	*nargs;
{
TextFieldWidget tfw=(TextFieldWidget)w;

if ((*nargs) == 0) TF.active_cursor=!TF.active_cursor;
else {
 switch (*args[0]) {
	case 'a':
	case 'A': TF.active_cursor=True;  break;

	case 'i':
	case 'I': TF.active_cursor=False; break;

 	case 't':
	case 'T': TF.active_cursor=!TF.active_cursor; break;

	default:
	   XtWarning("TextField: CursorState(): wrong parameter");
	   return;
 }
}
		   
DrawCursor(tfw,
	   GetTextStart(tfw),
 	   TF.internalHeight+SHADOW_WIDTH(tfw)+TF.font->max_bounds.ascent);
}

/* Methods */

/*ARGSUSED*/
static void TfwExpose(w,ev,reg)
Widget w;
XEvent *ev;
Region reg;

{ 
TextFieldWidget tfw=(TextFieldWidget)w;
ReDisplay(tfw,False);

#ifdef XAW3D
 (*((TextFieldWidgetClass)(XtClass((Widget)tfw)))->threeD_class.shadowdraw)(w,ev,reg,
		(Boolean)(! (tfw->core.sensitive && !TF.readonly)));
#endif
}


static void ClassInitialize()
{
  static XtConvertArgRec formatCvtFromStringArgs[]={
	{ XtWidgetBaseOffset,
	  (XtPointer) XtOffsetOf(TextFieldRec,textField.input_format),
	  sizeof(String)
	},
	{ XtImmediate,
	  (XtPointer)True,
	  sizeof(XtPointer)
	}
  };
  static XtConvertArgRec formatCvtToStringArgs[]={
	{ XtWidgetBaseOffset,
	  (XtPointer) XtOffsetOf(TextFieldRec,textField.output_format),
	  sizeof(String)
	},
	{ XtImmediate,
	  (XtPointer)False,
	  sizeof(XtPointer)
	}
  };

  XawInitializeWidgetSet();
  XtAddConverter(XtRString, XtRJustify, XmuCvtStringToJustify,
		  (XtConvertArgList)NULL, 0);
#ifdef TEXT_COMPAT
  XtSetTypeConverter(XtRString, XtRResizeMode, CvtStringToResizeMode,
		  (XtConvertArgList)NULL, 0);
#endif
  XtSetTypeConverter(XtRString, XtRCFormat, CvtByFormat,
		  formatCvtFromStringArgs, XtNumber(formatCvtFromStringArgs), XtCacheAll, NULL);
  XtSetTypeConverter(XtRCFormat, XtRString, CvtByFormat,
		  formatCvtToStringArgs, XtNumber(formatCvtToStringArgs), XtCacheAll, NULL);
}

/*ARGSUSED*/
static void Initialize( treq, tnew, args, num_args)
 Widget	  treq, tnew;
 ArgList  args;
 Cardinal *num_args;
{
 TextFieldWidget tfw=(TextFieldWidget)tnew;
 Display	 *di=XtDisplay((Widget)tfw);
 Dimension       minh;
 char		 mess[256];
 int		 len;


 /* initialize Xmu's atom cache */
 (void)XmuInternAtom(di,XmuMakeAtom("NULL"));

 TF.norm=TF.insens=0;
 TF.selections=0;
 TF.active_cursor=False;
 /* set multicklick so that the count will be
  * reset upon first call of HighlightExtend()
  * anyway
  */
 TF.multiclick=4;
 TF.multiclick_time=(Time)0;

 Unselect(tfw,(Atom*)0);

 if (TF.use_string_in_place) {
	TF.buffer=TF.string;
	if (TF.maxlen < strlen(TF.string)) {
	  if (TF.maxlen) /* maxlen==0 means: set maxlen to strlen */
	    XtWarning("TextField: buffer length < strlen() (set to strlen())");
	  TF.maxlen=strlen(TF.string);
	}
 } else {
 	if (TF.maxlen<1) {
        	TF.maxlen=DEF_BUFFER_SIZE;
 	} else if (TF.maxlen>1023) {
		XtWarning("TextField: buffer length > 1023 (correct?)");
 	}
	
 	TF.buffer=(String)XtMalloc((unsigned) TF.maxlen+1);
 	TF.buffer[0]=TF.buffer[TF.maxlen]=(char)0;

 	if (strlen(TF.string)>TF.maxlen) {
		(void)sprintf(mess,"TextField: Buffer full (%i chars)",(int)TF.maxlen);
		XBell(XtDisplay((Widget)tfw),100);
		XtWarning(mess);
 	}

 	if (TF.string) (void)strncpy(TF.buffer,TF.string,TF.maxlen);

 	TF.string=TF.buffer;
 }

 len=strlen(TF.buffer);
 
 if (TF.text_cursor <  0 || TF.text_cursor>len) {
	XtWarning("TextField: cursor position out of range (using 0)");
	TF.text_cursor=0;
 }

 if (NEGDIMENSION(TF.internalHeight)) {
	XtWarning("TextField: internalHeight < 0 ? (using 0)");
	TF.internalHeight=0;
 }

 if (NEGDIMENSION(TF.internalWidth)) {
	XtWarning("TextField: internalWidth < 0 ? (using 0)");
	TF.internalWidth=0;
 }

 TF.visible_start=(short)0; TF.visible_end=(short)len;
 

 /* set Font */
 minh=TF.font->max_bounds.ascent
      + TF.font->max_bounds.descent
      + 2*(  TF.internalHeight + SHADOW_WIDTH(tfw));

 if (tfw->core.height < minh) {
   if (tfw->core.height > 0) {
     (void)sprintf(mess,"TextField: Height too small (using %i)",(int)minh);
     XtWarning(mess);
   }
   tfw->core.height=minh;
 }
 getGCs(tfw);

 if (tfw->core.width<=0 || RESIZE(tfw)) {
   tfw->core.width=  XTextWidth(TF.font,TF.buffer,len)
		   + 2*( TF.internalWidth + WIDTH 
		        +SHADOW_WIDTH(tfw));
 } else {
   switch (TF.justify) {
    case XtJustifyLeft: TF.visible_start=TF.text_cursor-TF.scroll_chars;
			if (TF.visible_start<0) TF.visible_start=0;
			adj_right(tfw);
			break;
    case XtJustifyRight:TF.visible_end=TF.text_cursor+TF.scroll_chars;
			if (TF.visible_end>len) TF.visible_end=len;
			adj_left(tfw);
			break;
    default:		adj_right (tfw);
   			adj_left (tfw);
   }
 }
}

static void Destroy(w)
Widget w;
{
TextFieldWidget tfw=(TextFieldWidget)w;

Unselect(tfw,(Atom*)0);
XtReleaseGC((Widget)tfw,TF.norm);
XtReleaseGC((Widget)tfw,TF.norm_hil);
XtReleaseGC((Widget)tfw,TF.insens);
XtReleaseGC((Widget)tfw,TF.insens_hil);

if (! TF.use_string_in_place) XtFree(TF.buffer);
}

static void Resize(w)
Widget w;
{
TextFieldWidget tfw=(TextFieldWidget)w;

TF.internalHeight=(int)(tfw->core.height
		  -TF.font->max_bounds.ascent
		  -TF.font->max_bounds.descent)/2 - SHADOW_WIDTH(tfw);

if (NEGDIMENSION(TF.internalHeight)) TF.internalHeight=0;

if (TF.justify==XtJustifyRight) {
  adj_left (tfw);
  adj_right(tfw);
} else {
  adj_right(tfw);
  adj_left (tfw);
}
}

#define NTF (ntfw->textField)

/*ARGSUSED*/
static Boolean SetValues( current, req, super, args, nargs)
Widget   current,req,super;
ArgList  args;
Cardinal *nargs;
{
 TextFieldWidget tfw=(TextFieldWidget)  current;
 TextFieldWidget ntfw=(TextFieldWidget) super;

 Boolean adjust=False,new_string,new_buffer;

 int     nlen;

 if (!XtIsSensitive((Widget)ntfw)&&XtIsSensitive((Widget)tfw)) { /* Switch to insensitive */
     NTF.active_cursor=False;
 }
 
 if (TF.font->fid !=NTF.font->fid ||
     tfw->core.background_pixel != ntfw->core.background_pixel ||
     TF.foreground!=NTF.foreground ||
     TF.insensitive_foreground!=NTF.insensitive_foreground) {
     XtReleaseGC((Widget)tfw,TF.norm);
     XtReleaseGC((Widget)tfw,TF.norm_hil);
     XtReleaseGC((Widget)tfw,TF.insens);
     XtReleaseGC((Widget)tfw,TF.insens_hil);
     getGCs(ntfw);
     adjust=(TF.font->fid!=NTF.font->fid);
 }

 if (!RESIZE(ntfw)) {
   if (adjust) {
     NTF.internalHeight=(int)( ntfw->core.height
			  -NTF.font->max_bounds.ascent
			  -NTF.font->max_bounds.descent)/2
			-SHADOW_WIDTH(ntfw);
     if (NEGDIMENSION(NTF.internalHeight)) {
	ntfw->core.height-=NTF.internalHeight;
	NTF.internalHeight=0;
     }
   } else {
   NTF.internalHeight=TF.internalHeight;
   }
   if ((int)NTF.internalWidth*2 > (int)tfw->core.width-30) {
	NTF.internalWidth=TF.internalWidth;
   } 
   adjust= (TF.internalWidth!=NTF.internalWidth);
 } else {
   if (adjust || (NTF.internalHeight!=TF.internalHeight)) {
    ntfw->core.height=NTF.font->max_bounds.ascent+NTF.font->max_bounds.descent
		    + 2*(SHADOW_WIDTH(ntfw)+NTF.internalHeight);
   }
 }

   if (NTF.use_string_in_place != TF.use_string_in_place) {
	XtWarning("TextField (SetValues): useStringInPlace cannot be changed");
	NTF.use_string_in_place = TF.use_string_in_place;
   }

   if (NTF.use_string_in_place) {
	if ( !  (new_buffer = (NTF.string != TF.string)) &&
   		(NTF.maxlen!=TF.maxlen)) {
	  XtWarning("TextField (SetValues): new length for old buffer req (use old)");
	  NTF.maxlen=TF.maxlen;
	}
	if (new_buffer) NTF.buffer=NTF.string;
   } else {
	if (0!=(new_buffer = (NTF.maxlen!=TF.maxlen))) {
		NTF.buffer=XtMalloc((unsigned)NTF.maxlen+1);
        	NTF.buffer[0]=NTF.buffer[NTF.maxlen]=0;
   	} else {
		NTF.buffer=TF.buffer;
   	}
   }

   new_string = (Boolean)(strcmp( NTF.string, TF.string));

   if (new_string || new_buffer){

        if (new_string || NTF.hil_start>NTF.maxlen) Unselect(tfw,(Atom*)0);

	if (!NTF.use_string_in_place) {
		(void)strncpy(NTF.buffer,NTF.string,NTF.maxlen);
		if (strlen(NTF.string)>NTF.maxlen) {
	   	XtWarning("TextField: Buffer full");
           	XBell(XtDisplay(ntfw),100);
        	}
        	if (new_buffer) XtFree(TF.buffer);
	}
	if (new_string){
	  NTF.visible_start=0;
	  NTF.visible_end=strlen(NTF.buffer);
	  adjust=True;
        } else {
	  if (NTF.visible_end>NTF.maxlen) {
		NTF.visible_end=NTF.maxlen;
		adjust=True;
          }
	  if (NTF.hil_end>NTF.maxlen) {
		NTF.hil_end=NTF.maxlen;
		adjust=True;
          }
	}
   }

   NTF.string=NTF.buffer;  
   nlen=strlen(NTF.buffer);
   if (NTF.text_cursor<0 ||
       NTF.text_cursor>nlen) {
	XtWarning("TextField: text cursor out of range");
	NTF.text_cursor=0;
   }
   if (NTF.scroll_chars<0) {
	XtWarning("TextField: scrollChars out of range");
	NTF.scroll_chars=1;
   }

   if (NTF.text_cursor>NTF.visible_end) {
     NTF.visible_end=NTF.text_cursor; adjust=True;
   }
   if (NTF.text_cursor<NTF.visible_start){
     NTF.visible_start=NTF.text_cursor; adjust=True;
   }

   if ((int)NTF.value.size<0) {
     XtWarning("TextField: valueSize<0");
     NTF.value.size=TF.value.size;
   }

   if (RESIZE(ntfw) && adjust) {
     ntfw->core.width=XTextWidth(NTF.font,NTF.buffer,strlen(NTF.buffer))
		      +2*(SHADOW_WIDTH(ntfw)+WIDTH+NTF.internalWidth);
     NTF.visible_end=nlen;
     NTF.visible_start=0;
   }
    adj_right(ntfw);
    adj_left(ntfw);

   return(True);
/* the following does not work correctly; no exposure events
 * are generated, if parent does not allow the new
 * geometry
 * return((ntfw->core.width==tfw->core.width)&&
 *        (ntfw->core.height==tfw->core.height));
 */
}
#undef NTF

static XtGeometryResult QueryGeometry(w,proposed,answer)
Widget w;
XtWidgetGeometry *proposed, *answer;
{
 TextFieldWidget tfw=(TextFieldWidget)w;
 Boolean ok=False;
 answer->request_mode=CWWidth|CWHeight;

 answer->width=XTextWidth(TF.font,TF.buffer,strlen(TF.buffer))
		+ 2*( SHADOW_WIDTH(tfw)+TF.internalWidth+WIDTH);

 answer->height= TF.font->max_bounds.ascent
		+TF.font->max_bounds.descent
		+2*(SHADOW_WIDTH(tfw)+TF.internalHeight);

 if ((proposed->request_mode&CWWidth) &&
      proposed->width==answer->width) ok=True;

 if ((proposed->request_mode&CWHeight) && proposed->width!=answer->width) ok=False;
 
 if (ok) 				return(XtGeometryYes);

 if (answer->width==tfw->core.width &&
     answer->height==tfw->core.height)  return(XtGeometryNo);
 else					return(XtGeometryAlmost);
}


/*ARGSUSED*/
static Boolean AcceptFocus(widget,time)
Widget widget;
Time   time;
{
/*
 * XtSetKeyboardFocus(XtParent(widget),widget);
 *
 * Doesn't work; The composite parent must
 * call XtSetKeyboardFocus. I dont want to use
 * XSetInputFocus(), since it interferes with 
 * XtSetKeyboardFocus.
 *
 * The Xt programmers manual is not consistent about
 * this method. Somewhere I read the method itself has
 * to set the KB focus, somewhere else was said, that
 * the method only tells the parent if the widget is 
 * willing to accept the focus. The latter is what we do.
 */
 return(XtIsSensitive(widget)&&XtIsManaged(widget));
}

/* Public Procedures */

/* convert the text string to the type indicated
 * by the valueType resource (valueSize must be set correctly)
 * and store the converted value at address valueAddress
 */

Boolean TextFieldString2Value(w)
Widget w;
{
 TextFieldWidget tfw=(TextFieldWidget)w;
 XrmValue from;
 if (! XtIsSubclass(w,textFieldWidgetClass)) {
   XtWarning("TextFieldString2Value: operand not a subclass of 'TextField'");
 }
 if (TF.value.addr==NULL) {
  XtWarning("TextField: valueAddress == NULL");
  return(False);
 }
 from.size=strlen(from.addr=TF.buffer);
 return(XtConvertAndStore((Widget)tfw,XtRString,&from,TF.value_type,&TF.value));
}

Boolean TextFieldValue2String(w)
Widget w;
{
 TextFieldWidget tfw=(TextFieldWidget)w;
 XrmValue to;
 Boolean  rval;
 int      len;

 if (! XtIsSubclass(w,textFieldWidgetClass)) {
   XtWarning("TextFieldString2Value: operand not a subclass of 'TextField'");
 }
 if (TF.value.addr==NULL) {
  XtWarning("TextField: valueAddress == NULL");
  return(False);
 }
 to.size=TF.maxlen; to.addr=TF.buffer;

 if (0!=(rval=XtConvertAndStore((Widget)tfw,
		TF.value_type,&TF.value,XtRString,&to))) {
   Unselect(tfw,(Atom*)0);
   len=strlen(TF.buffer);
   if (TF.text_cursor>len) TF.text_cursor=len;
   TF.visible_start=0; TF.visible_end=len;
   
   if (!resize_width(tfw)) {
     if (TF.justify==XtJustifyRight) {
       adj_left (tfw);
       adj_right(tfw);
     } else {
       adj_right(tfw);
       adj_left (tfw);
     }
   }
   ReDisplay(tfw,True);
 }
 return(rval);
}

/* Type converters */

static Boolean CvtByFormat(di,args,nargs,from,to)
Display 	*di;
XrmValuePtr	args;
Cardinal	*nargs;
XrmValuePtr	from;
XrmValuePtr	to;
{
 String  fmt;
 Boolean rval,fromString;
 char    mess[256];
 int	 typ=0;

 int     iarg=0;
 long	 larg=0;
 double  darg=0;

 if (*nargs != 2) {
  XtErrorMsg("wrong Parameters", "CvtByFormat",
	     "XtToolkitError",
	     "String to CFormat conversion needs 1 extra arg",
	     (String*)NULL,(Cardinal*)NULL);
 }
 (void)sprintf(mess,"CvtByFormat: ");

 fmt=*(String*)args[0].addr;
 fromString=(Boolean)(unsigned)(*(XtPointer*)args[1].addr);
 rval=True;

 if (fmt==NULL) {
  (void)sprintf(mess+13,"%s","no format string");
  XtWarning(mess); rval=False;
 } else if ((typ=strlen(fmt)-1)>80) {
  (void)sprintf(mess+13,"%s","format too long");
  XtWarning(mess); rval=False;
 } else if (fmt[0]!='%') {
  (void)sprintf(mess+13,"%s","possibly no format (missing %%)");
  XtWarning(mess); rval=False;
 }
 
 
 if (from->addr==NULL) {
  (void)sprintf(mess+13,"%s","no source address");
  XtWarning(mess); rval=False;
 } 
 if (to->addr==NULL) {
  (void)sprintf(mess+13,"%s","no target address");
  XtWarning(mess); rval=False;
 } 
 if (rval) 
  if (fromString) {
   rval=(sscanf(from->addr,fmt,to->addr)==1);
  } else
   switch(fmt[typ]) {
    case 'c':
    case 'd':
    case 'o':
    case 'u':
    case 'x':
    case 'X':  if (from->size==sizeof(char)) {
			if   (fmt[typ-1]=='l')	larg=(long)(*((char*)from->addr));
		   	else			iarg=(int) (*((char*)from->addr));
		} else if (from->size==sizeof(short)) {
			if   (fmt[typ-1]=='l')	larg=(long)(*((short*)from->addr));
		   	else			iarg=(int) (*((short*)from->addr));
		} else if (from->size==sizeof(int)) {
			if   (fmt[typ-1]=='l')	larg=(long)(*((int*)from->addr));
		   	else			iarg=(int) (*((int*)from->addr));
		} else if (from->size==sizeof(long)) {
			if   (fmt[typ-1]=='l')	larg=(long)(*((long*)from->addr));
		   	else			rval=False;
		} else {
			rval=False;
	       }
	       if (rval) {
                 if (fmt[typ-1]=='l') (void)sprintf(to->addr,fmt,larg);
                 else		      (void)sprintf(to->addr,fmt,iarg);
		
		 if (strlen(to->addr) > to->size) { /* !! String overflow !! */
  			XtErrorMsg("buffer overflow", "CvtByFormat",
	     			"XtToolkitError",
	     			"The target string is too small",
	     			(String*)NULL,(Cardinal*)NULL);

		 }

	       } 
	       break;

    case 'f':
    case 'e':
    case 'E':
    case 'g':
    case 'G':  if (from->size==sizeof(float)) {
		  darg=(double)(* ((float*)from->addr));
		} else if (from->size==sizeof(double)) {
		  darg=(double)(* ((double*)from->addr));
		} else {
			rval=False;
	       }
	       if (rval) {
		(void)sprintf(to->addr,fmt,darg);
		if (strlen(to->addr) > to->size) { /* !! String overflow !! */
  			XtErrorMsg("buffer overflow", "CvtByFormat",
	     			"XtToolkitError",
	     			"The target string is too small",
	     			(String*)NULL,(Cardinal*)NULL);

		}
	       }
	       break;

    default:   (void)sprintf(mess+13,"%s","wrong conversion character");
  	       XtWarning(mess); rval=False;
	       break;
 }

 if (!rval) {
  if (fromString) {
    (void)sprintf(mess,"%s using format '%s'",XtRCFormat,fmt);
    XtDisplayStringConversionWarning(di,(char*)from->addr,mess);
  } else {
    (void)sprintf(mess,"Cannot convert type %s to string %s using format %s",
	XtRCFormat,(char*)from->addr,fmt);
    XtWarning(mess);
  }
 }
 return(rval);
}

#ifdef TEXT_COMPAT
/* Do this to be compatible with the Text widget's 'resize' resource */
/* ARGSUSED */
static Boolean 
CvtStringToResizeMode(di, args, num_args, fromVal, toVal)
Display		*di;
XrmValuePtr 	args;		/* unused */
Cardinal	*num_args;	/* unused */
XrmValuePtr	fromVal;
XrmValuePtr	toVal;
{
  static XawTextResizeMode resizeMode;
  static  XrmQuark  QResizeNever, QResizeWidth, QResizeHeight, QResizeBoth;
  XrmQuark    q;
  char        lowerName[BUFSIZ];
  static Boolean inited = FALSE;
    
  if ( !inited ) {
    QResizeNever      = XrmPermStringToQuark(XtEtextResizeNever);
    QResizeWidth      = XrmPermStringToQuark(XtEtextResizeWidth);
    QResizeHeight     = XrmPermStringToQuark(XtEtextResizeHeight);
    QResizeBoth       = XrmPermStringToQuark(XtEtextResizeBoth);
    inited = TRUE;
  }

  XmuCopyISOLatin1Lowered (lowerName, (char *)fromVal->addr);
  q = XrmStringToQuark(lowerName);

  if      (q == QResizeNever)          resizeMode = XawtextResizeNever;
  else if (q == QResizeWidth)          resizeMode = XawtextResizeWidth;
  else if (q == QResizeHeight)         resizeMode = XawtextResizeHeight;
  else if (q == QResizeBoth)           resizeMode = XawtextResizeBoth;
  else {
    return False;
  }
  toVal->size=sizeof(XawTextResizeMode);
  toVal->addr=(XtPointer)&resizeMode;
  return True;
}
#endif

#undef TF
#undef SetPoint
