/*
 *	xparts.c - X bit for the chooser
 *	Copyright (C) 2000  Fred Barnes <frmb2@ukc.ac.uk>
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <X11/Xlib.h>
#include <X11/Xos.h>
#include <X11/Xatom.h>
#include <X11/Xutil.h>
#include <errno.h>

#include "support.h"
#include "cfg.h"
#include "interface.h"
#include "error.h"
#include "gadgets.h"


/* Globals */

MyXContext _xcontext;
GlobalConfigData _globconfig;


/* Statics */

#define MAX_NUM_GADGETS (10)
static int num_gadgets;
static Gadget *gadgets[MAX_NUM_GADGETS];
static Gadget *g_toplevel;
static Gadget *g_panel;
static Gadget *g_title;
static Gadget *g_hostlabel;
static Gadget *g_hostedit;
static Gadget *g_errorlabel;
static Gadget *g_statuslabel;
static Gadget *g_loginbutton;
static Gadget *g_querybutton;
static Gadget *g_copylabel;
static Gadget *g_extratext;
static int mainloop_running, mainloop_termcode;
static void (*pp_select_function)(char *);
static void (*pp_check_function)(char *);
static struct timeval xp_timeout;
static int xp_fd;
static void (*xp_timeout_fcn)(void);
static void (*xp_recv_fcn)(void);
static Atom wm_name_atom;
static Atom string_atom;

static void g_hostedit_accept (Gadget *gadget, char *string);
static void g_hostedit_escape (Gadget *gadget, char *string);
static void g_hostedit_ctrlc (Gadget *gadget, char *string);
static void g_loginbutton_click (Gadget *gadget);
static void g_querybutton_click (Gadget *gadget);
static int kill_window (char *window_name, Window start_at, int depth);


/* Own prototypes */


/*
 *	int init_xparts (char **in_argv, int in_argc)
 *	Connects to the X server, etc.
 *	Returns 0 on success, -1 on failure
 */
int init_xparts (char **in_argv, int in_argc)
{
	int xoff, yoff;
	int xdim, ydim;

	_xcontext.connected = 0;
	_xcontext.argv = in_argv;
	_xcontext.argc = in_argc;
	if (init_config_file (&_globconfig)) {
		APP_MSG ("error processing config file");
		return -1;
	}
	_xcontext.display = XOpenDisplay (NULL);
	if (!_xcontext.display) {
		APP_MSG ("unable to connect to remote display");
		return -1;
	}
	_xcontext.connected = 1;
	_xcontext.screen = DefaultScreen (_xcontext.display);
	pp_select_function = NULL;
	pp_check_function = NULL;
	xp_fd = -1;
	xp_timeout_fcn = NULL;
	xp_recv_fcn = NULL;
	xp_timeout.tv_sec = 0;
	xp_timeout.tv_usec = 0;
	/* Create the gadgets */
	xoff = (DisplayWidth (_xcontext.display, _xcontext.screen) >> 1) - (384 >> 1);
	yoff = (DisplayHeight (_xcontext.display, _xcontext.screen) >> 1) - (196 >> 1);
	num_gadgets = 0;
	xdim = 384, ydim = 196;
	if (_globconfig.extratext) {
		ydim += 16;
	}
	gadgets[num_gadgets++] = g_toplevel = create_toplevel (&_xcontext, "toplevel", xoff, yoff, xdim, ydim);
	gadgets[num_gadgets++] = g_panel = create_panel (&_xcontext, "panel", g_toplevel, 0, 0, xdim, ydim);
	gadgets[num_gadgets++] = g_title = create_label (&_xcontext, "title", g_panel, 1, 1, xdim-2, 48, "X Window System");
	gadgets[num_gadgets++] = g_hostlabel = create_label (&_xcontext, "hostlabel", g_panel, 96, 66, 48, 24, "Host :");
	gadgets[num_gadgets++] = g_hostedit = create_edit (&_xcontext, "hostedit", g_panel, 160, 64, 128, 28, 128, g_hostedit_accept, g_hostedit_escape, g_hostedit_ctrlc);
	gadgets[num_gadgets++] = g_copylabel = create_label (&_xcontext, "copylabel", g_panel, 1, 163, xdim-2, 16, "Copyright (C) 2000 Fred Barnes <frmb2@ukc.ac.uk>");
	gadgets[num_gadgets++] = g_errorlabel = create_label (&_xcontext, "errorlabel", g_panel, 1, 96, xdim-2, 24, "");
	g_errorlabel->hide_action (g_errorlabel);
	gadgets[num_gadgets++] = g_statuslabel = create_label (&_xcontext, "statuslabel", g_panel, 1, 96, xdim-2, 24, "");
	g_statuslabel->hide_action (g_statuslabel);
	if (_globconfig.hidequery) {
		gadgets[num_gadgets++] = g_loginbutton = create_button (&_xcontext, "loginbutton", g_panel, 128, 128, 128, 32, "Login", g_loginbutton_click);
	} else {
		gadgets[num_gadgets++] = g_loginbutton = create_button (&_xcontext, "loginbutton", g_panel, 88, 128, 96, 32, "Login", g_loginbutton_click);
		gadgets[num_gadgets++] = g_querybutton = create_button (&_xcontext, "querybutton", g_panel, 200, 128, 96, 32, "Query", g_querybutton_click);
	}
	if (_globconfig.extratext) {
		/* Place just under g_copylabel */
		gadgets[num_gadgets++] = g_extratext = create_label (&_xcontext, "extratext", g_panel, 1,
			g_copylabel->y + (g_copylabel->fontinfo->ascent + g_copylabel->fontinfo->descent) + 4, xdim-2, 16, "");
	}
	return 0;
}


/*
 *	void stop_xparts (void)
 *	Stops the X parts
 */
void stop_xparts (void)
{
	if (!_xcontext.connected) {
		ERR_MSG ("Not connected!");
		return;
	}
	destroy_button (g_loginbutton);
	if (!_globconfig.hidequery) {
		destroy_button (g_querybutton);
	}
	destroy_label (g_statuslabel);
	destroy_label (g_errorlabel);
	destroy_edit (g_hostedit);
	destroy_label (g_hostlabel);
	destroy_label (g_copylabel);
	destroy_label (g_title);
	destroy_panel (g_panel);
	destroy_toplevel (g_toplevel);
	XCloseDisplay (_xcontext.display);
	_xcontext.connected = 0;
	return;
}


/*
 *	int kill_window (char *window_name, Window start_at, int depth)
 *	recursively kills a top-level window (in-case WM wrapped it up)
 */
static int kill_window (char *window_name, Window start_at, int depth)
{
	Window root_return, parent_return;
	Window *children;
	int n_children, i;
	Atom actual_type;
	int actual_format;
	unsigned long nitems, bytes_after;
	unsigned char *prop;
	int did_find;

	if (!XQueryTree (_xcontext.display, start_at, &root_return, &parent_return, &children, &n_children)) {
		return 0;
	}
	did_find = 0;
	for (i=0; (i<n_children) && !did_find; i++) {
		/* note: not XFree()ing memory return by XGetWindowProperty() in here */
		if (XGetWindowProperty (_xcontext.display, children[i], wm_name_atom, 0, 64, False, string_atom, &actual_type,
			&actual_format, &nitems, &bytes_after, &prop) != Success) {
			continue;
		}
		if ((actual_format == 8) && (actual_type == string_atom) && (nitems > 0)) {
			if (!strncmp (window_name, (char *)prop, strlen (window_name))) {
				did_find = 1;
				/* better kill it :-) */
				XDestroyWindow (_xcontext.display, children[i]);
				XFlush (_xcontext.display);
			}
		}
		if (depth > 1) {
			did_find = kill_window (window_name, children[i], depth-1);
		}
		if ((actual_format > 0) && (nitems > 0)) {
			XFree (prop);
		}
	}
	XFree (children);
	return did_find;
}


/*
 *	void xparts_kill_window (char *window_name)
 *	kills a top-level window based on WM_NAME property
 */
void xparts_kill_window (char *window_name)
{
	Window the_root;

	wm_name_atom = XInternAtom (_xcontext.display, "WM_NAME", True);
	if (wm_name_atom == None) {
		return;
	}
	string_atom = XInternAtom (_xcontext.display, "STRING", True);
	if (string_atom == None) {
		return;
	}
	the_root = RootWindow (_xcontext.display, _xcontext.screen);
	kill_window (window_name, the_root, 3);
	return;
}


/*
 *	int xparts_mainloop (void)
 *	Does the mainloop thing
 */
int xparts_mainloop (void)
{
	XEvent ev;
	int i, result, hi_fd, selret;
	fd_set read_set;

	if (!_xcontext.connected) {
		ERR_MSG ("Not connected!");
		return -1;
	}
	mainloop_running = 1;
	mainloop_termcode = 0;
	result = XGrabKeyboard (_xcontext.display, g_hostedit->window, False, GrabModeAsync, GrabModeAsync, CurrentTime);
	if (result) {
		ERR_MSG ("Unable to XGrabKeyboard ()");
		return -1;
	}
	while (mainloop_running) {
		if ((xp_fd > -1) && !XPending (_xcontext.display)) {
			FD_ZERO (&read_set);
			FD_SET (xp_fd, &read_set);
			FD_SET (ConnectionNumber (_xcontext.display), &read_set);
			if (xp_fd > ConnectionNumber (_xcontext.display)) {
				hi_fd = xp_fd;
			} else {
				hi_fd = ConnectionNumber (_xcontext.display);
			}
			selret = select (hi_fd+1, &read_set, NULL, NULL, &xp_timeout);
			if (!selret) {
				if (xp_timeout_fcn) {
					xp_timeout_fcn ();
				}
				continue;
			} else if (selret == -1) {
				if (errno == EINTR) {
					continue;
				}
				ERR_MSG ("select () == -1: %s", strerror(errno));
				return -1;
			}
			if (FD_ISSET (xp_fd, &read_set)) {
				if (xp_recv_fcn) {
					xp_recv_fcn ();
				}
			}
			if (!FD_ISSET (ConnectionNumber (_xcontext.display), &read_set)) {
				continue;
			}
		}
		XNextEvent (_xcontext.display, &ev);
		for (i=0; i<num_gadgets; i++) {
			if (ev.xany.window == gadgets[i]->window) {
				break;
			}
		}
		if (i == num_gadgets) {
			continue;
		}
		switch (ev.type) {
		case Expose:
			if (gadgets[i]->expose_event) {
				gadgets[i]->expose_event (&ev, gadgets[i]);
			}
			break;
		case ButtonPress:
			if (gadgets[i]->mousedown_event) {
				gadgets[i]->mousedown_event (&ev, gadgets[i]);
			}
			break;
		case ButtonRelease:
			if (gadgets[i]->mouseup_event) {
				gadgets[i]->mouseup_event (&ev, gadgets[i]);
			}
			break;
		case KeyPress:
		case KeyRelease:
			if (gadgets[i]->keypress_event) {
				gadgets[i]->keypress_event (&ev, gadgets[i]);
			}
			break;
		case EnterNotify:
			if (gadgets[i]->mouseenter_event) {
				gadgets[i]->mouseenter_event (&ev, gadgets[i]);
			}
			break;
		case LeaveNotify:
			if (gadgets[i]->mouseleave_event) {
				gadgets[i]->mouseleave_event (&ev, gadgets[i]);
			}
			break;
		default:
			break;
		}
	}
	XUngrabKeyboard (_xcontext.display, CurrentTime);
	return mainloop_termcode;
}


/*
 *	void xparts_readtimeout (int fd, struct timeval *tv, void (*recv_fcn)(void), void (*timeout_fcn)(void));
 *	Sets the timeout to `to' -> `timeout_fcn()' or read ready on `fd' -> `recv_fcn'
 */
void xparts_readtimeout (int fd, struct timeval *tv, void (*recv_fcn)(void), void (*timeout_fcn)(void))
{
	xp_fd = fd;
	if (xp_fd == -1) {
		return;
	}
	xp_timeout.tv_sec = tv->tv_sec;
	xp_timeout.tv_usec = tv->tv_usec;
	xp_timeout_fcn = timeout_fcn;
	xp_recv_fcn = recv_fcn;
	return;
}


/*
 *	void set_error_message (char *message)
 *	Displays an error message
 */
void set_error_message (char *message)
{
	g_errorlabel->settext_action (g_errorlabel, message);
	g_statuslabel->hide_action (g_statuslabel);
	g_errorlabel->show_action (g_errorlabel);
	return;
}


/*
 *	void set_status_message (char *message)
 *	Displays a status message
 */
void set_status_message (char *message)
{
	g_statuslabel->settext_action (g_statuslabel, message);
	g_errorlabel->hide_action (g_errorlabel);
	g_statuslabel->show_action (g_statuslabel);
	return;
}


/*
 *	void give_select_function (void (*fcn)(char *))
 *	Provides the select function for selecting a host
 */
void give_select_function (void (*fcn)(char *))
{
	pp_select_function = fcn;
	return;
}


/*
 *	void give_check_function (void (*fcn)(char *))
 *	Provides the check function for checking a hosts availability
 */
void give_check_function (void (*fcn)(char *))
{
	pp_check_function = fcn;
	return;
}


/*
 *	void mainloop_terminate (int termcode)
 *	Terminates the mainloop
 */
void mainloop_terminate (int termcode)
{
	mainloop_running = 0;
	mainloop_termcode = termcode;
	return;
}


/*
 *	void g_hostedit_escape (Gadget *gadget, char *string)
 *	Called when ESC pressed on host edit
 */
static void g_hostedit_escape (Gadget *gadget, char *string)
{
	if (pp_check_function) {
		pp_check_function (string);
	}
	return;
}


/*
 *	void g_hostedit_ctrlc (Gadget *gadget, char *string)
 *	Called when ctrl-c is pressed on host edit
 */
static void g_hostedit_ctrlc (Gadget *gadget, char *string)
{
	mainloop_terminate (255);
	return;
}


/*
 *	void g_hostedit_accept (Gadget *gadget, char *string)
 *	Called when enter/return is pressed in the edit box
 */
static void g_hostedit_accept (Gadget *gadget, char *string)
{
	if (pp_select_function) {
		pp_select_function (string);
	}
	return;
}


/*
 *	void g_loginbutton_click (Gadget *gadget)
 *	Called when login button is clicked
 */
static void g_loginbutton_click (Gadget *gadget)
{
	if (pp_select_function) {
		pp_select_function (text_of_edit (g_hostedit));
	}
	return;
}


/*
 *	void g_querybutton_click (Gadget *gadget)
 *	Called when query button is clicked
 */
static void g_querybutton_click (Gadget *gadget)
{
	if (pp_check_function) {
		pp_check_function (text_of_edit (g_hostedit));
	}
	return;
}



