/*
  XBubble - xbubble.c

  Copyright (C) 2002  Ivan Djelic <ivan@savannah.gnu.org>
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <unistd.h>
#include <sys/time.h>

#if TIME_WITH_SYS_TIME
#include <time.h>
#endif

#include <X11/Xlib.h>
#include <X11/Xutil.h>
/*
#include <mcheck.h>
*/
#include "data.h"
#include "game.h"
#include "screens.h"
#include "init.h"

Display *display;
Visual *visual;
Window root, win;
int screen, depth;
Colormap colormap;

int scale, win_width, win_height, fps;
unsigned long frame_duration;
char *data_dir;
int colors[NB_CELLS];

#define DEFAULT_FPS        100
#define DEFAULT_HANDICAP   7000
#define DEFAULT_PERIOD     12
#define level_handicap(X)  ( DEFAULT_HANDICAP + 3000*( NORMAL - (X)))

void usage() {
  printf("Usage: xbubble [-d display] [-s scale] [-f framerate] ");
  printf("[-r resourcepath]\n\n");
  exit(1);
}

static void randomize_colors( int nb_rows ) {
  int i;
  /* preload bubbles */
  for ( i = 0; i < nb_rows*COLS; i++ )
    colors[i] = rnd( NB_COLORS );
}

void play_single_game( int *handicap ) {
  int score[2]  = { 0, 0 };
  Game game;
  game = new_game( SINGLE_PLAYER, handicap, False, NULL, 1, score, NORMAL );
  play_game(game);
  delete_game( game, False );
}

void play_challenge_game( int *handicap ) {
  int score[2]  = { 0, 0 };
  Game game;
  /* preload bubbles */
  randomize_colors(5);
  game = new_game( SINGLE_PLAYER, handicap, True, colors, 1, score, NORMAL );
  play_game(game);
  delete_game( game, False );
}

void play_match( enum GameMode mode, int *handicap, enum Level level ) {
  int games[2]  = { 0, 0 };
  int rounds[2] = { 0, 0 };
  int round, not_done, score[2];
  enum GameResult result;
  Game game;
  do {
    score[0] = 0;
    score[1] = 0;
    not_done = 1;
    for ( round = 1; not_done; round++ ) {
      /* preload bubbles */
      randomize_colors(5);
      game = new_game( mode, handicap, True, colors, round, score, level );
      result = play_game(game);
      not_done = ((( score[0] + score[1] < 2 )||( score[0] == score[1] ))&&
		  ( result != ABORTED ));
      delete_game( game, not_done );
    }
    /* update stats */
    rounds[0] += score[0];
    rounds[1] += score[1];
    if ( result != ABORTED ) {
      games[0] += ( score[0] > score[1] )? 1 : 0 ;
      games[1] += ( score[1] > score[0] )? 1 : 0 ;
    }
  }
  while ( screen3( mode, rounds, games ) == S3_CONTINUE );
}

void play_xbubble() {
  int i, quit, level;
  int choice[2], handicap[2];

  /* empty preloaded bubbles array */
  for ( i = 0; i < NB_CELLS; i++ )
    colors[i] = -1; /* empty */
  quit = 0;

  while ( ! quit ) {
    handicap[0] = level_handicap(NORMAL);
    handicap[1] = level_handicap(NORMAL);
    switch ( screen1() ) {

    case S1_SINGLE_PLAYER:
      /* select handicap */
      choice[0] = screen5();
      if ( choice[0] != S5_ESCAPE ) {
	handicap[0] = level_handicap( choice[0] );
	play_single_game( handicap );
      }
      break;

    case S1_TWO_PLAYERS:
      /* select players handicaps */
      if ( screen4(choice) == S4_OK ) {
	handicap[0] = level_handicap( choice[0] );
	handicap[1] = level_handicap( choice[1] );
	play_match( TWO_PLAYERS, handicap, NORMAL );
      }
      break;

    case S1_PLAYER_VS_COMPUTER:
      /* select computer skill */
      level = screen2();
      if ( level != S2_ESCAPE ) {
	level += VERY_EASY - S2_VERY_EASY;
	play_match( PLAYER_VS_COMPUTER, handicap, level );
      }
      break;

    case S1_CHALLENGE:
      /* select handicap */
      choice[0] = screen5();
      if ( choice[0] != S5_ESCAPE ) {
	handicap[0] = level_handicap( choice[0] );
	play_challenge_game( handicap );
      }
      break;

    case S1_QUIT:
      quit = 1;
      break;
    default:
      break;
    }
  }
}

int main( int argc, char ** argv ) {
  double zoom;
  int c;
  char *display_name, app_name[128];
  unsigned long timer_resolution;
  int max_win_width, pscale;
  struct timeval tv;
  XSizeHints xsh;
  XWMHints wm_hint;
  XClassHint class_hint;

  /* default values */
  fps = DEFAULT_FPS;
  data_dir = DATADIR;
  pscale = 0;
  display_name = NULL;
  /* mtrace(); */

  /* parse options */
  while (( c = getopt( argc, argv, "s:f:r:d:")) != -1 ) {
    switch (c) {
    case 's': /* scale */
      if ( sscanf( optarg, "%d", &pscale ) != 1 )  {
	fprintf(stderr, "Invalid scale: %s.\n", optarg);
	usage();
      }
      break;
    case 'f': /* frame rate */
      if ( sscanf( optarg, "%d", &fps ) != 1 )  {
	fprintf(stderr, "Invalid frame rate: %s.\n", optarg);
	usage();
      }
      break;
    case 'r': /* resources */
      data_dir = optarg;
      break;
    case 'd': /* display */
      display_name = optarg;
      break;
    default:
      usage();
    }
  }
  init_display( display_name );
  /* XSynchronize( display, True ); */
  /* adjust frame rate */
  fps = clip( fps, 10, 100 );
  frame_duration = 1000000 / fps;
  timer_resolution = 1000000 / CLK_TCK;
  if ( frame_duration % timer_resolution )
    frame_duration = ((frame_duration/timer_resolution)+1) * timer_resolution;
  fps = 1000000 / frame_duration;

  /* adjust scale */
  max_win_width = game_win_width(MAX_SCALE);
  if ( ! pscale ) /* automagically adjust window size to 75% of screen width */
    pscale = 75*WidthOfScreen(DefaultScreenOfDisplay(display))/max_win_width;
  pscale = clip( pscale, 5, 100 );
  scale = pscale*MAX_SCALE/100;
  zoom = 1.0*scale/MAX_SCALE;
  win_width = game_win_width(scale);
  win_height = game_win_height(scale);

  /* create main window */
  win = XCreateSimpleWindow( display, root, 0, 0, win_width, win_height,0,0,0);
  XSelectInput( display, win,
		ExposureMask|StructureNotifyMask|KeyPressMask|KeyReleaseMask );
  if ( depth == 8 )
    XSetWindowColormap( display, win, colormap );
  /* we don't want window resizing */
  xsh.flags = PBaseSize | PMinSize | PMaxSize;
  xsh.min_width = xsh.max_width = xsh.base_width = win_width;
  xsh.min_height = xsh.max_height = xsh.base_height = win_height;
  XSetWMNormalHints( display, win, &xsh );

  /* set window title name, name, class, etc. */
  sprintf( app_name, "XBubble @ %d fps  v%s", fps, VERSION );
  XStoreName( display, win, app_name );
  class_hint.res_name = "xbubble";
  class_hint.res_class = "XBubble";
  wm_hint.flags = InputHint | StateHint;
  wm_hint.input = True;
  wm_hint.initial_state = NormalState;
  XSetClassHint( display, win, &class_hint );
  XSetWMHints( display, win, &wm_hint );

  /* new randomize seed */
  gettimeofday( &tv, NULL );
  srand( tv.tv_usec );

  splash_screen(zoom);
  play_xbubble();

  cleanup_graphics();
  XDestroyWindow( display, win );
  if ( depth == 8 )
    XFreeColormap( display, colormap );
  XCloseDisplay( display );
  exit(0);
}
