/*
  XBubble - sprite.c

  Copyright (C) 2002  Ivan Djelic <ivan@savannah.gnu.org>
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdlib.h>
#include <X11/Xlib.h>

#include "utils.h"
#include "rectangle.h"
#include "frame.h"
#include "sprite.h"

extern Display *display;
extern Window root;

struct _Sprite {
  int layer, frame, clock, x, y, dx, dy, dw, dh;
  int new, finished, cycle;
  int needs_draw;
  int needs_erase;  
  int needs_redraw;
  Set overlap;
  RectangleList redraw_boxes;
  Set animation;
};

struct _SpritePool {
  int nb_layers;
  Set pool, *layer;
  RectangleList redraw_boxes;
  GC mask_gc, normal_gc;
};

/*********************** Sprite routines *********************************/

Sprite new_sprite( int layer, int overlap_max, Set a, int frame, int cycle ) {
  Sprite s = ( Sprite ) xmalloc( sizeof( struct _Sprite ));
  s->layer = layer;
  s->animation = a;
  s->frame = frame;
  s->cycle = cycle;
  s->clock = 0;
  s->dx = -1;
  s->dy = -1;
  s->overlap = new_set(overlap_max);
  s->redraw_boxes = new_rectangle_list(overlap_max);
  return s;
}

void delete_sprite(Sprite s) { 
  delete_set(s->overlap);
  delete_rectangle_list(s->redraw_boxes);
  free(s);
}

void set_sprite_position( Sprite s, int x, int y ) {
  if (( x != s->x )||( y != s->y )) {
    s->x = x;
    s->y = y;
    s->needs_erase = 1;
  }
}

void set_sprite_animation( Sprite s, Set a, int frame, int cycle ) {
  s->animation = a;
  s->frame = frame;
  s->clock = 0;
  s->cycle = cycle;
  s->needs_erase = 1;
}

Set get_sprite_animation( Sprite s ) {
  return s->animation;
}

void set_sprite_layer( Sprite s, int layer ) {
  if ( layer != s->layer ) {
    s->layer = layer;
    s->needs_erase = 1;
  }
}

void set_sprite_frame( Sprite s, int frame ) {
  int frame_max = s->animation->size;
  if ( frame >= frame_max)
    frame = ( s->cycle )? frame % frame_max : frame_max-1;
  if ( s->frame !=  frame ) {
    if ( ((Frame) s->animation->element[s->frame])->pixmap != 
	 ((Frame) s->animation->element[frame])->pixmap )
      s->needs_erase = 1;
    s->frame = frame;
  }
}

void increase_sprite_frame( Sprite s, int increase ) {
  set_sprite_frame( s, s->frame + increase );
}

void increase_sprite_clock( Sprite s, int dt ) {
  s->clock += dt;
  while ( s->clock >= ((Frame) s->animation->element[s->frame])->delay ) {
    s->clock -= ((Frame) s->animation->element[s->frame])->delay;
    set_sprite_frame( s, s->frame + 1 );
  }
}

/************************ SpritePool routines ******************************/

SpritePool new_sprite_pool( int nb_layers, int nb_sprites_max ) {
  int i;
  XGCValues gcv;
  unsigned long gcm;
  SpritePool sp = (SpritePool) xmalloc( sizeof( struct _SpritePool ));
  sp->layer = (Set*) xmalloc( sizeof( Set ) * nb_layers );
  sp->pool = new_set( nb_sprites_max );
  sp->redraw_boxes = new_rectangle_list( nb_sprites_max );
  for ( i = 0; i < nb_layers; i++ )
    sp->layer[i] = new_set( nb_sprites_max );
  sp->nb_layers = nb_layers;
  gcm = GCFunction | GCGraphicsExposures;
  gcv.graphics_exposures = False;
  gcv.function = GXcopy;
  sp->normal_gc = XCreateGC( display, root, gcm, &gcv);
  sp->mask_gc = XCreateGC( display, root, gcm, &gcv);
  return sp;
}

void delete_sprite_pool( SpritePool sp ) {
  int i;
  XFreeGC( display, sp->mask_gc);
  XFreeGC( display, sp->normal_gc);
  for ( i = 0; i < sp->nb_layers; i++ )
    delete_set(sp->layer[i]);
  delete_set(sp->pool);
  delete_rectangle_list(sp->redraw_boxes);
  free(sp->layer);
  free(sp);
}

void empty_sprite_pool( SpritePool sp ) {  
  empty_rectangle_list( sp->redraw_boxes );
  empty_set( sp->pool );
}

void add_sprite_to_pool( SpritePool sp, Sprite s ) {
  add_element_to_set( sp->pool, s);
  s->new = 1;
  s->needs_erase = 0;
  s->needs_draw = 0;
  s->needs_redraw = 0;
  s->finished = 0;
}

void remove_sprite_from_pool( SpritePool sp, Sprite s ) {
  if ( s->new )
    remove_element_from_set( sp->pool, s);
  else
    s->finished = 1;
}

static int sprite_overlap( Sprite s, int x, int y, int w, int h ) {
  return (( s->dx < x + w )&&( s->dy < y + h )&&
	  ( s->dx + s->dw > x )&&( s->dy + s->dh > y ));
}

RectangleList get_sprite_pool_redraw_list( SpritePool sp ) {
  return sp->redraw_boxes;
}

/**********************************************************************/

static void draw_sprite( SpritePool sp, Sprite s, Pixmap dest ) {
  Frame frame = (Frame) s->animation->element[s->frame];
  if ( frame->has_mask ) {
    XSetClipMask( display, sp->mask_gc, frame->mask );
    XSetClipOrigin( display, sp->mask_gc, s->dx, s->dy);
    XCopyArea( display, frame->pixmap, dest, sp->mask_gc, 
	       0, 0, s->dw, s->dh, s->dx, s->dy);
  }
  else
    XCopyArea( display, frame->pixmap, dest, sp->normal_gc, 
	       0, 0, s->dw, s->dh, s->dx, s->dy);
}

static void redraw_sprite( SpritePool sp, Sprite s, Pixmap dest ) {
  int i;
  Rectangle *r;
  Frame frame;
  GC redraw_gc = sp->normal_gc;
  frame = (Frame) s->animation->element[s->frame];
  if ( frame->has_mask ) {
    XSetClipMask( display, sp->mask_gc, frame->mask );
    XSetClipOrigin( display, sp->mask_gc, s->dx, s->dy );
    redraw_gc = sp->mask_gc;
  }
  for ( i = 0; i < s->redraw_boxes->size; i++ ) {
    r = &s->redraw_boxes->element[i];
    XCopyArea( display, frame->pixmap, dest, redraw_gc, r->x1, r->y1,
	       r->x2 - r->x1, r->y2 - r->y1, r->x1 + s->dx, r->y1 + s->dy );
  }
}

static void erase_sprite( SpritePool sp, Sprite s, Pixmap dest, Pixmap bg ) {
  XCopyArea( display, bg, dest, sp->normal_gc, 
	     s->dx, s->dy, s->dw, s->dh, s->dx, s->dy);
}

static void sprite_partial_redraw( Sprite s, int x, int y, int w, int h ) {
  int x1, y1, x2, y2, tmp;
  /* intersect rectangles */
  x1 = ( s->dx > x ) ? 0 : x - s->dx;
  y1 = ( s->dy > y ) ? 0 : y - s->dy;
  x2 = ( s->dx + s->dw < x + w ) ? s->dw : x + w - s->dx;
  y2 = ( s->dy + s->dh < y + h ) ? s->dh : y + h - s->dy;
  if ( x1 >= x2 ) {
    tmp = x2;
    x2 = x1;
    x1 = tmp;
  }
  if ( y1 >= y2 ) {
    tmp = y2;
    y2 = y1;
    y1 = tmp;
  }
  add_disjoint_rectangle_to_list( s->redraw_boxes, x1, y1, x2, y2);
}

void redraw_sprite_pool( SpritePool sp, Pixmap pixmap, Pixmap background ) {
  int i, j, layer;
  Sprite s, t;
  Frame frame;

  /* empty layers */
  for ( layer = 0; layer < sp->nb_layers; layer++ )
    empty_set( sp->layer[layer] );
  empty_rectangle_list( sp->redraw_boxes );
  
  for ( i = 0; i < sp->pool->size; i++ ) {
    s = sp->pool->element[i];
    /* check if sprite needs to be erased */
    if ((( s->needs_erase )&&( ! s->new ))||( s->finished )) {
      erase_sprite( sp, s, pixmap, background);
      /*
      merge_rectangle_with_list( sp->redraw_boxes, s->dx, s->dy, 
				 s->dx + s->dw, s->dy + s->dh);
      */
      add_disjoint_rectangle_to_list( sp->redraw_boxes, s->dx, s->dy, 
				      s->dx + s->dw, s->dy + s->dh);
      /* partially redraw overlapping sprites */
      for ( j = 0; j < s->overlap->size; j++ ) {
	t = s->overlap->element[j];
	t->needs_redraw = 1;
	sprite_partial_redraw( t, s->dx, s->dy, s->dw, s->dh );
	remove_element_from_set( t->overlap, s);
      }
    }
    /* check if sprite needs full redraw */
    if ((( s->needs_erase )&&( ! s->finished ))||( s->new )) {
      s->new = 0;
      s->needs_erase = 0;
      s->needs_draw = 1;
      /*  compute new sprite location */
      frame = s->animation->element[s->frame];
      s->dw = frame->width;
      s->dh = frame->height;
      s->dx = s->x - frame->cx;
      s->dy = s->y - frame->cy;

      /* (re)compute overlapping sets */
      empty_set( s->overlap );      
      for ( j = 0; j < sp->pool->size; j++ ) {
	t = sp->pool->element[j];
	if (( s != t )&&( ! t->new )&&
	    ( sprite_overlap( t, s->dx, s->dy, s->dw, s->dh ))) {
	  add_element_to_set( t->overlap, s);
	  add_element_to_set( s->overlap, t);
	  /* redraw overlapping sprites on upper layers */
	  if ( t->layer > s->layer ) {
	    t->needs_redraw = 1;
	    sprite_partial_redraw( t, s->dx, s->dy, s->dw, s->dh );
	  }
	}
      }
    }
    if ( s->finished ) {
      remove_element_from_set_at( sp->pool, i);
      i--;
    }
    else /* sort sprites into layers */
      add_element_to_set( sp->layer[s->layer], s );
  }
  
  /* redraw each layer starting from the bottom */
  for ( layer = 0; layer < sp->nb_layers; layer++ )
    for ( i = 0; i < sp->layer[layer]->size; i++ ) {
      s = sp->layer[layer]->element[i];
      if ( s->needs_draw ) {
	draw_sprite( sp, s, pixmap );
	merge_rectangle_with_list( sp->redraw_boxes, 
				   s->dx, s->dy, s->dx + s->dw, s->dy + s->dh);
	/*
	add_disjoint_rectangle_to_list( sp->redraw_boxes, s->dx, s->dy, 
					s->dx + s->dw, s->dy + s->dh);
	*/
      }
      else
	if ( s->needs_redraw )
	  redraw_sprite( sp, s, pixmap );
      s->needs_draw = 0;
      s->needs_redraw = 0;
      empty_rectangle_list(s->redraw_boxes);
    }
}

void draw_sprite_pool( SpritePool sp, Pixmap pixmap, Pixmap background,
		       int width, int height ) {
  int i;
  Sprite s;
  /* redraw background */
  XCopyArea( display, background, pixmap, sp->normal_gc, 
	     0, 0, width, height, 0, 0 );
  /* flag all sprites as new */
  for ( i = 0; i < sp->pool->size; i++ ) {
    s = sp->pool->element[i];
    s->new = 1;
  }
  redraw_sprite_pool( sp, pixmap, background );
}
