/*
  XBubble - init.c

  Copyright (C) 2002  Ivan Djelic <ivan@savannah.gnu.org>
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <stdlib.h>
#include <limits.h>
#include <string.h>
#include <ctype.h>
#include <math.h>
#include <X11/Xlib.h>

#include "rgba.h"
#include "data.h"
#include "dialog.h"
#include "frame.h"
#include "bubble.h"
#include "board.h"

extern Display *display;
extern Visual *visual;
extern Window root, win;
extern int screen, depth;
extern Colormap colormap;

extern char * data_dir;
extern int scale, win_width, win_height;

XFontStruct *title_font, *dialog_font, *menu_font;
GC dialog_gc, dialog_text_gc, menu_gc, selected_gc;
Pixmap win_bg, board_bg;
Pixmap cup_on, cup_off, cup_on_mask, cup_off_mask;
int board_bg_offset_x, board_bg_offset_y;

Set bubble_animation[NB_COLORS][NB_BUBBLE_STATES];
Set countdown_animation, frame_light_animation, canon_animation;

char* game_fonts[] = {
  /* first try a nice true type font */
  "-*-curlz mt-*-*-*-*-%d-*-*-*-*-*-iso8859-1",
  /* this one is pretty standard - it should be available */
  "-bitstream-charter-*-*-*-*-%d-*-*-*-*-*-*-*",
  /* we supply this one ( from the freefont package ) */
  "-freefont-baskerville-*-i-*-*-%d-*-*-*-*-*-iso8859-1",
  NULL
};

#define SUPPLIED_FONT 2

int progress = 0;
char* bubble_name[NB_COLORS] = BUBBLE_IMAGES;
char* bubble_explosion[NB_COLORS] = BUBBLE_EXPLOSIONS;

char * data_file( char * file ) {
  int path_max;
  char * abs_data_dir;
  static int offset = 0;
  static char buffer[1024];
  if ( ! offset ) {
#ifdef PATH_MAX
    path_max = PATH_MAX;
#else
    path_max = pathconf (path, _PC_PATH_MAX);
    if (path_max <= 0)
      path_max = 4096;
#endif
    abs_data_dir = (char *) xmalloc( path_max * sizeof(char));
    realpath( data_dir, abs_data_dir );
    offset = strlen(abs_data_dir);
    if ( offset > 1024-128 )
      offset = 1024-128;
    strncpy( buffer, abs_data_dir, offset );
    buffer[offset++] = '/';
    buffer[1023] = '\0';
    free( abs_data_dir );
  }
  strncpy( buffer + offset, file, 126);
  return buffer;
}

void init_display( char * display_name ) {
  int i;
  XColor colors[256];
  display = XOpenDisplay( display_name );
  if ( display == NULL ) {
    fprintf( stderr,"xbubble : can't open display.\n");
    exit(1);
  }
  screen = DefaultScreen( display);
  visual = DefaultVisual( display, screen);
  root = RootWindow( display, screen);
  depth = DefaultDepth( display, screen);
  if ( depth < 8 ) {
    fprintf( stderr, "Sorry, your display color depth is not supported.\n");
    XCloseDisplay(display);
    exit(0);
  }
  if ( depth == 8 ) { /* allocate our own RRRGGGBB palette */
    for ( i = 0; i < 256; i++ ) {
      colors[i].pixel = i;
      colors[i].red =   (( i & 0xe0 ) << 8 );
      colors[i].green = (( i & 0x1c ) << 11 );
      colors[i].blue =  (( i & 0x03 ) << 14 );
      colors[i].flags = DoRed | DoGreen | DoBlue;
    }
    colormap = XCreateColormap( display, root, visual, AllocAll );
    XStoreColors( display, colormap, colors, 256 );
  }
}

void add_font_path( char * path ) {
  int i, npaths;
  char ** fontpath, ** new_fontpath;
  fontpath = XGetFontPath( display, &npaths );    
  if ( fontpath != NULL ) {
    /* check if path is already present */
    for ( i = 0; i < npaths; i++ )
      if ( strncmp( path, fontpath[i], strlen(path) ) == 0 ) {
	XFreeFontPath(fontpath);
	return;
      }
    new_fontpath = (char **) xmalloc( sizeof(char *)*( npaths + 1 ));
    memcpy( new_fontpath, fontpath, npaths*sizeof(char *));
    new_fontpath[npaths] = path;
    XSetFontPath( display, new_fontpath, npaths+1 );
    XFreeFontPath(fontpath);
    free(new_fontpath);
  }
}

XFontStruct * load_font( int pixel_size ) {
  XFontStruct * xfont;
  char name[128];
  int i;
  static int added_font_path = 0;
  /* try to load each font in turn */
  for ( i = 0; game_fonts[i] != NULL; i++ ) {
    sprintf( name, game_fonts[i], pixel_size );
    xfont = XLoadQueryFont( display, name );
    if ( xfont != NULL )
      return xfont;
    /* let's try again if we can supply this font */
    if (( i == SUPPLIED_FONT )&&( ! added_font_path )) {
      add_font_path(data_file(""));
      added_font_path = 1;
      i--;
    }
  }
  /* at least we have the "fixed" font ... uurgh */
  if (( xfont = XLoadQueryFont( display, "fixed" )) != NULL )
    return xfont;
  fprintf( stderr, "xbubble : couldn't load any font.\n" );
  exit(1);
}

Frame load_frame( char *file, double zoom ) {
  Pixmap pixmap, mask, *pmask;
  RgbaImage ri, ri2;
  Frame frame = (Frame) xmalloc( sizeof( struct _Frame ));
  ri = create_rgba_image_from_png_file(file);
  if ( ri == NULL )
    exit(1);
  ri2 = scale_down_rgba_image( ri, zoom, zoom );
  if ( ri2 == NULL ) {
    fprintf( stderr, "Error scaling %s. Aborting...\n", file );
    exit(1);
  }
  pmask = ( ri2->has_alpha )? &mask : NULL;
  if ( ! create_pixmaps_from_rgba_image( ri2, &pixmap, pmask )) {
    fprintf( stderr, "Couldn't create pixmaps for image %s. Aborting...\n", 
	     file );
    exit(1);
  }
  frame->width = ri2->width;
  frame->height = ri2->height;
  /* center sprite */
  frame->cx = ri2->width/2;
  frame->cy = ri2->height/2;
  frame->delay = 40; /* default frame duration = 40 ms */
  frame->has_mask = ri2->has_alpha;
  frame->pixmap = pixmap;
  frame->mask = mask;
  delete_rgba_image(ri);
  delete_rgba_image(ri2);
  return frame;
}

void delete_frame( Frame frame ) {
  XFreePixmap( display, frame->pixmap);
  if ( frame->has_mask )
    XFreePixmap( display, frame->mask);
  free(frame);
}

Set load_animation( char *filename, double zoom ) {
  Frame frame;
  Set animation = NULL;
  int cx, cy, delay, max_nb_frames = 0, lc = 0;
  char line[128], framename[128];
  FILE *fd;
  /* open animation description file */
  fd = fopen( data_file(filename), "r" );
  if ( fd == NULL ) {
    fprintf(stderr, "Couldn't open description file %s.\n", 
	    data_file(filename) );
    return NULL;
  }
  /* count frames */
  while ( fgets( line, 128, fd ) != NULL ) {
    if ( isalnum(line[0]))
      max_nb_frames++;
  }
  rewind(fd);
  /* load frames */
  if ( max_nb_frames > 0 ) {
    animation = new_set(max_nb_frames);
    while ( fgets( line, 128, fd ) != NULL ) {
      lc++;
      /* parse line */
      if ( isalnum(line[0])) {
	if ( sscanf( line, "%127s%d%d%d", framename, &cx, &cy, &delay) == 4 ) {
	  frame = load_frame( data_file(framename), zoom );
	  add_element_to_set( animation, frame );
	  frame->cx = cx*zoom;
	  frame->cy = cy*zoom;
	  frame->delay = delay;
	}
	else
	  fprintf(stderr,"Error parsing line %d of file %s.\n", lc, filename);
      }
    }
    if ( animation->size == 0 ) {
      fprintf(stderr,"Animation %s has no valid frames.\n",filename);
      exit(1);
    }
  }
  fclose(fd);
  return animation;
}

void load_scaled_image( char *file, Pixmap *pixmap, 
			Pixmap *mask, double zoom ) {
  RgbaImage ri, ri2;
  if (( ri = create_rgba_image_from_png_file(file)) == NULL )
    exit(1);
  ri2 = ri;
  if (( zoom < 1.0 )&&
      (( ri2 = scale_down_rgba_image( ri, zoom, zoom )) == NULL )) {
    fprintf( stderr, "Error scaling %s. Aborting...\n", file );
    exit(1);
  }
  if ( ! create_pixmaps_from_rgba_image( ri2, pixmap, mask )) {
    fprintf( stderr, "Couldn't create pixmaps for image %s.\n", file );
    exit(1);
  }
  delete_rgba_image(ri);
  if ( ri2 != ri )
    delete_rgba_image(ri2);
}

void delete_animation( Set animation ) {
  int i;
  for ( i = 0; i < animation->size; i++ )
    delete_frame( animation->element[i] );
  delete_set( animation );
}

void load_animations( double zoom ) {
  RgbaImage ri, ri2;
  Pixmap canon_pixmap, canon_mask; 
  Set a;
  Frame dead_frame, countdown_frame, frame;
  int i, color, state;
  char digit[2] = "0";  
  GC gc;
  XGCValues gcv;
  unsigned long gcm;
  frame_light_animation = load_animation( "frame_light.txt", zoom );
  dead_frame = load_frame( data_file("dead.png"), zoom );

  /* bubble animation */
  for ( color = 0; color < NB_COLORS; color++ ) {
    a = new_set(1); 
    frame = load_frame( data_file( bubble_name[color] ), zoom );
    add_element_to_set( a, frame );
    for ( state = 0; state < NB_BUBBLE_STATES; state++ ) {
      bubble_animation[color][state] = a;
      if ( state == EXPLODING )
	bubble_animation[color][state] = 
	  load_animation( bubble_explosion[color], zoom );
      if ( state == DEAD ) {
	bubble_animation[color][state] = new_set(2);
	add_element_to_set( bubble_animation[color][state], a->element[0] );
	add_element_to_set( bubble_animation[color][state], dead_frame );
      }
    }
  }
  /* make canon animation */
  ri2 = create_rgba_image_from_png_file(data_file(CANON_IMAGE));  
  ri2->hotx = CANON_CX;
  ri2->hoty = CANON_CY;
  ri = scale_down_rgba_image( ri2, zoom, zoom );
  delete_rgba_image(ri2);
  canon_animation = new_set( NB_ANGLES );
  for ( i = -CANON_ANGLE_MAX; i <= CANON_ANGLE_MAX; i++ ) {
    ri2 = rotate_rgba_image( ri, -i*ANGLE_STEP, True );
    create_pixmaps_from_rgba_image( ri2, &canon_pixmap, &canon_mask );
    frame = (Frame) xmalloc( sizeof( struct _Frame ));
    frame->width = ri2->width;
    frame->height = ri2->height;
    frame->cx = ri2->hotx;
    frame->cy = ri2->hoty;
    frame->delay = 0;
    frame->has_mask = ri2->has_alpha;
    frame->pixmap = canon_pixmap;
    frame->mask = canon_mask;
    delete_rgba_image(ri2);
    add_element_to_set( canon_animation, frame );
  }
  delete_rgba_image(ri);

  /* make countdown animation */
  gcm = GCFunction | GCGraphicsExposures;
  gcv.graphics_exposures = False;
  gcv.function = GXcopy;
  gc = XCreateGC( display, root, gcm, &gcv );
  XSetForeground( display, gc, get_pixel( 0x30, 0x14, 0x40 ));
  XSetFont( display, gc, menu_font->fid );

  countdown_frame = load_frame( data_file(COUNTDOWN_IMAGE), zoom );
  countdown_animation = new_set(6);
  for ( i = 0; i < 6; i++ ) {
    frame = (Frame) xmalloc( sizeof( struct _Frame ));
    frame->width = countdown_frame->width;
    frame->height = countdown_frame->height;
    frame->cx = countdown_frame->cx;
    frame->cy = countdown_frame->cy;
    frame->delay = countdown_frame->delay;
    frame->has_mask = countdown_frame->has_mask;
    frame->pixmap = XCreatePixmap( display, root, frame->width, frame->height,
				   depth );
    frame->mask = countdown_frame->mask;
    digit[0] = '0' + i;
    XCopyArea( display, countdown_frame->pixmap, frame->pixmap, gc, 0, 0,
	       frame->width, frame->height, 0, 0 );
    XDrawString( display, frame->pixmap, gc, 
		 ( frame->width - XTextWidth( menu_font, digit, 1 ))/2,
		 ( frame->height + menu_font->ascent - menu_font->descent )/2,
		 digit, 1 );
    add_element_to_set( countdown_animation, frame );
  }
  XFreeGC( display, gc );
  countdown_frame->has_mask = 0;
  delete_frame( countdown_frame );
}

void load_images( double zoom ) {
  GC bg_gc;
  XGCValues gcv;
  unsigned int x, y, width, height, d;
  unsigned long gcm;
  Window w;
  Frame frame;
  Pixmap tile, side, mask_side, tee, mask_tee, bottom, mask_bottom, bar;
  load_scaled_image( data_file(BOARD_BG_TILE), &tile, NULL, zoom );
  load_scaled_image( data_file(BOARD_SIDE), &side, &mask_side, zoom );
  load_scaled_image( data_file(BOTTOM_BAR), &bar, NULL, zoom );
  load_scaled_image( data_file(TEE), &tee, &mask_tee, zoom );
  load_scaled_image( data_file(BOTTOM), &bottom, &mask_bottom, zoom );
  load_scaled_image( data_file(CUP_ON), &cup_on,&cup_on_mask,zoom);
  load_scaled_image( data_file(CUP_OFF), &cup_off, &cup_off_mask,
		     zoom);
  gcm = GCFunction | GCGraphicsExposures;
  gcv.graphics_exposures = False;
  gcv.function = GXcopy;
  bg_gc = XCreateGC( display, root, gcm, &gcv);
  XSetFillStyle( display, bg_gc, FillTiled );
  XSetTile( display, bg_gc, tile );
  width = board_win_width(scale);
  height = board_win_height(scale);
  /* make board background */
  board_bg = XCreatePixmap( display, root, width, height, depth );
  /* tile background */
  XFillRectangle( display, board_bg, bg_gc, 0, 0, width, height);
  /* add left side */  
  XSetClipMask( display, bg_gc, mask_side );
  XSetClipOrigin( display, bg_gc, 0, 0 );
  XCopyArea( display, side, board_bg, bg_gc, 0, 0, scale_x( 0, scale), 
	     height, 0, 0 );
  /* add right side */
  XSetClipOrigin( display, bg_gc, width - scale_x( 0, scale), 0 );
  XCopyArea( display, side, board_bg, bg_gc, 0, 0, scale_x( 0, scale), 
	     height, width - scale_x( 0, scale), 0 );
  /* add bottom bar */
  x = scale_x( 0, scale );
  y = scale_y( (ROWS-1)*ROW_HEIGHT, scale );
  width = scale_x( BOARD_WIDTH, scale ) - x;
  height = 20*scale/MAX_SCALE;
  XSetTile( display, bg_gc, bar );
  XSetClipMask( display, bg_gc, None );
  XSetClipOrigin( display, bg_gc, 0, 0 );
  XFillRectangle( display, board_bg, bg_gc, x, y, width, height );
  /* add bottom */ 
  XGetGeometry( display, bottom, &w, &x, &y, &width, &height, &d, &d );
  XSetTile( display, bg_gc, bottom );
  XSetClipMask( display, bg_gc, mask_bottom );
  x = ( board_win_width(scale) - width )/2;
  y = board_win_height(scale) - height;
  XSetClipOrigin( display, bg_gc, x, y );
  XCopyArea( display, bottom, board_bg, bg_gc, 0, 0, width, height, x, y );
  /* add tee */ 
  XGetGeometry( display, tee, &w, &x, &y, &width, &height, &d, &d );
  XSetTile( display, bg_gc, tee );
  XSetClipMask( display, bg_gc, mask_tee );
  x = scale_x( 1.0, scale ) - width/2;
  y = board_win_height(scale) - height;
  XSetClipOrigin( display, bg_gc, x, y );
  XCopyArea( display, tee, board_bg, bg_gc, 0, 0, width, height, x, y );
  /* add top */
  frame = frame_light_animation->element[0];
  XSetClipMask( display, bg_gc, None );
  XGetGeometry( display, frame->pixmap, &w, &x, &y, &width, &height, &d, &d );
  x = ( board_win_width(scale) - width )/2;
  XCopyArea( display, frame->pixmap, board_bg, bg_gc, 0, 0, 
	     frame->width, frame->height, x, 0 );
  /* cleanup */
  XFreeGC( display, bg_gc);
  XFreePixmap( display, bar );
  XFreePixmap( display, tile );
  XFreePixmap( display, side );
  XFreePixmap( display, mask_side );
  XFreePixmap( display, bottom );
  XFreePixmap( display, mask_bottom );
  XFreePixmap( display, tee );
  XFreePixmap( display, mask_tee );
}

void init_gcs() {
  unsigned long gcm;
  XGCValues gcv;
  /* load dialog font */
  dialog_font = load_font( 3*scale/2 );
  gcm = GCFunction | GCGraphicsExposures;
  gcv.graphics_exposures = False;
  gcv.function = GXcopy;
  /* yellow for hilighting menu selection */
  selected_gc = XCreateGC( display, root, gcm, &gcv);
  XSetFont( display, selected_gc, menu_font->fid );
  XSetForeground( display, selected_gc, get_pixel( 0xff, 0xff, 0 ));
  /* grey for menu items */
  menu_gc = XCreateGC( display, root, gcm, &gcv );
  XSetFont( display, menu_gc, menu_font->fid );
  XSetForeground( display, menu_gc, get_pixel( 0xc0, 0xc0, 0xc0 ));
}

void cleanup_graphics() {
  int i, color;
  delete_animation( frame_light_animation );
  delete_animation( canon_animation );
  for ( i = 1; i < countdown_animation->size; i++ )
    ((Frame) countdown_animation->element[i])->has_mask = 0;
  delete_animation( countdown_animation );
  delete_frame( bubble_animation[0][DEAD]->element[1] );
  for ( color = 0; color < NB_COLORS; color++ ) {
    delete_animation( bubble_animation[color][STUCK] );
    delete_animation( bubble_animation[color][EXPLODING] );
    delete_set( bubble_animation[color][DEAD] );
  }
  XFreeGC( display, dialog_gc );
  XFreeGC( display, dialog_text_gc );
  XFreeGC( display, menu_gc );
  XFreeGC( display, selected_gc );
  XFreePixmap( display, win_bg );
  XFreePixmap( display, board_bg );
  XFreePixmap( display, cup_on );
  XFreePixmap( display, cup_off );
  XFreePixmap( display, cup_on_mask );
  XFreePixmap( display, cup_off_mask );
  XFreeFont( display, title_font );
  XFreeFont( display, dialog_font );
  XFreeFont( display, menu_font );
}

void splash_screen( double zoom ) {
  Window splash1, splash2;
  XGCValues gcv;
  int height1, height2;
  unsigned long gcm;
  char * title = "XBubble";
  char * subtitle = "Loading graphics ...";
  load_scaled_image( data_file(WIN_BG), &win_bg, NULL, 1.0 );
  XSetWindowBackgroundPixmap( display, win, win_bg );
  XMapWindow( display, win );
  /* load title & menu fonts */
  title_font = load_font( 3*scale );
  menu_font = load_font( scale );
  height1 = title_font->ascent + title_font->descent;
  height2 = menu_font->ascent + menu_font->descent;
  gcm = GCFunction | GCGraphicsExposures;
  gcv.graphics_exposures = False;
  gcv.function = GXcopy;
  /* make GCs */
  dialog_gc = XCreateGC( display, root, gcm, &gcv );
  dialog_text_gc = XCreateGC( display, root, gcm, &gcv );
  /* GC for tiling dialogs background */
  XSetFillStyle( display, dialog_gc, FillTiled );
  XSetTile( display, dialog_gc, win_bg );
  splash1 = create_dialog( win_width/2, ( win_height - height2 )/2, title, 
			   title_font, get_pixel( 0xff, 0xe4, 0xff ), 0, 1.0 );
  splash2 = create_dialog( win_width/2, ( win_height + height1 )/2, subtitle, 
			   menu_font, get_pixel( 0xff, 0x40, 0xc4 ), 0, 1.0 );
  load_animations(zoom);
  load_images(zoom);
  init_gcs();
  XDestroyWindow( display, splash1 );
  XDestroyWindow( display, splash2 ); 
}
