/*
  XBubble - cell.c
 
  Copyright (C) 2002  Ivan Djelic <ivan@savannah.gnu.org>
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
*/

#include <string.h>
#include <math.h>

#include "utils.h"
#include "data.h"
#include "bubble.h"
#include "cell.h"

/*
 * The playing board is paved with hexagonal cells ( drawing a Voronoi
 * diagram of bubble centers ). Bubbles are periodically lowered by one
 * row, thus the cells pavement depends on "parity" : 
 *
 * if ( parity == 1 )             if ( parity == -1 )
 * ____________________           ____________________
 * |/ \ / \ / \ / \ / \|          |\ / \ / \ / \ / \ /|
 * | 0 | 1 | 2 | 3 | 4 |  row 0   | | 1 | 2 | 3 | 4 | |
 * |\ / \ / \ / \ / \ /|          |/ \ / \ / \ / \ / \|
 * | | 6 | 7 | 8 | 9 | |  row 1   | 5 | 6 | 7 | 8 | 9 |
 * |/ \ / \ / \ / \ / \|          |\ / \ / \ / \ / \ /|
 * |10 |   |   |   |   |  row 2   | |11 |   |   |   | |
 *       .........                     .........
 *                         ...
 * "board->array->cells" stores the content of each cell ( either a stuck 
 * bubble or EMPTY_CELL ), it is indexed as shown above.
 */

CellArray new_cell_array( int moving_ceiling ) {
  int i;
  CellArray ca;
  ca = (CellArray) xmalloc( sizeof( struct _CellArray ));
  memset( ca->tagged, 0, sizeof(int)*NB_CELLS );
  ca->first_row = 0;
  ca->parity = 1;
  ca->moving_ceiling = moving_ceiling;
  for ( i = 0; i < NB_CELLS; i++ )
    ca->cell[i] = EMPTY_CELL;
  return ca;
}

void delete_cell_array( CellArray ca ) {
  free(ca);
}

void empty_cell_array( CellArray ca ) {
  int i;
  ca->first_row = 0;
  for ( i = 0; i < NB_CELLS; i++ )
    ca->cell[i] = EMPTY_CELL;
}

int cellCL( CellArray ca, int row, int column ) {
  if (( column < row_start( ca, row) )||( column > COLS-1 )||
      ( row < ca->first_row )||( row >= ROWS ))
    return OUT_OF_BOUNDS;
  return COLS*row + column;
}

static int cellXY( CellArray ca, double x, double y ) {
  double rx, ry;
  int column, row, rest, a, b, c, e;
  e = ca->parity;
  rx = (x-(e+1.0)/4);
  ry = 2*ROW_HEIGHT*(y-0.5) + 1;
  a = (int) floor(rx+ry);
  b = (int) floor(ry-rx);
  c = (int) floor(2*x + (1-e)/2);
  row = (a+b)/3;
  rest = e*(row%2);
  column = (c+rest)>>1;
  return cellCL( ca, row, column );
}

int neighbor_cell( CellArray ca, int cell, Quadrant quadrant ) {
  int column, row, shift;
  row = cell / COLS;
  column = cell % COLS;
  shift = 1 - row_start( ca, row);

  switch ( quadrant ) {
  case EAST: column++; break;
  case WEST: column--; break;
  case NORTH_EAST: column += shift; row--; break;
  case SOUTH_EAST: column += shift; row++; break;
  case NORTH_WEST: column += shift-1; row--; break;
  case SOUTH_WEST: column += shift-1; row++; break;
  }
  return cellCL( ca, row, column );
}

void cell_center( CellArray ca, int cell, double *x, double *y) {
  int column, row;
  row = cell / COLS;
  column = cell % COLS;
  *x = column + 0.5*(1 - row_start(ca, row));
  *y = row*ROW_HEIGHT + 0.5;
}

int floating_cell( CellArray ca, int cell ) {
  Quadrant q;
  for ( q = 0; q <= 5; q++ )
    if ( is_not_empty( ca, neighbor_cell( ca, cell, q )))
      return 0;
  return 1;
}

int find_random_empty_cell( CellArray ca ) {
  int col, column, r, c;
  /* pick a random column */
  column = rnd( COLS );
  /* search for target cell */
  for ( r = ca->first_row; r < ROWS; r++ ) {
    col = column + row_start( ca, r);
    if ( col >= COLS )
      col = COLS - 1;
    c = col + r*COLS;
    if ( ca->cell[c] == EMPTY_CELL )
      return c;
  }
  return OUT_OF_BOUNDS;
}

int cell_array_overflow( CellArray ca ) {
  int i;
  for ( i = NB_CELLS - COLS; i < NB_CELLS; i++ )
    if ( ca->cell[i] != EMPTY_CELL )
      return 1;
  return 0;
}

int cell_array_empty( CellArray ca ) {
  int i, first, last;
  first = first_cell(ca);
  last = first + COLS - 1;
  for ( i = first; i <= last; i++ )
    if ( ca->cell[i] != EMPTY_CELL )
      return 0;
  return 1;
}

void hook_cell( CellArray ca, int i) {
  Quadrant q;
  if (( is_not_empty( ca, i))&&( ! ca->tagged[i] )) {
    ca->tagged[i] = 1;
    /* hook neighbors */
    for ( q = 0; q <= 5; q++ )
      hook_cell( ca, neighbor_cell( ca, i, q ));
  }
}

int count_floating_cells( CellArray ca, Vector floating ) {
  int i;
  empty_vector( floating );
  /* hook first row and propagate */
  for ( i = 0; i < COLS; i++ )
    hook_cell( ca, i + first_cell(ca));
  /* count floating bubbles */
  for ( i = first_cell(ca); i < NB_CELLS; i++ )
    if ( ca->cell[i] != EMPTY_CELL ) {
      if ( ! ca->tagged[i] )
	add_element_to_vector( floating, i);
      ca->tagged[i] = 0;
    }
  return floating->size;
}

void lower_cell_array( CellArray ca ) {
  int i;
  if ( ca->first_row < ROWS - 1 ) {
    ca->parity *= -1;
    for ( i = NB_CELLS-1; i >= first_cell(ca) + COLS; i-- )
      ca->cell[i] = ca->cell[ i - COLS ];
    for ( i = first_cell(ca); i < first_cell(ca) + COLS; i++ )
      ca->cell[i] = EMPTY_CELL;
    if ( ca->moving_ceiling )
      ca->first_row++;
  }
}

int target_cell( CellArray ca, double x, double y, double vx, double vy,
		 double * target_y, Vector path ) {
  int row, col, first_col, min_col, max_col, min_row, max_row, cell;
  int bouncing = 1;
  double v2, dp, vp, vp2, xt, yt, xc, xa, ya, half_range, dist, min_dist;
  v2 = vx*vx + vy*vy;
  
  /* sanity check */
  if ( vy >= 0 )
    return OUT_OF_BOUNDS;

  while ( bouncing ) {
    /* intersect trajectory with ceiling */
    xc = x + ( ceiling_y(ca) - y )*vx/vy;
    if ( xc < 0.5 ) { /* bubble hits left wall */
      xt = 0.5;
      yt = y + ( xt - x )*vy/vx;
    }
    else {
      if ( xc > COLS - 0.5001 ) { /* bubble hits right wall */
	xt = COLS - 0.5001; /* always stay inside board cells */
	yt = y + ( xt - x )*vy/vx; 
      }
      else { /* bubble hits ceiling */
      bouncing = 0;
      xt = xc;
      yt = ceiling_y(ca);
      }
    }
    min_dist = ( yt - y )/vy;
    /* now scan cells and look for a colliding bubble */

    /* row range */
    max_row = clip( (int) floor( y / ROW_HEIGHT ) + 1, 0, ROWS-1 );
    min_row = clip( (int) floor( yt / ROW_HEIGHT ) - 1, 0, ROWS-1 );

    /* scan from bottom to top */
    for ( row = max_row; row >= min_row; row-- ) {
      cell = COLS*row;
      /* column range */
      half_range = sqrt( 1 + vx*vx/vy/vy );
      first_col = row_start( ca, row);
      xc = x + ( row*ROW_HEIGHT + 0.5 - y )*vx/vy - ( 1.0 - first_col )/2;
      min_col = clip( (int) floor( xc - half_range + 1 ), first_col, COLS-1 );
      max_col = clip( (int) floor( xc + half_range ), first_col, COLS-1 );

      for ( col = min_col; col <= max_col; col++ ) {
	if ( ca->cell[ col + cell ] != EMPTY_CELL ) {
	  /* compute collision location */
	  cell_center( ca, col + cell, &xa, &ya );
	  dp = vx*( xa - x ) + vy*( ya - y );
	  if ( dp > 0 ) {
	    vp = vy*( xa - x ) - vx*( ya - y );
	    vp2 = vp*vp;
	    if ( vp2 <= v2 ) {
	      dist = ( dp - sqrt( v2 - vp2 ))/v2;
	      if (( dist >= 0 )&&( dist <= min_dist )) {
		min_dist = dist;
		/* store location */
		xt = x + vx*dist;
		yt = y + vy*dist;
		bouncing = 0;
		/* check next row and finish */
		if ( min_row < row - 1 )
		  min_row = row - 1;
	      }
	    }
	  }
	}
	else
	  if ( path != NULL )
	    add_element_to_vector( path, col + cell );
      }
    }
    /* prepare for next part of trajectory */
    vx *= -1;
    x = xt;
    y = yt;
  }
  *target_y = yt;
  return cellXY( ca, xt, yt );
}

void set_explosive_bubble( CellArray ca, int cell, int depth, Set explosive,
			   Vector explosive_depth ) {
  Quadrant q;
  int neighbor;
  Bubble bubble, bubble2;
  bubble = ca->cell[cell];
  add_element_to_set( explosive, bubble);
  /* tag bubble to avoid visiting it twice and store recursion depth */
  ca->tagged[cell] = 1;
  if ( explosive_depth != NULL )
    add_element_to_vector( explosive_depth, depth );
  /* propagate explosion to neighbors */
  for ( q = EAST; q <= SOUTH_EAST; q++ ) {
    neighbor = neighbor_cell( ca, cell, q );
    if (( neighbor != OUT_OF_BOUNDS )&&( ! ca->tagged[neighbor] )) {
      bubble2 = ca->cell[neighbor];
      if (( bubble2 != EMPTY_CELL )&&( bubble2->color == bubble->color ))
	set_explosive_bubble( ca, neighbor, depth+1, 
			      explosive, explosive_depth );
    }
  }
}

int count_explosive_bubbles( CellArray ca, int cell, Set explosive, 
			     Vector explosive_depth ) {
  int i;
  Bubble bubble;
  empty_set(explosive);
  if ( explosive_depth != NULL )
    empty_vector(explosive_depth);
  set_explosive_bubble( ca, cell, 1, explosive, explosive_depth );
  for ( i = 0; i < explosive->size; i++ ) {
    bubble = explosive->element[i];
    ca->tagged[bubble->cell] = 0;
  }
  return explosive->size;
}
