/*
  XBubble - board.c
 
  Copyright (C) 2002  Ivan Djelic <ivan@savannah.gnu.org>
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
*/

#include <stdlib.h>
#include <math.h>

#include "utils.h"
#include "cell.h"
#include "data.h"
#include "frame.h"
#include "sprite.h"
#include "bubble.h"
#include "board.h"

extern int scale;
extern Set countdown_animation, frame_light_animation, canon_animation;

/* sprite locations */
#define FRAME_X          ( BOARD_WIDTH*0.5 )
#define FRAME_Y          ( -OFFSET_Y*0.5 )
#define NEW_X            1.0
#define NEW_Y            ( BOARD_HEIGHT - 0.75 )
#define COUNTDOWN_X      1.3
#define COUNTDOWN_Y      ( NEW_Y - 1.3 )
#define CANON_X          ( BOARD_WIDTH*0.5 )
#define CANON_Y          ( ROWS*ROW_HEIGHT + 0.5 )
 
/* speeds are expressed in cells/ms */
#define LAUNCHING_SPEED  0.02     /* velocity of launched bubbles */
#define EXPLODING_SPEED  0.005    /* vertical velocity after explosion */
#define RISING_SPEED     0.04     /* velocity of incoming bubbles */
#define GRAVITY          0.00003  /* gravity in cell.ms^-2 */

/* times expressed in milliseconds */
#define PROPAGATION_TIME 40       /* explosion propagation time */
#define CANON_LOAD_TIME  500
#define COUNTDOWN        5000 
#define SLOW_BLINKING    250      /* blinking light period */
#define FAST_BLINKING    125

enum CountdownState { INACTIVE, ACTIVE, VISIBLE };

struct _Board {
  enum CountdownState countdown_state;
  int period, launch_count, locked;
  int ready_to_fire, launch_requested, needs_hook_update, dead;
  int empty, quiet, clock, was_lowered, color, next_color;
  int last_fire_delay, max_fire_delay, count;
  int canon_angle, nb_evaluations, frame_on, canon_direction;
  double canon_virtual_angle, vx[NB_ANGLES], vy[NB_ANGLES];
  CellArray array;
  Set bubbles, explosive_bubbles;
  SpritePool sprite_pool;
  Vector input, output, floating_cells, explosive_depth;
  Sprite canon_sprite, countdown_sprite, frame_light_sprite;
};

enum { BOTTOM_LAYER = 0, CANON_LAYER, MEDIUM_LAYER, TOP_LAYER };

static Bubble new_board_bubble( Board board, int color ) {
  Bubble bubble = new_bubble( color, NEW_X, NEW_Y, MEDIUM_LAYER);
  add_element_to_set( board->bubbles, bubble );
  add_sprite_to_pool( board->sprite_pool, bubble->sprite );
  return bubble;
}

Board new_board( int period, int handicap, int moving_ceiling ) {
  int i;
  Board board = (Board) xmalloc( sizeof( struct _Board));
  board->array = new_cell_array( moving_ceiling );
  board->bubbles = new_set( 2*NB_CELLS );
  board->sprite_pool = new_sprite_pool( TOP_LAYER+1, 2*NB_CELLS );
  board->input = new_vector( NB_CELLS );
  board->output = new_vector( NB_CELLS );
  board->explosive_bubbles = new_set( NB_CELLS );
  board->explosive_depth = new_vector( NB_CELLS );
  board->floating_cells = new_vector( NB_CELLS );
  board->max_fire_delay = handicap;
  board->period = period;

  /* frame blinking light */
  board->frame_light_sprite = new_sprite( BOTTOM_LAYER, 30, 
					  frame_light_animation, 0, 0);
  set_sprite_position( board->frame_light_sprite, scale_x( FRAME_X, scale ), 
		       scale_y( FRAME_Y, scale ));
  add_sprite_to_pool( board->sprite_pool, board->frame_light_sprite );
  board->frame_on = 0;

  /* countdown */
  board->countdown_sprite = new_sprite( BOTTOM_LAYER, 30, 
					countdown_animation, 0, 0);
  set_sprite_position( board->countdown_sprite, scale_x( COUNTDOWN_X, scale ),
		       scale_y( COUNTDOWN_Y, scale ));
  board->countdown_state = INACTIVE;

  /* canon */
  board->canon_angle = 0;
  board->canon_virtual_angle = 0.0;
  board->canon_direction = 0;
  board->canon_sprite = new_sprite( CANON_LAYER, 30, canon_animation, 0, 0);
  add_sprite_to_pool( board->sprite_pool, board->canon_sprite );
  move_board_canon( board, 0);

  /* precompute launching vectors */
  for ( i = 0; i < NB_ANGLES; i++ ) {
    board->vx[i] = LAUNCHING_SPEED*sin((i-CANON_ANGLE_MAX)*ANGLE_STEP );
    board->vy[i] =-LAUNCHING_SPEED*cos((i-CANON_ANGLE_MAX)*ANGLE_STEP );
  }
  board->launch_count = 0;
  board->last_fire_delay = 1000;
  board->locked = 0;
  board->needs_hook_update = 0;
  board->ready_to_fire = 0;
  board->launch_requested = 0;
  board->empty = 0;
  board->dead = 0;
  board->quiet = 0;
  board->clock = 0;
  /* start with a new bubble */
  board->color = rnd( NB_COLORS );
  new_board_bubble( board, board->color );
  return board;
}

void delete_board(Board board) {
  int i;
  delete_cell_array( board->array );
  for ( i = 0; i < board->bubbles->size; i++ )
    delete_bubble( board->bubbles->element[i] );
  delete_set( board->bubbles );
  delete_set( board->explosive_bubbles );
  delete_vector( board->input );
  delete_vector( board->output );
  delete_vector( board->explosive_depth );
  delete_vector( board->floating_cells );
  delete_sprite_pool( board->sprite_pool );
  delete_sprite( board->canon_sprite );
  delete_sprite( board->countdown_sprite );
  delete_sprite( board->frame_light_sprite );
  free( board );
}

static void lock_board(Board board) {
  board->locked++;
}

static void unlock_board(Board board) {
  board->locked--;
}

int board_overflow(Board board) {
  return cell_array_overflow(board->array);
}

static void switch_frame_on( Board board ) {
  set_sprite_frame( board->frame_light_sprite, 1);
  board->frame_on = 1;
}

static void switch_frame_off( Board board ) {
  set_sprite_frame( board->frame_light_sprite, 0);
  board->frame_on = 0;
}

static void start_countdown( Board board ) {
  board->count = board->max_fire_delay;
  board->countdown_state = ACTIVE;
}

static void cancel_countdown( Board board ) {
  switch ( board->countdown_state ) {
  case VISIBLE:
    remove_sprite_from_pool( board->sprite_pool, board->countdown_sprite);
  case ACTIVE:
    board->countdown_state = INACTIVE;
    board->last_fire_delay = board->max_fire_delay - board->count;
    break;
  default:
    break;
  }
}
  
static void update_countdown( Board board, int dt ) {
  board->count -= dt;
  switch ( board->countdown_state ) {
  case ACTIVE:
    if ( board->count <= 0 )
      /* launch bubble automatically */
      board->launch_requested = 1;
    else
      if ( board->count <= COUNTDOWN ) {
	add_sprite_to_pool( board->sprite_pool, board->countdown_sprite );
	set_sprite_frame( board->countdown_sprite, 1+board->count/1000 ); 
	board->countdown_state = VISIBLE;
      }
    break;
  case VISIBLE:
    if ( board->count > 0 )
      set_sprite_frame( board->countdown_sprite, 1+board->count/1000 );  
    else {
      remove_sprite_from_pool( board->sprite_pool, board->countdown_sprite);
      board->countdown_state = ACTIVE;
    }
    break;
  default:
    break;
  }
}

int get_last_fire_delay( Board board ) {
  return board->last_fire_delay;
}

void explode_bubbles( Board board ) {
  Bubble bubble;
  int i, clock;
  for ( i = 0; i < board->explosive_bubbles->size; i++ ) {
    bubble = board->explosive_bubbles->element[i];
    /* use recursion depth as initial clock to simulate propagation */ 
    clock = (1 - board->explosive_depth->element[i])*PROPAGATION_TIME;
    set_bubble_state( bubble, EXPLODING, TOP_LAYER, clock);
    bubble->vx = 0;
    bubble->vy = -EXPLODING_SPEED;
    board->array->cell[bubble->cell] = EMPTY_CELL;
    board->needs_hook_update = 1;
    if ( i >= 3 ) /* send some bubbles to opponent */ 
      add_element_to_vector( board->output, bubble->color );
  }
}

void explode_board( Board board ) {
  int i, clock;
  Bubble bubble;
  if ( ! board_overflow(board) ) {
    for( i = 0; i < board->bubbles->size; i++ ) {
      bubble = board->bubbles->element[i];
      if (( bubble->state != RUNAWAY )&&( bubble->state != STUCK )) {
	bubble->vx = 0;
	bubble->vy = 0;
	set_bubble_state( bubble, EXPLODING, TOP_LAYER, 0 );
      }
    }
    clock = 0;
    for ( i = NB_CELLS-1; i >= 0; i-- )
      if ( board->array->cell[i] != EMPTY_CELL ) {
	bubble = board->array->cell[i];
	bubble->vx = 0;
	bubble->vy = 0;
	set_bubble_state( bubble, EXPLODING, TOP_LAYER, clock );
	clock -= PROPAGATION_TIME;
      }
    empty_cell_array( board->array );
  }
  /* prevent from new row of bubbles */
  board->launch_count = 0;
}

void drop_bubble( Board board, Bubble bubble ) {
  set_bubble_state( bubble, FALLING, TOP_LAYER, 0);
  /* add a small random part in falling speed */
  bubble->vx = 0;
  bubble->vy = rnd(100)*EXPLODING_SPEED/100;
}

void stick_bubble( Board board, Bubble bubble, int target_cell, int clock) {
  double x, y;
  cell_center( board->array, target_cell, &x, &y );
  set_bubble_state( bubble, STUCK, BOTTOM_LAYER, clock);
  set_bubble_position( bubble, x, y);
  bubble->cell = target_cell;
  board->array->cell[target_cell] = bubble;
  board->empty = 0;
}

void animate_bubble( Board board, Bubble bubble, int dt ) {
  int i, processed = 0;
  double x, y, t;

  switch ( bubble->state ) {
  case NEW:
    if ( ! board->ready_to_fire ) {
      /* compute new position of bubble */
      t = M_PI * bubble->clock/2/CANON_LOAD_TIME;
      x = CANON_X - ( CANON_X - NEW_X )*cos(t);
      y = NEW_Y + ( CANON_Y - NEW_Y )*sin(t);
      set_bubble_position( bubble, x, y);
      bubble->clock += dt;

      /* if bubble has reached canon then we're ready to fire */
      if ( bubble->clock > CANON_LOAD_TIME ) {
	set_bubble_state( bubble, READY, MEDIUM_LAYER, 0 );
	set_bubble_position( bubble, CANON_X, CANON_Y );
	board->ready_to_fire = 1;
	/* add a new bubble */
	board->next_color = rnd( NB_COLORS );
	new_board_bubble( board, board->next_color );
	start_countdown(board);
      }
    }
    break;

  case READY:
    update_countdown( board, dt );
    if (( ! board->locked )&&( board->launch_requested )) {
      /* launch bubble */
      set_bubble_state( bubble, LAUNCHED, MEDIUM_LAYER, 0);
      bubble->vx = board->vx[board->canon_angle+CANON_ANGLE_MAX];
      bubble->vy = board->vy[board->canon_angle+CANON_ANGLE_MAX];
      bubble->cell = target_cell( board->array, bubble->x, bubble->y,
				  bubble->vx, bubble->vy, 
				  &bubble->target_y, NULL );
      /* lock board until bubble is stuck */
      lock_board(board);
      board->launch_count++;
      board->ready_to_fire = 0;
      board->launch_requested = 0;
      board->color = board->next_color;
      cancel_countdown(board);
    }
    break;

  case LAUNCHED: 
    /* move bubble */
    bubble->x += dt*bubble->vx;
    bubble->y += dt*bubble->vy;
    /* bounce bubble against walls */
    if ( bubble->x < 0.5 ) {
      bubble->x = 1.0 - bubble->x;
      bubble->vx = -bubble->vx;
    }
    if ( bubble->x > COLS - 0.5001 ) {
      bubble->x = 2*COLS - 1.0002 - bubble->x;
      bubble->vx = -bubble->vx;
    }
    /* check if bubble has reached its target cell */
    if ( bubble->y <= bubble->target_y ) {
      stick_bubble( board, bubble, bubble->cell, 0);

      /* check if bubble triggers an explosion */
      if ( count_explosive_bubbles( board->array, bubble->cell, 
				    board->explosive_bubbles, 
				    board->explosive_depth ) >= 3 )
	explode_bubbles(board);
      else
	empty_set( board->explosive_bubbles );
      unlock_board(board);
      board->launch_requested = 0; /* prevent launch until next frame */
    }
    else
      set_bubble_position( bubble, bubble->x, bubble->y);
    break;
    
  case STUCK:
    if ( ! bubble->clock )
      increase_sprite_clock( bubble->sprite, dt );
    break;
    
  case EXPLODING:
    bubble->clock += dt;
    /* if bubble has already been processed, skip it */
    for ( i = 0; i < board->explosive_bubbles->size; i++ )
      if ( board->explosive_bubbles->element[i] == bubble ) {
	processed = 1;
	break;
      }
    if (( bubble->clock <= 0 )||( processed ))
      break;

    if ( bubble->clock > get_bubble_animation_duration(bubble) ) {
      set_bubble_state( bubble, RUNAWAY, TOP_LAYER, 0);
      remove_sprite_from_pool( board->sprite_pool, bubble->sprite );
      break;
    }

  case FALLING:
    bubble->y += dt*bubble->vy + 0.5*GRAVITY*dt*dt;
    bubble->vy += dt*GRAVITY;
    set_bubble_position( bubble, bubble->x, bubble->y);
    increase_sprite_clock( bubble->sprite, dt );
    /* detect runaway bubbles */
    if ( bubble->y > BOARD_HEIGHT - 0.5 ) {
      /* send bubble to opponent */
      if ( bubble->state == FALLING )
	add_element_to_vector( board->output, bubble->color );
      set_bubble_state( bubble, RUNAWAY, TOP_LAYER, 0);
      remove_sprite_from_pool( board->sprite_pool, bubble->sprite );
    }
    break;
    
  case RISING:
    set_bubble_position( bubble, 
			 bubble->x+dt*bubble->vx, bubble->y+dt*bubble->vy );
    increase_sprite_clock( bubble->sprite, dt );
    /* check if bubble has reached its target cell */
    if ( bubble->y <= cell_y( bubble->cell )) {
      stick_bubble( board, bubble, bubble->cell, 1);
      unlock_board(board);
    }
    break;
    
  case DEAD:
    if ( bubble->clock <= 200 ) {
      if ( bubble->clock >= 0 ) 
	increase_sprite_clock( bubble->sprite, dt );
      board->quiet = 0;
      bubble->clock += dt;
    }
    break;
    
  default:
    break;
  }
}

void send_bubble( Board board, int color ) {
  add_element_to_vector( board->input, color );
}

int place_received_bubbles( Board board ) {
  Bubble bubble;
  int color, i, target_cell;
  double x, y;

  if ( ! board->locked ) {
    for ( i = 0; i < board->input->size; i++ ) {
      color = board->input->element[i];
      /* find a random target cell */
      target_cell = find_random_empty_cell(board->array);
      if ( target_cell != OUT_OF_BOUNDS ) {
        /* create rising bubble */
	bubble = new_board_bubble( board, color);
	set_bubble_state( bubble, RISING, TOP_LAYER, 0);
	cell_center( board->array, target_cell, &x, &y);
	set_bubble_position( bubble, x, BOARD_HEIGHT - 0.5);
	bubble->vx = 0;
	bubble->vy = -RISING_SPEED;
	bubble->cell = target_cell;
	/* lock board until all rising bubbles have reached their target */
	lock_board(board);
	/* lock cell to avoid 2 bubbles having the same target */
	board->array->cell[target_cell] = bubble;
      }
      board->input->element[i] = target_cell;
    }
    /* unlock cells */
    for ( i = 0; i < board->input->size; i++ ) {
      target_cell = board->input->element[i];
      if ( target_cell != OUT_OF_BOUNDS )
	board->array->cell[target_cell] = EMPTY_CELL;
    }
    empty_vector(board->input);
    return 1;
  }
  return 0;
}

int receive_bubble(Board board) {
  if ( board->output->size > 0 )
    return board->output->element[--board->output->size];
  return -1;
}

void drop_bubbles(Board board) {
  int i, cell;
  Bubble bubble;	
  count_floating_cells( board->array, board->floating_cells );
  for ( i = 0; i < board->floating_cells->size; i++ ) {
    cell = board->floating_cells->element[i];
    bubble = board->array->cell[cell];
    drop_bubble( board, bubble );
    board->array->cell[cell] = EMPTY_CELL;
  }
  board->empty = cell_array_empty( board->array );
}

int lower_board(Board board) {
  int i, color;
  Bubble bubble;
  if ( board->locked )
    return 0;
  lower_cell_array( board->array );
  for ( i = 0; i < board->bubbles->size; i++ ) {
    bubble = board->bubbles->element[i];
    if ( bubble->state == STUCK ) 
      set_bubble_position( bubble, bubble->x, bubble->y+ROW_HEIGHT );
    /* update cell record */
    bubble->cell += COLS;
  }
  if ( ! board->array->moving_ceiling ) { /* add a new row of bubbles */
    board->array->cell[0] = EMPTY_CELL;
    for ( i = row_start( board->array, 0); i < COLS; i++ ) {
      color = rnd( NB_COLORS );
      bubble = new_board_bubble( board, color);
      /* stick bubble with clock = 1 to avoid animation */
      stick_bubble( board, bubble, i, 1);
    }
  }
  else /* lower frame light */
    set_sprite_position( board->frame_light_sprite, scale_x( FRAME_X, scale ),
			 scale_y( FRAME_Y+board->array->first_row*ROW_HEIGHT,
				  scale ));
  return 1;
}

void kill_board( Board board ) {
  int i, clock;
  Bubble bubble;
  for( i = 0; i < board->bubbles->size; i++ ) {
    bubble = board->bubbles->element[i];
    if (( bubble->state != RUNAWAY )&&( bubble->state != STUCK ))
      set_bubble_state( bubble, DEAD, BOTTOM_LAYER, 0 );
  }
  clock = 0;
  for ( i = NB_CELLS-1; i >= 0; i-- )
    if ( board->array->cell[i] != EMPTY_CELL ) {
      bubble = board->array->cell[i];
      set_bubble_state( bubble, DEAD, BOTTOM_LAYER, clock );
      clock -= PROPAGATION_TIME;
    }
  board->dead = 1;
}

void animate_bubbles( Board board, int dt ) {
  int i;
  Bubble bubble;
  board->quiet = 1;
  /* update state of each bubble */
  for ( i = 0; i < board->bubbles->size; i++ ) {
    bubble = board->bubbles->element[i];
    if ( bubble->state != RUNAWAY ) {
      animate_bubble( board, bubble, dt );
      if (( bubble->state != NEW )&&
	  ( bubble->state != READY )&&
	  ( bubble->state != DEAD ))
	board->quiet = 0;
    }
    else { /* destroy bubble */
      remove_element_from_set_at( board->bubbles, i--);
      delete_sprite( bubble->sprite );
      free( bubble );
    }
  }
  empty_set( board->explosive_bubbles );
}  

void move_board_canon( Board board, int dt ) {
  int frame, x, y;
  Frame cframe;
  board->canon_virtual_angle += board->canon_direction*dt*CANON_ROTATING_SPEED;
  board->canon_virtual_angle = clip( board->canon_virtual_angle,
				     -CANON_ANGLE_MAX, CANON_ANGLE_MAX );
  board->canon_angle = (int) floor( board->canon_virtual_angle + 0.5 );
  frame = (board->canon_angle+CANON_ANGLE_MAX)*canon_animation->size/NB_ANGLES;
  /* compute canon center coordinates */
  cframe = (Frame) canon_animation->element[frame];
  x = scale_x( CANON_X, scale );
  y = scale_y( CANON_Y, scale );
  set_sprite_frame( board->canon_sprite, frame);
  set_sprite_position( board->canon_sprite, x, y);
  board->canon_direction = 0;
}

void animate_board( Board board, int dt ) {
  int blinking;
  board->clock += dt;
  board->was_lowered = 0;
  move_board_canon( board, dt );

  if ( ! board_overflow(board) ) {
    /* lower all bubbles if enough bubbles were launched */
    if (( board->launch_count > board->period )&&( lower_board(board) )) {
      board->launch_count = 0;
      switch_frame_off(board);
      board->was_lowered = 1;
    }
    /* blink frame when launch count is almost equal to period */
    if ( board->launch_count >= board->period - 1 ) {
      blinking = ( board->launch_count >= board->period )?
	FAST_BLINKING : SLOW_BLINKING;
      if (( board->frame_on )&&(( board->clock/blinking ) % 2 ))
	switch_frame_off(board);
      if (( ! board->frame_on )&&(( board->clock/blinking ) % 2 == 0 ))
	switch_frame_on(board);
    }

    /* add external bubbles from opponent */
    if ( ! board_overflow(board) )
      place_received_bubbles(board);

    if ( ! board_overflow(board) ) {
      animate_bubbles( board, dt );
      
      /* drop some bubbles */ 
      if ( board->needs_hook_update ) {
	drop_bubbles(board);
	board->needs_hook_update = 0;
      }
    }
  }
  else { /* game end animation */
    if ( ! board->quiet ) {
      if ( ! board->dead )
	kill_board(board);
      board->quiet = 1; /* reset to 0 in animate_bubble() */
      animate_bubbles( board, dt);
    }
  }
  board->launch_requested = 0;
}

void rotate_canon_left( Board board ) {
  board->canon_direction = -1;
}

void rotate_canon_right( Board board ) {
  board->canon_direction = 1;
}

void canon_stop( Board board ) {
  board->canon_direction = 0;
}

void fire_board_canon(Board board) {
  board->launch_requested = 1;
}

void load_bubbles( Board board, int *colors ) {
  int cell;
  Bubble bubble;
  for ( cell = 0; cell < NB_CELLS; cell++ )
    if (( colors[cell] >= 0 )&&
	( cell % COLS >= row_start( board->array, cell/COLS ))) {
      bubble = new_board_bubble( board, colors[cell]);
      /* stick bubble with clock = 1 to skip animation */
      stick_bubble( board, bubble, cell, 1);
    }
}

SpritePool get_board_sprite_pool( Board board ) {
  return board->sprite_pool;
}

int board_quiet( Board board ) {
  return board->quiet;
}

int board_empty( Board board ) {
  return board->empty;
}

int get_canon_angle( Board board ) {
  return board->canon_angle;
}

int ready_to_fire( Board board ) {
  return (( board->ready_to_fire )&&( ! board->locked ));
}

int board_was_lowered( Board board ) {
  return board->was_lowered;
}

CellArray get_board_array( Board board ) {
  return board->array;
}

void get_board_info( Board board, double *cx, double *cy, 
		     double **vx, double **vy, int *color, int *next_color,
		     int *launch_count, int *period ) {
  *cx = CANON_X;
  *cy = CANON_Y;
  *vx = board->vx;
  *vy = board->vy;
  *color = board->color;
  *next_color = board->next_color;
  *launch_count = board->launch_count;
  *period = board->period;
}
