#!/usr/bin/env python
# -*- coding: utf-8 -*-

#This file is part of SVGUIEditor, a library implementing an editor for SVGUI
#interfaces
#
#Copyright (C) 2007: Edouard TISSERANT, Laurent BESSARD and Jonathan HURTREL
#
#See COPYING file for copyrights details.
#
#This library is free software; you can redistribute it and/or
#modify it under the terms of the GNU General Public
#License as published by the Free Software Foundation; either
#version 2.1 of the License, or (at your option) any later version.
#
#This library is distributed in the hope that it will be useful,
#but WITHOUT ANY WARRANTY; without even the implied warranty of
#MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#General Public License for more details.
#
#You should have received a copy of the GNU General Public
#License along with this library; if not, write to the Free Software
#Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

from xml.dom import minidom
import cPickle
import os, sys, re
from types import *

from SVGUIGenerator import *
from svgui import *

[ITEM_CONTAINER, ITEM_BUTTON, ITEM_TEXT, ITEM_SCROLLBAR, ITEM_ROTATING, 
 ITEM_NOTEBOOK, ITEM_NOTEBOOKCONTENT, ITEM_TRANSFORM] = range(8)

def GetElementType(element):
    if isinstance(element, svgui.controls_Button):
        return ITEM_BUTTON
    elif isinstance(element, svgui.controls_ScrollBar):
        return ITEM_SCROLLBAR
    elif isinstance(element, svgui.controls_TextCtrl):
        return ITEM_TEXT
    elif isinstance(element, svgui.controls_NoteBook):
        return ITEM_NOTEBOOK
    elif isinstance(element, svgui.NoteBook_NoteBookContent):
        return ITEM_NOTEBOOKCONTENT
    elif isinstance(element, svgui.controls_RotatingCtrl):
        return ITEM_ROTATING
    elif isinstance(element, svgui.controls_Transform):
        return ITEM_TRANSFORM
    elif isinstance(element, svgui.Container):
        return ITEM_CONTAINER
    return ITEM_UNEDITABLE

def GetElementName(element):
    if isinstance(element, svgui.controls_Button):
        return "Button"
    elif isinstance(element, svgui.controls_ScrollBar):
        return "ScrollBar"
    elif isinstance(element, svgui.controls_TextCtrl):
        return "TextCtrl"
    elif isinstance(element, svgui.controls_NoteBook):
        return "NoteBook"
    elif isinstance(element, svgui.NoteBook_NoteBookContent):
        return "NoteBookContent"
    elif isinstance(element, svgui.controls_RotatingCtrl):
        return "RotatingCtrl"
    elif isinstance(element, svgui.controls_Transform):
        return "Transform"
    elif isinstance(element, svgui.Container):
        return "Container"
    return None

def GetElementTreeInfos(element):
    element_infos = {"name" : element.getname(), "data" : element.getid(), "children" : []}
    if GetElementType(element) in [ITEM_CONTAINER, ITEM_NOTEBOOKCONTENT]:
        for child in element.getelements():
            element_infos["children"].append(GetElementTreeInfos(child))
    elif GetElementType(element) == ITEM_NOTEBOOK:
        for content in element.getcontents():
            element_infos["children"].append(GetElementTreeInfos(content))
    return element_infos

#-------------------------------------------------------------------------------
#                         Undo Buffer for SVGUIControler
#-------------------------------------------------------------------------------

# Length of the buffer
UNDO_BUFFER_LENGTH = 20

"""
Class implementing a buffer of changes made on the current editing model
"""
class UndoBuffer:

    # Constructor initialising buffer
    def __init__(self, currentstate, issaved = False):
        self.Buffer = []
        self.CurrentIndex = -1
        self.MinIndex = -1
        self.MaxIndex = -1
        # if current state is defined
        if currentstate:
            self.CurrentIndex = 0
            self.MinIndex = 0
            self.MaxIndex = 0
        # Initialising buffer with currentstate at the first place
        for i in xrange(UNDO_BUFFER_LENGTH):
            if i == 0:
                self.Buffer.append(currentstate)
            else:
                self.Buffer.append(None)
        # Initialising index of state saved
        if issaved:
            self.LastSave = 0
        else:
            self.LastSave = -1
    
    # Add a new state in buffer
    def Buffering(self, currentstate):
        self.CurrentIndex = (self.CurrentIndex + 1) % UNDO_BUFFER_LENGTH
        self.Buffer[self.CurrentIndex] = currentstate
        # Actualising buffer limits
        self.MaxIndex = self.CurrentIndex
        if self.MinIndex == self.CurrentIndex:
            # If the removed state was the state saved, there is no state saved in the buffer
            if self.LastSave == self.MinIndex:
                self.LastSave = -1
            self.MinIndex = (self.MinIndex + 1) % UNDO_BUFFER_LENGTH
        self.MinIndex = max(self.MinIndex, 0)
    
    # Return current state of buffer
    def Current(self):
        return self.Buffer[self.CurrentIndex]
    
    # Change current state to previous in buffer and return new current state
    def Previous(self):
        if self.CurrentIndex != self.MinIndex:
            self.CurrentIndex = (self.CurrentIndex - 1) % UNDO_BUFFER_LENGTH
            return self.Buffer[self.CurrentIndex]
        return None
    
    # Change current state to next in buffer and return new current state
    def Next(self):
        if self.CurrentIndex != self.MaxIndex:
            self.CurrentIndex = (self.CurrentIndex + 1) % UNDO_BUFFER_LENGTH
            return self.Buffer[self.CurrentIndex]
        return None
    
    # Return True if current state is the first in buffer
    def IsFirst(self):
        return self.CurrentIndex == self.MinIndex
    
    # Return True if current state is the last in buffer
    def IsLast(self):
        return self.CurrentIndex == self.MaxIndex

    # Note that current state is saved
    def CurrentSaved(self):
        self.LastSave = self.CurrentIndex
        
    # Return True if current state is saved
    def IsCurrentSaved(self):
        return self.LastSave == self.CurrentIndex


#-------------------------------------------------------------------------------
#                           Controler for SVGUIEditor
#-------------------------------------------------------------------------------

class SVGUIControler:
    
    def __init__(self):
        self.LastNewIndex = 0
        self.Reset()
        
    def Reset(self):
        self.current_id = 0
        self.FilePath = ""
        self.FileName = ""
        self.Interface = None
        self.InterfaceBuffer = None
        self.CutBuffer = None
        
    # Returns a new id
    def GetNewId(self):
        self.current_id += 1
        return self.current_id
    
    def CreateNewInterface(self, name="Interface"):
        self.Reset()
        self.Interface = svgui.Container()
        self.Interface.setname(name)
        self.Interface.setid(self.GetNewId())
        self.InterfaceBuffer = UndoBuffer(self.Copy(self.Interface), False)
        self.SetFilePath("")
    
    def HasOpenedInterface(self):
        return self.Interface != None
    
    def GetInterfaceInfos(self):
        if self.Interface != None:
            return GetElementTreeInfos(self.Interface)
    
    def GetElementByName(self, name):
        if name:
            if self.Interface.getname() == name:
                return self.Interface
            else:
                element = self.Interface.getelementbyname(name)
                if element is not None:
                    return element
        return None
    
    def GetElementById(self, id):
        if id is not None:
            if self.Interface.getid() == id:
                return self.Interface
            else:
                element = self.Interface.getelementbyid(id)
                if element is not None:
                    return element
        return None
    
    def IsExistingName(self, name):
        return self.GetElementByName(name) is not None

    def IsExistingId(self, id):
        return self.GetElementById(id) is not None
    
    def GetElementAttributes(self, id):
        element = self.GetElementById(id)
        if element is not None:
            return element.getElementAttributes()
        return None
    
    def SetElementAttribute(self, id, attr, value):
        element = self.GetElementById(id)
        if element is not None:
            element.setElementValue(attr, value)
            self.BufferInterface()
    
    def IsElementInterface(self, id):
        return id is not None and self.Interface.getid() == id
        
    def IsElementContainer(self, id):
        element = self.GetElementById(id)
        if element is not None:
            return GetElementType(element) in [ITEM_CONTAINER, ITEM_NOTEBOOKCONTENT]
        return False
    
    def IsElementNoteBook(self, id):
        element = self.GetElementById(id)
        if element is not None:
            return GetElementType(element) == ITEM_NOTEBOOK
        return False
    
    def IsElementNoteBookContent(self, id):
        element = self.GetElementById(id)
        if element is not None:
            return GetElementType(element) == ITEM_NOTEBOOKCONTENT
        return False
    
    def GetElementType(self, id):
        element = self.GetElementById(id)
        if element is not None:
            return GetElementType(element)
        return None
    
    def GetElementName(self, id):
        element = self.GetElementById(id)
        if element is not None:
            return element.getname()
        return ""
    
    def GetFilePath(self):
        return self.FilePath
    
    def GetSVGFilePath(self):
        return os.path.splitext(self.FilePath)[0] + ".svg"
    
    # Return file name and point out if file is up to date
    def GetFilename(self):
        if self.InterfaceBuffer and self.InterfaceBuffer.IsCurrentSaved():
            return self.FileName
        else:
            return "~%s~"%self.FileName
    
    def SetFilePath(self, filepath):
        self.FilePath = filepath
        if filepath == "":
            self.LastNewIndex += 1
            self.FileName = "Unnamed%d"%self.LastNewIndex
        else:
            self.FileName = os.path.splitext(os.path.basename(filepath))[0]
    
    def OpenXMLFile(self, filepath):
        xmlfile = open(filepath, 'r')
        tree = minidom.parse(xmlfile)
        xmlfile.close()
        
        self.Interface = svgui.Container()
        for child in tree.childNodes:
            if child.nodeType == tree.ELEMENT_NODE and child.nodeName == "Interface":
                self.Interface.loadXMLTree(child, ["xmlns", "xmlns:xsi", "xsi:schemaLocation"])
                self.SetFilePath(filepath)
                self.InterfaceBuffer = UndoBuffer(self.Copy(self.Interface), True)
                self.current_id = self.Interface.getmaxid()
        return True
    
    def SaveXMLFile(self, filepath = None):
        if not filepath and self.FilePath == "":
            return False
        else:
            text = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"no\"?>\n"
            extras = {"xmlns":"http://www.w3.org/2001/XMLSchema",
                      "xmlns:xsi":"http://www.w3.org/2001/XMLSchema-instance",
                      "xsi:schemaLocation" : "def_xsd.xsd"}
            text += self.Interface.generateXMLText("Interface", 0, extras)

            if filepath is not None:
                xmlfile = open(filepath,"w")
            else:
                xmlfile = open(self.FilePath,"w")
            xmlfile.write(text)
            xmlfile.close()
            self.InterfaceBuffer.CurrentSaved()
            if filepath is not None:
                self.SetFilePath(filepath)
            return True
    
    def GenerateProgram(self, backend, window_size):
        elements = self.GetElementsByType()
        if backend == "C":
            folder = os.path.split(self.GetFilePath())[0]
            generator = SVGUICGenerator(elements, self.GetSVGFilePath(), self.GetFilePath())
        elif backend == "Haxe":
            folder = os.path.join(os.path.split(self.GetFilePath())[0], "build")
            if not os.path.exists(folder):
                os.mkdir(folder)
            generator = SVGUIHaxeGenerator(elements, self.GetSVGFilePath(), self.GetFilePath())
        generator.GenerateProgram(window_size, folder)
    
    def GetControlTypes(self):
        if self.Interface is not None:
            return self.Interface.getcontentchoices()
        return []
    
    def GetElementsNames(self):
        names = []
        if self.Interface != None:
            names.append(self.Interface.getname())
            names.extend(self.Interface.getelementsnames())
        return names
    
    def AddElement(self, parent_id, type, name):
        if self.Interface is not None:
            self.current_id += 1
            element = self.GetElementById(parent_id)
            if element is not None and GetElementType(element) in [ITEM_CONTAINER, ITEM_NOTEBOOKCONTENT]:
                element.addelement(type, name, self.current_id)
                self.BufferInterface()
                return self.current_id
            elif element is not None and GetElementType(element) == ITEM_NOTEBOOK:
                element.addcontent(name, self.current_id)
                self.BufferInterface()
                return self.current_id
        return None
    
    def DeleteElement(self, id):
        if self.Interface is not None:
            self.Interface.removeelementbyid(id)
            self.BufferInterface()
    
    def CutElement(self, id):
        if self.Interface is not None:
            self.CutBuffer = self.Interface.removeelementbyid(id)
            self.BufferInterface()
    
    def PasteElement(self, parent_id):
        element = self.GetElementById(parent_id)            
        if element is not None and GetElementType(element) in [ITEM_CONTAINER, ITEM_NOTEBOOKCONTENT]:
            if GetElementType(self.CutBuffer) != ITEM_NOTEBOOKCONTENT:
                element.appendcontent({"name" : GetElementName(self.CutBuffer), "value" : self.CutBuffer})
                paste_id = self.CutBuffer.getid()
                self.CutBuffer = None
                self.BufferInterface()
                return paste_id
        elif element is not None and GetElementType(element) == ITEM_NOTEBOOK:
            if GetElementType(self.CutBuffer) == ITEM_NOTEBOOKCONTENT:
                element.appendNoteBookContent(self.CutBuffer)
                paste_id = self.CutBuffer.getid()
                self.CutBuffer = None
                self.BufferInterface()
                return paste_id
        return None
    
    def MoveElement(self, element_id, target_id):
        if self.IsElementContainer(element_id):
            element = self.GetElementById(element_id)
            if element.getelementbyid(target_id):
                return False
        element = self.Interface.removeelementbyid(element_id)
        target = self.GetElementById(target_id)
        if target is not None and GetElementType(target) in [ITEM_CONTAINER, ITEM_NOTEBOOKCONTENT]:
            target.appendcontent({"name" : GetElementName(element), "value" : element})
            self.BufferInterface()
            return True
        elif target is not None and GetElementType(target) == ITEM_NOTEBOOK:
            target.appendNoteBookContent(element)
            self.BufferInterface()
            return True
        return False
        
    def IsCutBufferEmpty(self):
        return self.CutBuffer is None
    
    def GetElementsByType(self, element_type=None, root=None):
        elements = []
        if root is None:
            root = self.Interface
        if root is not None:
            root_type = GetElementType(root)
            if root_type == element_type or element_type is None:
                elements.append(root)
            if root_type in [ITEM_CONTAINER, ITEM_NOTEBOOKCONTENT]:
                for child in root.getelements():
                    elements.extend(self.GetElementsByType(element_type, child))
            if root_type == ITEM_NOTEBOOK:
                for child in root.getcontents():
                    elements.extend(self.GetElementsByType(element_type, child))
        return elements

#-------------------------------------------------------------------------------
#                      Current Buffering Management Functions
#-------------------------------------------------------------------------------

    """
    Return a copy of the project
    """
    def Copy(self, model):
        return cPickle.loads(cPickle.dumps(model))

    def BufferInterface(self):
        self.InterfaceBuffer.Buffering(self.Copy(self.Interface))
    
    def InterfaceIsSaved(self):
        if self.InterfaceBuffer:
            return self.InterfaceBuffer.IsCurrentSaved()
        else:
            return True

    def LoadPrevious(self):
        self.Interface = self.Copy(self.InterfaceBuffer.Previous())
    
    def LoadNext(self):
        self.Interface = self.Copy(self.InterfaceBuffer.Next())
    
    def GetBufferState(self):
        first = self.InterfaceBuffer.IsFirst()
        last = self.InterfaceBuffer.IsLast()
        return not first, not last

        