/***************************************
  $Header: /home/amb/wwwoffle/RCS/wwwoffle.c 2.29 1999/10/15 09:26:54 amb Exp $

  WWWOFFLE - World Wide Web Offline Explorer - Version 2.5b.
  A user level program to interact with the server.
  ******************/ /******************
  Written by Andrew M. Bishop

  This file Copyright 1996,97,98,99 Andrew M. Bishop
  It may be distributed under the GNU Public License, version 2, or
  any higher version.  See section COPYING of the GNU Public license
  for conditions under which this file may be redistributed.
  ***************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <limits.h>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "version.h"
#include "wwwoffle.h"
#include "document.h"
#include "misc.h"
#include "config.h"
#include "sockets.h"
#include "errors.h"


static void usage(int verbose);


/*+ The action to perform. +*/
typedef enum _Action
{
 None,                          /*+ Undecided. +*/

 Online,                        /*+ Tell the server that we are online. +*/
 Autodial,                      /*+ Tell the server that we are in autodial mode. +*/
 Offline,                       /*+ Tell the server that we are offline. +*/

 Fetch,                         /*+ Tell the server to fetch the requested pages. +*/

 Config,                        /*+ Tell the server to re-read the configuration file. +*/

 Purge,                         /*+ Tell the server to purge pages. +*/

 Kill,                          /*+ Tell the server to exit. +*/

 Get,                           /*+ Tell the server to GET pages. +*/
 Post,                          /*+ Tell the server to POST a page. +*/
 Put,                           /*+ Tell the server to PUT a page. +*/

 Output,                        /*+ Get a page from the server and output it. +*/
 OutputWithHeader               /*+ Get a page and the headers from the server and output it. +*/
}
Action;


static void add_url_list(char **links);
static void add_url_file(char *url_file);

/*+ The list of URLs or files. +*/
char **url_file_list=NULL;
/*+ The number of URLs or files. +*/
int n_url_file_list=0;


/*++++++++++++++++++++++++++++++++++++++
  The main program.
  ++++++++++++++++++++++++++++++++++++++*/

int main(int argc, char** argv)
{
 int i;
 int recursive_depth=0,recursive_mode=0,force=0;
 int stylesheets=0,images=0,frames=0,scripts=0,objects=0;

 Action action=None;

 char *env=NULL;
 char *host=NULL;
 int port=0;

 /* Parse the command line options */

 ConfigFile=NULL;

 if(argc==1)
    usage(0);

 for(i=1;i<argc;i++)
   {
    if(!strcmp(argv[i],"-h"))
       usage(1);

    if(!strcmp(argv[i],"-o"))
      {
       if(action!=None)
         {fprintf(stderr,"wwwoffle: Only one command at a time.\n\n");usage(0);}
       action=Output;
       argv[i]=NULL;
       continue;
      }

    if(!strcmp(argv[i],"-O"))
      {
       if(action!=None)
         {fprintf(stderr,"wwwoffle: Only one command at a time.\n\n");usage(0);}
       action=OutputWithHeader;
       argv[i]=NULL;
       continue;
      }

    if(!strcmp(argv[i],"-post"))
      {
       if(action!=None)
         {fprintf(stderr,"wwwoffle: Only one command at a time.\n\n");usage(0);}
       action=Post;
       argv[i]=NULL;
       continue;
      }

    if(!strcmp(argv[i],"-put"))
      {
       if(action!=None)
         {fprintf(stderr,"wwwoffle: Only one command at a time.\n\n");usage(0);}
       action=Put;
       argv[i]=NULL;
       continue;
      }

    if(!strcmp(argv[i],"-F"))
      {
       if(action!=None && action!=Get)
         {fprintf(stderr,"wwwoffle: Only one command at a time.\n\n");usage(0);}
       action=Get;
       if(force)
         {fprintf(stderr,"wwwoffle: Only one '-F' argument may be given.\n"); exit(1);}
       force=1;
       argv[i]=NULL;
       continue;
      }

    if(!strncmp(argv[i],"-g",2))
      {
       if(action!=None && action!=Get)
         {fprintf(stderr,"wwwoffle: Only one command at a time.\n\n");usage(0);}
       action=Get;
       if(strchr(argv[i]+2,'S'))
         {
          if(stylesheets)
            {fprintf(stderr,"wwwoffle: Only one '-gS' argument may be given.\n"); exit(1);}
          stylesheets=1;
         }
       if(strchr(argv[i]+2,'i'))
         {
          if(images)
            {fprintf(stderr,"wwwoffle: Only one '-gi' argument may be given.\n"); exit(1);}
          images=1;
         }
       if(strchr(argv[i]+2,'f'))
         {
          if(frames)
            {fprintf(stderr,"wwwoffle: Only one '-gf' argument may be given.\n"); exit(1);}
          frames=1;
         }
       if(strchr(argv[i]+2,'s'))
         {
          if(scripts)
            {fprintf(stderr,"wwwoffle: Only one '-gs' argument may be given.\n"); exit(1);}
          scripts=1;
         }
       if(strchr(argv[i]+2,'o'))
         {
          if(objects)
            {fprintf(stderr,"wwwoffle: Only one '-go' argument may be given.\n"); exit(1);}
          objects=1;
         }
       if(argv[i][2]==0)
          fprintf(stderr,"wwwoffle: The '-g' option does nothing on its own.\n");

       argv[i]=NULL;
       continue;
      }

    if(!strcmp(argv[i],"-i"))
      {
       fprintf(stderr,"wwwoffle: The '-i' option is being replaced by '-gi'.\n");

       if(action!=None && action!=Get)
         {fprintf(stderr,"wwwoffle: Only one command at a time.\n\n");usage(0);}
       action=Get;
       if(images)
         {fprintf(stderr,"wwwoffle: Only one '-i' argument may be given.\n"); exit(1);}
       images=1;
       argv[i]=NULL;
       continue;
      }

    if(!strcmp(argv[i],"-f"))
      {
       fprintf(stderr,"wwwoffle: The '-f' option is being replaced by '-gf'.\n");

       if(action!=None && action!=Get)
         {fprintf(stderr,"wwwoffle: Only one command at a time.\n\n");usage(0);}
       action=Get;
       if(frames)
         {fprintf(stderr,"wwwoffle: Only one '-f' argument may be given.\n"); exit(1);}
       frames=1;
       argv[i]=NULL;
       continue;
      }

    if(!strncmp(argv[i],"-R",2) || !strncmp(argv[i],"-r",2) || !strncmp(argv[i],"-d",2))
      {
       if(action!=None && action!=Get)
         {fprintf(stderr,"wwwoffle: Only one command at a time.\n\n");usage(0);}
       action=Get;

       if(recursive_depth)
         {fprintf(stderr,"wwwoffle: Only one '-d', '-r' or '-R' argument may be given.\n"); exit(1);}

       if(argv[i][1]=='d')
          recursive_mode=1;
       else if(argv[i][1]=='r')
          recursive_mode=2;
       else /* argv[i][1]=='R' */
          recursive_mode=3;

       if(argv[i][2])
          recursive_depth=atoi(&argv[i][2]);
       else
          recursive_depth=1;
       if(recursive_depth<=0)
         {fprintf(stderr,"wwwoffle: The '-%c' argument may only be followed by a positive integer.\n",argv[i][1]); exit(1);}

       argv[i]=NULL;
       continue;
      }

    if(!strcmp(argv[i],"-p"))
      {
       char *colon;

       if(++i>=argc)
         {fprintf(stderr,"wwwoffle: The '-p' argument requires a hostname and optionally a port number.\n"); exit(1);}

       if(ConfigFile)
         {fprintf(stderr,"wwwoffle: The '-p' and '-c' options cannot be used together.\n"); exit(1);}

       if((colon=strchr(argv[i],':')))
         {
          *colon++=0;

          port=atoi(colon);

          if(port<=0 || port>=65536)
            {fprintf(stderr,"wwwoffle: The port number %d '%s' is invalid.\n",port,argv[i]); exit(1);}
         }

       host=argv[i];

       argv[i-1]=NULL;
       argv[i]=NULL;
       continue;
      }

    if(!strcmp(argv[i],"-c"))
      {
       if(++i>=argc)
         {fprintf(stderr,"wwwoffle: The '-c' argument requires a configuration file name.\n"); exit(1);}

       if(host)
         {fprintf(stderr,"wwwoffle: The '-p' and '-c' options cannot be used together.\n"); exit(1);}

       ConfigFile=argv[i];

       argv[i-1]=NULL;
       argv[i]=NULL;
       continue;
      }

    if(!strcmp(argv[i],"-on") || !strcmp(argv[i],"-online"))
      {
       if(action!=None)
         {fprintf(stderr,"wwwoffle: Only one command at a time.\n\n");usage(0);}
       action=Online;
       argv[i]=NULL;
       continue;
      }

    if(!strcmp(argv[i],"-auto") || !strcmp(argv[i],"-autodial"))
      {
       if(action!=None)
         {fprintf(stderr,"wwwoffle: Only one command at a time.\n\n");usage(0);}
       action=Autodial;
       argv[i]=NULL;
       continue;
      }

    if(!strcmp(argv[i],"-off") || !strcmp(argv[i],"-offline"))
      {
       if(action!=None)
         {fprintf(stderr,"wwwoffle: Only one command at a time.\n\n");usage(0);}
       action=Offline;
       argv[i]=NULL;
       continue;
      }

    if(!strcmp(argv[i],"-fetch"))
      {
       if(action!=None)
         {fprintf(stderr,"wwwoffle: Only one command at a time.\n\n");usage(0);}
       action=Fetch;
       argv[i]=NULL;
       continue;
      }

    if(!strcmp(argv[i],"-config"))
      {
       if(action!=None)
         {fprintf(stderr,"wwwoffle: Only one command at a time.\n\n");usage(0);}
       action=Config;
       argv[i]=NULL;
       continue;
      }

    if(!strcmp(argv[i],"-purge"))
      {
       if(action!=None)
         {fprintf(stderr,"wwwoffle: Only one command at a time.\n\n");usage(0);}
       action=Purge;
       argv[i]=NULL;
       continue;
      }

    if(!strcmp(argv[i],"-kill"))
      {
       if(action!=None)
         {fprintf(stderr,"wwwoffle: Only one command at a time.\n\n");usage(0);}
       action=Kill;
       argv[i]=NULL;
       continue;
      }

    if(argv[i][0]=='-' && argv[i][1])
      {
       fprintf(stderr,"wwwoffle: Unknown option '%s'.\n\n",argv[i]);
       usage(0);
      }

    add_url_file(argv[i]);
   }

 if(action==None)
    action=Get;

 if((action==Post || action==Put) && n_url_file_list>1)
   {
    fprintf(stderr,"wwwoffle: The -post and -put options only allow one URL.\n\n");
    usage(0);
   }

 /* Initialise things. */

 if(!ConfigFile && !host && (env=getenv("WWWOFFLE_PROXY")))
   {
    if(*env=='/')
       ConfigFile=env;
    else
      {
       char *colon1,*colon2;

       host=(char*)malloc(strlen(env)+1);
       strcpy(host,env);

       if((colon1=strchr(host,':')))
         {
          *colon1=0;
          if((colon2=strchr(colon1+1,':')))
            {
             *colon2=0;
             if(action==Get || action==Output || action==OutputWithHeader)
                port=atoi(colon1+1);
             else
                port=atoi(colon2+1);
            }
          else
             port=atoi(colon1+1);

          if(port<=0 || port>=65536)
            {fprintf(stderr,"wwwoffle: The port number %d '%s' is invalid.\n",port,env); exit(1);}
         }
      }
   }

 InitErrorHandler("wwwoffle",0,1);

 if(ConfigFile)
   {
    if(ReadConfigFile(2))
       PrintMessage(Fatal,"Error in configuration file '%s'.",ConfigFile);

    host=GetLocalHost(0);
   }

 /* The connections to the WWWOFFLE server. */

 if(action!=Get && action!=Post && action!=Put && action!=Output && action!=OutputWithHeader)
   {
    int socket;
    char *line=NULL;

    socket=OpenClientSocket(host?host:"localhost",port?port:WWWOFFLE_Port,ConnectTimeout);
    init_buffer(socket);

    if(socket==-1)
       PrintMessage(Fatal,"Cannot open connection to wwwoffle server %s port %d.",host?host:"localhost",port?port:WWWOFFLE_Port);

    /* Send the message. */

    if(PassWord)
       write_formatted(socket,"WWWOFFLE PASSWORD %s\r\n",PassWord);

    if(action==Online)
       write_string(socket,"WWWOFFLE ONLINE\r\n");
    else if(action==Autodial)
       write_string(socket,"WWWOFFLE AUTODIAL\r\n");
    else if(action==Offline)
       write_string(socket,"WWWOFFLE OFFLINE\r\n");
    else if(action==Fetch)
       write_string(socket,"WWWOFFLE FETCH\r\n");
    else if(action==Config)
       write_string(socket,"WWWOFFLE CONFIG\r\n");
    else if(action==Purge)
       write_string(socket,"WWWOFFLE PURGE\r\n");
    else if(action==Kill)
       write_string(socket,"WWWOFFLE KILL\r\n");
    else
       write_string(socket,"WWWOFFLE BOGUS\r\n");

    while((line=read_line(socket,line)))
      {
       fputs(line,stdout);
       fflush(stdout);
      }

    CloseSocket(socket);
   }

 /* The connections to the http proxy. */

 else if(action==Get)
   {
    URL *Url;
    struct stat buf;
    char *refresh=NULL;

    if(recursive_depth || recursive_mode || force || stylesheets || images || frames || scripts || objects)
       refresh=CreateRefreshPath(recursive_depth,recursive_mode,force,
                                 stylesheets,images,frames,scripts,objects);

    for(i=0;i<n_url_file_list;i++)
       if(strcmp(url_file_list[i],"-") && stat(url_file_list[i],&buf))
         {
          int socket;
          char buffer[READ_BUFFER_SIZE];

          socket=OpenClientSocket(host?host:"localhost",port?port:HTTP_Port,ConnectTimeout);
          init_buffer(socket);

          if(socket==-1)
             PrintMessage(Fatal,"Cannot open connection to wwwoffle server %s port %d.",host?host:"localhost",port?port:HTTP_Port);

          Url=SplitURL(url_file_list[i]);

          if(refresh)
             printf("Getting: %s (with recursive options).\n",Url->name);
          else
             printf("Getting: %s\n",Url->name);

          if(Url->user)
            {
             char *userpass1=(char*)(malloc(strlen(Url->user)+(Url->pass?strlen(Url->pass):1)+3)),*userpass2;

             sprintf(userpass1,"%s:%s",Url->user,Url->pass?Url->pass:"");
             userpass2=Base64Encode(userpass1,strlen(userpass1));

             if(refresh)
                write_formatted(socket,"GET %s/?%s HTTP/1.0\r\n"
                                       "Authorization: Basic %s\r\n"
                                       "Pragma: wwwoffle\r\n"
                                       "Accept: */*\r\n"
                                       "\r\n",
                                refresh,Url->name,userpass2);
             else
                write_formatted(socket,"GET %s HTTP/1.0\r\n"
                                       "Authorization: Basic %s\r\n"
                                       "Pragma: wwwoffle\r\n"
                                       "Accept: */*\r\n"
                                       "\r\n",
                                Url->name,userpass2);

             free(userpass1);
             free(userpass2);
            }
          else
            {
             if(refresh)
                write_formatted(socket,"GET %s/?%s HTTP/1.0\r\n"
                                       "Pragma: wwwoffle\r\n"
                                       "Accept: */*\r\n"
                                       "\r\n",
                                refresh,Url->name);
             else
                write_formatted(socket,"GET %s HTTP/1.0\r\n"
                                       "Pragma: wwwoffle\r\n"
                                       "Accept: */*\r\n"
                                       "\r\n",
                                Url->name);
            }

          while(read_data(socket,buffer,READ_BUFFER_SIZE)>0)
             ;

          CloseSocket(socket);

          FreeURL(Url);
         }
       else if(!strcmp(url_file_list[i],"-") || S_ISREG(buf.st_mode))
         {
          int file;
          char *buffer=(char*)malloc(strlen(url_file_list[i])+256);
          char **links;

          if(strcmp(url_file_list[i],"-"))
            {
             file=open(url_file_list[i],O_RDONLY);
             if(file==-1)
               {PrintMessage(Warning,"Cannot open file '%s' for reading.",url_file_list[i]);continue;}

             printf("Reading: %s\n",url_file_list[i]);
            }
          else
            {
             file=fileno(stdin);

             printf("Reading: <stdin>\n");
            }

          init_buffer(file);

          strcpy(buffer,"file://localhost");
          if(*url_file_list[i]=='/')
             strcat(buffer,url_file_list[i]);
          else
            {
             char cwd[PATH_MAX+1];

             if(!getcwd(cwd,PATH_MAX))
             cwd[0]=0;

             strcat(buffer,cwd);
             strcat(buffer,"/");
             strcat(buffer,url_file_list[i]);
            }

          Url=SplitURL(buffer);
          ParseHTML(file,Url);

          if(stylesheets && (links=GetReferences(RefStyleSheet)))
             add_url_list(links);

          if(images && (links=GetReferences(RefImage)))
             add_url_list(links);

          if(frames && (links=GetReferences(RefFrame)))
             add_url_list(links);

          if(scripts && (links=GetReferences(RefScript)))
             add_url_list(links);

          if(objects && (links=GetReferences(RefObject)))
             add_url_list(links);

          if(objects && (links=GetReferences(RefInlineObject)))
             add_url_list(links);

          if((links=GetReferences(RefLink)))
             add_url_list(links);

          FreeURL(Url);
          free(buffer);
          if(file!=0)
             close(file);
         }
       else
          PrintMessage(Warning,"The file '%s' is not a regular file.",url_file_list[i]);
   }
 else if(action==Post || action==Put)
   {
    URL *Url;
    int socket,n,length=0;
    char buffer[READ_BUFFER_SIZE];
    char *data=(char*)malloc(READ_BUFFER_SIZE+1);

    socket=OpenClientSocket(host?host:"localhost",port?port:HTTP_Port,ConnectTimeout);
    init_buffer(socket);

    if(socket==-1)
       PrintMessage(Fatal,"Cannot open connection to wwwoffle server %s port %d.",host?host:"localhost",port?port:HTTP_Port);

    Url=SplitURL(url_file_list[0]);

    if(action==Post)
       printf("Posting: %s\n",Url->name);
    else
       printf("Putting: %s\n",Url->name);

    while((n=read_data(0,data+length,READ_BUFFER_SIZE))>0)
      {
       length+=n;
       data=(char*)realloc((void*)data,length+READ_BUFFER_SIZE+1);
      }

    if(Url->user)
      {
       char *userpass1=(char*)(malloc(strlen(Url->user)+(Url->pass?strlen(Url->pass):1)+3)),*userpass2;

       sprintf(userpass1,"%s:%s",Url->user,Url->pass?Url->pass:"");
       userpass2=Base64Encode(userpass1,strlen(userpass1));

       if(action==Post)
          write_formatted(socket,"POST %s HTTP/1.0\r\n"
                                 "Authorization: Basic %s\r\n"
                                 "Content-Type: application/x-www-form-urlencoded\r\n"
                                 "Content-Length: %d\r\n"
                                 "Pragma: wwwoffle\r\n"
                                 "Accept: */*\r\n"
                                 "\r\n",
                          Url->name,userpass2,length);
       else
          write_formatted(socket,"PUT %s HTTP/1.0\r\n"
                                 "Authorization: Basic %s\r\n"
                                 "Content-Length: %d\r\n"
                                 "Pragma: wwwoffle\r\n"
                                 "Accept: */*\r\n"
                                 "\r\n",
                          Url->name,userpass2,length);

       free(userpass1);
       free(userpass2);
      }
    else
      {
       if(action==Post)
          write_formatted(socket,"POST %s HTTP/1.0\r\n"
                                 "Content-Type: application/x-www-form-urlencoded\r\n"
                                 "Content-Length: %d\r\n"
                                 "Pragma: wwwoffle\r\n"
                                 "Accept: */*\r\n"
                                 "\r\n",
                          Url->name,length);
       else
          write_formatted(socket,"PUT %s HTTP/1.0\r\n"
                                 "Content-Length: %d\r\n"
                                 "Pragma: wwwoffle\r\n"
                                 "Accept: */*\r\n"
                                 "\r\n",
                          Url->name,length);
      }

    write_data(socket,data,length);

    write_data(socket,"\r\n",2);

    while(read_data(socket,buffer,READ_BUFFER_SIZE)>0)
       ;

    CloseSocket(socket);

    FreeURL(Url);
    free(data);
   }
 else /* action==Output or action==OutputWithHeader */
   {
    URL *Url;
    int socket;
    char *line=NULL,buffer[READ_BUFFER_SIZE];
    int nbytes;

    if(!n_url_file_list)
       PrintMessage(Fatal,"No URL specified to output.");

    socket=OpenClientSocket(host?host:"localhost",port?port:HTTP_Port,ConnectTimeout);
    init_buffer(socket);

    if(socket==-1)
       PrintMessage(Fatal,"Cannot open connection to wwwoffle server %s port %d.",host?host:"localhost",port?port:HTTP_Port);

    Url=SplitURL(url_file_list[0]);

    if(action!=Output && action!=OutputWithHeader)
       fprintf(stderr,"Getting: %s\n",Url->name);

    if(Url->user)
      {
       char *userpass1=(char*)(malloc(strlen(Url->user)+(Url->pass?strlen(Url->pass):1)+3)),*userpass2;

       sprintf(userpass1,"%s:%s",Url->user,Url->pass?Url->pass:"");
       userpass2=Base64Encode(userpass1,strlen(userpass1));

       write_formatted(socket,"GET %s HTTP/1.0\r\n"
                              "Authorization: Basic %s\r\n"
                              "Pragma: wwwoffle\r\n"
                              "Accept: */*\r\n"
                              "\r\n",
                       Url->name,userpass2);

       free(userpass1);
       free(userpass2);
      }
    else
       write_formatted(socket,"GET %s HTTP/1.0\r\n"
                              "Pragma: wwwoffle\r\n"
                              "Accept: */*\r\n"
                              "\r\n",
                       Url->name);

    if((line=read_line(socket,line)))
      {
       char *willget="HTTP/1.0 404 WWWOFFLE Will Get\r\n";
       int status;

       sscanf(line,"%*s %d",&status);

       if(!strcmp(willget,line))
          fprintf(stderr,"The URL is not in the cache but has been requested.\n");
       else if((status>=300 && status<400) && action!=OutputWithHeader)
          fprintf(stderr,"The URL has been moved, check with a browser.\n");
       else if(status!=200 && action!=OutputWithHeader)
          fprintf(stderr,"The URL returns an error message, check with a browser.\n");
       else
         {
          if(action==OutputWithHeader)
             fputs(line,stdout);

          while((line=read_line(socket,line)))
            {
             if(action==OutputWithHeader)
                fputs(line,stdout);
             if(line[0]=='\r' || line[0]=='\n')
                break;
            }

          while((nbytes=read_data(socket,buffer,READ_BUFFER_SIZE))>0)
             fwrite(buffer,1,nbytes,stdout);
         }
      }
    else
       PrintMessage(Fatal,"Cannot read from wwwoffle server.");

    CloseSocket(socket);

    FreeURL(Url);
   }

 /* exit. */

 return(0);
}


/*++++++++++++++++++++++++++++++++++++++
  Print the program usage in long or short format.

  int verbose True for long format.
  ++++++++++++++++++++++++++++++++++++++*/

static void usage(int verbose)
{
 fprintf(stderr,
         "\n"
         "WWWOFFLE - World Wide Web Offline Explorer - Version %s\n"
         "\n",WWWOFFLE_VERSION);

 if(verbose)
    fprintf(stderr,
            "(c) Andrew M. Bishop 1996,97,98,99 [       amb@gedanken.demon.co.uk ]\n"
            "                                   [http://www.gedanken.demon.co.uk/]\n"
            "\n");

 fprintf(stderr,
         "Usage: wwwoffle -h\n"
         "       wwwoffle -online | -autodial | -offline | -fetch\n"
         "       wwwoffle -config | -purge | -kill\n"
         "       wwwoffle [-o|-O] <url>\n"
         "       wwwoffle [-g[Sisfo]] [-F] [-(d|r|R)[<depth>]] <url> ...\n"
         "       wwwoffle [-g[Sisfo]] [-F] [-(d|r|R)[<depth>]] [<file>|-] ...\n"
         "       wwwoffle -post <url> | -put <url>\n"
         "\n"
         "Any of these can also take:  [-p <host>[:<port>] | -c <config-file>]\n"
         "The environment variable WWWOFFLE_PROXY can be set instead of -p or -c options.\n"
         "\n");

 if(verbose)
    fprintf(stderr,
            "wwwoffle -h          : Display this help.\n"
            "\n"
            "wwwoffle -on[line]   : Indicate to the server that the network is active.\n"
            "                       (Proxy requests will be fetched from remote hosts.)\n"
            "\n"
            "wwwoffle -auto[dial] : Indicate to the server that the network is automatic.\n"
            "                       (Proxy requests will be fetched from remote hosts\n"
            "                        ONLY if they are not already cached.)\n"
            "\n"
            "wwwoffle -off[line]  : Indicate to the server that the network is inactive.\n"
            "                       (Proxy requests will be fetched from cache or recorded.)\n"
            "\n"
            "wwwoffle -fetch      : Force the server to fetch the pages that are recorded.\n"
            "\n"
            "wwwoffle -config     : Force the server to re-read the configuration file.\n"
            "\n"
            "wwwoffle -purge      : Force the server to purge pages from the cache.\n"
            "\n"
            "wwwoffle -kill       : Force the server to exit cleanly.\n"
            "\n"
            "wwwoffle <url> ...   : Fetch the specified URLs.\n"
            "\n"
            "wwwoffle <file> ...  : Fetch the URLs that are links in the specified file.\n"
            "\n"
            " -o                  : Fetch the URL and output it on the standard output.\n"
            " -O                  : As above but include the HTTP header.\n"
            "\n"
            " -g[Sisfo]           : Fetch the items included in the specified URLs.\n"
            "                       (S=stylesheets, i=images, f=frames, s=scripts, o=objects)\n"
            " -F                  : Force the url to be refreshed even if already cached.\n"
            " -(d|r|R)[<depth>]   : Fetch pages linked to the URLs and their links,\n"
            "                       going no more than <depth> steps (default 1).\n"
            "                        (-d => URLs in the same directory or sub-directory)\n"
            "                        (-r => URLs on the same host)\n"
            "                        (-R => URLs on any host)\n"
            "\n"
            "wwwoffle -post <url> : Create a request using the POST method, the data is read\n"
            "                       from stdin and appended to the request.  The user should\n"
            "                       ensure that the data is correctly url-encoded.\n"
            "wwwoffle -put <url>  : Create a request using the PUT method, the data is read\n"
            "                       from stdin and appended to the request.\n"
            "\n"
            " -p <host>[:<port>]  : The host name and port number to talk to the demon on.\n"
            "                       (Defaults to localhost for the server and\n"
            "                        %d for control port, %d for http proxy port).\n"
            "\n"
            " -c <config-file>    : The name of the configuration file with the hostname,\n"
            "                       port number and the password (if any).\n"
            "\n"
            "WWWOFFLE_PROXY       : An environment variable that can be set to either the\n"
            "                       name of the config file (absolute path) or the hostname\n"
            "                       and port number (both proxy and control) for the proxy.\n"
            "                       e.g. \"/var/spool/wwwoffle/wwwoffle.conf\",\n"
            "                       \"localhost:8080:8081\" or \"localhost:8080\" are valid.\n"
            "\n",DEF_WWWOFFLE_PORT,DEF_HTTP_PORT);

 if(verbose)
    exit(0);
 else
    exit(1);
}


/*++++++++++++++++++++++++++++++++++++++
  Add a list of URLs to the list.

  char **links The list of URLs to add.
  ++++++++++++++++++++++++++++++++++++++*/

static void add_url_list(char **links)
{
 int i;

 for(i=0;links[i];i++)
   {
    URL *linkUrl=SplitURL(links[i]);

    if(strcmp(linkUrl->proto,"file"))
       add_url_file(linkUrl->name);

    FreeURL(linkUrl);
   }
}


/*++++++++++++++++++++++++++++++++++++++
  Add a URL or a file to the list.

  char *url_file The URL or file to add.
  ++++++++++++++++++++++++++++++++++++++*/

static void add_url_file(char *url_file)
{
 if(!(n_url_file_list%16))
   {
    if(n_url_file_list)
       url_file_list=(char**)realloc(url_file_list,(n_url_file_list+16)*sizeof(char*));
    else
       url_file_list=(char**)malloc(16*sizeof(char*));
   }

 url_file_list[n_url_file_list]=(char*)malloc(strlen(url_file)+1);

 strcpy(url_file_list[n_url_file_list],url_file);

 n_url_file_list++;
}
