// TimerActivityMonitor.hh
//
// Copyright (C) 2001, 2002, 2003, 2004, 2005, 2006, 2007 Rob Caelers <robc@krandor.org>
// All rights reserved.
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// $Id: TimerActivityMonitor.hh 1142 2007-02-19 22:57:16Z rcaelers $
//

#ifndef TIMERACTIVITYMONITOR_HH
#define TIMERACTIVITYMONITOR_HH

#include "Core.hh"
#include "IActivityMonitor.hh"
#include "ITimer.hh"

//! An Activity Monitor that takes its activity state from a timer.
/*! This Activity Monitor is 'active' if the timer is running or if
 *  the timer has its idle time not at maximum, 'idle' otherwise.
 */
class TimerActivityMonitor : public IActivityMonitor
{
public:
  //! Constructs an activity monitor that depends on specified timer.
  TimerActivityMonitor(Timer *t) :
    timer(t),
    suspended(false),
    forced_idle(false)
      
  {
    Core *core =  Core::get_instance();
    monitor = core->get_activity_monitor();
  }

  virtual ~TimerActivityMonitor()
  {
  }
  
  //! Stops the activity monitoring.
  void terminate()
  {
  }

  //! Suspends the activity monitoring.
  void suspend()
  {
    suspended = true;
  }

  //! Resumes the activity monitoring.
  void resume()
  {
    suspended = false;
  }

  //! Returns the current state
  ActivityState get_current_state()
  {
    if (forced_idle)
      {
        TRACE_ENTER("ActivityState::get_current_state");
        ActivityState local_state = monitor->get_current_state();

        TRACE_MSG(local_state)
        if (local_state != ACTIVITY_IDLE &&
            local_state != ACTIVITY_SUSPENDED)
          {
            forced_idle = false;
          }

        TRACE_EXIT();
      }

    if (forced_idle)
      {
        return ACTIVITY_IDLE;
      }

    if (suspended)
      {
        return ACTIVITY_SUSPENDED;
      }
    
    Timer::TimerState state = timer->get_state();
    time_t idle = timer->get_elapsed_idle_time();
    time_t reset = timer->get_auto_reset();

    if (state == ITimer::STATE_STOPPED && idle >= reset)
      {
        return ACTIVITY_IDLE;
      }
    else
      {
        return ACTIVITY_ACTIVE;
      }
  }

  //! Force state to be idle.
  void force_idle()
  {
    TRACE_ENTER("TimerActivityMonitor::force_idle");
    forced_idle = true;
    TRACE_EXIT();
  }

  
  // Returns the collected statistics.
  void get_statistics(ActivityMonitorStatistics &stats) const
  {
    (void) stats;
  }


  //! Sets the statistics.
  void set_statistics(const ActivityMonitorStatistics &stats)
  {
    (void) stats;
  }


  //! Resets all collected statistics.
  void reset_statistics()
  {
  }

  
  //! Sets the activity listener of this monitor.
  void set_listener(ActivityMonitorListener *l)
  {
    (void)l;
  }
  
private:
  //! Reference monitor
  IActivityMonitor *monitor;
  
  //! Reference timer.
  Timer *timer;

  //! Monitor suspended?
  bool suspended;

  //! Is this timer forced idle?
  bool forced_idle;
};

#endif // TIMERACTIVITYMONITOR_HH
