/* $Id: wl.c,v 1.11 2005/11/10 09:10:02 hacki Exp $ */

/*
 * Copyright (c) 2005 Marcus Glocker <marcus@nazgul.ch>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <sys/param.h>
#include <sys/ioctl.h>
#include <sys/socket.h>

#include <err.h>
#include <ifaddrs.h>
#include <net/if.h>
#include <net/if_media.h>
#include <net80211/ieee80211.h>
#include <net80211/ieee80211_ioctl.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include "wl.h"

/*
 * global variables for this file
 */
static char *speed[25] = {
	"DS1", "1",
	"DS2", "2",
	"DS5", "5.5",
	"DS11", "11",
	"OFDM6", "6",
	"OFDM9", "9",
	"OFDM12", "12s",
	"OFDM18", "18",
	"OFDM24", "24",
	"OFDM36", "36",
	"OFDM48", "48",
	"OFDM54", "54",
	NULL
}; 

/*
 * get_wep()
 *	get wep status
 * Return:
 *	0 = wep disabled, 1 = wep enabled, -1 = error
 */
int
get_wep(const char *interface)
{
	int			r = 0, s, inwkey;
	struct ieee80211_nwkey	nwkey;

	if ((s = socket(AF_INET, SOCK_DGRAM, 0)) == -1)
		return -1;

	memset(&nwkey, 0, sizeof(nwkey));
	strlcpy(nwkey.i_name, interface, sizeof(nwkey.i_name));
	if ((inwkey = ioctl(s, SIOCG80211NWKEY, (caddr_t)&nwkey)) == -1) {
		close(s);
		return -1;
	}
	close(s);

	if (inwkey == 0 && nwkey.i_wepon > 0)
		r = 1;

	return r;
}

/*
 * get_channel()
 *	get channel number
 * Return:
 *	<channel number> = success, 0 = no data, -1 = error
 */
int
get_channel(const char *interface)
{
	int			s, ichan;
	struct ieee80211chanreq	channel;

	if ((s = socket(AF_INET, SOCK_DGRAM, 0)) == -1)
		return -1;

	memset(&channel, 0, sizeof(channel));
	strlcpy(channel.i_name, interface, sizeof(channel.i_name));
	if ((ichan = ioctl(s, SIOCG80211CHANNEL, (caddr_t)&channel)) == -1) {
		close(s);
		return -1;
	}
	close(s);

	if (ichan == 0) {
		if (channel.i_channel < 1000)
			return channel.i_channel;
	}

	return 0;
}

/*
 * get_signal()
 *	get signal strength
 * Return:
 *	<signal strength> = success, 0 = no data, -1 = error
 */
int
get_signal(const char *interface, const char *network)
{
	int				i = 0, s, len;
	char				network_id[IEEE80211_NWID_LEN + 1];
	struct ieee80211_nodereq_all	na;
	struct ieee80211_nodereq	nr[8];

	if ((s = socket(AF_INET, SOCK_DGRAM, 0)) == -1)
		return -1;

	memset(&na, 0, sizeof(na));
	memset(&nr, 0, sizeof(nr));
	na.na_node = nr;
	na.na_size = sizeof(nr);
	strlcpy(na.na_ifname, interface, sizeof(na.na_ifname));
	if (ioctl(s, SIOCG80211ALLNODES, &na) == -1) {
		close(s);
		return -1;
	}
	close(s);

	for (i = 0; i < na.na_nodes; i++) {
		if (nr[i].nr_nwid_len < sizeof(network_id))
			len = nr[i].nr_nwid_len + 1;
		else
			len = sizeof(network_id);
		strlcpy(network_id, nr[i].nr_nwid, len);
		if (!strcmp(network_id, network))
			return nr[i].nr_rssi;
	}

	return 0;
}

/*
 * get_speed()
 *	get media speed
 * Return:
 *	<pointer to speed> = success, NULL = no data / error
 */
char *
get_speed(const char *interface)
{
	int					s, mword;
	struct ifmediareq			ifmr;
	const struct ifmedia_description	*desc;
	const struct ifmedia_description	ifm_subtype_descriptions[] =
	    IFM_SUBTYPE_DESCRIPTIONS;

	if ((s = socket(AF_INET, SOCK_DGRAM, 0)) == -1)
		return NULL;

	memset(&ifmr, 0, sizeof(ifmr));
	strlcpy(ifmr.ifm_name, interface, sizeof(ifmr.ifm_name));
	if (ioctl(s, SIOCGIFMEDIA, (caddr_t)&ifmr) == -1) {
		close(s);
		return NULL;
	}
	close(s);

	mword = ifmr.ifm_active;
	for (desc = ifm_subtype_descriptions; desc->ifmt_string != NULL;
	    desc++) {
		if (IFM_TYPE_MATCH(desc->ifmt_word, mword) &&
		    IFM_SUBTYPE(desc->ifmt_word) == IFM_SUBTYPE(mword))
			return translate_speed(desc->ifmt_string);
	}

	return NULL;
}

/*
 * get_status()
 *	get network status
 * Return:
 *	<pointer to status> = success, NULL = no data / error
 */
char *
get_status(const char *interface)
{
	int					s, bitno;
	static char				status[64];
	struct ifmediareq			ifmr;
	const struct ifmedia_status_description	*ifms;
	const struct ifmedia_status_description	ifm_status_descriptions[] =
	    IFM_STATUS_DESCRIPTIONS;
	const int				ifm_status_valid_list[] =
	    IFM_STATUS_VALID_LIST;

	if ((s = socket(AF_INET, SOCK_DGRAM, 0)) == -1)
		return NULL;

	memset(&ifmr, 0, sizeof(ifmr));
	strlcpy(ifmr.ifm_name, interface, sizeof(ifmr.ifm_name));
	if (ioctl(s, SIOCGIFMEDIA, (caddr_t)&ifmr) == -1) {
		close(s);
		return NULL;
	}
	close(s);

	for (bitno = 0; ifm_status_valid_list[bitno] != 0; bitno++) {
		for (ifms = ifm_status_descriptions; ifms->ifms_valid != 0;
		    ifms++) {
			if (ifms->ifms_type != IFM_TYPE(ifmr.ifm_current) ||
			    ifms->ifms_valid != ifm_status_valid_list[bitno])
				continue;
			strlcpy(status, IFM_STATUS_DESC(ifms, ifmr.ifm_status),
				sizeof(status));
			return status;
		}
		
	}

	return NULL;
}

/*
 * get_nwid()
 *	get wireless network id
 * Return:
 * 	pointer to network id = success, NULL = no data / error
 */
char *
get_nwid(const char *interface)
{
	int			s, len, inwid;
	static char		network_id[IEEE80211_NWID_LEN + 1];
	struct ifreq		ifr;
	struct ieee80211_nwid	nwid;

	if ((s = socket(AF_INET, SOCK_DGRAM, 0)) == -1)
		return NULL;

	memset(&ifr, 0, sizeof(ifr));
	ifr.ifr_data = (caddr_t)&nwid;
	strlcpy(ifr.ifr_name, interface, sizeof(ifr.ifr_name));
	if ((inwid = ioctl(s, SIOCG80211NWID, (caddr_t)&ifr)) == -1) {
		close(s);
		return NULL;
	}
	close(s);

	if (inwid == 0) {
		if (nwid.i_len < sizeof(network_id))
			len = nwid.i_len + 1;
		else
			len = sizeof(network_id);
		strlcpy(network_id, nwid.i_nwid, len);

		return network_id;
	}

	return NULL;
}

/*
 * get_first_wnic()
 *	scans interfaces and returns the first found wireless interface
 * Return:
 *	<pointer to wnic> = success, NULL = no nic found
 */
char *
get_first_wnic(void)
{
	char		*r = NULL;
	char		nic[IF_NAMESIZE];
	struct ifaddrs	*ifap = NULL, *ifa = NULL;

	memset(nic, 0, sizeof(nic));

	if (getifaddrs(&ifap) != 0)
		errx(1, "getifaddrs");

	for (ifa = ifap; ifa; ifa = ifa->ifa_next) {
		if (strcmp(nic, ifa->ifa_name)) {
			if (get_wep(ifa->ifa_name) != -1) {
				r = strdup(ifa->ifa_name);
				break;
			}
		}
		strlcpy(nic, ifa->ifa_name, sizeof(nic));
	}

	freeifaddrs(ifap);

	return r;
}

/*
 * check_nic()
 *	check if the monitored interface still exists
 * Return:
 *	0 = interface gone, 1 = interface exists
 */
int
check_nic(const char *interface)
{
	int		r = 0;
	char		nic[IF_NAMESIZE];
	struct ifaddrs	*ifap = NULL, *ifa = NULL;

	memset(nic, 0, sizeof(nic));

	if (getifaddrs(&ifap) != 0)
		errx(1, "getifaddrs");

	for (ifa = ifap; ifa; ifa = ifa->ifa_next) {
		if (strcmp(nic, ifa->ifa_name)) {
			if (!strcmp(ifa->ifa_name, interface)) {
				r = 1;
				break;
			}
		}
		strlcpy(nic, ifa->ifa_name, sizeof(nic));
	}

	freeifaddrs(ifap);

	return r;
}

/*
 * translate_speed()
 *	translate the result of media speed to human readable string
 * Return:
 *	<pointer to speed> = success, NULL = no data / error 
 */
char *
translate_speed(const char *mode)
{
	int	i;

	for (i = 0; speed[i] != NULL; i++) {
		if (!strcmp(mode, speed[i]))
			return speed[i + 1];
	}

	return NULL;
}
