# lprng-lib.pl
# Functions for lprng-style printer management

# list_printers()
# Returns a list of known printer names
sub list_printers
{
local($l, @rv);
foreach $l (&list_printcap()) {
	$l->{'name'} =~ /^([^\|]+)/;
	push(@rv, $1);
	}
return @rv;
}

# get_printer(name, [nostatus])
sub get_printer
{
local($l, %prn, @w, @n, $w, %cap, @jobs);
foreach $l (&list_printcap()) {
	@n = split(/\|/, $l->{'name'});
	if ($n[0] eq $_[0]) {
		# found the printer.. get info from printcap
		$prn{'name'} = $n[0];
		if (@n > 2) { $prn{'alias'} = [ @n[1..$#n-1] ]; }
		if (@n > 1) { $prn{'desc'} = $n[$#n]; }
		$prn{'desc'} = $l->{'cm'} if ($l->{'cm'});
		$prn{'iface'} = $l->{'if'};
		$prn{'banner'} = !defined($l->{'sh'});
		if ($l->{'rm'}) {
			$prn{'rhost'} = $l->{'rm'};
			$prn{'rqueue'} = $l->{'rp'};
			}
		elsif ($l->{'lp'} =~ /^(\S+)\%(\d+)$/) {
			$prn{'dhost'} = $1;
			$prn{'dport'} = $2;
			}
		else {
			$prn{'dev'} = $l->{'lp'};
			}
		$prn{'msize'} = $l->{'mx#'};

		if (!$_[1]) {
			if (!$l->{'rm'}) {
				# call lpc to get status
				$out = `lpc -P$prn{'name'} status 2>&1`;
				if ($out =~ /\n(\S+)\s+(\S+)\s+(\S+)/) {
					$prn{'accepting'} = $3 eq 'enabled';
					$prn{'enabled'} = $2 eq 'enabled';
					}
				}
			else {
				# remote printers are always accepting
				$prn{'accepting'} = 1;
				$prn{'enabled'} = 1;
				}
			}

		return \%prn;
		}
	}
return undef;
}

# get_jobs(printer)
sub get_jobs
{
local @jobs;
open(LPQ, "lpq -P$_[0] |");
while(<LPQ>) {
	chop;
	if (/^\s+Rank\s+Owner/i) { $doneheader++; }
	elsif ($doneheader && /^(\S+)\s+(\S+)\s+\S+\s+(\d+)\s+(.*\S)\s+(\d+)\s+(\S+)$/) {
		# Local print queue
		local(%job, $f, @pq);
		$job{'id'} = $3;
		$job{'user'} = $2;
		$job{'size'} = $5;
		$job{'file'} = $4;
		$job{'printing'} = ($1 eq "active");
		$job{'user'} =~ s/\+\d+$//g;
		local $d = "$config{'spool_dir'}/$_[0]";
		opendir(DIR, $d);
		while($f = readdir(DIR)) {
			if ($f =~ /df.(\d+)/ && $1 == $job{'id'}) {
				push(@pq, "$d/$f");
				}
			}
		closedir(DIR);
		$job{'printfile'} = @pq ? \@pq : undef;
		push(@jobs, \%job);
		}
	elsif (/^(\S+):\s+(\S+)\s+\[job\s+(\d+)/) {
		# remote print queue
		local(%job);
		$job{'id'} = $3;
		$job{'user'} = $1;
		if (<LPQ> =~ /^\s+(.*\S)\s+(\d+)/) {
			$job{'file'} = $1;
			$job{'size'} = $2;
			push(@jobs, \%job);
			}
		}
	}
close(LPQ);
return @jobs;
}

sub printer_support
{
return $_[0] !~ /^(why|allow|default|ctype|riface|sysv)$/;
}


# create_printer(&details)
# Create a new printer in /etc/printcap
sub create_printer
{
local(%cap);
$cap{'sd'} = "$config{'spool_dir'}/$_[0]->{'name'}";
local @st = stat($config{'spool_dir'});
&lock_file($cap{'sd'});
mkdir($cap{'sd'}, 0755);
&unlock_file($cap{'sd'});
&lock_file("$cap{'sd'}/control.$_[0]->{'name'}");
open(TOUCH, ">$cap{'sd'}/control.$_[0]->{'name'}");
close(TOUCH);
&unlock_file("$cap{'sd'}/control.$_[0]->{'name'}");
&lock_file("$cap{'sd'}/status.$_[0]->{'name'}");
open(TOUCH, ">$cap{'sd'}/status.$_[0]->{'name'}");
close(TOUCH);
&unlock_file("$cap{'sd'}/status.$_[0]->{'name'}");
&lock_file("$cap{'sd'}/unspooler.$_[0]->{'name'}");
open(TOUCH, ">$cap{'sd'}/unspooler.$_[0]->{'name'}");
close(TOUCH);
&unlock_file("$cap{'sd'}/unspooler.$_[0]->{'name'}");
&system_logged("chown -R $st[4]:$st[5] $cap{'sd'}");
&system_logged("chmod 2700 $cap{'sd'}");
&system_logged("chmod 600 $cap{'sd'}/*");
&system_logged("chmod 700 $cap{'sd'}/printfilter")
	if (-r "$cap{'sd'}/printfilter");

&lock_file($config{'printcap_file'});
open(CAP, ">> $config{'printcap_file'}");
print CAP &make_printcap($_[0], \%cap),"\n";
close(CAP);
&unlock_file($config{'printcap_file'});
&system_logged("lpc reread all >/dev/null 2>&1");
&apply_status($_[0]);
}

# modify_printer(&details)
sub modify_printer
{
local(@old, $o, $old, @cap);
&lock_file($config{'printcap_file'});
@old = &list_printcap();
foreach $o (@old) {
	$o->{'name'} =~ /^([^\|]+)/;
	if ($1 eq $_[0]->{'name'}) {
		# found current details
		$old = $o;
		last;
		}
	}
if (!$old) { &error("Printer '$_[0]->{'name'}' no longer exists"); }
open(CAP, $config{'printcap_file'});
@cap = <CAP>;
close(CAP);
splice(@cap, $old->{'line'},
       $old->{'eline'} - $old->{'line'} + 1, &make_printcap($_[0], $old)."\n");
open(CAP, "> $config{'printcap_file'}");
print CAP @cap;
close(CAP);
&unlock_file($config{'printcap_file'});
&system_logged("lpc reread all >/dev/null 2>&1");
&apply_status($_[0]);
}

# delete_printer(name)
sub delete_printer
{
local(@old, $o, $old, @cap);
&lock_file($config{'printcap_file'});
@old = &list_printcap();
foreach $o (@old) {
	$o->{'name'} =~ /^([^\|]+)/;
	if ($1 eq $_[0]) {
		# found current details
		$old = $o;
		last;
		}
	}
if (!$old) { &error("Printer '$_[0]' no longer exists"); }
open(CAP, $config{'printcap_file'});
@cap = <CAP>;
close(CAP);
splice(@cap, $old->{'line'}, $old->{'eline'} - $old->{'line'} + 1);
open(CAP, ">$config{'printcap_file'}");
print CAP @cap;
close(CAP);
&unlock_file($config{'printcap_file'});
if ($old->{'sd'} =~ /$_[0]$/) {
	&system_logged("rm -rf $old->{'sd'}");
	}
&system_logged("lpc reread all >/dev/null 2>&1");
}

# cancel_job(printer, job)
# Calls lprm to remove some job
sub cancel_job
{
local($out);
$out = &backquote_logged("lprm -P$_[0] $_[1] 2>&1");
if ($?) { &error("lprm failed : $out"); }
}

# make_printcap(&details, &old)
# Updates or creates a printcap line
sub make_printcap
{
local(%prn, %cap, $a, $rv, $c);
%prn = %{$_[0]}; %cap = %{$_[1]};
$cap{'if'} = $prn{'iface'} ? $prn{'iface'} : undef;
$cap{'sh'} = $prn{'banner'} ? undef : "";
$cap{'lp'} = $prn{'dhost'} ? "$prn{'dhost'}%$prn{'dport'}" :
	     $prn{'dev'} ? $prn{'dev'} : undef;
$cap{'rm'} = $prn{'rhost'} ? $prn{'rhost'} : undef;
$cap{'rp'} = $prn{'rqueue'} ? $prn{'rqueue'} : undef;
$cap{'mx#'} = $prn{'msize'} ? $prn{'msize'} : undef;
$cap{'cm'} = $prn{'desc'} ? $prn{'desc'} : undef;
$rv = $prn{'name'};
foreach $a (@{$prn{'alias'}}) { $rv .= "|$a"; }
$rv .= "|$prn{'desc'}";
foreach $c (keys %cap) {
	if ($c =~ /^(\S\S)(#?)$/ && defined($cap{$c})) {
		if ($cap{$c} eq "") { $rv .= ":$c"; }
		elsif ($2 eq "#") { $rv .= ":$c$cap{$c}"; }
		else { $rv .= ":$c=$cap{$c}"; }
		}
	}
$rv .= ":";
return $rv;
}

# list_printcap()
# Returns an array of associative arrays containing printcap fields
sub list_printcap
{
local(@rv, @line, $line, $cont, $lnum, $i);
open(CAP, $config{'printcap_file'});
$lnum = 0;
while($line = <CAP>) {
	$line =~ s/^#.*$//g;	# remove comments
	$line =~ s/\s+$//g;	# remove trailing spaces
	if ($line =~ /\S/) {
		$ncont = ($line =~ s/\\$//g);
		if ($cont || $line =~ /^\s+/) {
			$line[$#line] .= $line;
			$eline[@line - 1] = $lnum;
			}
		else {
			push(@line, $line);
			$eline[@line - 1] = $sline[@line - 1] = $lnum;
			}
		$cont = $ncont;
		}
	$lnum++;
	}
close(CAP);
for($i=0; $i<@line; $i++) {
	local(%cap);
	@w = split(/:+/, $line[$i]);
	$cap{'name'} = $w[0];
	$cap{'line'} = $sline[$i];
	$cap{'eline'} = $eline[$i];
	foreach $w (@w[1..$#w]) {
		if ($w =~ /^([A-z0-9]+)=(.*)$/) { $cap{$1} = $2; }
		elsif ($w =~ /^([A-z0-9]+)#(.*)$/) { $cap{$1."#"} = $2; }
		elsif ($w =~ /^([A-z0-9]+)$/) { $cap{$w} = ""; }
		}
	push(@rv, \%cap);
	}
return @rv;
}

# apply_status(&details)
# Calls lpc to enable or disable a printer.
# Restarting lpd doesn't seem to be necessary?
sub apply_status
{
local $out;
if ($prn{'enabled'}) {
	$out = &backquote_logged("lpc -P$prn{'name'} start 2>&1");
	}
else {
	$out = &backquote_logged("lpc -P$prn{'name'} stop 2>&1");
	}
if ($prn{'accepting'}) {
	$out = &backquote_logged("lpc -P$prn{'name'} enable 2>&1");
	}
else {
	$out = &backquote_logged("lpc -P$prn{'name'} disable 2>&1");
	}
}

# sched_running()
# Returns the pid if lpsched is running, 0 if not, -1 if cannot be stopped
sub sched_running
{
local $out = `lpc lpd all 2>&1`;
return $out =~ /pid\s+(\d+)/ ? $1 : 0;
}

# start_sched()
# Start lpsched
sub start_sched
{
local $out = &backquote_logged("lpd 2>&1");
if ($? || $out =~ /error/i) { &error("<tt>$out</tt>"); }
}

# stop_sched(pid)
# Stop the running lpsched process
sub stop_sched
{
&kill_logged('INT', $_[0]) || &error(&text("estop", $!));
}

if (-r "/dev/lp0") {
	@device_files = ("/dev/lp0", "/dev/lp1",
			 "/dev/ttyS0", "/dev/ttyS1", "/dev/null");
	}
else {
	@device_files = ("/dev/lp1", "/dev/lp2",
			 "/dev/ttyS0", "/dev/ttyS1", "/dev/null");
	}
@device_names = ("Parallel port 1", "Parallel port 2",
		 "Serial Port 1 (COM1)", "Serial Port 2 (COM2)",
		 "Null Device");

