# read_zone_file(file, origin)
# Reads a DNS zone file and returns a data structure of records. The origin
# must be a domain without the trailing dot, or just .
sub read_zone_file
{
local($file, $lnum, $line, $t, @tok, @lnum,
      $i, @rv, $origin, $num, $j, @inc, @oset);
$origin = $_[1];
$file = &absolute_path($_[0]);
open(FILE, $config{'chroot'}.$file);
$lnum = 0;
while($line = <FILE>) {
	# strip comments
	$line =~ s/\r|\n//g;
	$line =~ s/;.*$//g;
	$line =~ s/#.*$//g;

	# split line into tokens
	local $oset = 0;
	while(1) {
		if ($line =~ /^(\s*)\"([^"]+)"(.*)/ ||
		    $line =~ /^(\s*)([A-z0-9\.\-_\$\/\*]+)(.*)/ ||
		    $line =~ /^(\s*)([^A-z0-9\.\-_\$\/\*\s])(.*)/) {
			$oset += length($1);
			push(@tok, $2); push(@lnum, $lnum); push(@oset, $oset);
			$line = $3; $oset += length($2);
			}
		else { last; }
		}
	$lnum++;
	}
close(FILE);

# parse into data structures
$i = 0; $num = 0;
while($i < @tok) {
	if ($tok[$i] =~ /^\$origin$/i) {
		# $ORIGIN directive (may be relative or absolute)
		if ($tok[$i+1] =~ /^(\S+)\.$/) {
			$origin = $1 ? $1 : ".";
			}
		elsif ($origin eq ".") { $origin = $tok[$i+1]; }
		else { $origin = "$tok[$i+1].$origin"; }
		$i += 2;
		}
	elsif ($tok[$i] =~ /^\$include$/i) {
		# including another file
		if ($lnum[$i+1] == $lnum[$i+2]) {
			# $INCLUDE zonefile origin
			local $inc_origin;
			if ($tok[$i+2] =~ /^(\S+)\.$/) {
				$inc_origin = $1 ? $1 : ".";
				}
			elsif ($origin eq ".") { $inc_origin = $tok[$i+2]; }
			else { $inc_origin = "$tok[$i+2].$origin"; }
			@inc = &read_zone_file($tok[$i+1], $inc_origin);
			$i += 3;
			}
		else {
			# $INCLUDE zonefile
			@inc = &read_zone_file($tok[$i+1], $origin);
			$i += 2;
			}
		foreach $j (@inc) { $j->{'num'} = $num++; }
		push(@rv, @inc);
		}
	elsif ($tok[$i] =~ /^\$(\S+)/i) {
		# some other special directive
		local $ln = $lnum[$i];
		while($lnum[$i] == $ln) {
			$i++;
			}
		}
	else {
		# some other directive
		local(%dir, @values, $l);
		$dir{'line'} = $lnum[$i];
		$dir{'file'} = $file;
		if ($tok[$i] =~ /^(in|hs)$/i) {
			# starting with a class
			$dir{'class'} = uc($tok[$i]);
			$i++;
			}
		elsif ($tok[$i] =~ /^\d/ && $tok[$i] !~ /in-addr/i &&
		       $oset[$i] > 0) {
			# starting with a TTL and class
			$dir{'ttl'} = $tok[$i];
			$dir{'class'} = uc($tok[$i+1]);
			$i += 2;
			}
		elsif ($tok[$i+1] =~ /^(in|hs)$/i) {
			# starting with a name and class
			$dir{'name'} = $tok[$i];
			$dir{'class'} = uc($tok[$i+1]);
			$i += 2;
			}
		elsif ($oset[$i] > 0) {
			# starting with nothing
			$dir{'class'} = "IN";
			}
		elsif ($tok[$i+1] =~ /^\d/) {
			# starting with a name, ttl and class
			$dir{'name'} = $tok[$i];
			$dir{'ttl'} = $tok[$i+1];
			$dir{'class'} = uc($tok[$i+2]);
			$i += 3;
			}
		else {
			# starting with a name
			$dir{'name'} = $tok[$i];
			$dir{'class'} = "IN";
			$i++;
			}
		if ($dir{'name'} eq '') {
			$#rv >= 0 ||
				&error(&text('efirst', $lnum[$i]+1, $file));
			$dir{'name'} = $rv[$#rv]->{'name'};
			}
		$dir{'type'} = uc($tok[$i++]);

		# read values until end of line (unless a ( is found)
		$l = $lnum[$i];
		while($lnum[$i] == $l && $i < @tok) {
			if ($tok[$i] eq "(") {
				while($tok[++$i] ne ")") {
					push(@values, $tok[$i]);
					}
				$i++; # skip )
				last;
				}
			push(@values, $tok[$i++]);
			}
		$dir{'values'} = \@values;
		$dir{'eline'} = $lnum[$i-1];
		if ($dir{'name'} eq "@") {
			$dir{'name'} = $origin eq "." ? "." : "$origin.";;
			}
		elsif ($dir{'name'} !~ /\.$/) {
			$dir{'name'} .= $origin eq "." ? "." : ".$origin.";
			}
		$dir{'num'} = $num++;
		push(@rv, \%dir);
		}
	}
return @rv;
}

# create_record(file, name, ttl, class, type, values)
# Add a new record of some type to some zone file
sub create_record
{
open(ZONE, ">>".$config{'chroot'}.&absolute_path($_[0]));
print ZONE &make_record(@_[1..$#_]);
close(ZONE);
}

# modify_record(file, old, name, ttl, class, type, values)
# Updates an existing record in some zone file
sub modify_record
{
local(@zone, $lines, $file);
$file = $config{'chroot'}.&absolute_path($_[0]);
open(ZONE, $file);
@zone = <ZONE>;
close(ZONE);
$lines = $_[1]->{'eline'} - $_[1]->{'line'} + 1;
splice(@zone, $_[1]->{'line'}, $lines, &make_record(@_[2..$#_]));
open(ZONE, "> $file");
print ZONE @zone;
close(ZONE);
}

# delete_record(file, old)
# Deletes a record in some zone file
sub delete_record
{
local(@zone, $lines, $file);
$file = $config{'chroot'}.&absolute_path($_[0]);
open(ZONE, $file);
@zone = <ZONE>;
close(ZONE);
$lines = $_[1]->{'eline'} - $_[1]->{'line'} + 1;
splice(@zone, $_[1]->{'line'}, $lines);
open(ZONE, "> $file");
print ZONE @zone;
close(ZONE);
}

# make_record(name, ttl, class, type, values)
# Returns a string for some zone record
sub make_record
{
return $_[0] . ($_[1] ? "\t$_[1]" : "") . "\t$_[2]\t$_[3]\t$_[4]\n";
}

# bump_soa_record(file, &records)
# Increase the serial number in some SOA record by 1
sub bump_soa_record
{
local($i, $r, $v, $vals);
for($i=0; $i<@{$_[1]}; $i++) {
	$r = $_[1]->[$i];
	if ($r->{'type'} eq "SOA") {
		$v = $r->{'values'};
		if ($config{'soa_style'} == 1 && $v->[2] =~ /^(\d{8})(\d\d)$/) {
			if ($1 eq &date_serial())
				{ $serial = sprintf "%d%2.2d", $1, $2+1; }
			else { $serial = &date_serial()."00"; }
			}
		else { $serial = $v->[2]+1; }
		$vals = "$v->[0] $v->[1] (\n\t\t\t$serial\n\t\t\t$v->[3]\n".
			"\t\t\t$v->[4]\n\t\t\t$v->[5]\n\t\t\t$v->[6] )";
		&modify_record($_[0], $r, $r->{'name'}, $r->{'ttl'},
				$r->{'class'}, $r->{'type'}, $vals);
		}
	}
}



# date_serial()
# Returns a string like YYYYMMDD
sub date_serial
{
local $now = time();
local @tm = localtime($now);
return sprintf "%4.4d%2.2d%2.2d", $tm[5]+1900, $tm[4]+1, $tm[3];
}

# get_zone_defaults(&array)
sub get_zone_defaults
{
if (!&read_file("$module_config_directory/zonedef", $_[0])) {
	$_[0]->{'refresh'} = 10800; $_[0]->{'retry'} = 3600;
	$_[0]->{'expiry'} = 432000; $_[0]->{'minimum'} = 38400;
	}
}

# save_zone_defaults(&array)
sub save_zone_defaults
{
&write_file("$module_config_directory/zonedef", $_[0]);
}

# allowed_zone_file(&access, file)
sub allowed_zone_file
{
return 0 if ($_[1] =~ /\.\./);
return 0 if (-l $_[1] && !&allowed_zone_file($_[0], readlink($_[1])));
local $l = length($_[0]->{'dir'});
return length($_[1]) > $l && substr($_[1], 0, $l) eq $_[0]->{'dir'};
}

# sort_records(list)
sub sort_records
{
return @_ if (!@_ || !$config{'records_order'});
if ($config{'records_order'} == 1) {
	# Sort by name
	if ($_[0]->{'type'} eq "PTR") {
		return sort ptr_sort_func @_;
		}
	else {
		return sort { $a->{'name'} cmp $b->{'name'} } @_;
		}
	}
else {
	# Sort by value
	if ($_[0]->{'type'} eq "A") {
		return sort ip_sort_func @_;
		}
	elsif ($_[0]->{'type'} eq "MX") {
		return sort { $a->{'values'}->[1] cmp $b->{'values'}->[1] } @_;
		}
	else {
		return sort { $a->{'values'}->[0] cmp $b->{'values'}->[0] } @_;
		}
	}
}

sub ptr_sort_func
{
$a->{'name'} =~ /^(\d+)\.(\d+)\.(\d+)\.(\d+)/;
local ($a1, $a2, $a3, $a4) = ($1, $2, $3, $4);
$b->{'name'} =~ /^(\d+)\.(\d+)\.(\d+)\.(\d+)/;
return	$a4 < $4 ? -1 :
	$a4 > $4 ? 1 :
	$a3 < $3 ? -1 :
	$a3 > $3 ? 1 :
	$a2 < $2 ? -1 :
	$a2 > $2 ? 1 :
	$a1 < $1 ? -1 :
	$a1 > $1 ? 1 : 0;
}

sub ip_sort_func
{
$a->{'values'}->[0] =~ /^(\d+)\.(\d+)\.(\d+)\.(\d+)/;
local ($a1, $a2, $a3, $a4) = ($1, $2, $3, $4);
$b->{'values'}->[0] =~ /^(\d+)\.(\d+)\.(\d+)\.(\d+)/;
return	$a1 < $1 ? -1 :
	$a1 > $1 ? 1 :
	$a2 < $2 ? -1 :
	$a2 > $2 ? 1 :
	$a3 < $3 ? -1 :
	$a3 > $3 ? 1 :
	$a4 < $4 ? -1 :
	$a4 > $4 ? 1 : 0;
}

# arpa_to_ip(name)
# Converts an address like 4.3.2.1.in-addr.arpa. to 1.2.3.4
sub arpa_to_ip
{
if ($_[0] =~ /^([\d\-\.]+)\.in-addr\.arpa/i) {
	return join('.',reverse(split(/\./, $1)));
	}
return $_[0];
}

# ip_to_arpa(address)
# Converts an IP address like 1.2.3.4 to 4.3.2.1.in-addr.arpa.
sub ip_to_arpa
{
if ($_[0] =~ /^([\d\-\.]+)$/) {
	return join('.',reverse(split(/\./,$1))).".in-addr.arpa.";
	}
return $_[0];
}

# absolute_path(path)
# If a path does not start with a /, prepend the base directory
sub absolute_path
{
if ($_[0] =~ /^\//) { return $_[0]; }
return &base_directory()."/".$_[0];
}



