# user-lib.pl
# Common functions for Unix user management

do '../web-lib.pl';
&init_config();
do "./$gconfig{'os_type'}-lib.pl";

# password_file(file)
# Returns true if some file looks like a valid Unix password file
sub password_file
{
if (!$_[0]) { return 0; }
elsif (open(SHTEST, $_[0])) {
	local($line);
	$line = <SHTEST>;
	close(SHTEST);
	return $line =~ /^\S+:\S*:/;
	}
else { return 0; }
}

# list_users()
# Returns an array of hashtable, each containing info about one user. Each hash
# will always contain the keys
#  user, pass, uid, gid, real, home, shell
# In addition, if the system supports shadow passwords it may also have:
#  change, min, max, warn, inactive, expire
# Or if it supports FreeBSD master.passwd info, it will also have
#  class, change, expire
sub list_users
{
# read the password file
local(@rv, $_, %idx, $lnum, @pw, $p, $i, $j);
if (&passfiles_type() == 1) {
	# read the master.passwd file only
	$lnum = 0;
	open(PASSWD, $config{'master_file'});
	while(<PASSWD>) {
		s/\r|\n//g;
		if (/\S/ && !/^[#\+\-]/) {
			@pw = split(/:/, $_, -1);
			push(@rv, { 'user' => $pw[0],	'pass' => $pw[1],
				    'uid' => $pw[2],	'gid' => $pw[3],
				    'class' => $pw[4],	'change' => $pw[5],
				    'expire' => $pw[6],	'real' => $pw[7],
				    'home' => $pw[8],	'shell' => $pw[9],
				    'line' => $lnum,	'num' => scalar(@rv) });
			}
		$lnum++;
		}
	close(PASSWD);
	}
else {
	# start by reading /etc/passwd
	$lnum = 0;
	open(PASSWD, $config{'passwd_file'});
	while(<PASSWD>) {
		s/\r|\n//g;
		if (/\S/ && !/^[#\+\-]/) {
			@pw = split(/:/, $_, -1);
			push(@rv, { 'user' => $pw[0],	'pass' => $pw[1],
				    'uid' => $pw[2],	'gid' => $pw[3],
				    'real' => $pw[4],	'home' => $pw[5],
				    'shell' => $pw[6],	'line' => $lnum,
				    'num' => scalar(@rv) });
			$idx{$pw[0]} = $rv[$#rv];
			}
		$lnum++;
		}
	close(PASSWD);
	if (&passfiles_type() == 2) {
		# read the shadow file data
		$lnum = 0;
		open(SHADOW, $config{'shadow_file'});
		while(<SHADOW>) {
			s/\r|\n//g;
			if (/\S/ && !/^[#\+\-]/) {
				@pw = split(/:/, $_, -1);
				$p = $idx{$pw[0]};
				$p->{'pass'} = $pw[1];
				$p->{'change'} = $pw[2] < 0 ? "" : $pw[2];
				$p->{'min'} = $pw[3] < 0 ? "" : $pw[3];
				$p->{'max'} = $pw[4] < 0 ? "" : $pw[4];
				$p->{'warn'} = $pw[5] < 0 ? "" : $pw[5];
				$p->{'inactive'} = $pw[6] < 0 ? "" : $pw[6];
				$p->{'expire'} = $pw[7] < 0 ? "" : $pw[7];
				$p->{'sline'} = $lnum;
				}
			$lnum++;
			}
		close(SHADOW);
		for($i=0; $i<@rv; $i++) {
			if (!defined($rv[$i]->{'sline'})) {
				# not in shadow!
				for($j=$i; $j<@rv; $j++) { $rv[$j]->{'num'}--; }
				splice(@rv, $i--, 1);
				}
			}
		}
	}
return @rv;
}

# create_user(&details)
# Creates a new user with the given details
sub create_user
{
local(%u) = %{$_[0]};
local $lref;
if (&passfiles_type() == 1) {
	# just need to add to master.passwd
	$lref = &read_file_lines($config{'master_file'});
	splice(@$lref, &nis_index($lref), 0,
	       "$u{'user'}:$u{'pass'}:$u{'uid'}:$u{'gid'}:$u{'class'}:".
	       "$u{'change'}:$u{'expire'}:$u{'real'}:$u{'home'}:$u{'shell'}");
	}
else {
	# add to /etc/passwd
	$lref = &read_file_lines($config{'passwd_file'});
	splice(@$lref, &nis_index($lref), 0,
	       "$u{'user'}:".(&passfiles_type() == 2 ? "x" : $u{'pass'}).
	       ":$u{'uid'}:$u{'gid'}:$u{'real'}:$u{'home'}:$u{'shell'}");
	if (&passfiles_type() == 2) {
		# add to shadow as well..
		$lref = &read_file_lines($config{'shadow_file'});
		splice(@$lref, &nis_index($lref), 0,
		       "$u{'user'}:$u{'pass'}:$u{'change'}:$u{'min'}:".
		       "$u{'max'}:$u{'warn'}:$u{'inactive'}:$u{'expire'}:");
		}
	}
&flush_file_lines();
}

# modify_user(&old, &details)
sub modify_user
{
local(%u) = %{$_[1]};
local(@passwd, @shadow);
if (&passfiles_type() == 1) {
	# just need to update master.passwd
	&replace_file_line($config{'master_file'}, $_[0]->{'line'},
	      "$u{'user'}:$u{'pass'}:$u{'uid'}:$u{'gid'}:$u{'class'}:".
	      "$u{'change'}:$u{'expire'}:$u{'real'}:$u{'home'}:$u{'shell'}\n");
	}
else {
	# update /etc/passwd
	&replace_file_line($config{'passwd_file'}, $_[0]->{'line'},
		"$u{'user'}:".(&passfiles_type() == 2 ? "x" : $u{'pass'}).
		":$u{'uid'}:$u{'gid'}:$u{'real'}:$u{'home'}:$u{'shell'}\n");
	if (&passfiles_type() == 2) {
		# update shadow file as well..
		&replace_file_line($config{'shadow_file'}, $_[0]->{'sline'},
			"$u{'user'}:$u{'pass'}:$u{'change'}:$u{'min'}:".
			"$u{'max'}:$u{'warn'}:$u{'inactive'}:$u{'expire'}:\n");
		}
	}
}

# delete_user(&details)
sub delete_user
{
if (&passfiles_type() == 1) {
	&replace_file_line($config{'master_file'}, $_[0]->{'line'});
	}
else {
	&replace_file_line($config{'passwd_file'}, $_[0]->{'line'});
	if (&passfiles_type() == 2) {
		&replace_file_line($config{'shadow_file'}, $_[0]->{'sline'});
		}
	}
}

# list_groups()
# Returns a list of all the local groups as an array of hashtables. Each
# will contain group, pass, gid, members
sub list_groups
{
local(@rv, $lnum, $_, %idx, $g, $i, $j);
$lnum = 0;
open(GROUP, $config{'group_file'});
while(<GROUP>) {
	s/\r|\n//g;
	if (/\S/ && !/^[#\+\-]/) {
		@gr = split(/:/, $_, -1);
		push(@rv, { 'group' => $gr[0],	'pass' => $gr[1],
			    'gid' => $gr[2],	'members' => $gr[3],
			    'line' => $lnum,	'num' => scalar(@rv) });
		$idx{$gr[0]} = $rv[$#rv];
		}
	$lnum++;
	}
close(GROUP);
if (&groupfiles_type() == 2) {
	# read the gshadow file data
	$lnum = 0;
	open(SHADOW, $config{'gshadow_file'});
	while(<SHADOW>) {
		s/\r|\n//g;
		if (/\S/ && !/^[#\+\-]/) {
			@gr = split(/:/, $_, -1);
			$g = $idx{$gr[0]};
			$g->{'pass'} = $gr[1];
			$g->{'sline'} = $lnum;
			}
		$lnum++;
		}
	close(SHADOW);
	for($i=0; $i<@rv; $i++) {
		if (!defined($rv[$i]->{'sline'})) {
			# not in shadow!
			for($j=$i; $j<@rv; $j++) { $rv[$j]->{'num'}--; }
			splice(@rv, $i--, 1);
			}
		}
	}
return @rv;
}

# create_group(&details)
sub create_group
{
local(%g) = %{$_[0]};
local $lref;
$lref = &read_file_lines($config{'group_file'});
splice(@$lref, &nis_index($lref), 0,
       "$g{'group'}:".(&groupfiles_type() == 2 ? "x" : $g{'pass'}).
       ":$g{'gid'}:$g{'members'}");
close(GROUP);
if (&groupfiles_type() == 2) {
	$lref = &read_file_lines($config{'gshadow_file'});
	splice(@$lref, &nis_index($lref), 0,
	       "$g{'group'}:$g{'pass'}::$g{'members'}");
	}
&flush_file_lines();
}

# modify_group(&old, &details)
sub modify_group
{
local(%g) = %{$_[1]};
&replace_file_line($config{'group_file'}, $_[0]->{'line'},
		   "$g{'group'}:".(&groupfiles_type() == 2 ? "x" : $g{'pass'}).
		   ":$g{'gid'}:$g{'members'}\n");
if (&groupfiles_type() == 2) {
	&replace_file_line($config{'gshadow_file'}, $_[0]->{'sline'},
			   "$g{'group'}:$g{'pass'}::$g{'members'}\n");
	}
}

# delete_group(&details)
sub delete_group
{
&replace_file_line($config{'group_file'}, $_[0]->{'line'});
if (&groupfiles_type() == 2) {
	&replace_file_line($config{'gshadow_file'}, $_[0]->{'sline'});
	}
}


############################################################################
# Misc functions
############################################################################
# recursive_change(dir, olduid, oldgid, newuid, newgid)
# Change the UID or GID of a directory and all files in it, if they match the
# given UID/GID
sub recursive_change
{
local(@list, $f, @stbuf);
(@stbuf = stat($_[0])) || return;
(-l $_[0]) && return;
if (($_[1] < 0 || $_[1] == $stbuf[4]) &&
    ($_[2] < 0 || $_[2] == $stbuf[5])) {
	# Found match..
	chown($_[3] < 0 ? $stbuf[4] : $_[3],
	      $_[4] < 0 ? $stbuf[5] : $_[4], $_[0]);
	}
if (-d $_[0]) {
	opendir(DIR, $_[0]);
	@list = readdir(DIR);
	closedir(DIR);
	foreach $f (@list) {
		if ($f eq "." || $f eq "..") { next; }
		&recursive_change("$_[0]/$f", $_[1], $_[2], $_[3], $_[4]);
		}
	}
}

# making_changes()
# Called before changes are made to the password or group file
sub making_changes
{
if ($config{'pre_command'} =~ /\S/) {
	`$config{'pre_command'} 2>&1`;
	}
}

# made_changes()
# Called after the password or group file has been changed, to run the
# post-changes command.
sub made_changes
{
if ($config{'post_command'} =~ /\S/) {
	system("$config{'post_command'} </dev/null >/dev/null 2>&1");
	}
}

# other_modules(function, &user)
# Call some function in the useradmin_update.pl file in other modules
sub other_modules
{
local($m, %minfo);
local $func = $_[0];
opendir(DIR, "..");
foreach $m (readdir(DIR)) {
	if ($m !~ /^\./ && -r "../$m/useradmin_update.pl" &&
	    (%minfo = &get_module_info($m)) &&
	    &check_os_support(\%minfo)) {
		&foreign_require($m, "useradmin_update.pl");
		&foreign_call($m, $func, $_[1]);
		}
	}
}

# can_edit_user(&acl, &user)
sub can_edit_user
{
local $m = $_[0]->{'uedit_mode'};
local %u;
if ($m == 0) { return 1; }
elsif ($m == 1) { return 0; }
elsif ($m == 2 || $m == 3) {
	map { $u{$_}++ } split(/\s+/, $_[0]->{'uedit'});
	return $m == 2 ? $u{$_[1]->{'user'}}
		       : !$u{$_[1]->{'user'}};
	}
elsif ($m == 4) {
	return (!$_[0]->{'uedit'} || $_[1]->{'uid'} >= $_[0]->{'uedit'}) &&
	       (!$_[0]->{'uedit2'} || $_[1]->{'uid'} <= $_[0]->{'uedit2'});
	}
else { $_[1]->{'gid'} == $_[0]->{'uedit'}; }
}

# can_edit_group(&acl, &group)
sub can_edit_group
{
local $m = $_[0]->{'gedit_mode'};
local %g;
if ($m == 0) { return 1; }
elsif ($m == 1) { return 0; }
elsif ($m == 2 || $m == 3) {
	map { $g{$_}++ } split(/\s+/, $_[0]->{'gedit'});
	return $m == 2 ? $g{$_[1]->{'group'}}
		       : !$g{$_[1]->{'group'}};
	}
else { return (!$_[0]->{'gedit'} || $_[1]->{'gid'} >= $_[0]->{'gedit'}) &&
	      (!$_[0]->{'gedit2'} || $_[1]->{'gid'} <= $_[0]->{'gedit2'}); }
}

# nis_index(&lines)
sub nis_index
{
local $i;
for($i=0; $i<@{$_[0]}; $i++) {
	last if ($_[0]->[$i] =~ /^[\+\-]/);
	}
return $i;
}

# copy_skel_files(source, dest, uid, gid)
sub copy_skel_files
{
local ($f, $df);
foreach $f (split(/\s+/, $_[0])) {
	if (-d $f) {
		# copy all files in a directory
		opendir(DIR, $f);
		foreach $df (readdir(DIR)) {
			if ($df eq "." || $df eq "..") { next; }
			&copy_file("$f/$df", $_[1], $_[2], $_[3]);
			}
		closedir(DIR);
		}
	elsif (-r $f) {
		# copy just one file
		&copy_file($f, $_[1], $_[2], $_[3]);
		}
	}
}

# copy_file(file, destdir, uid, gid)
# Copy a file or directory and chown it
sub copy_file
{
local($base, $subs);
$_[0] =~ /\/([^\/]+)$/; $base = $1;
$subs = $config{'files_remove'};
$base =~ s/$subs//g;
system("cp -r $_[0] $_[1]/$base >/dev/null 2>/dev/null");
system("chown -R $_[2]:$_[3] $_[1]/$base >/dev/null 2>/dev/null");
}


1;
