# web-lib.pl
# Common functions and definitions for web admin programs

# Vital libraries
use Socket;

# Configuration and spool directories
if (!defined($ENV{'WEBMIN_CONFIG'})) { die "WEBMIN_CONFIG not set"; }
$config_directory = $ENV{'WEBMIN_CONFIG'};
if (!defined($ENV{'WEBMIN_VAR'})) { die "WEBMIN_VAR not set"; }
$var_directory = $ENV{'WEBMIN_VAR'};

# read_file(file, &assoc, [&order])
# Fill an associative array with name=value pairs from a file
sub read_file
{
open(ARFILE, $_[0]) || return 0;
while(<ARFILE>) {
	s/\r|\n//g;
        if (!/^#/ && /^([^=]+)=(.*)$/) {
		$_[1]->{$1} = $2;
		push(@{$_[2]}, $1);
        	}
        }
close(ARFILE);
return 1;
}
 
# write_file(file, array)
# Write out the contents of an associative array as name=value lines
sub write_file
{
local($arr);
$arr = $_[1];
open(ARFILE, "> $_[0]");
foreach $k (keys %$arr) {
        print ARFILE "$k=$$arr{$k}\n";
        }
close(ARFILE);
}

# html_escape
# Convert &, < and > codes in text to HTML entities
sub html_escape
{
local($tmp);
$tmp = $_[0];
$tmp =~ s/&/&amp;/g;
$tmp =~ s/</&lt;/g;
$tmp =~ s/>/&gt;/g;
$tmp =~ s/\"/&#34;/g;
return $tmp;
}

# obtain_lock
# Get a lock on a file, or wait.
# If the file is locked, and the locking process no longer exists ignore the
# lock and carry on.
sub obtain_lock {
  while(stat("$_[0].lock")) {
	sleep(1);
	open(LOCK, "$_[0].lock");
	chop($lpid = <LOCK>);
	close(LOCK);
	if ($lpid && !kill(0, "/proc/$lpid")) {
		# The process holding this lock is gone!
		# print STDERR "Process $lpid does not exist\n";
		last;
		}
	}
  open(LOCK,">$_[0].lock");
  print LOCK "$$\n";
  close(LOCK);
}

# release_lock
# Release the lock we are holding on a file
sub release_lock {
  unlink("$_[0].lock");
}

# test_lock
# Is this file locked?
sub test_lock
{
if (-r "$_[0].lock") {
	open(LOCK, "$_[0].lock");
	<LOCK> =~ /(\d+)/;
	close(LOCK);
	if ($1 && kill(0, $1)) { return 1; }
	}
return 0;
}

# tempname
# Returns a mostly random temporary file name
sub tempname
{
$tempfilecount++;
return "/tmp/".$$."_".$tempfilecount."_".$scriptname;
}

# trunc
# Truncation a string to the shortest whole word less than the given width
sub trunc {
  local($str,$c);
  if (length($_[0]) <= $_[1])
    { return $_[0]; }
  $str = substr($_[0],0,$_[1]);
  do {
    $c = chop($str);
    } while($c !~ /\S/);
  return $str;
}

# indexof
# Returns the index of some value in an array, or -1
sub indexof {
  local($i);
  for($i=1; $i <= $#_; $i++) {
    if ($_[$i] eq $_[0]) { return $i - 1; }
    }
  return -1;
  }

# unique
# Returns the unique elements of some array
sub unique
{
local(%found, @rv, $e);
foreach $e (@_) {
	if (!$found{$e}++) { push(@rv, $e); }
	}
return @rv;
}

# sysprint(handle, [string]+)
sub sysprint
{
local($str, $fh);
$str = join('', @_[1..$#_]);
$fh = $_[0];
syswrite $fh, $str, length($str);
}

# check_ipaddress(ip)
# Check if some IP address is properly formatted
sub check_ipaddress
{
return $_[0] =~ /^(\d+)\.(\d+)\.(\d+)\.(\d+)$/ &&
	$1 >= 0 && $1 <= 255 &&
	$2 >= 0 && $2 <= 255 &&
	$3 >= 0 && $3 <= 255 &&
	$4 >= 0 && $4 <= 255;
}

# generate_icon(image, title, link)
sub generate_icon
{
if ($_[2]) {
	print "<table border><tr><td>\n",
	      "<a href=\"$_[2]\"><img src=\"$_[0]\" border=0 ",
	      "width=48 height=48></a></td></tr></table>\n";
	print "<a href=\"$_[2]\">$_[1]</a>\n";
	}
else {
	print "<table border><tr><td>\n",
	      "<img src=\"$_[0]\" border=0 width=48 height=48>",
	      "</td></tr></table>\n$_[1]\n";
	}
}

# urlize
# Convert a string to a form ok for putting in a URL
sub urlize {
  local($tmp, $tmp2, $c);
  $tmp = $_[0];
  $tmp2 = "";
  while(($c = chop($tmp)) ne "") {
	if ($c !~ /[A-z0-9]/) {
		$c = sprintf("%%%2.2X", ord($c));
		}
	$tmp2 = $c . $tmp2;
	}
  return $tmp2;
}

# include
# Read and output the named file
sub include
{
open(INCLUDE, $_[0]);
while(<INCLUDE>) {
	print;
	}
close(INCLUDE);
}

# copydata
# Read from one file handle and write to another
sub copydata
{
local($line, $out, $in);
$out = $_[1];
$in = $_[0];
while($line = <$in>) {
	print $out $line;
	}
}

# ReadParseMime
# Read data submitted via a POST request using the multipart/form-data coding
sub ReadParseMime
{
local($boundary,$line,$foo);
$ENV{CONTENT_TYPE} =~ /boundary=(.*)$/;
$boundary = $1;
<STDIN>;	# skip first boundary
while(1) {
	$name = "";
	# Read section headers
	while(1) {
		$line = <STDIN>;
		chop($line); chop($line);	# lose /r/n
		if (!$line) { last; }
		elsif ($line =~ /^Content-Disposition: form-data(.*)/) {
			$rest = $1;
			while ($rest =~ /([a-zA-Z]*)=\"([^\"]*)\"(.*)/) {
				if ($1 eq name) {
					$name = $2;
					}
				else {
					$foo = $name . "_$1";
					$in{$foo} = $2;
					}
				$rest = $3;
				}
			}
		elsif ($line =~ /^Content-Type: (.*)/) {
			$foo = $name . "_content_type";
			$in{$foo} = $1;
			}
		}
	# Read data
	$in{$name} .= "\0" if ($in{$name});
	while(1) {
		$line = <STDIN>;
		if (!$line) { return; }
		if (index($line,"$boundary") != -1) { last; }
		$in{$name} .= $line;
		}
	chop($in{$name}); chop($in{$name});
	if (index($line,"$boundary--") != -1) { last; }
	}
}

# ReadParse([&assoc])
# Fills the given associative array with CGI parameters, or uses the global
# %in if none is given. Also sets the global variables $in and @in
sub ReadParse
{
local $a = $_[0] ? $_[0] : \%in;
local $i;
if ($ENV{'REQUEST_METHOD'} eq 'POST') {
	read(STDIN, $in, $ENV{'CONTENT_LENGTH'});
	}
else {
	$in = $ENV{'QUERY_STRING'};
	}
@in = split(/\&/, $in);
foreach $i (@in) {
	local ($k, $v) = split(/=/, $i, 2);
	$k =~ s/\+/ /g; $k =~ s/%(..)/pack("c",hex($1))/ge;
	$v =~ s/\+/ /g; $v =~ s/%(..)/pack("c",hex($1))/ge;
	$a->{$k} = defined($a->{$k}) ? $a->{$k}."\0".$v : $v;
	}
}

# PrintHeader
# Outputs the HTTP header for HTML
sub PrintHeader
{
print "pragma: no-cache\n" if ($pragma_no_cache);
print "Content-type: text/html",
	$_[0] ? "; Charset=$_[0]" : "","\n\n";
}

# header(title, image, [help], [config], [nomodule], [nowebmin], [rightside],
#	 [header], [body])
# Output a page header with some title and image. The header may also
# include a link to help, and a link to the config page.
# The header will also have a link to to webmin index, and a link to the
# module menu if there is no config link
sub header
{
local($l, $ll, %access, $lang);
foreach $l (&list_languages()) {
	$lang = $l if ($l->{'lang'} eq $current_lang);
	}
&PrintHeader($lang->{'charset'});
print "<html>\n";
if (@_ > 0) {
	print "<head>\n";
	print "<title>$_[0]</title>\n";
	print $_[7] if ($_[7]);
	print "<SCRIPT LANGUAGE=\"JavaScript\">\n";
	printf "defaultStatus=\"%s%s logged into Webmin %s on %s (%s %s)\";\n",
		$ENV{'REMOTE_USER'},
		$ENV{'SSL_USER'} ? " (SSL certified)" : "",
		&get_webmin_version(),
		&get_system_hostname(),
		$gconfig{'real_os_type'} ? $gconfig{'real_os_type'}
					 : $gconfig{'os_type'},
		$gconfig{'real_os_version'} ? $gconfig{'real_os_version'}
					    : $gconfig{'os_version'};
	print "</SCRIPT>\n";
	print "</head>\n";
	}
local $bgcolor = defined($gconfig{'cs_page'}) ? $gconfig{'cs_page'} : "ffffff";
local $link = defined($gconfig{'cs_link'}) ? $gconfig{'cs_link'} : "0000ee";
local $text = defined($gconfig{'cs_text'}) ? $gconfig{'cs_text'} : "000000";
print "<body bgcolor=#$bgcolor link=#$link vlink=#$link text=#$text $_[8]>\n";
if (@_ > 1) {
	print "<table width=100%><tr>\n";
	print "<td width=15% valign=top align=left>";
	if ($ENV{'HTTP_WEBMIN_SERVERS'}) {
		print "<a href='$ENV{'HTTP_WEBMIN_SERVERS'}'>",
		      "$text{'header_servers'}</a><br>\n";
		}
	if (!$_[5]) {
		print "<a href='/?cat=$module_info{'category'}'>",
		      "$text{'header_webmin'}</a><br>\n";
		}
	if (!$_[4]) { print "<a href=\"\">$text{'header_module'}</a><br>\n"; }
	if (defined($_[2])) {
		print &hlink($text{'header_help'}, $_[2]),"<br>\n";
		}
	if ($_[3]) {
		local %access = &get_module_acl();
		if (!$access{'noconfig'}) {
			print "<a href=\"/config.cgi?$module_name\">",
			      $text{'header_config'},"</a><br>\n";
			}
		}
	print "</td>\n";
	local $title = $_[0];
	$title =~ s/&auml;//g;
	$title =~ s/&ouml;//g;
	$title =~ s/&uuml;//g;
	if ($_[1]) {
		print "<td align=center width=70%>",
		      "<img alt=\"$_[0]\" src=\"$_[1]\"></td>\n";
		}
	elsif ($lang->{'titles'} && !$gconfig{'texttitles'}) {
		print "<td align=center width=70%>";
		foreach $l (split(//, $title)) {
			$ll = ord($l);
			print "<img src=/images/letters/$ll.gif align=bottom>";
			}
		print "</td>\n";
		}
	else {
		print "<td align=center width=70%><h1>$_[0]</h1></td>\n";
		}
	print "<td width=15% valign=top align=right>";
	print $_[6];
	print "</td></tr></table>\n";
	}
}

# footer(page, name)
# Output a footer for returning to some page
sub footer
{
if (@_) {
	local $url = $_[0];
	$url = "/?cat=$module_info{'category'}" if ($url eq '/');
	print "<a href=\"$url\"><img alt=\"<-\" align=middle border=0 src=/images/left.gif></a>\n";
	print "&nbsp;&nbsp;<a href=\"$url\">",&text('main_return', $_[1]),
	      "</a><p>\n";
	}
print "</body></html>\n";
}

# redirect
# Output headers to redirect the browser to some page
sub redirect
{
local($port, $prot, $url);
$port = $ENV{'SERVER_PORT'} == 443 && uc($ENV{'HTTPS'}) eq "ON" ? "" :
	$ENV{'SERVER_PORT'} == 80 && uc($ENV{'HTTPS'}) ne "ON" ? "" :
		":$ENV{'SERVER_PORT'}";
$prot = uc($ENV{'HTTPS'}) eq "ON" ? "https" : "http";
if ($_[0] =~ /^(http|https|ftp|gopher):/) {
	$url = $_[0];
	}
elsif ($_[0] =~ /^\//) {
	$url = "$prot://$ENV{'SERVER_NAME'}$port$_[0]";
	}
else {
	$ENV{'SCRIPT_NAME'} =~ /^(.*)\/[^\/]+$/;
	$url = "$prot://$ENV{'SERVER_NAME'}$port$1/$_[0]";
	}
print "Location: $url\n\n";
}

# kill_byname(name, signal)
# Use the command defined in the global config to find and send a signal
# to a process matching some name
sub kill_byname
{
local(@pids);
@pids = &find_byname($_[0]);
if (@pids) { kill($_[1], @pids); }
}

# find_byname(name)
# Finds a process by name, and returns a list of matching PIDs
sub find_byname
{
local($cmd, @pids);
$cmd = $gconfig{'find_pid_command'};
$cmd =~ s/NAME/$_[0]/g;
@pids = split(/\n/, `$cmd`);
return @pids;
}

# error([message]+)
# Display an error message and exit. The variable $whatfailed must be set
# to the name of the operation that failed.
sub error
{
&header($text{'error'}, "");
print "<hr>\n";
print "<h3>",($whatfailed ? "$whatfailed : " : ""),@_,"</h3>\n";
print "<hr>\n";
&footer();
exit;
}

# error_setup(message)
# Register a message to be prepended to all error strings
sub error_setup
{
$whatfailed = $_[0];
}

# wait_for(handle, regexp, regexp, ...)
# Read from the input stream until one of the regexps matches..
sub wait_for
{
local($hit, $c, $i, $sw, $rv, $ha); undef($wait_for_input);
$ha = $_[0];
$code =
"undef(\$hit);\n".
"while(1) {\n".
" if ((\$c = getc($ha)) eq \"\") { return -1; }\n".
" \$wait_for_input .= \$c;\n";
#" \$wait_for_input .= \$c;\nprint \$wait_for_input,\"\\n\";";
for($i=1; $i<@_; $i++) {
        $sw = $i>1 ? "elsif" : "if";
        $code .= " $sw (\$wait_for_input =~ /$_[$i]/i) { \$hit = $i-1; }\n";
        }
$code .=
" if (defined(\$hit)) {\n".
"  \@matches = (-1, \$1, \$2, \$3, \$4, \$5, \$6, \$7, \$8, \$9);\n".
"  return \$hit;\n".
"  }\n".
" }\n";
$rv = eval $code;
if ($@) { &error("wait_for error : $@\n"); }
return $rv;
}

# fast_wait_for(handle, string, string, ...)
sub fast_wait_for
{
local($inp, $maxlen, $ha, $i, $c, $inpl);
for($i=1; $i<@_; $i++) {
	$maxlen = length($_[$i]) > $maxlen ? length($_[$i]) : $maxlen;
	}
$ha = $_[0];
while(1) {
	if (($c = getc($ha)) eq "") {
		&error("fast_wait_for read error : $!");
		}
	$inp .= $c;
	if (length($inp) > $maxlen) {
		$inp = substr($inp, length($inp)-$maxlen);
		}
	$inpl = length($inp);
	for($i=1; $i<@_; $i++) {
		if ($_[$i] eq substr($inp, $inpl-length($_[$i]))) {
			return $i-1;
			}
		}
	}
}

# has_command(command)
# Returns the full path if some command is in the path, undef if not
sub has_command
{
local($d);
if (!$_[0]) { return undef; }
if ($_[0] =~ /^\//) { return (-x $_[0]) ? $_[0] : undef; }
foreach $d (split(/:/ , $ENV{PATH})) {
	if (-x "$d/$_[0]") { return "$d/$_[0]"; }
	}
return undef;
}

@make_date_marr = ("Jan", "Feb", "Mar", "Apr", "May", "Jun",
	 	   "Jul", "Aug", "Sep", "Oct", "Nov", "Dec");

# make_date(seconds)
# Converts a Unix date/time in seconds to a human-readable form
sub make_date
{
local(@tm);
@tm = localtime($_[0]);
return sprintf "%d/%s/%d %2.2d:%2.2d",
		$tm[3], $make_date_marr[$tm[4]], $tm[5]+1900, $tm[2], $tm[1];
}

# file_chooser_button(input, type, [form])
# Return HTML for a file chooser button, if the browser supports Javascript.
# Type values are 0 for file or directory, or 1 for directory only
sub file_chooser_button
{
local($form);
$form = @_ > 2 ? $_[2] : 0;
return "<input type=button onClick='ifield = document.forms[$form].$_[0]; chooser = window.open(\"/chooser.cgi?type=$_[1]&file=\"+ifield.value, \"chooser\", \"toolbar=no,menubar=no,scrollbar=no,width=400,height=300\"); chooser.ifield = ifield' value=\"...\">\n";
}

# read_acl(&array, &array)
# Reads the acl file into the given associative arrays
sub read_acl
{
local($user, $_, @mods);
open(ACL, &acl_filename());
while(<ACL>) {
	if (/^(\S+):\s*(.*)/) {
		local(@mods);
		$user = $1;
		@mods = split(/\s+/, $2);
		if ($_[0]) { foreach $m (@mods) { ${$_[0]}{$user,$m}++; } }
		if ($_[1]) { $_[1]->{$user} = \@mods; }
		}
	}
close(ACL);
}

# acl_filename()
# Returns the file containing the webmin ACL
sub acl_filename
{
return "$config_directory/webmin.acl";
}

# acl_check()
# Does nothing, but kept around for compatability
sub acl_check
{
}

# get_miniserv_config(&array)
# Store miniserv configuration into the given array
sub get_miniserv_config
{
return &read_file($ENV{'MINISERV_CONFIG'}, $_[0]);
}

# put_miniserv_config(&array)
# Store miniserv configuration from the given array
sub put_miniserv_config
{
&write_file($ENV{'MINISERV_CONFIG'}, $_[0]);
}

# restart_miniserv()
# Send a HUP signal to miniserv
sub restart_miniserv
{
local($pid, %miniserv, $addr, $i);
&get_miniserv_config(\%miniserv) || return;
open(PID, $miniserv{'pidfile'}) || &error("Failed to open pid file");
chop($pid = <PID>);
close(PID);
if (!$pid) { &error("Invalid pid file"); }
kill('HUP', $pid);

# wait for miniserv to come back up
$addr = inet_aton($miniserv{'bind'} ? $miniserv{'bind'} : "127.0.0.1");
for($i=0; $i<20; $i++) {
	sleep(1);
	socket(STEST, PF_INET, SOCK_STREAM, getprotobyname("tcp"));
	$rv = connect(STEST, sockaddr_in($miniserv{'port'}, $addr));
	close(STEST);
	if ($rv) { last; }
	}
if ($i == 20) { &error("Failed to restart Webmin server!"); }
}

# check_os_support(&minfo)
sub check_os_support
{
local @oss = split(/\s+/, $_[0]->{'os_support'});
if (!@oss)
	{ return 1; }
elsif (&indexof($gconfig{'os_type'}, @oss) != -1)
	{ return 1; }
elsif (&indexof("$gconfig{'os_type'}/$gconfig{'os_version'}", @oss) != -1)
	{ return 1; }
else
	{ return 0; }
}

# http_download(host, port, page, destfile)
# Download data from a HTTP url to a local file
sub http_download
{
$SIG{ALRM} = "download_timeout";
alarm(60);
if ($gconfig{'http_proxy'} =~ /^http:\/\/(\S+):(\d+)/ && !&no_proxy($_[0])) {
	# going through proxy
	&open_socket($1, $2, "SOCK");
	print SOCK "GET http://$_[0]:$_[1]$_[2] HTTP/1.0\r\n";
	}
else {
	# can connect directly
	&open_socket($_[0], $_[1], "SOCK");
	print SOCK "GET $_[2] HTTP/1.0\r\n";
	}
alarm(0);
print SOCK "Host: $_[0]\r\n";
print SOCK "User-agent: Webmin\r\n";
print SOCK "\r\n";
&complete_http_download("SOCK", $_[3]);
}

# complete_http_download(socket, destfile)
# Do a HTTP download, after the headers have been sent
sub complete_http_download
{
local($line, %header, $s);
$s = $_[0];

# read headers
alarm(60);
($line = <$s>) =~ s/\r|\n//g;
if ($line !~ /^HTTP\/1\..\s+200\s+/) { &error("Download failed : $line"); }
while(<$s> =~ /^(\S+):\s+(.*)$/) { $header{lc($1)} = $2; }
alarm(0);

# read data
open(PFILE, "> $_[1]");
while(read($s, $buf, 1024) > 0) { print PFILE $buf; }
close(PFILE);
close($s);
}


# ftp_download(host, file, destfile)
# Download data from an FTP site to a local file
sub ftp_download
{
local($buf, @n);

$SIG{ALRM} = "download_timeout";
alarm(60);
if ($gconfig{'ftp_proxy'} =~ /^http:\/\/(\S+):(\d+)/ && !&no_proxy($_[0])) {
	# download through http-style proxy
	&open_socket($1, $2, "SOCK");
	print SOCK "GET ftp://$_[0]$_[1] HTTP/1.0\r\n";
	print SOCK "User-agent: Webmin\r\n";
	print SOCK "\r\n";
	&complete_http_download("SOCK", $_[2]);
	}
else {
	# connect to host and login
	&open_socket($_[0], 21, "SOCK");
	alarm(0);
	&ftp_command("", 2);
	&ftp_command("user anonymous", 3);
	&ftp_command("pass root\@".&get_system_hostname(), 2);

	# request the file
	&ftp_command("type i", 2);
	&ftp_command("pasv", 2) =~ /\(([0-9,]+)\)/;
	@n = split(/,/ , $1);
	&open_socket("$n[0].$n[1].$n[2].$n[3]", $n[4]*256 + $n[5], "CON");
	&ftp_command("retr $_[1]", 1);

	# transfer data
	open(PFILE, "> $_[2]");
	while(read(CON, $buf, 1024) > 0) { print PFILE $buf; }
	close(PFILE);
	close(CON);

	# finish off..
	&ftp_command("", 2);
	&ftp_command("quit", 2);
	close(SOCK);
	}
}

# no_proxy(host)
# Checks if some host is on the no proxy list
sub no_proxy
{
foreach $n (split(/\s+/, $gconfig{'noproxy'})) {
	if ($_[0] =~ /$n/) { return 1; }
	}
return 0;
}

# open_socket(host, port, handle)
sub open_socket
{
local($addr, $h); $h = $_[2];
socket($h, PF_INET, SOCK_STREAM, getprotobyname("tcp")) ||
	&error("Failed to create socket : $!");
($addr = inet_aton($_[0])) ||
	&error("Failed to lookup IP address for $_[0]");
connect($h, sockaddr_in($_[1], $addr)) ||
	&error("Failed to connect to $_[0]:$_[1] : $!");
select($h); $| =1; select(STDOUT);
}


# download_timeout()
# Called when a download times out
sub download_timeout
{
&error("Timeout downloading $in{url}");
}


# ftp_command(command, expected)
# Send an FTP command, and die if the reply is not what was expected
sub ftp_command
{
local($line, $code, $reply);
$what = $_[0] ne "" ? "<i>$_[0]</i>" : "initial connection";
if ($_[0] ne "") {
        print SOCK "$_[0]\r\n";
        }
alarm(60);
if (!($line = <SOCK>)) {
        &error("Failed to read reply to $what");
        }
$line =~ /^(...)(.)(.*)$/;
if (int($1/100) != $_[1]) {
        &error("$what failed : $3");
        }
$code = $1; $reply = $3;
if ($2 eq "-") {
        # Need to skip extra crap..
        while(1) {
                if (!($line = <SOCK>)) {
                        &error("Failed to read reply to $what");
                        }
                $line =~ /^(....)(.*)$/; $reply .= $2;
		if ($1 eq "$code ") { last; }
                }
        }
alarm(0);
return $reply;
}

# to_ipaddress(hostname)
# Converts a hostname to an a.b.c.d format IP address
sub to_ipaddress
{
local(@ip);
@ip = unpack("CCCC", gethostbyname($_[0]));
if (@ip) { return join("." , @ip); }
else { return undef; }
}

# icons_table(&links, &titles, &icons, [columns])
# Renders a 4-column table of icons
sub icons_table
{
local($i);
local $cols = $_[3] ? $_[3] : 4;
local $per = int(100.0 / $cols);
print "<table width=100% cellpadding=5> <tr>\n";
for($i=0; $i<@{$_[0]}; $i++) {
	if ($i%$cols == 0) { print "<tr>\n"; }
	print "<td width=$per% align=center valign=top>\n";
	&generate_icon($_[2]->[$i], $_[1]->[$i], $_[0]->[$i]);
	print "</td>\n";
        if ($i%$cols == $cols-1) { print "</tr>\n"; }
        }
while($i++%$cols) { print "<td width=$per%></td>\n"; }
print "</table><p>\n";
}

# replace_file_line(file, line, [newline]*)
# Replaces one line in some file with 0 or more new lines
sub replace_file_line
{
local(@lines);
open(FILE, $_[0]);
@lines = <FILE>;
close(FILE);
if (@_ > 2) { splice(@lines, $_[1], 1, @_[2..$#_]); }
else { splice(@lines, $_[1], 1); }
open(FILE, "> $_[0]");
print FILE @lines;
close(FILE);
}

# read_file_lines(file)
# Returns a reference to an array containing the lines from some file. This
# array can be modified, and will be written out when flush_file_lines()
# is called.
sub read_file_lines
{
if (!$file_cache{$_[0]}) {
        local(@lines, $_);
        open(READFILE, $_[0]);
        while(<READFILE>) {
                s/\r|\n//g;
                push(@lines, $_);
                }
        close(READFILE);
        $file_cache{$_[0]} = \@lines;
        }
return $file_cache{$_[0]};
}

# flush_file_lines()
sub flush_file_lines
{
foreach $f (keys %file_cache) {
        open(FLUSHFILE, "> $f");
        foreach $line (@{$file_cache{$f}}) {
                print FLUSHFILE $line,"\n";
                }
        close(FLUSHFILE);               
        }                               
}                                       

# unix_user_input(fieldname, user)
# Returns HTML for an input to select a Unix user
sub unix_user_input
{
return "<input name=$_[0] size=8 value=\"$_[1]\"> ".
       &user_chooser_button($_[0], 0)."\n";
}

# unix_group_input(fieldname, user)
# Returns HTML for an input to select a Unix group
sub unix_group_input
{
return "<input name=$_[0] size=8 value=\"$_[1]\"> ".
       &group_chooser_button($_[0], 0)."\n";
}

# hlink(text, page, [module])
sub hlink
{
local $mod = $_[2] ? $_[2] : $module_name;
return "<a onClick='window.open(\"/help.cgi/$mod/$_[1]\", \"help\", \"toolbar=no,menubar=no,scrollbars=yes,width=400,height=300,resizable=yes\"); return false' href=\"/help.cgi/$mod/$_[1]\">$_[0]</a>";
}

# user_chooser_button(field, multiple, [form])
# Returns HTML for a javascript button for choosing a Unix user or users
sub user_chooser_button
{
local $form = @_ > 2 ? $_[2] : 0;
local $w = $_[1] ? 500 : 300;
return "<input type=button onClick='ifield = document.forms[$form].$_[0]; chooser = window.open(\"/user_chooser.cgi?multi=$_[1]&user=\"+escape(ifield.value), \"chooser\", \"toolbar=no,menubar=no,scrollbars=yes,width=$w,height=200\"); chooser.ifield = ifield' value=\"...\">\n";
}

# group_chooser_button(field, multiple, [form])
# Returns HTML for a javascript button for choosing a Unix group or groups
sub group_chooser_button
{
local $form = @_ > 2 ? $_[2] : 0;
local $w = $_[1] ? 500 : 300;
return "<input type=button onClick='ifield = document.forms[$form].$_[0]; chooser = window.open(\"/group_chooser.cgi?multi=$_[1]&group=\"+escape(ifield.value), \"chooser\", \"toolbar=no,menubar=no,scrollbars=yes,width=$w,height=200\"); chooser.ifield = ifield' value=\"...\">\n";
}

# foreign_check(module)
# Checks if some other module exists and is supported on this OS
sub foreign_check
{
local %minfo;
&read_file("../$_[0]/module.info", \%minfo) || return 0;
return &check_os_support(\%minfo);
}

# foreign_require(module, file)
# Brings in functions from another module
sub foreign_require
{
chdir("../$_[0]");
eval <<EOF;
package $_[0];
\$ENV{'FOREIGN_MODULE_NAME'} = '$_[0]';
do "./$_[1]";
EOF
if ($@) { &error($@); }
chdir("../$module_name");
return 1;
}

# foreign_call(module, function, [arg]*)
# Call a function in another module
sub foreign_call
{
chdir("../$_[0]");
local @args = @_[2 .. @_-1];
$main::foreign_args = \@args;
local @rv = eval <<EOF;
package $_[0];
&$_[1](\@{\$main::foreign_args});
EOF
if ($@) { &error($@); }
chdir("../$module_name");
return wantarray ? @rv : $rv[0];
}

# foreign_config(module)
# Get the configuration from another module
sub foreign_config
{
local %fconfig;
&read_file("$config_directory/$_[0]/config", \%fconfig);
return %fconfig;
}

# get_system_hostname()
# Returns the hostname of this system
sub get_system_hostname
{
if (!$get_system_hostname) {
	chop($get_system_hostname = `hostname 2>/dev/null`);
	if ($?) {
		use Sys::Hostname;
		$get_system_hostname = eval "hostname()";
		if ($@) { $get_system_hostname = "UNKNOWN"; }
		}
	}
return $get_system_hostname;
}

# get_webmin_version()
# Returns the version of Webmin currently being run
sub get_webmin_version
{
if (!$get_webmin_version) {
	open(VERSION, "../version") || open(VERSION, "version") || return 0;
	chop($get_webmin_version = <VERSION>);
	close(VERSION);
	}
return $get_webmin_version;
}

# get_module_acl([user], [module])
# Returns an array containing access control options for the given user
sub get_module_acl
{
local %rv;
local $u = $_[0] ? $_[0] : $ENV{'REMOTE_USER'};
local $m = $_[1] ? $_[1] : $module_name;
&read_file("../$m/defaultacl", \%rv);
&read_file("$config_directory/$m/$u.acl", \%rv);
return %rv;
}

# save_module_acl(&acl, [user], [module])
# Updates the acl hash for some user and module (or the current one)
sub save_module_acl
{
local $u = $_[1] ? $_[1] : $ENV{'REMOTE_USER'};
local $m = $_[2] ? $_[2] : $module_name;
&write_file("$config_directory/$m/$u.acl", $_[0]);
}

# init_config()
# Sets the following variables
#  %config - Per-module configuration
#  %gconfig - Global configuration
#  $tb - Background for table headers
#  $cb - Background for table bodies
#  $scriptname - Base name of the current perl script
#  $module_name - The name of the current module
#  $module_config_directory - The config directory for this module
sub init_config
{
# Read the webmin global config file. This contains the OS type and version,
# OS specific configuration and global options such as proxy servers
&read_file("$config_directory/config", \%gconfig);

# Set PATH and LD_LIBRARY_PATH
$ENV{'PATH'} = $gconfig{'path'};
if ($gconfig{'ld_env'}) {
	$ENV{$gconfig{'ld_env'}} = $gconfig{'ld_path'};
	}

# Work out which module we are in, and read the per-module config file
if ($ENV{'FOREIGN_MODULE_NAME'}) {
	# In a foreign call - use the module name given
	$module_name = $ENV{'FOREIGN_MODULE_NAME'};
	}
elsif ($ENV{'SCRIPT_NAME'}) {
	if ($ENV{'SCRIPT_NAME'} =~ /^\/([^\/]+)\//) {
		# Get module name from CGI path
		$module_name = $1;
		}
	}
elsif ($0 =~ /([^\/]+)\/[^\/]+$/) {
	# Get module name from command line
	$module_name = $1;
	}
if ($module_name) {
	$module_config_directory = "$config_directory/$module_name";
	&read_file("$module_config_directory/config", \%config);
	%module_info = &get_module_info($module_name);
	}

# Set some useful variables
$tb = defined($gconfig{'cs_header'}) ? "bgcolor=#$gconfig{'cs_header'}"
				     : "bgcolor=#9999ff";
$cb = defined($gconfig{'cs_table'}) ? "bgcolor=#$gconfig{'cs_table'}"
				    : "bgcolor=#cccccc";
$0 =~ /([^\/]+)$/;
$scriptname = $1;

# Load language strings into %text
local $u = $ENV{'REMOTE_USER'};
$current_lang = $gconfig{"lang_$u"} ? $gconfig{"lang_$u"} :
		$gconfig{"lang"} ? $gconfig{"lang"} : $default_lang;
%text = &load_language();

# Check if the HTTP user can access this module
if ($module_name && !$main::no_acl_check && !$ENV{'FOREIGN_MODULE_NAME'}) {
	local(%acl, %minfo);
	&read_acl(\%acl, undef);
	%minfo = &get_module_info($module_name);
	$acl{$u,$module_name} || $acl{$u,'*'} ||
		&error("Access denied : User <i>$u</i> is not allowed to ".
		       "use the <i>$minfo{'desc'}</i> module");
	$main::no_acl_check++;
	}
return 1;
}

$default_lang = "en";

# load_language([module])
# Returns a hashtable mapping text codes to strings in the appropriate language
sub load_language
{
local $mod = $_[0] ? "../$_[0]" : ".";
local %text;
if ($module_name) {
	&read_file("../lang/$default_lang", \%text);
	&read_file("../lang/$current_lang", \%text);
	}
&read_file("$mod/lang/$default_lang", \%text);
&read_file("$mod/lang/$current_lang", \%text);
return %text;
}

# text(message, [substitute]+)
sub text
{
local $rv = $text{$_[0]};
local $i;
for($i=1; $i<@_; $i++) {
	$rv =~ s/\$$i/$_[$i]/g;
	}
return $rv;
}

# encode_base64(string)
# Encodes a string into base64 format
sub encode_base64
{
    local $res;
    pos($_[0]) = 0;                          # ensure start at the beginning
    while ($_[0] =~ /(.{1,45})/gs) {
        $res .= substr(pack('u', $1), 1)."\n";
        chop($res);
    }
    $res =~ tr|` -_|AA-Za-z0-9+/|;
    local $padding = (3 - length($_[0]) % 3) % 3;
    $res =~ s/.{$padding}$/'=' x $padding/e if ($padding);
    return $res;
}

# get_module_info(module, [noclone])
# Returns a hash containg a module name, desc and os_support
sub get_module_info
{
return () if ($_[0] =~ /^\./);
local (%rv, $clone);
if ($module_name && &read_file("../$_[0]/module.info", \%rv)) {
	$clone = -l "../$_[0]";
	}
elsif (&read_file("$_[0]/module.info", \%rv)) {
	$clone = -l $_[0];
	}
else { return (); }
$rv{"desc"} = $rv{"desc_$current_lang"} if ($rv{"desc_$current_lang"});
if ($clone && !$_[1]) {
	$rv{'clone'} = $rv{'desc'};
	&read_file("$config_directory/$_[0]/clone", \%rv);
	}
$rv{'dir'} = $_[0];
return %rv;
}

# list_languages()
# Returns an array of supported languages
sub list_languages
{
if (!@list_languages_cache) {
	local ($o, $_);
	open(LANG, "../lang_list.txt") || open(LANG, "lang_list.txt");
	while(<LANG>) {
		if (/^(\S+)\s+(.*)/) {
			local $l = { 'desc' => $2 };
			foreach $o (split(/,/, $1)) {
				if ($o =~ /^([^=]+)=(.*)$/) {
					$l->{$1} = $2;
					}
				}
			$l->{'index'} = scalar(@rv);
			push(@list_languages_cache, $l);
			}
		}
	close(LANG);
	}
return @list_languages_cache;
}

# read_env_file(file, &array)
sub read_env_file
{
open(FILE, $_[0]) || return 0;
while(<FILE>) {
	s/#.*$//g;
	if (/([A-z0-9_]+)\s*=\s*"(.*)"/ ||
	    /([A-z0-9_]+)\s*=\s*'(.*)'/ ||
	    /([A-z0-9_]+)\s*=\s*(\S+)/) {
		$_[1]->{$1} = $2;
		}
	}
close(FILE);
return 1;
}

# write_env_file(file, &array, export)
sub write_env_file
{
local $k;
local $exp = $_[2] ? "export " : "";
open(FILE, ">$_[0]");
foreach $k (keys %{$_[1]}) {
	local $v = $_[1]->{$k};
	if ($v =~ /^\S+$/) {
		print FILE "$exp$k=$v\n";
		}
	else {
		print FILE "$exp$k=\"$v\"\n";
		}
	}
close(FILE);
}

1;  # return true?

