# fdisk-lib.pl
# Functions for disk management under linux

do '../web-lib.pl';
&init_config();
&foreign_require("mount", "mount-lib.pl");
&foreign_require("raid", "raid-lib.pl");
&foreign_require("proc", "proc-lib.pl");
%access = &get_module_acl();
$| = 1;

# list_disks()
# Returns a list of structures, one per disk
sub list_disks
{
local (@rv, $f, $i);
local $idx = 0;
if ((-d "/proc/ide" || -d "/proc/scsi") && -r "/proc/partitions") {
	# Get the list of IDE disks from the kernel
	opendir(IDE, "/proc/ide");
	foreach $f (sort { $a cmp $b } readdir(IDE)) {
		next if ($f !~ /hd(\S)$/);
		local $disk = { "device" => "/dev/$f",
				"prefix" => "/dev/$f",
				"type" => "ide" };
		open(MEDIA, "/proc/ide/$f/media");
		local $media = <MEDIA>;
		close(MEDIA);
		next if ($media !~ /^disk/);
		open(GEOM, "/proc/ide/$f/geometry");
		while(<GEOM>) {
			if (/logical\s+(\d+)\/(\d+)\/(\d+)/) {
				$disk->{'cylinders'} = $1;
				$disk->{'heads'} = $2;
				$disk->{'sectors'} = $3;
				}
			}
		close(GEOM);
		open(MODEL, "/proc/ide/$f/model");
		chop($disk->{'model'} = <MODEL>);
		close(MODEL);
		$disk->{'index'} = $idx++;
		$disk->{'desc'} = &text('select_device', 'IDE',
					uc(substr($disk->{'device'}, -1)));
		push(@rv, $disk);
		}
	closedir(IDE);

	# Get the list of SCSI disks from the kernel and fdisk
	local @scsi;
	open(PARTS, "/proc/partitions");
	while(<PARTS>) {
		s/\r|\n//g;
		if (/\d+\s+\d+\s+\d+\s+sd(\S)$/) {
			push(@scsi, "/dev/sd$1");
			}
		elsif (/\d+\s+\d+\s+\d+\s+(rd\/c(\d+)d\d+)$/) {
			local ($d, $c) = ("/dev/$1", $2);
			push(@scsi, $d);
			open(RD, "/proc/rd/c$c/current_status");
			while(<RD>) {
				if (/^Configuring\s+(.*)/i) {
					$dac{$d} = $1;
					}
				elsif (/\s+(\S+):\s+([^, ]+)/ && $1 eq $d) {
					$raid{$d} = $2;
					}
				}
			close(RD);
			}
		}
	close(PARTS);
	local @pscsi;
	open(SCSI, "/proc/scsi/scsi");
	while(<SCSI>) {
		s/\s/ /g;
		if (/^Host:/) {
			push(@pscsi, $_);
			}
		elsif (/^\s+\S/) {
			$pscsi[$#pscsi] .= $_;
			}
		}
	close(SCSI);
	@pscsi = grep { /Type:\s+Direct-Access/ } @pscsi;
	for($i=0; $i<@scsi; $i++) {
		&open_fdisk("-l", $scsi[$i]);
		while(<$fh>) {
			if (/Disk\s+([^ :]+):\s+(\d+)\s+\S+\s+(\d+)\s+\S+\s+(\d+)/) {
				local @st = stat($1);
				local $disk = { 'device' => $1,
						'heads' => $2,
						'sectors' => $3,
						'cylinders' => $4,
						'index' => $idx++,
						'type' => 'scsi' };
				if ($pscsi[$i] =~ /Vendor:\s+(\S+).*Model:\s+(.*)\s+Rev:/i) {
					$disk->{'model'} = "$1 $2";
					}
				if ($pscsi[$i] =~ /Host:\s+scsi(\d+).*Id:\s+(\d+)/i) {
					$disk->{'controller'} = int($1);
					$disk->{'scsiid'} = int($2);
					}
				if ($dac{$disk->{'device'}}) {
					$disk->{'model'} = $dac{$disk->{'device'}};
					$disk->{'raid'} = $raid{$disk->{'device'}};
					$disk->{'device'} =~ /c(\d+)d(\d+)$/;
					$disk->{'desc'} = &text('select_mylex', "$1", "$2");
					$disk->{'prefix'} = $disk->{'device'}.'p';
					}
				else {
					$disk->{'desc'} = &text('select_device', 'SCSI',
								uc(substr($disk->{'device'}, -1)));
					$disk->{'prefix'} = $disk->{'device'};
					}
				push(@rv, $disk);
				}
			}
		&close_fdisk();
		}
	}
else {
	# Get the list from fdisk -l
	local @cdstat = stat("/dev/cdrom");
	&open_fdisk("-l");
	while(<$fh>) {
		if (/Disk\s+([^ :]+):\s+(\d+)\s+\S+\s+(\d+)\s+\S+\s+(\d+)/) {
			local @st = stat($1);
			if ($st[1] != $cdstat[1]) {
				local $disk = { 'device' => $1,
						'heads' => $2,
						'sectors' => $3,
						'cylinders' => $4,
						'index' => $idx++ };
				$disk->{'type'} =
				    $disk->{'device'} =~ /hd.$/ ? 'ide'
								: 'scsi';
				$disk->{'desc'} = &text('select_device', uc($disk->{'type'}),
							uc(substr($disk->{'device'}, -1)));
				push(@rv, $disk);
				}
			}
		}
	}
return @rv;
}

# list_partitions(disk)
# Returns a list of structures, one per partition
sub list_partitions
{
local(@rv, $wf);
local $idx = 0;
&open_fdisk($_[0]);
&wprint("p\nq\n");
&wait_for($fh, 'System\r');
while(1) {
	$wf = &wait_for($fh,
	  '\n(\/dev\/\S+(\d+))[ \t*]+\d+\s+(\d+)\s+(\d+)\s+(\S+)\s+(\S{1,2})\s+(.*)\r',
	  '\n(\/dev\/\S+(\d+))[ \t*]+(\d+)\s+(\d+)\s+(\S+)\s+(\S{1,2})\s+(.*)\r');
	last if ($wf < 0);
	$matches[4] =~ s/\+$//g;
	push(@rv, { 'number' => $matches[2],
		    'device' => $matches[1],
		    'type' => $matches[6],
		    'start' => $matches[3],
		    'end' => $matches[4],
		    'blocks' => $matches[5],
		    'extended' => $matches[6] eq '5' || $matches[6] eq 'f' ? 1 : 0,
		    'desc' => $matches[1] =~ /(.)d(\S)(\d+)$/ ?
			      &text('select_part', $1 eq 's' ? 'SCSI' : 'IDE', uc($2), "$3") :
			      $matches[1] =~ /rd\/c(\d+)d(\d+)p(\d+)$/ ? 
			      &text('select_mpart', "$1", "$2", "$3") : "???",
		    'index' => $idx++ });
	}
&close_fdisk();
return @rv;
}

# change_type(disk, partition, type)
# Changes the type of an existing partition
sub change_type
{
&open_fdisk("$_[0]");
&wprint("t\n");
&wait_for($fh, 'Partition.*:');
&wprint("$_[1]\n");
&wait_for($fh, 'Hex.*:');
&wprint("$_[2]\n");
&wait_for($fh, 'Command.*:');
&wprint("w\n"); sleep(1);
&close_fdisk();
}

# delete_partition(disk, partition)
# Delete an existing partition
sub delete_partition
{
&open_fdisk("$_[0]");
&wprint("d\n");
&wait_for($fh, 'Partition.*:');
&wprint("$_[1]\n");
&wait_for($fh, 'Command.*:');
&wprint("w\n");
&wait_for($fh, 'Syncing'); sleep(3);
&close_fdisk();
}

# create_partition(disk, partition, start, end, type)
# Create a new partition with the given extent and type
sub create_partition
{
&open_fdisk("$_[0]");
&wprint("n\n");
local $wf = &wait_for($fh, 'primary.*\r\n', 'First.*:');
if ($_[1] > 4) {
	&wprint("l\n");
	}
else {
	&wprint("p\n");
	&wait_for($fh, 'Partition.*:');
	&wprint("$_[1]\n");
	}
&wait_for($fh, 'First.*:') if ($wf != 1);
&wprint("$_[2]\n");
&wait_for($fh, 'Last.*:');
&wprint("$_[3]\n");
&wait_for($fh, 'Command.*:');

&wprint("t\n");
&wait_for($fh, 'Partition.*:');
&wprint("$_[1]\n");
&wait_for($fh, 'Hex.*:');
&wprint("$_[4]\n");
&wait_for($fh, 'Command.*:');
&wprint("w\n");
&wait_for($fh, 'Syncing'); sleep(3);
&close_fdisk();
}

# create_extended(disk, partition, start, end)
sub create_extended
{
&open_fdisk("$_[0]");
&wprint("n\n");
&wait_for($fh, 'primary.*\r\n');
&wprint("e\n");
&wait_for($fh, 'Partition.*:');
&wprint("$_[1]\n");
&wait_for($fh, 'First.*:');
&wprint("$_[2]\n");
&wait_for($fh, 'Last.*:');
&wprint("$_[3]\n");
&wait_for($fh, 'Command.*:');

&wprint("w\n");
&wait_for($fh, 'Syncing'); sleep(3);
&close_fdisk();
}

# list_tags()
# Returns a list of known partition tag numbers
sub list_tags
{
return sort { hex($a) <=> hex($b) } (keys %tags);
}

# tag_name(tag)
# Returns a human-readable version of a tag
sub tag_name
{
return $tags{$_[0]} ? $tags{$_[0]}
		    : $hidden_tags{$_[0]};
}

# conv_type(tag)
# Given a partition tag, returns the filesystem type (assuming it is supported)
sub conv_type
{
if ($_[0] eq "4" || $_[0] eq "6" ||
    $_[0] eq "1" || $_[0] eq "e") { $rv = "msdos"; }
elsif ($_[0] eq "b" || $_[0] eq "c") { return "vfat"; }
elsif ($_[0] eq "83") { $rv = "ext2"; }
elsif ($_[0] eq "81") { $rv = "minix"; }
else { return undef; }
if (&has_command("mkfs.$rv")) { return $rv; }
return undef;
}

# fstype_name(type)
# Returns a readable name for a filesystem type
sub fstype_name
{
return $fstypes{$_[0]};
}

sub mkfs_options
{
if ($_[0] eq "ext2") {
	&opt_input("ext2_b", "bytes", 1);
	&opt_input("ext2_f", "bytes", 0);
	&opt_input("ext2_i", "", 1);
	&opt_input("ext2_m", "%", 0);
	&opt_input("ext2_g", "", 1);
	print "<td align=right><b>$text{'ext2_c'}</b></td>\n";
	print "<td><input type=radio name=ext2_c value=1> $text{'yes'}\n";
	print "<input type=radio name=ext2_c value=0 checked> $text{'no'}",
	      "</td> </tr>\n";
	}
elsif ($_[0] eq "msdos" || $_[0] eq "vfat") {
	&opt_input("msdos_f", "", 1);
	&opt_input("msdos_F", "bits", 0);
	&opt_input("msdos_i", "", 1);
	&opt_input("msdos_n", "", 0);
	&opt_input("msdos_r", "", 1);
	&opt_input("msdos_s", "sectors", 0);
	print "<tr> <td align=right><b>$text{'msdos_c'}</b></td>\n";
	print "<td><input type=radio name=msdos_c value=1> $text{'yes'}\n";
	print "<input type=radio name=msdos_c value=0 checked> $text{'no'}",
	      "</td> </tr>\n";
	}
elsif ($_[0] eq "minix") {
	&opt_input("minix_n", "", 1);
	&opt_input("minix_i", "", 0);
	&opt_input("minix_b", "", 1);
	print "<td align=right><b>$text{'minix_c'}</b></td>\n";
	print "<td><input type=radio name=msdos_c value=1> $text{'yes'}\n";
	print "<input type=radio name=msdos_c value=0 checked> $text{'no'}",
	      "</td> </tr>\n";
	}
}

# mkfs_parse(type, device)
# Returns a command to build a new filesystem of the given type on the
# given device. Options are taken from %in.
sub mkfs_parse
{
local($cmd);
if ($_[0] eq "ext2") {
	$cmd = "mkfs -t ext2";
	$cmd .= &opt_check("ext2_b", '\d+', "-b");
	$cmd .= &opt_check("ext2_f", '\d+', "-f");
	$cmd .= &opt_check("ext2_i", '\d{4,}', "-i");
	$cmd .= &opt_check("ext2_m", '\d+', "-m");
	$cmd .= &opt_check("ext2_g", '\d+', "-g");
	$cmd .= $in{'ext2_c'} ? " -c" : "";
	$cmd .= " $_[1]";
	}
elsif ($_[0] eq "msdos" || $_[0] eq "vfat") {
	$cmd = "mkfs -t $_[0]";
	$cmd .= &opt_check("msdos_f", '[1-2]', "-f");
	$cmd .= &opt_check("msdos_F", '\d+', "-F");
	$cmd .= &opt_check("msdos_i", '[0-9a-f]{8}', "-i");
	$cmd .= &opt_check("msdos_n", '\S{1,11}', "-n");
	$cmd .= &opt_check("msdos_r", '\d+', "-r");
	$cmd .= &opt_check("msdos_s", '\d+', "-s");
	$cmd .= $in{'msdos_c'} ? " -c" : "";
	$cmd .= " $_[1]";
	}
elsif ($_[0] eq "minix") {
	local(@plist, $disk, $part, $i, @pinfo);
	$cmd = "mkfs -t minix";
	$cmd .= &opt_check("minix_n", '14|30', "-n ");
	$cmd .= &opt_check("minix_i", '\d+', "-i ");
	$cmd .= $in{'minix_c'} ? " -c" : "";
	$cmd .= " $_[1]";
	$cmd .= &opt_check("minix_b", '\d+', " ");
	}
return $cmd;
}

# can_tune(type)
# Returns 1 if this filesystem type can be tuned
sub can_tune
{
return ($_[0] eq "ext2");
}

# tunefs_options(type)
# Output HTML for tuning options for some filesystem type
sub tunefs_options
{
if ($_[0] eq "ext2") {
	&opt_input("tunefs_c", "", 1);

	print "<td align=right><b>$text{'tunefs_e'}</b></td> <td>\n";
	print "<input type=radio name=tunefs_e_def value=1 checked> ",
	      "$text{'opt_default'}\n";
	print "&nbsp; <input type=radio name=tunefs_e_def value=0>\n";
	print "<select name=tunefs_e>\n";
	print "<option value=continue> $text{'tunefs_continue'}\n";
	print "<option value=remount-ro> $text{'tunefs_remount'}\n";
	print "<option value=panic> $text{'tunefs_panic'}\n";
	print "</select></td> </tr>\n";

	print "<tr> <td align=right><b>$text{'tunefs_u'}</b></td> <td>\n";
	print "<input type=radio name=tunefs_u_def value=1 checked> ",
	      "$text{'opt_default'}\n";
	print "&nbsp; <input type=radio name=tunefs_u_def value=0>\n";
	print "<input name=tunefs_u size=8> ",
	      &user_chooser_button("tunefs_u", 0),"</td>\n";

	print "<td align=right><b>$text{'tunefs_g'}</b></td> <td>\n";
	print "<input type=radio name=tunefs_g_def value=1 checked> ",
	      "$text{'opt_default'}\n";
	print "&nbsp; <input type=radio name=tunefs_g_def value=0>\n";
	print "<input name=tunefs_g size=8> ",
	      &group_chooser_button("tunefs_g", 0),"</td> </tr>\n";

	&opt_input("tunefs_m", "%", 1);
	$tsel = "<select name=tunefs_i_unit>\n".
		"<option value=d> $text{'tunefs_days'}\n".
		"<option value=w> $text{'tunefs_weeks'}\n".
		"<option value=m> $text{'tunefs_months'}\n".
		"</select>\n";
	&opt_input("tunefs_i", $tsel, 0);
	}
}

# tunefs_parse(type, device)
# Returns the tuning command based on user inputs
sub tunefs_parse
{
if ($_[0] eq "ext2") {
	$cmd = "tune2fs";
	$cmd .= &opt_check("tunefs_c", '\d+', "-c");
	$cmd .= $in{'tunefs_e_def'} ? "" : " -e$in{'tunefs_e'}";
	$cmd .= $in{'tunefs_u_def'} ? "" : " -u".getpwnam($in{'tunefs_u'});
	$cmd .= $in{'tunefs_g_def'} ? "" : " -g".getgrnam($in{'tunefs_g'});
	$cmd .= &opt_check("tunefs_m",'\d+',"-m");
	$cmd .= &opt_check("tunefs_i", '\d+', "-i").
		($in{'tunefs_i_def'} ? "" : $in{'tunefs_i_unit'});
	$cmd .= " $_[1]";
	}
return $cmd;
}

# need_reboot(disk)
# Returns 1 if a reboot is needed after changing the partitions on some disk
sub need_reboot
{
local $un = `uname -r`;
return $un =~ /^2\.0\./ || $un =~ /^1\./ || $un =~ /^0\./;
}

# device_status(device)
# Returns an array of  directory, type, mounted
sub device_status
{
local ($mounted) = grep { $_->[1] eq $_[0] }
		        &foreign_call("mount", "list_mounted");
local ($mount) = grep { $_->[1] eq $_[0] }
		      &foreign_call("mount", "list_mounts");

if ($mounted) { return ($mounted->[0], $mounted->[2], 1); }
elsif ($mount) { return ($mount->[0], $mount->[2], 0); }
else {
	local $conf = &foreign_call("raid", "get_raidtab");
	foreach $c (@$conf) {
		if (&indexof($_[0], @{$c->{'device'}})) {
			return ( $c->{'value'}, "raid", 1 );
			}
		}
	return ();
	}
}

# can_fsck(type)
# Returns 1 if some filesystem type can fsck'd
sub can_fsck
{
return ($_[0] eq "ext2" && &has_command("fsck.ext2") ||
	$_[0] eq "minix" && &has_command("fsck.minix"));
}

# fsck_command(type, device)
# Returns the fsck command to unconditionally check a filesystem
sub fsck_command
{
if ($_[0] eq "ext2") {
	return "fsck -t ext2 -p $_[1]";
	}
elsif ($_[0] eq "minix") {
	return "fsck -t minix -a $_[1]";
	}
}

# fsck_error(code)
# Returns a description of an exit code from fsck
sub fsck_error
{
return $text{"fsck_err$_[0]"} ? $text{"fsck_err$_[0]"}
			      : &text("fsck_unknown", $_[0]);
}

# partition_select(name, value, mode, &found)
# Returns HTML for selecting a disk or partition
# mode 0 = floppies and disk partitions
#      1 = disks
#      2 = floppies and disks and disk partitions
#      3 = disk partitions
sub partition_select
{
local $rv = "<select name=$_[0]>\n";
local ($found, $d, $p);
if ($_[2] == 0 || $_[2] == 2) {
	$rv .= sprintf "<option %s value=/dev/fd0>%s\n",
		$_[1] eq "/dev/fd0" ? "selected" : "",
		&text('select_fd', 0);
	$rv .= sprintf "<option %s value=/dev/fd1>%s\n",
		$_[1] eq "/dev/fd1" ? "selected" : "",
		&text('select_fd', 1);
	$found++ if ($_[1] =~ /^\/dev\/fd[01]$/);
	}
local @dlist = &list_disks();
foreach $d (@dlist) {
	local $dev = $d->{'device'};
	if ($_[2] == 1 || $_[2] == 2) {
		local $name = $d->{'desc'};
		$name .= " ($d->{'model'})" if ($d->{'model'});
		$rv .= sprintf "<option value=%s %s>%s\n",
			$dev, $_[1] eq $dev ? "selected" : "", $name;
		$found++ if ($dev eq $_[1]);
		}
	if ($_[2] == 0 || $_[2] == 2 || $_[2] == 3) {
		local @parts = &list_partitions($dev);
		foreach $p (@parts) {
			next if ($p->{'extended'});
			local $name = &text('select_part', uc($d->{'type'}),
					    uc(substr($dev, -1)),
					    $p->{'number'});
			$name = $p->{'desc'};
			$name .= " (".&tag_name($p->{'type'}).")"
				if (&tag_name($p->{'type'}));
			$rv .= sprintf "<option %s value=%s>%s\n",
				  $_[1] eq $p->{'device'} ? "selected" : "",
				  $p->{'device'}, $name;
			$found++ if ($_[1] eq $p->{'device'});
			}
		}
	}
if (!$found && $_[1] && !$_[3]) {
	$rv .= "<option selected>$_[1]\n";
	}
if ($_[3]) {
	${$_[3]} = $found;
	}
$rv .= "</select>\n";
return $rv;
}

#############################################################################
# Internal functions
#############################################################################
sub open_fdisk
{
local $fpath = &has_command("fdisk");
&error("<tt>fdisk</tt> command not found") if (!$fpath);
($fh, $fpid) = &foreign_call("proc", "pty_process_exec", $fpath, @_);
}

sub close_fdisk
{
close($fh); kill('TERM', $fpid);
}

sub wprint
{
syswrite($fh, $_[0], length($_[0]));
}

sub opt_input
{
print $_[2] ? "<tr>" : "";
print "<td align=right><b>$text{$_[0]}</b></td>\n";
print "<td nowrap><input type=radio name=$_[0]_def value=1 checked> ",
      $text{'opt_default'},"\n";
print "&nbsp; <input type=radio name=$_[0]_def value=0>\n";
print "<input name=$_[0] size=6> $_[1]</td>";
print $_[2] ? "\n" : "</tr>\n";
}

sub opt_check
{
if ($in{"$_[0]_def"}) { return ""; }
elsif ($in{$_[0]} !~ /^$_[1]$/) {
	&error(&text('opt_error', $in{$_[0]}, $text{$_[0]}));
	}
else { return " $_[2] $in{$_[0]}"; }
}

%tags = ('0', 'Empty',
	 '1', 'FAT12',
	 '2', 'XENIX root',
	 '3', 'XENIX usr',
	 '4', 'FAT16 <32M',
	 '6', 'FAT16',
	 '7', 'HPFS/NTFS',
	 '8', 'AIX',
	 '9', 'AIX bootable',
	 'a', 'OS/2 boot manager',
	 'b', 'Win95 FAT32',
	 'c', 'Win95 FAT32 LBA',
	 'e', 'Win95 FAT16 LBA',
	'10', 'OPUS',
	'11', 'Hidden FAT12',
	'12', 'Compaq diagnostic',
	'14', 'Hidden FAT16 < 32M',
	'16', 'Hidden FAT16',
	'17', 'Hidden HPFS/NTFS',
	'18', 'AST Windows swapfile',
	'1b', 'Hidden Win95 FAT (1b)',
	'1c', 'Hidden Win95 FAT (1c)',
	'1e', 'Hidden Win95 FAT (1e)',
	'24', 'NEC DOS',
	'3c', 'PartitionMagic recovery',
	'40', 'Venix 80286',
	'41', 'PPC PReP boot',
	'42', 'SFS',
	'4d', 'QNX 4.x',
	'4e', 'QNX 4.x 2nd partition',
	'4f', 'QNX 4.x 3rd partition',
	'50', 'OnTrack DM',
	'51', 'OnTrack DM6 Aux1',
	'52', 'CP/M',
	'53', 'OnTrack DM6 Aux3',
	'54', 'OnTrack DM6',
	'55', 'EZ-Drive',
	'56', 'Golden Bow',
	'5c', 'Priam Edisk',
	'61', 'SpeedStor',
	'63', 'GNU HURD or SysV',
	'64', 'Novell Netware 286',
	'65', 'Novell Netware 386',
	'70', 'DiskSecure Multi-Boot',
	'75', 'PC/IX',
	'80', 'Old Minix',
	'81', 'Minix / Old Linux / Solaris',
	'82', 'Linux swap',
	'83', 'Linux',
	'84', 'OS/2 hidden C: drive',
	'85', 'Linux extended',
	'86', 'NTFS volume set (86)',
	'87', 'NTFS volume set (87)',
	'93', 'Amoeba',
	'94', 'Amoeba BBT',
	'a0', 'IBM Thinkpad hibernation',
	'a5', 'BSD/386',
	'a6', 'OpenBSD',
	'a7', 'NeXTSTEP',
	'b7', 'BSDI filesystem',
	'b8', 'BSDI swap',
	'c1', 'DRDOS/sec FAT12',
	'c4', 'DRDOS/sec FAT16 <32M',
	'c6', 'DRDOS/sec FAT16',
	'c7', 'Syrinx',
	'db', 'CP/M / CTOS',
	'e1', 'DOS access',
	'e3', 'DOS read-only',
	'e4', 'SpeedStor',
	'eb', 'BeOS',
	'f1', 'SpeedStor',
	'f4', 'SpeedStor large partition',
	'f2', 'DOS secondary',
	'fd', 'Linux raid',
	'fe', 'LANstep',
	'ff', 'BBT'
	);

%hidden_tags = (
	 '5', 'Extended',
	 'f', 'Win95 extended LBA',
	);
	
%fstypes = ("ext2","Linux Native",
	    "minix","Minix",
	    "msdos","MS-DOS",
	    "vfat","Windows 95");

# can_edit_disk(device)
sub can_edit_disk
{
foreach (split(/\s+/, $access{'disks'})) {
        return 1 if ($_ eq "*" || $_ eq $_[0]);
        }
return 0;
}

