#include "fm.h"
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>

#include <signal.h>
#include <setjmp.h>

#include "html.h"
#include "Str.h"
#include "myctype.h"
#include "regex.h"

#define NOPROXY_NETADDR  /* allow IP address for no_proxy */

#ifdef INET6
/* see rc.c, "dns_order" and dnsorders[] */
int ai_family_order_table[3][3] = {
    { PF_UNSPEC, PF_UNSPEC, PF_UNSPEC },	/* 0:unspec */
    { PF_INET, PF_INET6, PF_UNSPEC },		/* 1:inet inet6 */
    { PF_INET6, PF_INET, PF_UNSPEC }		/* 2:inet6 inet */
};
#endif /* INET6 */

static JMP_BUF  AbortLoading;

static int
                DefaultPort[] = {
  80,				/* http */
  70,				/* gopher */
  21,				/* ftp */
  21,				/* ftpdir */
  0,				/* local - not defined */
  119,				/* nntp */
  119,				/* news */
  0,				/* mailto - not defined */
#ifdef USE_SSL
  443,                          /* https */
#endif
};

struct cmdtable schemetable[] = {
  {"http", SCM_HTTP},
  {"gopher", SCM_GOPHER},
  {"ftp", SCM_FTP},
  {"local", SCM_LOCAL},
  {"file", SCM_LOCAL},
  {"nntp", SCM_NNTP},
  {"news", SCM_NEWS},
  {"mailto", SCM_MAILTO},
#ifdef USE_SSL
  {"https", SCM_HTTPS},
#endif
  {NULL, SCM_UNKNOWN},
};

static struct table2 DefaultGuess[] = {
  {"html", "text/html"},
  {"HTML", "text/html"},
  {"htm", "text/html"},
  {"HTM", "text/html"},
  {"shtml", "text/html"},
  {"SHTML", "text/html"},
  {"gif", "image/gif"},
  {"GIF", "image/gif"},
  {"jpeg", "image/jpeg"},
  {"jpg", "image/jpeg"},
  {"JPEG", "image/jpeg"},
  {"JPG", "image/jpeg"},
  {"png", "image/png"},
  {"PNG", "image/png"},
  {"xbm", "image/xbm"},
  {"XBM", "image/xbm"},
  {"au", "audio/basic"},
  {"AU", "audio/basic"},
  {"gz", "application/x-gzip"},
  {"Z", "application/x-compress"},
  {"tar", "application/x-tar"},
  {"zip", "application/x-zip"},
  {"lha", "application/x-lha"},
  {"lzh", "application/x-lha"},
  {"LZH", "application/x-lha"},
  {"ps", "application/postscript"},
  {"pdf", "application/pdf"},
  {NULL, NULL}
};


/* #define HTTP_DEFAULT_FILE	"/index.html" */

#ifndef HTTP_DEFAULT_FILE
#define HTTP_DEFAULT_FILE "/"
#endif

static char *
DefaultFile(int scheme)
{
  switch(scheme) {
  case SCM_HTTP:
#ifdef USE_SSL
  case SCM_HTTPS:
#endif
    return allocStr(HTTP_DEFAULT_FILE, 0);
  case SCM_GOPHER:
    return allocStr("1", 0);
  case SCM_LOCAL:
  case SCM_FTP:
    return allocStr("/", 0);
  }
  return NULL;
}

static          MySignalHandler
KeyAbort(SIGNAL_ARG)
{
  LONGJMP(AbortLoading, 1);
}

#ifdef USE_SSL
SSL_CTX * ssl_ctx = NULL;

void 
free_ssl_ctx()
{
  if (ssl_ctx != NULL)
    SSL_CTX_free(ssl_ctx);
}

SSL *
openSSLHandle(int sock)
{
  SSL *handle;
  if (ssl_ctx == NULL) {
#if SSLEAY_VERSION_NUMBER < 0x0800
    ssl_ctx = SSL_CTX_new();
    X509_set_default_verify_paths(ssl_ctx->cert);
#else
    SSLeay_add_ssl_algorithms();
    ssl_ctx = SSL_CTX_new(SSLv23_client_method());
    SSL_CTX_set_options(ssl_ctx, SSL_OP_ALL);
    SSL_CTX_set_default_verify_paths(ssl_ctx);
#endif
    atexit(free_ssl_ctx);
  }
  handle = SSL_new(ssl_ctx);
  SSL_set_fd(handle, sock);
  if (SSL_connect(handle) <= 0)
    return NULL;
  return handle;
}

static void
SSL_write_from_file(SSL *ssl, char *file)
{
  FILE *fd;
  int c;
  char buf[1];
  fd = fopen(file, "r");
  if (fd != NULL) {
    while((c = fgetc(fd)) != EOF) {
      buf[0] = c;
      SSL_write(ssl, buf, 1);
    }
    fclose(fd);
  }
}

#endif

static void
write_from_file(int sock, char *file)
{
  FILE *fd;
  int c;
  char buf[1];
  fd = fopen(file, "r");
  if (fd != NULL) {
    while((c = fgetc(fd)) != EOF) {
      buf[0] = c;
      write(sock, buf, 1);
    }
    fclose(fd);
  }
}

ParsedURL *
baseURL(Buffer *buf)
{
  if (buf->bufferprop & BP_NO_URL) {
    /* no URL is defined for the buffer */
    return NULL;
  }
  if (buf->baseURL != NULL) {
    /* <BASE> tag is defined in the document */
    return buf->baseURL;
  }
  else
    return &buf->currentURL;
}

int
openSocket(char *hostname,
	   char *remoteport_name,
	   unsigned short remoteport_num)
{
  int             sock;
#ifdef INET6
  int *af;
  struct addrinfo hints, *res0, *res;
  int error;
#else
  struct sockaddr_in hostaddr;
  struct hostent *entry;
  struct servent *serv;
  struct protoent *proto;
  int             a1, a2, a3, a4;
  unsigned long   adr;
#endif
  MySignalHandler(*trap) ();

  if (SETJMP(AbortLoading) != 0) {
    goto error;
  }
  trap = signal(SIGINT, KeyAbort);
  if (fmInitialized)
    term_cbreak();
  if (hostname == NULL)
    goto error;

#ifdef INET6
  for (af = ai_family_order_table[DNS_order]; ; af++) {
      memset(&hints, 0, sizeof(hints));
      hints.ai_family = *af;
      hints.ai_socktype = SOCK_STREAM;
      if (remoteport_num != 0) {
	  char portbuf[10];
	  snprintf(portbuf, sizeof(portbuf), "%d", remoteport_num);
	  error = getaddrinfo(hostname, portbuf, &hints, &res0);
      } else {
	  error = -1;
      }
      if (error && remoteport_name && remoteport_name[0] != '\0') {
	  /* try default port */
	  error = getaddrinfo(hostname, remoteport_name, &hints, &res0);
      }
      if (error) {
	  if (*af == PF_UNSPEC) {
	      goto error;
	  }
	  /* try next ai family */
	  continue;
      }
      sock = -1;
      for (res = res0; res; res = res->ai_next) {
	  sock = socket(res->ai_family, res->ai_socktype, res->ai_protocol);
	  if (sock < 0) {
	      continue;
	  }
	  if (connect(sock, res->ai_addr, res->ai_addrlen) < 0) {
	      close(sock);
	      sock = -1;
	      continue;
	  }
	  break;
      }
      if (sock < 0) {
	  freeaddrinfo(res0);
	  if (*af == PF_UNSPEC) {
	      goto error;
	  }
	  /* try next ai family */
	  continue;
      }
      freeaddrinfo(res0);
      break;
  }
#else
  if ((serv = getservbyname(remoteport_name, "tcp")) == NULL) {
    serv = New(struct servent);
  }
  serv->s_port = htons(remoteport_num);
  bzero((char *) &hostaddr, sizeof(struct sockaddr_in));
  if ((proto = getprotobyname("tcp")) == NULL) {
    /* protocol number of TCP is 6 */
    proto = New(struct protoent);
    proto->p_proto = 6;
  }
  if ((sock = socket(AF_INET, SOCK_STREAM, proto->p_proto)) < 0) {
    goto error;
  }
  regexCompile("[0-9][0-9]*\\.[0-9][0-9]*\\.[0-9][0-9]*\\.[0-9][0-9]*",0);
  if (regexMatch(hostname,1)) {
    sscanf(hostname,"%d.%d.%d.%d",&a1,&a2,&a3,&a4);
    adr = htonl((a1<<24)|(a2<<16)|(a3<<8)|a4);
    bcopy((void*)&adr,(void*)&hostaddr.sin_addr,sizeof(long));
    hostaddr.sin_family = AF_INET;
    hostaddr.sin_port = serv->s_port;
    if (connect(sock, (struct sockaddr *) & hostaddr,
                sizeof(struct sockaddr_in)) < 0) {
      goto error;
    }
  }
  else {
    char **h_addr_list ;
    int result ;
    if ((entry = gethostbyname(hostname)) == NULL) {
      goto error;
    }
    hostaddr.sin_family = AF_INET;
    hostaddr.sin_port = serv->s_port;
    for ( h_addr_list = entry->h_addr_list ; *h_addr_list ; h_addr_list ++ )
    {
      bcopy((void*)h_addr_list[0], (void*)&hostaddr.sin_addr, entry->h_length);

      if ((result=connect(sock, (struct sockaddr *) & hostaddr,
                          sizeof(struct sockaddr_in))) == 0) {
        break;
      }
    }
    if ( result < 0 )
      goto error;
  }
#endif

  if (fmInitialized)
    term_raw();
  signal(SIGINT, trap);
  return sock;
error:
  if (fmInitialized)
    term_raw();
  signal(SIGINT, trap);
  return -1;

}


void
parseURL(char *url, ParsedURL * p_url, ParsedURL * current)
{
  char           *p, *q;
  char            tmp[LINELEN];
  int             i;

  p = url;
  p_url->scheme = SCM_MISSING;
  p_url->port = 0;
  p_url->user = NULL;
  p_url->pass = NULL;
  p_url->host = NULL;
  p_url->is_nocache = 0;
  p_url->file = NULL;
  p_url->label = NULL;

  if (*url == '#') { /* label only */
    if (current)
      copyParsedURL(p_url,current);
    goto do_label;
  }
  /* search for scheme */
  while (*p && (isalpha(*p) || *p == '.' || *p == '+' || *p == '-'))
    p++;
  if (*p == ':') {		/* scheme found */
    p_url->scheme = SCM_UNKNOWN;
    for(i = 0; (q = schemetable[i].cmdname) != NULL; i++) {
      int len = strlen(q);
      if (!strncasecmp(q, url, len) && url[len] == ':') {
	p_url->scheme = schemetable[i].cmd;
	break;
      }
    }
    p++;
  } else {			/* scheme not found */
    if (current)
      copyParsedURL(p_url,current);
    else
      p_url->scheme = SCM_LOCAL;
    p = url;
    goto analyze_file;
  }
  /* get host and port */
  if (p[0] != '/' || p[1] != '/') {	/* scheme:foo or scheme:/foo */
    p_url->host = NULL;
    if (p_url->scheme != SCM_UNKNOWN)
      p_url->port = DefaultPort[p_url->scheme];
    else
      p_url->port = 0;
    goto analyze_file;
  }
  if (p_url->scheme == SCM_LOCAL) {	/*   file://foo		*/
    if (p[2] == '/' || p[2] == '~') {	/*   file:///foo	*/
      p += 2;				/*   file://~user	*/
      goto analyze_file;
    }
#ifdef CYGWIN
    goto analyze_file;			/*   file://DRIVE/foo or file://machine/share/foo	*/
#endif
  }
  p += 2;				/* scheme://foo		*/
analyze_url:
  q = p;
  while (*p && *p != ':' && *p != '/' && *p != '@')
    p++;
  switch (*p) {
  case '\0':				/* scheme://host		*/
					/* scheme://user@host		*/
					/* scheme://user:pass@host	*/
    p_url->host = allocStr(q, 0);
    p_url->port = DefaultPort[p_url->scheme];
    p_url->file = DefaultFile(p_url->scheme);
    p_url->label = NULL;
    return;
  case ':':
    p_url->host = allocStr(q, p - q);
    q = ++p;
    while (*p && *p != '/' && *p != '@')
      p++;
    if(*p == '@'){			/* scheme://user:pass@...	*/
      p_url->pass = allocStr(q, p - q);
      q = ++p;
      p_url->user = p_url->host;
      p_url->host = NULL;
      goto analyze_url;
    }
    strncpy(tmp, q, p - q);
    tmp[p - q] = '\0';
    p_url->port = atoi(tmp);
    if (*p == '\0') {			/* scheme://host:port		*/
					/* scheme://user@host:port	*/
					/* scheme://user:pass@host:port	*/
      p_url->file = DefaultFile(p_url->scheme);
      p_url->label = NULL;
      return;
    }
    break;
  case '@':				/* scheme://user@...		*/
    p_url->user = allocStr(q, p - q);
    q = ++p;
    goto analyze_url;
  case '/':				/* scheme://host/...		*/
					/* scheme://user@host/...	*/
					/* scheme://user:pass@host/...	*/
    p_url->host = allocStr(q, p - q);
    p_url->port = DefaultPort[p_url->scheme];
    break;
  }
analyze_file:
  if (p_url->scheme == SCM_LOCAL && p_url->user == NULL &&
      p_url->host != NULL && strcmp(p_url->host, "localhost")) {
    p_url->scheme = SCM_FTP;		/* ftp://host/...		*/
    if (p_url->port == 0)
      p_url->port = DefaultPort[SCM_FTP];
  }

  q = p;
  if (p_url->scheme == SCM_GOPHER) {
    if (*q == '/')
      q++;
    if (*q && q[0] != '/' && q[1] != '/' && q[2] == '/')
      q++;
  }
  if (*p == '/')
    p++;
  if (*p == '\0') {			/* scheme://host[:port]/ */
    p_url->file = DefaultFile(p_url->scheme);
    p_url->label = NULL;
    return;
  }
  if (p_url->scheme == SCM_GOPHER && *p == 'R') {
    p++;
    q = tmp;
    *q++ = *p++;
    while (*p && *p != '/')
      p++;
    while (*p)
      *q++ = *p++;
    *q = '\0';
    p_url->file = allocStr(tmp, 0);
  } else {
    char *cgi = strchr(p, '?');
  again:
    while (*p && *p != '#')
      p++;
    if (*p == '#' && p_url->scheme == SCM_LOCAL) {
      /*
       * According to RFC2396, # means the beginning of
       * URI-reference, and # should be escaped.  But,
       * if the scheme is SCM_LOCAL, the special
       * treatment will apply to # for convinience.
       */
      if (p > q && *(p-1) == '/' && (cgi == NULL || p < cgi)) {
	/* # comes as the first character of the file name */
        /* that means, # is not a label but a part of the  */
        /* file name.                                      */
        p++;
	goto again;
      }
      else if (*(p+1) == '\0') {
	/* # comes as the last character of the file name  */
        /* that means, # is not a label but a part of the  */
        /* file name.                                      */
        p++;
      }
    }
    p_url->file = allocStr(q, p - q);
  }
do_label:
  if (p_url->scheme == SCM_MISSING) {
    p_url->scheme = SCM_LOCAL;
    p_url->file = allocStr(p, 0);
    p_url->label = NULL;
  } else if (*p == '#')
    p_url->label = allocStr(p + 1, 0);
  else
    p_url->label = NULL;
}

#define initParsedURL(p) bzero(p,sizeof(ParsedURL))

void
copyParsedURL(ParsedURL * p, ParsedURL * q)
{
  p->scheme = q->scheme;
  p->port = q->port;
  p->is_nocache = q->is_nocache;
  if (q->user)
    p->user = allocStr(q->user, 0);
  else
    p->user = NULL;
  if (q->pass)
    p->pass = allocStr(q->pass, 0);
  else
    p->pass = NULL;
  if (q->host)
    p->host = allocStr(q->host, 0);
  else
    p->host = NULL;
  if (q->file)
    p->file = allocStr(q->file, 0);
  else
    p->file = NULL;
  if (q->label)
    p->label = allocStr(q->label, 0);
  else
    p->label = NULL;
}

void
parseURL2(char *url, ParsedURL *pu, ParsedURL *current)
{
  char *p, *q;
  Str tmp;

  parseURL(url, pu, current);
  if (pu->scheme == SCM_MAILTO)
    return;

  if (pu->scheme == SCM_LOCAL && *pu->file == '~')
    pu->file = expandName(pu->file);

  if (current && pu->scheme == current->scheme) {
    if (pu->user == NULL) {
      pu->user = current->user;
    }
    if (pu->pass == NULL) {
      pu->pass = current->pass;
    }
    if (pu->host == NULL) {
      pu->host = current->host;
    }
    if (pu->file) {
      if (pu->scheme != SCM_GOPHER &&
	  pu->scheme != SCM_NEWS && 
	  pu->file[0] != '/') {
        p = pu->file;
	if (current->file) {
	  tmp = Strnew_charp(current->file);
	  if ((q = strchr(tmp->ptr,'?')) != NULL)
	    Strshrink(tmp,(tmp->ptr+tmp->length)-q);
	  while (tmp->length > 0) {
            if (Strlastchar(tmp) == '/')
              break;
            Strshrink(tmp,1);
	  }
	  Strcat_charp(tmp, p);
	  pu->file = allocStr(tmp->ptr, 0);
	}
      } else if (pu->scheme == SCM_GOPHER &&
		 pu->file[0] == '/') {
	p = pu->file;
	pu->file = allocStr(p + 1, 0);
      }
    } 
    else if (pu->label) {
      /* pu has only label */
      pu->file = current->file;
    }
  }
  if (pu->file) {
    if (pu->scheme == SCM_LOCAL && pu->file[0] != '/') {
      tmp = Strnew_charp(currentdir());
      if (Strlastchar(tmp) != '/')
	Strcat_char(tmp,'/');
      Strcat_charp(tmp,pu->file);
      pu->file = cleanupName(tmp->ptr);
    } else if (pu->scheme != SCM_GOPHER && pu->scheme != SCM_NEWS &&
	       pu->file[0] == '/') {
      pu->file = cleanupName(pu->file);
    }
  }
}

Str
parsedURL2Str(ParsedURL *pu)
{
  Str tmp = Strnew();
  static char *scheme_str[] = {
    "http","gopher","ftp","ftp","file","nntp","news","mailto",
#ifdef USE_SSL
    "https",
#endif
    };

  if (pu->scheme == SCM_UNKNOWN || pu->scheme == SCM_MISSING) {
    return Strnew_charp("???");
  }
  if (pu->host == NULL && pu->file == NULL && pu->label != NULL) {
    /* local label */
    return Sprintf("#%s",pu->label);
  }
  tmp = Strnew_charp(scheme_str[pu->scheme]);
  Strcat_char(tmp,':');
  if (pu->scheme == SCM_MAILTO) {
    Strcat_charp(tmp,pu->file);
    return tmp;
  }
  if (pu->scheme != SCM_NEWS) {
    Strcat_charp(tmp,"//");
  }
  if (pu->user) {
    Strcat_charp(tmp,pu->user);
/*
    if (pu->pass) {
      Strcat_char(tmp,':');
      Strcat_charp(tmp,pu->pass);
    }
*/
    Strcat_char(tmp,'@');
  }
  if (pu->host) {
    Strcat_charp(tmp,pu->host);
    if (pu->port != DefaultPort[pu->scheme]) {
      Strcat_char(tmp,':');
      Strcat(tmp,Sprintf("%d",pu->port));
    }
  }
  if (pu->file == NULL || pu->file[0] != '/')
    Strcat_char(tmp,'/');
  Strcat_charp(tmp,pu->file);
  if (pu->label) {
    Strcat_char(tmp,'#');
    Strcat_charp(tmp,pu->label);
  }
  return tmp;
}  

static char*
otherinfo(ParsedURL *target, ParsedURL *current, char *referer)
{
  Str s = Strnew();

  Strcat_charp(s,"User-Agent: ");
  if (UserAgent == NULL || *UserAgent == '\0')
     Strcat_charp(s,version);
  else
     Strcat_charp(s,UserAgent);
  Strcat_charp(s,"\r\n");
  Strcat_charp(s,"Accept: text/*, image/*, audio/*, application/*\r\n");
  Strcat_charp(s,"Accept-Language: ");
  if (AcceptLang != NULL && *AcceptLang != '\0') {
    Strcat_charp(s,AcceptLang);
    Strcat_charp(s,"\r\n");
  }
  else {
#if LANG == JA
    Strcat_charp(s,"ja; q=1.0, en; q=0.5\r\n");
#else  /* must be EN */
    Strcat_charp(s,"en; q=1.0\r\n");
#endif
  }
  if (target->host) {
    Strcat_charp(s,"Host: ");
    Strcat_charp(s,target->host);	
    if (target->port != DefaultPort[target->scheme])
      Strcat(s,Sprintf(":%d",target->port));
    Strcat_charp(s,"\r\n");
  }
  if (target->is_nocache) {
    Strcat_charp(s, "Pragma: no-cache\r\n");
    Strcat_charp(s, "Cache-control: no-cache\r\n");
  }
 if(!NoSendReferer){
  if (referer == NULL && current && current->scheme != SCM_LOCAL &&
      (current->scheme != SCM_FTP ||
       (current->user == NULL && current->pass == NULL))) {
    Strcat_charp(s,"Referer: ");
    Strcat(s,parsedURL2Str(current));
    Strcat_charp(s,"\r\n");
  }
  else if (referer != NULL && referer != NO_REFERER) {
    Strcat_charp(s,"Referer: ");
    Strcat_charp(s,referer);
    Strcat_charp(s,"\r\n");
  }
 }
  return s->ptr;
}

Str
HTTPrequest(ParsedURL *pu, ParsedURL *current, HRequest *hr, TextList *extra)
{
  Str tmp;
  TextListItem *i;
#ifdef USE_COOKIE
  Str cookie;
#endif
  switch (hr->command) {
  case HR_COMMAND_CONNECT:
    tmp = Strnew_charp("CONNECT ");
    break;
  case HR_COMMAND_POST:
    tmp = Strnew_charp("POST ");
    break;
  case HR_COMMAND_HEAD:
    tmp = Strnew_charp("HEAD ");
    break;
  case HR_COMMAND_GET:
  default:
    tmp = Strnew_charp("GET ");
  }
  if (hr->command == HR_COMMAND_CONNECT) {
    Strcat_charp(tmp,pu->host);
    Strcat(tmp,Sprintf(":%d",pu->port));
  }
  else if (hr->flag & HR_FLAG_LOCAL) {
    Strcat_charp(tmp,pu->file);
  }
  else {
    Strcat(tmp,parsedURL2Str(pu));
  }
  Strcat_charp(tmp," HTTP/1.0\r\n");
  if (hr->referer == NO_REFERER)
    Strcat_charp(tmp,otherinfo(pu,NULL,NULL));
  else
    Strcat_charp(tmp,otherinfo(pu,current,hr->referer));
  if (extra != NULL)
    for (i = extra->first; i != NULL; i = i->next)
      Strcat_charp(tmp,i->ptr);
#ifdef USE_COOKIE
  if (hr->command != HR_COMMAND_CONNECT &&
      use_cookie && (cookie = find_cookie(pu))) {
    Strcat_charp(tmp, "Cookie: ");
    Strcat(tmp, cookie);  
    Strcat_charp(tmp,"\r\n");
  }
#endif
  if (hr->command == HR_COMMAND_POST) {
    if (hr->request->enctype == FORM_ENCTYPE_MULTIPART){
      Strcat_charp(tmp,"Content-type: multipart/form-data; boundary=");
      Strcat_charp(tmp,hr->request->boundary);
      Strcat_charp(tmp,"\r\n");
      Strcat(tmp,Sprintf("Content-length: %ld\r\n",hr->request->length));
      Strcat_charp(tmp,"\r\n");
    } else {
      Strcat_charp(tmp,"Content-type: application/x-www-form-urlencoded\r\n");
      Strcat(tmp,Sprintf("Content-length: %ld\r\n",hr->request->length));
      Strcat_charp(tmp,"\r\n");
      Strcat_charp(tmp,hr->request->body);
      Strcat_charp(tmp,"\r\n");
    }
  }
  else
    Strcat_charp(tmp,"\r\n");
#ifdef DEBUG
  fprintf(stderr, "HTTPrequest: [ %s ]\n\n", tmp->ptr);
#endif
  return tmp;
}

URLFile
openURL(char *url, ParsedURL * pu, ParsedURL * current,
	URLOption *option, FormList *request, TextList *extra_header, 
	URLFile *ouf, char *status)
{
  Str            tmp = Strnew();
  FILE           *f, *fw;
  int             i, sock;
  char           *p,*q;
  URLFile         uf;
  char           *last_dot;
  HRequest        hr;
#ifdef USE_SSL
  SSL            *sslh;
#endif

  if (ouf) {
    uf = *ouf;
  } else {
    uf.scheme = SCM_MISSING;
    uf.stream.f = NULL;
    uf.stream_type = SMT_FILE;
    uf.iseof = NIL;
    uf.close = (void(*)())fclose;
    uf.encoding = ENC_7BIT;
    uf.is_cgi = NIL;
    uf.gzip = 0;
  }

retry:
  parseURL2(url,pu,current);
  if (pu->scheme == SCM_LOCAL && pu->file == NULL) {
    if (pu->label != NULL) {
      /* #hogege is not a label but a filename */
      Str tmp2 = Strnew_charp("#");
      Strcat_charp(tmp2,pu->label);
      pu->file = tmp2->ptr;
      pu->label = NULL;
    }
    else {
      /* given URL must be null string */
      return uf;
    }
  }

  uf.scheme = pu->scheme;
  pu->is_nocache = (option->flag & RG_NOCACHE);
  last_dot = "";
  for (p= pu->file; p && *p; p++) {
    if ( *p == '.') {
      last_dot = p;
    }
    else if (*p == '?')
      break;
  }
  if (*last_dot == '.') last_dot++;
  for (i = 0; last_dot[i] && IS_ALPHA(last_dot[i]); i++);
  uf.ext = allocStr(last_dot, i);

  hr.command = HR_COMMAND_GET;
  hr.flag = 0;
  hr.referer = option->referer;
  hr.request = request;
  
  switch (pu->scheme) {
  case SCM_LOCAL:
    if (request && request->body) {
      /* local CGI: POST */
      uf.stream.f = localcgi_post(pu->file,request);
      if (uf.stream.f == NULL) goto ordinary_local_file;
      uf.close = (void(*)())pclose;	
      uf.is_cgi = T;
    }	
    else if ((p = strchr(pu->file,'?')) != NULL) {
      /* lodal CGI: GET */
      for (q = pu->file; *q && *q != '?'; q++)
        Strcat_char(tmp,*q);
      uf.stream.f = localcgi_get(tmp->ptr,p+1);
      if (uf.stream.f == NULL) {
	pu->file = tmp->ptr;
	goto ordinary_local_file;
      }
      uf.close = (void(*)())pclose; 
      uf.is_cgi = T;
    }      
    else if (!strncmp(pu->file+strlen(pu->file)-4,".cgi",4)) {
      /* lodal CGI: GET */
      uf.stream.f = localcgi_get(pu->file,"");
      if (uf.stream.f == NULL) goto ordinary_local_file;
      uf.close = (void(*)())pclose; 
      uf.is_cgi = T;
    }      
    else {
    ordinary_local_file:
      uf.stream.f = examineFile(pu->file,&uf.close);
    }
    if (uf.stream.f == NULL && document_root != NULL) {
      Strcat_charp(tmp,document_root);
      if (Strlastchar(tmp) != '/')
	Strcat_char(tmp,'/');
      Strcat_charp(tmp,pu->file);
      uf.stream.f = examineFile(tmp->ptr,&uf.close);
    }
    if (uf.stream.f == NULL && strncasecmp(url,"http://",7) != 0) {
      /* retry it as http:// */
      url = Strnew_m_charp("http://",url,NULL)->ptr;
      goto retry;
    }
    return uf;
  case SCM_FTP:
    if (pu->file == NULL)
      pu->file = allocStr("/",0);
    if (FTP_proxy != NULL && 
	!Do_not_use_proxy &&
	!check_no_proxy(pu->host)) {
      sock = openSocket(FTP_proxy_parsed.host,
			schemetable[FTP_proxy_parsed.scheme].cmdname,
			FTP_proxy_parsed.port);
      if (sock < 0)
	goto no_ftp_proxy;
      tmp = HTTPrequest(pu,current,&hr,NULL);
      write(sock, tmp->ptr, tmp->length);
    } else {
  no_ftp_proxy:
      uf.stream.f = openFTP(pu);
      uf.scheme = pu->scheme;
      uf.close = (void(*)())closeFTP ;
      return uf;
    }
    break;
  case SCM_HTTP:
#ifdef USE_SSL
  case SCM_HTTPS:
#endif
    if (pu->file == NULL)
      pu->file = allocStr("/",0);
    if (request && request->method == FORM_METHOD_POST && request->body)
      hr.command = HR_COMMAND_POST;
    if (request && request->method == FORM_METHOD_HEAD)
      hr.command = HR_COMMAND_HEAD;
    if (HTTP_proxy != NULL && 
	!Do_not_use_proxy &&
	!check_no_proxy(pu->host)) {
      char *save_label;
#ifdef USE_SSL
      if (pu->scheme == SCM_HTTPS && *status == HTST_CONNECT) {
	sock = ouf->stream.ss->sock;
	if (!(sslh = openSSLHandle(sock))) {
	  *status = HTST_MISSING;
	  return uf;
	}
      } else {
	sock = openSocket(HTTP_proxy_parsed.host,
			  schemetable[HTTP_proxy_parsed.scheme].cmdname,
			  HTTP_proxy_parsed.port);
	sslh = NULL;
      }
#else
      sock = openSocket(HTTP_proxy_parsed.host,
			schemetable[HTTP_proxy_parsed.scheme].cmdname,
			HTTP_proxy_parsed.port);
#endif
      if (sock < 0)
	goto no_http_proxy;
      save_label = pu->label;
      pu->label = NULL;
#ifdef USE_SSL
      if (pu->scheme == SCM_HTTPS) {
	if(*status == HTST_NORMAL) {
	  hr.command = HR_COMMAND_CONNECT;
	  tmp = HTTPrequest(pu,current,&hr,NULL);
	  *status = HTST_CONNECT;
	} else {
	  hr.flag |= HR_FLAG_LOCAL;
	  tmp = HTTPrequest(pu,current,&hr,extra_header);
	  *status = HTST_NORMAL;
	}
      } else
#endif
      {
        tmp = HTTPrequest(pu,current,&hr,extra_header);
        *status = HTST_NORMAL;
        pu->label = save_label;
      }
    } else {
  no_http_proxy:
      sock = openSocket(pu->host,
			schemetable[pu->scheme].cmdname,
			pu->port);
      if (sock < 0) {
	*status = HTST_MISSING;
	return uf;
      }
#ifdef USE_SSL
      if (pu->scheme == SCM_HTTPS) {
	if (!(sslh = openSSLHandle(sock))) {
	  *status = HTST_MISSING;
	  return uf;
	}
      }
#endif
      hr.flag |= HR_FLAG_LOCAL;
      tmp = HTTPrequest(pu,current,&hr,extra_header);
      *status = HTST_NORMAL;
    }
#ifdef USE_SSL
    if (pu->scheme == SCM_HTTPS) {
      uf.stream_type = SMT_SSL;
      uf.stream.ss = newSSLStream(sslh, sock);
      if (sslh)
        SSL_write(sslh, tmp->ptr, tmp->length);
      else
        write(sock, tmp->ptr, tmp->length);
      if (hr.command == HR_COMMAND_POST &&
	request->enctype == FORM_ENCTYPE_MULTIPART) {
	if (sslh)
	  SSL_write_from_file(sslh, request->body);
	else
	  write_from_file(sock, request->body);
      }
      return uf;
    } else
#endif
    {
      write(sock, tmp->ptr, tmp->length);
#ifdef HTTP_DEBUG
      {
	FILE *ff = fopen("zzrequest","a");
	fwrite(tmp->ptr,sizeof(char),tmp->length,ff);
	fclose(ff);
      }
#endif
      if (hr.command == HR_COMMAND_POST &&
	  request->enctype == FORM_ENCTYPE_MULTIPART)
	write_from_file(sock, request->body);
    }
    break;
  case SCM_GOPHER:
    if (GOPHER_proxy != NULL && 
	!Do_not_use_proxy &&
	!check_no_proxy(pu->host)) {
      sock = openSocket(GOPHER_proxy_parsed.host,
			schemetable[GOPHER_proxy_parsed.scheme].cmdname,
			GOPHER_proxy_parsed.port);
      if (sock < 0)
	goto no_gopher_proxy;
      tmp = HTTPrequest(pu,current,&hr,NULL);
    } else {
  no_gopher_proxy:
      sock = openSocket(pu->host,
			schemetable[pu->scheme].cmdname,
			pu->port);
      if (sock < 0)
	return uf;
      if (pu->file == NULL)
	pu->file = "1";
      tmp = Strnew_charp(pu->file);
      Strcat_char(tmp,'\n');
    }
    write(sock, tmp->ptr, tmp->length);
    break;
  case SCM_NEWS:
    p = getenv("NNTPSERVER");
    if (p == NULL)
      return uf;
    sock = openSocket(p, "nntp", pu->port);
    if (sock < 0)
      return uf;
    f = fdopen(sock, "r");
    fw = fdopen(sock, "w");
    if (f == NULL || fw == NULL)
      return uf;
    tmp = Strfgets(f);
    if (tmp->length == 0)
      goto nntp_error;
    sscanf(tmp->ptr, "%d", &i);
    if (i != 200)
      goto nntp_error;
    fprintf(fw, "ARTICLE <%s>\r\n", pu->file);
    fflush(fw);
    tmp = Strfgets(f);
    if (tmp->length == 0)
      goto nntp_error;
    sscanf(tmp->ptr, "%d", &i);
    if (i != 220)
      goto nntp_error;
    uf.stream.f = f;
    uf.close = (void(*)())fclose;
    return uf;
nntp_error:
    fclose(f);
    fclose(fw);
    return uf;
  case SCM_UNKNOWN:
    return uf;
  }
  uf.stream.f = fdopen(sock, "r");
  uf.close = (void(*)())fclose;
  return uf;
}

char           *
guessContentTypeFromTable(struct table2 * table, char *filename)
{
  char           *p;
  p = &filename[strlen(filename) - 1];
  while (filename < p && *p != '.')
    p--;
  if (p == filename)
    return NULL;
  p++;
  while (table->item1) {
    if (!strcmp(p, table->item1))
      return table->item2;
    table++;
  }
  return NULL;
}

char           *
guessContentType(char *filename)
{
  if (filename == NULL)
    return NULL;
  return guessContentTypeFromTable(DefaultGuess, filename);
}

void
set_no_proxy(char *no_proxy_list)
{
  char *p;
  Str tmp;

  NO_proxy_domains = newTextList();
  p = no_proxy_list;
  tmp = Strnew_size(64);
  while (*p) {
    while (*p && IS_SPACE(*p)) p++;
    Strclear(tmp);
    while (*p && !IS_SPACE(*p) && *p != ',')
      Strcat_char(tmp,*p++);
    if (tmp->length > 0)
      pushText(NO_proxy_domains,tmp->ptr);
    while (*p && IS_SPACE(*p)) p++;
    if (*p == ',') p++;
  }
}

static int
domain_match(char *pat, char *domain)
{
  if (*pat == '.')
    pat++;
  for (;;) {
    if (!strcasecmp(pat,domain))
      return 1;
    domain = strchr(domain,'.');
    if (domain == NULL)
      return 0;
    domain++;
  }
}

int
check_no_proxy(char *domain)
{
  TextListItem *tl;

  if (NO_proxy_domains == NULL || NO_proxy_domains->nitem == 0)
    return 0;
  for (tl = NO_proxy_domains->first; tl != NULL; tl = tl->next) {
    if (domain_match(tl->ptr,domain))
      return 1;
  }
#ifdef NOPROXY_NETADDR
  if (! NOproxy_netaddr) {
      return 0;
  }
  /*
   * to check noproxy by network addr
   */
  {
#ifndef INET6
    struct hostent *he ;
    int n ;
    unsigned char **h_addr_list ;
    char addr[4*16], buf[5] ;

    he = gethostbyname( domain ) ;
    if ( !he )
        return( 0 ) ;
    for ( h_addr_list = (unsigned char**)he->h_addr_list
              ; *h_addr_list ; h_addr_list ++ )
    {
      sprintf( addr, "%d", h_addr_list[0][0] ) ;
      for ( n = 1 ; n < he->h_length ; n ++ )
      {
        sprintf( buf, ".%d", h_addr_list[0][n] ) ;
        strcat( addr, buf ) ;
      }
      for (tl = NO_proxy_domains->first; tl != NULL; tl = tl->next) {
        if ( strncmp(tl->ptr,addr,strlen(tl->ptr)) == 0 )
            return( 1 ) ;
      }
    }
#else /* INET6 */
    int error;
    struct addrinfo hints;
    struct addrinfo *res, *res0;
    char addr[4*16];
    int *af;

    for (af = ai_family_order_table[DNS_order]; ; af++) {
	memset(&hints, 0, sizeof(hints));
	hints.ai_family = *af;
	error = getaddrinfo(domain, NULL, &hints, &res0);
	if (error) {
	    if (*af == PF_UNSPEC) {
		break;
	    }
	    /* try next */
	    continue;
	}
	for (res = res0; res != NULL; res = res->ai_next) {
	    switch (res->ai_family) {
	    case AF_INET:
		inet_ntop(AF_INET, 
			  &((struct sockaddr_in *)res->ai_addr)->sin_addr,
			  addr, sizeof(addr));
		break;
	    case AF_INET6:
		inet_ntop(AF_INET6,
			  &((struct sockaddr_in6 *)res->ai_addr)->sin6_addr,
			  addr, sizeof(addr));
		break;
	    default:
		/* unknown */
		continue;
	    }
	    for (tl = NO_proxy_domains->first; tl != NULL; tl = tl->next) {
		if (strncmp(tl->ptr, addr, strlen(tl->ptr)) == 0) {
		    freeaddrinfo(res0);
		    return 1;
		}
	    }
	}
	freeaddrinfo(res0);
	if (*af == PF_UNSPEC) {
	    break;
	}
    }
#endif /* INET6 */
  }
#endif /* NOPROXY_NETADDR */
  return 0;
}
