/* $Id: terms.c,v 1.6 2000/01/21 08:32:03 aito Exp $ */
/*
 * An original curses library for EUC-kanji by Akinori ITO,     December 1989
 * revised by Akinori ITO, January 1995
 */
#include <stdio.h>
#include <signal.h>
#include <sys/types.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/time.h>
#include <unistd.h>
#include "config.h"
#include <string.h>
#ifdef USE_GPM
#include <gpm.h>
#endif
#ifdef MOUSE
static int is_xterm = 0;
#endif

#ifdef AIX
#include <sys/select.h>
#endif

#include "terms.h"
#include "fm.h"

char           *getenv(const char *);
MySignalHandler   reset_exit(SIGNAL_ARG), error_dump(SIGNAL_ARG);
void            setlinescols(void);
void flush_tty();

#ifdef MOUSE
void mouse_init(),mouse_end();
int mouseActive = 0;
#endif

#ifndef SIGIOT
#define SIGIOT SIGABRT
#endif

#ifdef TERMIO
#include        <sys/ioctl.h>
#include        <termio.h>
typedef struct termio TerminalMode;
#define TerminalSet(fd,x)       ioctl(fd,TCSETA,x)
#define TerminalGet(fd,x)       ioctl(fd,TCGETA,x)
#define MODEFLAG(d)     ((d).c_lflag)
#define IMODEFLAG(d)    ((d).c_iflag)
#endif

#ifdef TERMIOS
#include <termios.h>
#include <unistd.h>
typedef struct termios TerminalMode;
#define TerminalSet(fd,x)       tcsetattr(fd,TCSANOW,x)
#define TerminalGet(fd,x)       tcgetattr(fd,x)
#define MODEFLAG(d)     ((d).c_lflag)
#define IMODEFLAG(d)    ((d).c_iflag)
#endif

#ifdef SGTTY
#include        <sys/ioctl.h>
#include        <sgtty.h>
typedef struct sgttyb TerminalMode;
#define TerminalSet(fd,x)       ioctl(fd,TIOCSETP,x)
#define TerminalGet(fd,x)       ioctl(fd,TIOCGETP,x)
#define MODEFLAG(d)     ((d).sg_flags)
#endif

#define MAX_LINE        200
#define MAX_COLUMN      300

/* Screen properties */
#define S_SCREENPROP    0x0f
#define S_NORMAL        0x00
#define S_STANDOUT      0x01
#define S_UNDERLINE     0x02
#define S_BOLD          0x04
#define S_EOL           0x08

/* Sort of Character */
#define C_WHICHCHAR     0xc0
#define C_ASCII         0x00
#ifdef JP_CHARSET
#define C_WCHAR1        0x40
#define C_WCHAR2        0x80
#endif
#define C_CTRL          0xc0

#define CHMODE(c)       ((c)&C_WHICHCHAR)
#define SETCHMODE(var,mode)     var = (((var)&~C_WHICHCHAR) | mode)

/* Charactor Color */
#define COL_FCOLOR      0xf00
#define COL_FBLACK      0x800
#define COL_FRED        0x900
#define COL_FGREEN      0xa00
#define COL_FYELLOW     0xb00
#define COL_FBLUE       0xc00
#define COL_FMAGENTA    0xd00
#define COL_FCYAN       0xe00
#define COL_FWHITE      0xf00
#define COL_FTERM       0x000

#define S_COLORED       0xf00

#ifdef BG_COLOR
/* Background Color */
#define COL_BCOLOR      0xf000
#define COL_BBLACK      0x8000
#define COL_BRED        0x9000
#define COL_BGREEN      0xa000
#define COL_BYELLOW     0xb000
#define COL_BBLUE       0xc000
#define COL_BMAGENTA    0xd000
#define COL_BCYAN       0xe000
#define COL_BWHITE      0xf000
#define COL_BTERM       0x0000

#define S_BCOLORED      0xf000
#endif


#define S_GRAPHICS      0x10

#define S_DIRTY         0x20

#define SETPROP(var,prop) (var = (((var)&S_DIRTY) | prop))

/* Line status */
#define L_DIRTY         0x01
#define L_UNUSED        0x02
#define L_NEED_CE       0x04
#define L_CLRTOEOL      0x08

#define ISDIRTY(d)      ((d) & L_DIRTY)
#define ISUNUSED(d)     ((d) & L_UNUSED)
#define NEED_CE(d)      ((d) & L_NEED_CE)

typedef unsigned short l_prop;

typedef struct scline {
  char            lineimage[MAX_COLUMN];
  l_prop          lineprop[MAX_COLUMN];
  short           isdirty;
  short           eol;
}               Screen;

static
TerminalMode    d_ioval;
static
int             tty;
static
FILE           *ttyf;

static
char            bp[1024], funcstr[256];

char           *T_cd, *T_ce, *T_kr, *T_kl, *T_cr, *T_bt, *T_ta, *T_sc,
               *T_rc, *T_so, *T_se, *T_us, *T_ue, *T_cl, *T_cm, *T_al,
               *T_sr, *T_md, *T_me, *T_ti, *T_te, *T_nd, *T_as, *T_ae,
               *T_eA, *T_ac, *T_op;

int             LINES, COLS;
static int      tab_step = 8;
static int      CurLine, CurColumn;
static Screen   ScreenElem[MAX_LINE], *ScreenImage[MAX_LINE];
static l_prop   CurrentMode = 0;
static int      graph_enabled = 0;

char            DisplayCode = DISPLAY_CODE;

static char     gcmap[96];

extern int      tgetent(char*,char*);
extern int      tgetnum(char*);
extern int      tgetflag(char*);
extern char    *tgetstr(char*,char**);
extern char    *tgoto(char *, int, int);
extern int      tputs(char*,int,int(*)(char));
void            putchars(unsigned char, unsigned char, FILE *);
void            clear(),wrap(),touch_line(), touch_column(int);
#ifdef JP_CHARSET
void            switch_wchar(FILE*);
void            switch_ascii(FILE*);
#endif
void            need_clrtoeol(void), clrtoeol(void);

int write1(char);

/* #define writestr(s)  tputs(s,1,write1) */

static void writestr(char *s)
{
  tputs(s,1,write1);
}

#define MOVE(line,column)       writestr(tgoto(T_cm,column,line));

int
set_tty(void)
{
  char *ttyn;
  if (isatty(0))                /* stdin */
    ttyn = ttyname(0);
  else
#ifndef __EMX__
    ttyn = "/dev/tty";
#else
    ttyn = "con";
#endif
  tty = open(ttyn, O_RDWR);
  if (tty < 0) {
    /* use stderr instead of stdin... is it OK???? */
    tty = 2;
  }
  ttyf = fdopen(tty, "w");
  TerminalGet(tty, &d_ioval);
  return 0;
}

void
ttymode_set(int mode,int imode)
{
  int             er;
  TerminalMode    ioval;

  TerminalGet(tty, &ioval);
  MODEFLAG(ioval) |= mode;
#ifndef SGTTY
  IMODEFLAG(ioval) |= imode;
#endif

  er = TerminalSet(tty, &ioval);

  if (er == -1) {
    printf("Error occured while set %x: errno=%d\n", mode, errno);
    reset_exit(SIGNAL_ARGLIST);
  }
}

void
ttymode_reset(int mode,int imode)
{
  int             er;
  TerminalMode    ioval;

  TerminalGet(tty, &ioval);
  MODEFLAG(ioval) &= ~mode;
#ifndef SGTTY
  IMODEFLAG(ioval) &= ~imode;
#endif

  er = TerminalSet(tty, &ioval);

  if (er == -1) {
    printf("Error occured while reset %x: errno=%d\n", mode, errno);
    reset_exit(SIGNAL_ARGLIST);
  }
}

#ifndef SGTTY
void
set_cc(int spec, int val)
{
  int             er;
  TerminalMode    ioval;

  TerminalGet(tty, &ioval);
  ioval.c_cc[spec] = val;
  er = TerminalSet(tty, &ioval);
  if (er == -1) {
    printf("Error occured: errno=%d\n", errno);
    reset_exit(SIGNAL_ARGLIST);
  }
}
#endif

void
reset_tty(void)
{
  if (DisplayCode != 'E' && DisplayCode != 'S')
    writestr("\033(B");  /* designate US_ASCII */
  writestr(T_op);       /* turn off */
  writestr(T_me);
  if (!Do_not_use_ti_te) {
    if (T_te && *T_te)
      writestr(T_te);
    else
      writestr(T_cl);
  }
  writestr(T_se); /* reset terminal */
  fflush(ttyf);
  TerminalSet(tty, &d_ioval);
  close(tty);
}

MySignalHandler
reset_exit(SIGNAL_ARG)
{
  reset_tty();
#ifdef MOUSE
  if (mouseActive)
    mouse_end();
#endif
  exit(0);
  SIGNAL_RETURN;
}

MySignalHandler
error_dump(SIGNAL_ARG)
{
  signal(SIGIOT,SIG_DFL);
  reset_tty();
  abort();
  SIGNAL_RETURN;
}

void
set_int(void)
{
  signal(SIGHUP, reset_exit);
  signal(SIGINT, reset_exit);
  signal(SIGQUIT, reset_exit);
  signal(SIGTERM, reset_exit);
  signal(SIGILL, error_dump);
  signal(SIGIOT, error_dump);
  signal(SIGFPE, error_dump);
  signal(SIGBUS, error_dump);
/*  signal(SIGSEGV, error_dump); */
}


static void
setgraphchar(void)
{
  int c, i, n;

  for (c = 0; c < 96; c++)
    gcmap[c] = (char) (c + ' ');
  
  if (!T_ac)
    return;

  n = strlen(T_ac);
  for (i = 0; i < n-1; i += 2) {
    c = (unsigned)T_ac[i] - ' ';
    if (c >= 0 && c < 96)
      gcmap[c] = T_ac[i+1];
  }
}

#define graphchar(c) (((unsigned)(c)>=' ' && (unsigned)(c)<128)? gcmap[(c)-' '] : (c))

extern char *allocStr(char*,int);
#define GETSTR(v,s) {v = pt; suc = tgetstr(s,&pt); if (!suc) v = ""; else v = allocStr(suc,0); }

void
getTCstr(void)
{
  char           *suc;
  char           *pt = funcstr;

  tgetent(bp, getenv("TERM"));

  GETSTR(T_ce, "ce");           /* clear to the end of line */
  GETSTR(T_cd, "cd");           /* clear to the end of display */
  GETSTR(T_kr, "nd");           /* cursor right */
  if (suc == NULL)
    GETSTR(T_kr, "kr");
  if (tgetflag("bs"))
    T_kl = "";                        /* cursor left */
  else {
    GETSTR(T_kl, "le");
    if (suc == NULL)
      GETSTR(T_kl, "kb");
    if (suc == NULL)
      GETSTR(T_kl, "kl");
  }
  GETSTR(T_cr, "cr");           /* carrige return */
  GETSTR(T_ta, "ta");           /* tab */
  GETSTR(T_sc, "sc");           /* save cursor */
  GETSTR(T_rc, "rc");           /* restore cursor */
  GETSTR(T_so, "so");           /* standout mode */
  GETSTR(T_se, "se");           /* standout mode end */
  GETSTR(T_us, "us");           /* underline mode */
  GETSTR(T_ue, "ue");           /* underline mode end */
  GETSTR(T_md, "md");           /* bold mode */
  GETSTR(T_me, "me");           /* bold mode end */
  GETSTR(T_cl, "cl");           /* clear screen */
  GETSTR(T_cm, "cm");           /* cursor move */
  GETSTR(T_al, "al");           /* append line */
  GETSTR(T_sr, "sr");           /* scroll reverse */
  GETSTR(T_ti, "ti");           /* terminal init */
  GETSTR(T_te, "te");           /* terminal end */
  GETSTR(T_nd, "nd");           /* move right one space */
  GETSTR(T_eA, "eA");           /* enable alternative charset */
  GETSTR(T_as, "as");           /* alternative (graphic) charset start */ 
  GETSTR(T_ae, "ae");           /* alternative (graphic) charset end */
  GETSTR(T_ac, "ac");           /* graphics charset pairs */
  GETSTR(T_op, "op");           /* set default color pair to its original value */
#ifdef CYGWIN
/* for TERM=pcansi on MS-DOS prompt.
  T_as = "\033[12m";
  T_ae = "\033[10m";
  T_ac = "l\001k\002m\003j\004x\005q\006n\020a\024v\025w\026u\027t\031";
*/
  T_as[0] = '\0';
  T_ae[0] = '\0';
  T_ac[0] = '\0';
#endif
 
  LINES = COLS = 0;
  setlinescols();
  setgraphchar();
}

#ifndef TIOCGWINSZ
#include <sys/ioctl.h>
#endif

void
setlinescols(void)
{
  char           *p;
  int             i;
#if defined(TERMIOS) && defined(TIOCGWINSZ)
  struct winsize  wins;

  i = ioctl(tty, TIOCGWINSZ, &wins);
  if (i >= 0 && wins.ws_row != 0 && wins.ws_col != 0) {
    LINES = wins.ws_row;
    COLS = wins.ws_col;
  }
#endif
  if (LINES <= 0 &&
      (p = getenv("LINES")) != NULL &&
      (i = atoi(p)) >= 0)
    LINES = i;
  if (COLS <= 0 &&
      (p = getenv("COLUMNS")) != NULL &&
      (i = atoi(p)) >= 0)
    COLS = i;
  if (LINES <= 0)
    LINES = tgetnum("li");      /* number of line */
  if (COLS <= 0)
    COLS = tgetnum("co");       /* number of column */
  if (COLS > MAX_COLUMN)
    COLS = MAX_COLUMN;
  if (LINES > MAX_LINE)
    LINES = MAX_LINE;
}

void
setupscreen(void)
{
  int             i;

  for (i = 0; i < LINES; i++) {
    ScreenImage[i] = &ScreenElem[i];
    ScreenImage[i]->lineprop[0] = S_EOL;
    ScreenImage[i]->isdirty = 0;
  }
  for (; i < MAX_LINE; i++) {
    ScreenElem[i].isdirty = L_UNUSED;
  }

  clear();
}

/*
 * Screen initialize
 */
int
initscr(void)
{
  if (set_tty() < 0)
    return -1;
  set_int();
  getTCstr();
  if (T_ti && !Do_not_use_ti_te)
    writestr(T_ti);
  setupscreen();
  return 0;
}

static int      wmode = C_ASCII;
static char     wbuf;

int
write1(char c)
{
#ifdef SCREEN_DEBUG
  usleep(50);
#endif
#ifdef JP_CHARSET
  if (c & 0x80) {
    switch (wmode) {
    case C_ASCII:
      switch_wchar(ttyf);
    case C_WCHAR2:
      wmode = C_WCHAR1;
      wbuf = c;
      break;
    case C_WCHAR1:
      wmode = C_WCHAR2;
      putchars((unsigned char) wbuf, (unsigned char) c, ttyf);
      break;
    }
  } else {
    switch (wmode) {
    case C_ASCII:
      break;
    case C_WCHAR1:
      /* ignore byte */
      wmode = C_ASCII;
      switch_ascii(ttyf);
      break;
    case C_WCHAR2:
      wmode = C_ASCII;
      switch_ascii(ttyf);
      break;
    }
    putc(c, ttyf);
  }
#else
  putc(c, ttyf);
#endif
#ifdef SCREEN_DEBUG
  fflush(ttyf);
#endif
  return 0;
}

#ifdef JP_CHARSET
void
endline(void)
{                               /* End of line */
  if (wmode != C_ASCII) {
    switch_ascii(ttyf);
    wmode = C_ASCII;
  }
}

void
switch_ascii(FILE * f)
{
  extern char *GetSOCode(char);
  if (CODE_JIS(DisplayCode)) {
    fputs(GetSOCode(DisplayCode), f);
  }
}

void
switch_wchar(FILE * f)
{
  extern char *GetSICode(char);
  if (CODE_JIS(DisplayCode)) {
    fputs(GetSICode(DisplayCode), f);
  }
}

void
putchars(unsigned char c1, unsigned char c2, FILE * f)
{
  switch (DisplayCode) {
  case CODE_EUC:
    putc(c1, f);
    putc(c2, f);
    break;
  case CODE_JIS1:
  case CODE_JIS2:
  case CODE_JIS3:
  case CODE_JIS4:
    putc(c1 & 0x7f, f);
    putc(c2 & 0x7f, f);
    break;
  case CODE_SJIS:
    c1 &= 0x7f;
    c2 &= 0x7f;
    c1 -= 0x20;
    c2 -= 0x20;
    if ((c1 & 1) == 0)
      c2 += 94;
    c1 = ((c1 - 1) >> 1) + 0x81;
    c2 += 0x3f;
    if (c1 > 0x9f)
      c1 += 0x40;
    if (c2 > 0x7e)
      c2++;
    putc(c1, f);
    putc(c2, f);
    break;
  }
}
#endif

void
move(int line, int column)
{
  if (line >= 0 && line < LINES)
    CurLine = line;
  if (column >= 0 && column < COLS)
    CurColumn = column;
}

#ifdef BG_COLOR
#define M_SPACE (S_SCREENPROP|S_COLORED|S_BCOLORED|S_GRAPHICS)
#else
#define M_SPACE (S_SCREENPROP|S_COLORED|S_GRAPHICS)
#endif

static int 
need_redraw(char c1, l_prop pr1, char c2, l_prop pr2)
{
  if (c1 != c2)
    return 1;
  if (c1 == ' ')
    return (pr1 ^ pr2) & M_SPACE & ~S_DIRTY;

  if ((pr1 ^ pr2) & ~S_DIRTY)
    return 1;

  return 0;
}

#define M_CEOL (~(M_SPACE|C_WHICHCHAR))

void
addch(char c)
{
  char           *p;
  l_prop         *pr;
  int             dest, i;
  short          *dirty;

  if (CurColumn == COLS)
    wrap();
  if (CurColumn >= COLS)
    return;
  p = ScreenImage[CurLine]->lineimage;
  pr = ScreenImage[CurLine]->lineprop;
  dirty = &ScreenImage[CurLine]->isdirty;

  if (pr[CurColumn] & S_EOL) {
    if (c == ' ' &&
#ifdef JP_CHARSET
        CHMODE(CurrentMode) != C_WCHAR1 &&
#endif
        !(CurrentMode & M_SPACE)) {
      CurColumn++;
      return;
    }
    for (i = CurColumn; i >= 0 && (pr[i] & S_EOL); i--) {
      p[i] = ' ';
      SETPROP(pr[i], (pr[i] & M_CEOL) | C_ASCII);
    }
  }
  
  if (c == '\t' || c == '\n' || c == '\r' || c == '\b')
    SETCHMODE(CurrentMode, C_CTRL);
#ifdef JP_CHARSET
  else if (CHMODE(CurrentMode) == C_WCHAR1)
    SETCHMODE(CurrentMode, C_WCHAR2);
  else if (c & 0x80)
    SETCHMODE(CurrentMode, C_WCHAR1);
#endif
  else if (c & ~0x1f)
    SETCHMODE(CurrentMode, C_ASCII);
  else
    return;

  /* Required to erase bold or underlined character for some
     terminal emulators. */
  if (((pr[CurColumn] & S_BOLD) &&
       need_redraw(p[CurColumn], pr[CurColumn], c, CurrentMode)) ||
      ((pr[CurColumn] & S_UNDERLINE) && !(CurrentMode & S_UNDERLINE))) {
    touch_line();
    if (CurColumn < COLS-1) {
      touch_column(CurColumn+1);
      if (pr[CurColumn+1] & S_EOL) {
        p[CurColumn+1] = ' ';
        SETPROP(pr[CurColumn+1], (pr[CurColumn+1] & M_CEOL) | C_ASCII);
      } 
#ifdef JP_CHARSET
      else if (CHMODE(pr[CurColumn+1]) == C_WCHAR1 && CurColumn < COLS-2)
        touch_column(CurColumn+2);
#endif
    }
  }

#ifdef JP_CHARSET
  if (CurColumn >= 1 && CHMODE(pr[CurColumn-1]) == C_WCHAR1 &&
      CHMODE(CurrentMode) != C_WCHAR2) {
    p[CurColumn-1] = ' ';
    SETPROP(pr[CurColumn-1], (pr[CurColumn-1] & ~C_WHICHCHAR) | C_ASCII);
    touch_line();
    touch_column(CurColumn-1);
  }
  
  if (CurColumn < COLS - 1 && CHMODE(pr[CurColumn+1]) == C_WCHAR2 &&
      CHMODE(CurrentMode) != C_WCHAR1) {
    p[CurColumn+1] = ' ';
    SETPROP(pr[CurColumn+1], (pr[CurColumn+1] & ~C_WHICHCHAR) | C_ASCII);
    touch_line();
    touch_column(CurColumn+1);
  }
    
  if (CurColumn == COLS - 1 && CHMODE(CurrentMode) == C_WCHAR1) {
    wrap();
    p = ScreenImage[CurLine]->lineimage;
    pr = ScreenImage[CurLine]->lineprop;
  }
#endif
  if (CHMODE(CurrentMode) != C_CTRL) {
    if (need_redraw(p[CurColumn], pr[CurColumn], c, CurrentMode)) {
      p[CurColumn] = c;
      SETPROP(pr[CurColumn], CurrentMode);
      touch_line();
      touch_column(CurColumn);
#ifdef JP_CHARSET
      if (CHMODE(CurrentMode) == C_WCHAR1)
        touch_column(CurColumn+1);
      else if (CHMODE(CurrentMode) == C_WCHAR2)
        touch_column(CurColumn-1);
#endif
    }
    CurColumn++;
  } else if (c == '\t') {
    dest = (CurColumn + tab_step) / tab_step * tab_step;
    if (dest >= COLS) {
      wrap();
      touch_line();
      dest = tab_step;
      p = ScreenImage[CurLine]->lineimage;
      pr = ScreenImage[CurLine]->lineprop;
    }
    for (i = CurColumn; i < dest; i++) {
      if (need_redraw(p[i], pr[i], ' ', CurrentMode)) {
        p[i] = ' ';
        SETPROP(pr[i], CurrentMode);
        touch_line();
        touch_column(i);
      }
    }
    CurColumn = i;
  } else if (c == '\n') {
    wrap();
  } else if (c == '\r') {       /* Carrige return */
    CurColumn = 0;
  } else if (c == '\b' && CurColumn > 0) {      /* Backspace */
    CurColumn--;
#ifdef JP_CHARSET
    if (CurColumn > 0 &&
        CHMODE(pr[CurColumn]) == C_WCHAR2)
      CurColumn--;
#endif
  }
}

void
wrap(void)
{
  if (CurLine == LINES - 1)
    return;
  CurLine++;
  CurColumn = 0;
}

void
touch_column(int col)
{
  if (col >= 0 && col < COLS)
    ScreenImage[CurLine]->lineprop[col] |= S_DIRTY;
}

void
touch_line(void)
{
  if (!(ScreenImage[CurLine]->isdirty & L_DIRTY)) {
    int i;
    for (i = 0; i < COLS; i++)
      ScreenImage[CurLine]->lineprop[i] &= ~S_DIRTY;
    ScreenImage[CurLine]->isdirty |= L_DIRTY;
  }

}

void
standout(void)
{
  CurrentMode |= S_STANDOUT;
}

void
standend(void)
{
  CurrentMode &= ~S_STANDOUT;
}

void
toggle_stand(void)
{
  l_prop *pr = ScreenImage[CurLine]->lineprop;
  pr[CurColumn] ^= S_STANDOUT;
#ifdef JP_CHARSET
  if (CHMODE(pr[CurColumn]) == C_WCHAR1)
    pr[CurColumn+1] ^= S_STANDOUT;
#endif
}

void
bold(void)
{
  CurrentMode |= S_BOLD;
}

void
boldend(void)
{
  CurrentMode &= ~S_BOLD;
}

void
underline(void)
{
  CurrentMode |= S_UNDERLINE;
}

void
underlineend(void)
{
  CurrentMode &= ~S_UNDERLINE;
}

void
graphstart(void)
{
  CurrentMode |= S_GRAPHICS;
}

void
graphend(void)
{
  CurrentMode &= ~S_GRAPHICS;
}

int
graph_ok(void)
{
#ifndef KANJI_SYMBOLS
  if (no_graphic_char)
    return 0;
#endif
  return T_as[0] != 0 && T_ae[0] != 0 && T_ac[0] != 0;
}

void
setfcolor(int color)
{
  CurrentMode &= ~COL_FCOLOR;
  if ((color & 0xf) <= 7)
    CurrentMode |= (((color & 7) | 8) << 8);
}

static char    *
color_seq(int colmode)
{
  static char     seqbuf[32];
  sprintf(seqbuf, "\033[%dm", ((colmode >> 8) & 7) + 30);
  return seqbuf;
}

#ifdef BG_COLOR
void
setbcolor(int color)
{
  CurrentMode &= ~COL_BCOLOR;
  if ((color & 0xf) <= 7)
    CurrentMode |= (((color & 7) | 8) << 12);
}

static char    *
bcolor_seq(int colmode)
{
  static char     seqbuf[32];
  sprintf(seqbuf, "\033[%dm", ((colmode >> 12) & 7) + 40);
  return seqbuf;
}
#endif

#define RF_NEED_TO_MOVE    0
#define RF_CR_OK           1
#define RF_NONEED_TO_MOVE  2
#ifdef BG_COLOR
#define M_MEND (S_STANDOUT|S_UNDERLINE|S_BOLD|S_COLORED|S_BCOLORED|S_GRAPHICS)
#else
#define M_MEND (S_STANDOUT|S_UNDERLINE|S_BOLD|S_COLORED|S_GRAPHICS)
#endif
void
refresh(void)
{
  int             line, col, pcol;
  int             pline = CurLine;
  int             moved = RF_NEED_TO_MOVE;
  char           *pc;
  l_prop         *pr, mode = 0;
  l_prop          color = COL_FTERM;
#ifdef BG_COLOR
  l_prop          bcolor = COL_BTERM;
#endif
  short          *dirty;

  for (line = 0; line < LINES; line++) {
    dirty = &ScreenImage[line]->isdirty;
    if (*dirty & L_DIRTY) {
      *dirty &= ~L_DIRTY;
      pc = ScreenImage[line]->lineimage;
      pr = ScreenImage[line]->lineprop;
      for (col = 0; col < COLS && !(pr[col] & S_EOL); col++) {
        if (*dirty & L_NEED_CE && col >= ScreenImage[line]->eol) {
          if (need_redraw(pc[col], pr[col],  ' ', 0))
            break;
        } else {
          if (pr[col] & S_DIRTY)
            break;
        }
      }
      if (*dirty & (L_NEED_CE|L_CLRTOEOL)) {
        pcol = ScreenImage[line]->eol;
	if (pcol >= COLS) {
	  *dirty &= ~(L_NEED_CE|L_CLRTOEOL);
	  pcol = col;
	}
      } else {
        pcol = col;
      }
      if (line < LINES - 2 && pline == line - 1 && pcol == 0) {
        switch (moved) {
        case RF_NEED_TO_MOVE:
          MOVE(line,0);
          moved = RF_CR_OK;
          break;
        case RF_CR_OK:
          write1('\n');
          write1('\r');
          break;
        case RF_NONEED_TO_MOVE:
          moved = RF_CR_OK;
          break;
        }
      }
      else {
        MOVE(line, pcol);
        moved = RF_CR_OK;
      }
      if (*dirty & (L_NEED_CE|L_CLRTOEOL)) {
        writestr(T_ce);
        if (col != pcol)
          MOVE(line, col);
      }
      pline = line;
      pcol = col;
      for (; col < COLS; col++) {
        if (pr[col] & S_EOL)
          break;

        /* some terminal emulators do linefeed when a character
           is put on COLS-th column. this behavior is different
           from one of vt100, but such terminal emulators are
           used as vt100-compatible emulators. This behaviour
           causes scroll when a character is drawn on (COLS-1,LINES-1)
           point. To avoid the scroll, I prohibit to draw character
           on (COLS-1,LINES-1).
           */
#ifndef BG_COLOR
        if (line == LINES-1 && col == COLS-1)
          break;
#endif
        if ((!(pr[col] & S_STANDOUT) && (mode & S_STANDOUT)) ||
            (!(pr[col] & S_UNDERLINE) && (mode & S_UNDERLINE)) ||
            (!(pr[col] & S_BOLD) && (mode & S_BOLD)) ||
            (!(pr[col] & S_COLORED) && (mode & S_COLORED))
#ifdef BG_COLOR
            || (!(pr[col] & S_BCOLORED) && (mode & S_BCOLORED))
#endif
            || (!(pr[col] & S_GRAPHICS) && (mode & S_GRAPHICS))) {
          if ((!(pr[col] & S_COLORED) && (mode & S_COLORED))
#ifdef BG_COLOR
              || (!(pr[col] & S_BCOLORED) && (mode & S_BCOLORED))
#endif
            )
            writestr(T_op);
          if (!(pr[col] & S_GRAPHICS) && (mode & S_GRAPHICS))
            writestr(T_ae);
          writestr(T_me);
          mode &= ~M_MEND;
        }
        if ((*dirty & L_NEED_CE && col >= ScreenImage[line]->eol)?
            need_redraw(pc[col], pr[col],  ' ', 0) : (pr[col] & S_DIRTY)) {
          if (pcol == col-1)
            writestr(T_nd);
          else if (pcol != col)
            MOVE(line, col);

          if ((pr[col] & S_COLORED) && (pr[col] ^ mode) & COL_FCOLOR) {
            color = (pr[col] & COL_FCOLOR);
            mode = ((mode & ~COL_FCOLOR) | color);
            writestr(color_seq(color));
          }
#ifdef BG_COLOR
          if ((pr[col] & S_BCOLORED) && (pr[col] ^ mode) & COL_BCOLOR) {
            bcolor = (pr[col] & COL_BCOLOR);
            mode = (mode & ~COL_BCOLOR | bcolor);
            writestr(bcolor_seq(bcolor));
          }
#endif
          if ((pr[col] & S_STANDOUT) && !(mode & S_STANDOUT)) {
            writestr(T_so);
            mode |= S_STANDOUT;
          }
          if ((pr[col] & S_UNDERLINE) && !(mode & S_UNDERLINE)) {
            writestr(T_us);
            mode |= S_UNDERLINE;
          }
          if ((pr[col] & S_BOLD) && !(mode & S_BOLD)) {
            writestr(T_md);
            mode |= S_BOLD;
          }
          if ((pr[col] & S_GRAPHICS) && !(mode & S_GRAPHICS)) {
            if (!graph_enabled) {
              graph_enabled = 1;
              writestr(T_eA);
            }
            writestr(T_as);
            mode |= S_GRAPHICS;
          }
          write1((pr[col] & S_GRAPHICS) ? graphchar(pc[col]) : pc[col]);
          pcol = col +1;
        }
      }
      if (col == COLS)
        moved = RF_NEED_TO_MOVE;
      for (; col < COLS && !(pr[col] & S_EOL); col++)
        pr[col] |= S_EOL;
    }
    *dirty &= ~(L_NEED_CE|L_CLRTOEOL);
    if (mode & M_MEND) {
      if (mode & (S_COLORED
#ifdef BG_COLOR
                  |S_BCOLORED
#endif
        ))
        writestr(T_op);
      if (mode & S_GRAPHICS)
        writestr(T_ae);
      writestr(T_me);
      mode &= ~M_MEND;
    }
#ifdef JP_CHARSET
    endline();
#endif
  }
  MOVE(CurLine, CurColumn);
  fflush(ttyf);
}

void
clear(void)
{
  int             i, j;
  l_prop         *p;
  writestr(T_cl);
  move(0, 0);
  for (i = 0; i < LINES; i++) {
    ScreenImage[i]->isdirty = 0;
    p = ScreenImage[i]->lineprop;
    for (j = 0; j < COLS; j++) {
      p[j] = S_EOL;
    }
  }
  CurrentMode = C_ASCII;
}

void
scroll_raw(void)
{                               /* raw scroll */
  MOVE(LINES-1, 0);
  write1('\n');
}

void
scroll(void)
{                               /* scroll up */
  int             cli = CurLine, cco = CurColumn;
  Screen         *t = ScreenImage[0];
  int             i;

  scroll_raw();
  move(cli, cco);
  for (i = 1; i < LINES; i++) {
    ScreenImage[i - 1] = ScreenImage[i];
  }
  ScreenImage[LINES - 1] = t;
  for (i = 0; i < COLS; i++)
    t->lineprop[i] = S_EOL;
  t->isdirty = 0;
}

void
rscroll(void)
{                               /* scroll up */
  int             cli = CurLine, cco = CurColumn;
  Screen         *t = ScreenImage[LINES-1];
  int             i, j;
  int             redraw_all = 1;

  if (T_sr && *T_sr) {
    MOVE(0,0);
    writestr(T_sr);
    redraw_all = 0;
  }
  move(cli, cco);
  for (i = LINES-1; i >= 0; i--) {
    ScreenImage[i] = ScreenImage[i-1];
    if (redraw_all) {
      ScreenImage[i]->isdirty |= L_DIRTY;
      for (j = 0; j < COLS; j++)
        ScreenImage[i]->lineprop[j] |= S_DIRTY;
    }
  }
  ScreenImage[0] = t;
  for (i = 0; i < COLS; i++)
    t->lineprop[i] = S_EOL;
  t->isdirty = 0;
}

#if 0
void
need_clrtoeol(void)
{                               /* Clear to the end of line as the need arises */
  l_prop         *lprop = ScreenImage[CurLine]->lineprop;

  if (lprop[CurColumn] & S_EOL)
    return;

  if (!(ScreenImage[CurLine]->isdirty & (L_NEED_CE|L_CLRTOEOL)) ||
      ScreenImage[CurLine]->eol > CurColumn)
    ScreenImage[CurLine]->eol = CurColumn;

  ScreenImage[CurLine]->isdirty |= L_NEED_CE;
}
#endif

void
clrtoeol(void)
{                               /* Clear to the end of line */
  int             i;
  l_prop         *lprop = ScreenImage[CurLine]->lineprop;

  if (lprop[CurColumn] & S_EOL)
    return;

  if (!(ScreenImage[CurLine]->isdirty & (L_NEED_CE|L_CLRTOEOL)) ||
      ScreenImage[CurLine]->eol > CurColumn)
    ScreenImage[CurLine]->eol = CurColumn;

  ScreenImage[CurLine]->isdirty |= L_CLRTOEOL;
  touch_line();
  for (i = CurColumn; i < COLS && !(lprop[i] & S_EOL); i++) {
    lprop[i] = S_EOL|S_DIRTY;
  }
}

#ifdef BG_COLOR
void
clrtoeol_with_bcolor(void)
{
  int i, cli, cco;
  l_prop pr;
  
  if (!(CurrentMode & S_BCOLORED)) {
    clrtoeol();
    return;
  }
  cli = CurLine;
  cco = CurColumn;
  pr = CurrentMode;
  CurrentMode = CurrentMode & (M_CEOL|S_BCOLORED) | C_ASCII;
  for (i = CurColumn; i < COLS; i++)
    addch(' ');
  move(cli,cco);
  CurrentMode = pr;
}

void
clrtoeolx(void)
{
  clrtoeol_with_bcolor();
}
#else

void
clrtoeolx(void)
{
  clrtoeol();
}
#endif

void
clrtobot_eol(void (* clrtoeol)())
{ 
  int             l, c;

  l = CurLine;
  c = CurColumn;
  (*clrtoeol)();
  CurColumn = 0;
  CurLine++;
  for (; CurLine < LINES; CurLine++)
    (*clrtoeol)();
  CurLine = l;
  CurColumn = c;
}

void
clrtobot(void)
{
  clrtobot_eol(clrtoeol);
}

void
clrtobotx(void)
{
  clrtobot_eol(clrtoeolx);
}

#if 0
void
no_clrtoeol(void)
{
  int             i;
  l_prop         *lprop = ScreenImage[CurLine]->lineprop;
  
  ScreenImage[CurLine]->isdirty &= ~L_CLRTOEOL;
}
#endif

void
addstr(char *s)
{
  while (*s != '\0')
    addch(*(s++));
}

void
addnstr(char *s, int n)
{
  int i;

#ifdef JP_CHARSET
  for(i = 0; i < n - 1 && *s != '\0'; i++)
    addch(*(s++));
  if (*s != '\0') {
    if (*s & 0x80) {                    /* WCHAR */
      if (CHMODE(CurrentMode) == C_WCHAR1)       /* WCHAR second byte */
        addch(*s);
    } else                              /* Ascii or WCHAR2 */
      addch(*s);
  }
#else
  for(i = 0; i < n && *s != '\0'; i++)
    addch(*(s++));
#endif
}

void
addnstr_sup(char *s, int n)
{
  int i;

#ifdef JP_CHARSET
  for(i = 0; i < n - 1 && *s != '\0'; i++)
    addch(*(s++));
  if (*s != '\0') {
    if (*s & 0x80) {                    /* WCHAR */
      if (CHMODE(CurrentMode) == C_WCHAR1) {     /* WCHAR second byte */
        addch(*s);
        i++;
      }
    } else {                            /* Ascii or WCHAR2 */
      addch(*s);
      i++;
    }
  }
#else
  for(i = 0; i < n && *s != '\0'; i++)
    addch(*(s++));
#endif
  for(; i< n; i++)
    addch(' ');
}

void
crmode(void)
#ifndef SGTTY
{
  ttymode_reset(ICANON,IXON);
  ttymode_set(ISIG,0);
#ifdef TERMIOS
  set_cc(VMIN, 1);
#else
  set_cc(VEOF, 1);
#endif
}
#else
{
  ttymode_set(CBREAK,0);
}
#endif

void
nocrmode(void)
#ifndef SGTTY
{
  ttymode_set(ICANON,0);
#ifdef TERMIOS
  set_cc(VMIN, 4);
#else
  set_cc(VEOF, 4);
#endif
}
#else
{
  ttymode_reset(CBREAK,0);
}
#endif

void
term_echo(void) {
  ttymode_set(ECHO,0);
}

void
term_noecho(void) {
  ttymode_reset(ECHO,0);
}

void
term_raw(void)
#ifndef SGTTY
#ifdef IEXTEN
#define TTY_MODE ISIG|ICANON|ECHO|IEXTEN
#else
#define TTY_MODE ISIG|ICANON|ECHO
#endif
{
  ttymode_reset(TTY_MODE,IXON|IXOFF);
#ifdef TERMIOS
  set_cc(VMIN, 1);
#else
  set_cc(VEOF, 1);
#endif
}
#else
{
  ttymode_set(RAW,0);
}
#endif

void
term_cooked(void)
#ifndef SGTTY
{
  ttymode_set(TTY_MODE,0);
#ifdef TERMIOS
  set_cc(VMIN, 4);
#else
  set_cc(VEOF, 4);
#endif
}
#else
{
  ttymode_reset(RAW,0);
}
#endif

void
term_cbreak(void)
{
  term_cooked();
  term_noecho();
}

char
getch(void)
{
  char            c;

  read(tty, &c, 1);
  return c;
}

#ifdef USE_GPM
char
wgetch(void)
{
  char            c;

  read(tty, &c, 1);
  return c;
}

int
do_getch()
{
  if (is_xterm)
    return getch();
  else
    return Gpm_Getch();
}
#endif

void
bell(void)
{
  write1(7);
}

void
sleep_till_anykey(int sec, int purge)
{
  fd_set rfd;
  struct timeval tim;
  int             er;
  TerminalMode    ioval;

  TerminalGet(tty, &ioval);
  term_raw();
  
  tim.tv_sec = sec;
  tim.tv_usec = 0;

  FD_ZERO(&rfd);
  FD_SET(tty,&rfd);

  if (select(tty+1,&rfd,0,0,&tim) > 0 && purge)
      getch();
  er = TerminalSet(tty, &ioval);
  if (er == -1) {
    printf("Error occured: errno=%d\n", errno);
    reset_exit(SIGNAL_ARGLIST);
  }
}

#ifdef MOUSE

#define XTERM_ON   {fputs("\033[?1001s\033[?1000h",ttyf); flush_tty();}
#define XTERM_OFF  {fputs("\033[?1000l\033[?1001r",ttyf); flush_tty();}
void mouse_init()
{
  char *term;
#ifdef USE_GPM
  Gpm_Connect conn;
  extern int gpm_process_mouse(Gpm_Event*,void*);

  if (mouseActive) return;
  term = getenv("TERM");
  if (!strncmp(term,"kterm",5) || !strncmp(term,"xterm",5)) {
    is_xterm = 1;
  }
  else {
    conn.eventMask = ~0;
    conn.defaultMask = GPM_MOVE|GPM_HARD;
    conn.maxMod = 0;
    conn.minMod = 0;
    Gpm_Open(&conn,0); /* don't care even if it fails */
    gpm_handler = gpm_process_mouse;
  }
#else
  term = getenv("TERM"); 
  if (!strncmp(term,"kterm",5) || !strncmp(term,"xterm",5)) {
    is_xterm = 1;
  }
#endif
  mouseActive = 1;
}

void mouse_end()
{
  if (mouseActive == 0) return;
  if (is_xterm) {
    XTERM_OFF;
  }
#ifdef USE_GPM
  else
    Gpm_Close();
#endif
  mouseActive = 0;
}

void mouse_active()
{
  if (!mouseActive)
    mouse_init();
  if (is_xterm)
    XTERM_ON;
}

void mouse_inactive()
{
  if (is_xterm)
    XTERM_OFF;
}
  
#endif

void flush_tty()
{
	fflush(ttyf);
}
