
#include <stdio.h>
#include <ctype.h>
#include "string.h"
#include <errno.h>

#include "viz.h"

extern errno;

extern char *malloc();

void
printlist(level, list)
int level;
MEMBER *list;
{
    MEMBER *p;

    if (level > 0)
	(void) fprintf(stderr, "%*.*s ", level*4, level*4, " ");

    for (p = list; p != NIL; p = p->next) {

	if (p->count == UPTO_EOF)
	    (void) fprintf(stderr, "<inf>");
	else if (p->count < 0)
	    (void) fprintf(stderr, "$%c", - p->count);
	else
	    (void) fprintf(stderr, "%d", p->count);

	if (p->type == T_LISTHEAD) {

	    (void) fprintf(stderr, "(sublist) ");

	} else if (p->type == T_COPYOUT) {

	    (void) fprintf(stderr, "[%s] ", p->u.copyout);

	} else if (p->type == T_SEEK) {

	    (void) fprintf(stderr, "%l!%d ", p->u.seek.count,
						p->u.seek.direction);

	} else if (p->type == T_MATH) {

	    (void) fprintf(stderr, "$%c%c%d ",
		p->u.math.reg_no, p->u.math.operator, p->u.math.operand);

	} else if (p->type == T_NEWLINE) {

	    (void) fputs("\\n ", stderr);

	} else {

	    (void) fprintf(stderr, "(%c,%c,%d",
		p->u.iospec.ichar, p->u.iospec.ochar, p->u.iospec.size);
	    if (p->u.iospec.reg_no > 0)
		(void) fprintf(stderr, ">%c", p->u.iospec.reg_no);
	    (void) fprintf(stderr, ") ");
	}
    }
    (void) fprintf(stderr, "\n");

    for (p = list; p != NIL; p = p->next) {
	if (p->type == T_LISTHEAD)
	    printlist(level+1, p->u.sublist);
    }
}

/* Create a new list with mem its first member.
 * Do this by
 *   - allocating a T_LISTHEAD member that points to mem as its sublist;
 *   - give the listhead a repeat count of 1;
 *   - give the listhead a NIL next field.
 * Return ptr to the listhead.
 */
MEMBER *
newlist(mem)
MEMBER *mem;
{
    MEMBER *p;

    p = (MEMBER *) malloc(sizeof(MEMBER));
    if (p != NIL) {
	p->next = NIL;
	p->type = T_LISTHEAD;
	p->count = 1;
	p->u.sublist = mem;
    }
    return p;
}

/* Add a member to a list; returns 1st arg */
MEMBER *addmember(list, member)
MEMBER *list;	/* listhead member */
MEMBER *member;
{
    MEMBER *p;

    p = list->u.sublist;
    while (p->next)
	p = p->next;

    p->next = member;
    member->next = NIL;

    return list;
}

/* Makes the core of an iospec -- everything except for a reg_no is
 * filled in (fmt is NULL, but is implied by the ichar/ochar setting).
 */
IOSPEC
makecore(ichar, ochar)
int ichar;
int ochar;
{
    IOSPEC io;
    int defaultichar();
    int defaultochar();

    if (ichar == 0 && ochar == 0) {
	ichar = 'C';
	ochar = 'a';
    } else if (ichar == 0) {
	ichar = defaultichar(ochar);
    } else if (ochar == 0) {
	ochar = defaultochar(ichar);
    }

    io.size = getsize(ichar);
    io.ichar = ichar;
    io.ochar = ochar;
    io.fmt = NULL;
    io.reg_no = 0;

    return io;
}

/* Makes the core of an iospec when a printf-style outform is given */
IOSPEC
makecorepct(ichar, fmt)
int ichar;
char *fmt;
{
    IOSPEC io;

    if (ichar == 0) {
	io.ichar = 'C';
	io.ochar = 'a';
    } else {
	io.ichar = ichar;
	io.ochar = defaultochar(ichar);
    }

    io.size = getsize(ichar);
    io.fmt = fmt;
    io.reg_no = 0;

    return io;
}

/* Condenses a list.
 * Returns the total number of changes made to the list.

 * condenselist() does the following:
 *	- combines similar adjacent members into single members;
 *	- combines similar adjacent sublists into single sublists;
 *	- merges sublists that have unit repeat count or are of
 *	  unit length into the body of the list;
 */
int
condenselist(list)
MEMBER *list;	/* pointer to a listhead member */
{
    /* Work over each member of list.  Each pass is recursive on
     * sublists, so that all sublists are completely processed before
     * a current list is processed.

     * Combining duplicates is done first, so that sublists can be
     * merged into the current list, if at all possible.  However,
     * once a list has been merged, there may be a new duplicate
     * at the merge point.  Therefore, if there are any merged sublists,
     * we have to repeat the member-combining passes.

     * Pass 1: comb_dup_nl()
     *    Combine adjacent duplicate non-lists into single members.

     * Pass 2: comb_dup_list()
     *    Combine adjacent duplicate lists into single lists.

     * Pass 3: merge_sublist()
     *    Merge sublists that have unit repeat counts or are one
     *    element long into current list.

     */
    int nmerge;
    int nchange = 0;

    do {
	nchange += comb_dup_nl(list);
	nchange += comb_dup_list(list);
	nmerge = merge_sublist(list);
	nchange += nmerge;
    } while (nmerge > 0);
    return nchange;
}


int
comb_dup_nl(list)
MEMBER *list;	/* Pointer to a member of list; can be listhead or otherwise */
{
    register MEMBER *p, *q;

    /* Combine adjacent duplicate non-list members into single members. */

    int nchange = 0;

    /* Process sublists first */
    for (p = list; p != NIL; p = p->next) {
	if (p->type == T_LISTHEAD)
	    nchange += comb_dup_nl(p->u.sublist);
    }

    /* Now process duplicate non-list members */
    p = list;
    while ( (q = p->next) != NIL) {
	if (p->type == q->type) {
	    if (
		 (p->type == T_IOSPEC &&
		    (p->count == UPTO_EOF || p->count >= 0) &&
				(q->count == UPTO_EOF || q->count >= 0) &&
		    p->u.iospec.reg_no == q->u.iospec.reg_no &&
		    p->u.iospec.size == q->u.iospec.size &&
		    p->u.iospec.ichar == q->u.iospec.ichar &&
		    p->u.iospec.ochar == q->u.iospec.ochar &&
		    p->u.iospec.fmt == q->u.iospec.fmt) ||
		(p->type == T_MATH  &&
		    p->u.math.reg_no == q->u.math.reg_no &&
		    p->u.math.operator == q->u.math.operator &&
		    p->u.math.operand == q->u.math.operand) ||
		(p->type == T_COPYOUT  &&
		    (strcmp(p->u.copyout, q->u.copyout) == 0)) ||
		(p->type == T_NEWLINE)
	      ) {
		/* Two successive non-list members do the same thing;
		 * join them.
		 */
		nchange++;
		if (p->count == UPTO_EOF || q->count == UPTO_EOF)
		    p->count = UPTO_EOF;
		else
		    p->count += q->count;
		p->next = q->next;
		/* Don't advance member pointer p; that way, we'll
		 * compare the combined member to next one.
		 */
	     }  else {
		/* Can't reduce these members; advance to next member */
		p = q;
	    }
	} else {
	    /* Can't reduce these members; advance to next member */
	    p = q;
	}
    }
    return nchange;
}

int
comb_dup_list(list)
MEMBER *list;	/* Pointer to a member of list; can be listhead or otherwise */
{
    register MEMBER *p, *q;
    int nchange;

    /* Combine adjacent duplicate lists into single lists. */

    for (nchange = 0, p=list; p != NIL; ) {

	if (p->type != T_LISTHEAD) {
	    /* Not a list */
	    p = p->next;
	    continue;
	}
	nchange += comb_dup_list(p->u.sublist);

	q = p->next;
	if (q == NIL)
	    return nchange;
	
	if (q->type != T_LISTHEAD) {
	    /* Not a sublist */
	    p = q->next;
	    continue;
	}

	if (((p->count > 0 && q->count > 0) ||
		(p->count == UPTO_EOF && q->count == UPTO_EOF)) &&
	    duplicates(p->u.sublist, q->u.sublist)) {
	    /* Two successive lists do the same thing; join them. */
	    nchange++;
	    if (p->count == UPTO_EOF || q->count == UPTO_EOF)
		p->count = UPTO_EOF;
	    else
		p->count += q->count;
	    p->next = q->next;

	    /* Don't advance member pointer p; that way, we'll
	     * compare the combined member to next one.
	     */
	} else {
	    /* Can't reduce these members; advance to next member */
	    p = q;
	}
    }
    return nchange;
}

int
merge_sublist(list)
MEMBER *list;	/* Pointer to a member of list; can be listhead or otherwise */
{
    /* Merge sublists of list that have unit repeat counts or are one
     * element long into list.

     * Return resulting list length.
     */

    register MEMBER *p, *end;
    int nchange;
    long test;

    for (nchange=0, p=list; p != NIL; p = p->next) {

	/* p points to successive members of list */

	if (p->type != T_LISTHEAD) {
	    /* Not a listhead; nothing to merge here */
	    continue;
	}

	/* First, merge p's sublists into it. */
	nchange += merge_sublist(p->u.sublist);


	/* In following line, note that a single-member sublist
	 * has the value p->u.sublist->next == NIL.
	 */

	if ( (p->count < 0 && p->count != UPTO_EOF) ||
			(p->count > 1 &&  p->u.sublist->next != NIL)) {
	    /* Can't merge sublist into current list:
	     * either a count comes from a register, or
	     * sublist has multiple members and our repeat count > 1.
	     */
	    continue;
	}

	/* Get here if the sublist is repeated once or if it's
	 * got just 1 member.  Merge the sublist into current list.
	 */
	nchange++;
	if (p->count == UPTO_EOF || p->count > 1) {
	    /* Sublist must have only one member.  Fold the repeat
	     * counts together, so that the listhead member contains
	     * a count of 1, and it can then be discarded.
	     */

	    /* If either count is UPTO_EOF, use that as total count.
	     * Otherwise, don't merge unless the product of the two
	     * values can still fit into a long.
	     */
	    if (p->count == UPTO_EOF || p->u.sublist->count == UPTO_EOF) {
		p->u.sublist->count = UPTO_EOF;
	    } else {
		test = p->u.sublist->count * p->count;
		if (test / p->count != p->u.sublist->count)
		    continue;
		else
		    p->u.sublist->count = test;
	    }
	}
	/* The listhead now has a repeat count of one.  Replace it with
	 * the sublist.  Find end of list, and connect it to successor
	 * of current listhead.
	 */
	end = p->u.sublist;
	while (end->next)
	    end = end->next;
	end->next = p->next;

	/* Copy the first member of the sublist onto the
	 * listhead member.
	
	 * [[ It would be better to just discard the listhead member
	 * and insert the sublist member, but since the lists are
	 * singly-linked, we don't easily know who is pointing to
	 * the listhead, so we can't adjust their pointers.
	 * An alternative would be to make the listhead member
	 * into a no-op that points to the former first member
	 * of the sublist, and optionally remove the no-op on
	 * a second pass.  But since the list member structures
	 * are small and discarded listheads will probably be
	 * only occasional, copying is cheaper than a ton of compares
	 * on a second pass. ]]
	 */
	*p = *(p->u.sublist);
    }
    return nchange;
}

duplicates(list1, list2)
MEMBER *list1, *list2;	/* listhead or other list member */
{
    /* Returns non-zero if two lists are the same, including their
     * count fields.  However, if either count field is < 0, the two
     * lists will automatically compare unequal, so that we don't have to
     * try to determine if the count registers will necessarily contain
     * the same value at execution time.

     * Note that if you want to compare the xxx and yyy in the
     * lists 3(xxx) and 14(yyy), you should pass pointers to the first
     * members of xxx and yyy, so that you don't compare the repeat
     * counts "3" and "14", which will of course compare unequal.

     */

    for (; list1 != NIL && list2 != NIL;
	    list1 = list1->next, list2 = list2->next) {

	if (list1->type != list2->type)
	    return 0;

	if (list1->count < 0 || list2->count < 0 ||
	    list1->count != list2->count)
	    return 0;
	
	switch (list1->type) {
	
	case T_COPYOUT:

	    if (strcmp(list1->u.copyout, list2->u.copyout) != 0)
		return 0;
	    break;

	case T_MATH:

	    if (list1->u.math.reg_no != list2->u.math.reg_no ||
		list1->u.math.operator != list2->u.math.operator ||
		list1->u.math.operand != list2->u.math.operand)
		return 0;
	    break;

	case T_IOSPEC:

	    if (list1->u.iospec.size != list2->u.iospec.size ||
		list1->u.iospec.ichar != list2->u.iospec.ichar ||
		list1->u.iospec.ochar != list2->u.iospec.ochar ||
		list1->u.iospec.fmt != list2->u.iospec.fmt ||
		list1->u.iospec.reg_no != list2->u.iospec.reg_no)
		return 0;
	    break;

	case T_LISTHEAD:

	    if (duplicates(list1->u.sublist, list2->u.sublist) == 0)
		return 0;
	
	    break;

	default:
	    (void) fprintf(stderr,
		"%s: unknown type %d in duplicates()\n", prog, list1->type);
	}
    }
    if (list1 != NIL || list2 != NIL)
	return 0; /* list lengths differ */
    else
	return 1;

    /* NOTREACHED */
}

/* Returns the number of members of a sublist; returns 0 if list is
 * not a listhead.
 */
listlen(list)
MEMBER *list;	/* Points to a listhead */
{
    int n;
    if (list->type != T_LISTHEAD)
	return 0;

    for (n=0, list = list->u.sublist; list != NIL; n++, list = list->next)
	;
    return n;
}
